% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/particle_smoother.R
\name{particle_smoother}
\alias{particle_smoother}
\alias{particle_smoother.lineargaussian}
\alias{particle_smoother.nongaussian}
\alias{particle_smoother.ssm_nlg}
\alias{particle_smoother.ssm_sde}
\title{Particle Smoothing}
\usage{
particle_smoother(model, particles, ...)

\method{particle_smoother}{lineargaussian}(
  model,
  particles,
  method = "psi",
  seed = sample(.Machine$integer.max, size = 1),
  ...
)

\method{particle_smoother}{nongaussian}(
  model,
  particles,
  method = "psi",
  seed = sample(.Machine$integer.max, size = 1),
  max_iter = 100,
  conv_tol = 1e-08,
  ...
)

\method{particle_smoother}{ssm_nlg}(
  model,
  particles,
  method = "bsf",
  seed = sample(.Machine$integer.max, size = 1),
  max_iter = 100,
  conv_tol = 1e-08,
  iekf_iter = 0,
  ...
)

\method{particle_smoother}{ssm_sde}(
  model,
  particles,
  L,
  seed = sample(.Machine$integer.max, size = 1),
  ...
)
}
\arguments{
\item{model}{A model object of class \code{bssm_model}.}

\item{particles}{Number of particles as a positive integer. Suitable values
depend on the model, the data, and the chosen algorithm. While larger values
provide more accurate estimates, the run time also increases with respect to
the number of particles, so it is generally a good idea to test the filter
first with a small number of particles, e.g., less than 100.}

\item{...}{Ignored.}

\item{method}{Choice of particle filter algorithm.
For Gaussian and non-Gaussian models with linear dynamics,
options are \code{"bsf"} (bootstrap particle filter, default for
non-linear models)
and \code{"psi"} (\eqn{\psi}-APF, the default for other models), and
for non-linear models option \code{"ekf"} (extended Kalman particle filter)
is also available.}

\item{seed}{Seed for the C++ RNG (positive integer).}

\item{max_iter}{Maximum number of iterations used in Gaussian approximation,
as a positive integer.
Default is 100 (although typically only few iterations are needed).}

\item{conv_tol}{Positive tolerance parameter used in Gaussian approximation.
Default is 1e-8.}

\item{iekf_iter}{Non-negative integer. If zero (default), first
approximation for non-linear Gaussian models is obtained from extended
Kalman filter. If \code{iekf_iter > 0}, iterated extended Kalman filter is
used with \code{iekf_iter} iterations.}

\item{L}{Positive integer defining the discretization level for SDE model.}
}
\value{
List with samples (\code{alpha}) from the smoothing distribution
and corresponding weights (\code{weights}),
as well as smoothed means and covariances (\code{alphahat} and \code{Vt})
of the states and
estimated log-likelihood (\code{logLik}).
}
\description{
Function \code{particle_smoother} performs particle smoothing
based on either bootstrap particle filter (Gordon et al. 1993),
\eqn{\psi}-auxiliary particle filter (\eqn{\psi}-APF) (Vihola et al. 2020),
extended Kalman particle filter (Van Der Merwe et al. 2001),
or its version based on iterated EKF (Jazwinski, 1970).
The smoothing phase is based on the filter-smoother algorithm by
Kitagawa (1996).
}
\details{
See one of the vignettes for \eqn{\psi}-APF in case of nonlinear models.
}
\examples{
set.seed(1)
x <- cumsum(rnorm(100))
y <- rnorm(100, x)
model <- ssm_ulg(y, Z = 1, T = 1, R = 1, H = 1, P1 = 1)
system.time(out <- particle_smoother(model, particles = 1000))
# same with simulation smoother:
system.time(out2 <- sim_smoother(model, particles = 1000, 
  use_antithetic = TRUE))
ts.plot(out$alphahat, rowMeans(out2), col = 1:2)

}
\references{
Gordon, NJ, Salmond, DJ, Smith, AFM (1993).
Novel approach to nonlinear/non-Gaussian Bayesian state estimation.
IEE Proceedings-F, 140, 107-113.
https://doi.org/10.1049/ip-f-2.1993.0015

Vihola, M, Helske, J, Franks, J. Importance sampling type estimators
based on approximate marginal Markov chain Monte Carlo.
Scand J Statist. 2020; 1-38.
https://doi.org/10.1111/sjos.12492

Van Der Merwe, R, Doucet, A, De Freitas, N,  Wan, EA (2001).
The unscented particle filter.
In Advances in neural information processing systems, p 584-590.

Jazwinski, A 1970. Stochastic Processes and Filtering Theory.
Academic Press.

Kitagawa, G (1996). Monte Carlo filter and smoother for non-Gaussian
nonlinear state space models.
Journal of Computational and Graphical Statistics, 5, 1-25.
https://doi.org/10.2307/1390750
}
