#' Get point data from a c3d object
#'
#' Get the point data of an c3d object in a data frame.
#'
#' The point data of imported c3d objects in `c3dr` is saved as a list of lists.
#' This is good for internal handling, but for analysis a table format (a data
#' frame) is often more convenient.`c3d_data()` returns the point data from an
#' imported c3d object as a data frame.
#'
#' Analyses of data frames may require them to have different formats. For
#' `c3d_data` output, different data formats (`"wide"`, `"long"`, `"longest"`)
#' are available. See the section below for more details. You can convert
#' between different formats with [c3d_convert()].
#'
#' # Data Formats
#'
#' ## Wide
#'
#' The wide format has three numeric columns per point (x, y, z). The column
#' names have the structure `pointname_type`, so for example the x-coordinate of
#' a point named `C7` has the name `C7_x`. Each row corresponds to one recording
#' frame.
#'
#' ## Long
#'
#' The long format has one column per point. The column names correspond to the
#' names of the points. Each recording frame corresponds to three rows of data
#' (x, y, z). In additional to the point columns containing numeric data there
#' are two additional columns: One that indicates the frame number (`frame`,
#' numeric) and one that indicates the coordinate type (`type`, either x, y, or
#' z, as a character).
#'
#' ## Longest
#'
#' The longest format has one data column (`value`, numeric). The other columns
#' indicate the frame number (`frame`, numeric), the coordinate type (`type`,
#' either x, y, or z, as a character), and the point name (`point`, character).
#' Thus, each row of the data frame corresponds to one data entry.
#'
#' @param x A c3d object, as imported by [c3d_read()].
#' @param format Either `"wide"` (default), `"long"`, or `"longest"` to
#'   determine the format of the resulting data frame. See the Data Formats
#'   section for more details.
#'
#' @return A data frame of class `c3d_data` with the c3d point data. The
#'   structure of the data frame depends on the 'format' argument.
#'
#' @examples
#' # Import example data
#' d <- c3d_read(c3d_example())
#'
#' # wide format (default)
#' w <- c3d_data(d)
#' head(w)
#'
#' # long format
#' l <- c3d_data(d, format = "long")
#' head(l)
#'
#' # longest format
#' ll <- c3d_data(d, format = "longest")
#' head(ll)
#' @export
c3d_data <- function(x, format = "wide") {
  if (!inherits(x, "c3d")) stop("'x' needs to be a list of class 'c3d'.")
  if (x$parameters$POINT$USED == 0) stop("No point data available")
  # change data format from nested list to data.frame (wide format)
  out <- as.data.frame(
    matrix(unlist(x$data), nrow = x$header$nframes, byrow = TRUE)
  )
  # get label names
  colnames(out) <- paste0(
    rep(x$parameters$POINT$LABELS[seq_len(ncol(out) / 3)], each = 3),
    c("_x", "_y", "_z")
  )
  class(out) <- c("c3d_data_wide", "c3d_data", "data.frame")

  if (format == "wide") {
    out
  } else {
    c3d_convert(out, format)
  }
}


#' Convert between c3d point data formats
#'
#' Convert between different representations of point data in c3d files.
#'
#' @param data A data frame of class `c3d_data` as generated by [c3d_data()].
#'   The data can have any of the three data formats (see Data Formats section
#'   below).
#' @inheritParams c3d_data
#'
#' @inheritSection c3d_data Data Formats
#'
#' @return A data frame of class `c3d_data` in the format specified by the
#'   `format` argument.
#'
#' @examples
#' # Import example data
#' d <- c3d_read(c3d_example())
#' # get point data in wide format
#' w <- c3d_data(d, format = "wide")
#' # convert to long data
#' l <- c3d_convert(w, "long")
#' head(l)
#' @export
c3d_convert <- function(data, format) {
  if (!inherits(data, "c3d_data")) {
    stop(
      "'data' needs to be a data frame of class",
      " 'c3d_data' as generated by 'cd3_data()'."
    )
  }

  # input: wide
  if (inherits(data, "c3d_data_wide")) {
    if (format == "wide") {
      message("Data is already in 'wide' format.")
      data
    } else if (format == "long") {
      c3d_wide_to_long(data)
    } else if (format == "longest") {
      c3d_long_to_longest(c3d_wide_to_long(data))
    } else {
      stop("'format' argument must be either 'wide', 'long', or 'longest'")
    }
  } else if (inherits(data, "c3d_data_long")) { # input: long
    if (format == "wide") {
      c3d_long_to_wide(data)
    } else if (format == "long") {
      message("Data is already in 'long' format.")
      data
    } else if (format == "longest") {
      c3d_long_to_longest(data)
    } else {
      stop("'format' argument must be either 'wide', 'long', or 'longest'")
    }
  } else if (inherits(data, "c3d_data_longest")) { # input: longest
    if (format == "wide") {
      c3d_long_to_wide(c3d_longest_to_long(data))
    } else if (format == "long") {
      c3d_longest_to_long(data)
    } else if (format == "longest") {
      message("Data is already in 'longest' format.")
      data
    } else {
      stop("'format' argument must be either 'wide', 'long', or 'longest'")
    }
  }
}

#' Convert from wide to long data
#'
#' Internal function for `c3d_data` objects to convert from wide to long data
#' format.
#'
#' @noRd
c3d_wide_to_long <- function(x) {
  if (!inherits(x, "c3d_data_wide")) {
    stop("'x' needs to be a data frame of class 'c3d' in 'wide' format.")
  }
  # get new column names
  old_names <- colnames(x)[0:(ncol(x) / 3 - 1) * 3 + 1]
  new_names <- sub("_([xyz])$", "", old_names)

  # get split factors for reshape
  splt <- split(seq_len(ncol(x)), rep(1:(ncol(x) / 3), each = 3))
  r <- stats::reshape(
    data = x,
    varying = splt,
    v.names = new_names,
    direction = "long",
    timevar = "type",
    times = c("x", "y", "z"),
    idvar = "frame"
  )
  # reorder columns
  r <- r[, c(ncol(r), seq_along(r)[-ncol(r)])]
  # reorder rows
  r <- r[order(r$frame, r$type), ]
  rownames(r) <- NULL

  # attach class
  class(r) <- c("c3d_data_long", "c3d_data", "data.frame")
  r
}

#' Convert from long to wide data
#'
#' Internal function for `c3d_data` objects to convert from wide to long data
#' format.
#'
#' @noRd
c3d_long_to_wide <- function(x) {
  if (!inherits(x, "c3d_data_long")) {
    stop("'x' needs to be a data frame of class 'c3d' in 'long' format.")
  }

  # Get column names excluding 'frame' and 'type'
  value_cols <- setdiff(names(x), c("frame", "type"))

  # Reshape from long to wide
  r <- stats::reshape(
    data = x,
    timevar = "type",
    idvar = "frame",
    v.names = value_cols,
    direction = "wide"
  )

  # Fix the column ordering
  result <- data.frame(matrix(nrow = nrow(r)))[-1] # empty data frame

  # Fill in the data in correct order
  for (i in seq_along(value_cols)) {
    col_base <- value_cols[i]
    result[[paste0(col_base, "_x")]] <- r[[paste0(col_base, ".x")]]
    result[[paste0(col_base, "_y")]] <- r[[paste0(col_base, ".y")]]
    result[[paste0(col_base, "_z")]] <- r[[paste0(col_base, ".z")]]
  }

  # attach class
  class(result) <- c("c3d_data_wide", "c3d_data", "data.frame")
  result
}

#' Convert from long to longest data
#'
#' Internal function for `c3d_data` objects to convert from long to longest data
#' format.
#'
#' @noRd
c3d_long_to_longest <- function(x) {
  if (!inherits(x, "c3d_data_long")) {
    stop("'x' needs to be a data frame of class 'c3d' in 'long' format.")
  }

  vary <- names(x)[-(1:2)]
  r <- stats::reshape(
    data = x,
    varying = list(vary),
    v.names = "value",
    direction = "long",
    timevar = "point",
    times = vary
  )

  # reorder rows based on frame-type-point
  # preserve original point order (ordering by a factor)
  # also delete id column
  p_factor <- factor(r$point, levels = unique(r$point))
  r <- r[order(r$frame, p_factor, r$type), -ncol(r)]
  rownames(r) <- NULL

  # attach class
  class(r) <- c("c3d_data_longest", "c3d_data", "data.frame")
  r
}

#' Convert from longest to long data
#'
#' Internal function for `c3d_data` objects to convert from longest to long data
#' format.
#'
#' @noRd
c3d_longest_to_long <- function(x) {
  if (!inherits(x, "c3d_data_longest")) {
    stop("'x' needs to be a data frame of class 'c3d' in 'longest' format.")
  }

  # reshape from longest to long format
  r <- stats::reshape(
    data = x,
    timevar = "point",
    idvar = c("frame", "type"),
    v.names = "value",
    direction = "wide"
  )

  # rename columns by removing "value." prefix that reshape adds
  names(r)[-(1:2)] <- sub("^value\\.", "", names(r)[-(1:2)])

  rownames(r) <- NULL
  attr(r, "reshapeWide") <- NULL

  # attach class
  class(r) <- c("c3d_data_long", "c3d_data", "data.frame")
  r
}

#' Get analog data from a c3d object
#'
#' Get the analog data of a c3d object in a data frame.
#'
#' The analog data of imported c3d objects in `c3dr` is saved as a list of
#' lists. This is good for internal handling, but for analysis a table format (a
#' data frame) is often more convenient.`c3d_analog()` returns the analog data
#' from an imported c3d object as a data frame.
#'
#' @param x A c3d object, as imported by [c3d_read()].
#'
#' @return A data.frame with with n rows and m columns, where n is the number of
#'   frames recorded times the number of analog subframes per frame, and m as
#'   the number of recorded analog channels.
#'
#' @examples
#' # Import example data
#' d <- c3d_read(c3d_example())
#'
#' # get analog data
#' a <- c3d_analog(d)
#' head(a)
#' @export
c3d_analog <- function(x) {
  # change data format from list of matrices to data.frame
  if (x$parameters$ANALOG$USED == 0) stop("No analog data available")
  out <- as.data.frame(do.call(rbind, x$analog))
  colnames(out) <- x$parameters$ANALOG$LABELS[seq_len(ncol(out))]
  class(out) <- c("c3d_analog", "data.frame")
  out
}
