% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/operators.R
\name{divergence}
\alias{divergence}
\alias{\%divergence\%}
\title{Numerical and Symbolic Divergence}
\usage{
divergence(
  f,
  var,
  params = list(),
  coordinates = "cartesian",
  accuracy = 4,
  stepsize = NULL,
  drop = TRUE
)

f \%divergence\% var
}
\arguments{
\item{f}{array of \code{characters} or a \code{function} returning a \code{numeric} array.}

\item{var}{vector giving the variable names with respect to which the derivatives are to be computed and/or the point where the derivatives are to be evaluated. See \code{\link{derivative}}.}

\item{params}{\code{list} of additional parameters passed to \code{f}.}

\item{coordinates}{coordinate system to use. One of: \code{cartesian}, \code{polar}, \code{spherical}, \code{cylindrical}, \code{parabolic}, \code{parabolic-cylindrical} or a vector of scale factors for each varibale.}

\item{accuracy}{degree of accuracy for numerical derivatives.}

\item{stepsize}{finite differences stepsize for numerical derivatives. It is based on the precision of the machine by default.}

\item{drop}{if \code{TRUE}, return the divergence as a scalar and not as an \code{array} for vector-valued functions.}
}
\value{
Scalar for vector-valued functions when \code{drop=TRUE}, \code{array} otherwise.
}
\description{
Computes the numerical divergence of \code{functions} or the symbolic divergence of \code{characters}
in arbitrary \href{https://en.wikipedia.org/wiki/Orthogonal_coordinates#Table_of_orthogonal_coordinates}{orthogonal coordinate systems}.
}
\details{
The divergence of a vector-valued function \eqn{F_i} produces a scalar value 
\eqn{\nabla \cdot F} representing the volume density of the outward flux of the 
vector field from an infinitesimal volume around a given point. 
The \code{divergence} is computed in arbitrary orthogonal coordinate systems using the 
scale factors \eqn{h_i}:

\deqn{\nabla \cdot F = \frac{1}{J}\sum_i\partial_i\Biggl(\frac{J}{h_i}F_i\Biggl)}

where \eqn{J=\prod_ih_i}. When \eqn{F} is an \code{array} of vector-valued functions 
\eqn{F_{d_1\dots d_n,i}}, the \code{divergence} is computed for each vector:

\deqn{(\nabla \cdot F)_{d_1\dots d_n} = \frac{1}{J}\sum_i\partial_i\Biggl(\frac{J}{h_i}F_{d_1\dots d_n,i}\Biggl)}
}
\section{Functions}{
\itemize{
\item \code{f \%divergence\% var}: binary operator with default parameters.

}}
\examples{
### symbolic divergence of a vector field
f <- c("x^2","y^3","z^4")
divergence(f, var = c("x","y","z"))

### numerical divergence of a vector field in (x=1, y=1, z=1)
f <- function(x,y,z) c(x^2, y^3, z^4)
divergence(f, var = c(x=1, y=1, z=1))

### vectorized interface
f <- function(x) c(x[1]^2, x[2]^3, x[3]^4)
divergence(f, var = c(1,1,1)) 

### symbolic array of vector-valued 3-d functions
f <- array(c("x^2","x","y^2","y","z^2","z"), dim = c(2,3))
divergence(f, var = c("x","y","z"))

### numeric array of vector-valued 3-d functions in (x=0, y=0, z=0)
f <- function(x,y,z) array(c(x^2,x,y^2,y,z^2,z), dim = c(2,3))
divergence(f, var = c(x=0, y=0, z=0))
 
### binary operator
c("x^2","y^3","z^4") \%divergence\% c("x","y","z")

}
\references{
Guidotti, E. (2020). "calculus: High dimensional numerical and symbolic calculus in R". \url{https://arxiv.org/abs/2101.00086}
}
\seealso{
Other differential operators: 
\code{\link{curl}()},
\code{\link{derivative}()},
\code{\link{gradient}()},
\code{\link{hessian}()},
\code{\link{jacobian}()},
\code{\link{laplacian}()}
}
\concept{differential operators}
