% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/other_methods.R
\name{wrapper}
\alias{wrapper}
\alias{sva.wrapper}
\alias{ruv.wrapper}
\alias{leapp.wrapper}
\title{Wrapper functions for some previous methods}
\usage{
sva.wrapper(
  formula,
  X.data = NULL,
  Y,
  r,
  sva.method = c("irw", "two-step"),
  B = 5
)

ruv.wrapper(
  formula,
  X.data = NULL,
  Y,
  r,
  nc,
  lambda = 1,
  ruv.method = c("RUV2", "RUV4", "RUVinv")
)

leapp.wrapper(
  formula,
  X.data = NULL,
  Y,
  r,
  search.tuning = F,
  ipod.method = c("hard", "soft")
)
}
\arguments{
\item{formula}{a formula indicating the known covariates including both primary variables and nuisance variables, which are seperated by \code{|}. The variables before \code{|} are primary variables and the variables after \code{|} are nuisance variables. It's OK if there is no nuisance variables, then \code{|} is not needed and \code{formula} becomes a typical formula with all the covariates considered primary. When there is confusion about where the intercept should be put, \code{cate} will include it in X.nuis.}

\item{X.data}{the data frame used for \code{formula}}

\item{Y}{outcome, n*p matrix}

\item{r}{number of latent factors, can be estimated using the function \code{est.confounder.num}}

\item{sva.method}{parameter for \code{\link[sva]{sva}}.
whether to use an iterative reweighted algorithm (irw) or a two-step algorithm (two-step).}

\item{B}{parameter for \code{\link[sva]{sva}}. the number of iterations of the irwsva algorithm}

\item{nc}{parameter for \link{ruv} functions: position of the negative controls}

\item{lambda}{parameter for \code{\link[ruv:RUVinv]{RUVinv}}}

\item{ruv.method}{either using \code{\link[ruv:RUV2]{RUV2}}, \code{\link[ruv:RUV4]{RUV4}} or
\code{\link[ruv:RUVinv]{RUVinv}} functions}

\item{search.tuning}{logical parameter for \code{\link[leapp]{leapp}}, whether using BIC to search for tuning parameter of IPOD.}

\item{ipod.method}{parameter for \code{\link[leapp]{leapp}}. "hard": hard thresholding in the IPOD algorithm;
"soft": soft thresholding in the IPOD algorithm}
}
\value{
All functions return \code{beta.p.value} which are the p-values after adjustment.
For the other returned objects, refer to \link{cate} for their meaning.
}
\description{
These functions provide an uniform interface to three existing methods: SVA, RUV, LEAPP
The wrapper functions transform the data into desired forms and call the corresponding functions in the package
\link{sva}, \link{ruv}, \link{leapp}
}
\details{
The \code{beta.p.values} returned is a length \code{p} vector, each for the overall effects of all the primary variables.

Only 1 variable of interest is allowed for \code{leapp.wrapper}. The method can be slow.
}
\examples{
## this is the simulation example in Wang et al. (2015).
n <- 100
p <- 1000
r <- 2
set.seed(1)
data <- gen.sim.data(n = n, p = p, r = r,
                     alpha = rep(1 / sqrt(r), r),
                     beta.strength = 3 * sqrt(1 + 1) / sqrt(n),
                     Gamma.strength = c(seq(3, 1, length = r)) * sqrt(p),
                     Sigma = 1 / rgamma(p, 3, rate = 2),
                     beta.nonzero.frac = 0.05)
X.data <- data.frame(X1 = data$X1)
sva.results <- sva.wrapper(~ X1 | 1, X.data, data$Y,
                           r = r, sva.method = "irw")
ruv.results <- ruv.wrapper(~ X1 | 1, X.data, data$Y, r = r,
                           nc = sample(data$beta.zero.pos, 30), ruv.method = "RUV4")
leapp.results <- leapp.wrapper(~ X1 | 1, X.data, data$Y, r = r)
cate.results <- cate(~ X1 | 1, X.data, data$Y, r = r)

## p-values after adjustment
par(mfrow = c(2, 2))
hist(sva.results$beta.p.value)
hist(ruv.results$beta.p.value)
hist(leapp.results$beta.p.value)
hist(cate.results$beta.p.value)

## type I error
mean(sva.results$beta.p.value[data$beta.zero.pos] < 0.05)

## power
mean(sva.results$beta.p.value[data$beta.nonzero.pos] < 0.05)

## false discovery proportion for sva
discoveries.sva <- which(p.adjust(sva.results$beta.p.value, "BH") < 0.2)
fdp.sva <- length(setdiff(discoveries.sva, data$beta.nonzero.pos)) / max(length(discoveries.sva), 1)
fdp.sva

}
