% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smi.R
\name{smi}
\alias{smi}
\title{Single-Mediator-Imputation Estimation Method}
\usage{
smi(
  fit.r = NULL,
  fit.m,
  fit.y,
  treat,
  covariates,
  sims = 100,
  conf.level = 0.95,
  conditional = TRUE,
  cluster = NULL,
  long = FALSE,
  mc.cores = 1L
)
}
\arguments{
\item{fit.r}{a fitted model object for treatment. Can be of class 'CBPS' or
'SumStat'. Default is 'NULL'. Only necessary if 'conditional' is 'FALSE'.}

\item{fit.m}{a fitted model object for mediator. Can be of class 'lm', 'glm',
'multinom', or 'polr'.}

\item{fit.y}{a fitted model object for outcome. Can be of class 'lm' or 'glm'.}

\item{treat}{a character string indicating the name of the treatment variable
used in the models. The treatment can be categorical with two or more
categories (two- or multi-valued factor).}

\item{covariates}{a vector containing the name of the covariate variable(s)
used in the models. Each covariate can be categorical with two or more
categories (two- or multi-valued factor) or continuous (numeric).}

\item{sims}{number of Monte Carlo draws for nonparametric bootstrap.}

\item{conf.level}{level of the returned two-sided confidence intervals,
which are estimated by the nonparametric percentile bootstrap method.
Default is .95, which returns the 2.5 and 97.5 percentiles of the simulated
quantities.}

\item{conditional}{a logical value. If 'TRUE', the function will return the
estimates conditional on those covariate values; and all covariates in
mediator and outcome models need to be centered prior to fitting.
Default is 'TRUE'. If 'FALSE', 'fit.r' needs to be specified.}

\item{cluster}{a vector of cluster indicators for the bootstrap. If provided,
the cluster bootstrap is used. Default is 'NULL'.}

\item{long}{a logical value. If 'TRUE', the output will contain the entire
sets of estimates for all bootsrap samples. Default is 'FALSE'.}

\item{mc.cores}{The number of cores to use. Must be exactly 1 on Windows.}
}
\value{
\item{result}{a matrix containing the point estimates of the initial disparity,
disparity remaining, and disparity reduction, and the percentile bootsrap
confidence intervals for each estimate.}
\item{all.result}{a matrix containing the point estimates of the initial disparity,
disparity remaining, and disparity reduction for all bootsrap samples. Returned
if 'long' is 'TRUE'.}
}
\description{
'smi' is used to estimate the initial disparity, disparity reduction, and
disparity remaining for causal decomposition analysis, using the
single-mediator-imputation estimation method proposed by Park et. al. (2021+).
}
\details{
This function returns the point estimates of the initial disparity,
disparity reduction, and disparity remaining for a categorical
treatment and a variety of types of outcome and mediator(s) in causal
decomposition analysis. It also returns nonparametric
percentile bootstrap confidence intervals for each estimate.

The definition of the initial disparity, disparity reduction, and
disparity remaining can be found in help('mmi'). As opposed to the 'mmi'
function, this function uses the single-mediator-imputation method
suggested by Park et al. (2021+). See the reference for more details.

If one wants to make the inference conditional on baseline covariates,
set 'conditional = T' and center the data before fitting the models.

As of version 0.0.1, the mediator model ('fit.m') can be of class 'lm', 'glm',
'multinom', or 'polr', corresponding respectively to the linear regression
models and generalized linear models, multinomial log-linear models, and
ordered response models. The outcome model ('fit.y') can be of class 'lm' or 'glm'.
Also, the treatment model ('fit.r') can be of class 'CBPS' or 'SumStat', both of
which use the propensity score weighting. It is only necessary when 'conditional = F'.
}
\examples{
data(sdata)

#-----------------------------------------------------------#
# Example 1-a: Continuous Outcome
#-----------------------------------------------------------#
require(PSweight)
fit.r1 <- SumStat(R ~ C.num + C.bin, data = sdata, weight = "IPW")
\donttest{require(CBPS)
fit.r2 <- CBPS(R ~ C.num + C.bin, data = sdata, method = "exact",
          standardize = "TRUE")}

# Continuous mediator
fit.m1 <- lm(M.num ~ R + C.num + C.bin, data = sdata)
fit.y1 <- lm(Y.num ~ R + M.num + S + C.num + C.bin, data = sdata)
res.1a1 <- smi(fit.r = fit.r1, fit.m = fit.m1,
          fit.y = fit.y1, sims = 40, conditional = FALSE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.1a1

\donttest{# Binary mediator
fit.m2 <- glm(M.bin ~ R + C.num + C.bin, data = sdata,
          family = binomial(link = "logit"))
fit.y2 <- lm(Y.num ~ R + M.bin + S + C.num + C.bin, data = sdata)
res.1a2 <- smi(fit.r = fit.r1, fit.m = fit.m2,
          fit.y = fit.y2, sims = 40, conditional = FALSE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.1a2

# Categorical mediator
require(MASS)
fit.m3 <- polr(M.cat ~ R + C.num + C.bin, data = sdata)
fit.y3 <- lm(Y.num ~ R + M.cat + S + C.num + C.bin, data = sdata)
res.1a3 <- smi(fit.r = fit.r1, fit.m = fit.m3,
          fit.y = fit.y3, sims = 40, conditional = FALSE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.1a3

require(nnet)
fit.m4 <- multinom(M.cat ~ R + C.num + C.bin, data = sdata)
res.1a4 <- smi(fit.r = fit.r1, fit.m = fit.m4,
          fit.y = fit.y3, sims = 40, conditional = FALSE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.1a4}

#-----------------------------------------------------------#
# Example 1-b: Binary Outcome
#-----------------------------------------------------------#
\donttest{# Continuous mediator
fit.y1 <- glm(Y.bin ~ R + M.num + S + C.num + C.bin,
          data = sdata, family = binomial(link = "logit"))
res.1b1 <- smi(fit.r = fit.r1, fit.m = fit.m1,
          fit.y = fit.y1, sims = 40, conditional = FALSE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.1b1

# Binary mediator
fit.y2 <- glm(Y.bin ~ R + M.bin + S + C.num + C.bin,
          data = sdata, family = binomial(link = "logit"))
res.1b2 <- smi(fit.r = fit.r1, fit.m = fit.m2,
          fit.y = fit.y2, sims = 40, conditional = FALSE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.1b2

# Categorical mediator
fit.y3 <- glm(Y.bin ~ R + M.cat + S + C.num + C.bin,
          data = sdata, family = binomial(link = "logit"))
res.1b3 <- smi(fit.r = fit.r1, fit.m = fit.m3,
          fit.y = fit.y3, sims = 40, conditional = FALSE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.1b3

res.1b4 <- smi(fit.r = fit.r1, fit.m = fit.m4,
          fit.y = fit.y3, sims = 40, conditional = FALSE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.1b4}

#--------------------------------------------------------------#
# Example 2-a: Continuous Outcome, Conditional on Covariates
#--------------------------------------------------------------#
\donttest{# For conditional = T, need to create data with centered covariates
# copy data
sdata.c <- sdata
# center quantitative covariate(s)
sdata.c$C.num <- scale(sdata.c$C.num, center = TRUE, scale = FALSE)
# center binary (or categorical) covariates(s)
# only neccessary if the desired baseline level is NOT the default baseline level.
sdata.c$C.bin <- relevel(sdata.c$C.bin, ref = "1")

# Continuous mediator
fit.y1 <- lm(Y.num ~ R + M.num + S + C.num + C.bin, data = sdata.c)
fit.m1 <- lm(M.num ~ R + C.num + C.bin, data = sdata.c)
res.2a1 <- smi(fit.m = fit.m1,
          fit.y = fit.y1, sims = 40, conditional = TRUE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.2a1

# Binary mediator
fit.y2 <- lm(Y.num ~ R + M.bin + S + C.num + C.bin, data = sdata.c)
fit.m2 <- glm(M.bin ~ R + C.num + C.bin, data = sdata.c,
          family = binomial(link = "logit"))
res.2a2 <- smi(fit.m = fit.m2,
          fit.y = fit.y2, sims = 40, conditional = TRUE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.2a2

# Categorical mediator
fit.y3 <- lm(Y.num ~ R + M.cat + S + C.num + C.bin, data = sdata.c)
fit.m3 <- polr(M.cat ~ R + C.num + C.bin, data = sdata.c)
res.2a3 <- smi(fit.m = fit.m3,
          fit.y = fit.y3, sims = 40, conditional = TRUE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.2a3

fit.m4 <- multinom(M.cat ~ R + C.num + C.bin, data = sdata.c)
res.2a4 <- smi(fit.m = fit.m4,
          fit.y = fit.y3, sims = 40, conditional = TRUE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.2a4}

#-----------------------------------------------------------#
# Example 2-b: Binary Outcome, Conditional on Covariates
#-----------------------------------------------------------#
\donttest{# Continuous mediator
fit.y1 <- glm(Y.bin ~ R + M.num + S + C.num + C.bin,
          data = sdata.c, family = binomial(link = "logit"))
res.2b1 <- smi(fit.m = fit.m1,
          fit.y = fit.y1, sims = 40, conditional = TRUE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.2b1

# Binary mediator
fit.y2 <- glm(Y.bin ~ R + M.bin + S + C.num + C.bin,
          data = sdata.c, family = binomial(link = "logit"))
res.2b2 <- smi(fit.m = fit.m2,
          fit.y = fit.y2, sims = 40, conditional = TRUE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.2b2

# Categorical mediator
fit.y3 <- glm(Y.bin ~ R + M.cat + S + C.num + C.bin,
          data = sdata.c, family = binomial(link = "logit"))
res.2b3 <- smi(fit.m = fit.m3,
          fit.y = fit.y3, sims = 40, conditional = TRUE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.2b3

res.2b4 <- smi(fit.m = fit.m4,
          fit.y = fit.y3, sims = 40, conditional = TRUE,
          covariates = c("C.num", "C.bin"), treat = "R")
res.2b4}
}
\references{
Park, S., Kang, S., and Lee, C. (2021+). "Choosing an Optimal Method for Causal
Decomposition Analysis: A Better Practice for Identifying Contributing Factors to
Health Disparities".
}
\seealso{
\code{\link{mmi}}
}
\author{
Suyeon Kang, University of California, Riverside, \email{skang062@ucr.edu};
Soojin Park, University of California, Riverside, \email{soojinp@ucr.edu}.
}
