% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/invCIR.r
\name{quickInverse}
\alias{quickInverse}
\title{Point and Interval Inverse Estimation ("Dose-Finding"), using CIR and IR}
\usage{
quickInverse(
  y,
  x = NULL,
  wt = NULL,
  target,
  estfun = cirPAVA,
  intfun = morrisCI,
  delta = TRUE,
  conf = 0.9,
  resolution = 100,
  extrapolate = FALSE,
  adaptiveShrink = FALSE,
  starget = target[1],
  parabola = FALSE,
  ...
)
}
\arguments{
\item{y}{can be either of the following: y values (response rates), a 2-column matrix with positive/negative response counts by dose, a \code{\link{DRtrace}} object or a \code{\link{doseResponse}} object.}

\item{x}{dose levels (if not included in y).}

\item{wt}{weights (if not included in y).}

\item{target}{A vector of target response rate(s), for which the percentile dose estimate is needed. See Note.}

\item{estfun}{the function to be used for point estimation. Default \code{\link{cirPAVA}}.}

\item{intfun}{the function to be used for interval estimation. Default \code{\link{morrisCI}} (see help on that function for additional options).}

\item{delta}{logical: should intervals be calculated using the delta ("local") method (default, \code{TRUE}) or back-drawn directly from the forward bounds? See Details.}

\item{conf}{numeric, the interval's confidence level as a fraction in (0,1). Default 0.9.}

\item{resolution}{numeric: how fine should the grid for the inverse-interval approximation be? Default 100, which seems to be quite enough. See 'Details'.}

\item{extrapolate}{logical: should extrapolation beyond the range of estimated y values be allowed? Default \code{FALSE}. Note this affects only the point estimate; interval boundaries are not extrapolated.}

\item{adaptiveShrink}{logical, should the y-values be pre-shrunk towards an experiment's target? Recommended when the data were obtained via an adaptive dose-finding design. See \code{\link{DRshrink}} and the Note below.}

\item{starget}{The shrinkage target. Defaults to \code{target[1]}.}

\item{parabola}{logical: should the confidence-interval's interpolation between points with observations follow a parabola (\code{TRUE}) creating broader intervals between observations, or a straight line (\code{FALSE}, default)?}

\item{...}{Other arguments passed on to \code{\link{doseFind}} and \code{\link{quickIsotone}}, and onwards from there.}
}
\value{
A data frame with 4 elements:
\itemize{
\item {\code{target}  }  { The user-provided target values of y, at which x is estimated}
\item {\code{point} } {  The point estimates of x}
\item {\code{lowerPPconf,upperPPconf}  }  { the interval-boundary estimates for a 'PP'=\code{100*conf} confidence interval}
}
}
\description{
Convenience wrapper for point and interval estimation of the "dose" that would generate a \code{target} "response" value, using CIR and IR.
}
\details{
The inverse point estimate is calculated in a straightforward manner from a forward estimate, using \code{\link{doseFind}}. For the inverse interval, the default option (\code{delta=TRUE}) calls \code{\link{deltaInverse}} for a "local" (Delta) inversion of the forward intervals. 
If \code{delta=FALSE}, a second call to \code{\link{quickIsotone}} genderates a high-resolution grid outlining the forward intervals. Then the algorithm "draws a horizontal line" at \code{y=target} to find the right and left bounds on this grid. Note that the right (upper) dose-finding confidence bound is found on the lower forward confidence bound, and vice versa.
}
\note{
If the data were obtained from an adaptive dose-finding design and you seek to estimate a dose other than the experiment's target, note that away from the target the estimates are likely biased (Flournoy and Oron, 2019). Use \code{adaptiveShrink=TRUE} to mitigate the bias. In addition, either provide the true target as \code{starget}, or a vector of values to \code{target}, with the first value being the true target.
}
\examples{
# Interesting run (#664) from a simulated up-and-down ensemble:
# (x will be auto-generated as dose levels 1:5)
dat=doseResponse(y=c(1/7,1/8,1/2,1/4,4/17),wt=c(7,24,20,12,17))
# The experiment's goal is to find the 30th percentile
inv1=quickInverse(dat,target=0.3)
# With old PAVA as the forward estimator:
inv0=quickInverse(dat,target=0.3,estfun=oldPAVA)


### Showing the data and the estimates
par(mar=c(3,3,1,1),mgp=c(2,.5,0),tcl=-0.25)
plot(dat,ylim=c(0.05,0.55),refsize=4,las=1) # uses plot.doseResponse()

# The true response function; true target is where it crosses the y=0.3 line
lines(seq(1,5,0.1),pweibull(seq(1,5,0.1),shape=1.1615,scale=8.4839),col=4)
abline(h=0.3,col=2,lty=3)
# Plotting the point estimates, as "tick" marks on the y=0.3 line
lines(rep(inv1$point,2),c(0.25,0.35)) # CIR
lines(rep(inv0$point,2),c(0.25,0.35),lty=2) # IR
# You could plot the CIs too, but they are very broad and much more similar than the 
# point estimates. The broadness likely reflects the shallow slope, which itself reflects the 
# monotonicity violations.
# Here's code to plot the CIR 90\% CI as a light-green rectangle:
# rect(inv1$lower90conf,0.25,inv1$upper90conf,0.35,col=rgb(0,1,0,alpha=0.3),border=NA)

legend('topleft',pch=c(NA,'X',NA,NA),lty=c(1,NA,2,1),col=c(4,1,1,1),
	legend=c('True Curve','Observations','IR Estimate','CIR Estimate'),bty='n')

}
\references{
Flournoy N and Oron AP, 2020. Bias Induced by Adaptive Dose-Finding Designs. Journal of Applied Statistics 47, 2431-2442.
}
\seealso{
\code{\link{quickIsotone}},\code{\link{doseFind}},\code{\link{deltaInverse}}
}
