#' Get Data Augmented With Clinical Significance Categories
#'
#' To obtain patient-wise results, use `get_augmented_data()`.
#'
#' This function returns the patient-wise results, containing the considered pre
#' and post intervention value, its raw change as well as the RCI and the
#' individual category a patient belongs to.
#'
#' @inheritParams get_data
#'
#' @importFrom dplyr left_join case_when relocate
#' @importFrom rlang .data
#'
#' @return A tibble with used data and clinical significance categories
#' @export
#'
#' @examples
#' results <- jacobson_1989 %>%
#' clinical_significance(
#'   id = subject,
#'   time = time,
#'   outcome = gds,
#'   pre = "pre",
#'   reliability = 0.80,
#'   m_functional = 30,
#'   sd_functional = 7,
#'   type = "c"
#' )
#'
#' get_augmented_data(results)

get_augmented_data <- function(x) {
  assert_class(x, "clinisig")

  clinisig_method <- get_method(x)

  if (clinisig_method == "HLM") {
    hlm_categories <- x[["categories"]]
    hlm_coefficients <- x[["rci"]][["coefficients"]] %>%
      select(id, eb_estimate, sd)

    categories <- hlm_categories %>%
      left_join(hlm_coefficients, by = "id") %>%
      relocate(eb_estimate:sd, .after = post)
  } else {
    categories <- x[["categories"]]
  }

  categories %>%
    mutate(
      category = case_when(
        recovered ~ "Recovered",
        improved ~ "Improved",
        unchanged ~ "Unchanged",
        deteriorated ~ "Deteriorated",
        harmed ~ "Harmed"
      ),
      category = factor(category, levels = c("Recovered", "Improved", "Unchanged", "Deteriorated", "Harmed"))
    )
}
