#' Graphical investigation for the number of datasets generated by multiple imputation
#'
#' @description
#' For an object generated by the function \code{clusterMI}, the \code{choosem} function browses the sequence of the contributory partitions and computes the consensus partition at each step. Then, the rand index between successive consensus partitions is plotted.
#' @details
#' The number of imputed datasets (\code{m}) should be sufficiently large to improve the partition accuracy.
#' The \code{choosem} function can be used to check if this number is suitable.
#' This function computes the consensus partition by considering only the first imputed datasets.
#' By this way, a sequence of \code{m} consensus partitions is obtained.
#' Then, the rand index between successive partitions is computed and reported in a graph.
#' The rand index measures the proximity between two partitions.
#' If the rand index between the last consensus partitions of the sequence reaches its maximum values (1),
#' then it means last imputed dataset does not modify the consensus partition.
#' Consequently, the number of imputed datasets can be considered as sufficiently large.
#' @return A list of two objects
#'   \item{part}{\code{m}-columns matrix that contains in column p the consensus partition using only the p first imputed datasets}
#'   \item{rand}{a \code{m}-1 vector given the rand index between the \code{m} successive consensus partitions}
#' @param output an output from the clusterMI function
#' @param graph a boolean indicating if a graphic is plotted
#' @param nnodes number of CPU cores for parallel computing. By default \code{nnodes = 1}.
#' @export
#' @seealso \code{\link{clusterMI}}, \code{\link{imputedata}}
#' @references
#'  Audigier, V. and Niang, N., Clustering with missing data: which equivalent for Rubin's rules? Advances in Data Analysis and Classification <doi:10.1007/s11634-022-00519-1>, 2022.
#' @examples
#' data(wine, package = "clusterMI")
#' 
#' set.seed(123456)
#' ref <- wine$cult
#' nb.clust <- 3
#' wine.na <- wine
#' wine.na$cult <- NULL
#' wine.na <- prodna(wine.na)
#' 
#' #imputation
#' m <- 5 # number of imputed data sets. Should be larger in practice
#' res.imp <- imputedata(data.na = wine.na, nb.clust = nb.clust, m = m)
#' 
#' #pooling
#' nnodes <- 2 # number of CPU cores for parallel computing
#' res.pool <- clusterMI(res.imp, instability = FALSE, nnodes = nnodes)
#' 
#' res.choosem <- choosem(res.pool)

choosem <- function(output,graph=TRUE,nnodes = 1){
  if(is.null(nnodes)){
    nnodes.intern <- output$call$nnodes
  }else{
    nnodes.intern <- nnodes
  }
  if(nnodes.intern>1){
    cl <- parallel::makeCluster(nnodes.intern, type = "PSOCK")
    parallel::clusterExport(cl, list("output","fastnmf"), envir = environment())
    part <- parallel::parSapply(cl, seq(1,output$call$output$call$m,1),
                                FUN = function(m,output){
                                  
                                  res.out<-fastnmf(output$call$res.analyse[seq.int(m)],
                                                   nb.clust =output$call$output$call$nb.clust,
                                                   threshold = output$call$parameter.nmf$threshold,
                                                   method.init = output$call$parameter.nmf$method.init,
                                                   parameter.kmeans = output$call$parameter.nmf$parameter.kmeans,
                                                   parameter.minibatchkmeans = output$call$parameter.nmf$parameter.minibatchkmeans,
                                                   printflag = FALSE)$best$clust
                                  
                                  return(res.out)
                                },output=output)
    parallel::stopCluster(cl)
    
  }else{
    
    part<-sapply(seq(1,output$call$output$call$m,1),FUN=function(m,output){
      res.out<-fastnmf(output$call$res.analyse[seq.int(m)],
                       nb.clust =output$call$output$call$nb.clust,
                       method.init = output$call$parameter.nmf$method.init,
                       threshold = output$call$parameter.nmf$threshold,
                       parameter.kmeans = output$call$parameter.nmf$parameter.kmeans,
                       parameter.minibatchkmeans = output$call$parameter.nmf$parameter.minibatchkmeans,
                       printflag = FALSE
                       )$best$clust
      
      return(res.out)
    },output=output)
  }
  colnames(part)<-seq(1,output$call$output$call$m,1)
  res.out<-rep(NA,ncol(part)-1)
  for(ii in 1:(ncol(part)-1)){
    res.out[ii]<-randindex(part[,ii],part[,ii+1])
  }
  if(graph){
    oldpar <- par(no.readonly = TRUE)
    on.exit(par(oldpar))
    par(mfrow=c(1,1),mar=c(5, 4, 3, 1));plot(2:(ncol(part)),res.out,ylab="rand index",xlab="m",type="b")}
  res.out<-list(part=part,rand=res.out)
  return(res.out)
}