% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{BaselearnerCustom}
\alias{BaselearnerCustom}
\title{Create custom base-learner factory by using R functions.}
\format{\code{\link{S4}} object.}
\description{
\code{BaselearnerCustom} creates a custom base-learner factory by
  setting custom \code{R} functions. This factory object can be registered
  within a base-learner list and then used for training.
}
\section{Usage}{

\preformatted{
BaselearnerCustom$new(data_source, data_target, instantiateData, train,
  predict, extractParameter)
}
}

\section{Arguments}{

\describe{
\item{\code{data_source} [\code{Data} Object]}{
  Data object which contains the source data.
}
\item{\code{data_target} [\code{Data} Object]}{
  Data object which gets the transformed source data.
}
\item{\code{instantiateData} [\code{function}]}{
  \code{R} function to transform the source data. For details see the
  \code{Details}.
}
\item{\code{train} [\code{function}]}{
  \code{R} function to train the base-learner on the target data. For
  details see the \code{Details}.
}
\item{\code{predict} [\code{function}]}{
  \code{R} function to predict on the object returned by \code{train}.
  For details see the \code{Details}.
}
\item{\code{extractParameter} [\code{function}]}{
  \code{R} function to extract the parameter of the object returned by
  \code{train}. For details see the \code{Details}.
}
}
}

\section{Details}{

  The function must have the following structure:

  \code{instantiateData(X) { ... return (X.trafo) }} With a matrix argument
  \code{X} and a matrix as return object.

  \code{train(y, X) { ... return (SEXP) }} With a vector argument \code{y}
  and a matrix argument \code{X}. The target data is used in \code{X} while
  \code{y} contains the response. The function can return any \code{R}
  object which is stored within a \code{SEXP}.

  \code{predict(model, newdata) { ... return (prediction) }} The returned
  object of the \code{train} function is passed to the \code{model}
  argument while \code{newdata} contains a new matrix used for predicting.

  \code{extractParameter() { ... return (parameters) }} Again, \code{model}
  contains the object returned by \code{train}. The returned object must be
  a matrix containing the estimated parameter. If no parameter should be
  estimated one can return \code{NA}.

  For an example see the \code{Examples}.

  This class is a wrapper around the pure \code{C++} implementation. To see
  the functionality of the \code{C++} class visit
  \url{https://schalkdaniel.github.io/compboost/cpp_man/html/classblearnerfactory_1_1_custom_blearner_factory.html}.
}

\section{Fields}{

  This class doesn't contain public fields.
}

\section{Methods}{

\describe{
\item{\code{getData()}}{Get the data matrix of the target data which is used
  for modeling.}
\item{\code{transformData(X)}}{Transform a data matrix as defined within the
  factory. The argument has to be a matrix with one column.}
\item{\code{summarizeFactory()}}{Summarize the base-learner factory object.}
}
}

\examples{
# Sample data:
data.mat = cbind(1, 1:10)
y = 2 + 3 * 1:10

# Create new data object:
data.source = InMemoryData$new(data.mat, "my.data.name")
data.target = InMemoryData$new()

instantiateDataFun = function (X) {
  return(X)
}
# Ordinary least squares estimator:
trainFun = function (y, X) {
  return(solve(t(X) \%*\% X) \%*\% t(X) \%*\% y)
}
predictFun = function (model, newdata) {
  return(as.matrix(newdata \%*\% model))
}
extractParameter = function (model) {
  return(as.matrix(model))
}

# Create new custom linear base-learner factory:
custom.lin.factory = BaselearnerCustom$new(data.source, data.target,
  instantiateDataFun, trainFun, predictFun, extractParameter)

# Get the transformed data:
custom.lin.factory$getData()

# Summarize factory:
custom.lin.factory$summarizeFactory()

# Transform data manually:
custom.lin.factory$transformData(data.mat)

}
