#' List all processors available in a project
#'
#' @param proj_id a GCS project id.
#' @param loc a two-letter region code; "eu" or "us".
#' @param token an authentication token generated by \code{dai_auth()} or
#' another auth function.
#'
#' @return a dataframe.
#' @export
#'
#' @details For more information about processors, see the Google Document
#' AI documentation at \url{https://cloud.google.com/document-ai/docs/}.
#'
#' @examples
#' \dontrun{
#' df <- get_processors()
#' }
get_processors <- function(proj_id = get_project_id(),
						   loc = "eu",
						   token = dai_token()
						   ) {
	# checks
	if (!(is.character(proj_id) && length(proj_id) == 1)) {
    stop("Invalid proj_id parameter.")
    }

	loc <- tolower(loc)

    if (!(length(loc) == 1) || !(loc %in% c("eu", "us"))) {
    stop("Invalid loc parameter.")
    }

	base_url <- glue::glue("https://{loc}-documentai.googleapis.com")
	path <- glue::glue("/v1/projects/{proj_id}/locations/{loc}/processors")
	url <- glue::glue("{base_url}{path}")
	response <- httr::GET(url, httr::config(token = token))
	parsed <- httr::content(response)
	df <- as.data.frame(data.table::rbindlist(parsed$processors))
	df$id <- basename(df$name)
	df <- df[, c(8, 1:7)]
	return(df)
}

#' Get information about processor
#'
#' @param proj_id a GCS project id.
#' @param proc_id a Document AI processor id.
#' @param loc a two-letter region code; "eu" or "us".
#' @param token an authentication token generated by \code{dai_auth()} or
#' another auth function.
#'
#' @return a list.
#' @export
#'
#' @examples
#' \dontrun{
#' info <- get_processor_info()
#'
#' info <- get_processor_info(proc_id = get_processors()$id[1])
#' }
get_processor_info <- function(proj_id = get_project_id(),
							   proc_id = Sys.getenv("DAI_PROCESSOR_ID"),
							   loc = "eu",
							   token = dai_token()
							   ) {
    # checks
	if (!(is.character(proj_id) && length(proj_id) == 1)) {
		stop("Invalid proj_id parameter.")
		}

	if (!(is.character(proc_id) && length(proc_id) == 1) || proc_id == "") {
		stop("Invalid proc_id parameter.")
		}

	loc <- tolower(loc)

    if (!(length(loc) == 1) || !(loc %in% c("eu", "us"))) {
    stop("Invalid loc parameter.")
    }

	base_url <- glue::glue("https://{loc}-documentai.googleapis.com")
	path <- glue::glue("/v1/projects/{proj_id}/locations/{loc}/processors/{proc_id}")
	url <- glue::glue("{base_url}{path}")
	response <- httr::GET(url, httr::config(token = token))
	parsed <- httr::content(response)
	return(parsed)
}

#' List available processor versions
#'
#' @param proj_id a GCS project id.
#' @param proc_id a Document AI processor id.
#' @param loc a two-letter region code; "eu" or "us".
#' @param token an authentication token generated by \code{dai_auth()} or
#' another auth function.
#'
#' @return a dataframe.
#' @export
#'
#' @examples
#' \dontrun{
#' df <- get_processor_versions()
#'
#' df <- get_processor_versions(proc_id = get_processors()$id[1])
#' }
get_processor_versions <- function(proj_id = get_project_id(),
								   proc_id = Sys.getenv("DAI_PROCESSOR_ID"),
								   loc = "eu",
								   token = dai_token()
								   ) {
    # checks
	if (!(is.character(proj_id) && length(proj_id) == 1)) {
		stop("Invalid proj_id parameter.")
		}

	if (!(is.character(proc_id) && length(proc_id) == 1) || proc_id == "") {
		stop("Invalid proc_id parameter.")
		}

	loc <- tolower(loc)

    if (!(length(loc) == 1) || !(loc %in% c("eu", "us"))) {
    stop("Invalid loc parameter.")
    }

	base_url <- glue::glue("https://{loc}-documentai.googleapis.com")
	path <- glue::glue("/v1/projects/{proj_id}/locations/{loc}/processors/{proc_id}/processorVersions")
	url <- glue::glue("{base_url}{path}")
	response <- httr::GET(url, httr::config(token = token))
	parsed <- httr::content(response)
	df <- as.data.frame(data.table::rbindlist(parsed$processorVersions))
	df$shortName <- basename(df$name)
	df <- df[, c(6, 1:5)]
	return(df)
}