% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dmult.R
\name{dmult}
\alias{dmult}
\title{Multiply by diagonal matrix}
\usage{
dmult(M, v, side = c("left", "right"))
}
\arguments{
\item{M}{matrix}

\item{v}{vector with entries forming a diagonal matrix matching the dimensions of \code{M} depending on the value of \code{side}}

\item{side}{is the matrix \code{M} \code{"left"} or \code{"right"} multiplied by the diagonal matrix}
}
\value{
matrix product
}
\description{
Multiply by diagonal matrix using efficient algorithm
}
\details{
Naively multiplying by a diagonal matrix with \code{p} entries takes \eqn{\mathcal{O}(p^2)}, even though must values in the diagonal matrix are zero.  R has built in syntax to scale \emph{columns}, so \code{diag(v) \%*\% M} can be evaluated with \code{v * M}.  This is often difficult to remember, hard to look up, and scaling \emph{rows} requires \code{t(t(M) * v)}.  This is hard to read and write, and requires two transpose operations.

Here, \code{dmult()} uses \code{Rcpp} to evaluate the right multiplication efficiently, and uses \code{v * M} for left multiplication.  This gives good performance and readability.

In principle, the \code{Rcpp} code can be modified to use BLAS for better performance by treating a \code{NumericMatrix} as a \code{C} array.  But this is not currently a bottleneck
}
\examples{
# right multiply
# mat \%*\% diag(v)
n <- 30
p <- 20
mat <- matrix(n * p, n, p)
v <- rnorm(p)

A <- dmult(mat, v, side = "right")
B <- mat \%*\% diag(v)
range(A - B)

# left multiply
# diag(v) \%*\% mat
n <- 30
p <- 20
mat <- matrix(n * p, p, n)
v <- rnorm(p)

A <- dmult(mat, v, side = "left")
B <- diag(v) \%*\% mat
range(A - B)
}
