% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dnnModel.R
\name{deconvDigitalDLSorter}
\alias{deconvDigitalDLSorter}
\title{Deconvolute bulk RNA-Seq samples using a pre-trained DigitalDLSorter model}
\usage{
deconvDigitalDLSorter(
  data,
  model = NULL,
  normalize = TRUE,
  scaling = "standardize",
  simplify.set = NULL,
  simplify.majority = NULL,
  use.generator = FALSE,
  batch.size = 64,
  verbose = TRUE
)
}
\arguments{
\item{data}{Matrix or data frame with bulk RNA-Seq samples with genes as rows
in SYMBOL notation and samples as columns.}

\item{model}{Pre-trained DNN model to use to deconvolute \code{data}. Up to
now, the available models are intended to deconvolute samples from breast
cancer (\code{breast.chung.generic} and \code{breast.chung.specific}) and
colorectal cancer (\code{colorectal.li}). These pre-trained models are
stored in the \pkg{digitalDLSorteRdata} package, so it must be installed
together with \pkg{digitalDLSorteR} to use this function.}

\item{normalize}{Normalize data before deconvolution (\code{TRUE} by default).}

\item{scaling}{How to scale data before training. It may be:
\code{"standardize"} (values are centered around the mean with a unit
standard deviation) or \code{"rescale"} (values are shifted and rescaled so
that they end up ranging between 0 and 1). If \code{normalize = FALSE}, data
is not scaled.}

\item{simplify.set}{List specifying which cell types should be compressed into
a new label whose name will be the list name item. See examples and
vignettes for details.}

\item{simplify.majority}{List specifying which cell types should be compressed
into the cell type with the highest proportion in each sample. Unlike
\code{simplify.set}, this argument allows to maintain the complexity of the
results while compressing the information, as no new labels are created.}

\item{use.generator}{Boolean indicating whether to use generators for
prediction (\code{FALSE} by default).}

\item{batch.size}{Number of samples per batch. Only when \code{use.generator
= TRUE}.}

\item{verbose}{Show informative messages during execution.}
}
\value{
A data frame with samples (\eqn{i}) as rows and cell types (\eqn{j})
as columns. Each entry represents the predicted proportion of cell type
\eqn{j} in sample \eqn{i}.
}
\description{
Deconvolute bulk gene expression samples (bulk RNA-Seq) to enumerate and
quantify the proportion of cell types present in a bulk sample using Deep
Neural Network models. This function is intended for users who want to use
pre-trained models integrated in the package. So far, the available models
allow to deconvolute the immune infiltration of breast cancer (using data from
Chung et al., 2017) and the immune infiltration of colorectal cancer (using
data from Li et al., 2017) samples. For the former, two models are available
at two different levels of specificity: specific cell types
(\code{breast.chung.specific}) and generic cell types
(\code{breast.chung.generic}). See \code{breast.chung.generic},
\code{breast.chung.specific}, and \code{colorectal.li} documentation from the
\pkg{digitalDLSorteRdata} package for more details.
}
\details{
This function is intended for users who want to use \pkg{digitalDLSorteR} to
deconvolute their bulk RNA-Seq samples using pre-trained models. For users who
want to build their own models from other scRNA-Seq datasets, see the
\code{\link{createDDLSobject}} and \code{\link{deconvDDLSObj}}
functions.
}
\examples{
\dontrun{
set.seed(123)
sce <- SingleCellExperiment::SingleCellExperiment(
  assays = list(
    counts = matrix(
      rpois(30, lambda = 5), nrow = 15, ncol = 20,
      dimnames = list(paste0("Gene", seq(15)), paste0("RHC", seq(20)))
    )
  ),
  colData = data.frame(
    Cell_ID = paste0("RHC", seq(20)),
    Cell_Type = sample(x = paste0("CellType", seq(6)), size = 20,
                       replace = TRUE)
  ),
  rowData = data.frame(
    Gene_ID = paste0("Gene", seq(15))
  )
)
DDLS <- createDDLSobject(
  sc.data = sce,
  sc.cell.ID.column = "Cell_ID",
  sc.gene.ID.column = "Gene_ID",
  sc.filt.genes.cluster = FALSE, 
  sc.log.FC = FALSE
)
probMatrixValid <- data.frame(
  Cell_Type = paste0("CellType", seq(6)),
  from = c(1, 1, 1, 15, 15, 30),
  to = c(15, 15, 30, 50, 50, 70)
)
DDLS <- generateBulkCellMatrix(
  object = DDLS,
  cell.ID.column = "Cell_ID",
  cell.type.column = "Cell_Type",
  prob.design = probMatrixValid,
  num.bulk.samples = 50,
  verbose = TRUE
)
# training of DDLS model
tensorflow::tf$compat$v1$disable_eager_execution()
DDLS <- trainDDLSModel(
  object = DDLS,
  on.the.fly = TRUE,
  batch.size = 15,
  num.epochs = 5
)
# simulating bulk RNA-Seq data
countsBulk <- matrix(
  stats::rpois(100, lambda = sample(seq(4, 10), size = 100, replace = TRUE)),
  nrow = 40, ncol = 15,
  dimnames = list(paste0("Gene", seq(40)), paste0("Bulk", seq(15)))
)
# this is only an example. See vignettes to see how to use pre-trained models
# from the digitalDLSorteRmodels data package
results1 <- deconvDigitalDLSorter(
  data = countsBulk,
  model = trained.model(DDLS),
  normalize = TRUE
)
# simplify arguments
simplify <- list(CellGroup1 = c("CellType1", "CellType2", "CellType4"),
                 CellGroup2 = c("CellType3", "CellType5"))
# in this case the names of the list will be the new labels
results2 <- deconvDigitalDLSorter(
  countsBulk,
  model = trained.model(DDLS),
  normalize = TRUE,
  simplify.set = simplify
)
# in this case the cell type with the highest proportion will be the new label
results3 <- deconvDigitalDLSorter(
  countsBulk,
  model = trained.model(DDLS),
  normalize = TRUE,
  simplify.majority = simplify
)
}

}
\references{
Chung, W., Eum, H. H., Lee, H. O., Lee, K. M., Lee, H. B., Kim, K.
T., et al. (2017). Single-cell RNA-seq enables comprehensive tumour and
immune cell profiling in primary breast cancer. Nat. Commun. 8 (1), 15081.
doi: \doi{10.1038/ncomms15081}.
}
\seealso{
\code{\link{deconvDDLSObj}}
}
