% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_model.R
\name{make_model_object}
\alias{make_model_object}
\title{Create the TMB model object for the disaggregation model}
\usage{
make_model_object(
  data,
  priors = NULL,
  family = "gaussian",
  link = "identity",
  field = TRUE,
  iid = TRUE,
  silent = TRUE
)
}
\arguments{
\item{data}{disag_data object returned by \code{\link{prepare_data}} function that contains all the necessary objects for the model fitting}

\item{priors}{list of prior values}

\item{family}{likelihood function: \emph{gaussian}, \emph{binomial} or \emph{poisson}}

\item{link}{link function: \emph{logit}, \emph{log} or \emph{identity}}

\item{field}{logical. Flag the spatial field on or off}

\item{iid}{logical. Flag the iid effect on or off}

\item{silent}{logical. Suppress verbose output.}
}
\value{
The TMB model object returned by \code{\link[TMB]{MakeADFun}}.
}
\description{
\emph{make_model_object} function takes a \emph{disag_data} object created by \code{\link{prepare_data}}
and creates a TMB model object to be used in fitting.
}
\details{
\strong{The model definition}

The disaggregation model make predictions at the pixel level:
\deqn{link(pred_i) = \beta_0 + \beta X + GP(s_i) + u_i}{ link(predi) = \beta 0 + \beta X + GP + u}

And then aggregates these predictions to the polygon level using the weighted sum (via the aggregation raster, \eqn{agg_i}{aggi}):
\deqn{cases_j = \sum_{i \epsilon j} pred_i \times agg_i}{ casesj = \sum (predi x aggi)}
\deqn{rate_j = \frac{\sum_{i \epsilon j} pred_i \times agg_i}{\sum_{i \epsilon j} agg_i}}{ratej = \sum(predi x aggi) / \sum (aggi)}

The different likelihood correspond to slightly different models (\eqn{y_j}{yi} is the response count data):
\itemize{
  \item Gaussian:
   If \eqn{\sigma} is the dispersion of the pixel data, \eqn{\sigma_j}{\sigmaj} is the dispersion of the polygon data, where
   \eqn{\sigma_j = \sigma \sqrt{\sum agg_i^2} / \sum agg_i }{\sigmaj = \sigma x { \sqrt \sum (aggi ^ 2) } / \sum aggi}
   \deqn{dnorm(y_j/\sum agg_i, rate_j, \sigma_j)}{dnorm(yj / \sum aggi, ratej, \sigmaj)} - predicts incidence rate.
  \item Binomial:
   For a survey in polygon j, \eqn{y_j}{yj} is the number positive and \eqn{N_j}{Nj} is the number tested.
   \deqn{dbinom(y_j, N_j, rate_j)}{dbinom(yj, Nj, ratej)} - predicts prevalence rate.
  \item Poisson:
   \deqn{dpois(y_j, cases_j)}{dpois(yj, casesj)} - predicts incidence count.
}

Specify priors for the regression parameters, field and iid effect as a single named list. Hyperpriors for the field
are given as penalised complexity priors you specify \eqn{\rho_{min}} and \eqn{\rho_{prob}} for the range of the field
where \eqn{P(\rho < \rho_{min}) = \rho_{prob}}, and \eqn{\sigma_{min}} and \eqn{\sigma_{prob}} for the variation of the field
where \eqn{P(\sigma > \sigma_{min}) = \sigma_{prob}}. Also, specify pc priors for the iid effect.

The precise names and default values for these priors are:
\itemize{
\item priormean_intercept: 0
\item priorsd_intercept: 10.0
\item priormean_slope: 0.0
\item priorsd_slope: 0.5
\item prior_rho_min: A third the length of the diagonal of the bounding box.
\item prior_rho_prob: 0.1
\item prior_sigma_max: sd(response/mean(response))
\item prior_sigma_prob: 0.1
\item prior_iideffect_sd_max: 0.1
\item prior_iideffect_sd_prob: 0.01
}

The \emph{family} and \emph{link} arguments are used to specify the likelihood and link function respectively.
The likelihood function can be one of \emph{gaussian}, \emph{poisson} or \emph{binomial}.
The link function can be one of \emph{logit}, \emph{log} or \emph{identity}.
These are specified as strings.

The field and iid effect can be turned on or off via the \emph{field} and \emph{iid} logical flags. Both are default TRUE.

The \emph{iterations} argument specifies the maximum number of iterations the model can run for to find an optimal point.

The \emph{silent} argument can be used to publish/supress verbose output. Default TRUE.
}
\examples{
\dontrun{
polygons <- list()
n_polygon_per_side <- 10
n_polygons <- n_polygon_per_side * n_polygon_per_side
n_pixels_per_side <- n_polygon_per_side * 2

for(i in 1:n_polygons) {
  row <- ceiling(i/n_polygon_per_side)
  col <- ifelse(i \%\% n_polygon_per_side != 0, i \%\% n_polygon_per_side, n_polygon_per_side)
  xmin = 2*(col - 1); xmax = 2*col; ymin = 2*(row - 1); ymax = 2*row
  polygons[[i]] <- list(cbind(c(xmin, xmax, xmax, xmin, xmin),
                              c(ymax, ymax, ymin, ymin, ymax)))
}

polys <- lapply(polygons,sf::st_polygon)
N <- floor(runif(n_polygons, min = 1, max = 100))
response_df <- data.frame(area_id = 1:n_polygons, response = runif(n_polygons, min = 0, max = 1000))

spdf <- sf::st_sf(response_df, geometry = polys)

# Create raster stack
r <- terra::rast(ncol=n_pixels_per_side, nrow=n_pixels_per_side)
terra::ext(r) <- terra::ext(spdf)
r[] <- sapply(1:terra::ncell(r), function(x){
rnorm(1, ifelse(x \%\% n_pixels_per_side != 0, x \%\% n_pixels_per_side, n_pixels_per_side), 3))}
r2 <- terra::rast(ncol=n_pixels_per_side, nrow=n_pixels_per_side)
terra::ext(r2) <- terra::ext(spdf)
r2[] <- sapply(1:terra::ncell(r), function(x) rnorm(1, ceiling(x/n_pixels_per_side), 3))
cov_stack <- c(r, r2)
names(cov_stack) <- c('layer1', 'layer2')

test_data <- prepare_data(polygon_shapefile = spdf,
                          covariate_rasters = cov_stack)

 result <- make_model_object(test_data)
 }

}
