% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/discrim_regularized.R
\name{discrim_regularized}
\alias{discrim_regularized}
\title{Regularized discriminant analysis}
\usage{
discrim_regularized(
  mode = "classification",
  engine = "klaR",
  frac_common_cov = NULL,
  frac_identity = NULL
)
}
\arguments{
\item{mode}{A single character string for the type of model. The only
possible value for this model is "classification".}

\item{engine}{A single character string specifying what computational engine to use
for fitting.}

\item{frac_common_cov, frac_identity}{Numeric values between zero and one.}
}
\description{
\code{discrim_regularized()} defines a model that estimates a multivariate
distribution for the predictors separately for the data in each class. The
structure of the model can be LDA, QDA, or some amalgam of the two. Bayes'
theorem is used to compute the probability of each class, given the
predictor values.

There are different ways to fit this model. See the engine-specific pages
for more details:

\Sexpr[stage=render,results=rd]{parsnip:::make_engine_list("discrim_regularized", "discrim")}

More information on how \pkg{parsnip} is used for modeling is at
\url{https://www.tidymodels.org/}.
}
\details{
There are many ways of regularizing models. For example, one form of
regularization is to penalize model parameters. Similarly, the classic
James–Stein regularization approach shrinks the model structure to a less
complex form.

The model fits a very specific type of regularized model by Friedman (1989)
that uses two types of regularization. One modulates how class-specific the
covariance matrix should be. This allows the model to balance between LDA
and QDA. The second regularization component shrinks the covariance matrix
towards the identity matrix.

For the penalization approach, \code{\link[=discrim_linear]{discrim_linear()}} with a \code{mda} engine can be
used. Other regularization methods can be used with \code{\link[=discrim_linear]{discrim_linear()}} and
\code{\link[=discrim_quad]{discrim_quad()}} can used via the \code{sparsediscrim} engine for those functions.

This function only defines what \emph{type} of model is being fit. Once an engine
is specified, the \emph{method} to fit the model is also defined.

The model is not trained or fit until the \code{\link[=fit.model_spec]{fit.model_spec()}} function is used
with the data.
}
\examples{
parabolic_grid <-
  expand.grid(X1 = seq(-5, 5, length = 100),
              X2 = seq(-5, 5, length = 100))

rda_mod <-
  discrim_regularized(frac_common_cov = .5, frac_identity = .5) \%>\%
  set_engine("klaR") \%>\%
  fit(class ~ ., data = parabolic)

parabolic_grid$rda <-
  predict(rda_mod, parabolic_grid, type = "prob")$.pred_Class1

library(ggplot2)
ggplot(parabolic, aes(x = X1, y = X2)) +
  geom_point(aes(col = class), alpha = .5) +
  geom_contour(data = parabolic_grid, aes(z = rda), col = "black", breaks = .5) +
  theme_bw() +
  theme(legend.position = "top") +
  coord_equal()
}
\references{
\url{https://www.tidymodels.org}, \href{https://www.tmwr.org}{\emph{Tidy Models with R}}

Friedman, J (1989). Regularized Discriminant Analysis. \emph{Journal of the
American Statistical Association}, 84, 165-175.
}
\seealso{
\Sexpr[stage=render,results=rd]{parsnip:::make_seealso_list("discrim_regularized", "discrim")}
}
