#' @include model_logger.R
#' @include system_base.R
#' @importFrom bbmle mle2 parnames summary
#' @import dplyr
#' @importFrom graphics legend lines
#' @importFrom magrittr %>%
#' @importFrom rlang :=
#' @importFrom stats formula lm model.matrix na.omit median qnorm sd var
#' @import tibble

setClassUnion("characterOrNULL", c("character", "NULL"))
setOldClass(c("spec_tbl_df", "tbl_df", "tbl", "data.frame"))
utils::globalVariables("where")

#' @title Market model classes
#'
#' @slot logger Logger object.
#' @slot key_columns Vector of column names that uniquely identify data records. For
#' panel data this vector should contain an entity and a time point identifier.
#' @slot time_column Column name for the time point data.
#' @slot explanatory_columns Vector of explanatory column names for all model's
#' equations.
#' @slot data_columns Vector of model's data column names. This is the union of the
#' quantity, price and explanatory columns.
#' @slot columns Vector of primary key and data column names for all model's equations.
#' @slot model_tibble Model data \code{tibble}.
#' @slot model_type_string Model type string description.
#' @slot system Model's system of equations.
#' @name market_models
#' @seealso initialize_market_model
NULL

#' @describeIn market_models Base class for market models
setClass(
  "market_model",
  representation(
    ## Logging
    logger = "model_logger",

    ## Column fields
    key_columns = "vector",
    time_column = "characterOrNULL",
    explanatory_columns = "vector",
    data_columns = "vector",
    columns = "vector",

    ## Model data
    model_tibble = "tbl_df",
    model_type_string = "character",
    market_type_string = "character",
    system = "system_base"
  )
)

#' @title Model initialization
#'
#' @details
#' The following two subsections describe the common initialization steps of all market
#' model classes.
#'
#' \subsection{Variable construction}{
#' The constructor prepares the model's variables using the passed
#' specifications. The specification variables are expected to be of type
#' \code{language}. The right hand side specifications of the system are
#' expected to follow the syntax of \code{\link[stats]{formula}}. The
#' construction of the model's data uses the variables extracted by these
#' specification. The demand variables are extracted by a
#' formula that uses the \code{quantity} on the left hand side and the
#' \code{demand} on the right hand side of the formula. The supply
#' variables are constructed by the \code{quantity} and the
#' \code{supply} inputs. In the case of the
#' \code{\linkS4class{diseq_stochastic_adjustment}} model, the price dynamics'
#' variables are extracted using the \code{price dynamics} input.
#' The \code{price dynamics} for the \code{\linkS4class{diseq_stochastic_adjustment}}
#' should contain only terms other than that of excess demand. The excess demand term of
#' the price equation is automatically generated by the constructor.
#' }
#'
#' \subsection{Data preparation}{
#'   1. If the passed data set contains rows with NA values, they are dropped. If the
#' verbosity level allows warnings, a warning is emitted reporting how many rows were
#' dropped.
#'
#'   2. After dropping the rows, factor levels may be invalidated. If needed, the
#' constructor readjusts the factor variables by removing the unobserved levels. Factor
#' indicators and interaction terms are automatically created.
#'
#'   3. The primary key column is constructed by pasting the values of the columns of
#' the subject and time variables.
#'
#'   4. In the cases of the \code{\linkS4class{diseq_directional}},
#' \code{\linkS4class{diseq_deterministic_adjustment}}, and
#' the \code{\linkS4class{diseq_stochastic_adjustment}} models, a column with lagged
#' prices is constructed. Since lagged prices are unavailable for the observations of
#' the first time point, these observations are dropped. If the verbosity level allows
#' the emission of information messages, the constructor prints the number of dropped
#' observations.
#'
#'   5. In the cases of the \code{\linkS4class{diseq_directional}}
#' and the \code{\linkS4class{diseq_stochastic_adjustment}} models, a column with price
#' differences is created.
#' }
#'
#' @param .Object The object to be Constructed.
#' @param verbose Verbosity level.
#' @param subject The subject identifier of the data set.
#' @param time The time identifier of the data set.
#' @param quantity The quantity variable of the system.
#' @param price The price variable of the system.
#' @param demand A formula representation of the right hand side of the
#'   demand equation.
#' @param supply A formula representation of the right hand side of the
#'   supply equation.
#' @param price_dynamics A formula representation of the price equation.
#' @param correlated_shocks Should the model be estimated using correlated shocks?
#' @param data The data set.
#' @return The initialized model.
#' @name initialize_market_model
NULL

make_specification <- function(data, quantity, price, demand, supply,
                               subject, time, price_dynamics) {
  no_parts <- 7
  if (missing(price_dynamics)) {
    no_parts <- 6
  }
  found_parts <- 0

  fm <- NULL
  dnames <- names(data)
  n <- sys.nframe()
  while (!identical(sys.frame(which = n), globalenv())) {
    squantity <- toString(substitute(quantity, env = sys.frame(which = n)))
    if (squantity %in% dnames) {
      quantity <- squantity
      found_parts <- found_parts + 1
    }

    sprice <- toString(substitute(price, env = sys.frame(which = n)))
    if (sprice %in% dnames) {
      price <- sprice
      found_parts <- found_parts + 1
    }

    ssubject <- toString(substitute(subject, env = sys.frame(which = n)))
    if (ssubject %in% dnames) {
      subject <- ssubject
      found_parts <- found_parts + 1
    }

    stime <- toString(substitute(time, env = sys.frame(which = n)))
    if (stime %in% dnames) {
      time <- stime
      found_parts <- found_parts + 1
    }

    sdemand <- all.vars(substitute(demand, env = sys.frame(which = n)))
    if (all(sdemand %in% dnames)) {
      demand <- paste0(sdemand, collapse = " + ")
      found_parts <- found_parts + 1
    }

    ssupply <- all.vars(substitute(supply, env = sys.frame(which = n)))
    if (all(ssupply %in% dnames)) {
      supply <- paste0(ssupply, collapse = " + ")
      found_parts <- found_parts + 1
    }

    if (!missing(price_dynamics)) {
      sprice_dynamics <- all.vars(substitute(
        price_dynamics,
        env = sys.frame(which = n)
      ))
      if (all(sprice_dynamics %in% dnames)) {
        price_dynamics <- paste0(sprice_dynamics, collapse = " + ")
        found_parts <- found_parts + 1
      }
    }

    if (found_parts == no_parts) {
      rhs <- paste(demand, supply, sep = " | ")
      if (!missing(price_dynamics)) {
        rhs <- paste(rhs, price_dynamics, sep = " | ")
      }
      fm <- formula(paste0(
        paste(quantity, price, subject, time, sep = " | "), " ~ ",
        rhs
      ))
      break
    }
    n <- n - 1
  }
  specification <- Formula::Formula(eval(fm))

  specification
}

setMethod(
  "initialize", "market_model",
  function(.Object, model_type_string, verbose,
           specification,
           correlated_shocks,
           data,
           system_initializer) {

    ## Model assignments
    .Object@model_type_string <- model_type_string
    .Object@logger <- new("model_logger", verbose)
    .Object@system@correlated_shocks <- correlated_shocks
    print_info(.Object@logger, "This is '", model_name(.Object), "' model")

    .Object@key_columns <- all.vars(formula(specification, lhs = 3:4, rhs = 0))
    .Object@time_column <- all.vars(formula(specification, lhs = 4, rhs = 0))

    .Object@explanatory_columns <- all.vars(specification[[3]])

    .Object@data_columns <- all.vars(specification)
    .Object@columns <- unique(c(.Object@key_columns, .Object@data_columns))

    ## Data assignment
    .Object@model_tibble <- data

    ## Create model tibble
    len <- nrow(.Object@model_tibble)
    .Object@model_tibble <- .Object@model_tibble %>%
      dplyr::select(!!!.Object@columns) %>%
      na.omit()
    drops <- len - nrow(.Object@model_tibble)
    if (drops) {
      print_warning(.Object@logger, "Dropping ", drops, " rows due to omitted values.")
    }

    remove_unused_levels <- function(x) {
      initial_levels <- levels(x)
      x <- factor(x)
      remaining_levels <- levels(x)
      removed_levels <- initial_levels[!(initial_levels %in% remaining_levels)]
      if (length(removed_levels)) {
        print_warning(
          .Object@logger, "Removing unobserved '",
          paste0(removed_levels, collapse = ", "), "' level(s)."
        )
      }
      x
    }
    .Object@model_tibble <- .Object@model_tibble %>%
      dplyr::mutate(dplyr::across(
        where(is.factor),
        remove_unused_levels
      ))

    ## Create primary key column
    key_columns_syms <- rlang::syms(.Object@key_columns)
    .Object@model_tibble <- .Object@model_tibble %>%
      dplyr::mutate(pk = as.integer(paste0(!!!key_columns_syms)))

    ## Do we need to use lags?
    if (.Object@model_type_string %in% c(
      "Directional", "Deterministic Adjustment", "Stochastic Adjustment"
    )) {
      ## Generate lags
      key_syms <- rlang::syms(.Object@key_columns[.Object@key_columns !=
        .Object@time_column])
      price_column <- all.vars(formula(specification, lhs = 2, rhs = 0))
      price_sym <- rlang::sym(price_column)
      time_sym <- rlang::sym(.Object@time_column)
      lagged_price_column <- paste0("LAGGED_", price_column)
      lagged_price_sym <- rlang::sym(lagged_price_column)

      .Object@model_tibble <- .Object@model_tibble %>%
        dplyr::group_by(!!!key_syms) %>%
        dplyr::mutate(
          !!lagged_price_sym := dplyr::lag(!!price_sym, order_by = !!time_sym)
        ) %>%
        dplyr::ungroup()

      drop_rows <- .Object@model_tibble %>%
        dplyr::select(!!lagged_price_sym) %>%
        is.na() %>%
        c()
      .Object@model_tibble <- .Object@model_tibble[!drop_rows, ]
      print_info(
        .Object@logger, "Dropping ",
        sum(drop_rows), " rows by generating '", lagged_price_column, "'."
      )

      ## Generate first differences
      diff_column <- paste0(price_column, "_DIFF")
      diff_sym <- rlang::sym(diff_column)

      .Object@model_tibble <- .Object@model_tibble %>%
        dplyr::group_by(!!!key_syms) %>%
        dplyr::mutate(!!diff_sym := !!price_sym - !!lagged_price_sym) %>%
        dplyr::ungroup()
    }

    .Object@system <- system_initializer(
      specification,
      .Object@model_tibble, correlated_shocks
    )

    print_verbose(
      .Object@logger, "Using columns ",
      paste0(.Object@columns, collapse = ", "), "."
    )

    .Object
  }
)

initialize_from_formula <- function(model_type, specification, data,
                                    correlated_shocks, verbose,
                                    estimation_options) {
  specification <- Formula::Formula(specification)
  quantity <- terms(specification, lhs = 1, rhs = 0)[[2]]
  price <- terms(specification, lhs = 2, rhs = 0)[[2]]
  demand <- terms(specification, lhs = 0, rhs = 1)[[2]]
  supply <- terms(specification, lhs = 0, rhs = 2)[[2]]
  subject <- terms(specification, lhs = 3, rhs = 0)[[2]]
  time <- terms(specification, lhs = 4, rhs = 0)[[2]]
  args <- list(
    model_type,
    quantity = quantity, price = price,
    demand = demand, supply = supply,
    subject = subject, time = time,
    data = data, correlated_shocks = correlated_shocks, verbose = verbose
  )
  if (length(specification)[2] > 2) {
    price_dynamics <- terms(specification, lhs = 0, rhs = 3)[[2]]
    args <- append(args, price_dynamics)
  }
  model <- do.call(new, args)
  if (length(estimation_options)) {
    do.call(diseq::estimate, c(list(model), estimation_options))
  } else {
    diseq::estimate(model)
  }
}

#' @title Single call estimation
#'
#' @details
#' The functions of this section combine model initialization and estimation
#' into a single call. They also provide a less verbose interface to the
#' functionality of the package. The functions expect a formula following the
#' specification described in \link[=market_model_formula]{formula}, a
#' dataset, and optionally further initialization (see
#' \link[=initialize_market_model]{model initialization}) and
#' estimation (see \link[=estimate]{model estimation}) options.
#'
#' Each of these functions parses the passed formula, initializes the model
#' specified by the function's name, fit the model to the passed data using
#' the estimation options and returns fitted model.
#'
#' @param specification The model's formula.
#' @param data The data to be used with the model.
#' @param correlated_shocks Should the model's system entail correlated shocks?
#' By default the argument is set to \code{TRUE}.
#' @param verbose The verbosity with which operations on the model print
#' messages. By default the value is set to \code{0}, which prints only errors.
#' @param estimation_options A list with options to be used in the estimation
#' call. See \code{\link[diseq]{estimate}} for the available options.
#' @return The fitted model.
#' @name single_call_estimation
NULL


#' @title Market model formula.
#' @details Market model formulas adhere to the following specification:
#'
#' \code{quantity | price | subject | time ~ demand | supply}
#'
#' where
#' \itemize{
#' \item{quantity} The model's traded (observed) quantity variable.
#' \item{price} The model's price variable.
#' \item{quantity} The model's subject (e.g. firm) identification variable.
#' \item{quantity} The model's time identification variable.
#' \item{demand} The right hand side of the model's demand equation.
#' \item{supply} The right hand side of the model's supply equation.
#' }
#'
#' The \code{\linkS4class{diseq_stochastic_adjustment}} additionally specify
#' price dynamics by appending the right hand side of the equation at the end
#' of the formula, i.e.
#'
#' \code{quantity | price | subject | time ~ demand | supply | price_dynamics}
#'
#' The left hand side part of the model formula specifies the elements that
#' are needed for initializing the model. The market models of the package
#' prepare the data based on these four variables using their respective
#' identification assumptions. See \link[=market_models]{market model classes}
#' for more details.
#'
#' The function provides access to the formula used in model initialization.
#' @param x A market model object.
#' @return The model's formula
#' @examples
#' \donttest{
#' model <- simulate_model(
#'   "diseq_stochastic_adjustment", list(
#'     # observed entities, observed time points
#'     nobs = 500, tobs = 3,
#'     # demand coefficients
#'     alpha_d = -0.1, beta_d0 = 9.8, beta_d = c(0.3, -0.2), eta_d = c(0.6, -0.1),
#'     # supply coefficients
#'     alpha_s = 0.1, beta_s0 = 5.1, beta_s = c(0.9), eta_s = c(-0.5, 0.2),
#'     # price equation coefficients
#'     gamma = 1.2, beta_p0 = 3.1, beta_p = c(0.8)
#'   ),
#'   seed = 31
#' )
#'
#' # access the model's formula
#' formula(model)
#' }
#' @aliases market_model_formula
#' @export
setMethod(
  "formula", signature(x = "market_model"),
  function(x) formula(x@system@formula)
)

#' Prints a short description of the model.
#'
#' Sends basic information about the model to standard output.
#' @param object A model object.
#' @examples
#' \donttest{
#' model <- simulate_model(
#'   "diseq_stochastic_adjustment", list(
#'     # observed entities, observed time points
#'     nobs = 500, tobs = 3,
#'     # demand coefficients
#'     alpha_d = -0.1, beta_d0 = 9.8, beta_d = c(0.3, -0.2), eta_d = c(0.6, -0.1),
#'     # supply coefficients
#'     alpha_s = 0.1, beta_s0 = 5.1, beta_s = c(0.9), eta_s = c(-0.5, 0.2),
#'     # price equation coefficients
#'     gamma = 1.2, beta_p0 = 3.1, beta_p = c(0.8)
#'   ),
#'   seed = 31
#' )
#'
#' # print short model information
#' show(model)
#' }
#' @rdname show
#' @export
setMethod("show", signature(object = "market_model"), function(object) {
  cat(sprintf(
    "%s Model for Markets in %s\n",
    object@model_type_string, object@market_type_string
  ))
  show_implementation(object@system)
  cat(sprintf(
    "  %-18s: %s\n", "Shocks",
    ifelse(object@system@correlated_shocks, "Correlated", "Independent")
  ))
})

#' @title Model and fit summaries
#' @description Methods that summarize models and their estimates.
#' @param object An object to be summarized.
#' @name summaries
#' @examples
#' \donttest{
#' model <- simulate_model(
#'   "diseq_stochastic_adjustment", list(
#'     # observed entities, observed time points
#'     nobs = 500, tobs = 3,
#'     # demand coefficients
#'     alpha_d = -0.1, beta_d0 = 9.8, beta_d = c(0.3, -0.2), eta_d = c(0.6, -0.1),
#'     # supply coefficients
#'     alpha_s = 0.1, beta_s0 = 5.1, beta_s = c(0.9), eta_s = c(-0.5, 0.2),
#'     # price equation coefficients
#'     gamma = 1.2, beta_p0 = 3.1, beta_p = c(0.8)
#'   ),
#'   seed = 556
#' )
#'
#' # print model summary
#' summary(model)
#' }
NULL

#' @describeIn summaries Summarizes the model.
#' @description \code{market_model}: Prints basic information about the
#' passed model object. In addition to the output of
#' the \code{\link{show}} method, \code{summary} prints
#' \itemize{
#' \item the number of observations,
#' \item the number of observations in each equation for models with sample
#' separation, and
#' \item various categories of variables.
#' }
#' @export
setMethod("summary", signature(object = "market_model"), function(object) {
  show(object)
  cat(sprintf("  %-18s: %d\n", "Nobs", nrow(object@model_tibble)))
  summary_implementation(object@system)
  cat(sprintf(
    "  %-18s: %s\n", "Key Var(s)",
    paste0(object@key_columns, collapse = ", ")
  ))
  if (!is.null(object@time_column)) {
    cat(sprintf(
      "  %-18s: %s\n", "Time Var",
      paste0(object@time_column, collapse = ", ")
    ))
  }
})

#' Minus log-likelihood.
#'
#' Returns the opposite of the log-likelihood. The likelihood functions are based on
#' Maddala and Nelson (1974) \doi{10.2307/1914215}. The likelihoods expressions
#' that the function uses are derived in
#' Karapanagiotis (2020) \doi{10.2139/ssrn.3525622}. The function calculates
#' the model's log likelihood by evaluating the log likelihood of each observation in
#' the sample and summing the evaluation results.
#' @param object A model object.
#' @param parameters A vector of parameters at which the function is to be evaluated.
#' @return The opposite of the sum of the likelihoods evaluated for each observation.
#' @rdname minus_log_likelihood
#' @export
setGeneric("minus_log_likelihood", function(object, parameters) {
  standardGeneric("minus_log_likelihood")
})

#' Gradient
#'
#' Returns the gradient of the opposite of the log-likelihood evaluated at the passed
#' parameters.
#' @param object A model object.
#' @param parameters A vector of parameters at which the gradient is to be evaluated.
#' @return The opposite of the model log likelihood's gradient.
#' @rdname gradient
#' @export
setGeneric("gradient", function(object, parameters) {
  standardGeneric("gradient")
})

#' Hessian
#'
#' Returns the hessian of the opposite of the log-likelihood evaluated at the passed
#' parameters.
#' @param object A model object.
#' @param parameters A vector of parameters at which the hessian is to be evaluated.
#' @return The opposite of the model log likelihood's hessian.
#' @rdname hessian
#' @export
setGeneric("hessian", function(object, parameters) {
  standardGeneric("hessian")
})

validate_gradient_option <- function(object, option) {
  allowed <- c("calculated", "numerical")
  if (!(option %in% allowed)) {
    print_error(
      object@logger,
      paste0(
        "Invalid `gradient` option '", option, "'. Valid options are ('",
        paste0(allowed, collapse = "', '"), "')."
      )
    )
  }
}

validate_hessian_option <- function(object, option) {
  allowed <- c("skip", "calculated", "numerical")
  if (!(option %in% allowed)) {
    print_error(
      object@logger,
      paste0(
        "Invalid `hessian` option '", option, "'. Valid options are ('",
        paste0(allowed, collapse = "', '"), "')."
      )
    )
  }
}

validate_standard_error_option <- function(object, option) {
  allowed <- c("homoscedastic", "heteroscedastic")
  if (!(option %in% allowed || all(option %in% object@columns))) {
    print_error(
      object@logger,
      paste0(
        "Invalid `standard_error` option '", option, "'. Valid options are ('",
        paste0(allowed, collapse = "', '"), "') or a vector of model variable names."
      )
    )
  }
}


#' Maximize the log-likelihood.
#'
#' Maximizes the log-likelihood using the
#' \href{https://www.gnu.org/software/gsl/doc/html/multimin.html}{\code{GSL}}
#' implementation of the BFGS algorithm. This function is primarily intended for
#' advanced usage. The \code{\link{estimate}} functionality is a fast,
#' analysis-oriented alternative. If the
#' \href{https://www.gnu.org/software/gsl/doc/html/multimin.html}{\code{GSL}} is not
#' available, the function returns a trivial result list with status set equal to -1.
#' If the
#' \href{https://en.cppreference.com/w/cpp/algorithm/execution_policy_tag_t}{C++17
#' execution policies}
#' are available, the implementation of the optimization is parallelized.
#' @param object A model object.
#' @param start Initializing vector.
#' @param step Optimization step.
#' @param objective_tolerance Objective optimization tolerance.
#' @param gradient_tolerance Gradient optimization tolerance.
#' @param max_it Maximum allowed number of iterations.
#' @return A list with the optimization output.
#' @rdname maximize_log_likelihood
#' @seealso estimate
#' @examples
#' \donttest{
#' model <- simulate_model(
#'   "equilibrium_model", list(
#'     # observed entities, observed time points
#'     nobs = 500, tobs = 3,
#'     # demand coefficients
#'     alpha_d = -0.9, beta_d0 = 14.9, beta_d = c(0.3, -0.2), eta_d = c(-0.03, -0.01),
#'     # supply coefficients
#'     alpha_s = 0.9, beta_s0 = 3.2, beta_s = c(0.03), eta_s = c(0.05, 0.02)
#'   ),
#'   seed = 99
#' )
#'
#' # maximize the model's log-likelihood
#' mll <- maximize_log_likelihood(
#'   model,
#'   start = NULL, step = 1e-5,
#'   objective_tolerance = 1e-4, gradient_tolerance = 1e-3, max_it = 1e+3
#' )
#' mll
#' }
#' @export
setGeneric("maximize_log_likelihood", function(object, start, step, objective_tolerance,
                                               gradient_tolerance, max_it) {
  standardGeneric("maximize_log_likelihood")
})

#' Likelihood scores.
#'
#' It calculates the gradient of the likelihood at the given parameter point
#' for each observation in the sample. It, therefore, returns an n x k matrix,
#' where n denotes the number of observations in the sample and k the number
#' of estimated parameters. The ordering of the parameters is the same as
#' the one that is used in the summary of the results. The method can be
#' called either using directly a fitted model object, or by separately
#' providing a model object and a parameter vector.
#' @param object A model object.
#' @param parameters A vector with model parameters.
#' @param fit A fitted model object.
#' @return The score matrix.
#' @rdname scores
#' @examples
#' \donttest{
#' model <- simulate_model(
#'   "diseq_basic", list(
#'     # observed entities, observed time points
#'     nobs = 500, tobs = 3,
#'     # demand coefficients
#'     alpha_d = -0.9, beta_d0 = 8.9, beta_d = c(0.6), eta_d = c(-0.2),
#'     # supply coefficients
#'     alpha_s = 0.9, beta_s0 = 4.2, beta_s = c(0.03, 1.2), eta_s = c(0.1)
#'   ),
#'   seed = 7523
#' )
#'
#' # estimate the model object (BFGS is used by default)
#' fit <- estimate(model)
#'
#' # Calculate the score matrix
#' head(scores(model, coef(fit)))
#' }
#' @export
setGeneric("scores", function(object, parameters, fit = missing()) {
  standardGeneric("scores")
})

setGeneric("set_heteroscedasticity_consistent_errors", function(object, ...) {
  standardGeneric("set_heteroscedasticity_consistent_errors")
})

setGeneric("set_clustered_errors", function(object, ...) {
  standardGeneric("set_clustered_errors")
})

#' Model description.
#'
#' A unique identifying string for the model.
#' @param object A model object.
#' @return A string representation of the model.
#' @rdname model_name
#' @export
setGeneric("model_name", function(object) {
  standardGeneric("model_name")
})

#' @title Market side descriptive statistics
#' @details Calculates and returns basic descriptive statistics for the model's demand
#' or supply side data. Factor variables are excluded from the calculations. The function
#' calculates and returns:
#' \itemize{
#' \item \code{nobs} Number of observations.
#' \item \code{nmval} Number of missing values.
#' \item \code{min} Minimum observation.
#' \item \code{max} Maximum observation.
#' \item \code{range} Observations' range.
#' \item \code{sum} Sum of observations.
#' \item \code{median} Median observation.
#' \item \code{mean} Mean observation.
#' \item \code{mean_se} Mean squared error.
#' \item \code{mean_ce} Confidence interval bound.
#' \item \code{var} Variance.
#' \item \code{sd} Standard deviation.
#' \item \code{coef_var} Coefficient of variation.
#' }
#' @param object A model object.
#' @return A data \code{tibble} containing descriptive statistics.
#' @examples
#' # initialize the basic model using the houses dataset
#' model <- new(
#'   "diseq_basic", # model type
#'   subject = ID, time = TREND, quantity = HS, price = RM,
#'   demand = RM + TREND + W + CSHS + L1RM + L2RM + MONTH,
#'   supply = RM + TREND + W + L1RM + MA6DSF + MA3DHF + MONTH,
#'   fair_houses(), # data
#'   correlated_shocks = FALSE # allow shocks to be correlated
#' )
#'
#' # get descriptive statistics of demand side variables
#' demand_descriptives(model)
#'
#' # get descriptive statistics of supply side variables
#' supply_descriptives(model)
#' @name market_descriptives
NULL

setGeneric("descriptives", function(object, variables) {
  standardGeneric("descriptives")
})

#' @describeIn market_descriptives Demand descriptive statistics.
#' @export
setGeneric("demand_descriptives", function(object) {
  standardGeneric("demand_descriptives")
})

#' @describeIn market_descriptives Supply descriptive statistics.
#' @export
setGeneric("supply_descriptives", function(object) {
  standardGeneric("supply_descriptives")
})

setMethod(
  "set_heteroscedasticity_consistent_errors", signature(object = "market_model"),
  function(object, fit) {
    fit@details$original_hessian <- fit@details$hessian
    scores <- scores(object, coef(fit))
    adjustment <- MASS::ginv(t(scores) %*% scores)
    fit@details$hessian <- fit@details$hessian %*% adjustment %*% fit@details$hessian
    fit@vcov <- MASS::ginv(fit@details$hessian)
    fit
  }
)

setMethod(
  "set_clustered_errors", signature(object = "market_model"),
  function(object, fit, cluster_errors_by) {
    if (!(cluster_errors_by %in% names(object@model_tibble))) {
      print_error(
        object@logger, "Cluster variable is not among model data variables."
      )
    }
    fit@details$original_hessian <- fit@details$hessian
    cluster_var <- rlang::syms(cluster_errors_by)
    clustered_scores <- tibble::tibble(
      object@model_tibble %>% dplyr::select(!!!cluster_var),
      tibble::as_tibble(scores(object, coef(fit)))
    ) %>%
      dplyr::group_by(!!!cluster_var) %>%
      dplyr::group_map(~ t(as.matrix(.)) %*% (as.matrix(.)))
    fit@details$number_of_clusters <- length(clustered_scores)
    adjustment <- MASS::ginv(Reduce("+", clustered_scores))
    fit@details$hessian <- fit@details$hessian %*% adjustment %*% fit@details$hessian
    fit@vcov <- MASS::ginv(fit@details$hessian) * (
      fit@details$number_of_clusters / (fit@details$number_of_clusters - 1)
    )
    fit
  }
)

#' @rdname model_name
setMethod("model_name", signature(object = "market_model"), function(object) {
  paste0(
    object@model_type_string, " with ",
    ifelse(object@system@correlated_shocks, "correlated", "independent"), " shocks"
  )
})

#' Number of observations.
#'
#' Returns the number of observations that are used by an initialized model. The number
#' of used observations may differ from the numbers of observations of the data set
#' that was passed to the model's initialization.
#' @param object A model object.
#' @return The number of used observations.
#' @rdname nobs
#' @export
setMethod(
  "nobs", signature(object = "market_model"),
  function(object) {
    nrow(object@model_tibble)
  }
)

setMethod(
  "descriptives", signature(object = "market_model"),
  function(object, variables = NULL) {
    if (is.null(variables)) {
      variables <- object@columns
    }
    variables <- variables[sapply(
      variables,
      function(c) !is.factor(object@model_tibble[[c]])
    )]

    tibble::as_tibble(apply(
      object@model_tibble[, variables], 2,
      function(x) {
        c(
          nobs = length(x), nmval = sum(is.na(x)),
          min = min(x), max = max(x), range = max(x) - min(x),
          sum = sum(x), median = median(x), mean = mean(x),
          mean_se = sqrt(var(x) / length(x)),
          mean_ce = qnorm(0.975) * sqrt(var(x) / length(x)),
          var = var(x), sd = sd(x), coef_var = sd(x) / mean(x)
        )
      }
    ), rownames = "col")
  }
)

#' @rdname market_descriptives
setMethod("demand_descriptives", signature(object = "market_model"), function(object) {
  descriptives(object, gsub(
    object@system@demand@variable_prefix, "",
    all.vars(object@system@demand@formula[[3]])
  ))
})

#' @rdname market_descriptives
setMethod("supply_descriptives", signature(object = "market_model"), function(object) {
  descriptives(object, gsub(
    object@system@supply@variable_prefix, "",
    all.vars(object@system@supply@formula[[3]])
  ))
})

setGeneric("calculate_initializing_values", function(object) {
  standardGeneric("calculate_initializing_values")
})

setMethod(
  "calculate_initializing_values", signature(object = "market_model"),
  function(object) {
    dlm <- stats::lm(
      object@system@quantity_vector ~
      object@system@demand@independent_matrix - 1
    )
    names(dlm$coefficients) <- colnames(
      object@system@demand@independent_matrix
    )

    slm <- stats::lm(
      object@system@quantity_vector ~
      object@system@supply@independent_matrix - 1
    )
    names(slm$coefficients) <- colnames(
      object@system@supply@independent_matrix
    )

    ## Set demand initializing values
    if (any(is.na(dlm$coefficients))) {
      print_warning(
        object@logger,
        "Setting demand side NA initial values to zero: ",
        paste0(names(dlm$coefficients[is.na(dlm$coefficients)]), collapse = ", "), "."
      )
      dlm$coefficients[is.na(dlm$coefficients)] <- 0
    }
    start_names <- c(
      prefixed_price_variable(object@system@demand),
      prefixed_control_variables(object@system@demand)
    )
    start <- c(dlm$coefficients[start_names])

    ## Set supply initializing values
    if (any(is.na(slm$coefficients))) {
      print_warning(
        object@logger,
        "Setting supply side NA initial values to zero: ",
        paste0(names(slm$coefficients[is.na(slm$coefficients)]), collapse = ", ")
      )
      slm$coefficients[is.na(slm$coefficients)] <- 0
    }
    start_names <- c(
      prefixed_price_variable(object@system@supply),
      prefixed_control_variables(object@system@supply)
    )
    start <- c(start, slm$coefficients[start_names])

    if (object@model_type_string %in% c(
      "Deterministic Adjustment",
      "Stochastic Adjustment"
    )) {
      start <- c(start, gamma = 1)
      names(start)[length(start)] <- price_differences_variable(object@system)
    }

    start <- c(start, 1, 1)
    names(start)[(length(start) - 1):length(start)] <- c(
      prefixed_variance_variable(object@system@demand),
      prefixed_variance_variable(object@system@supply)
    )

    if (object@system@correlated_shocks) {
      start <- c(start, rho = 0)
      names(start)[length(start)] <- correlation_variable(object@system)
    }

    start
  }
)

setGeneric("prepare_initializing_values", function(object, initializing_vector) {
  standardGeneric("prepare_initializing_values")
})

setMethod(
  "prepare_initializing_values", signature(object = "market_model"),
  function(object, initializing_vector) {
    if (is.null(initializing_vector)) {
      print_verbose(object@logger, "Initializing using linear regression estimations.")
      initializing_vector <- calculate_initializing_values(object)
    }
    names(initializing_vector) <- likelihood_variables(object@system)
    print_debug(
      object@logger, "Using starting values: ",
      paste(names(initializing_vector), initializing_vector,
        sep = " = ",
        collapse = ", "
      )
    )

    initializing_vector
  }
)

#' @title Market side aggregation.
#'
#' @details Calculates the sample's aggregate demand or supply at the passed set of
#' parameters. If the model is static, as is for example the case of
#' \code{\linkS4class{equilibrium_model}}, then all observations are aggregated. If the
#' used data have a time dimension and aggregation per date is required, it can be
#' manually performed using the \code{\link{demanded_quantities}} and
#' \code{\link{supplied_quantities}} functions. If the model has a dynamic component,
#' such as the \code{\linkS4class{diseq_deterministic_adjustment}}, then demanded
#' and supplied quantities are automatically aggregated for each time point.
#' @param fit A fitted market model object.
#' @param model A model object.
#' @param parameters A vector of model's parameters.
#' @return The sum of the estimated demanded or supplied quantities evaluated at the
#' given parameters.
#' @name market_aggregation
#' @examples
#' \donttest{
#' fit <- diseq_basic(
#'   HS | RM | ID | TREND ~
#'   RM + TREND + W + CSHS + L1RM + L2RM + MONTH |
#'     RM + TREND + W + L1RM + MA6DSF + MA3DHF + MONTH,
#'   fair_houses(),
#'   correlated_shocks = FALSE
#' )
#'
#' # get estimated aggregate demand
#' aggregate_demand(fit)
#'
#' # simulate the deterministic adjustment model
#' model <- simulate_model(
#'   "diseq_deterministic_adjustment", list(
#'     # observed entities, observed time points
#'     nobs = 500, tobs = 3,
#'     # demand coefficients
#'     alpha_d = -0.6, beta_d0 = 9.8, beta_d = c(0.3, -0.2), eta_d = c(0.6, -0.1),
#'     # supply coefficients
#'     alpha_s = 0.2, beta_s0 = 4.1, beta_s = c(0.9), eta_s = c(-0.5, 0.2),
#'     # price equation coefficients
#'     gamma = 0.9
#'   ),
#'   seed = 1356
#' )
#'
#' # estimate the model object
#' fit <- estimate(model)
#'
#' # get estimated aggregate demand
#' aggregate_demand(fit)
#'
#' # get estimated aggregate demand
#' aggregate_supply(fit)
#' }
#' @seealso demanded_quantities, supplied_quantities
NULL

aggregate_equation <- function(model, parameters, equation) {
  model@system <- set_parameters(model@system, parameters)
  quantities <- quantities(slot(model@system, equation))
  result <- NULL
  if (!is.null(model@time_column)) {
    time_symbol <- rlang::sym(model@time_column)
    aggregate_symbol <- rlang::sym(colnames(quantities))
    result <- model@model_tibble[, model@time_column] %>%
      dplyr::mutate(!!aggregate_symbol := quantities) %>%
      dplyr::group_by(!!time_symbol) %>%
      dplyr::summarise(!!aggregate_symbol := sum(!!aggregate_symbol))
  } else {
    result <- sum(quantities)
  }
  result
}


#' @describeIn market_aggregation Demand aggregation.
#' @export
setGeneric("aggregate_demand", function(fit, model, parameters) {
  standardGeneric("aggregate_demand")
})

#' @rdname market_aggregation
setMethod(
  "aggregate_demand", signature(fit = "missing", model = "market_model"),
  function(model, parameters) {
    aggregate_equation(model, parameters, "demand")
  }
)

#' @title Estimated market quantities.
#'
#' @details Calculates and returns the estimated demanded or supplied quantities for
#' each observation at the passed vector of parameters.
#' @param fit A fitted model object.
#' @param model A model object.
#' @param parameters A vector of model's parameters.
#' @return A vector with the demanded quantities evaluated at the given parameter
#' vector.
#' @examples
#' \donttest{
#' fit <- diseq_basic(
#'   HS | RM | ID | TREND ~
#'   RM + TREND + W + CSHS + L1RM + L2RM + MONTH |
#'     RM + TREND + W + L1RM + MA6DSF + MA3DHF + MONTH,
#'   fair_houses(),
#'   correlated_shocks = FALSE
#' )
#'
#' # get estimated demanded and supplied quantities
#' head(cbind(
#'   demanded_quantities(fit),
#'   supplied_quantities(fit)
#' ))
#' }
#' @name market_quantities
NULL

#' @describeIn market_quantities Estimated demanded quantities.
#' @export
setGeneric("demanded_quantities", function(fit, model, parameters) {
  standardGeneric("demanded_quantities")
})

#' @rdname market_quantities
setMethod(
  "demanded_quantities", signature(fit = "missing", model = "market_model"),
  function(model, parameters) {
    model@system <- set_parameters(model@system, parameters)
    quantities(model@system@demand)
  }
)

#' @describeIn market_aggregation Supply aggregation.
#' @export
setGeneric("aggregate_supply", function(fit, model, parameters) {
  standardGeneric("aggregate_supply")
})

#' @rdname market_aggregation
setMethod(
  "aggregate_supply", signature(fit = "missing", model = "market_model"),
  function(model, parameters) {
    aggregate_equation(model, parameters, "supply")
  }
)

#' @describeIn market_quantities Estimated supplied quantities.
#' @export
setGeneric("supplied_quantities", function(fit, model, parameters) {
  standardGeneric("supplied_quantities")
})

#' @rdname market_quantities
setMethod(
  "supplied_quantities", signature(fit = "missing", model = "market_model"),
  function(model, parameters) {
    model@system <- set_parameters(model@system, parameters)
    quantities(model@system@supply)
  }
)
