% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/paragraph2vec.R
\name{paragraph2vec}
\alias{paragraph2vec}
\title{Train a paragraph2vec also known as doc2vec model on text}
\usage{
paragraph2vec(
  x,
  type = c("PV-DBOW", "PV-DM"),
  dim = 50,
  window = ifelse(type == "PV-DM", 5L, 10L),
  iter = 5L,
  lr = 0.05,
  hs = FALSE,
  negative = 5L,
  sample = 0.001,
  min_count = 5L,
  threads = 1L,
  encoding = "UTF-8",
  ...
)
}
\arguments{
\item{x}{a data.frame with columns doc_id and text or the path to the file on disk containing training data.\cr
Note that the text column should be of type character, should contain less than 1000 words where space or tab is 
used as a word separator and that the text should not contain newline characters as these are considered document delimiters.}

\item{type}{character string with the type of algorithm to use, either one of
\itemize{
\item{'PV-DM': Distributed Memory paragraph vectors}
\item{'PV-DBOW': Distributed Bag Of Words paragraph vectors}
}
Defaults to 'PV-DBOW'.}

\item{dim}{dimension of the word and paragraph vectors. Defaults to 50.}

\item{window}{skip length between words. Defaults to 10 for PV-DM and 5 for PV-DBOW}

\item{iter}{number of training iterations. Defaults to 20.}

\item{lr}{initial learning rate also known as alpha. Defaults to 0.05}

\item{hs}{logical indicating to use hierarchical softmax instead of negative sampling. Defaults to FALSE indicating to do negative sampling.}

\item{negative}{integer with the number of negative samples. Only used in case hs is set to FALSE}

\item{sample}{threshold for occurrence of words. Defaults to 0.001}

\item{min_count}{integer indicating the number of time a word should occur to be considered as part of the training vocabulary. Defaults to 5.}

\item{threads}{number of CPU threads to use. Defaults to 1.}

\item{encoding}{the encoding of \code{x} and \code{stopwords}. Defaults to 'UTF-8'. 
Calculating the model always starts from files allowing to build a model on large corpora. The encoding argument 
is passed on to \code{file} when writing \code{x} to hard disk in case you provided it as a data.frame.}

\item{...}{further arguments passed on to the C++ function \code{paragraph2vec_train} - for expert use only}
}
\value{
an object of class \code{paragraph2vec_trained} which is a list with elements 
\itemize{
\item{model: a Rcpp pointer to the model}
\item{data: a list with elements file: the training data used, n (the number of words in the training data), n_vocabulary (number of words in the vocabulary) and n_docs (number of documents)}
\item{control: a list of the training arguments used, namely min_count, dim, window, iter, lr, skipgram, hs, negative, sample}
}
}
\description{
Construct a paragraph2vec model on text. 
The algorithm is explained at \url{https://arxiv.org/pdf/1405.4053.pdf}.
People also refer to this model as doc2vec.\cr
The model is an extension to the word2vec algorithm, 
where an additional vector for every paragraph is added directly in the training.
}
\examples{
\dontshow{if(require(tokenizers.bpe))\{}
library(tokenizers.bpe)
## Take data and standardise it a bit
data(belgium_parliament, package = "tokenizers.bpe")
str(belgium_parliament)
x <- subset(belgium_parliament, language \%in\% "french")
x$text   <- tolower(x$text)
x$text   <- gsub("[^[:alpha:]]", " ", x$text)
x$text   <- gsub("[[:space:]]+", " ", x$text)
x$text   <- trimws(x$text)
x$nwords <- txt_count_words(x$text)
x <- subset(x, nwords < 1000 & nchar(text) > 0)

## Build the model
model <- paragraph2vec(x = x, type = "PV-DM",   dim = 15,  iter = 5)
\donttest{
model <- paragraph2vec(x = x, type = "PV-DBOW", dim = 100, iter = 20)
}
str(model)
embedding <- as.matrix(model, which = "words")
embedding <- as.matrix(model, which = "docs")
head(embedding)

## Get vocabulary
vocab <- summary(model, type = "vocabulary",  which = "docs")
vocab <- summary(model, type = "vocabulary",  which = "words")
\dontshow{\} # End of main if statement running only if the required packages are installed}
}
\references{
\url{https://arxiv.org/pdf/1405.4053.pdf}, \url{https://groups.google.com/g/word2vec-toolkit/c/Q49FIrNOQRo/m/J6KG8mUj45sJ}
}
\seealso{
\code{\link{predict.paragraph2vec}}, \code{\link{as.matrix.paragraph2vec}}
}
