#' @title Show a `ggraph`/`ggplot2` representation
#'   of your drake project.
#' @description This function requries packages `ggplot2` and `ggraph`.
#'   Install them with `install.packages(c("ggplot2", "ggraph"))`.
#' @export
#' @seealso [vis_drake_graph()], [sankey_drake_graph()],
#'   [render_drake_ggraph()]
#' @return A `ggplot2` object, which you can modify with more layers,
#'   show with `plot()`, or save as a file with `ggsave()`.
#' @inheritParams drake_graph_info
#' @inheritParams render_drake_ggraph
#' @examples
#' \dontrun{
#' test_with_dir("Quarantine side effects.", {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' config <- drake_config(my_plan)
#' # Plot the network graph representation of the workflow.
#' drake_ggraph(config) # Save to a file with `ggplot2::ggsave()`.
#' })
#' }
drake_ggraph <- function(
  config = drake::read_drake_config(),
  build_times = "build",
  digits = 3,
  targets_only = FALSE,
  split_columns = NULL,
  main = NULL,
  from = NULL,
  mode = c("out", "in", "all"),
  order = NULL,
  subset = NULL,
  make_imports = TRUE,
  from_scratch = FALSE,
  full_legend = FALSE,
  group = NULL,
  clusters = NULL,
  show_output_files = TRUE
){
  assert_pkgs(c("ggplot2", "ggraph"))
  graph_info <- drake_graph_info(
    config = config,
    from = from,
    mode = mode,
    order = order,
    subset = subset,
    build_times = build_times,
    digits = digits,
    targets_only = targets_only,
    split_columns = split_columns,
    make_imports = make_imports,
    from_scratch = from_scratch,
    full_legend = full_legend,
    group = group,
    clusters = clusters,
    show_output_files = show_output_files
  )
  if (is.null(main)){
    main <- graph_info$default_title
  }
  render_drake_ggraph(graph_info, main = main)
}

#' @title Render a static `ggplot2`/`ggraph` visualization from
#'   [drake_graph_info()] output.
#' @description This function requries packages `ggplot2` and `ggraph`.
#'   Install them with `install.packages(c("ggplot2", "ggraph"))`.
#' @export
#' @seealso [vis_drake_graph()], [sankey_drake_graph()], [drake_ggraph()]
#' @return A `ggplot2` object, which you can modify with more layers,
#'   show with `plot()`, or save as a file with `ggsave()`.
#' @inheritParams drake_graph_info
#' @param graph_info list of data frames generated by
#'   [drake_graph_info()].
#'   There should be 3 data frames: `nodes`, `edges`,
#'   and `legend_nodes`.
#' @param main character string, title of the graph
#' @examples
#' \dontrun{
#' test_with_dir("Quarantine side effects.", {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' # Instead of jumpting right to vis_drake_graph(), get the data frames
#' # of nodes, edges, and legend nodes.
#' config <- drake_config(my_plan) # Internal configuration list
#' drake_ggraph(config) # Jump straight to the static graph.
#' # Get the node and edge info that vis_drake_graph() just plotted:
#' graph <- drake_graph_info(config)
#' render_drake_ggraph(graph)
#' })
#' }
render_drake_ggraph <- function(
  graph_info,
  main = graph_info$default_title
){
  assert_pkgs(c("ggplot2", "ggraph"))
  graph <- igraph::graph_from_data_frame(
    d = graph_info$edges,
    directed = TRUE,
    vertices = graph_info$nodes
  )
  colors <- graph_info$nodes$color
  names(colors) <- graph_info$nodes$status
  shapes <- graph_info$nodes$shape
  names(shapes) <- graph_info$nodes$type
  shapes <- gsub("dot", "circle", shapes)
  layout <- ggraph::create_layout(graph, layout = "sugiyama")
  tmp <- layout$x
  layout$x <- -layout$y
  layout$y <- tmp
  layout$label <- paste0("\n\n", layout$label)
  status <- type <- label <- node1.name <- node2.name <- NULL
  ggraph::ggraph(layout) +
    ggraph::geom_node_point(
      ggplot2::aes(color = status, shape = type),
      size = 5,
      alpha = 0.5
    ) +
    ggraph::geom_node_text(ggplot2::aes(label = label)) +
    ggraph::geom_edge_link(
      arrow = ggplot2::arrow(length = ggplot2::unit(4, "mm")),
      alpha = 0.25
    ) +
    ggplot2::xlim(padded_scale(layout$x)) +
    ggplot2::ylim(padded_scale(layout$y)) +
    ggplot2::scale_color_manual(values = colors) +
    ggplot2::scale_shape_manual(values = shapes) +
    ggplot2::ggtitle(main) +
    ggplot2::labs(x = "", y = "")
}
