% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pvalue-permutations.R
\name{pt_oob}
\alias{pt_oob}
\title{Permutation Test With Out-Of-Bag Scores}
\usage{
pt_oob(x_train, x_test, scorer, n_pt = 2000)
}
\arguments{
\item{x_train}{Training (reference/validation) sample.}

\item{x_test}{Test sample.}

\item{scorer}{Function which returns a named list with outlier scores from
the training and test sample. The first argument to \code{scorer} must be
\code{x_train}; the second, \code{x_test}. The returned named list contains
two elements: \emph{train} and \emph{test}, each of which is a vector of
(outlier) scores. See notes below for more information.}

\item{n_pt}{The number of permutations.}
}
\value{
A named list of class \code{outlier.test} containing:
\itemize{
   \item \code{statistic}: observed WAUC statistic
   \item \code{seq_mct}: sequential Monte Carlo test, when applicable
   \item \code{p_value}: p-value
   \item \code{outlier_scores}: outlier scores from training and test set
}
}
\description{
Test for no adverse shift with outlier scores. Like goodness-of-fit testing,
this two-sample comparison takes the training set, \code{x_train} or
\code{os_train}, as the reference. The method checks whether the test set,
\code{x_test} or \code{os_test}, is worse off relative to this reference
set.
}
\details{
The empirical null distribution uses \code{n_pt} permutations to estimate
the p-value. For speed, this is implemented as a sequential Monte Carlo test
with the \pkg{simctest} package. See Gandy (2009) for details. The prefix
\emph{pt} refers to permutation test. This approach does not use the
asymptotic null distribution for the weighted AUC (WAUC), the test
statistic. This is the recommended approach for small samples.
}
\section{Notes}{

The scoring function, \code{scorer}, predicts out-of-bag scores to mimic
out-of-sample behaviour. The suffix \emph{oob} stands for out-of-bag to
highlight this point. This out-of-bag variant avoids refitting the
underlying algorithm from \code{scorer} at every permutation. It can, as a
result, be computationally appealing.
}

\examples{
\donttest{
library(dsos)
set.seed(12345)
data(iris)
idx <- sample(nrow(iris), 2 / 3 * nrow(iris))
xy_train <- iris[idx, ]
xy_test <- iris[-idx, ]

# First example: residual diagnostics
scorer_1 <- function(x_train, x_test) score_rd(x_train, x_test, response_name = "Species")
rd_test <- pt_oob(xy_train, xy_test, scorer = scorer_1)
rd_test

# Second example: prediction uncertainty
scorer_2 <- function(x_train, x_test) score_rue(x_train, x_test, response_name = "Species")
rue_test <- pt_oob(xy_train, xy_test, scorer = scorer_2)
rue_test

# Third example: sample memberships (class probabilities)
setosa <- iris[1:50, 1:4] # Training sample: Species == 'setosa'
versicolor <- iris[51:100, 1:4] # Test sample: Species == 'versicolor'
scorer_3 <- function(x_train, x_test) score_cp(x_train, x_test)
cp_test <- pt_oob(setosa, versicolor, scorer = scorer_3)
cp_test
}

}
\references{
Kamulete, V. M. (2022).
\emph{Test for non-negligible adverse shifts}.
In The 38th Conference on Uncertainty in Artificial Intelligence. PMLR.

Gandy, A. (2009).
\emph{Sequential implementation of Monte Carlo tests with uniformly bounded resampling risk}.
Journal of the American Statistical Association, 104(488), 1504-1511.

Li, J., & Fine, J. P. (2010).
\emph{Weighted area under the receiver operating characteristic curve and its application to gene selection}.
Journal of the Royal Statistical Society: Series C (Applied Statistics), 59(4), 673-692.
}
\seealso{
[pt_refit()] for (slower) p-value approximation via refitting.
[at_oob()] for p-value approximation from asymptotic null distribution.

Other permutation-test: 
\code{\link{pt_from_os}()},
\code{\link{pt_refit}()}
}
\concept{permutation-test}
