% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Class.Constructors.R
\name{make.design}
\alias{make.design}
\title{Creates a Survey.Design object}
\usage{
make.design(region = make.region(), transect.type = "line",
  design = "systematic", samplers = numeric(0),
  line.length = numeric(0), effort.allocation = numeric(0),
  design.angle = 0, spacing = numeric(0), edge.protocol = "minus",
  bounding.shape = "rectangle", truncation = 1, coverage.grid = NULL)
}
\arguments{
\item{region}{an object of class Region defining the survey region.}

\item{transect.type}{character variable specifying either "line" or "point"}

\item{design}{a character variable describing the type of design. Either "random",
"systematic", "eszigzag" (equal-spaced zigzag) or "eszigzagcom" (equal spaced zigzag with complementary lines). See details for more information.}

\item{samplers}{the number of samplers you wish the design to generate
(note that the number actually generated may differ slightly due to the
shape of the study region for some designs). This may be one value of a value
per strata.}

\item{line.length}{the total line length you desire or a vector of line lengths
the same length as the number of strata.}

\item{effort.allocation}{numeric values used to indicate the proportion of effort
to be allocated to each strata from number of samplers or line length. If length is
0 (the default) and only a total line length or total number of samplers is supplied,
effort allocated based on stratum area.}

\item{design.angle}{numeric value detailing the angle of the design. Can provide
multiple values relating to strata. The use of the angle varies with design, it
can be either the angle of the grid of points, the angle of lines or the design
axis for the zigzag design. See details.}

\item{spacing}{used by systematic designs, numeric value to define spacing
between transects. Can be a vector of values with one value per strata.}

\item{edge.protocol}{character value indicating whether a "plus" sampling or
"minus" sampling protocol is used.}

\item{bounding.shape}{only applicable to zigzag designs. A character value saying
whether the zigzag transects should be generated using a minimum bounding
"rectangle" or a "convex hull".}

\item{truncation}{A single numeric value describing the longest distance at which
an object may be observed. Truncation distance is constant across strata.}

\item{coverage.grid}{An object of class Coverage.Grid for use when
running the coverage simulation.}
}
\value{
object of a class which inherits from class Survey.Design
}
\description{
Creates a description of a survey design. Designs may use different
types of either point or line transect designs across strata but cannot mix
point and line transect design types within a single design object.
}
\details{
For point transect designs the user may either specify "random" or
"systematic" for the design argument. If the user specifies "random", they
should also provide a value for effort detailing the number of point transects
they wish their survey to have. For stratified designs they may specify a vector
of numbers detailing the number of transects per strata or alternatively use the
effort.allocation argument to allocate a total effort amount proportionally. If
effort.allocation is left blank then effort will be allocated according to strata
area. If the user specified"systematic" they may either provide their desired number
of samplers or a value for spacing which defines the gap between each of the
points (again a vector of spacing values can be provided for each strata).
Optionally the user may select a design.angle. For both random and systematic
point transect designs the user may select either a minus or plus sampling edge
protocol.

For line transect designs the user may either specify "random" (randomly
placed full width lines), "systematic" (systematically placed full width lines),
"eszigzag" (equally spaced zigzag lines) or "eszigzagcom" (two sets of complementary
equally spaced zigzag lines). If the user specifies "random", they
should provide the either the number of samplers they wish the design to generate
or the line length they wish to achieve, either by strata or as a total. If the
user specifies "systematic" they should specify either the number of samplers,
the desired line length or the spacing between lines. The design angle for these
parallel line designs refers to the angle of the lines where 0 is a vertical line
and moving round in a clockwise direction. If the user specifies a zigzag design they
should specify the systematic spacing value, number of samplers or line length
to be used and should choose between generating the design in a minimum bounding
rectangle or a convex hull. The default is minimum bounding rectangle which gives
more even coverage but the convex hull is generally more efficient. The designs
may be generated using plus or minus sampling protocols. Similar to the point
transect designs different values may be specified for each strata for all of
the above options. The design angle for the zigzag designs refers to the angle
of a line which would run through the middle of each zigzag transect if the
zigzags were to be generated within a rectangle. The design angle for zigzags
should usually run along the longest dimension of the study region.

See the Multi Strata Vignette for more complex examples of defining designs
across multiple strata.
}
\examples{
#Point transect example
shapefile.name <- system.file("extdata", "TrackExample.shp", package = "dssd")
region <- make.region(region.name = "study area",
                     shape = shapefile.name)
\donttest{
# Generate coverage grid
cover <- make.coverage(region,
                       n.grid.points = 500)

# Define design
design <- make.design(region = region,
                      transect.type = "point",
                      design = "random",
                      samplers = 25,
                      design.angle = 45,
                      edge.protocol = "minus",
                      truncation = 3,
                      coverage.grid = cover)

# Generate a single survey instance
survey <- generate.transects(design)
plot(region, survey, covered.area = TRUE)

# Warning! this will take some time to run
design <- run.coverage(design, reps = 500)
# Plot the coverage
plot(design)
# Display the design statistics
design

#Multi-strata line transect example
shapefile.name <- system.file("extdata", "AreaRProjStrata.shp", package = "dssd")
region <- make.region(region.name = "study area",
                     strata.name = c("North", "NW", "West Upper",
                                     "West Lower", "SW", "South"),
                     shape = shapefile.name)
plot(region)
# Make a coverage grid
cover <- make.coverage(region,
                       n.grid.points = 500)
# Define the design
design <- make.design(region = region,
                      transect.type = "line",
                      design = c("systematic", "systematic",
                                 "eszigzag", "systematic",
                                 "systematic", "eszigzagcom"),
                      line.length = 5000*1000, #5000km x 1000m (projection in m)
                      design.angle = c(160, 135, 170, 135, 50, 60),
                      edge.protocol = "minus",
                      truncation = 3000,
                      coverage.grid = cover)

# Create a single set of transects to check
survey <- generate.transects(design)
plot(region, survey, covered.area = TRUE)

# Warning! this will quite a long time to run as it is a complex example.
design <- run.coverage(design, reps = 500)
# Plot the coverage
plot(design)
# Display the design statistics
design
}

# Fast running example for CRAN testing purposes
# This spacing is too sparse to assess coverage in a real example and
# the number of repetitions is too low to assess design statistics
cover <- make.coverage(region,
                       n.grid.points = 50)
design <- make.design(region = region,
                      transect.type = "point",
                      design = "random",
                      samplers = 25,
                      design.angle = 45,
                      edge.protocol = "minus",
                      truncation = 3,
                      coverage.grid = cover)
survey <- generate.transects(design)
plot(region, survey, covered.area = TRUE)
design <- run.coverage(design, reps = 3)
plot(design)
design

}
\author{
Laura Marshall
}
