% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lb-keogh.R
\name{lb_keogh}
\alias{lb_keogh}
\title{Keogh's DTW lower bound}
\usage{
lb_keogh(x, y, window.size = NULL, norm = "L1", lower.env = NULL,
  upper.env = NULL, force.symmetry = FALSE, error.check = TRUE)
}
\arguments{
\item{x}{A time series (reference).}

\item{y}{A time series with the same length as \code{x} (query).}

\item{window.size}{Window size for envelop calculation. See details.}

\item{norm}{Vector norm. Either \code{"L1"} for Manhattan distance or \code{"L2"} for Euclidean.}

\item{lower.env}{Optionally, a pre-computed lower envelop for \strong{\code{y}} can be provided
(non-proxy version only).}

\item{upper.env}{Optionally, a pre-computed upper envelop for \strong{\code{y}} can be provided
(non-proxy version only).}

\item{force.symmetry}{If \code{TRUE}, a second lower bound is calculated by swapping \code{x} and
\code{y}, and whichever result has a \emph{higher} distance value is returned. The proxy
version can only work if a square matrix is obtained, but use carefully.}

\item{error.check}{Check data inconsistencies?}
}
\value{
A list with:
\itemize{
  \item \code{d}: The lower bound of the DTW distance.
  \item \code{upper.env}: The time series of \code{y}'s upper envelop.
  \item \code{lower.env}: The time series of \code{y}'s lower envelop.
}
}
\description{
This function calculates a lower bound (LB) on the Dynamic Time Warp (DTW) distance between two
time series. It uses a Sakoe-Chiba constraint.
}
\details{
The windowing constraint uses a centered window. The calculations expect a value in
\code{window.size} that represents the distance between the point considered and one of the edges
of the window. Therefore, if, for example, \code{window.size = 10}, the warping for an
observation \eqn{x_i} considers the points between \eqn{x_{i-10}} and \eqn{x_{i+10}}, resulting
in \code{10(2) + 1 = 21} observations falling within the window.

The reference time series should go in \code{x}, whereas the query time series should go in
\code{y}.
}
\note{
The lower bound is defined for time series of equal length only and is \strong{not} symmetric.

If you wish to calculate the lower bound between several time series, it would be better to use
the version registered with the \code{proxy} package, since it includes some small optimizations.
The convention mentioned above for references and queries still holds. See the examples.

The proxy version of \code{force.symmetry} should only be used when only \code{x} is provided or
both \code{x} and \code{y} are identical. It compares the lower and upper triangular of the
resulting distance matrix and forces symmetry in such a way that the tightest lower bound is
obtained.
}
\examples{

# Sample data
data(uciCT)

# Lower bound distance between two series
d.lbk <- lb_keogh(CharTraj[[1]], CharTraj[[2]], window.size = 20)$d

# Corresponding true DTW distance
d.dtw <- dtw(CharTraj[[1]], CharTraj[[2]],
             window.type = "slantedband", window.size = 20)$distance

d.lbk <= d.dtw

# Calculating the LB between several time series using the 'proxy' package
# (notice how both argments must be lists)
D.lbk <- proxy::dist(CharTraj[1], CharTraj[2:5], method = "LB_Keogh",
                     window.size = 20, norm = "L2")

# Corresponding true DTW distance
# (see dtwclust-package description for an explanation of DTW2)
D.dtw <- proxy::dist(CharTraj[1], CharTraj[2:5], method = "DTW2",
                     window.type = "slantedband", window.size = 20)

D.lbk <= D.dtw

}
\references{
Keogh E and Ratanamahatana CA (2005). ``Exact indexing of dynamic time warping.'' \emph{Knowledge
and information systems}, \strong{7}(3), pp. 358-386.
}

