if(interactive()){
  library(survival); library(dynamichazard); library(testthat)

  if(grepl("testthat$", getwd()))
    source("../../R/test_utils.R") else
      source("./R/test_utils.R")
}

# Had issues with win builder. Thus, these lines
test_name <- "boot"
cat("\nRunning", test_name, "\n")

test_that("Throws error when risk_set or data is not saved",{
  for(tmp in list(c(T, F),
                  c(F, T),
                  c(F, F))){
    fit <- ddhazard(
      formula = survival::Surv(stop, event) ~ group,
      data = head_neck_cancer, max_T = 40,
      by = 1, a_0 = c(0, 0),
      control = list(save_data = tmp[1], save_risk_set = tmp[2]))

    expect_error(ddhazard_boot(fit, 2),
                 regexp = "^Cannot bootstrap estimates when ddhazard has been")
  }
})

test_that("boot yields previously computed values with head_neck_cancer", {
  fit <- ddhazard(
    formula = survival::Surv(stop, event) ~ group,
    data = head_neck_cancer, max_T = 20,
    by = 1, Q_0 = diag(1000, 2), Q = diag(1e-2, 2),
    control = list(save_data = T, save_risk_set = T, eps = .01,
                   ridge_eps = 5e-3))

  set.seed(993)
  suppressWarnings(tmp <- ddhazard_boot(fit, do_sample_weights = F, do_stratify_with_event = F, R = 30))

  # plot(fit)
  # get_expect_equal(tmp, file = "tmp.txt")

  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c(-2.9417270151733201, -2.9417271494125758, -2.8751008812968308, -2.7944202142630039, -2.7218473374240366, -2.6454718499478842, -2.6548810638159783, -2.7347206446107339, -2.8005733099400301, -2.8613648474096909, -2.9213549135070527, -2.9859723293007789, -3.0339289593190788, -3.0675486510431185, -3.0755737654547737, -3.0985890531657061, -3.1251804112205708, -3.1437771523950744, -3.1430265035546250, -3.1483871572853541, -3.1376621000064713, 0.4404171871819423, 0.4404171192811067, 0.4736959887657881, 0.5200446109080480,  0.5521174112845471, 0.6022336481103565, 0.6068304383076504, 0.5726439210576278, 0.5629964410440592,  0.5469270404499240, 0.5294419316856761, 0.5071540820900921, 0.4977363624237865, 0.4999564634380535,  0.5146366934172963, 0.5098673080230457, 0.5031829300055697, 0.5042201332917474, 0.5139403746897799,  0.5206467192162000, 0.5345147175479759 ))

  expect_equal(unname(c(tmp$t)),
               c(-2.5652300641714802509, -2.7407427179497880765, -3.2990484299736690943, -3.0138480982085353865, -2.9525411260444300865, -2.8579896961567738956, -3.3280291977603746467, -2.9802399246285538048, -3.9107347541730366203, -3.2952975840541993691, -3.0029244117794067215, -3.0587475506128374292, -3.1709418628159391496, -2.8717271612754471377, -2.9854900314240584436, -3.4608133459788463782, -3.7764649066915954023, -3.2315002327524000592, -3.3005379690674381266, -3.1831192375560179997, -2.9785860443315574919, -2.8553032570234861964, -2.7400203765294435243, -3.0422650244955957710, -2.8880842609406172095, -2.9620668539104078043, -2.9681433722994681190, -2.9611947985380564496, -3.0753972364249260174, -2.6658162370159868004, -2.5652300254556275050, -2.7407431562053869989, -3.2990485221481415934, -3.0138481259701297432, -2.9525411226985620594, -2.8579897645543868023, -3.3280368530502086877, -2.9802400617945412620, -3.9107515730130524290, -3.2952979050775508263, -3.0029246922364927208, -3.0587478282609437130, -3.1709417543292879316, -2.8717276943902776409, -2.9854901080253997137, -3.4608133392505484593, -3.7764826903250190426, -3.2315000728647431671, -3.3005379962277654471, -3.1831194830328404244, -2.9785859755416703720, -2.8553037375634788830, -2.7400205641726680383, -3.0422655764775674569, -2.8880845010942208795, -2.9620672016564801865, -2.9681435138854457279, -2.9611947521440944442, -3.0753972560839897277, -2.6658164567612212537, -2.4934600559550217369, -2.6416317550864336638, -3.2251311194347356270, -2.9480569735341779491, -2.9105597018539972609, -2.8126936320723565643, -2.8271498451453571654, -2.8965483645823129066, -3.3235531547322412216, -3.2144753910878542946, -2.9151818787811407674, -2.9843138432090992040, -3.1376278655615132962, -2.7989089602665448986, -2.9224551933854554520, -3.3828494422067736380, -3.3534939506044407764, -3.1777726493601319291, -3.2425928490136497473, -3.1104895327531565385, -2.9080002854227253728, -2.7811196216625497790, -2.6578059515663885470, -2.9526284418350816630, -2.8071785878845307849, -2.8864456488931917733, -2.9232546453397043607, -2.9062163479919069786, -3.0120554471371461247, -2.5868370303805390265, -2.3935497163517176489, -2.5006833123695075827, -3.1535290993351408062, -2.8586561573798627833, -2.8280298800693457473, -2.7523158625118142595, -2.3216818842590201122, -2.8137340057068160881, -3.0209802143407511466, -3.1322897086938610300, -2.8290580185833973381, -2.8761991592456217504, -3.1093123198308068034, -2.6926779773554958552, -2.8721008501933011914, -3.2772257270740272617, -3.1800864391832184275, -3.1192714825050589234, -3.1618219906281765930, -3.0193726072781141845, -2.8541024078368177896, -2.6743123749827759639, -2.5270133209541989494, -2.8386960071782598725, -2.7037963627389780541, -2.8053927450637190510, -2.8531436133674534794, -2.8466520130696766344, -2.9919034821062444607, -2.5187490965009624588, -2.3369049943652440149, -2.4132207757385346625, -3.1191887507717024342, -2.7950778692097468792, -2.7617240837814800436, -2.7228786149564347951, -2.3423232698462408408, -2.7498467518313063351, -2.2680379698871928618, -3.0278393328371304705, -2.7812125032814942749, -2.8391403009764886178, -3.0807812088084247648, -2.5692479378398753731, -2.8176090881162481061, -3.2176409629691731240, -2.3716067570623584615, -3.0378773606224753756, -3.1310130683724195499, -2.9469936714203401884, -2.8224413159252450534, -2.5856317822950090068, -2.3816018820506865694, -2.7090106429925935316, -2.6127908509250312719, -2.7390760637312880910, -2.7458965897110583221, -2.8032001421417733766, -2.9368791717842270828, -2.4372303576811704673, -2.2857204152277272868, -2.3286912079855484059, -3.0598107215104075607, -2.7032348805124910562, -2.7138505463066135981, -2.6740233898231364762, -1.8670464804444903972, -2.6919564644792184360, -1.6642256288764056826, -2.9313242509571617589, -2.6958681607649266354, -2.7486349076258371760, -2.9852325497896621265, -2.4564883131039483288, -2.7147487692875982290, -3.1189842219117016597, -0.4683801029638561264, -2.9213831265653862346, -3.0375566493001961987, -2.8690838959447027001, -2.7832868890460797928, -2.5019112992137224616, -2.3672434403323663332, -2.5773933931011820953, -2.5667494979107021713, -2.6276497504697551300, -2.6285888888734025848, -2.7427986263458872251, -2.8682669959519735414, -2.3603548955755000449, -2.2670880941304454304, -2.3518196875484260744, -3.0440362798897160346, -2.6699123267969002704, -2.7036613893628040906, -2.6879112418619399705, -1.7221739390343049347, -2.7200129670405601701, -1.6219944619249746509, -2.9697682584682003615, -2.6440935673013861162, -2.6890991264339545452, -2.9576309223575898422, -2.4662881145358390711, -2.6295991867809105358, -3.1536220957724956371, -1.1586075447011536799, -2.9276257261393703324, -3.0512471869320139106, -2.8387250268897581407, -2.8124600613972079444, -2.5349287786463827743, -2.4156111874367085512, -2.5638641919194631313, -2.6395880384519556650, -2.5382379914909667384, -2.6137097575821628404, -2.7392082388147409588, -2.8638661977606045816, -2.3232128105630094339, -2.2997679368125116817, -2.5187010141883106051, -3.1254924416289027178, -2.7545017641166746891, -2.7332212194962917096, -2.7382849308851739956, -2.2585945949569135038, -2.8086669254041178867, -2.4347045674785854352, -3.0428555901722251775, -2.7226388799902689897, -2.7633064530715572005, -3.0652765015707155172, -2.5814814073049205945, -2.7001986164403093404, -3.2232563692090372420, -2.0672809079530716225, -3.0145495451486032934, -3.1269271997239780347, -2.9482957875071393872, -2.8705378349350301193, -2.6097472420731899767, -2.5259885533702428617, -2.6199242756443812397, -2.7188767375212790078, -2.6307411409419803050, -2.6531959839660079759, -2.7835998549295766225, -2.9190405025514385429, -2.4272551971066720355, -2.3451836881163261594, -2.6642806141684798327, -3.1871507880403697754, -2.8118549140061728231, -2.7717628135813292545, -2.7654183127491895888, -2.4690820560201780509, -2.8693826974230129956, -2.7111202031241252719, -3.0975199808671929347, -2.7949941513300577967, -2.8550288773157470601, -3.1632008378911091739, -2.6684015735236306099, -2.7766410642865322345, -3.2748558829752791866, -2.5236988274102785113, -3.0682461515652366124, -3.1919129106117276606, -3.0060397908318168270, -2.9326812861985600556, -2.6533404344284012844, -2.6058903965675872527, -2.6931220831053783726, -2.7578404806853846765, -2.6942096963270345533, -2.6535608936021302107, -2.8141047328095485369, -2.9618910155882218760, -2.5305983934039781857, -2.4166288060222775869, -2.7619999003986919206, -3.2626295791467856411, -2.8874522544651504141, -2.7687768363027664975, -2.7723224675978550557, -2.7685540063433213831, -2.9360947336610987435, -3.0255074782995223792, -3.1504935367402899793, -2.8756734250837787670, -2.9302933888410618302, -3.2420709086880048488, -2.7495854598612745612, -2.8057901270955611928, -3.3213652864105309703, -2.6573279283283759433, -3.1455923697365744118, -3.2434563848378679118, -3.0710334775414769304, -2.9965510667544261736, -2.6979468178191523542, -2.6834172106602638408, -2.8003324443069996796, -2.8016998806813351308, -2.7330456824500037882, -2.6909109686344496737, -2.8655655304280878148, -2.9757718366041925329, -2.6006034877643209313, -2.4959826709297843195, -2.8286763675778119342, -3.3433836868657587083, -2.9220126017946563834, -2.7833905150106423854, -2.8068094570631583196, -2.8210625992008488971,
                 -3.0023180281238106559, -3.2357353768424266249, -3.1895756130884467261, -2.9580335758465574614, -3.0073336900786928894, -3.2891392765042692936, -2.8177490271850285097, -2.8684313236745118125, -3.3940184650707112546, -2.2787398308120923041, -3.2260697017340733694, -3.2831237662925807363, -3.1646950102199586929, -3.0405671756619963020, -2.7360153684482400038, -2.7608887014416039207, -2.8797176047456272130, -2.8286058149635131542, -2.7889731295680162404, -2.7624089273186842597, -2.9045077012140425587, -2.9976676792911201019, -2.6395527252458297340, -2.5359457088192929319, -2.8704724697287198332, -3.3968613637526807736, -2.9684586403425186063, -2.8184815327595202916, -2.8406291859896608543, -3.0233267980043443401, -3.0475963373736738582, -3.6639606190542948205, -3.2645776218548818726, -3.0212888690025923033, -3.0789438658180889874, -3.3587241644950340813, -2.8625686130138787711, -2.9486285567366601335, -3.4433445841958350009, -2.5106957307878774444, -3.2759296528184669306, -3.3154101875053632043, -3.2579666023456690382, -3.1025297413682437764, -2.7854647589807117747, -2.8152022550815303248, -2.9617086275641462123, -2.8674154334156574109, -2.8769761626510117480, -2.8130253732078993600, -2.9721469420560269192, -3.0221924534997106626, -2.7063557826801258877, -2.5648099434201681035, -2.8874593005412330093, -3.4253447264383427573, -3.0309377386170739399, -2.8422506882845501863, -2.8649818856264972489, -2.9852932217269589898, -3.0854014314577185019, -3.8916011299054207306, -3.3231080134566775008, -3.0688227908534555510, -3.1221439506780725992, -3.4413899819170921113, -2.8853621497482224711, -3.0111269957605397707, -3.4585483434085553256, -2.7403232322488642048, -3.3207075725534584976, -3.3162989871248842455, -3.3271618651663850130, -3.1385189336694958584, -2.8374777230051289045, -2.8950299586290526577, -3.0231464681283850560, -2.8837385083110986805, -2.9348858489898268509, -2.8310885855907956810, -3.0252605814550159735, -3.0152285782760799293, -2.7507938092961246035, -2.5861803609461833098, -2.8648162849731870772, -3.4416820686538112106, -3.0784462169585240510, -2.8330606952721071323, -2.8597729048673934393, -3.0881242776539443717, -3.1302181848370969064, -3.8845210159138527928, -3.3533170939621541606, -3.0931288738071796018, -3.1384980303205618490, -3.5040631811048243982, -2.8908918178376032948, -3.0584962478280037779, -3.4544031289320442646, -2.9073968216014201005, -3.3503515252605553520, -3.3083647932121977853, -3.3858792011715870274, -3.1610761034442487194, -2.8720715688937668020, -2.9469883684015667491, -3.0519240461307428625, -2.9049534717434748465, -2.9893751670251487162, -2.8298985396604647846, -3.0651392035776785505, -2.9878563386880871988, -2.7886966782303028722, -2.5757550776750868238, -2.8020829397723407972, -3.4349468844071928686, -3.0996604737816286779, -2.7906114966461066373, -2.8247131848589064340, -2.9772756457980982248, -3.1484390449664987344, -3.6275716479739856801, -3.3553836101324154839, -3.0943084675974161790, -3.1280229103080303688, -3.5471064033529846604, -2.8665806689049948552, -3.0795302650615905549, -3.4172231318833414804, -1.9748360062769583223, -3.3535674501177541451, -3.2693808710990159838, -3.4231562041904983751, -3.1593018650135014092, -2.8776822946522049307, -2.9714244256764397178, -3.0478988680564707003, -2.8969593643478677158, -3.0167477874731414289, -2.7980610757799255772, -3.0811168716806469625, -2.9289092179531470705, -2.7944680258866116951, -2.6259432370107544941, -2.7741150175757671725, -3.4320822269824287076, -3.1083820114223263964, -2.7740974886141422928, -2.8070564009949157125, -2.9926618379522866853, -3.1959706792262587705, -3.4324366616170189914, -3.3776605628765836720, -3.1154717349610216992, -3.1337218893181213453, -3.5993347325026028294, -2.9025001574569579965, -3.1162206337391142874, -3.4429622370736501225, -2.6251710522758040511, -3.3717798620488808048, -3.2516071078379029657, -3.4393808083508714546, -3.1704984354265577373, -2.9233556158931843783, -3.0204605955760195357, -3.0883617260998712517, -2.9273606255439341695, -3.0761555046003175029, -2.7628818836929625036, -3.1114412525686612199, -2.9011392242194364854, -2.7842317534532656786, -2.6510603885647552325, -2.7890732508204991724, -3.4554363006453656659, -3.1470643722907167827, -2.8087268677979904119, -2.8074692278709330040, -3.1061243538103431305, -3.2347402877953679301, -2.8915082528995905697, -3.4020070755070253021, -3.1448142776935545939, -3.1409387453766441922, -3.6351642925849421140, -2.9451212013393450739, -3.1431471834211603955, -3.4894306367262419855, -3.2709240098373633110, -3.3924250233677484268, -3.2652084909904606214, -3.4619654987981030025, -3.2077756193442454880, -2.9615062783099723731, -3.0643160515881642247, -3.1330375868183697463, -2.9587993418017912717, -3.1290284292246037801, -2.7476696010964110073, -3.1560687354241339087, -2.8690437120714191721, -2.7702047749936506982, -2.6510389258412119240, -2.7855292878007471558, -3.4716178104564483498, -3.1781394293611331392, -2.8264929139668604208, -2.8044848471091117226, -3.3201642166929934419, -3.2536213693287945325, -3.1719736394888400355, -3.4170344328114126142, -3.1953086212640724462, -3.1509246904525465283, -3.6549052649787894254, -2.9688774467284937408, -3.1612514870047268012, -3.5147193072618976295, -3.6927341917073972510, -3.4033072213881947476, -3.2526954975021844518, -3.4778616175194216886, -3.2380367133273750824, -2.9924140761877526806, -3.0872528253251285690, -3.1558219970215741768, -2.9810471141066834377, -3.1636870051042360608, -2.7534060016603625343, -3.1945958581462505776, -2.8541864213745213874, -2.7395929875614095650, -2.6257318395995037363, -2.7482815381577951896, -3.4696586673244338428, -3.1902809221719392596, -2.8164558886674444516, -2.7767340472978094823, -3.4317099916258775316, -3.2527204450667936975, -3.1966054299100474800, -3.4104323499717921209, -3.2322450714136046201, -3.1401219769457484610, -3.6587897327476630593, -2.9739171850049177515, -3.1589890875299504458, -3.5189103173204530073, -3.8611658165772899665, -3.3930630330123086047, -3.2138769186549596135, -3.4754421252436311640, -3.2504962291401122698, -3.0046949207365019063, -3.0894548507363537304, -3.1568006108819481881, -2.9827793161806175348, -3.1803297695522800836, -2.7346462175599373090, -3.2163261114796544149, -2.8133659126654451121, -2.6786851918929257543, -2.6282648118041671559, -2.7278759564664394333, -3.4609107330622932430, -3.1956825789411111849, -2.7784976339096520981, -2.7593820827778690052, -3.5669570015035367838, -3.2439322505031999988, -3.2999781255942011171, -3.4396251311750316582, -3.2558250115925755352, -3.1350836514987334702, -3.6838956916562151100, -3.0015520607284749133, -3.1760085758398988176, -3.5485120344826093230, -3.7511587802918016088, -3.4008720301365165106, -3.2261392635077119628, -3.4668221980718318953, -3.2452157777067132116, -3.0107332029575344379, -3.1012067789896966552, -3.1647025182082280104, -3.0032734051159368782, -3.1937391688037806148, -2.7292649996885103292, -3.2213895641395855485, -2.8041434671374632082, -2.6494490785921414044, -2.6092554271373251673, -2.6794778862850145984, -3.4366472577560380763, -3.1847182238794018438, -2.7169281346073139716, -2.7205820477192648177, -3.6338664472725938914, -3.2193130553912454594, -3.0112880975621258983, -3.4541520805745653533, -3.2675630234576633626, -3.1111596441652711320, -3.6963699413821133888, -3.0153180937380734683,
                 -3.1776013463772501133, -3.5632337838053018686, -3.3911276640376462055, -3.3909744510595989375, -3.2187352368498163280, -3.4409003374928914809, -3.2241890936821753222, -3.0000309655015824362, -3.0985399839976190961, -3.1531778891737216597, -3.0066899278321734812, -3.1931304004400091756, -2.7060698530079401181, -3.2116071670654378067, -2.7739822364181159209, -2.5947108932860709629, -0.1265769806826728960, 0.2049006905030757164,  0.9190067058765167829, 0.5468005436176046619, -0.0822243820186379648, 0.2425091086992939016, -0.0728220661082566861, 0.4823490678599188786, 0.1471834968816877753, 0.7334563850975585053,  0.6319384187794150343, 0.6904582508265754148, 0.6878992046408594074, 0.3036356534552105968,  0.3757927150397460392, 0.9083028712082552580, -0.3817696734241763368, 0.6448236878752631185,  0.7652538778082437210, 0.7171860337363108462, 0.4183242231842630443, 0.4479569020968171000,  0.0911228164580402816, 0.5771736157615454665, 0.3608621464710363158, 0.5594519632791369323,  0.2871527210174751876, 0.3119248535025420099, 0.6314203254558762302, -0.0051838052882720812, -0.1265771073055881601, 0.2049003872631007805, 0.9190063659050968781, 0.5468003105085663673, -0.0822243518529705009, 0.2425091787169921798, -0.0728249861401757820, 0.4823488834028533123,  0.1471781082042616373, 0.7334562575942548124, 0.6319380493807580379, 0.6904578387909237058,  0.6878990608764955494, 0.3036355650014470053, 0.3757926530749289840, 0.9083026412564582097, -0.3817765630340410610, 0.6448235138961115887, 0.7652537723363527222, 0.7171856390829833838,  0.4183242313367389054, 0.4479568027623596627, 0.0911226167220683947, 0.5771730625302887718,  0.3608621593183624632, 0.5594516862157394854, 0.2871526988397087310, 0.3119248406598204015,  0.6314202417525528466, -0.0051839592837938930, -0.0985362834293646850, 0.2559243480027129647,  0.9643087996403986173, 0.5794885672820513367, -0.0657171253442475067, 0.2597560770511904149,  0.1250907835471125329, 0.5343358744053767762, 0.3338948021550455225, 0.7835241127365863978,  0.6808519777183850152, 0.7327794987710553043, 0.6978985149396186305, 0.3420939179200925828,  0.4082729336953018762, 0.9554868243163519015, -0.2226115880901383959, 0.6721910642592519025,  0.8020885785645629484, 0.7630536389385020879, 0.4546327522423226908, 0.4840429593145237352,  0.1183138680740163096, 0.6279323686196925935, 0.4057483523330215336, 0.5973889808125198986,  0.3070018369534349145, 0.3356057083610674008, 0.6685850350224319394, 0.0311046886028674696, -0.0502547637000890779, 0.3302332686434213405, 1.0187011290666962982, 0.6256307239221663430, -0.0268782169952309080, 0.2986221431953181016, 0.3225833113431393340, 0.5893618307786474553,  0.4252484789385704289, 0.8332607858637244291, 0.7231306377055051193, 0.8052254340656785914,  0.7211935963899108737, 0.4163402689752640495, 0.4416670822301319577, 1.0230805788218360064, -0.1561499396941981033, 0.7072756233527360825, 0.8642742440649316960, 0.8295170064905226237,  0.4863586388615174227, 0.5457206214008664125, 0.1739704445985187364, 0.6928426194494822221,  0.4701055041246272048, 0.6412913134196531750, 0.3511016567568672753, 0.3570800371688533925,  0.6811724654496282838, 0.0638321306568029601, -0.0492267065038353435, 0.3724676401158170114,  1.0361609603031214366, 0.6458653365007900327, 0.0014365443136744871, 0.3005745449970350647,  0.2957579393781011312, 0.6224666040294684066, 0.6551842538005054895, 0.8909718048849509797,  0.7514509541904701440, 0.8225601356096989836, 0.7289251844774478073, 0.4940460036211751227,  0.4624525175729968796, 1.0472664301121237251, 0.1482824755694556185, 0.7625553832372781171,  0.8891784096815547045, 0.8847557420893124025, 0.4818664231225102679, 0.5833741622117415426,  0.2257823880906871805, 0.7645063621761833961, 0.5082514560722353192, 0.6737409125570181789,  0.4193103070223701545, 0.3593398507485268123, 0.7077602559046880515, 0.1007849616407449078, -0.0210505401562087885, 0.4392488746096057661, 1.0751577262798819312, 0.6963960662386311506,  0.0309411316833658745, 0.3254837200574871581, 0.4697802684105908400, 0.6993535424895267472,  0.8738928448406505511, 0.9537425562465632334, 0.8138098630406602751, 0.8899471237774951859,  0.7747035111262235985, 0.5935067616974496474, 0.5145510265105156522, 1.1065667248958630520,  0.8889736610820853091, 0.8385374509044677405, 0.9623747005953693368, 0.9472240641869287270,  0.4983000949474839514, 0.6223816457808293290, 0.2593820371854719253, 0.8461404131891211433,  0.5332080792473714581, 0.7302626121981232732, 0.4891100417976289028, 0.3904428082370142716,  0.7637431688018747611, 0.1600339863226590276, -0.0074378691850686685, 0.4293171487977593959,  1.0797964292129156316, 0.7308484340776826294, 0.0428898691509931665, 0.3200176616177011213,  0.5247256796620882735, 0.6941243485809457381, 0.9145594023518661686, 0.9367416448968550746,  0.8545936151799254921, 0.9482447827470613655, 0.7766055488730235767, 0.6001642459059265411,  0.5542481876730406265, 1.1073412749861188953, 0.6248416920168022859, 0.8556148412265740522,  0.9496128004107126097, 0.9823211003821896714, 0.4741651039358709685, 0.6060479904306830257,  0.2593992490079577440, 0.8517911045954054128, 0.5010596508962898987, 0.7761742890874183676,  0.5148127930071978442, 0.3896080820785863263, 0.7674624854948748354, 0.1958071291056372454, -0.0010357641140773532, 0.3362226322854485749, 1.0394579633985425726, 0.6957002829747216399,  0.0276979179415037802, 0.3005353764520863891, 0.3184509978222714310, 0.6420423172859452965,  0.6715636192232561363, 0.8888811856189273719, 0.8248431360329678830, 0.9086384595525556218,  0.7129635677205025113, 0.5480052583436429980, 0.5131154439078078333, 1.0790397554307857853,  0.2812659007310706616, 0.8198577676325011776, 0.8976753361290219502, 0.9258747405356397842,  0.4514739855018092829, 0.5795954613350928764, 0.2215813869556123306, 0.8171094470823849765,  0.4673855653999668402, 0.7387893351212565518, 0.5048940410219432406, 0.3728523878928829416,  0.7234894770308151202, 0.1293952529380480720, 0.0365252191735989107, 0.2634295247169728960,  1.0219455749508787878, 0.6867249030064757065, 0.0330949081940429268, 0.3046281995231330253,  0.2518276366721654558, 0.6179755208870845173, 0.6067772237498945120, 0.8678392865839013037,  0.8214712793450538175, 0.8936826953111689509, 0.6636461654141732058, 0.5263578643117010625,  0.4969021299301563110, 1.0806765065028745010, 0.1271345119043393312, 0.8091963709990687148,  0.8707955204049568909, 0.9039855560145841906, 0.4479228711563455478, 0.5811328935099335791,  0.2048217836983370621, 0.8130031279548866063, 0.4613960980246933907, 0.7292219928767973425,  0.5314625360763565176, 0.3777292324934961032, 0.7029123232132729049, 0.0802782897471503859,  0.0428429722782585620, 0.2153631381313257931, 0.9835061015560769837, 0.6509623772341327497,  0.0618022302525854106, 0.3128666203828726067, 0.1368406233877759726, 0.5781485865409432545,  0.5146450534742703642, 0.8380023068262671959, 0.8017278373224143717, 0.8847380504730962825,  0.6200845905354404053, 0.5009781974380549396, 0.5091626086831552156, 1.0778420880032415496,  0.0900741551313576083, 0.7688029143785445330, 0.8478046598819770630, 0.8675132756194248174,  0.4339413397374251580,
                 0.5687077547151926371, 0.1749867324941983271, 0.7746406570451778162,  0.4382517897536227225, 0.7264454032529876759, 0.5149833831134783546, 0.3522395320319981482,  0.6958233602635689463, 0.0435477839794370658, 0.0321562970575497986, 0.1880827131086126314,  0.9480387955765006147, 0.6349613841068741937, 0.0793279183031843083, 0.3107635474822866839,  0.1231232621329553523, 0.5368982390082976952, 0.4619405708050382531, 0.8169872864864538009,  0.7791261736125439930, 0.8659371045696825364, 0.5920373101900380242, 0.4776332990287799340,  0.4938061711140156462, 1.0726406225073823464, 0.2511603788079159183, 0.7325047088486099645,  0.8355626607873853917, 0.8164581348805277639, 0.4263484820943204134, 0.5568559485242875473,  0.1566928904261217781, 0.7483496875654866987, 0.4241721608920052122, 0.7329002849320607194,  0.4667581804774425525, 0.3405230816548510120, 0.6883842853922099714, 0.0173948832590533203,  0.0419261960424879843, 0.1696031252714773063, 0.9265698918231846104, 0.6142584123856182909,  0.0937774035847300791, 0.3069807972379855476, 0.0387734884142730551, 0.5033329716839681778,  0.3284605018228609619, 0.7749834700619850913, 0.7620653663202673345, 0.8430080511251052400,  0.5535625730089233087, 0.4646188132679907046, 0.4557147230461863208, 1.0785074598831287940,  0.1660478825716210083, 0.7118874248972855145, 0.8222678996145154073, 0.7590859077362750451,  0.4120330692957130392, 0.5411039072587784382, 0.1424990773056009075, 0.7280098567381080299,  0.4053819014486563788, 0.7061514918069584645, 0.4246252199396947646, 0.3109730535405486607,  0.6691238360051011691, -0.0097834809613421003, 0.0692130545997137192, 0.1682392721803301905,  0.9171817857136851382, 0.6042219328813833723, 0.1260904284714920998, 0.3185940264189461857,  0.0475455439045686726, 0.4837162168753363356, 0.2576802665512004098, 0.7460436867419659723,  0.7565426526634557725, 0.8343416420486858831, 0.5217838343932519329, 0.4695938788105688544,  0.4313550631810342817, 1.1030637162870873258, 0.0939915247438506962, 0.7032881504141115236,  0.8280948144430866353, 0.7148115781564456928, 0.4104371113629439183, 0.5387306852372870658,  0.1340921601291078513, 0.7242726372477895636, 0.4022092697238732750, 0.6937248489961677089,  0.3985040967884270202, 0.2915070726160527426, 0.6666364283340506791, -0.0207845724424363110,  0.1128409259782612306, 0.1870537706739526240, 0.9176873238788774145, 0.6063481581740277626,  0.1776986664117488068, 0.3461448352125386929, 0.0108685945437293346, 0.4758502950470122617,  0.2606683299001626430, 0.7317802140384802589, 0.7604311862219051976, 0.8386428952965994110,  0.4997619326375800641, 0.4892806043209227962, 0.4194179401180480049, 1.1433385774820283842,  0.0544968836039653737, 0.7074258391115286759, 0.8505933034036085028, 0.6812207963350151996,  0.4199738220117291143, 0.5502027622455880618, 0.1363332396258470358, 0.7382069228560879104,  0.4103174326106724390, 0.6925996772467951645, 0.3862596151323612337, 0.2817665255247289169,  0.6801411542848456726, -0.0187044729707632196, 0.1737419697082950298, 0.2262601234156915164,  0.9291289755140255124, 0.6214274482287585055, 0.2482432502362855731, 0.3896142495984482590,  0.0574046176139216963, 0.4833008046961556614, 0.3422853876195570688, 0.7321024477535760244,  0.7736230998598918607, 0.8558529730311281725, 0.4872239798349555784, 0.5254452088015895139,  0.4207023348545817498, 1.2011631653938645847, 0.4414467915855763147, 0.7253641294911592485,  0.8923857701944176712, 0.6594302075866710355, 0.4412332632265478227, 0.5763849355270135311,  0.1489636553180084533, 0.7696730273883219953, 0.4330354066905769406, 0.7047737391351004410,  0.3892532078724468270, 0.2823495067080179499, 0.7108484161277428548, 0.0006623360572635684,  0.1788986359032556372, 0.2314019955298125819, 0.9297262963240388034, 0.6385039639616703067,  0.2858218904046860853, 0.4085991170142459494, 0.0455917655982459868, 0.4603609658848000774,  0.3988535573332164796, 0.7096723597938503492, 0.7627105316007909419, 0.8518822703140760400,  0.4617321465884402509, 0.5066587504091256688, 0.4026053472891742979, 1.2038338550816005679,  0.1973045235958039711, 0.7239440324773419588, 0.9095600753200846489, 0.6492366181407507320,  0.4442984528424104851, 0.5632849726957531722, 0.1373775283459249819, 0.7599602776866516418,  0.4163665922881462644, 0.6856447668443041987, 0.3851475336147880957, 0.2626992953335353098,  0.7065082448459469866, 0.0243324151991070603, 0.1964502570495942901, 0.2282302030957586436,  0.9252546698697724503, 0.6243508886903087474, 0.2865539293972360624, 0.4197281110016936045, -0.0063025728341738474, 0.4469843003309637020, 0.5642420526378727486, 0.6943422516538585443,  0.7453640103344046919, 0.8550361766454104284, 0.4435853479590147197, 0.4841555994675371122,  0.3934892876706090337, 1.1919750784349012473, -0.0465753835476583056, 0.7215153976070722841,  0.9180478983707562879, 0.6367401553082100429, 0.4356041644403390034, 0.5484971866885612890,  0.1218083933655835505, 0.7484295764145364460, 0.4073271914768710200, 0.6733260218364780503,  0.3797378085118869873, 0.2482926842659293198, 0.7044797323900755526, 0.0597491141434794981,  0.2262972231032616688, 0.2391123608699483316, 0.9278717337263072684, 0.6171685598148006768,  0.3021138562527317384, 0.4423055384482092456, -0.0914086836452215473, 0.4442992605418059071,  0.4810311787030531194, 0.6877936061342945218, 0.7288878939483054431, 0.8651339897407714918,  0.4326577403220896456, 0.4703250184698821723, 0.3931012615581573888, 1.1893949811101909297, -0.2046505150122484296, 0.7282345260404210485, 0.9423385893699470772, 0.6327971924403141823,  0.4343591064986509487, 0.5405337214812160340, 0.1129395115739605382, 0.7470792202576348462,  0.4067332178391331876, 0.6687538235831570477, 0.3820103026832545323, 0.2403039819675418520,  0.7117976672952250894, 0.1093736901935989936, 0.2683920461556221770, 0.2673150939171480278,  0.9386549773137646824, 0.6179184946072924234, 0.3333117823542555702, 0.4779280124465282720, -0.1371830794566830936, 0.4522407916987863929, 0.4782408530550537051, 0.6920064974231525667,  0.7166891283276495273, 0.8848303599988411872, 0.4288627625134636134, 0.4651139598846908974,  0.4025058087572862542, 1.1960363666513516012, -0.2651641056553342146, 0.7453192260763479293,  0.9824788862952213986, 0.6390189854612399190, 0.4411640978689643222, 0.5403197848437543138,  0.1107234352418353107, 0.7558493513225730265, 0.4156399931727464692, 0.6718162346341531865,  0.3971394800298718164, 0.2393787825218749721, 0.7324455253291939361, 0.1757686308286958154,  0.2808054531467217840, 0.2922306366218790918, 0.9564026862340182511, 0.6253847878546310302,  0.3800515760859862757, 0.5052751862766828861, -0.1893632130673357783, 0.4693401038326643904,  0.4563021670028560850, 0.6780759983267755642, 0.7086939001249058512, 0.9013839734205386645,  0.4185906833764717616, 0.4527379026034424325, 0.3991507597945093688, 1.1851487807000198327, -0.2183223927577809254, 0.7495017198956739346, 0.9817563321160455159, 0.6536057896773834752,  0.4559605111561428892, 0.5466653579974667387, 0.1113167138271281831, 0.7612420551423320081,  0.4111322798283194579, 0.6714190990788795332, 0.4111322946222571439, 0.2454424408503068644,
                 0.7519797265133529862, 0.2093799506858945447, 0.3029826582515715572, 0.3324365939879594389,  0.9813632962910023849, 0.6398120677457460603, 0.4404674976295580580, 0.5445923064902353783, -0.2151891660000859186, 0.4944169544474770195, 0.5587951066395042687, 0.6711378908496363760,  0.7047107168218004203, 0.9266170057227314238, 0.4134838878744800383, 0.4465705002879763463,  0.4028290857423946214, 1.1797282813568117366, -0.0750171739472248467, 0.7626053756596713473,  0.9927290304683636935, 0.6782390966746114636, 0.4780633681382663869, 0.5605363439650613566,  0.1172709231651102024, 0.7756552758382060375, 0.4143121298954660081, 0.6767106040066097616,  0.4349745441352929065, 0.2583139357891865995, 0.7830463779299368099, 0.2581644839327729701 ))

  set.seed(994)
  suppressWarnings(tmp <- ddhazard_boot(fit, do_sample_weights = T, do_stratify_with_event = F, R = 30))

  # plot(fit)
  # get_expect_equal(tmp, file = "tmp.txt")
  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c(-2.9417270151733201, -2.9417271494125758, -2.8751008812968308, -2.7944202142630039, -2.7218473374240366, -2.6454718499478842, -2.6548810638159783, -2.7347206446107339, -2.8005733099400301, -2.8613648474096909, -2.9213549135070527, -2.9859723293007789, -3.0339289593190788, -3.0675486510431185, -3.0755737654547737, -3.0985890531657061, -3.1251804112205708, -3.1437771523950744, -3.1430265035546250, -3.1483871572853541, -3.1376621000064713, 0.4404171871819423, 0.4404171192811067, 0.4736959887657881, 0.5200446109080480,  0.5521174112845471, 0.6022336481103565, 0.6068304383076504, 0.5726439210576278, 0.5629964410440592,  0.5469270404499240, 0.5294419316856761, 0.5071540820900921, 0.4977363624237865, 0.4999564634380535,  0.5146366934172963, 0.5098673080230457, 0.5031829300055697, 0.5042201332917474, 0.5139403746897799,  0.5206467192162000, 0.5345147175479759 ))

  expect_equal(unname(c(tmp$t)),
               c(-2.9396013299254866, -2.7989642092648177, -3.0194920004177619, -2.8934397173815980, -3.0448815479259927, -2.9504134984001569, -3.0648943449859209, -3.0863051390228149, -2.8710596678488036, -3.0153097704021214, -2.9154178059457903, -2.9672120853455177, -2.9993264517650737, -3.0812614913745136, -2.9603018091716171, -2.8383812385542466, -2.8855495698796703, -2.9058822032590492, -3.1883903186124880, -3.0690426119672347, -2.8864923046970574, -2.9366143500409980, -2.9669739063487381, -2.8560627590978442, -3.0523902020492426, -3.1169088017999087, -3.0200466824103476, -2.8838592368793732, -2.8210004344859829, -3.0035438747907706, -2.9396014923518630, -2.7989642897883691, -3.0194923898294839, -2.8934397860568013, -3.0448815471076012, -2.9504137339653784, -3.0648943998260396, -3.0863052097835189, -2.8710599747696097, -3.0153097040333274, -2.9154182726140347, -2.9672122160180505, -2.9993265691017412, -3.0812614449241837, -2.9603020961035575, -2.8383815790141309, -2.8855498402572035, -2.9058824401894991, -3.1883907766957087, -3.0690424665219220, -2.8864928605186009, -2.9366145348177515, -2.9669740136726599, -2.8560629157344128, -3.0523901423155526, -3.1169088771754057, -3.0200466451505439, -2.8838595829223364, -2.8210006749287153, -3.0035441098213838, -2.8722235295167926, -2.7311724164392706, -2.9409890714740534, -2.8236969295725691, -2.9749723894771032, -2.8802237497694190, -2.9830098820687234, -3.0106933346479239, -2.7963265263308763, -2.9400845434379561, -2.8368451121563840, -2.9104142422490971, -2.9191097749392831, -3.0227567745002930, -2.8892130244874927, -2.7737324492564266, -2.8227378287615794, -2.8381875051855343, -3.0822464980117203, -2.9982990265860248, -2.8167498012352250, -2.8785761203762776, -2.9020526710779455, -2.7882478889752438, -2.9786715364492093, -3.0357280381383731, -2.9664896251067043, -2.8225902926883726, -2.7632082757640299, -2.9357056565709234, -2.8032688437736777, -2.6456903239930210, -2.8384110893421330, -2.7658577862171514, -2.8987024361645006, -2.7743207201440958, -2.8889074248051028, -2.8987930090688971, -2.6965668972219805, -2.8552963797161963, -2.7370390861844953, -2.8260012104240122, -2.8016092014117793, -2.9535513931512867, -2.7880591163054094, -2.6879772336687138, -2.7196771917429312, -2.7439229116893902, -2.9601897143453724, -2.9172919699948898, -2.6937210548923414, -2.7938731742335046, -2.7992599321118457, -2.7173885430184233, -2.8957151295734542, -2.9455257189527515, -2.9056222443129767, -2.7254243618586091, -2.7006271468553344, -2.8499915238330438, -2.7362006197110471, -2.5764841360651047, -2.7336828767716916, -2.7061439426470160, -2.8454403158418984, -2.6534073477333857, -2.8185034580980162, -2.8004888371799161, -2.6151462029792474, -2.7808617095864605, -2.6394540740121419, -2.7689490702427495, -2.7192937674404409, -2.9191679987587857, -2.6893234529460797, -2.6208688475666913, -2.6336209186411228, -2.6775625088527359, -2.8762795545898858, -2.8551027541955993, -2.5894423100432356, -2.7045254522321782, -2.7253533841992232, -2.6728344439937923, -2.8239891450203558, -2.8775051322441527, -2.8512464949584544, -2.6379009995047018, -2.6468950914287137, -2.7412535852378790, -2.6251207376543300, -2.4780212267068276, -2.6228241328201238, -2.6277738756219513, -2.7812126879825740, -2.5229106572862499, -2.7440316306132431, -2.7032682897534341, -2.5198389281818656, -2.7469579939209661, -2.5214218936368695, -2.6781350858105046, -2.6522090475917879, -2.8677196790691752, -2.5859835896277827, -2.5413600187793297, -2.5694559326095536, -2.5964598060552100, -2.7704941422738223, -2.7832512562900269, -2.4983518251269761, -2.6189983798881316, -2.6530687099498542, -2.6195136231873501, -2.7583524052679502, -2.8221058635486349, -2.8065151688202792, -2.5506331313251036, -2.5805764833414133, -2.6027970493911794, -2.6112444385629292, -2.4812579481009385, -2.5979513605501161, -2.6622495224213019, -2.7963714830034414, -2.5114295951816059, -2.7773255185388543, -2.7082084860363329, -2.5440805358519936, -2.7846786631038247, -2.5177814133850007, -2.6771289828650024, -2.6722404569298734, -2.8816318419075313, -2.5855694044648128, -2.5405574867004566, -2.5631083590271837, -2.6263382932162944, -2.7657858309629924, -2.8035899033918694, -2.5005542754015964, -2.6411668430547066, -2.6591762808433224, -2.6344112060798959, -2.7763890593747975, -2.8095078102357300, -2.8358967363664749, -2.5264558521040659, -2.5959080028161856, -2.5785458480268262, -2.6967111068880483, -2.5478586519580428, -2.6860125391781478, -2.7372395361960926, -2.8574061214035504, -2.5875555747369932, -2.8724699732757042, -2.7688565158799570, -2.6129145531687668, -2.8874058152164688, -2.6170994503689000, -2.7542444066634117, -2.7824718547493736, -2.9704100367270714, -2.6640564713441197, -2.6107658956822144, -2.6166563989019975, -2.7093467551212105, -2.8771905004267024, -2.8827332048265184, -2.5983697856627646, -2.7047894049723311, -2.7283721950713629, -2.7119267571205627, -2.8567039834022241, -2.8860332109953202, -2.8978895826378084, -2.6459689475687442, -2.6723196207734841, -2.6464828128004436, -2.7837456573713095, -2.5953369291884214, -2.7483583570053778, -2.8158494099902627, -2.9044607198015790, -2.6671037394816270, -2.9546519518023100, -2.8238097465848302, -2.6738046829961348, -2.9769361780287507, -2.7054206282630284, -2.8150575089190930, -2.8647177043055780, -3.0333078972228646, -2.7200554759664080, -2.6728442113791231, -2.6710889079167073, -2.7794902740131420, -2.9515424675273971, -2.9573159759721284, -2.6698395572363371, -2.7526866764311824, -2.7940132355134701, -2.7657927234690916, -2.9246082810846170, -2.9446845239297557, -2.9605862916702033, -2.7329756017767513, -2.7381456329711216, -2.7046835638321127, -2.8703811935415104, -2.6666783664471128, -2.8162613000010608, -2.8763241657468677, -2.9646485697666032, -2.7308065091742435, -3.0252951218232873, -2.8643586442595774, -2.7363012565082543, -3.0404215831859411, -2.7812733326934320, -2.8660094291408535, -2.9369112290431394, -3.0908804180204683, -2.7959927357962497, -2.7299001608831266, -2.7368409271547480, -2.8565458703610265, -3.0068788201843999, -3.0242819533960361, -2.7474065498872879, -2.7974626447704294, -2.8556346725205413, -2.8019268120215042, -3.0008581728878214, -3.0131999677072510, -3.0076866923081131, -2.8129079634612397, -2.8052840454146106, -2.7654704195608257, -2.9532609905207425, -2.7221084346619251, -2.8944672500109556, -2.9310235448101438, -3.0165833931266808, -2.7806713228837463, -3.0716012295409758,
                 -2.9134177225986084, -2.7885388252386947, -3.0919948540802396, -2.8355949718077111, -2.9324809128422524, -2.9818129726749967, -3.1426949379830682, -2.8614750145466243, -2.7785725660674157, -2.7956767315088440, -2.9102008408339790, -3.0871047299128054, -3.0834670965121926, -2.8285692742449982, -2.8380802895891390, -2.9167906568709743, -2.8381831864994824, -3.0568239835715967, -3.0859893187119543, -3.0470221701735345, -2.8778479702419784, -2.8772882227744434, -2.8475620664578885, -3.0397858068409085, -2.7769338772099932, -2.9697639691357800, -2.9834011146344559, -3.0963022933560995, -2.8464098125972144, -3.1147600231637789, -2.9717246778592568, -2.8365369908536180, -3.1440302373601456, -2.9022974430296582, -3.0017829237603357, -3.0227592364119786, -3.1967351672052553, -2.9417171029819693, -2.8382001523956943, -2.8563312321846195, -2.9730183707363516, -3.1731092568755157, -3.1429490765904715, -2.9029567743199061, -2.8898510940878546, -2.9804376323971917, -2.9037062046642479, -3.1173928962979289, -3.1504990468517668, -3.0952109796256817, -2.9455975874839990, -2.9419372084303239, -2.9379548989206108, -3.1159782974930237, -2.8095791602242230, -3.0297335474953333, -3.0284068071480355, -3.1522347908904966, -2.8919904342685876, -3.1311805137311657, -3.0079063830624637, -2.8672420335503799, -3.1699079370155343, -2.9456854636250629, -3.0566792596385728, -3.0569397491897337, -3.2390828514071046, -3.0073970526756306, -2.8814703609133732, -2.9103594986671539, -3.0184434236440523, -3.2419808015376277, -3.1916893023649693, -2.9661602776613658, -2.9361859655593694, -3.0248631138977795, -2.9553183735786379, -3.1570667340112424, -3.1934495085452288, -3.1379614797035438, -2.9908950889286712, -2.9877321203174341, -3.0139410494823125, -3.1745519987140982, -2.8299079007498769, -3.0655937356985907, -3.0584093717771617, -3.1887212150097941, -2.9248833779125483, -3.1244011541013537, -3.0337087604103603, -2.8842925244756876, -3.1863457678419032, -2.9810432318951858, -3.1068787203698762, -3.0804523532269559, -3.2607446607813473, -3.0469096806750113, -2.9182286134891844, -2.9491714970121889, -3.0527430036535717, -3.2906167638759878, -3.2168939052892016, -3.0115285132866445, -2.9805432671234526, -3.0535858150051634, -2.9993719209508747, -3.1766778723295639, -3.2267555463585924, -3.1629253429973279, -3.0333541698115685, -3.0322179500200805, -3.0663781305857096, -3.2117326274016742, -2.8206614032858783, -3.0756039921111631, -3.0620942750991662, -3.2024175721730388, -2.9257939265151913, -3.0874720408440126, -3.0298216302841081, -2.8718933298091818, -3.1756986346936245, -2.9887012633076382, -3.1330042388161261, -3.0769312488797729, -3.2583012142637346, -3.0604604553460533, -2.9277308879108457, -2.9627859959215286, -3.0609080358880529, -3.3105795135603047, -3.2171184572146854, -3.0301887811962147, -3.0009634515678991, -3.0554649551706379, -3.0196904613152507, -3.1702917956956100, -3.2357792551401623, -3.1633031694699891, -3.0499436238901989, -3.0556711390807867, -3.0919348025519797, -3.2488372638967022, -2.8420332953184095, -3.0826473817460149, -3.0966188186341572, -3.2148812664166133, -2.9658503599004291, -3.1049114610784923, -3.0416616489817936, -2.8895578372197064, -3.1891186813136607, -3.0252676583131599, -3.1754538115384143, -3.1122964947348413, -3.2604050144263992, -3.0816647131488333, -2.9461615576496132, -2.9766892385083694, -3.0844102861935689, -3.3558333024691289, -3.2324501817269815, -3.0737284983890372, -3.0140369039581825, -3.0809607616605352, -3.0437910831875663, -3.1848302356787155, -3.2353035821357228, -3.1773359636703775, -3.0737457347936417, -3.0767376308288128, -3.1259729255082420, -3.2696376181825713, -2.8659739038469185, -3.1037929715501709, -3.1389575635336704, -3.2273233803121313, -3.0141027499586452, -3.1218741752738621, -3.0711161564065859, -2.9264236358878106, -3.2064588831935139, -3.0628593649958722, -3.2111651310638245, -3.1628412164278275, -3.2702435665387264, -3.1216237797472379, -2.9644411744985200, -2.9942867874307200, -3.1134630695476160, -3.4023604947247899, -3.2551700739987837, -3.1110913379953180, -3.0221874608792381, -3.1179298482976145, -3.0733368405216304, -3.2009614817876164, -3.2451903851696136, -3.1959521733927940, -3.1122969525060551, -3.0981481329887361, -3.1638755679994208, -3.2783454747512533, -2.8723924735128530, -3.1256911960810974, -3.1625199973705476, -3.2391190643320473, -3.0491064920968904, -3.1372024469298880, -3.0948249838341737, -2.9433287799867203, -3.2140997680531966, -3.0903053464127446, -3.2386479626749303, -3.1975771028477191, -3.2764749776381237, -3.1624340077213358, -2.9780896781488910, -3.0108151245434835, -3.1377885874412401, -3.4358231285099086, -3.2740171694929994, -3.1359116522657113, -3.0321668167115141, -3.1397750782104699, -3.0858493756817023, -3.2177931428708062, -3.2472056395124023, -3.2146564853064468, -3.1381953199248338, -3.1119963111214908, -3.1878455107830894, -3.2686832089731594, -2.8564667818064935, -3.1282589978821047, -3.1663457312254679, -3.2321165617607313, -3.0625001792872943, -3.1317858164794186, -3.0979051984169796, -2.9393765159849465, -3.2002988247389901, -3.0986011031631815, -3.2486551320280181, -3.2148391427768575, -3.2631592999125787, -3.1859734988948314, -2.9704211026699929, -3.0076257696500392, -3.1429440772086186, -3.4497221906929716, -3.2737140013159172, -3.1413798246290456, -3.0222047275640294, -3.1424238189630680, -3.0792674425474718, -3.2149820801375921, -3.2287535517118600, -3.2144791320162884, -3.1447869199351457, -3.1085302886045012, -3.1919914980678485, -3.2811407571564706, -2.8639525136711046, -3.1433978365742896, -3.1546173653012031, -3.2232819326653539, -3.0654026007738580, -3.1337835408201657, -3.1039003877518341, -2.9443451201679491, -3.2057481527687854, -3.1138562380997401, -3.2616091363436861, -3.2354191331899202, -3.2689002803848521, -3.2054102321416651, -2.9722120632615807, -3.0200577601252343, -3.1365228243296492, -3.4717555573486067, -3.2788615026968078, -3.1495167898495264, -3.0304207592321042, -3.1402394039387245, -3.0737353116332562, -3.2227871222097471, -3.2280232115565135, -3.2233766693057988, -3.1547554863621379, -3.1228328291290017, -3.2018592584470698, -3.2797898091609907, -2.8528311571687075, -3.1444072984263620, -3.1254749978484067, -3.1979120338906388, -3.0485640966052729, -3.1175723131104109, -3.0919153832013979, -2.9317940076717610, -3.1936143673013224, -3.1140726792589808, -3.2610565163160605, -3.2418320763270518, -3.2590709128201500,
                 -3.2109616466801287, -2.9570970591983410, -3.0162727252083767, -3.1131051492373376, -3.4779221572103349, -3.2686662272634885, -3.1415161002051097, -3.0223410543124407, -3.1225118760980188, -3.0517817548294541, -3.2141619707043549, -3.2108165459471847, -3.2172505243715905, -3.1485481705875165, -3.1246264606223830, -3.1944940411754557,  0.3320935069414260, 0.2505254970100418, 0.5766672061579198, 0.2448174885670596, 0.6278487915288379,  0.2234228827234392, 0.6865168634718350, 0.6250451820780565, 0.2896833359179761, 0.4840864323943599,  0.4537707229980060, 0.5254524474281549, 0.4815978895168921, 0.7009566986816539, 0.4371651068409051,  0.3549629889690428, 0.3479981428472072, 0.4449215043335775, 0.8328565972192965, 0.5198008341839073,  0.3654175537832234, 0.5876985367093309, 0.5207604788255613, 0.3165319435722850, 0.5380314400881514,  0.7619201736255233, 0.5203810240539591, 0.3079076667058396, 0.3973537732069855, 0.5234226067691599,  0.3320934711537261, 0.2505253978581944, 0.5766667815917272, 0.2448176647928121, 0.6278485553742772,  0.2234231142031247, 0.6865164436690744, 0.6250448026316409, 0.2896834126119014, 0.4840863204230176,  0.4537705318199866, 0.5254522982631251, 0.4815976313695397, 0.7009564830180359, 0.4371649222693821,  0.3549630393708829, 0.3479981162357488, 0.4449214338146274, 0.8328563609253125, 0.5198006845380527,  0.3654174570078430, 0.5876983661887580, 0.5207603079460883, 0.3165320304343763, 0.5380312677910186,  0.7619197223134935, 0.5203809774457463, 0.3079076744363861, 0.3973537603299496, 0.5234222884882218,  0.3641439424398970, 0.2842864660387431, 0.6202939272147923, 0.2792097864859047, 0.6665561407822292,  0.2570756737403546, 0.7339177214091845, 0.6683767947344379, 0.3306655311997388, 0.5260298566188668,  0.4949120012156921, 0.5501193760721014, 0.5284913606574655, 0.7315554420713614, 0.4749724378905183,  0.3858980573367524, 0.3779344781894157, 0.4790217747741763, 0.8976759926940221, 0.5584953419320747,  0.3987978569890935, 0.6106799693629662, 0.5534501740998140, 0.3491930244296043, 0.5800159305136934,  0.8105521301580338, 0.5443525841474409, 0.3345210341430312, 0.4225765839701738, 0.5578343768737776,  0.4073361510580674, 0.3266748862158517, 0.6859690070176826, 0.3153283118862135, 0.7094671638625625,  0.3113169807437559, 0.7898536053625451, 0.7312072447437794, 0.3903133396186205, 0.5743059978359897,  0.5441878294838636, 0.5997079538213266, 0.5999761941529337, 0.7793507840695335, 0.5334881797486404,  0.4321544185697883, 0.4378313261260187, 0.5323423633737446, 0.9727604505021263, 0.6048779549153451,  0.4621556653432807, 0.6467240583832936, 0.6081874897518461, 0.3879140754726338, 0.6321997354147045,  0.8670885381224580, 0.5854656319526732, 0.3864121009770251, 0.4588327543631145, 0.5991233154771499,  0.4349830292509522, 0.3529770374998747, 0.7452614861243384, 0.3498084577098937, 0.7352691011414823,  0.3652479706411662, 0.8180567975863325, 0.7751510954634745, 0.4234763381087894, 0.6072342604484363,  0.5846010076564345, 0.6256806221514373, 0.6374156387096970, 0.7886763296858110, 0.5807178381228387,  0.4556804229313838, 0.4690397598792506, 0.5550489122031018, 1.0169433881297210, 0.6305998211939976,  0.5086933535598692, 0.6772871261578786, 0.6327303517033405, 0.3996513917609082, 0.6646819767396657,  0.8942130694434052, 0.6098338931338034, 0.4264970676758900, 0.4794062809175780, 0.6488802715685907,  0.4863449082902295, 0.4081945098558618, 0.8176116423132358, 0.4030311691257025, 0.7846934021272870,  0.4395092910919844, 0.8715258918686137, 0.8404292024411172, 0.4868839383305896, 0.6385600391385381,  0.6495860188648447, 0.6780668354165893, 0.6756352494362282, 0.8203338971713798, 0.6493232214348904,  0.4999369267721266, 0.5154202726201083, 0.6008924166743417, 1.0880872057063760, 0.6730229604839122,  0.5713608553962012, 0.7255563555577851, 0.6744518455643718, 0.4415021956579382, 0.7112322190162973,  0.9373977599455862, 0.6376081644665613, 0.4767969970872510, 0.5216217100006082, 0.7270640965113275,  0.4940731563493990, 0.4155928285141398, 0.8432620787506315, 0.3813201295130136, 0.7874146196302524,  0.4477601008896062, 0.8589559460686687, 0.8342852171933118, 0.4729198485995600, 0.6297181813964089,  0.6462415541761789, 0.6962330265850960, 0.6706106560764313, 0.8249794502565638, 0.6706639724167694,  0.5089348118315757, 0.5239655658867051, 0.5861187413400135, 1.0976584363996693, 0.6638637047044337,  0.5888987187202037, 0.7205537691268299, 0.6814130872168461, 0.4478411086466668, 0.7187540325205034,  0.9408088386490066, 0.6226762713998091, 0.4974458995638125, 0.5127601974828744, 0.7540665134806106,  0.4516898802022245, 0.3861229871373463, 0.8057869428515507, 0.3402911710366805, 0.7651943442440103,  0.4057583219923432, 0.8217107685212224, 0.7938701747628572, 0.4372635002333868, 0.5854666372358653,  0.5928086437408238, 0.6615631508913319, 0.6097856609024915, 0.7891305776331734, 0.6397755389703651,  0.4716562933976255, 0.4900071097133954, 0.5534137786340583, 1.0345583442070323, 0.6243061271938504,  0.5462184434385867, 0.6912863932265545, 0.6468768398916063, 0.4227903221888195, 0.6861856251128582,  0.9055442858504250, 0.5930341458097563, 0.4488963768031625, 0.4694932232452098, 0.7282695584288494,  0.4306638600531004, 0.3889945523814058, 0.8009576878706290, 0.3180027958251186, 0.7703644708452618,  0.3853709187715078, 0.8093111959793251, 0.7786126296976910, 0.4291862883495902, 0.5620390608885610,  0.5639920614720753, 0.6509954473360344, 0.5733630529569025, 0.7769797659329947, 0.6408393736833469,  0.4596183416662997, 0.4801034446269258, 0.5465728911291530, 1.0040368068932517, 0.6057575601005909,  0.5314130036067207, 0.6833136934419504, 0.6370267327337582, 0.4214437313000279, 0.6799041428548518,  0.8968334231691869, 0.5823370866756368, 0.4248171518547074, 0.4480452982411373, 0.7359708538122426,  0.4010423231535017, 0.3646794670720084, 0.7811043077894309, 0.2991017978357676, 0.7554926189568771,  0.3649237766238303, 0.7979948885660375, 0.7627099634547942, 0.4101053552857835, 0.5509214364419277,  0.5342797152731548, 0.6363226347330857, 0.5345097276232487, 0.7597914767410353, 0.6166249119059179,  0.4398161339126188, 0.4484426693048268, 0.5254689669280788, 0.9777530549420739, 0.5836316117254139,  0.5004096628340420,
                 0.6621775871964296, 0.6203225768371147, 0.4230087973898279, 0.6588731894677093,  0.8703221325074412, 0.5732681817810868, 0.3954268254346318, 0.4149287992176959, 0.7330752856464204,  0.3708026017734523, 0.3446860968262320, 0.7492382103799954, 0.2911132310272315, 0.7392994540609871,  0.3509773353763892, 0.7977138321106580, 0.7401818265986472, 0.3946557673751973, 0.5409815970160680,  0.5154410706322747, 0.6133339941078887, 0.5088807488805626, 0.7390117265243372, 0.5927561149282491,  0.4220081502954827, 0.4193162417488236, 0.5161562373129276, 0.9383943718034435, 0.5629554696907643,  0.4669244808054892, 0.6467113227547695, 0.6046272472537527, 0.4257168577275727, 0.6470809860781421,  0.8364854316202233, 0.5636811892642328, 0.3760099194934320, 0.3819402463973225, 0.7129325369087293,  0.3367338352272361, 0.3222899756028038, 0.7227251720912848, 0.2815061714963508, 0.7068265598606402,  0.3296974033474672, 0.7911579710006316, 0.7127833386037616, 0.3732293625871239, 0.5318854336776355,  0.4899287968235183, 0.5870084223676357, 0.4893139354361045, 0.7170182282628774, 0.5615106765477844,  0.3975082757495447, 0.3787458305721818, 0.4981683445156807, 0.8928369186151258, 0.5449883693302827,  0.4345908524263500, 0.6187505173795892, 0.5939187942101314, 0.4090136991719674, 0.6285916266592895,  0.8046885314943839, 0.5425755378808885, 0.3472459275027920, 0.3522703901556988, 0.6782996910829190,  0.3124982076173229, 0.3158235433646123, 0.7097075157585007, 0.2851776946970611, 0.6866950888220564,  0.3236622302894569, 0.7993109138510690, 0.7001088041835566, 0.3672440111548214, 0.5375859995703823,  0.4792183830104878, 0.5722678987072428, 0.4824512127093791, 0.7072137204984997, 0.5430194941191326,  0.3876482334096457, 0.3491934433817831, 0.4939933015736399, 0.8608516099371498, 0.5393053731187267,  0.4145638687458670, 0.6000978517532456, 0.5976107628336613, 0.4052204566661476, 0.6239277256333159,  0.7856420653506716, 0.5325819733585861, 0.3316805833333961, 0.3342111711816586, 0.6566488400204877,  0.2979084591633790, 0.3227010940205274, 0.7090140777859250, 0.3015363671111470, 0.6772057151864005,  0.3315427004573710, 0.8215709546180399, 0.7004684809953992, 0.3752759609310142, 0.5556815915905269,  0.4799299456607324, 0.5674757351363239, 0.4876572642221146, 0.7090006871542244, 0.5374491713090235,  0.3901224568673803, 0.3302395216414918, 0.5018670159317591, 0.8424516964487647, 0.5455870366190301,  0.4066107126577437, 0.5888173363300773, 0.6143563646910444, 0.4124521940518792, 0.6319412908932460,  0.7769207995332380, 0.5333118595940834, 0.3258984161244838, 0.3247258686808844, 0.6480412924006831,  0.2930743015185399, 0.3446638212409549, 0.7206781547027508, 0.3313696037228945, 0.6784764579374009,  0.3553592050069499, 0.8586770517833227, 0.7158980017104645, 0.3988669615021454, 0.5876747350108666,  0.4941437709985482, 0.5741434184994010, 0.5068269981066991, 0.7225553604358774, 0.5445952581245944,  0.4064776538243794, 0.3227438951522138, 0.5228990532539038, 0.8390332163520333, 0.5638285100075432,  0.4115502832744994, 0.5866630778876637, 0.6449192674051828, 0.4315086687733120, 0.6530627670585534,  0.7798371283547153, 0.5451563913227488, 0.3321321466968549, 0.3253690953371490, 0.6525574046347229,  0.2819768189550660, 0.3365030919584292, 0.7268791775795498, 0.3289860733504207, 0.6733056074647775,  0.3405613183583908, 0.8438062142552244, 0.7101310053279315, 0.3907097583349771, 0.5932022092177812,  0.4785858936377388, 0.5607319245268497, 0.4880070551708043, 0.7248272778171998, 0.5385713670730230,  0.4079452689731301, 0.3063632038518704, 0.5248817685553112, 0.8107594153393696, 0.5623589114853755,  0.3901581713723982, 0.5804962103072906, 0.6497054469794453, 0.4403807422203200, 0.6502024758233266,  0.7826079886620506, 0.5370607145825692, 0.3251877513934437, 0.3213411522554447, 0.6435706253875011,  0.2770963233700148, 0.3320544498796091, 0.7230309884835904, 0.3247399086353607, 0.6744709627943205,  0.3252459914603145, 0.8357001864987104, 0.6963303631228976, 0.3760804744433995, 0.5905645218060851,  0.4626579296457702, 0.5457883827591008, 0.4633072438953819, 0.7220922732462220, 0.5202172475235971,  0.4074046082485274, 0.2941037956005318, 0.5175583145959660, 0.7878854669014619, 0.5545223359772521,  0.3746151484457783, 0.5804183988275576, 0.6456048284203758, 0.4414036044452960, 0.6451833808578580,  0.7816032198563937, 0.5297199620355015, 0.3116532728245665, 0.3192970747854509, 0.6327949510168801,  0.2795483096251581, 0.3378360914772241, 0.7254187739183687, 0.3298076946587247, 0.6829186909237268,  0.3188930924826814, 0.8345399040152767, 0.6903650151445120, 0.3713888668428800, 0.5968874958519557,  0.4540325652097434, 0.5372913954900929, 0.4462759917699977, 0.7270161615961173, 0.5077439229065764,  0.4157384337205273, 0.2888352451342170, 0.5177604188058527, 0.7733909809808467, 0.5537301873146986,  0.3668869595262315, 0.5858948039790169, 0.6498967540649783, 0.4509118012003904, 0.6475743296959580,  0.7887606199195364, 0.5290049624827273, 0.3058219861882207, 0.3245535844906117, 0.6303229803574393,  0.2899788975528311, 0.3543545452100793, 0.7363978432100888, 0.3441878395683240, 0.7002535797374206,  0.3223512848180173, 0.8421508722149686, 0.6938220233805670, 0.3767076688028033, 0.6132051369463716,  0.4536706539403400, 0.5360231241635283, 0.4370512379873207, 0.7408910963617521, 0.5029879697083320,  0.4343635528473602, 0.2924894873506522, 0.5266622901988537, 0.7684511789664594, 0.5613345725147287,  0.3677189928102314, 0.5989232072607276, 0.6628536580863168, 0.4689918841799066, 0.6591516172170370,  0.8054079547666559, 0.5362746410921845, 0.3083439842032925, 0.3379883883029440, 0.6367203083416814,  0.2895491078825189, 0.3591791425731984, 0.7347424947023158, 0.3653862350225892, 0.7177963000595025,  0.3320530906911925, 0.8529635733025503, 0.7008104767798385, 0.3826758783018289, 0.6201345207346645,  0.4520311769492697, 0.5360454651782646, 0.4301404728418902, 0.7472261443682011, 0.4977672546084849,  0.4440781038071706, 0.2934782675252380, 0.5424332580102142, 0.7596970369302596, 0.5679280622612993,  0.3718667942178685, 0.6055621143180121, 0.6757935703592732, 0.4846288838082953, 0.6640154041628935,  0.8168777709069946, 0.5404350218590104, 0.3116861531308337,
                 0.3353423114855133, 0.6400073247905157,  0.2950316403908166, 0.3727335954867030, 0.7389934532570213, 0.3945435315658608, 0.7433149078265279,  0.3508966309901104, 0.8719564658931147, 0.7162420903695386, 0.3975865787495118, 0.6352779084713088,  0.4568557880281978, 0.5415245127737852, 0.4297180606863852, 0.7605908914201137, 0.4983448755275794,  0.4616211315262371, 0.3019797256120352, 0.5659592352729585, 0.7588844115393401, 0.5812833535900798,  0.3831117841193760, 0.6185509220661916, 0.6952588551483661, 0.5077160035888328, 0.6763242547934774,  0.8360458520655455, 0.5510877070013314, 0.3218656973022740, 0.3382609217247659, 0.6508928501499358 ))

  set.seed(995)
  suppressWarnings(tmp <- ddhazard_boot(fit, do_sample_weights = F, do_stratify_with_event = T, R = 30))

  # plot(fit)
  # get_expect_equal(tmp, file = "tmp.txt")

  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c(-2.9417270151733201, -2.9417271494125758, -2.8751008812968308, -2.7944202142630039, -2.7218473374240366, -2.6454718499478842, -2.6548810638159783, -2.7347206446107339, -2.8005733099400301, -2.8613648474096909, -2.9213549135070527, -2.9859723293007789, -3.0339289593190788, -3.0675486510431185, -3.0755737654547737, -3.0985890531657061, -3.1251804112205708, -3.1437771523950744, -3.1430265035546250, -3.1483871572853541, -3.1376621000064713, 0.4404171871819423, 0.4404171192811067, 0.4736959887657881, 0.5200446109080480,  0.5521174112845471, 0.6022336481103565, 0.6068304383076504, 0.5726439210576278, 0.5629964410440592,  0.5469270404499240, 0.5294419316856761, 0.5071540820900921, 0.4977363624237865, 0.4999564634380535,  0.5146366934172963, 0.5098673080230457, 0.5031829300055697, 0.5042201332917474, 0.5139403746897799,  0.5206467192162000, 0.5345147175479759 ))

  expect_equal(unname(c(tmp$t)),
               c(-2.9372490170014713406, -3.1262555299090974970, -2.8356962860042007613, -2.9228965492523832914, -3.0224023565769320143, -3.0283734527039611883, -2.8264401272743309157, -3.1169519930836657728, -3.3464684090414853479, -3.0995248965175035138, -3.0844276233264382725, -3.2865481632546278412,                     NA, -2.9710807094301667775, -3.0023613438016623078, -3.2153098210553605618, -3.5510268449199728913, -2.8784754398212739623, -2.7683350978086451732, -2.9966829946193240808, -2.9467505605000616775, -2.8116902689592819797, -3.8307141377963587203, -4.0802687108067852151, -3.4258785529751367704, -2.9860122111259252975, -2.9376589114697484462, -2.9600371099618838500, -3.1014270414670344067, -3.0382284461459412306, -2.9372490711352519099, -3.1262553990713608520, -2.8356965096699431328, -2.9228969823545751794, -3.0224023310387604369, -3.0283734611528885239, -2.8264402135244388603, -3.1169521510566244871, -3.3464687680251818946, -3.0995246574748898105, -3.0844276593370323347, -3.2865481575095607525, NA, -2.9710809019179285073, -3.0023614723341758292, -3.2153100416241007231, -3.5510346259194087715, -2.8784757558715270420, -2.7683351824562505428, -2.9966829200443596548, -2.9467508654035743376, -2.8116907074871706484, -3.8307350102590769225, -4.0802813020414623324, -3.4258786829835372600, -2.9860122813493794602, -2.9376588971631498737, -2.9600372777741346120, -3.1014271460012241555, -3.0382287227877387004, -2.8741694706481548671, -3.0683317901644597825, -2.7652644498560370323, -2.8483991816557363563, -2.9743847914109049491, -2.9542501977843818040, -2.8078068043911859064, -3.0428434926225742529, -3.2483479798249916826, -3.0407927264130485412, -3.0213289923601283071, -3.2261347050947435378,                     NA, -2.8918228857809440413, -2.9156030411043372119, -3.1257105693028188753, -3.4720650694388952573, -2.8239992733721033247, -2.6913284627027294071, -2.9346909456900425184, -2.8630996919355169439, -2.7255927731349638421, -3.4431498233695738698, -3.5278854271509869633, -3.3720419224680346915, -2.9006483260544575487, -2.8897101121773114407, -2.8900464066841791144, -3.0264948693344133623, -2.9607828966838871665, -2.8108624748750909994, -2.9896916224971286091, -2.7014864656874979509, -2.7449138603096230327, -2.9084633534212587236, -2.8532044031098791059, -2.7868603119753325714, -2.9610184871446256771, -3.1107240408966561063, -2.9627213261088654050, -2.9668346775785319203, -3.1652447017965190668, NA, -2.7944707288968437808, -2.8299370053575740869, -3.0149797075416655545, -2.7673744490577920629, -2.7489402720863291840, -2.6089800480613636324, -2.8634877467956227903, -2.7874613693766363554, -2.6656785528220177106, -2.4377888721081504286, -2.5685099428185775849, -3.3079972530585424195, -2.8046451948130313170, -2.8470998601140973250, -2.8047102805300347939, -2.9631311289817841015, -2.8843036941750641589, -2.7532046514556984818, -2.9634327248711374025, -2.6384345712870254097, -2.6232505872632501109, -2.8390858013145443550, -2.7486228207989675809, -2.7177836033749609435, -2.8921033888209350948, -3.0621545505881297977, -2.8871630508457326592, -2.8944765323188277328, -3.0871519263705589076,                     NA, -2.7003299046415638607, -2.7490683416802101924, -2.9586191458668769094, -2.7683595466277939323, -2.6279754426789798671, -2.5638978241837624417, -2.8054768423578564374, -2.6841911752978817596, -2.5809364011303079778, -2.3388647744633419556, -2.4966594796570298342, -3.2596080124371535902, -2.7452638995384406329, -2.8091303097248485621, -2.7550387947333447869, -2.9185379903446793115, -2.8318590058568555712, -2.6931166480022534770, -2.9049877506434529373, -2.5937230047538424316, -2.5475523816822471801, -2.7709646042865241355, -2.6465571087020447472, -2.6699374613748432239, -2.8020308088682992498, -2.9758717031483739035, -2.8162679830087142818, -2.8182845802955913683, -3.0330012966021167031, NA, -2.6585829873604325257, -2.6697121929105493976, -2.9172010070256866321, -2.0801443626555662370, -2.5367585800792715744, -2.5090558550916171043, -2.7553293234854083771, -2.5828438382736611523, -2.5066455548882586868, -0.3925942864977726710, -1.9007533243851340021, -3.2060668880436393735, -2.6797041369758538565, -2.7949832896490907075, -2.6604634465587637671, -2.8771945848970590376, -2.7653760782829621689, -2.7171851206668846679, -2.9609111023003826801, -2.6373510367729302573, -2.5756586981783113899, -2.8122069308518575781, -2.6881777693067268054, -2.6759934156837665853, -2.7934857891417159692, -2.9846205227852027342, -2.8466966518789793561, -2.8466559752390252491, -3.0434519626175893237,                     NA, -2.6104457671612477654, -2.7262051054264899719, -2.9230101374343173681, -1.5827947953203782561, -2.5611459667019653708, -2.4848902308939435635, -2.7812974986483909312, -2.5593585082595544655, -2.5100900391074527640, -1.6711744222496316414, -1.8105918363504514002, -3.1782590194619033674, -2.6805441458831680279, -2.7993358010339615127, -2.6063385893364934276, -2.8649243183562917991, -2.7728189645705132449, -2.7717012914565737347, -3.0346231632663189792, -2.7226936060578745824, -2.7096590359520713598, -2.8827280525263594413, -2.7494590389223665383, -2.7203130323076383235, -2.8416529077078558352, -3.0304183858831210685, -2.8939270414061546433, -2.9256629447607060968, -3.1002613962702629813, NA, -2.6794831941262780006, -2.8235633065078711290, -2.9519047041855555058, -2.3158339180131282120, -2.6431703450311334613, -2.5229583476012980015, -2.8726946377903219698, -2.6350884681720798675, -2.5876206550735778578, -2.4166618242682682549, -2.5887473314091296039, -3.2426650328443176186, -2.7644795577181175972, -2.8541332296960746362, -2.7112709194272524016, -2.9486694704448845883, -2.8418975329453166623, -2.7982248374574814065, -3.0753051779327176263, -2.7726976886870033923, -2.8150585900956435204, -2.9216939038117519800, -2.8017950537163351932, -2.7325163930958935588, -2.9145986350280024268, -3.0569814194995261758, -2.9244707776844691693, -2.9793616912615439318, -3.1293659291367261766,                     NA, -2.7418115369055788832, -2.8854931690565197577, -2.9933260389101565835, -2.4672064338761354740, -2.6806695765549983790, -2.6045223164592288256, -2.9455604908108385942, -2.7338876990556482127, -2.6897752245942418980, -2.6518445966955854765, -2.8094220890985810435, -3.2703895858698204258, -2.8347801812752511097, -2.8979527363868964329, -2.7936048269324751381, -2.9984285350100488010, -2.9225482551545973742, -2.8652599184231601903, -3.1147739350300396310, -2.8019063802217614167, -2.8782280503594490639, -2.9312492016555320440, -2.8722338650261578508, -2.7247489210927242631, -2.9787231918298973099, -3.1030777720804190167, -2.9983891783213505455, -3.0520298941632204581, -3.1507920916898619623, NA, -2.7941762304947515005, -2.9424838614005226489, -3.0691060243713552680, -2.8728304181622901226, -2.7180157880423276850, -2.6699736041902606409, -2.9915763165381505928, -2.7887630512126468929, -2.7804338066337606072, -2.4285845530532128578, -2.7489054727565136460, -3.2776604657291477096, -2.8864838666865364480, -2.9373046423996789223, -2.8743525023120173323, -3.0820215986561136923, -2.9749102256716852821, -2.9186549316273220356, -3.1666537248911752478, -2.8901475566809358142, -2.9121743355490075089, -2.9505884509484654465, -2.9268826553574727178, -2.7319084460426599215, -3.0540777915694308220, -3.1933733294837134409, -3.0613059732951404257, -3.1246440380805293735, -3.1659840132854575323,                     NA, -2.8259103614990110920, -2.9720415523360239618, -3.1016394592892653215, -3.1836606640145017622, -2.7735932670573011372,
                 -2.7209993286037348348, -3.0389989241753867510, -2.8538598269853374667, -2.8609600510207546797, -2.9322143209245186135, -3.0222931480508195001, -3.3385217530492776739, -2.9481628536340367397, -2.9999599449728169276, -2.9720066122471111569, -3.1321146794608361041, -3.0131097781118585388, -2.9757604215165063266, -3.2263535295891303534, -2.9733771426431436780, -2.9617922928352569123, -2.9711171718315911860, -2.9823427263559931788, -2.7580585822364200865, -3.1199887771644982948, -3.2656407070518529423, -3.0910926744519895415, -3.2028076207415852039, -3.2290895929361984606, NA, -2.9086755646264905728, -3.0168034978767295229, -3.1650555808810758585, -3.3194632252540725581, -2.8729058377807783486, -2.7621856379886651212, -3.0900864084738564053, -2.9351485613647527728, -2.9364768251866451010, -3.4369214296883718873, -3.3856159820939146776, -3.4075141622358300353, -3.0294960320728931968, -3.0678128265432338395, -3.0399438333472934559, -3.1738604247167541139, -3.0687383651872313095, -3.0147413605694612926, -3.2611156598909478710, -3.0310206120787652750, -2.9899293758349205774, -2.9718760805646780021, -3.0279749290573465537, -2.7979422968165086338, -3.1683574790950901345, -3.3338993201898614593, -3.0994459054737362180, -3.2647249215905338460, -3.2760921810582548375,                     NA, -2.9878649825922094507, -3.0547158489089940581, -3.2353091411543184464, -3.4941256832824714174, -2.9584042785563213762, -2.8091404709303526843, -3.1235278222774129908, -2.9997660053631882704, -2.9962557328020498204, -3.3985213669636258338, -3.3867975520811111068, -3.4513077771867592958, -3.0953703414964608420, -3.1208646598910454451, -3.0795091225323538708, -3.1977692263626598113, -3.1344237702210442365, -3.0366894873499337137, -3.2719147015019953706, -3.0642606732032482419, -3.0379902193107413133, -2.9693757944078380540, -3.0405308713859398040, -2.8116675499588508202, -3.2016365645926496697, -3.3753351888654572477, -3.1102725029500239984, -3.3004961792675935328, -3.2962816427775369910, NA, -3.0539973152873391982, -3.0892065992049326795, -3.2929534136711295922, -3.6111870899384674871, -3.0201073819661758968, -2.8291323649545128127, -3.1403361845986896839, -3.0503311665234433825, -3.0428067034372938693, -3.2291467825250297352, -3.5326872329749292589, -3.4834628784495551557, -3.1484707005213197206, -3.1485989449201601253, -3.1037473996664681763, -3.1928512715978922820, -3.1768765002446022194, -3.0305701494495420611, -3.2586936541639506260, -3.0732461451135435482, -3.0581366516091588181, -2.9417856654501042257, -3.0197524247351648086, -2.7990495357053588599, -3.2085195784799154239, -3.3901387356795207317, -3.0902880613374135166, -3.3101771960584303045, -3.2894730264381841778,                     NA, -3.0958283805002868583, -3.0973021314737767540, -3.3260229773320659774, -3.4981285590168296196, -3.0585107330417140581, -2.8222807468818897902, -3.1294281587896266572, -3.0749893953981066730, -3.0645042029574063314, -2.1198587721625852254, -3.3157917645153549557, -3.4937424717211165159, -3.1775599237808660469, -3.1510104073637221767, -3.1010034290894483711, -3.1588854788164337783, -3.1962647833875483805, -3.0606921266756379474, -3.2473060605960206715, -3.0978120591834485964, -3.0802573438686007279, -2.9269164096426152355, -3.0489062358968923583, -2.8291466938924894059, -3.2306406257400590576, -3.4117548534260007109, -3.1172838417824237922, -3.3204616037869829626, -3.3204294920697212667, NA, -3.1419610860162050514, -3.1349768300520057807, -3.3648571138551370829, -3.6949662877788029824, -3.1030085772617881368, -2.8023535604402471932, -3.1427562735989171827, -3.0881499761972972529, -3.0755765253348279487, -2.2266427877209173936, -3.1276731673909692333, -3.5084052725082544022, -3.1966864020465246554, -3.1645424990164836565, -3.0855814239086281070, -3.1736384817631142496, -3.2067057959453268978, -3.0936118380503287995, -3.2261746835207669726, -3.1254970404852175925, -3.1199858783533738205, -2.9120398267704779016, -3.0802482242381818978, -2.8497669329649624181, -3.2421227011284390329, -3.4402571783819433371, -3.1448732280374991355, -3.3199495889272041538, -3.3665499123167350604,                     NA, -3.1671826781018768138, -3.1663050936829932880, -3.3976357807274140121, -3.7329559965051770654, -3.1405338110368656857, -2.7946891111068907776, -3.1431468935712114643, -3.0906781008800523480, -3.0896392525159646070, -2.9283231992404008537, -3.3210646423602776522, -3.5277984720898283300, -3.2209651918342792065, -3.1900911368549365044, -3.1075171840447923799, -3.1883483167783652767, -3.2739090231003009634, -3.1303667999243556253, -3.2163988196058070912, -3.1336453833283588466, -3.1381309609881862244, -2.9207646670180227311, -3.1263015286967208439, -2.8724244721508229716, -3.2555656437741280484, -3.4614746120789710027, -3.1737209553594762923, -3.3197939299276275626, -3.3950876239729552708, NA, -3.1841767008414780804, -3.2034112436419039938, -3.4108494195483749500, -3.5845421446032155188, -3.1593914231267916648, -2.7998960413735072272, -3.1194608267707488736, -3.0689139705776788958, -3.0814239553293183427, -3.2961548508323184059, -3.4495647664059911897, -3.5515949883324333669, -3.2373997158476623959, -3.1948280526387544676, -3.1189625791456196957, -3.1927258833344738242, -3.3271853849388124402, -3.1493077893255732036, -3.1851080840393843907, -3.1222844869588968209, -3.1347524077758839312, -2.9098075125696332144, -3.1524115123928901916, -2.8765549668381589221, -3.2480412028064695740, -3.4627691624494150169, -3.1820148102238099419, -3.2970077526659262723, -3.4061418317653027188,                     NA, -3.1811992662489632089, -3.2235702971419701868, -3.4045360672096109944, -3.2414472532016058715, -3.1597083085503174793, -2.7824810702919293703, -3.0716165200742469032, -3.0226638080769272499, -3.0508930325940246675, -3.0224258603297999848, -3.2213454671431254184, -3.5589814654658780135, -3.2343914054521771817, -3.1787218670078045690, -3.1079334237784483896, -3.1757054403739140014, -3.3668345212104746622, -3.1752087964387629704, -3.1822182428186422953, -3.1295297887792083991, -3.1671171817374643709, -2.9045666483997267804, -3.1586686683718605551, -2.8849549636696938393, -3.2332689181383296173, -3.4772489310203416046, -3.1813587313281606050, -3.2911982296357074063, -3.3997526853535453562, NA, -3.1983876488028517571, -3.2390972935507513775, -3.3935757868674985005, -3.2880876113490473678, -3.1540710295863325641, -2.7948268560841103003, -3.0831200408430117221, -3.0063292896175024538, -3.0366000198366833729, -3.4731317270848838241, -2.8788673409711069695, -3.5629580996132408366, -3.2653504511200388549, -3.1537847865849797380, -3.1026747541020793797, -3.1731668557993630841, -3.3931042988770636626, -3.1881022523859154916, -3.1611567436869685288, -3.1199581699456930828, -3.1832447881728311856, -2.8828192809485577719, -3.1477624754088928682, -2.8774850283147515029, -3.1986068387791752876, -3.4761750979884267210, -3.1615391814942062076, -3.2682234504619072091, -3.3773170500898248569,                     NA, -3.1999607452729845747, -3.2398797968854822216, -3.3659222126767569527, -3.0296737600751200148, -3.1303527923476761607, -2.7872007053346496974, -3.0757390120817191814, -2.9695677358801066958, -3.0019009726036132157, -3.7544299317484401968, -2.0881485527731040719, -3.5520927644327184858, -3.2807782053028398117, -3.1114210326200271162, -3.0788489068757098899, -3.1521955379729504898, -3.4061782737498100815, 0.4562987657314457879, 0.7605975165275645766,  0.3579168375301289284, 0.4835358921817262123, 0.5530213582031593411,
                 0.4636679722222322164,  0.1299411205200183550, 0.6140956870672940049, 0.9255902055811882079, 0.5876249078016784200,  0.6631816775372103567, 0.8329707511787808327, NA, 0.4730679043132340889,  0.6373474620792040080, 0.8352188824060163830, 0.0005665521259682789, 0.2967203530116326871,  0.0358912809920864176, 0.4304674808933681152, 0.4264466268840910734, 0.3170104171974251761, -0.4090781110824666666, 0.0241526786789151224, 1.0480087965730546795, 0.5668609923879410761,  0.4535290426234138139, 0.4350309845708948409, 0.6736655411439218533, 0.8335298368893563348,  0.4562987148143780591, 0.7605972044484196060, 0.3579168624297865420, 0.4835356493371932851,  0.5530212538197039063, 0.4636678410005369289, 0.1299411492886044517, 0.6140954860183174269,  0.9255900491402873076, 0.5876246589145164290, 0.6631814487155049953, 0.8329705608642773429,                     NA, 0.4730677548843549851, 0.6373470829659680925, 0.8352188533348104960,  0.0005622701640832783, 0.2967203502517666847, 0.0358912018704829194, 0.4304674053095604580,  0.4264464936005477602, 0.3170104377865288203, -0.4090863825306665369, 0.0241461429333463332,  1.0480086202928884376, 0.5668607073095134341, 0.4535290145213314306, 0.4350308286496747945,  0.6736653267153251967, 0.8335293304078649212, 0.4858809055508510610, 0.7950912164890433509,  0.3913157420136313958, 0.5202259353920366536, 0.5744839147095890386, 0.5012125973421526215,  0.1237109711671847273, 0.6563227235597746123, 0.9915696208625719432, 0.6184204594978914837,  0.6940238953360822194, 0.8707811272897614252, NA, 0.5119888485075851925,  0.6863732007506653243, 0.8880262384515504870, 0.0363547072821366335, 0.3190171251492013305,  0.0721352793335763298, 0.4640768244491320083, 0.4709582668498346614, 0.3569936414931301760, -0.2589564820866485650, 0.3108901085709192547, 1.0840254890480314209, 0.6138394782138146821,  0.4708293898237995712, 0.4725838155170184463, 0.7175081405647952826, 0.8749928052079916085,  0.5166172284209911147, 0.8452857987160923292, 0.4276991018450284066, 0.5726085727221745802,  0.6206672003554163375, 0.5538319569229368522, 0.1356775034094920562, 0.7146768342783698102,  1.0818213772547826679, 0.6612950021039867288, 0.7349934240420457066, 0.9149723352586711655,                     NA, 0.5578521294101161931, 0.7295176122770417049, 0.9477590162939124596,  0.4156603910636025301, 0.3662679734239370211, 0.1116489905433495911, 0.5017416033786095131,  0.4959302199161557123, 0.3820444444273084494, 0.1356803278776951283, 0.8091537759987197376,  1.1361360467424548304, 0.6642334083935867994, 0.4997502609383888128, 0.5251775804145820681,  0.7689537333659653973, 0.9242834358882436874, 0.5416226731855808829, 0.8498683493163398106,  0.4523383254461273983, 0.6317530724482207649, 0.6525511546616156178, 0.6025147163599910982,  0.1682461307207767376, 0.7695800245694520703, 1.1040529643245835523, 0.6976925979166741509,  0.7706121405648369560, 0.9698769467774936981, NA, 0.5939308372924900770,  0.7630816153795215717, 0.9660641467502921742, 0.4020878695306121520, 0.4307167612739873919,  0.1270458829610502383, 0.5173278119349901116, 0.5306199685995380833, 0.4117006097996691594,  0.1707952107208340342, 0.8352025010069400457, 1.1661090457184415392, 0.6739064280842496091,  0.5123474582933169597, 0.5428924561639004720, 0.8117575449718239522, 0.9444112799889827770,  0.5903559118964852859, 0.8929399422352094495, 0.4924030230837260214, 0.6912044728546103967,  0.6992874696638731180, 0.6788717061734006464, 0.1978403268741746879, 0.8476995208964909212,  1.1682910144133438202, 0.7470767917770619881, 0.8408248889519268632, 1.0246012097025116727,                     NA, 0.6113075488543782132, 0.8182520023732340331, 0.9913328553578459879,  0.7932725449034754295, 0.4874493304947856709, 0.1443688862256565786, 0.5513898968148991386,  0.5801743799776698518, 0.4513770025889629611, 0.9492937250749653444, 1.1497379965160641202,  1.2073243600923149188, 0.6976458263404735627, 0.5362115165114204052, 0.5962926124453709065,  0.8592839059550502201, 0.9970298548112348502, 0.6188699737273481993, 0.8444998161909956780,  0.4861606227651344980, 0.6799204794844295741, 0.6865080302493034026, 0.6700839235226820900,  0.2050516502679601283, 0.8739600437590906434, 1.1711699962709538614, 0.7456415634539770299,  0.8518285610813751685, 1.0214079082746692073, NA, 0.6355287018569647506,  0.7968737336388371784, 0.9988446796356094115, 1.0556024754784314457, 0.4854572728229739065,  0.1404279109064000552, 0.5406238073210701822, 0.5992883403754006855, 0.4477469400418092604,  0.4503833681837505765, 1.1938651643424955662, 1.2176391965274828699, 0.7090548033683485851,  0.5609970623932036826, 0.6175207239720480468, 0.8911283344319523847, 0.9960014271782865469,  0.6202920226191208641, 0.8024232055070279479, 0.4584222152542677398, 0.6085389989528554588,  0.6609004814962791130, 0.6630546660534102354, 0.1975872381545196732, 0.8447478406587674016,  1.1345246488789926254, 0.7344940534121018327, 0.8437797085506761885, 0.9842526551686368030,                     NA, 0.6062731094646480745, 0.7634992944045361218, 0.9752675053465564847,  0.6551481214523589003, 0.4654291160515156567, 0.0992030352442300756, 0.4881727361173954227,  0.5448465890984063309, 0.4056258786508529779, 0.1723027993086599374, 0.7953071722562958801,  1.1654904139210304770, 0.6737533955622097004, 0.5545490532455155108, 0.5617588785650466932,  0.8713533506272217455, 0.9526536835605405873, 0.6499234071601202034, 0.7875139418843365435,  0.4548388782186211232, 0.5551597186480994317, 0.6598242407316594615, 0.6853196497169038448,  0.2126167707953769137, 0.8403162044976644207, 1.1350117894602629320, 0.7497950917874338650,  0.8638392655848410318, 0.9765062905865909970, NA, 0.6000250051758468928,  0.7565029902537352413, 0.9797106611019269495, 0.5839319782398458836, 0.4782182112878672897,  0.0714063020971792833, 0.4568946295049777229, 0.5065631390726650363, 0.3776481357361635149,  0.1086648059942182387, 0.6926139321671717086, 1.1439415888139958799, 0.6635120886366472348,  0.5714838769405705676, 0.5303359779200548241, 0.8791493446833819059, 0.9271950275304572386,  0.6352400500588560872, 0.7647635079833841054, 0.4551101266423199232, 0.5218151107209220152,  0.6725022270881185760, 0.6828816819433379814, 0.2307653324345471935, 0.8334333862786680314,  1.1123114098263238159, 0.7182119356691405443, 0.8585386511954981037, 0.9654637535484484889,                     NA, 0.5879237677039890198, 0.7423881673939972670, 0.9506021296963756972,  0.3561808193302024028, 0.4802115425595110398, 0.0433510893038860923, 0.4374118517894858282,  0.4907519705649955322, 0.3459877098156046160, 0.2211457590786334582, 0.7273595934534927698,  1.1315995368491051920, 0.6580343698050122869, 0.5811687893734821531, 0.4897575506859802585,  0.8490869758882804152, 0.9128302631879607887, 0.6292636116988670203, 0.7484715672154156341,  0.4317232575503844316, 0.5068476001607631343, 0.6658723658725304029, 0.6819527309111345081,  0.2398887435508306187, 0.8070498296276588102, 1.0639705856956847008, 0.6930318823444496790,  0.8520681482952181529, 0.9668362482968462235, NA, 0.5860075576632606253,  0.7384591993177466174, 0.9452664157785168886, 0.1872389640800658783, 0.4745015915509919191,  0.0218897611357279087, 0.4066332483661801445, 0.4717853121235200398, 0.3268406053425298996,  0.0374585536512565542,
                 0.5875459786191696843, 1.0993260917833880264, 0.6519387192651274754,  0.5694428313517616580, 0.4426080270630624036, 0.8372651997730107265, 0.9064151788946194221,  0.6098516535488437418, 0.7274912079447976021, 0.4125196745579163826, 0.4846234375014116802,  0.6558225538475335181, 0.6781341580603608143, 0.2378138412778780797, 0.7871471862197542135,  1.0232862103340161219, 0.6843115405722102196, 0.8408106584588475041, 0.9452660417321282482,                     NA, 0.5575890768450515456, 0.7198647042766486148, 0.9167615936141160216,  0.1169347504500548307, 0.4347884715416570622, -0.0042094561390637712, 0.3799279422265277595,  0.4557928553611121369, 0.3093436745158090750, -0.1510174739977875324, 0.3992486041599766500,  1.0546744002551475816, 0.6222408936269425705, 0.5614583772661279060, 0.4103649983651410649,  0.8314261051763113253, 0.8836859707693848209, 0.6037274845602952356, 0.7210150397657808519,  0.4056398270953183172, 0.4769576704764409980, 0.6603898298566671654, 0.6894919729254804786,  0.2482855801775877835, 0.7810915899884414149, 0.9960340750551506250, 0.6909666130488145708,  0.8431460315890867419, 0.9390815797813709676, NA, 0.5387695635822893347,  0.7131666998064511986, 0.9008259002512059022, 0.0337975622047065050, 0.4068368481725557206, -0.0213188574648958615, 0.3673520302486540934, 0.4533354902229184136, 0.3026912758499457068, -0.1239323644057364615, 0.3972659466572922016, 1.0255839996548712367, 0.6048709674601575603,  0.5663121591298885260, 0.3919680365973459701, 0.8399965070218579077, 0.8684475204143999871,  0.6105877008713531673, 0.7280791655784760730, 0.4100764864224797357, 0.4775881364456311950,  0.6747535503002179968, 0.7178218096063362363, 0.2723297173944156935, 0.7874324739737663670,  0.9844730716375909019, 0.7108063320784718808, 0.8585713375150214599, 0.9489246381237991246,                     NA, 0.5297892100699053142, 0.7168626175428431946, 0.8970254405291873123, -0.0140920434590253718, 0.3904307849943827735, -0.0273944852916642514, 0.3685938069306323728,  0.4629378721616920767, 0.3056677628437179939, -0.0412998613885507959, 0.3265322883424179867,  1.0090171423982636778, 0.5984255625696005554, 0.5842031334261842401, 0.3856575344475107192,  0.8638186475002485043, 0.8635031183560799573, 0.6311060757627311446, 0.7486415509127475332,  0.4256540348234399418, 0.4922531650290619121, 0.7008084463603653225, 0.7629982708224102295,  0.3098105629315522913, 0.8071728089239169623, 0.9883669150043490337, 0.7467895261303189036,  0.8868727479964551019, 0.9747284273631737772, NA, 0.5313011129876386018,  0.7330147648206334354, 0.9062246324852297752, 0.0627287084922521221, 0.3852240770525764968, -0.0225127550688556458, 0.3846073880451962301, 0.4855077379744111021, 0.3191640775330449875,  0.4146842065589685933, 0.4430238709852679646, 1.0056947829639217851, 0.6035417961776867735,  0.6149474447056044735, 0.3926933471323984515, 0.9029174901004056730, 0.8686731396835354335,  0.6134238972739658458, 0.7612889731448571062, 0.4213666200461275713, 0.4977018499041765409,  0.7070072580336332901, 0.7594483498879847083, 0.3022234659379539612, 0.8063813205231964432,  0.9818537815486174658, 0.7355012263140793261, 0.9066546934777323097, 0.9627724181231964717,                     NA, 0.5215854149265355399, 0.7179257036453713381, 0.9057914342927495532, -0.0370017440061350816, 0.3687858095175328144, -0.0173308827288809197, 0.3735413883706250604,  0.5097044088603981393, 0.3322656353878764568, 0.3759695479793148110, 0.5392041626714144664,  0.9937427023022225026, 0.6094538799393681705, 0.6284033969118936191, 0.4018568003535756272,  0.8930724461633526845, 0.8728367772072814779, 0.5933383709915295290, 0.7755284254809180489,  0.4240240314378076469, 0.4903543066029317776, 0.7207016944658911628, 0.7449405621556428603,  0.2939462346446364283, 0.8147723793361612010, 0.9729977458739339502, 0.7233434181111023653,  0.9270377519543064881, 0.9487795925131270458, NA, 0.5200780054871927405,  0.6999457310587183789, 0.9046627690717652381, -0.0511106772935098486, 0.3598346816190287623, -0.0149538294651255788, 0.3736276858168848802, 0.5348115839139805372, 0.3430464180785075534,  0.1010699832586591795, 0.4473099173205130064, 0.9809768992810331545, 0.6046992253089968905,  0.6414537834774665104, 0.4065915183550190548, 0.8913728730756621532, 0.8491718656440538915,  0.5792642153481623035, 0.7983110924654157614, 0.4352644512858783177, 0.4928855848079108926,  0.7400096476247324251, 0.7358684609668377785, 0.2929486362889043294, 0.8309296764575893057,  0.9731796113247498381, 0.7188071672624674058, 0.9561967527099068942, 0.9440906986321587757,                     NA, 0.5255224141594940779, 0.6864729078844044041, 0.9129315584571243702,  0.0354836026309643948, 0.3594157154617182437, -0.0066591843714334242, 0.3859369755523444634,  0.5713173394534989402, 0.3628265527677123070, -0.0424702981638358651, 0.3922403562742301508,  0.9760833613578416834, 0.6076351450594752457, 0.6645871723778509210, 0.4216261347839130869,  0.8986554963865266421, 0.8304405272102530766, 0.5725176545946868423, 0.8330875569306968043,  0.4550388493804428114, 0.5052424203685396131, 0.7687104277777898620, 0.7351492036941000840,  0.3001614536072170214, 0.8572997375023003741, 0.9844559584073944158, 0.7236650551455482416,  0.9959361374956100521, 0.9486198679962674074, NA, 0.5389398202405594196,  0.6795571754333266101, 0.9305472410112699544, 0.2271830381140543809, 0.3674567008633543574,  0.0114282981936605632, 0.4105340631759704650, 0.6192195626290996069, 0.3915707154406132262,  0.0684476674892871007, 0.5215867150135459207, 0.9809122912886170109, 0.6192094341270485414,  0.6977359835925305287, 0.4483565977456888674, 0.9159143781924314887, 0.8165039291637901098,  0.5617394325782397724, 0.8467895412642084674, 0.4620165910760486994, 0.4913454627214262627,  0.7855792545677589001, 0.7427225439854699962, 0.3050730660318403675, 0.8826613928654951646,  0.9812623758639205107, 0.7368547163721959059, 1.0146343109564153195, 0.9623329976235518446,                     NA, 0.5380900949874826145, 0.6779264019247748241, 0.9464584250532885878,  0.2103739241714049357, 0.3825880747785647684, 0.0154231746734327380, 0.4122070445944157058,  0.6447080020135382838, 0.4162746527395139173, -0.1094643431102914621, 0.7130379279215761645,  0.9846141536344410472, 0.6068936940850720996, 0.7309204423104932458, 0.4645772977909486956,  0.9302504930865599375, 0.8072649693601456899, 0.5563700906604909235, 0.8698452677392884969,  0.4757563526238194895, 0.4844132089814798814, 0.8099404711236571908, 0.7575164043558220150,  0.3173811236937940095, 0.9175273161126296095, 0.9860702435267350019, 0.7593001238992732604,  1.0399665500900967530, 0.9851125425218104636, NA, 0.5429741691390492786,  0.6825356364479971960, 0.9708466717517647560, 0.3591750054240411827, 0.4061352726387843504,  0.0280984762475568750, 0.4236127796872680706, 0.6795676541210371457, 0.4496540504891561318, -0.2208252049627673907, 1.1363892490243781008, 0.9971818998818803292, 0.6007498310632737315,  0.7721380284488321699, 0.4893082411553317801, 0.9536643670363390557, 0.8026619951123025176 ))

  set.seed(999)
  suppressWarnings(tmp <- ddhazard_boot(fit, do_sample_weights = T, do_stratify_with_event = T, R = 30))
  # plot(fit)
  # get_expect_equal(tmp, file = "tmp.txt")

  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c(-2.9417270151733201, -2.9417271494125758, -2.8751008812968308, -2.7944202142630039, -2.7218473374240366, -2.6454718499478842, -2.6548810638159783, -2.7347206446107339, -2.8005733099400301, -2.8613648474096909, -2.9213549135070527, -2.9859723293007789, -3.0339289593190788, -3.0675486510431185, -3.0755737654547737, -3.0985890531657061, -3.1251804112205708, -3.1437771523950744, -3.1430265035546250, -3.1483871572853541, -3.1376621000064713, 0.4404171871819423, 0.4404171192811067, 0.4736959887657881, 0.5200446109080480,  0.5521174112845471, 0.6022336481103565, 0.6068304383076504, 0.5726439210576278, 0.5629964410440592,  0.5469270404499240, 0.5294419316856761, 0.5071540820900921, 0.4977363624237865, 0.4999564634380535,  0.5146366934172963, 0.5098673080230457, 0.5031829300055697, 0.5042201332917474, 0.5139403746897799,  0.5206467192162000, 0.5345147175479759 ))

  expect_equal(unname(c(tmp$t)),
               c(-3.16854430720318447, -2.96808455563678297, -2.93508380767045995, -2.99912817133640619, -2.81159492574507341, -2.83461938345575115, -2.95873610250501828, -2.96007788990117904, -3.05268237210650106, -3.01568288614035396, -3.02256627060218630, -2.84618397136823242, -2.93136326040002437, -2.83943465549750185, -2.90458422629431157, -2.89061341159735319, -3.05049614462486574, -2.81177370775273650, -3.07748366333583245, -3.01249981610069151, -2.90768642847751746, -2.87837141728142365, -2.87191728474086361, -2.80130427835999285, -2.92559428369930519, -2.79274918661565685, -2.65837419823345700, -2.85066149234685051, -3.04436041352953701, -3.11159676646947281, -3.16854446868543871, -2.96808467751182370, -2.93508394015364615, -2.99912824347485341, -2.81159520431886767, -2.83461960437617133, -2.95873626669048129, -2.96007803836107097, -3.05268240742304764, -3.01568289989867200, -3.02256627143865231, -2.84618414705473999, -2.93136333064511900, -2.83943510926995035, -2.90458444632526014, -2.89061368857854406, -3.05049634722705587, -2.81177398217479935, -3.07748365754760123, -3.01249974001799270, -2.90768649551353864, -2.87837175118952793, -2.87191743257136967, -2.80130447338732314, -2.92559447380770354, -2.79274965101592088, -2.65837421257124262, -2.85066163196565592, -3.04436037059550824, -3.11159676146996222, -3.09943978350997185, -2.91375351361768198, -2.87916817648080192, -2.92591759143622010, -2.73819396894466571, -2.77175851175672294, -2.88415008159476294, -2.89842246468721321, -2.99226802656187507, -2.95567984808903628, -2.94615817976334071, -2.77973955762929714, -2.87162064485816559, -2.77959571105472314, -2.84919755173900313, -2.83644350596534256, -2.98272834439056389, -2.74453350456162326, -3.00368692746676924, -2.94903552424170945, -2.85252102287204812, -2.80257728390943139, -2.82018044612921992, -2.72542901602803189, -2.84536173962081218, -2.70802736148191991, -2.58746184845127836, -2.78892464666757212, -2.98145473842164677, -3.05719471455444625, -3.01266368529604733, -2.84788531362875696, -2.80440176629340154, -2.85476804376841953, -2.66902719156460932, -2.71230214490497312, -2.78112371713822526, -2.80790494750392217, -2.90866297854150169, -2.87541769206081810, -2.85680605595708936, -2.71096607114825439, -2.80157055847433689, -2.69073404075646527, -2.77831428610753051, -2.75646570570722282, -2.90647753620211624, -2.67884046327440162, -2.93551625457630694, -2.88428479799411619, -2.79090070936592305, -2.70671439937318947, -2.75606441305724603, -2.63089199535690232, -2.75277766632592424, -2.62885807449045128, -2.48275118141929552, -2.73713665183494204, -2.88321151723400160, -2.98908559818911268, -2.95592082032690628, -2.74832687257666120, -2.72667985995928897, -2.77175542359802218, -2.60037370876086227, -2.63262593425294433, -2.72891424736805899, -2.72421281636977763, -2.85327052132255554, -2.84234839282697482, -2.78856131941584851, -2.64887759709025250, -2.72594168753384070, -2.59292462997890860, -2.70914808561697118, -2.67514081922404667, -2.79208703501046385, -2.62947776010163636, -2.87426497309145645, -2.83528914429522771, -2.72332776204878346, -2.62196399805053471, -2.68837237115743610, -2.54726245814038732, -2.67624404482476042, -2.55531954811138506, -2.43000256615489985, -2.68404217306038273, -2.79316502766128050, -2.91566274028957473, -2.88785915718475916, -2.63925069822555125, -2.64031450271625845, -2.69118645387927202, -2.56427181098481771, -2.56224592593829437, -2.68878136012670499, -2.65595886717821195, -2.78387639578943968, -2.77590935376175052, -2.71619575023578008, -2.59409341253453540, -2.65803782162154656, -2.51303989399373862, -2.61604684164444290, -2.57655545587932533, -2.66985902549966392, -2.53871510824775948, -2.78731544113962348, -2.79354110565949920, -2.65167803529563040, -2.53348424907609182, -2.62434456930739657, -2.47741891921936741, -2.59713264844549485, -2.50083549349671186, -2.36359611940650760, -2.62043240363072094, -2.67815639526342952, -2.86092142540385330, -2.89008593940460656, -2.64149114434143595, -2.66106223017016363, -2.69551518399985701, -2.58591392189670000, -2.56558260704705621, -2.69870449933571432, -2.67382332636389686, -2.78113472926584349, -2.76859636800811071, -2.71814310401137105, -2.62034845467305777, -2.69447608152991291, -2.52457952039435218, -2.62266020253050947, -2.55837068712960392, -2.70690121531370398, -2.53705832389983765, -2.77330872816147878, -2.78298467954753814, -2.64893537422400893, -2.55208546292300609, -2.64078214004657807, -2.47768284262121918, -2.62138325969358021, -2.52330039825843677, -2.38045616770630897, -2.65527150249573163, -2.68260590010202327, -2.85920407551689815, -2.94636515739217453, -2.71704721694332862, -2.73128087159154109, -2.77050624860873551, -2.66777302691203877, -2.64846284386318809, -2.80825057142079482, -2.74992370287929333, -2.84934549791112124, -2.84562371286789340, -2.79838223715005396, -2.69791986815488061, -2.76340989900045653, -2.60986646233894870, -2.72351008071104905, -2.63796941937257579, -2.79355008540406802, -2.61702079261460163, -2.83129928999752467, -2.85177932862856309, -2.72216954602133487, -2.65173133981052178, -2.70017731896397217, -2.56669275977187761, -2.70513377366907992, -2.62805504709037807, -2.47738515227403377, -2.73811423849205093, -2.79760061376837132, -2.93675994106720140, -2.99116121790855605, -2.77645449374368170, -2.78464901162433964, -2.82864850441033688, -2.73375854970506271, -2.70330348013695243, -2.88575621345661082, -2.81599968653673560, -2.89489668668124667, -2.91865829010327182, -2.86664269949826611, -2.76076010674885985, -2.82619994241516004, -2.68268018152489907, -2.79429323883915925, -2.72242262253129885, -2.85788663749693539, -2.68626798762564878, -2.88402503157539636, -2.89834289999254624, -2.79129721051167312, -2.72891637900934647, -2.76530654291850864, -2.65503849069515940, -2.78040217634908426, -2.71902147003473793, -2.56755244936146498, -2.80291815354150398, -2.87072401658904752, -3.01625998499472647, -3.02230602212408783, -2.83358114956923535, -2.83632839420391614, -2.89499124602532731, -2.79181343655122749, -2.75638491240316963, -2.94211897554846225, -2.87097829301355789, -2.95378926251522111, -2.97292858063353194, -2.93364152869527883, -2.80658610531487396, -2.88032764192655621, -2.74842713028149976, -2.88018658138588535, -2.80322842809237605, -2.90955690719235438, -2.75127136665509120, -2.93523237964153827, -2.94263891620277906, -2.84335452115892373, -2.78086294926625399, -2.83237346340693374, -2.73054238976605657, -2.85607232418925516, -2.78947907858493416, -2.63090283430395777, -2.86917654509560549, -2.93808812797572694, -3.08355036393836723, -3.07217383381310061, -2.89249699231006874, -2.88211461908758082, -2.94938344921025486, -2.85010855676966468, -2.82324049286424339, -3.00255820205057278,
                 -2.93742798254321924, -3.00986855503706963, -3.02875945007777103, -2.99457770975137105, -2.85589763292678445, -2.93153839125127424, -2.81358728737881947, -2.94468889979922999, -2.86610625603674496, -2.96589132340951300, -2.81318094975674038, -2.98923681334550695, -2.99202502890029676, -2.88933676848924792, -2.83556642969266104, -2.88520252024132295, -2.79752505159446185, -2.93549736335152023, -2.84900296600939429, -2.69690569674339642, -2.92780522773521357, -2.99222396335492702, -3.13323373617452861, -3.14238941132467309, -2.95967893770257673, -2.95167821779890849, -3.01428497572702847, -2.91334860430776876, -2.90240842213647454, -3.08054914047268413, -3.00312293473840564, -3.08033392615738100, -3.09643428802678322, -3.06321620807460748, -2.91190115573028230, -2.97258945825109233, -2.88313331347736090, -2.99795060420836545, -2.93803197320607801, -3.02154689237408558, -2.88275848092021780, -3.03366151935087114, -3.02927607750144556, -2.92426065867742091, -2.90486305996864669, -2.93768580293619852, -2.85385838621780108, -3.00481207708128029, -2.93384833590830940, -2.75912941096084063, -2.98486150005969675, -3.06303553943440443, -3.17730885140355790, -3.18651099185860787, -3.00844324993580781, -3.00729246906750225, -3.06501006324858860, -2.97186060576348909, -2.95596089817754093, -3.13396034526777667, -3.04832713380557330, -3.12559438776600196, -3.13812951173227228, -3.11511787734004830, -2.95592838850083339, -2.99590904677375525, -2.93955832615324963, -3.03739839236482023, -2.99127381801926662, -3.07088617923190954, -2.92712875822366447, -3.06457596183334546, -3.05690781000950418, -2.94083938205005246, -2.95586611684600742, -2.96987549747160839, -2.87598425587743822, -3.04782264737986797, -2.99997168958816829, -2.79861652677922157, -3.03064769297890857, -3.12245604816185907, -3.19892521377288253, -3.20168029136517651, -3.03322266069712931, -3.04752799585780654, -3.09722956428277385, -3.00714335144914902, -2.99193781759609267, -3.16321927534221414, -3.08771500710723501, -3.15811916558951289, -3.16499061890227562, -3.15037153767276124, -2.99268877407976230, -3.00446122643491886, -2.97342392661842370, -3.07235446717645022, -3.02761131503553660, -3.11560708619168469, -2.96581073852646737, -3.08801521847365734, -3.06865629216405633, -2.94743721013491955, -3.00190261810884529, -2.98169741949783651, -2.88800174618208993, -3.07468103647361835, -3.04958480225139095, -2.83310962793028098, -3.06004411026202172, -3.15702441052295768, -3.20011475132981982, -3.18721004502780980, -3.03099516306042904, -3.06152677593345857, -3.10320348418714920, -3.01777814008091205, -3.00276939042680491, -3.16726871134551224, -3.10292738962588954, -3.16699382936437424, -3.16689124076182971, -3.15877425052920291, -3.00552257396012834, -2.98503658731956634, -2.98148674277733639, -3.08230397098275688, -3.03769502530550106, -3.13503225382982631, -2.98053634482798957, -3.08554151181444647, -3.05320674741456788, -2.92654255200286784, -3.02201858181839533, -2.96646815783555073, -2.86683347230192576, -3.07328026202314897, -3.07413897949183301, -2.83943002366922670, -3.06559070148537316, -3.16248648404477484, -3.17441175817158916, -3.20077564794479352, -3.03852428476599545, -3.09769970463212685, -3.12266004166247679, -3.06038336805468170, -3.03086067725486075, -3.18130848258217380, -3.13186389527200193, -3.18151491996225300, -3.17408820093204902, -3.18721144322427019, -3.02814096665536336, -2.98625710202555306, -3.00182186239749704, -3.11995412940216665, -3.04654021842078571, -3.16999882759708473, -3.00069412227684618, -3.09290964563625215, -3.06091736815709803, -2.93015772059678881, -3.05109832371371148, -2.97079671236840825, -2.86651900849205443, -3.10963807354285215, -3.10720648415428657, -2.87530195677771205, -3.07702563271168339, -3.19310516106009290, -3.17922827267468344, -3.23203948692618015, -3.06018909778776660, -3.13740876488195841, -3.14706978940414217, -3.09196492828155600, -3.06503590201516385, -3.19957510516347110, -3.16261792408438636, -3.18962972694645819, -3.19391348273108999, -3.20371083736567641, -3.05749721555571297, -3.00529705694490135, -3.02682679567343360, -3.14168809038547181, -3.05916493359406561, -3.19319897329940039, -3.02439828798669952, -3.11769747857861201, -3.07981000423714502, -2.95141203475754299, -3.08710089715990588, -2.98644522599183526, -2.87993377159102604, -3.14782186231933725, -3.14330679378102484, -2.89771044261924482, -3.08700940332126406, -3.22675636150523637, -3.20190494159699313, -3.25467422905198411, -3.08243412644462866, -3.17855333985825972, -3.17105571845950962, -3.10484676379870983, -3.08001621169633655, -3.21931633950505702, -3.18494421401873140, -3.19515996187872675, -3.20691009105775304, -3.21687982965559227, -3.08008414428448063, -3.01943718279409357, -3.04709379235341249, -3.15712281302119457, -3.07059216400079116, -3.20514476899832657, -3.03852642259643391, -3.13698162923858614, -3.09636937193826078, -2.97061835669872920, -3.11511555485136915, -2.99873192033999425, -2.88742217996638928, -3.17566970086630640, -3.18034861808853275, -2.91736819465154218, -3.09810932155931074, -3.24519985130158339, -3.22184018263928884, -3.25638923987305784, -3.08488134492242194, -3.20223097455849448, -3.17574540131220218, -3.09845204743073310, -3.07557454268058406, -3.21983676915523986, -3.18923576474793924, -3.18135994859625182, -3.20063896265658343, -3.20915593913761832, -3.08450927120522334, -3.01229949642833805, -3.04805387534971217, -3.15372500620834462, -3.06057053801766532, -3.19707105636703615, -3.03393152365193508, -3.13732044519619846, -3.09264509298781798, -2.96970321341752497, -3.12416509926151109, -2.98985445580297338, -2.86766006575932630, -3.18403274141555137, -3.19930902677838125, -2.91514807395818565, -3.08995406024339125, -3.24181031902232064, -3.22264888291181872, -3.26670926262505024, -3.10097241162701076, -3.21967460349812429, -3.17935780209960583, -3.08922608271627741, -3.07708709410585968, -3.22881468428632523, -3.20037763631322614, -3.18031540601287421, -3.21088754263692033, -3.21258704651139260, -3.09438691117793185, -3.00868176994239089, -3.05339038247419658, -3.16131646247979070, -3.05269362137206857, -3.19628983667101929, -3.04009028503751910, -3.14389675079458364, -3.09278013508649430, -2.98228662914506470, -3.13990665682505599, -2.98782728266889341, -2.86299372012461228, -3.19745800446412343, -3.21442224063449222, -2.92641386277827342, -3.09400424074302238, -3.22641835716143710, -3.23097578583225120, -3.26118763730617056, -3.10111710171788024, -3.22266273054158292, -3.16727460160525620, -3.06264779716992130, -3.06349753653624335, -3.22203239697763211, -3.19614569826347505, -3.16441249938907365, -3.20557120363015446, -3.19840577446448338, -3.08939320766827885, -2.98616264346467686, -3.04204491737987937,
                 -3.15364039911453720, -3.02705916531552433, -3.17862966419483062, -3.03078069269486328, -3.13654104818571433, -3.07554985496667888, -2.97878672577881254, -3.14061616999513182, -2.96732858633997676, -2.83542989996787353, -3.19479346479385695, -3.21402362602560743, -2.92036631147686121, -3.08236710277867942, -3.19002399030150485, -3.22316608415127348, 0.62375725621562927, 0.40074026624945963,  0.42257562161075979, 0.47469962686991762, 0.27409256552214600, 0.24370145532561216,  0.59023566450061304, 0.49868209298719707, 0.56530686220633442, 0.62849642353632917,  0.53181111665897263, 0.32440295075754955, 0.40181299341210730, 0.31907145743046506,  0.48019785773380663, 0.31662038927062036, 0.59712997249827915, 0.29469323065253028,  0.59917659874196116, 0.40128601050508100, 0.29660949662505648, 0.33460798116243590,  0.27780962528109149, -0.01391833757539517, 0.40515453229523385, 0.30871762024445487,  0.02042862137193890, 0.37734786348737215, 0.49507605569387741, 0.67508377766233008,  0.62375723997012733, 0.40074020438890084, 0.42257556996949547, 0.47469952853746628,  0.27409269525007596, 0.24370154771790475, 0.59023541926088341, 0.49868194412886124,  0.56530671425893264, 0.62849623807251898, 0.53181090717359814, 0.32440299216923224,  0.40181299031874973, 0.31907157386321883, 0.48019776642793788, 0.31662040550839965,  0.59712963259413510, 0.29469329852797732, 0.59917642058196985, 0.40128608379709468,  0.29660957235486540, 0.33460795553845218, 0.27780971561939749, -0.01391846595775031,  0.40515448251871289, 0.30871764707828225, 0.02042851130278073, 0.37734787273650278,  0.49507578237945660, 0.67508358150738257, 0.66314322825182948, 0.42543134266478566,  0.44695663938042907, 0.51522415366797003, 0.30906703225288462, 0.27265362936187693,  0.62834913901669254, 0.53081703554637549, 0.60010678240585080, 0.65722588400269621,  0.57371671972003901, 0.35873697422435702, 0.42860561610458187, 0.34130689074021536,  0.50267053852812027, 0.33709373911874857, 0.63467069381108221, 0.32598564019140996,  0.64260893923716877, 0.43574702516815866, 0.32067776573680107, 0.37344570986443204,  0.29655327284207511, 0.02016802585579425, 0.44787922399471697, 0.35042641882877196,  0.04545319078173601, 0.40625429396556317, 0.52672622055376195, 0.70422070931348291,  0.71211882715969532, 0.46364679928270625, 0.49051612312596299, 0.55717953550420418,  0.34476511265959275, 0.30714811318667351, 0.68491876382048789, 0.58279638798549427,  0.65750286279771963, 0.70532651325270046, 0.61827974791153317, 0.40277459233656404,  0.46902773596860181, 0.37680717814323716, 0.54219603855134646, 0.37774110782688131,  0.68383521638898470, 0.36579344596290203, 0.68541473405904962, 0.48290578227958608,  0.36068089630482913, 0.43193994685941783, 0.32615337226061569, 0.06982367030406716,  0.50113631020258764, 0.39146621631884709, 0.09076349547636975, 0.43960104276531414,  0.57878563727135068, 0.75008538459067298, 0.73377638465011552, 0.51476841255059036,  0.52987866277897544, 0.59773638634512183, 0.36381761683942754, 0.34769373577467777,  0.70192744165454246, 0.62294005620418080, 0.69000676561189489, 0.71498719780570985,  0.64768594491895870, 0.42847263881783598, 0.49726448052449640, 0.40926707494887299,  0.56876474903265761, 0.40712144256142291, 0.75717757041991685, 0.38341528482610765,  0.71972427924248794, 0.50548725125205141, 0.38741399120550069, 0.46757933650829653,  0.35170121754261252, 0.10102011822230422, 0.53472732813809543, 0.41670119310010845,  0.09362651026505894, 0.46036514582954952, 0.62564636964702958, 0.79389840350606300,  0.78499832410288684, 0.58600799438610618, 0.57797468995834567, 0.66206657008933778,  0.39172151512080050, 0.40197714346832381, 0.73094867071059433, 0.67647128196827711,  0.74814559893166455, 0.75916504162057685, 0.69139334749776626, 0.46727371610322799,  0.53462658707124677, 0.45836064005609400, 0.63079372867593642, 0.46395570667607899,  0.83817944059845173, 0.43220192285543990, 0.77094443968059612, 0.54336351165602781,  0.43229620727092710, 0.52844437894328689, 0.38757512518878612, 0.14245963565267949,  0.58021055658293497, 0.44781382141247961, 0.11879987004264682, 0.50437166531990041,  0.69705928083693469, 0.83978027265062372, 0.79704272718489388, 0.59475122232199928,  0.57167485248613870, 0.67211685205311134, 0.40009673096536497, 0.41003984414429306,  0.75281606211682972, 0.67212245476542798, 0.75393045874177034, 0.76825141740315872,  0.69253166007948308, 0.46021278546413119, 0.52102379158586887, 0.47078645524310925,  0.65426382341928602, 0.49139669627184868, 0.82721467383549530, 0.45011672870109143,  0.78193265229330100, 0.56688975142125830, 0.43240389850977301, 0.52558354745248637,  0.39340440578726510, 0.14861092000325357, 0.59036371928160825, 0.44173940420529123,  0.10879632834698136, 0.49345844707826503, 0.70398012767506157, 0.84653621794003886,  0.77679675693536765, 0.56200773139668603, 0.53961034584957568, 0.64437965087400872,  0.37251004735389526, 0.37806144025865984, 0.71448689676776977, 0.62838478082921800,  0.71725259790989571, 0.73718302191424612, 0.65467524138293509, 0.41631972867149303,  0.48908881598732745, 0.44762066032478942, 0.61572824859527353, 0.45437145819646518,  0.77342212624371454, 0.42017645126726388, 0.74680563533406152, 0.54828014046246420,  0.39631955326891799, 0.47471947601258463, 0.36896564477695376, 0.11044311885704122,  0.56770464846748614, 0.39814994587311392, 0.06906097883627027, 0.46388390909631316,  0.66267705217941619, 0.79549516415279220, 0.78609955863898795, 0.55583742445510254,  0.53219433380484904, 0.64472917670633445, 0.36817903192886048, 0.37231993862463286,  0.70028284863662660, 0.60988314499608265, 0.70972807003232696, 0.72833626536671536,  0.64077956537130487, 0.39551771748916753, 0.47703755153062183, 0.44816442676427926,  0.60185304533321793, 0.43850239450736650, 0.74625913383906373, 0.41448650377015750,  0.73603059989914665, 0.55648569896756217, 0.38075545952970613, 0.44701230778677625,  0.36437975345525375, 0.09242823627297231, 0.57312009155531196, 0.37601974440554087,  0.04857173804108667, 0.45796831331553878, 0.65062144896555996, 0.76607180498574634,  0.79781590536926128, 0.54071264600185553, 0.51411163665645832, 0.62945933528444176,  0.35902276421617352, 0.35676245254552541, 0.69254649314189176, 0.59077963866619965,  0.68221238950317242, 0.72500906057027736, 0.61715337535343673, 0.37768430232413280,  0.46003944632547045, 0.44298115524080761, 0.56506618976749479, 0.41459747177632883,  0.71746927878883748, 0.40258203698690753, 0.71564697434289370, 0.55688211460053094,  0.36844392359985945,
                 0.42667778971257186, 0.34661910452699640, 0.07374265831332757,  0.57103671110947540, 0.35896663530346501, 0.03540610169891242, 0.44139965494893529,  0.63322887858173160, 0.73816202707150025, 0.79485185582313811, 0.51405368428051312,  0.50120942539290381, 0.61487604609086977, 0.35532549863725149, 0.33347415744996278,  0.68170066818394837, 0.56137312084957258, 0.66436607755992982, 0.71665839759477545,  0.58904353816936261, 0.35697028232618871, 0.44648070466596373, 0.43673080387897489,  0.53553101145421056, 0.39393339638068225, 0.68482525099730329, 0.39105258620369732,  0.69457938954044762, 0.54552959983671734, 0.35784695868669691, 0.40685953784251638,  0.33752751541944892, 0.05425282900690688, 0.56128822144824386, 0.34386797972209365,  0.02524202883895388, 0.42679856893033252, 0.61844853787270804, 0.71641846719617119,  0.77886038312281758, 0.48628655733841197, 0.47514681921398905, 0.58999304154342369,  0.34843292249208824, 0.30580652218120413, 0.66540932877623149, 0.53126290352714256,  0.62907260507268004, 0.70331893027148717, 0.56088150334056697, 0.33115907895895769,  0.43756493782925654, 0.42134014703668077, 0.51463954700371728, 0.36517341864210179,  0.65903814674915451, 0.36850222687304113, 0.67341738029908138, 0.53468274643222924,  0.34633750371146221, 0.37457344518853541, 0.33453947269826034, 0.03776599006492869,  0.55375815812101037, 0.31080769385692719, 0.01583950360588687, 0.40818018884647839,  0.59657603900395384, 0.70185582283884440, 0.77981244068844746, 0.47209728421916031,  0.46179202093395311, 0.57894239064746933, 0.35416225876503449, 0.29172761319419782,  0.66211991556283434, 0.51507067052663558, 0.60792038818747396, 0.70312183971941711,  0.54552114944776275, 0.31784274773193538, 0.44128562123977200, 0.41781623218621888,  0.50628144934406927, 0.34852002875728361, 0.64562027305478198, 0.35849337401103798,  0.66617276106349621, 0.53773305914807179, 0.34898837862914855, 0.35564548895396347,  0.34450528748567977, 0.03785026526816887, 0.56147884275986870, 0.28931757711897199,  0.01912094470728010, 0.40142576617900638, 0.58921241328940144, 0.70228622372218497,  0.79629674482752300, 0.47068376363004483, 0.46051214836705096, 0.58082898538120664,  0.37199383667240821, 0.28959646067622619, 0.67086991636836057, 0.50992660420123070,  0.59857575462694856, 0.71412634107939055, 0.54241349768793112, 0.31564303670513638,  0.45714539391863640, 0.42514051573730216, 0.50797335530304810, 0.34273569963592598,  0.64332103352655512, 0.35871992720740897, 0.67032470981922598, 0.55420966286293094,  0.36352486975529680, 0.34713733617421716, 0.36665489880921120, 0.05117279965707977,  0.58301823875157144, 0.27864783877171861, 0.03287497912054099, 0.40578992691591775,  0.59663520789869740, 0.71632888500953484, 0.82828271999508685, 0.48224907967879110,  0.47207432818057188, 0.59618881294691584, 0.40179991652345065, 0.29989109071380826,  0.69160053717341685, 0.51742474138180783, 0.60191843152050073, 0.73694929129539577,  0.55237230671520143, 0.32577821232168230, 0.48620887881835217, 0.44339112480629361,  0.52153946221736680, 0.34856628947266061, 0.65445629257796056, 0.37043040999388765,  0.68746686805299495, 0.58485049910240217, 0.39137363947854981, 0.35113241042028109,  0.40135467186135598, 0.08037060894068565, 0.61913994643033732, 0.27938242149694137,  0.05888046316786189, 0.42168086900589230, 0.61909938997121294, 0.74463580533979756,  0.83105214624302393, 0.47771622563919136, 0.45853070096675230, 0.59354330179730097,  0.39940256270103736, 0.28922466621102694, 0.69712396445027325, 0.50758812857671354,  0.59413519578375262, 0.74748512212900997, 0.53860605310680509, 0.32120270438467247,  0.48926956658138010, 0.44361912342879550, 0.50574203412153662, 0.34661994170833960,  0.64638553087493023, 0.37069391693818521, 0.68850780923323485, 0.58795794689850223,  0.39036752031104116, 0.34039645579468164, 0.41105207080194750, 0.08558671472790702,  0.61942517361881289, 0.26625235551330040, 0.05640743237526748, 0.42552762301666225,  0.61485412230467551, 0.74034628547415482, 0.82501475604179442, 0.46324457298913629,  0.44056076840040870, 0.58607570587538516, 0.40487629859791008, 0.28285123967153164,  0.69744721241452357, 0.49779188042952571, 0.58986959660327298, 0.74641487817626007,  0.53044804971158188, 0.31488515690189695, 0.48652124413838405, 0.43666709513331337,  0.49686170361462811, 0.34409387367288624, 0.64691572726593716, 0.36290776891657722,  0.68048853290103140, 0.58373249567178620, 0.38133977369656447, 0.32303137606631604,  0.41083475449262657, 0.08280041430622953, 0.61189292378848592, 0.25079357926654910,  0.05631543579127379, 0.42438628007811136, 0.61017779090486679, 0.72847836073868655,  0.82795675308618311, 0.45550182601671174, 0.42873572703020163, 0.58619386894104275,  0.41883916316611947, 0.28564927540108442, 0.70457229235584162, 0.49594658036284556,  0.59394506286079241, 0.75255402924873716, 0.53003009661920331, 0.31666376203812041,  0.49181117032691463, 0.43582485539433635, 0.49477020558460955, 0.34883884085657801,  0.65635869716574957, 0.36252952002671884, 0.67991892838928236, 0.58802049374580334,  0.37990815226700586, 0.31285270814220595, 0.41838337790761715, 0.09030957985979048,  0.61246996370090634, 0.24142264504418887, 0.06354948423513335, 0.43046377829271032,  0.61527191130499115, 0.72424624999502962, 0.84081586331526281, 0.45649543019881511,  0.42458258895590179, 0.59554535386761132, 0.44126170258203345, 0.29758965850240165,  0.72016882315098885, 0.50294851615531677, 0.60804867978948773, 0.76680441341044925,  0.53905457031889548, 0.32744927713419725, 0.50647003136745616, 0.44214494670507698,  0.50066083246994653, 0.36286481359231004, 0.67579597663726021, 0.37026138396573116,  0.68798823204226445, 0.60203273419278025, 0.38747444333598108, 0.31101610096815285,  0.43490857756880330, 0.11059851503204987, 0.62175495521471502, 0.23984823869235256,  0.07967521643340796, 0.44530218291602952, 0.63078672169744743, 0.72918052472595629,  0.84366823569932925, 0.45457481819264756, 0.42088289175387467, 0.60310273471620035,  0.46025362923000568, 0.30585875979028287, 0.72387885068700153, 0.50313164593151660,  0.61789918247968445, 0.77073891864637822, 0.54792809279521848, 0.32881350316348823,  0.52247360272854559, 0.45118326782879625, 0.50515929115952996, 0.38121547258131466,  0.68646322497470780, 0.37581124033229485, 0.69367570556716185, 0.61453369192973062,  0.38938397800318897, 0.30915709980803729, 0.45036347664251036, 0.12564287288588288,  0.62243713674518752, 0.23863330245108683, 0.08286449063269355, 0.45255273628687498,
                 0.65488657516155069, 0.73323446713628582, 0.85333748512788998, 0.45984534003177896,  0.42350408572273240, 0.61785934423582800, 0.48684362999400299, 0.32127729695335261,  0.73440323459262069, 0.51095804085833185, 0.63532843999780009, 0.78097513862519152,  0.56482744451959443, 0.33744218628401879, 0.54713195865368724, 0.46632759132938384,  0.51636867034878331, 0.40721337726665474, 0.70556380267907659, 0.38822729723156507,  0.70514893424589409, 0.63506613587904415, 0.39825199732667504, 0.31380238783364051,  0.47372793095537608, 0.15172091997810555, 0.62990218012377319, 0.24400544212225200,  0.09267057317356760, 0.46697519092406659, 0.68896635280040941, 0.74501304578572036 ))
})

test_that("Boot gives previously computed values with timevarying covariates",{
  fit <- ddhazard(Surv(tstart, tstop, death == 2) ~ age +
                    log(albumin) + log(protime) + log(bili), pbc2,
                  id = pbc2$id, by = 100, max_T = 3600,
                  Q_0 = diag(rep(10000, 5)), Q = diag(rep(0.001, 5)),
                  control = list(save_risk_set = T, save_data = T))

  set.seed(999)
  suppressWarnings(tmp <- ddhazard_boot(fit, do_sample_weights = F, do_stratify_with_event = F, R = 30))

  # plot(dd_fit)
  # get_expect_equal(tmp, file = "tmp.txt")

  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c(-11.05250337824802465, -11.05250352469279917, -11.04508999427243232, -11.03079218210795354, -11.01276689498567407, -10.97180707356202767, -10.92868943459950515, -10.90098502723408558, -10.86985710925975468, -10.84982541828216718, -10.82653501151556341, -10.80572600014983564, -10.77373825434438714, -10.73524462930758183, -10.70591598692485391, -10.66720973370452441, -10.65998486910317666, -10.64608628893297393, -10.64680077372383948, -10.63576684813315154, -10.63361042480508445, -10.61687960464793434, -10.60104166304420303, -10.58927465566920034, -10.58443158792417371, -10.57623508244723176, -10.56319725965373912, -10.55230844022145398, -10.54592620274321035, -10.53960130032549003, -10.53665901819950612, -10.53041595305994349, -10.52510593969067543, -10.52505810336422165, -10.52727056580709508, -10.52837999933849922, -10.53342995392302583, 0.05296958898534846, 0.05296958905563476, 0.05415849402328772,   0.05309799072773889, 0.05327594826658033, 0.05104031066007266, 0.05053463696140417, 0.05133502712341615,   0.05213837065912127, 0.05261984592138964, 0.05264115568159874, 0.05080471436273008, 0.04938715582283902,   0.04766643888761872, 0.04588737086129124, 0.04447490675834360, 0.04477538409203886, 0.04467320797311228,   0.04451161946271013, 0.04370458232740190, 0.04408756070392547, 0.04470681659873713, 0.04423196799915047,   0.04453525705758350, 0.04543917052864419, 0.04721251203076599, 0.04888155727856350, 0.04815360863535564,   0.04869375124859262, 0.04842177412108322, 0.04858862179933521, 0.04943215584087465, 0.05038472787081971,   0.05165423949341428, 0.05234389971935057, 0.05364525710387259, 0.05600856244422565, -4.30686031039544037,  -4.30686033866322493, -4.28911812805663928, -4.27998912162376222, -4.26876503528201745, -4.19916544913661038,  -4.11221120301745113, -4.04961138518570252, -3.96364464853105281, -3.90951366273536394, -3.88425038595311189,  -3.83562755297431313, -3.75715042968144397, -3.70593275221107188, -3.63175497236033129, -3.54073945601396334,  -3.48890563835667411, -3.44833071631922117, -3.42631784404413819, -3.37191034007683976, -3.35812106341563288,  -3.31782911141499159, -3.30587318478505177, -3.27812439292552549, -3.24873043138299522, -3.20762588814112126,  -3.13636229932190158, -3.09659228749524873, -3.03798213884600976, -2.99469934639876056, -2.97373260897097325,  -2.95958975059265939, -2.95827222219042296, -2.99268351257244003, -3.01924136612427807, -3.04687672446439661,  -3.05065564335752359, 3.31864494491752815, 3.31864501612206597, 3.32609854359188972, 3.30588493258488381,   3.27259320775837326, 3.26794688321532467, 3.23337113422625322, 3.18067910501152218, 3.09072238149282486,   3.01345224060827999, 2.96755490883330930, 2.95972027067869092, 2.93977593563432205, 2.94219505205174414,   2.92326723828454060, 2.90901668611108644, 2.90126135891327008, 2.89732029823756898, 2.89760749424391362,   2.89505496127566975, 2.89351873839219564, 2.88121781279513955, 2.87745119001530814, 2.85615666345754526,   2.84511940653338957, 2.83143267284266420, 2.81785700089585989, 2.80444779531331756, 2.78275861269943725,   2.77701079671141393, 2.77765154508227008, 2.78227415883856244, 2.79480371926767868, 2.81180828891713741,   2.81443534114923422, 2.81609089945707813, 2.80460170174301870, 0.99194883257467559, 0.99194882894273229,   0.95856807961592283, 0.99902565656941378, 1.04011724688152807, 1.02440686902098155, 1.02810372591729982,   1.05013451503190458, 1.13254883267763273, 1.18633909796471015, 1.24757108028956654, 1.26817152654242049,   1.27485163969115001, 1.25863528878724895, 1.27084749931237551, 1.25798711924296081, 1.19128180739154543,   1.15089054247972755, 1.11611654351599121, 1.08520744458319229, 1.05353951495120324, 1.02758650157878861,   1.03566385260869986, 1.06065000568720036, 1.04293331650105259, 0.99989795301664408, 0.93962408658826890,   0.94297104440748714, 0.92088077832370374, 0.90363057994208040, 0.88604372273376464, 0.86473878872740251,   0.83405917696158494, 0.82327669479966514, 0.83300016580390523, 0.82406112838505341, 0.79124136299044501 ))

  expect_equal(unname(c(tmp$t)),
               c(-13.587819621696180405, NA, NA, -10.545246481320971199, NA,                     NA, -6.774268461636745897, -9.249431337425210842, -10.982376792590780923, -9.854388731740293395,                     NA, NA, NA, -5.162241984920820315, NA,                     NA, -11.605318771714383530, -12.944635091910502211, -10.755005951368501016, -6.989747314437287251, -12.585567473675212113, NA, -12.292168841765953147, NA, NA, -10.363277308918803499, NA, NA, -10.445206086312559535, NA, -13.587819599598450537, NA, NA, -10.545246538360622424, NA,                     NA, -6.774268486001650658, -9.249431393385259881, -10.982376815302700379, -9.854388812084380334,                     NA, NA, NA, -5.162241941453997285, NA,                     NA, -11.605318816537483428, -12.944635207444335379, -10.755005991712033975, -6.989747318246234187, -12.585567408532513767, NA, -12.292168840600657731, NA, NA, -10.363277276918335446, NA, NA, -10.445206210540925795, NA, -13.579307894065401285, NA, NA, -10.544970886929736054, NA,                     NA, -6.763824725929200810, -9.414398758412005463, -10.963480451394037374, -9.846302498293336924,                     NA, NA, NA, -5.134970484068780650, NA,                     NA, -11.602134992924375112, -12.938007781024900567, -10.737063716966027016, -6.967771087235892224, -12.583355457288355694, NA, -12.291627188093990242, NA, NA, -10.351558900145036546, NA, NA, -10.434422358279149989, NA, -13.554965760852716983, NA, NA, -10.547381381229630648, NA,                     NA, -6.733348876886113388, -9.146155525609405146, -10.948243987569142632, -9.848583049688041768,                     NA, NA, NA, -5.065825933243878865, NA,                     NA, -11.605409103001184334, -12.917062410619333690, -10.723886644785569189, -6.962417112327973001, -12.581371753778890721, NA, -12.269899194427285494, NA, NA, -10.326933315174306571, NA, NA, -10.404637310405059125, NA, -13.534834098914942047, NA, NA, -10.533245114425437094, NA,                     NA, -6.707611565385656505, -9.193171131980722777, -10.951984545803203019, -9.838422403393966675,                     NA, NA, NA, -5.291272178571439611, NA,                     NA, -11.605384057433196787, -12.874902775385965725, -10.700186764384511662, -6.958858411481577022, -12.560378545670182859, NA, -12.237630703007116750, NA, NA, -10.293280885372620759, NA, NA, -10.369149886575922181, NA, -13.508817473349113314, NA, NA, -10.497587340731364236, NA,                     NA, -6.665207812603069826, -9.080157916242097116, -10.925034126378992738, -9.796895176862943089,                     NA, NA, NA, -5.217723199543796042, NA,                     NA, -11.577332385451921226, -12.802986967218059178, -10.644380560301843985, -6.931342773518833766, -12.504045802234376339, NA, -12.189229301812517292, NA, NA, -10.247155260389082088, NA, NA, -10.311650499999521102, NA, -13.487987165550016755, NA, NA, -10.456336340007288399, NA,                     NA, -6.617812081132512958, -8.944052788082359484, -10.903913062953781932, -9.755321486925929264,                     NA, NA, NA, -5.210542841517451507, NA,                     NA, -11.537962206072339910, -12.733129187151341455, -10.582514292686500568, -6.903911242302651630, -12.442010360226515076, NA, -12.138475530562120142, NA, NA, -10.194408873374275259, NA, NA, -10.252858368227446562, NA, -13.480760155168786341, NA, NA, -10.418254997677486529, NA,                     NA, -6.583149547183973382, -8.736206590345862111, -10.905286016436948060, -9.711487012542590946,                     NA, NA, NA, -5.268918412023253062, NA,                     NA, -11.508152121302376969, -12.687887282610628148, -10.540299192819492902, -6.876487217468444157, -12.426499773836642504, NA, -12.116576960265469154, NA, NA, -10.152799914644909052, NA, NA, -10.225237259534686984, NA, -13.481139170889571588, NA, NA, -10.397526126128235902, NA,                     NA, -6.556594751341418892, -8.324975639176020792, -10.915309636748387945, -9.652797970398911076,                     NA, NA, NA, -5.559521792838925336, NA,                     NA, -11.467195828476928199, -12.661384121299052552, -10.496040237560375630, -6.845284002788886824, -12.388756891588375453, NA, -12.101668615530579487, NA, NA, -10.107155241003553670, NA, NA, -10.188562464957719556, NA, -13.481041133919376307, NA, NA, -10.393222974236921985, NA,                     NA, -6.542021390415750481, -8.330151449114062245, -10.949482705818670780, -9.605634492617809883,                     NA, NA, NA, -5.504796364977538659, NA,                     NA, -11.439164496910843027, -12.635297079069857773, -10.464163661368631963, -6.816020661952794768, -12.366627850278383960, NA, -12.094205602142958256, NA, NA, -10.097056202500887423, NA, NA, -10.190906178082538958, NA, -13.481084918388770433, NA, NA, -10.387798964814844993, NA,                     NA, -6.502297768602613104, -8.073741664461898893, -10.998550331533349933, -9.541001358145976496,                     NA, NA, NA, -5.631934594128740201, NA,                     NA, -11.418532801926382092, -12.620112998534589011, -10.418730997130541738, -6.798354483664969194, -12.354920507526669482, NA, -12.088018187965474226, NA, NA, -10.085912109592431563, NA, NA, -10.183056447460256067, NA, -13.454002746702631299, NA, NA, -10.380877360997471825, NA,                     NA, -6.457664762213230603, -8.239585042133986903, -11.034595884447956138, -9.481410284334312877,                     NA, NA, NA, -5.564997829604697799, NA,                     NA, -11.395699114048754907, -12.620922441104474032, -10.392961743707809674, -6.799332603477887815, -12.325932298256802255, NA, -12.086501055074245770, NA, NA, -10.091545466400022590, NA, NA, -10.169126167910894409, NA, -13.427174675582275398, NA, NA, -10.367475630278455867, NA,                     NA, -6.403588048590723503, -8.284741411838211889, -11.051951787456985699, -9.398458070774450945,                     NA, NA, NA, -5.507778536168091321, NA,                     NA, -11.364698727981208393, -12.631048242686157224, -10.365722577957791728, -6.788423236666442762, -12.295653935112188293, NA, -12.088504691631396071, NA, NA, -10.081653220165451401, NA, NA, -10.153672115273918664, NA, -13.418112310128593734, NA, NA, -10.336768822762511633, NA,                     NA, -6.351266640819268616, -8.214946796865060463, -11.044099175644975119, -9.314549460651329937,                     NA, NA, NA, -5.470897194621153758, NA,                     NA, -11.329100029702129149, -12.625521907790046683, -10.323213097480056888, -6.746626858691979756, -12.256428619422779391, NA, -12.081857101432838419, NA, NA, -10.077289415205111567, NA, NA, -10.123248184803108174, NA, -13.400573735850839441, NA, NA, -10.327349133349059684, NA,                     NA, -6.313551514570879597, -7.804548218679110860, -11.036932246427570092, -9.250633813230042080,                     NA, NA, NA, -5.542339915360336100, NA,                     NA, -11.342680384601218080, -12.619203235746221381, -10.271653254325343596, -6.706329543372691049, -12.238311757014166403, NA, -12.081458745519226383, NA, NA, -10.072854963954746665, NA, NA, -10.084591331669772174, NA, -13.375899864302175146, NA, NA, -10.312128301854338375, NA,                     NA, -6.261812531486746103, -7.686112523413124009, -11.023456059121464534, -9.204282744793275128,                     NA, NA, NA, -5.592949385462559597, NA,                     NA, -11.329325590791743394, -12.612970994991592022, -10.212299379105550301, -6.663827775976590040, -12.219602188021603695, NA, -12.055527731464280805, NA, NA, -10.054166037447991755, NA, NA, -10.035397189511453320, NA, -13.377342529794891135, NA, NA, -10.310939600591360943, NA,                     NA, -6.258044436063078919, -7.938658077931760815, -11.012070576297183777, -9.210149943773064507,                     NA, NA, NA, -5.419537986701471688, NA,                     NA, -11.340276908563579283, -12.625209498484503357, -10.226474355031623276, -6.658239591373057209, -12.215592140212365280, NA, -12.066851619777455795, NA, NA, -10.047474831168340614, NA, NA, -10.012353723734323196, NA, -13.362579148060497403, NA, NA, -10.325187537212137556, NA,                     NA, -6.239666296407285095, -8.078608008238209237, -10.991583505855661684, -9.200943102589288713,                     NA, NA, NA, -5.317784548997658511, NA,                     NA, -11.350930211193784203, -12.628747794981784125, -10.241750065421324933, -6.643781354120200433, -12.212692930536835689, NA, -12.080913478712771436, NA, NA, -10.044953512435013110, NA, NA, -9.990865151064195260, NA, -13.360501724974124471, NA, NA, -10.338406259504198559, NA,                     NA, -6.247770788931948260, -8.072575275419728413, -10.988676717068937450, -9.228976164479469446,                     NA, NA, NA, -5.276429500169882658, NA,                     NA, -11.376366441244657679, -12.632504727622817953, -10.269852826797238876, -6.636886973547787782, -12.232610247106414647, NA, -12.114872041061401475, NA, NA, -10.059657512360480780, NA, NA, -9.977254592671451761, NA, -13.354651623690338980, NA, NA, -10.347798129367660636, NA,                     NA, -6.235851066538769416, -8.000220766936077865, -10.980344746266064604, -9.239140627529517147,                     NA, NA, NA, -5.293720693991260440, NA,                     NA, -11.395147890622386200, -12.621119292637843046, -10.284097443373410030, -6.624927473160887814, -12.247766156154787254, NA, -12.137135148207537583, NA, NA, -10.059055288069497891, NA, NA, -9.947205795312543231, NA, -13.357460347545284662, NA, NA, -10.351741730749493442, NA,                     NA, -6.241929839778354250, -8.083380369227841200, -10.982079282034606038, -9.261642920135081170,                     NA,
                 NA, NA, -5.239212172442545778, NA,                     NA, -11.431084398982060080, -12.630069112135258180, -10.302365822315991295, -6.613290087386926253, -12.258177676095391107, NA, -12.163000687099092545, NA, NA, -10.073083367478636418, NA, NA, -9.964887737708203019, NA, -13.350889456512813780, NA, NA, -10.341982316418933507, NA,                     NA, -6.235645367302476316, -8.092845868155754019, -10.975336263657462510, -9.275363101129281418,                     NA, NA, NA, -5.174274817511252955, NA,                     NA, -11.455247234190217398, -12.625644702174771083, -10.307947736303715303, -6.589532757180684541, -12.249990617718527375, NA, -12.178259510372800634, NA, NA, -10.069437834731070680, NA, NA, -9.959498295592595696, NA, -13.339275126362705848, NA, NA, -10.327870575618764093, NA,                     NA, -6.225434482580952888, -7.963676891400296221, -10.955170819715652542, -9.287583853983944593,                     NA, NA, NA, -5.169333663036389304, NA,                     NA, -11.472538665152063331, -12.631857025445535569, -10.310110346947777060, -6.577389441018229199, -12.252282185030672679, NA, -12.203348082095201832, NA, NA, -10.059409333581768919, NA, NA, -9.967044380753504740, NA, -13.334647352585564661, NA, NA, -10.304486127202112655, NA,                     NA, -6.200192886132777303, -7.689863204260705132, -10.948354884158627343, -9.304645637449294071,                     NA, NA, NA, -5.314681158415144324, NA,                     NA, -11.491038008214774280, -12.633229430700271934, -10.304958067811629263, -6.567879773877873362, -12.270930343974960408, NA, -12.227736925612056496, NA, NA, -10.075766432332795119, NA, NA, -9.962019632891156107, NA, -13.338426020385512061, NA, NA, -10.308444817726655884, NA,                     NA, -6.209319076474787202, -7.832903000739019461, -10.948236126551087466, -9.328465067846883230,                     NA, NA, NA, -5.351166153571003115, NA,                     NA, -11.506176663310313302, -12.638957947110085911, -10.309571917381434858, -6.569900642152430414, -12.293137620533377330, NA, -12.252022517142215463, NA, NA, -10.084662356452879450, NA, NA, -9.968382485135986570, NA, -13.342002366277103675, NA, NA, -10.308705277784286025, NA,                     NA, -6.209079034263924157, -7.852007105525363073, -10.944912783969432013, -9.347723917250149839,                     NA, NA, NA, -5.338069932878189761, NA,                     NA, -11.517971659974680421, -12.642489401478012923, -10.307892951704550555, -6.571393134459521335, -12.310480881294207478, NA, -12.274965503426987112, NA, NA, -10.085659300618825540, NA, NA, -9.966890157631963731, NA, -13.340959797063039360, NA, NA, -10.301812217071487865, NA,                     NA, -6.199342344447696718, -7.948470397024507861, -10.939377108354403134, -9.353230008799998885,                     NA, NA, NA, -5.149679692798117436, NA,                     NA, -11.528062294120019260, -12.643516453202314764, -10.300554034647049662, -6.563815450886406389, -12.322198046266390037, NA, -12.293691531465432476, NA, NA, -10.083062618648790831, NA, NA, -9.964118036388700972, NA, -13.336580312511939539, NA, NA, -10.299963179013685632, NA,                     NA, -6.181035581615184071, -7.746914159993309568, -10.928756431890157330, -9.355362873872337559,                     NA, NA, NA, -5.113549854410588402, NA,                     NA, -11.533366210721590406, -12.649663026286923184, -10.297687269308649860, -6.555229329082009393, -12.325229070482485838, NA, -12.299514833907913669, NA, NA, -10.079662722401016239, NA, NA, -9.959646753202832770, NA, -13.329854052538852471, NA, NA, -10.292632063031692979, NA,                     NA, -6.172987033655194189, -7.846627077195516442, -10.929611550851559443, -9.352203704232636028,                     NA, NA, NA, -5.102253315441901194, NA,                     NA, -11.534915994289455909, -12.651831632728990584, -10.289878886229653077, -6.542256914823709302, -12.333802651004596385, NA, -12.310145836643782147, NA, NA, -10.066636497719988697, NA, NA, -9.957253562014683368, NA, -13.330559049528840632, NA, NA, -10.286363863330338475, NA,                     NA, -6.160628394714428246, -7.942085596353610555, -10.930205025534149854, -9.351376991362336355,                     NA, NA, NA, -5.081475605578092747, NA,                     NA, -11.540001136853158314, -12.656995840923185526, -10.283580023021386296, -6.531235980694286347, -12.344598947343712325, NA, -12.320500817200132460, NA, NA, -10.053215066193081029, NA, NA, -9.953317865949840026, NA, -13.330288888684604487, NA, NA, -10.289870467088700678, NA,                     NA, -6.151783247732064197, -8.048831358872659436, -10.927133288943439382, -9.354869433156197900,                     NA, NA, NA, -5.095182972423049073, NA,                     NA, -11.542368547248296196, -12.658967606641796877, -10.273646395732352943, -6.536712352344969723, -12.352487621962099595, NA, -12.335456438365602239, NA, NA, -10.030833108340491577, NA, NA, -9.960192806321847669, NA, -13.329782349445681078, NA, NA, -10.289496002293740062, NA,                     NA, -6.138005684126999206, -8.189640090355119639, -10.921703658763584954, -9.355295061913121657,                     NA, NA, NA, -5.139452249240424209, NA,                     NA, -11.542537978871127535, -12.658498498785835551, -10.261210126778676610, -6.538714730660428742, -12.358105622784682609, NA, -12.347967894730622618, NA, NA,  -9.999782988526092709, NA, NA, -9.961537499869150381, NA, -13.331730091134829763, NA, NA, -10.293038999368851805, NA,                     NA, -6.118689664917066118, -8.375661146998606199, -10.924802798395610637, -9.358870013904537544,                     NA, NA, NA, -5.092336814763638309, NA,                     NA, -11.545458510260035112, -12.659784040692800389, -10.252416624025258329, -6.540408955299116478, -12.364781471592818818, NA, -12.357035467225651715, NA, NA,  -9.972647156670184643, NA, NA, -9.962650947582423200, NA, -13.341235267204798021, NA, NA, -10.301308597201069617, NA,                     NA, -6.108471519784144377, -8.331085804418723484, -10.927936931527044706, -9.360846595637843492,                     NA, NA, NA, -5.141415088625416629, NA,                     NA, -11.547722955128826072, -12.659246990494306573, -10.240765940856579874, -6.546727655850582472, -12.371717616769082326, NA, -12.364573735242712971, NA, NA,  -9.961844835230877493, NA, NA, -9.969297092385572157, NA, -13.348502781952603868, NA, NA, -10.306463490429342400, NA,                     NA, -6.103671190866829299, -8.262749300838999389, -10.933266944365570339, -9.364239042856357997,                     NA, NA, NA, -5.234701649621783304, NA,                     NA, -11.549814244745405745, -12.662273038262528146, -10.230394471354626873, -6.555798450018044399, -12.377832626655846937, NA, -12.373984622837800984, NA, NA,  -9.952593979290266901, NA, NA, -9.978852235670787607, NA, -13.355241615621116225, NA, NA, -10.310942306762630238, NA,                     NA, -6.096165835307593817, -8.217416096646193324, -10.939541734308226850, -9.366037945239050089,                     NA, NA, NA, -5.235439088654968209, NA,                     NA, -11.554302208643433048, -12.666005532393779021, -10.212667300682117144, -6.562026418708614806, -12.385694262302932245, NA, -12.379791341415755213, NA, NA,  -9.937496125227507804, NA, NA, -9.988696538270444236, NA, -13.361439589668480110, NA, NA, -10.317363595584636116, NA,                     NA, -6.100098054547362203, -8.188132984294050232, -10.950003188201584692, -9.372581593507328890,                     NA, NA, NA, -5.009867311199169393, NA,                     NA, -11.558305631845662731, -12.673318812788654242, -10.220492415083112903, -6.567713494634578097, -12.393737332303265575, NA, -12.384462474199404269, NA, NA,  -9.941265025150080703, NA, NA, -9.997784505824732548, NA,   0.080098329478164504, NA, NA, 0.046696639462552092, NA,                     NA, 0.051037975671545881, 0.075490269784538849, 0.048017297665966352, 0.039622882834189264,                     NA, NA, NA, 0.021008504418855545, NA,                     NA, 0.039904191308153480, 0.061754554511508089, 0.060488939144720255, 0.034904289035988946,   0.050915000111280584, NA, 0.054527493014739088, NA, NA,   0.045716088785303276, NA, NA, 0.067393304690826078, NA,   0.080098329985386046, NA, NA, 0.046696639877114261, NA,                     NA, 0.051037976306606628, 0.075490271141427590, 0.048017297323172184, 0.039622882811291990,                     NA, NA, NA, 0.021008504951978021, NA,                     NA, 0.039904191454706729, 0.061754554551302687, 0.060488939587063903, 0.034904288438547801,   0.050915000036232665, NA, 0.054527492987012108, NA, NA,   0.045716088765925769, NA, NA, 0.067393305232577791, NA,   0.082448555421892486, NA, NA, 0.048654102468267804, NA,                     NA, 0.052049466222688412, 0.087193269865881579, 0.047672102447555618, 0.040878715621299459,                     NA, NA, NA, 0.087505933898440330, NA,                     NA, 0.041196336703302076, 0.062867406769221548, 0.060114904562287373, 0.034892573277434724,   0.052610678757840840, NA, 0.056025266233489343, NA, NA,   0.045525395441159147, NA, NA, 0.067590955477275533, NA,   0.078634446709614669, NA, NA, 0.049194844535813000, NA,                     NA, 0.050328052050117963, 0.075875757767716906, 0.046670030972493434, 0.039846399685406757,                     NA, NA, NA, 0.066098342344738700, NA,                     NA, 0.042949488789065855, 0.061467232530118286, 0.059284583014918013, 0.035468678393011238,   0.052461548571868737, NA,
                 0.055293224007341851, NA, NA,   0.044527671839169981, NA, NA, 0.066357307028266255, NA,   0.074772775809044928, NA, NA, 0.050756934851266436, NA,                     NA, 0.048415974545299167, 0.078450909967575283, 0.045833448500418220, 0.041390148393291484,                     NA, NA, NA, 0.050456700383369912, NA,                     NA, 0.046067500326266830, 0.062000964498306124, 0.059073733671869459, 0.036846440414361518,   0.054819212380663278, NA, 0.054958496846094052, NA, NA,   0.044300168554278034, NA, NA, 0.066665997583369108, NA,   0.073183484875191884, NA, NA, 0.049204500281451846, NA,                     NA, 0.044878092976338303, 0.070695242845572201, 0.042244928673699819, 0.039902620101456232,                     NA, NA, NA, 0.023817716287917561, NA,                     NA, 0.044229927677155313, 0.058109646002368974, 0.054976564910653808, 0.035752268329672446,   0.053467799060410985, NA, 0.053625183313078487, NA, NA,   0.041993773813128375, NA, NA, 0.063481504740443478, NA,   0.073964533934661861, NA, NA, 0.049060738554263739, NA,                     NA, 0.042801897308526550, 0.063888757662228277, 0.040498744478803174, 0.040398850821074229,                     NA, NA, NA, 0.009139026103293355, NA,                     NA, 0.043501046145505928, 0.056340601924213404, 0.052105318849039226, 0.036191009586215288,   0.054325418137649358, NA, 0.054026652057734742, NA, NA,   0.040991165663075493, NA, NA, 0.062002037357181496, NA,   0.076506289517177278, NA, NA, 0.050415731672051919, NA,                     NA, 0.042350531316956236, 0.055513751346169433, 0.040379744730106693, 0.042282112090740599,                     NA, NA, NA, 0.042551617372512757, NA,                     NA, 0.043534190238195769, 0.056427047168882707, 0.052339990038664741, 0.038415589163340091,   0.054779484755661564, NA, 0.054736035890593643, NA, NA,   0.040774548883699469, NA, NA, 0.061976257833895682, NA,   0.079347730859704729, NA, NA, 0.052267241870810069, NA,                     NA, 0.043321543443781606, 0.039350843612894405, 0.041980561525584104, 0.042823307670482424,                     NA, NA, NA, 0.061105720001913791, NA,                     NA, 0.045318876938643643, 0.058360071602287768, 0.052344056231528616, 0.039211875768622027,   0.053222246553642771, NA, 0.057169166063453139, NA, NA,   0.041562797079310813, NA, NA, 0.063144914687773779, NA,   0.077586289209934406, NA, NA, 0.053620770922204131, NA,                     NA, 0.041021820737370325, 0.042315780486283851, 0.045614346425549956, 0.041416533589378038,                     NA, NA, NA, 0.047097167517916511, NA,                     NA, 0.046970574256588689, 0.057686052970215271, 0.051967946376015761, 0.039380082360561522,   0.052531227988156362, NA, 0.056356907512672270, NA, NA,   0.040779623885716654, NA, NA, 0.065611914455445633, NA,   0.076347950577279111, NA, NA, 0.053381375769520956, NA,                     NA, 0.037402263213638105, 0.034600589123331374, 0.049581081555757238, 0.038393129331009732,                     NA, NA, NA, 0.045046283891231609, NA,                     NA, 0.048026750633810088, 0.055523190864179679, 0.049252808339169590, 0.039775323715585585,   0.052684988111359340, NA, 0.054447578331656352, NA, NA,   0.041838449167476630, NA, NA, 0.065870149486463969, NA,   0.072331267476105660, NA, NA, 0.050801724155909334, NA,                     NA, 0.033155623343870289, 0.045085448349568978, 0.051043848841655179, 0.036702522096474954,                     NA, NA, NA, 0.030615463479963696, NA,                     NA, 0.046303993239024385, 0.052011857878944315, 0.045537324921935907, 0.039021700379649429,   0.048748504266580284, NA, 0.051491702525585707, NA, NA,   0.040924002284938678, NA, NA, 0.063663515755305378, NA,   0.070034479512601153, NA, NA, 0.049851998337567593, NA,                     NA, 0.032337376826273884, 0.049077506004029055, 0.053094504785065721, 0.034960697242316457,                     NA, NA, NA, 0.041218463141729764, NA,                     NA, 0.047187569365215253, 0.050184568750557809, 0.042391173362932189, 0.038262823566765469,   0.046763664577701203, NA, 0.049750402958581380, NA, NA,   0.041438065096935636, NA, NA, 0.063914021301214985, NA,   0.068644937095835051, NA, NA, 0.048014670257595954, NA,                     NA, 0.030227160899822650, 0.045759628358070722, 0.052728083324966979, 0.030610108946394831,                     NA, NA, NA, 0.038523673019266291, NA,                     NA, 0.046972594355668321, 0.048298380839489258, 0.039448777087723361, 0.036193072844929744,   0.044561150437484584, NA, 0.048865888553511087, NA, NA,   0.040622753785514373, NA, NA, 0.061474692179577003, NA,   0.066651652871718384, NA, NA, 0.046379914484627485, NA,                     NA, 0.028590581203496682, 0.025649594662894425, 0.052136059859383133, 0.025841988027543589,                     NA, NA, NA, 0.013396615781677976, NA,                     NA, 0.046725431862681750, 0.047453695022716402, 0.035925952779518162, 0.034311154790109433,   0.042783204067020984, NA, 0.049408290872815661, NA, NA,   0.039843872288384076, NA, NA, 0.059172948599362674, NA,   0.064927613770342479, NA, NA, 0.045704337456636018, NA,                     NA, 0.027876555647216455, 0.020899004686802315, 0.051259275769716409, 0.025061835429902434,                     NA, NA, NA, 0.040534979833825702, NA,                     NA, 0.043267920955398241, 0.048532711491049381, 0.033305240566247285, 0.034092674201549947,   0.043646457884499598, NA, 0.047548970243825837, NA, NA,   0.039249777663289576, NA, NA, 0.058129729652962120, NA,   0.065050795100798725, NA, NA, 0.044412574240838248, NA,                     NA, 0.028708590414864534, 0.033677175193549277, 0.049941473624153340, 0.026846350926767678,                     NA, NA, NA, 0.031056872727587625, NA,                     NA, 0.043592907234932275, 0.050227085258630289, 0.035936958884785153, 0.035279726664106156,   0.045572025701849990, NA, 0.047095066298213861, NA, NA,   0.039005231338935811, NA, NA, 0.056171458155017770, NA,   0.062891170707644883, NA, NA, 0.044053960018631572, NA,                     NA, 0.028703271131925334, 0.041065517730729674, 0.048292630227360703, 0.028666433451478589,                     NA, NA, NA, 0.036642134961936877, NA,                     NA, 0.043739735849597915, 0.050673703602979889, 0.038580452177828989, 0.035405564368849111,   0.046740866676782962, NA, 0.048643040705105105, NA, NA,   0.038854550569705930, NA, NA, 0.055113924278624375, NA,   0.060726622021237520, NA, NA, 0.044799041070122891, NA,                     NA, 0.028755569418773663, 0.040451919956872986, 0.046727741173028192, 0.031018243331040522,                     NA, NA, NA, 0.030414398920545715, NA,                     NA, 0.042632402635458963, 0.052389403181389821, 0.039619529310946554, 0.036356950919425930,   0.045168723549485236, NA, 0.051599915309090867, NA, NA,   0.038066042702468529, NA, NA, 0.052216008778441354, NA,   0.059441405372001217, NA, NA, 0.046286779690961968, NA,                     NA, 0.027453142667231277, 0.038065438735747734, 0.045699981592020864, 0.030652830795547644,                     NA, NA, NA, 0.012367075859832143, NA,                     NA, 0.042329032798661834, 0.051959997329288264, 0.039242668834637943, 0.036892467505406853,   0.043596338151583217, NA, 0.052145148619176580, NA, NA,   0.036523522799029665, NA, NA, 0.048761205784079831, NA,   0.060278334936240519, NA, NA, 0.048120830403278581, NA,                     NA, 0.027946343326210186, 0.042242242871507740, 0.045693927569859016, 0.032056602795381417,                     NA, NA, NA, 0.024387345830466110, NA,                     NA, 0.042734530939014664, 0.052824474106555150, 0.039978990989510257, 0.037328810847454132,   0.042783671452356412, NA, 0.053308829224401333, NA, NA,   0.036018600474216358, NA, NA, 0.049241263322814859, NA,   0.061464610890645588, NA, NA, 0.049748023758640729, NA,                     NA, 0.028664907321902185, 0.044202712795395671, 0.045726765581280797, 0.034082406157317710,                     NA, NA, NA, 0.095488560323547328, NA,                     NA, 0.043528594435184295, 0.053995423963510612, 0.040467955948726937, 0.037725949173058541,   0.042077325886073561, NA, 0.054994241690692766, NA, NA,   0.035479878541659564, NA, NA, 0.049883989016506800, NA,   0.062119882673317521, NA, NA, 0.051219310067355042, NA,                     NA, 0.028204872250931629, 0.038538207033707757, 0.044928053452396630, 0.035290808880353283,                     NA, NA, NA, 0.071748910704452301, NA,                     NA, 0.043523947273934570, 0.052148505140114043, 0.040775029626009505, 0.037290711109282448,   0.040841556688826428, NA, 0.055704046324398890, NA, NA,   0.035038650465106375, NA, NA, 0.049302501945192694, NA,   0.063559755442808133, NA, NA, 0.052690409472092498, NA,                     NA, 0.027643269403629718, 0.026391600533833583, 0.044750807154618474, 0.037393592939656217,                     NA, NA, NA, 0.052667538837432450, NA,                     NA, 0.044269320604244169, 0.050515836801846896, 0.041083756345505214, 0.037330634590330876,   0.040592409092310509, NA, 0.057232927989468642, NA, NA,   0.035866245552926453, NA, NA, 0.049284388290271791, NA,   0.063606279552081896, NA, NA, 0.054345851758503774, NA,                     NA, 0.029095825136195269, 0.034194758779478800,
                 0.045434601374099708, 0.039278113012006871,                     NA, NA, NA, 0.028147240970497991, NA,                     NA, 0.045263662492739276, 0.049976417199454254, 0.041512559375775603, 0.036978591780398050,   0.041954088748083103, NA, 0.058969854100700762, NA, NA,   0.037222441467144381, NA, NA, 0.050077779559553032, NA,   0.064826902070555970, NA, NA, 0.056547547868612477, NA,                     NA, 0.030821719194458343, 0.036602417597119047, 0.046647484016282506, 0.041529396702960880,                     NA, NA, NA, 0.021401637956938606, NA,                     NA, 0.046934422304092956, 0.050209832070266497, 0.041847738384541242, 0.037600354299628165,   0.043842087059045562, NA, 0.061674203669270400, NA, NA,   0.038913719871300836, NA, NA, 0.051838346078181517, NA,   0.065773923190153877, NA, NA, 0.058697164852716741, NA,                     NA, 0.032782971985388529, 0.042804099787624839, 0.047787166981228542, 0.043378128685435648,                     NA, NA, NA, 0.116017814937288313, NA,                     NA, 0.048603732947249165, 0.050273407360813351, 0.042280378926261658, 0.038182368449490141,   0.046242380878394322, NA, 0.065011530257771649, NA, NA,   0.040597389841699472, NA, NA, 0.053967107077748981, NA,   0.066435002125645792, NA, NA, 0.058333457620240420, NA,                     NA, 0.032252552026815276, 0.032763346983177996, 0.045987040361416487, 0.042560075908511442,                     NA, NA, NA, 0.075800581019396601, NA,                     NA, 0.049521680099885793, 0.049779017093507093, 0.040835355034818201, 0.036887206728534759,   0.045477529670777631, NA, 0.065354556014487489, NA, NA,   0.041109235183092986, NA, NA, 0.053237514778015332, NA,   0.067832330072987987, NA, NA, 0.058456163302834277, NA,                     NA, 0.033368638358787700, 0.037619842303721208, 0.045666043544973819, 0.042142400147471511,                     NA, NA, NA, 0.054395103244913159, NA,                     NA, 0.051049602617668513, 0.049595510194098115, 0.039714324889590466, 0.036170280560197034,   0.045974833122003574, NA, 0.066772149754251137, NA, NA,   0.042035410319696036, NA, NA, 0.054103969647238036, NA,   0.067871738022035166, NA, NA, 0.058707262001552046, NA,                     NA, 0.032740142146332141, 0.042262323510566913, 0.044377024896948170, 0.041808967795723941,                     NA, NA, NA, 0.030680525450000551, NA,                     NA, 0.051710021685761720, 0.049326874060434803, 0.037623641870357410, 0.035744058911631162,   0.045687480143246320, NA, 0.067288884340576324, NA, NA,   0.040477527247057069, NA, NA, 0.052761476558074123, NA,   0.068739137325516564, NA, NA, 0.059229506590161463, NA,                     NA, 0.032432839055316479, 0.048140909848957525, 0.043445404024106252, 0.041860526109369321,                     NA, NA, NA, 0.009353703706941070, NA,                     NA, 0.052957734763887529, 0.049374021840936540, 0.035893338206865424, 0.035726213125728941,   0.045927718781050390, NA, 0.068097176618933358, NA, NA,   0.039256113788594796, NA, NA, 0.052102513662889177, NA,   0.070628891631164620, NA, NA, 0.060475927899636399, NA,                     NA, 0.032463957865969062, 0.056342883420099493, 0.043213297295646665, 0.042447883308298649,                     NA, NA, NA, 0.002042122640976248, NA,                     NA, 0.054978224602234331, 0.049915107239523443, 0.034769403497049108, 0.036340770841264150,   0.046953994923188136, NA, 0.069341888689333886, NA, NA,   0.038411244555194991, NA, NA, 0.052268808067964682, NA,   0.072911315899752333, NA, NA, 0.061221814402181121, NA,                     NA, 0.032759559911408581, 0.067329657571252102, 0.044439888457629334, 0.044087505865322174,                     NA, NA, NA, 0.060980705244043254, NA,                     NA, 0.058103539458912334, 0.050377112602531963, 0.034660034513338177, 0.036225959951250331,   0.047448239554808647, NA, 0.070111873034234176, NA, NA,   0.037921209425986743, NA, NA, 0.052741565724254189, NA,   0.075859150772766740, NA, NA, 0.062272381556210336, NA,                     NA, 0.032325592125019391, 0.064915280055754102, 0.046501350411877515, 0.046352404142070147,                     NA, NA, NA, 0.049773076235293069, NA,                     NA, 0.062114762225291567, 0.051314773221190248, 0.035039091098272121, 0.036317364402573708,   0.049023807516599210, NA, 0.071358979767063069, NA, NA,   0.036986838417928665, NA, NA, 0.053876428559297057, NA,   0.077319871449409994, NA, NA, 0.062494249307368910, NA,                     NA, 0.032882381049883128, 0.061017728623256474, 0.047638031396101108, 0.047537456432464979,                     NA, NA, NA, 0.035896873856875476, NA,                     NA, 0.062473130191448935, 0.053384574134868912, 0.034640068146084245, 0.037164251071727057,   0.050299163625514301, NA, 0.073135379713737070, NA, NA,   0.036385717257315250, NA, NA, 0.054800931174830064, NA,   0.079678568639637293, NA, NA, 0.063432783631546627, NA,                     NA, 0.033884584736548672, 0.058754489344078759, 0.049667890743389974, 0.049241093400254302,                     NA, NA, NA, 0.109398251063420326, NA,                     NA, 0.063702892306602907, 0.056110038769860225, 0.034226079649863558, 0.038391534883903636,   0.052529795970101809, NA, 0.075007220127730923, NA, NA,   0.036096321969882189, NA, NA, 0.056660914334477175, NA,   0.082980495757321290, NA, NA, 0.065347373119379160, NA,                     NA, 0.036628337440449250, 0.057758505620796291, 0.052871434966645897, 0.052112376662199728,                     NA, NA, NA, 0.419512985080708334, NA,                     NA, 0.065672445921069009, 0.059767239621973889, 0.036655587490850364, 0.040209806550359953,   0.055582463051387379, NA, 0.077344461809386922, NA, NA,   0.038121649013686372, NA, NA, 0.059546440412891158, NA,  -3.087667960524590249, NA, NA, -4.277797269668612401, NA,                     NA, -5.103255234179911781, -4.325887876262100917, -3.754223669468521596, -4.165471181425991709,                     NA, NA, NA, -2.529281005304389929, NA,                     NA, -4.377904287904337721, -3.667867459452057499, -4.356885945568219398, -4.314577399817849823,  -3.458084153292806828, NA, -4.821888330884594787, NA, NA,  -4.186273764098943140, NA, NA, -4.660252393776854518, NA,  -3.087667945853662221, NA, NA, -4.277797355215787967, NA,                     NA, -5.103255287692773479, -4.325887904992048227, -3.754223641254041954, -4.165471186690806960,                     NA, NA, NA, -2.529280998964340110, NA,                     NA, -4.377904256049998999, -3.667867558001481676, -4.356885952299039744, -4.314577372436118274,  -3.458084144483343891, NA, -4.821888422180959033, NA, NA,  -4.186273805185472874, NA, NA, -4.660252561470317190, NA,  -3.086640633417693014, NA, NA, -4.284632319829862368, NA,                     NA, -5.093507662060622998, -4.428467289158130704, -3.721996711584802497, -4.163176604672185199,                     NA, NA, NA, -2.054258897296058262, NA,                     NA, -4.378884067035546224, -3.629658342985105257, -4.329726473203086456, -4.259718260493023934,  -3.454328912585306277, NA, -4.802342341939901615, NA, NA,  -4.168104040625538076, NA, NA, -4.641439095820206973, NA,  -3.125311040102607407, NA, NA, -4.315466691855039727, NA,                     NA, -5.044518190750311604, -4.387717186548253601, -3.725888619066077823, -4.202613358509752040,                     NA, NA, NA, -2.149224429677625725, NA,                     NA, -4.413401052059547247, -3.577563007063884193, -4.355087149289945891, -4.275415446407939690,  -3.468431395096453151, NA, -4.720522916100708954, NA, NA,  -4.156025369404550140, NA, NA, -4.560341961011203260, NA,  -3.161259500918721876, NA, NA, -4.325914457772224608, NA,                     NA, -4.997871292925287534, -4.415267381869414365, -3.721053726116411831, -4.231903677776113426,                     NA, NA, NA, -2.369299514768200776, NA,                     NA, -4.440393479906863838, -3.495572266068459388, -4.386597369353514964, -4.329267288228631827,  -3.468293858584218015, NA, -4.642120754699208263, NA, NA,  -4.136874272798737096, NA, NA, -4.456377180017334183, NA,  -3.137794746765736331, NA, NA, -4.246449948311165734, NA,                     NA, -4.900690130079848217, -4.209790022589739777, -3.643880547598111530, -4.196288347342364666,                     NA, NA, NA, -2.481419355311677677, NA,                     NA, -4.397048940613842660, -3.339601909102189570, -4.333031901190373958, -4.328354449485801503,  -3.425816899876055643, NA, -4.493900434881172856, NA, NA,  -4.053570550778763604, NA, NA, -4.324906502250041029, NA,  -3.085524978104131399, NA, NA, -4.147594845278613107, NA,                     NA, -4.801073277600771405, -4.029682624919900036, -3.555546626067020100, -4.143850020264422618,                     NA, NA, NA, -2.551130221527610598, NA,                     NA, -4.344658128728799085, -3.160625417298004614, -4.260473486075730953, -4.311863639798164449,  -3.364508073165363378, NA, -4.338414835739529352, NA, NA,  -3.956213468530231481, NA, NA, -4.171111921817590940, NA,  -3.030196074521071203, NA, NA, -4.040895435159660209, NA,                     NA, -4.744988789352586167, -3.896839475517516860, -3.509483508357030601, -4.096875112380510053,                     NA, NA, NA, -2.340868121711978223, NA,                     NA, -4.309146844501951179, -3.010829473793594779, -4.176822607019195388, -4.279111177652890063,
                 -3.427817523328637428, NA, -4.254309136553441562, NA, NA,  -3.895675409964962199, NA, NA, -4.064821628134261289, NA,  -2.959149479108255321, NA, NA, -3.974240315807237245, NA,                     NA, -4.717975376089458273, -3.753660999829611988, -3.462451755402881748, -4.006243192591185220,                     NA, NA, NA, -2.371057163302873860, NA,                     NA, -4.263869596420450492, -2.894652367146708372, -4.091553311741170873, -4.198306371192628283,  -3.418544786305168781, NA, -4.179368883002421597, NA, NA,  -3.824835303392013053, NA, NA, -3.920647523028504278, NA,  -2.933315744598170571, NA, NA, -3.947132399372400346, NA,                     NA, -4.694405834121552878, -3.675587596787194933, -3.495172900304836183, -3.910184106489924094,                     NA, NA, NA, -2.441445166339222261, NA,                     NA, -4.212383195263353031, -2.835432792052783757, -4.017713917645029831, -4.140369071030882253,  -3.420668497947742193, NA, -4.107565017999658785, NA, NA,  -3.881799642964544717, NA, NA, -3.906369635395473772, NA,  -2.912106617878873482, NA, NA, -3.954584533777834920, NA,                     NA, -4.670714485638533198, -3.696067802628881793, -3.614571066972101399, -3.795792891520227030,                     NA, NA, NA, -2.564412020913764412, NA,                     NA, -4.187833135638538096, -2.852603200401872652, -4.011001930179205921, -4.178589140935649482,  -3.454249792786907136, NA, -4.100230713468619115, NA, NA,  -3.982565596312141576, NA, NA, -3.945834220984884677, NA,  -2.879815455024886450, NA, NA, -3.955752249073365423, NA,                     NA, -4.631744359474410544, -3.678871212312353034, -3.652185811287747441, -3.678353511343643412,                     NA, NA, NA, -2.606643073519749443, NA,                     NA, -4.153669833787906995, -2.830961529280084044, -3.990242554665998753, -4.216228991706254092,  -3.449017007923223410, NA, -4.076902248367995085, NA, NA,  -4.005596704210771186, NA, NA, -3.938501279987078441, NA,  -2.854184566742375750, NA, NA, -3.917427781006763254, NA,                     NA, -4.553957480577568440, -3.605037819518283726, -3.646955611312057144, -3.519798402667433379,                     NA, NA, NA, -2.488658156618761996, NA,                     NA, -4.085342394129290788, -2.789388977003807391, -3.954867497821657363, -4.215943776198782800,  -3.413319695527469833, NA, -4.048519542284710226, NA, NA,  -3.996512317089674582, NA, NA, -3.891092741949720324, NA,  -2.905217091708827137, NA, NA, -3.869159795502994825, NA,                     NA, -4.496373035747298630, -3.528637338372747401, -3.653780417890340360, -3.413370217057143829,                     NA, NA, NA, -2.478354720144511969, NA,                     NA, -4.024664229987189756, -2.718652912082330619, -3.915700551068446167, -4.219857654606306241,  -3.380362398455797113, NA, -3.996848345405719982, NA, NA,  -4.046980665287371437, NA, NA, -3.848740480844211387, NA,  -2.913218180040041627, NA, NA, -3.822169786035147787, NA,                     NA, -4.439130240329507870, -3.362794909645509644, -3.628313683706546833, -3.303496117655965847,                     NA, NA, NA, -2.668664186241172853, NA,                     NA, -3.967805228620878921, -2.626983738022658610, -3.856503892663857513, -4.211065535299941764,  -3.339781577611705909, NA, -3.922449969496745403, NA, NA,  -4.063815598481025582, NA, NA, -3.761920493742736404, NA,  -2.907377333211743853, NA, NA, -3.756608696999240937, NA,                     NA, -4.356877409529301914, -3.219616997643493406, -3.601353672752195756, -3.189903006162895682,                     NA, NA, NA, -2.498582125232998052, NA,                     NA, -3.895846850620793056, -2.522638193443741184, -3.775703596065686174, -4.182527865896848773,  -3.274304405563627007, NA, -3.849154929652351065, NA, NA,  -4.066844712739239753, NA, NA, -3.637187440546675088, NA,  -2.913073545938258402, NA, NA, -3.726974258305849386, NA,                     NA, -4.302829130679970859, -3.150462887052984051, -3.569203999512409187, -3.165669427282387094,                     NA, NA, NA, -2.440305645502197862, NA,                     NA, -3.847942826265083216, -2.455097105192762807, -3.762310933194428664, -4.171864971630984797,  -3.217172643897716622, NA, -3.836381983894225467, NA, NA,  -4.081333308881242239, NA, NA, -3.575859827499205945, NA,  -2.939496988670344813, NA, NA, -3.711474454280718760, NA,                     NA, -4.253855698245612160, -3.111885401702111587, -3.538790504494500055, -3.151632742157917999,                     NA, NA, NA, -2.330864048196094718, NA,                     NA, -3.797383203254668160, -2.401228834061010176, -3.755642051965911499, -4.186577192249059998,  -3.175247328668441060, NA, -3.841878361036263456, NA, NA,  -4.109720916694415749, NA, NA, -3.503223745025230151, NA,  -2.976050767426393406, NA, NA, -3.690188795047108172, NA,                     NA, -4.228163880977650280, -3.070510459968566419, -3.514192810097930497, -3.162926658310709627,                     NA, NA, NA, -2.332486967342030315, NA,                     NA, -3.763277990489465985, -2.351625046860403856, -3.782168568088545513, -4.195653963500291184,  -3.208489339523668882, NA, -3.870243660769284055, NA, NA,  -4.170392591753904199, NA, NA, -3.466208476500312052, NA,  -2.985380879671787557, NA, NA, -3.642094188692944101, NA,                     NA, -4.131800387320370582, -3.018106403756139144, -3.466967876083019817, -3.134789034468710156,                     NA, NA, NA, -2.441288750829561760, NA,                     NA, -3.714372189275915837, -2.264008436564063143, -3.772627538019751814, -4.187987605398398649,  -3.231700142284834154, NA, -3.868572496624035573, NA, NA,  -4.188291418331978377, NA, NA, -3.393274219291840144, NA,  -2.997598871725525527, NA, NA, -3.586643149817256848, NA,                     NA, -4.082779913625507184, -3.135859747684574650, -3.464236020634902768, -3.151145703714674617,                     NA, NA, NA, -2.321789226277076512, NA,                     NA, -3.737234712278905135, -2.261190580561865726, -3.760617164699248782, -4.200064943596490075,  -3.248978034539514503, NA, -3.913211819357061749, NA, NA,  -4.292663212548986529, NA, NA, -3.435028241077722200, NA,  -2.978505858622552349, NA, NA, -3.499963637589056820, NA,                     NA, -4.027788406256100018, -3.197370292401243042, -3.463860022308371978, -3.163813658322447875,                     NA, NA, NA, -1.801373000106515798, NA,                     NA, -3.773927174614843061, -2.225616251327817263, -3.715330917291781443, -4.170840669287350089,  -3.224342336752184845, NA, -3.945897620920818749, NA, NA,  -4.363526926030854902, NA, NA, -3.408410940907762310, NA,  -2.954881582130625794, NA, NA, -3.409216703805607374, NA,                     NA, -3.993573378696214871, -3.237168990654184064, -3.462327530466767112, -3.203173010210825122,                     NA, NA, NA, -1.944197131604475048, NA,                     NA, -3.826230392054677854, -2.277777063154710291, -3.662005424316562507, -4.179858230660246754,  -3.269431450861428612, NA, -4.019519029500550289, NA, NA,  -4.438111563068871668, NA, NA, -3.433200747376270900, NA,  -2.905683297769198870, NA, NA, -3.310702312198736674, NA,                     NA, -3.949486030768860623, -3.257460917751795915, -3.436049280941964046, -3.228980086354349410,                     NA, NA, NA, -2.154704787975282532, NA,                     NA, -3.876362966905185115, -2.336006632005982464, -3.598692001810704255, -4.154908942603472433,  -3.297147033611049771, NA, -4.110146792957565687, NA, NA,  -4.470147374737067025, NA, NA, -3.432945484893024979, NA,  -2.867659172253418820, NA, NA, -3.230998625565769000, NA,                     NA, -3.943596630356130284, -3.305870067645181543, -3.409038199192250040, -3.259739495582798074,                     NA, NA, NA, -2.335587451946912196, NA,                     NA, -3.873457506742596479, -2.392089409388713950, -3.536366083755515533, -4.134207246356539578,  -3.336858194192093396, NA, -4.198681193459142591, NA, NA,  -4.484308938497182595, NA, NA, -3.443544952542646254, NA,  -2.825347312257898480, NA, NA, -3.132008776643944614, NA,                     NA, -3.929146031367435565, -3.346900014985951799, -3.372770391246125321, -3.282705682464214458,                     NA, NA, NA, -2.369069098673973972, NA,                     NA, -3.869094392141477101, -2.428247620296397180, -3.475161161141734567, -4.084815689556767104,  -3.383157617125613026, NA, -4.284580152878629811, NA, NA,  -4.483407262877589616, NA, NA, -3.447613154586320228, NA,  -2.750152620447756657, NA, NA, -3.007830324758052676, NA,                     NA, -3.902945303695902979, -3.307520671258295852, -3.325909397891723795, -3.227694185074001521,                     NA, NA, NA, -1.607638544800947145, NA,                     NA, -3.840186058065355290, -2.444753652026681046, -3.392793813717076112, -3.991661070375351450,  -3.413625399348700462, NA, -4.362396330359898933, NA, NA,  -4.462155089745555969, NA, NA, -3.424903849337275652, NA,  -2.663765381378293995, NA, NA, -2.948368716756515795, NA,                     NA, -3.872495511843105032, -3.269672071279924097, -3.313604566267385199, -3.216333601421328403,                     NA, NA, NA, -1.839577399552235759, NA,                     NA, -3.787852128784872985, -2.479241063368137876, -3.348824227449291424, -3.944664329812104242,  -3.433778660028854901, NA, -4.376546055232635624, NA, NA,  -4.453932050496563377, NA, NA, -3.419747757762940044, NA,  -2.577258138257134235, NA, NA, -2.884869193000090259, NA,
                 NA, -3.812624526089333887, -3.240196705670118416, -3.284636438779428680, -3.203051629159769487,                     NA, NA, NA, -1.976718568049154579, NA,                     NA, -3.737184768390237277, -2.509870735018528531, -3.304396709681894695, -3.892187370862430384,  -3.439973598075623418, NA, -4.371800201281410558, NA, NA,  -4.439320271457731337, NA, NA, -3.391730755229706240, NA,  -2.522774101618028730, NA, NA, -2.833459386317047191, NA,                     NA, -3.794948181909831408, -3.198057964209909620, -3.282392745281652147, -3.196806084053724906,                     NA, NA, NA, -2.125474868514144244, NA,                     NA, -3.698573418905759969, -2.546685344483126112, -3.270667809607340271, -3.843821538358206613,  -3.458149159371154635, NA, -4.389951977067514832, NA, NA,  -4.443792236377335669, NA, NA, -3.379076787277291771, NA,  -2.468449541395794977, NA, NA, -2.832074740269166835, NA,                     NA, -3.797685482072648444, -3.158335983725260920, -3.293370788008470385, -3.214038652842178934,                     NA, NA, NA, -2.280367484632888520, NA,                     NA, -3.660861400768007812, -2.580808399006829568, -3.238211511796737163, -3.866412807186885736,  -3.478847046491115869, NA, -4.430055243556790145, NA, NA,  -4.441916787427478752, NA, NA, -3.416683170500007360, NA,  -2.424245717151427559, NA, NA, -2.837345712695992450, NA,                     NA, -3.792769437340254512, -3.122518944202075986, -3.330074448721403790, -3.237859035024555965,                     NA, NA, NA, -2.364095554823601564, NA,                     NA, -3.635265315953731591, -2.620709531977187812, -3.225182210333806587, -3.886030078208972505,  -3.514776644190049204, NA, -4.467175476077483331, NA, NA,  -4.434041070252500383, NA, NA, -3.447296942075594117, NA,  -2.399076234262333518, NA, NA, -2.849739269140680431, NA,                     NA, -3.789566526012396963, -3.092479807362350286, -3.364705146548979897, -3.267913427394359971,                     NA, NA, NA, -1.954580244596331173, NA,                     NA, -3.612975318485202703, -2.669144384100352596, -3.227111435986937060, -3.916456271437777303,  -3.580281825673321983, NA, -4.516208746268422836, NA, NA,  -4.425534273947775432, NA, NA, -3.480586622458860635, NA,  -2.411581862853878544, NA, NA, -2.891507947745852736, NA,                     NA, -3.834758091602699981, -3.140478945319695292, -3.422806962740251091, -3.313126131695673937,                     NA, NA, NA, -2.073382660764914398, NA,                     NA, -3.604927388266220945, -2.726838121654358016, -3.257368492416314432, -3.969864591447876734,  -3.672669501493317501, NA, -4.573820036621531138, NA, NA,  -4.485167924573740805, NA, NA, -3.539145160029753434, NA,  -2.425248679542315600, NA, NA, -2.956402842034122624, NA,                     NA, -3.833171510261763615, -3.174141027263697179, -3.460663501150531562, -3.329761022535177695,                     NA, NA, NA, -2.240518375532333906, NA,                     NA, -3.652489433293598875, -2.774714671608648153, -3.292027174435369297, -3.990749749193867491,  -3.734958242108275872, NA, -4.597316624508736460, NA, NA,  -4.500342822333680104, NA, NA, -3.581504344763225323, NA,  -2.440799878852790705, NA, NA, -3.015871183969996761, NA,                     NA, -3.822471490644641978, -3.210745270562484155, -3.509909046375879083, -3.343051949217521912,                     NA, NA, NA, -1.775713247382529669, NA,                     NA, -3.687937871738988349, -2.820653224882728960, -3.349804341266563235, -4.011568804520260656,  -3.770610093141910202, NA, -4.602146744154795854, NA, NA,  -4.517503888958729341, NA, NA, -3.615145981707574840, NA,  -2.441070123497367739, NA, NA, -3.041571332933629357, NA,                     NA, -3.803399470372901270, -3.249092531999415279, -3.550005784221148541, -3.329469933204149523,                     NA, NA, NA, 0.407476683101993586, NA,                     NA, -3.713583496352714786, -2.836377792775628404, -3.354238452940760062, -4.036233853944414030,  -3.808001007112920000, NA, -4.607524027643262521, NA, NA,  -4.521594239317907160, NA, NA, -3.652626168029077913, NA,   3.228200353491821772, NA, NA, 3.260865886632574373, NA,                     NA, 1.876774874772951129, 2.277388836836955921, 3.117166127265142972, 3.079854308196050106,                     NA, NA, NA, 0.604488861053732118, NA,                     NA, 3.717208883513810846, 3.675025744084955637, 3.076164577027343583, 2.132641396825959657,   3.564841267044595696, NA, 3.935708794409766398, NA, NA,   3.112406145856193351, NA, NA, 2.761536510035906833, NA,   3.228200321460329203, NA, NA, 3.260865931125040529, NA,                     NA, 1.876774891135909673, 2.277388854955511910, 3.117166134421282209, 3.079854344494780793,                     NA, NA, NA, 0.604488843878331572, NA,                     NA, 3.717208884284644466, 3.675025842733411263, 3.076164585651329109, 2.132641409270658261,   3.564841237511557015, NA, 3.935708838545385113, NA, NA,   3.112406154472671105, NA, NA, 2.761536621269396363, NA,   3.281228986666504532, NA, NA, 3.263663320883352981, NA,                     NA, 1.880814565389634652, 2.274546787691502203, 3.151933957958044985, 3.109841168351086260,                     NA, NA, NA, 0.049432393194798863, NA,                     NA, 3.718524656195432598, 3.678678562299959598, 3.115445623120057128, 2.133107069505689068,   3.579857473197454176, NA, 3.932862921050194416, NA, NA,   3.135111353870746242, NA, NA, 2.778326739610842644, NA,   3.309400597767263807, NA, NA, 3.247210766823185324, NA,                     NA, 1.860809690545065509, 2.162187028746267003, 3.144563968774472418, 3.122292809144127101,                     NA, NA, NA, 0.227211147537748737, NA,                     NA, 3.700706474745346419, 3.648906511720991563, 3.104821396071808515, 2.101142616840137478,   3.563152194415571294, NA, 3.900497563638176235, NA, NA,   3.125805117512384435, NA, NA, 2.788893904312184979, NA,   3.356300101796932722, NA, NA, 3.207070717112514568, NA,                     NA, 1.867947428684630928, 2.182930585586870365, 3.156159493128444904, 3.099529814416696283,                     NA, NA, NA, 0.327398222431364960, NA,                     NA, 3.657615113630611425, 3.602965761775765774, 3.075697504774527413, 2.075119898335989976,   3.514790567247503716, NA, 3.858707088216064118, NA, NA,   3.121544215208547079, NA, NA, 2.765996967788668215, NA,   3.401231098134993580, NA, NA, 3.207175021355492817, NA,                     NA, 1.931625651385194598, 2.232131203988934498, 3.202996668520358359, 3.076809608560314047,                     NA, NA, NA, 0.592416055602371427, NA,                     NA, 3.687727760629219809, 3.586609364343240181, 3.095146170225470339, 2.061866529647954582,   3.495642772677272081, NA, 3.835601017170333638, NA, NA,   3.160935148810624096, NA, NA, 2.752048336970844389, NA,   3.402210054964203145, NA, NA, 3.183283603226318981, NA,                     NA, 1.965291704273218087, 2.235560958766957373, 3.204665990407831622, 3.022802670299353345,                     NA, NA, NA, 0.739291709799082097, NA,                     NA, 3.703487406385991498, 3.524949335872570533, 3.081187661523752563, 2.018695858492556283,   3.457312253232445620, NA, 3.791973077062348096, NA, NA,   3.166181282340475356, NA, NA, 2.705876941784540524, NA,   3.360423952200775055, NA, NA, 3.138092987881514695, NA,                     NA, 1.989563642063393978, 2.175786251918510494, 3.190641759626814622, 2.959919034905913904,                     NA, NA, NA, 0.467540135571932858, NA,                     NA, 3.698641160077588808, 3.450677642878595464, 3.030186655588327227, 1.959512546937195232,   3.427785863472317729, NA, 3.722826547895256688, NA, NA,   3.149232161969778510, NA, NA, 2.633828233564777133, NA,   3.260429468890833959, NA, NA, 3.022105715996519670, NA,                     NA, 1.944720933401234308, 2.023435731786589198, 3.111378362809810838, 2.877072287723438482,                     NA, NA, NA, 0.241304065184183125, NA,                     NA, 3.674384443077062468, 3.291691599928831646, 2.930413102126797842, 1.850153568346212030,   3.407104325660661015, NA, 3.603652781164407237, NA, NA,   3.075034950868478223, NA, NA, 2.535664221575451460, NA,   3.238536246689260878, NA, NA, 2.922082169032123922, NA,                     NA, 1.901112518169710164, 1.993956310244486385, 3.011815720362899462, 2.839442726764252356,                     NA, NA, NA, 0.387682357594861693, NA,                     NA, 3.629368344509535493, 3.177521084920808647, 2.816399316502611772, 1.752578764956602919,   3.394363534097831536, NA, 3.504866430498762231, NA, NA,   3.081739951530251709, NA, NA, 2.464198014939879222, NA,   3.208686529936827014, NA, NA, 2.847628815236865396, NA,                     NA, 1.880240304706340515, 1.866534024456231844, 2.960792923966828916, 2.801620122294430359,                     NA, NA, NA, 0.387450921635163370, NA,                     NA, 3.578705142373943993, 3.153653296723436750, 2.812278989652126526, 1.705900859812779835,   3.394733689061277371, NA, 3.478189019417443184, NA, NA,   3.052842479683547161, NA, NA, 2.465841413079179922, NA,   3.216081053701049797, NA, NA, 2.840880582423570999, NA,                     NA, 1.879804326296606209, 1.898794191114991037, 2.916185953050459911, 2.761781724292191154,                     NA, NA, NA, 0.554721720560227505, NA,                     NA, 3.583891474671388799, 3.158166897897720204,
                 2.831421147525392534, 1.712046814377341741,   3.465241988248026317, NA, 3.477442279637290579, NA, NA,   3.069021173096718957, NA, NA, 2.467166140113334905, NA,   3.229608868031281332, NA, NA, 2.812934048613914051, NA,                     NA, 1.855586426672065858, 1.920458603964754651, 2.852406043996616347, 2.705592697596189922,                     NA, NA, NA, 0.481636511699503278, NA,                     NA, 3.571252900714774636, 3.153326134586253104, 2.854031284238863098, 1.690954983183373894,   3.507591152410797442, NA, 3.478733064764671035, NA, NA,   3.062763006041160807, NA, NA, 2.437641939577382288, NA,   3.262351795238837227, NA, NA, 2.796058125300871389, NA,                     NA, 1.858723846955978365, 1.952898534895356253, 2.833985810543629302, 2.709257408859062188,                     NA, NA, NA, 0.514395701124311211, NA,                     NA, 3.582118923626243845, 3.146699592245628008, 2.872627744935372984, 1.701281745767361908,   3.564207500637748272, NA, 3.465111483154899474, NA, NA,   3.104919368106071609, NA, NA, 2.455152688042030906, NA,   3.276281304444632614, NA, NA, 2.776377796428107114, NA,                     NA, 1.846395744492797864, 1.866205289170038251, 2.801732319829042250, 2.700025145195349197,                     NA, NA, NA, 0.702383899553258750, NA,                     NA, 3.521904456198760247, 3.140773138650221341, 2.911944704671406647, 1.735579569269010136,   3.580126404048669109, NA, 3.417944097380624324, NA, NA,   3.109496499020899751, NA, NA, 2.481565181381751639, NA,   3.297059600875677621, NA, NA, 2.744732610669121442, NA,                     NA, 1.820851619514626263, 1.857136536069474708, 2.792900579262012961, 2.670283530990111842,                     NA, NA, NA, 0.448228526771191071, NA,                     NA, 3.514790875188213981, 3.131115123922080024, 2.938411482107062156, 1.746787318748280748,   3.574527256889179050, NA, 3.417277993238311051, NA, NA,   3.130746776414634169, NA, NA, 2.485780283454904982, NA,   3.308552197278025186, NA, NA, 2.743822116098167285, NA,                     NA, 1.794702009701288237, 1.948375334663357439, 2.804367858480987685, 2.653075435855417386,                     NA, NA, NA, 0.569427293442274385, NA,                     NA, 3.479735593302771157, 3.126428091522157171, 2.916854141818865820, 1.763026380448047270,   3.573304768394804931, NA, 3.437753047939566819, NA, NA,   3.174241603152402558, NA, NA, 2.538376797818190145, NA,   3.339193873273715418, NA, NA, 2.733611747703567740, NA,                     NA, 1.770305877285430096, 2.009783845689825377, 2.813274269591109267, 2.635257159034936958,                     NA, NA, NA, 0.549961179418412605, NA,                     NA, 3.441739624805878250, 3.125452179726679880, 2.898109691935282672, 1.790702742508476941,   3.579328034841501083, NA, 3.445369511418996833, NA, NA,   3.220532110174082696, NA, NA, 2.580840850039214818, NA,   3.371190747375367280, NA, NA, 2.710637047873375227, NA,                     NA, 1.755990535539004416, 2.023621250546806749, 2.820324105515341184, 2.612062711361560208,                     NA, NA, NA, 0.614516592101345793, NA,                     NA, 3.414209432792202570, 3.113675381436664669, 2.891103696809161683, 1.811730023688659319,   3.590479997760770203, NA, 3.443492566510706787, NA, NA,   3.269362421379239603, NA, NA, 2.620412856159460624, NA,   3.382991657481115855, NA, NA, 2.666818256379736241, NA,                     NA, 1.721529997207964957, 2.006085524926763064, 2.810671505976249485, 2.600716746799072165,                     NA, NA, NA, 0.759469531774261464, NA,                     NA, 3.378953266264153488, 3.111173707158624513, 2.886220338815654429, 1.835905865384103164,   3.606917526696223231, NA, 3.450724407866636589, NA, NA,   3.315332436485178391, NA, NA, 2.664126136929188426, NA,   3.375097046376972720, NA, NA, 2.619561218517739931, NA,                     NA, 1.682451883502353640, 2.038917730857703781, 2.799625717415830994, 2.586657596450685936,                     NA, NA, NA, 0.677891647387884433, NA,                     NA, 3.350397446293547077, 3.117680614365267999, 2.874183254133590104, 1.868096028212964832,   3.626087124861198774, NA, 3.470075843459831333, NA, NA,   3.379166584418708652, NA, NA, 2.698872085311290192, NA,   3.350204500868286139, NA, NA, 2.561248359701934429, NA,                     NA, 1.645444072852895623, 2.041962774376676393, 2.794555104151910552, 2.574016797741650375,                     NA, NA, NA, 0.095235180511452161, NA,                     NA, 3.332525763084030590, 3.113102463290578381, 2.854277587639550973, 1.886539521782538920,   3.627551856614038783, NA, 3.482954092795894852, NA, NA,   3.434230319889856275, NA, NA, 2.703438135502776341, NA,   3.332790902670354605, NA, NA, 2.504044570422898630, NA,                     NA, 1.613232171925441660, 2.024703691081490398, 2.806718095890756270, 2.572042485593793781,                     NA, NA, NA, 0.286650707453472930, NA,                     NA, 3.332406408263562270, 3.121555266862563460, 2.837112144680057124, 1.888874610490932637,   3.623904435473706442, NA, 3.505295353152330762, NA, NA,   3.499744643890911000, NA, NA, 2.700058546288654604, NA,   3.289919450838763915, NA, NA, 2.446514992001886490, NA,                     NA, 1.576584760598337409, 1.943241812640547961, 2.781715033215237387, 2.554065233386250355,                     NA, NA, NA, 0.398895306272111394, NA,                     NA, 3.319419574729846278, 3.135063707262732091, 2.823819637422379181, 1.867973486169973718,   3.584960996993316495, NA, 3.537068402873925432, NA, NA,   3.512503179285614952, NA, NA, 2.686528662936770839, NA,   3.263793599963149994, NA, NA, 2.428300046865409900, NA,                     NA, 1.587050678011627580, 1.995441003916150091, 2.774029280638089290, 2.544421221081314854,                     NA, NA, NA, 0.595643035445955760, NA,                     NA, 3.286124517109271892, 3.160466924551737833, 2.825172627029894024, 1.867737720948244196,   3.552933319269998780, NA, 3.568521155473102446, NA, NA,   3.525432504829723257, NA, NA, 2.691767827135213675, NA,   3.238968194336186546, NA, NA, 2.406555275417276096, NA,                     NA, 1.602085631456553250, 2.005989516280514362, 2.761241727912435273, 2.533374633134493514,                     NA, NA, NA, 0.653756142160007103, NA,                     NA, 3.261504818893982360, 3.188177681262927887, 2.831208124579261742, 1.854349201581874107,   3.529848647302552322, NA, 3.596708926735668221, NA, NA,   3.536856098335411591, NA, NA, 2.690723679551680014, NA,   3.218826366143559436, NA, NA, 2.386413659234313478, NA,                     NA, 1.615639513330824784, 2.051695525727077918, 2.746311796546208939, 2.531988494218035068,                     NA, NA, NA, -0.087892137668173498, NA,                     NA, 3.232841871312505333, 3.214889606397770105, 2.843481176535438859, 1.847706839692830005,   3.506563362435390907, NA, 3.622870631203163772, NA, NA,   3.547583565036966480, NA, NA, 2.676068802172083316, NA,   3.188585853469036557, NA, NA, 2.365076772346223155, NA,                     NA, 1.622979182745510229, 2.002417845235033411, 2.741600599484756096, 2.538770451221326496,                     NA, NA, NA, 0.255406629233229765, NA,                     NA, 3.202001881102527037, 3.226617654627964349, 2.841688513931766913, 1.841204491862387993,   3.492829064447405152, NA, 3.632550191318089539, NA, NA,   3.551982521424168038, NA, NA, 2.675640783376405096, NA,   3.158985552653615692, NA, NA, 2.343254402202465947, NA,                     NA, 1.619310985961501359, 2.029330261131228053, 2.725547503778536829, 2.550151010723951117,                     NA, NA, NA, 0.434596823465360216, NA,                     NA, 3.169009367936018506, 3.238300985961096057, 2.846165604977143992, 1.834280251814616403,   3.469307521167436903, NA, 3.634369750372233199, NA, NA,   3.556900656465495025, NA, NA, 2.661604466818824122, NA,   3.147889427974054133, NA, NA, 2.324199707467162757, NA,                     NA, 1.653178678601974427, 2.064899022296584619, 2.726598737594756461, 2.564999933211551397,                     NA, NA, NA, 0.637992713097506270, NA,                     NA, 3.143804872489466717, 3.250326685554854045, 2.863530791136178699, 1.827656858844834753,   3.455578801230433594, NA, 3.655760047517219835, NA, NA,   3.594664851051069476, NA, NA, 2.676839097180010452, NA,   3.136182900139802499, NA, NA, 2.315442125863074185, NA,                     NA, 1.694131738132602427, 2.101100721661687487, 2.731677714264458601, 2.586847416120857446,                     NA, NA, NA, 0.811954788029270680, NA,                     NA, 3.115151498177218592, 3.260973315538634232, 2.885122231200393905, 1.838300983269717470,   3.440568125690217194, NA, 3.682917799096094846, NA, NA,   3.632453218373578263, NA, NA, 2.700669085889184640, NA,   3.123971245571688815, NA, NA, 2.306733120812562099, NA,                     NA, 1.739036636758735765, 2.145302096889043941, 2.740508200165153951, 2.615277381945741997,                     NA, NA, NA, 0.861262169732795346, NA,                     NA, 3.087042721205589757, 3.271867667216353581, 2.917750477959488720, 1.847224378223533137,   3.425350981325032684, NA, 3.707641994378415795, NA, NA,   3.670463524121203580, NA, NA, 2.716984925845204568, NA,
                 3.116008462395396528, NA, NA, 2.299927025505433598, NA,                     NA, 1.796672842580713692, 2.204447055496953745, 2.741748802410547192, 2.647455894854256808,                     NA, NA, NA, 0.384893742028523611, NA,                     NA, 3.054077751039752542, 3.286485587558218846, 2.957941706760103706, 1.864440156830964312,   3.419334041209202812, NA, 3.739427362072642147, NA, NA,   3.714822675684708031, NA, NA, 2.731752443502630801, NA,   3.106129186923187735, NA, NA, 2.289658724503894405, NA,                     NA, 1.853234694529040771, 2.178910750153713494, 2.748247750140098589, 2.691707726948386004,                     NA, NA, NA, 0.467204705639933437, NA,                     NA, 3.022770523357071859, 3.304542232064877627, 3.016902598420250659, 1.878350853576152391,   3.417406148029908053, NA, 3.775577434611405270, NA, NA,   3.750258096656248430, NA, NA, 2.737571828599946855, NA,   3.100449374417385329, NA, NA, 2.290202914704332571, NA,                     NA, 1.864003065662072967, 2.140147385088308152, 2.730480672057678859, 2.714037956981668387,                     NA, NA, NA, 0.554154272558110117, NA,                     NA, 3.022389538140644039, 3.298342236704310704, 3.067222592569139206, 1.862896286186282335,   3.403220117768594122, NA, 3.780959325803663340, NA, NA,   3.759166787250445729, NA, NA, 2.718148715301107021, NA,   3.087503186803484567, NA, NA, 2.290065683641528516, NA,                     NA, 1.883172415945876699, 2.108203558360934871, 2.714348816119104058, 2.726497679754599890,                     NA, NA, NA, -0.058786319994611924, NA,                     NA, 3.011589120712045098, 3.294719040857719605, 3.108075346829027641, 1.840044437698373780,   3.380282106828251809, NA, 3.775189587871251806, NA, NA,   3.767008605800070686, NA, NA, 2.692939344074732677, NA,   3.064520872676683272, NA, NA, 2.269732962739291260, NA,                     NA, 1.889708051146857359, 2.081459882672170192, 2.682924409599337423, 2.725280783528779338,                     NA, NA, NA, -2.581509984666792157, NA,                     NA, 2.993824635879922536, 3.274764370947458758, 3.099307703098814315, 1.817538397140333384,   3.350868494922803809, NA, 3.768465963877798863, NA, NA,   3.755377195010565217, NA, NA, 2.661305358433966539, NA,   1.265306222470304442, NA, NA, 0.983129589193998621, NA,                     NA, 1.017065152449590837, 0.589003462786476528, 1.018430054873038060, 1.124106389420284513,                     NA, NA, NA, 1.082149117077308320, NA,                     NA, 1.072092966323718422, 0.909625086001551009, 1.071639897256816409, 0.844577617597411900,   1.092561258209910235, NA, 1.038949170768870944, NA, NA,   1.014732711808240762, NA, NA, 1.138421216986263529, NA,   1.265306232236593775, NA, NA, 0.983129596736347366, NA,                     NA, 1.017065154614362754, 0.589003436218133669, 1.018430050937821463, 1.124106387336627710,                     NA, NA, NA, 1.082149091882674830, NA,                     NA, 1.072092966892312482, 0.909625067047899249, 1.071639898131811819, 0.844577602862138122,   1.092561260379418364, NA, 1.038949163178025437, NA, NA,   1.014732704847147726, NA, NA, 1.138421207733120832, NA,   1.121699083370837569, NA, NA, 0.949509856473429892, NA,                     NA, 1.009647271713033145, 0.359396088754979137, 0.955492378499042561, 1.051326668070421633,                     NA, NA, NA, 0.171439061100736700, NA,                     NA, 1.045503703281888574, 0.878913892569637589, 1.003840309376754281, 0.829279914763967341,   1.043431320776803339, NA, 1.015826411837153609, NA, NA,   0.963588055412754940, NA, NA, 1.108617346360504241, NA,   1.140515864972018178, NA, NA, 0.966207333874159136, NA,                     NA, 1.043553494139210613, 0.843971202796278686, 0.966027290483720824, 1.053732812723182422,                     NA, NA, NA, 0.296898218494179800, NA,                     NA, 1.064155868437993924, 0.935295642815594519, 1.044542521993743422, 0.855105853975426267,   1.076619451472907718, NA, 1.033288394048857306, NA, NA,   0.997212238541415497, NA, NA, 1.067967508612106542, NA,   1.161751655451926046, NA, NA, 0.996368569777236424, NA,                     NA, 1.114940580862771480, 0.756408970879646203, 0.995365724549254094, 1.020943695645449401,                     NA, NA, NA, 1.064516718886184155, NA,                     NA, 1.094810544230560012, 0.951032337774829761, 1.147724715901439430, 0.887513227234576707,   1.067823502393392410, NA, 1.076229195196301447, NA, NA,   1.029150967597868060, NA, NA, 1.043391514772855189, NA,   1.074737699592852369, NA, NA, 0.942485318520647208, NA,                     NA, 1.030722047875287206, 0.673175637048600950, 0.934968216068958413, 1.002938003163197811,                     NA, NA, NA, 1.090657047511057076, NA,                     NA, 1.042253377533620018, 0.926320579878110562, 1.121194389039621875, 0.906275436966622938,   1.052587252726008060, NA, 1.003531975630310713, NA, NA,   0.957679536510319562, NA, NA, 1.050724418775577540, NA,   1.006596652486369869, NA, NA, 0.907142495285025108, NA,                     NA, 0.993787701844790039, 0.676067629599375941, 0.915982843633481147, 0.989425834959653527,                     NA, NA, NA, 1.165562697207051635, NA,                     NA, 1.017997034348915086, 0.933752036955077469, 1.147143372287581231, 0.935431232842611649,   1.031639220061343565, NA, 0.946018340777982858, NA, NA,   0.924796157465827884, NA, NA, 1.078466838272314821, NA,   0.961153639814163796, NA, NA, 0.873295232694994539, NA,                     NA, 0.975228339836115232, 0.897576862189576330, 0.917023966574287974, 0.993968853656988394,                     NA, NA, NA, 0.817891001134847140, NA,                     NA, 1.008174156048561088, 0.933726694753238551, 1.151789872422698258, 0.947491306020807111,   1.085422231650577052, NA, 0.966696715451277333, NA, NA,   0.951948807602071456, NA, NA, 1.149598451684048506, NA,   1.009069946439633103, NA, NA, 0.977209442018580399, NA,                     NA, 1.080927847182693302, 1.516281311341671989, 1.009967105475058524, 1.044625567597576410,                     NA, NA, NA, 1.403664095339415718, NA,                     NA, 1.017465410203032494, 1.064549467010453165, 1.265783735159351053, 1.040244841441904100,   1.131250263339454776, NA, 1.058943520873401267, NA, NA,   1.070628963554307145, NA, NA, 1.238137952640823602, NA,   1.046769447639742445, NA, NA, 1.067019931164797875, NA,                     NA, 1.158481955623348547, 1.389953465304866631, 1.098214972715407134, 1.037229734025731265,                     NA, NA, NA, 1.376277422342268242, NA,                     NA, 1.015060258054122144, 1.218085070522000990, 1.397206829513694348, 1.139173224713716071,   1.135830124395663221, NA, 1.169552066297739890, NA, NA,   1.093971112888281860, NA, NA, 1.292422487556013300, NA,   1.135688275781382872, NA, NA, 1.229302255124709919, NA,                     NA, 1.261863211243490435, 1.859476983421588514, 1.192967595360016508, 1.079711925650131965,                     NA, NA, NA, 1.751689230567990974, NA,                     NA, 1.081465720059105129, 1.316881843355672821, 1.463063278813513390, 1.251101112014280536,   1.167273951581269786, NA, 1.250933328245879439, NA, NA,   1.242727946144689399, NA, NA, 1.328533777342236544, NA,   1.207373160451673977, NA, NA, 1.301752709734714797, NA,                     NA, 1.336364282395194669, 1.478426282858289031, 1.287271113491110563, 1.107235502521267456,                     NA, NA, NA, 1.645467598271238785, NA,                     NA, 1.088015692428972780, 1.385233729742171516, 1.467035972051451420, 1.285953609551262922,   1.136769222665502177, NA, 1.302288803237156234, NA, NA,   1.282742118641063023, NA, NA, 1.361579551647737807, NA,   1.206030004916152754, NA, NA, 1.310796843516631727, NA,                     NA, 1.303946535578728527, 1.233325691066664120, 1.335543534428687007, 1.135452883769065613,                     NA, NA, NA, 1.341867441952146578, NA,                     NA, 1.032262083296376032, 1.442896193165066121, 1.475158899116146616, 1.352747217881487662,   1.109263686888440459, NA, 1.314563039654336407, NA, NA,   1.273675762963935698, NA, NA, 1.368633920530966774, NA,   1.181876623572887919, NA, NA, 1.313567866035455722, NA,                     NA, 1.283726439614926829, 1.180318829152357640, 1.343691134803814080, 1.169804924312562999,                     NA, NA, NA, 1.286166051712225977, NA,                     NA, 0.972059465186502103, 1.454395151312573686, 1.486294000463050713, 1.349870192120145029,   1.055349712493220515, NA, 1.324947611234549516, NA, NA,   1.248379124404252583, NA, NA, 1.344716885729785316, NA,   1.185104821805905484, NA, NA, 1.315477840481198424, NA,                     NA, 1.286380316587188988, 1.766207816213777537, 1.376316670448006763, 1.254386799026023569,                     NA, NA, NA, 1.837105985511625406, NA,                     NA, 1.018592852394144321, 1.446145791198174546, 1.489669580327890319, 1.300819800437375884,   1.059320118742595795, NA, 1.365563413973762685, NA, NA,   1.285486643043280530, NA, NA, 1.279556849556759968, NA,   1.178852887835034835, NA, NA, 1.308628020125996105, NA,                     NA, 1.312646769464822949, 1.769669981168264572, 1.373134662384808857, 1.230337809013719585,                     NA,
                 NA, NA, 1.692863254151797170, NA,                     NA, 1.044654913735360235, 1.351756687474305885, 1.479150261866704508, 1.244192780849964208,   0.984675587931245322, NA, 1.352790163426493164, NA, NA,   1.255868894354347898, NA, NA, 1.220920338036541208, NA,   1.138601907296074689, NA, NA, 1.271203466417815564, NA,                     NA, 1.279251808070025254, 1.141956489204948078, 1.340984622574985741, 1.132977369412550228,                     NA, NA, NA, 1.292288519217280163, NA,                     NA, 1.021877127756780412, 1.250529959529877999, 1.398614064007530411, 1.132567549006586738,   0.882205108713892905, NA, 1.277787247878447330, NA, NA,   1.182987513278574099, NA, NA, 1.123370342751250250, NA,   1.149716406373992639, NA, NA, 1.235458205566871914, NA,                     NA, 1.289895197796902426, 0.758251439732939070, 1.336229259627650157, 1.078783269254004917,                     NA, NA, NA, 0.909601473568784624, NA,                     NA, 1.006364378590667519, 1.181080980224711086, 1.331928910493913376, 1.072881160840144066,   0.789102375937836942, NA, 1.221250878401829443, NA, NA,   1.106185413554924946, NA, NA, 1.029015533589358045, NA,   1.167537745902685664, NA, NA, 1.209335245337249232, NA,                     NA, 1.289351826077227869, 0.699084728388057020, 1.329377214195342383, 1.020856724313794928,                     NA, NA, NA, 0.853922750381080453, NA,                     NA, 1.016590308640584306, 1.116801718521792663, 1.288834875882563180, 1.001305971219041080,   0.798133459297804326, NA, 1.170376757357845188, NA, NA,   1.062510954372609007, NA, NA, 1.015841226420151733, NA,   1.187920039334742439, NA, NA, 1.187328575791362306, NA,                     NA, 1.294919713602713962, 0.744308456170960531, 1.328178112581706705, 0.980981562353251846,                     NA, NA, NA, 1.139088929456654631, NA,                     NA, 1.030656463880199603, 1.060649431567091128, 1.252478043573923339, 0.933068633454060881,   0.806042864654815783, NA, 1.137218396757121663, NA, NA,   1.013468281051595143, NA, NA, 1.006537363427364218, NA,   1.200989416660449560, NA, NA, 1.175760835310716956, NA,                     NA, 1.284324504308749226, 0.664485765997248845, 1.331626089383696421, 0.933876015321519759,                     NA, NA, NA, 0.808507160491150678, NA,                     NA, 1.064797196601926288, 1.009185050351464996, 1.204728464350449668, 0.873497895122031354,   0.808523812821577148, NA, 1.116996046390796860, NA, NA,   0.962648912064332896, NA, NA, 0.957053017519004401, NA,   1.220255087446890929, NA, NA, 1.172152915337392409, NA,                     NA, 1.283883979675385811, 0.649110041847309804, 1.342019260580085138, 0.885253473817406489,                     NA, NA, NA, -0.278730774489053146, NA,                     NA, 1.111626135161410733, 0.952379222758095834, 1.163953948474635514, 0.813140391719971989,   0.810748812077887404, NA, 1.100373364093229833, NA, NA,   0.906587520931251389, NA, NA, 0.914344245108139275, NA,   1.238652255882794373, NA, NA, 1.176627036327414721, NA,                     NA, 1.327162678492149972, 0.872581660077473842, 1.351541572223509347, 0.866431847772294428,                     NA, NA, NA, 0.039163901413492214, NA,                     NA, 1.162053729806475477, 1.001626016378976969, 1.128665272693166965, 0.825215654853380776,   0.897367947438926450, NA, 1.115763303326419376, NA, NA,   0.843392230158773848, NA, NA, 0.941640155001106538, NA,   1.276786414013518112, NA, NA, 1.196616188716464801, NA,                     NA, 1.404229165513591226, 1.397866918803592595, 1.392659362789604227, 0.858788978306715634,                     NA, NA, NA, 0.760548879736937300, NA,                     NA, 1.236748102115617254, 1.062487248227932835, 1.101741181835382832, 0.850534068021918777,   1.012965575434265908, NA, 1.136909507646239437, NA, NA,   0.806979427707043895, NA, NA, 0.981951220425578142, NA,   1.319709808985618293, NA, NA, 1.132449407011330900, NA,                     NA, 1.352795142052384492, 1.115558895071089074, 1.379195443481202599, 0.836080875676972424,                     NA, NA, NA, 1.188250845868388250, NA,                     NA, 1.268824941212484791, 1.071983976123047411, 1.036731123202333293, 0.849044160307756091,   1.076903838551396042, NA, 1.149724269308894486, NA, NA,   0.745396968622651590, NA, NA, 0.945698974630409861, NA,   1.318632530459488361, NA, NA, 1.038973265448528194, NA,                     NA, 1.287661251839132603, 1.065533550119720063, 1.362390705005948721, 0.827287324894647735,                     NA, NA, NA, 1.251775233979616875, NA,                     NA, 1.238687483896110386, 1.035152107365963303, 0.993660219450412940, 0.822742434409587275,   1.117332959475648702, NA, 1.142005567236855423, NA, NA,   0.681417562001698007, NA, NA, 0.907838640478514924, NA,   1.289873860044079557, NA, NA, 0.932441642622866440, NA,                     NA, 1.228252783765326761, 0.758979607814741386, 1.349003881326149346, 0.739948049422119447,                     NA, NA, NA, -0.517760835744116910, NA,                     NA, 1.198897228767427992, 0.992399213603804897, 0.929113106437839620, 0.753921579842014689,   1.143423170763419261, NA, 1.127444945887119632, NA, NA,   0.609219480344156294, NA, NA, 0.870593383402629595, NA,   1.264976830498052207, NA, NA, 0.919528422679855462, NA,                     NA, 1.212533995952858490, 1.101278921080548256, 1.396600206900185048, 0.724238346661745691,                     NA, NA, NA, -0.093755876523776593, NA,                     NA, 1.155279036073630694, 0.993499616604767710, 0.936127030689097861, 0.753630775977174983,   1.194242107690856036, NA, 1.118804477923337481, NA, NA,   0.572338542876108303, NA, NA, 0.877304165122622992, NA,   1.233388808254944902, NA, NA, 0.905524735817091919, NA,                     NA, 1.153637473314472306, 0.876512732942901440, 1.413504816529525421, 0.704167960308164975,                     NA, NA, NA, 0.147076364560457751, NA,                     NA, 1.113768408199603099, 0.992544776219865366, 0.939590188453604425, 0.746403411190795785,   1.224193400145062949, NA, 1.083893357325879547, NA, NA,   0.538541171900463134, NA, NA, 0.858644121471305755, NA,   1.216851056904205919, NA, NA, 0.899102651671992437, NA,                     NA, 1.096947932738620013, 0.628423517017324684, 1.445447325734992461, 0.688300670093842326,                     NA, NA, NA, 0.392709220521088476, NA,                     NA, 1.086960460021329844, 1.000034818074947651, 0.951424409821290173, 0.740257045045417117,   1.265955316592938695, NA, 1.050545657345982375, NA, NA,   0.512817058373752022, NA, NA, 0.844787241356501406, NA,   1.193402616080912360, NA, NA, 0.900493763329405783, NA,                     NA, 1.037337986178631155, 0.346833786670540989, 1.485040236164735461, 0.673515023852275352,                     NA, NA, NA, 0.715502088747810960, NA,                     NA, 1.064355882321898372, 1.008879447006052699, 0.962673070890044613, 0.734943658962303625,   1.312757215515636622, NA, 1.016355161954325848, NA, NA,   0.491655180966815264, NA, NA, 0.832768529843420313, NA,   1.165950368043188190, NA, NA, 0.903584263433771739, NA,                     NA, 0.968686557546770799, -0.012846954380927311, 1.534794741155505626, 0.654715116895286275,                     NA, NA, NA, 0.953540100299316462, NA,                     NA, 1.044184608503654133, 1.018836940085163345, 0.970889027568660024, 0.725836116888062044,   1.368083140433063338, NA, 0.981406343854288377, NA, NA,   0.474008771428854281, NA, NA, 0.822454666978510329, NA,   1.128875585976392593, NA, NA, 0.894089657456716869, NA,                     NA, 0.891034628146853191, -0.475149291297670473, 1.542242928981809591, 0.613663668368812920,                     NA, NA, NA, 0.056496453231117294, NA,                     NA, 1.009669734490761028, 1.022374783225200146, 0.957963007466903882, 0.716550539853706425,   1.433828991880685466, NA, 0.957933663016146575, NA, NA,   0.423665612205222586, NA, NA, 0.806202427868420668, NA,   1.111076173918857712, NA, NA, 0.909333612970688709, NA,                     NA, 0.869114896304417961, -0.311300209605316458, 1.551656753872525218, 0.567860862654193332,                     NA, NA, NA, 0.355739715689408209, NA,                     NA, 0.974666722798749108, 1.025947254057442581, 0.946469268888114224, 0.730095410932096511,   1.504178579798707283, NA, 0.934087380992065786, NA, NA,   0.453239906355537570, NA, NA, 0.812933698726916898, NA,   1.111238844792900693, NA, NA, 0.951016924548257592, NA,                     NA, 0.845278510025986973, -0.096003957577904697, 1.584550915710924102, 0.541672114786283077,                     NA, NA, NA, 0.834744396257198984, NA,                     NA, 1.014934647812774537, 1.031647158817056820, 0.962107991862088063, 0.740657850343180546,   1.569938234870758897, NA, 0.911166196257087835, NA, NA,   0.476922947027078159, NA, NA, 0.841244983072905628, NA,   1.092132530203802965, NA, NA, 0.975991589817999827, NA,                     NA, 0.799009347993687835, 0.063163739162103633, 1.607280941385802464, 0.495845310644838277,                     NA, NA, NA, -0.098362323540837071, NA,                     NA, 1.023734194252273966, 1.017835271250237206, 0.991185940787828068, 0.752416955238215501,   1.577991001668749060, NA,
                 0.876937952648258134, NA, NA,   0.504024759575150694, NA, NA, 0.853359933056268893, NA,   1.061435076655089516, NA, NA, 0.981602982149452630, NA,                     NA, 0.714792187811584601, 0.179016113660373150, 1.618071551959163301, 0.420798216521904855,                     NA, NA, NA, -4.774779067365225060, NA,                     NA, 1.027686070214723690, 0.980750896681414686, 0.951375214212884734, 0.762740695796394474,   1.587314413110313405, NA, 0.840786297535069727, NA, NA,   0.474570504586541042, NA, NA, 0.866923993115378044, NA ))

  set.seed(68245)
  suppressWarnings(tmp <- ddhazard_boot(fit, do_sample_weights = T, do_stratify_with_event = F, R = 30))

  # plot(dd_fit)
  # get_expect_equal(tmp, file = "tmp.txt")

  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c(-11.05250337824802465, -11.05250352469279917, -11.04508999427243232, -11.03079218210795354, -11.01276689498567407, -10.97180707356202767, -10.92868943459950515, -10.90098502723408558, -10.86985710925975468, -10.84982541828216718, -10.82653501151556341, -10.80572600014983564, -10.77373825434438714, -10.73524462930758183, -10.70591598692485391, -10.66720973370452441, -10.65998486910317666, -10.64608628893297393, -10.64680077372383948, -10.63576684813315154, -10.63361042480508445, -10.61687960464793434, -10.60104166304420303, -10.58927465566920034, -10.58443158792417371, -10.57623508244723176, -10.56319725965373912, -10.55230844022145398, -10.54592620274321035, -10.53960130032549003, -10.53665901819950612, -10.53041595305994349, -10.52510593969067543, -10.52505810336422165, -10.52727056580709508, -10.52837999933849922, -10.53342995392302583, 0.05296958898534846, 0.05296958905563476, 0.05415849402328772,   0.05309799072773889, 0.05327594826658033, 0.05104031066007266, 0.05053463696140417, 0.05133502712341615,   0.05213837065912127, 0.05261984592138964, 0.05264115568159874, 0.05080471436273008, 0.04938715582283902,   0.04766643888761872, 0.04588737086129124, 0.04447490675834360, 0.04477538409203886, 0.04467320797311228,   0.04451161946271013, 0.04370458232740190, 0.04408756070392547, 0.04470681659873713, 0.04423196799915047,   0.04453525705758350, 0.04543917052864419, 0.04721251203076599, 0.04888155727856350, 0.04815360863535564,   0.04869375124859262, 0.04842177412108322, 0.04858862179933521, 0.04943215584087465, 0.05038472787081971,   0.05165423949341428, 0.05234389971935057, 0.05364525710387259, 0.05600856244422565, -4.30686031039544037,  -4.30686033866322493, -4.28911812805663928, -4.27998912162376222, -4.26876503528201745, -4.19916544913661038,  -4.11221120301745113, -4.04961138518570252, -3.96364464853105281, -3.90951366273536394, -3.88425038595311189,  -3.83562755297431313, -3.75715042968144397, -3.70593275221107188, -3.63175497236033129, -3.54073945601396334,  -3.48890563835667411, -3.44833071631922117, -3.42631784404413819, -3.37191034007683976, -3.35812106341563288,  -3.31782911141499159, -3.30587318478505177, -3.27812439292552549, -3.24873043138299522, -3.20762588814112126,  -3.13636229932190158, -3.09659228749524873, -3.03798213884600976, -2.99469934639876056, -2.97373260897097325,  -2.95958975059265939, -2.95827222219042296, -2.99268351257244003, -3.01924136612427807, -3.04687672446439661,  -3.05065564335752359, 3.31864494491752815, 3.31864501612206597, 3.32609854359188972, 3.30588493258488381,   3.27259320775837326, 3.26794688321532467, 3.23337113422625322, 3.18067910501152218, 3.09072238149282486,   3.01345224060827999, 2.96755490883330930, 2.95972027067869092, 2.93977593563432205, 2.94219505205174414,   2.92326723828454060, 2.90901668611108644, 2.90126135891327008, 2.89732029823756898, 2.89760749424391362,   2.89505496127566975, 2.89351873839219564, 2.88121781279513955, 2.87745119001530814, 2.85615666345754526,   2.84511940653338957, 2.83143267284266420, 2.81785700089585989, 2.80444779531331756, 2.78275861269943725,   2.77701079671141393, 2.77765154508227008, 2.78227415883856244, 2.79480371926767868, 2.81180828891713741,   2.81443534114923422, 2.81609089945707813, 2.80460170174301870, 0.99194883257467559, 0.99194882894273229,   0.95856807961592283, 0.99902565656941378, 1.04011724688152807, 1.02440686902098155, 1.02810372591729982,   1.05013451503190458, 1.13254883267763273, 1.18633909796471015, 1.24757108028956654, 1.26817152654242049,   1.27485163969115001, 1.25863528878724895, 1.27084749931237551, 1.25798711924296081, 1.19128180739154543,   1.15089054247972755, 1.11611654351599121, 1.08520744458319229, 1.05353951495120324, 1.02758650157878861,   1.03566385260869986, 1.06065000568720036, 1.04293331650105259, 0.99989795301664408, 0.93962408658826890,   0.94297104440748714, 0.92088077832370374, 0.90363057994208040, 0.88604372273376464, 0.86473878872740251,   0.83405917696158494, 0.82327669479966514, 0.83300016580390523, 0.82406112838505341, 0.79124136299044501 ))

  expect_equal(unname(c(tmp$t)),
               c(-11.92968574511654722, -11.62300264831321428, -10.23422442629716578, -12.26091197688843515, -11.36420249938667837, -11.53807372967151323, -10.03342642658376249, -11.83843287200145156, NA, -6.72507439901146764,  -9.88681008454443955, -8.63461827154769779, -9.04064935655839186, -10.16366950255433643, NA,  -7.94280128857106593, -7.59085887289955963, -13.51195662945783837, -9.52376760782359355, NA, -11.77069829898177211, -9.84925736212199077, -11.34452706454714388, -7.86830479177370368, -11.23340348716389592, -11.26322504801983904, -12.02950660358850854, NA, -7.63341370892039262, -11.50270934258737654, -11.92968593116309428, -11.62300285741152450, -10.23422452498884105, -12.26091193978200700, -11.36420258261429161, -11.53807380554453665, -10.03342651317812262, -11.83843304500777016, NA, -6.72507451666405132,  -9.88681004021968945, -8.63461828613009530, -9.04064943774780083, -10.16366964497599668, NA,  -7.94280124565038115, -7.59085895429895885, -13.51195669770785202, -9.52376765174636830, NA, -11.77069845100713508, -9.84925737947472157, -11.34452719172168678, -7.86830471754829652, -11.23340362288815619, -11.26322507897764602, -12.02950690802509115, NA, -7.63341382510911970, -11.50270945675704226, -11.92556264948662559, -11.58683381197190521, -10.22453071038868799, -12.24724620901026206, -11.35271532816773288, -11.53417784984607763, -10.04520771612846808, -11.82893144270835606, NA, -6.70547988233832370,  -9.87566936359699632, -8.61975225655713828, -9.03793964254199977, -10.15795419870432781, NA,  -7.94195417915102553, -7.58410176185493867, -13.50770294392060400, -9.51147049999762118, NA, -11.75972389651822425, -9.85015458000852817, -11.33427505898756671, -7.85072873547325045, -11.23204586870453170, -11.25883915775226463, -12.10295257749627496, NA, -7.62967796931994258, -11.49731269565540259, -11.89254650842542027, -11.55319379995716744, -10.21075346071359391, -12.23247219692689391, -11.34910504047907054, -11.51410702431108213, -10.00504466512153456, -11.79845457470167958, NA, -6.65260815075901668,  -9.86864205601401778, -8.60152888992157472, -9.02352142738304153, -10.13311087962923018, NA,  -7.93550818479496556, -7.55914439174709685, -13.50168838684927231, -9.48864259422227008, NA, -11.75814706865195269, -9.84518157581213771, -11.31356568147931085, -7.82274319317787725, -11.22110356006111331, -11.24048344001660737, -11.97849000668912112, NA, -7.59890340086023297, -11.47927990308666146, -11.86204127425864918, -11.47708460725642965, -10.18056516290806179, -12.20996275750731996, -11.32797776338841800, -11.50397203890176279, -9.97457433038425023, -11.76181825829782923, NA, -6.61788898063658948,  -9.86403047921650789, -8.58727757223748078, -9.02037574524679897, -10.11126715615614380, NA,  -7.97252697592958093, -7.57601046573955372, -13.48642773357203062, -9.47135084679473493, NA, -11.76253953563142574, -9.84493021219124387, -11.28771740890966768, -7.80774347942243985, -11.21102234634116002, -11.20729904824796463, -11.88346541321712202, NA, -7.58759053961263419, -11.47289408207081252, -11.80781040477350352, -11.37563694389301361, -10.12783925795961437, -12.16742554167725388, -11.28085075438944429, -11.47139229148124073, -9.92628281193841566, -11.70872416967051421, NA, -6.56024773316974930,  -9.83542506119156101, -8.55676466483321541, -9.00176358971245172, -10.05488506058807907, NA,  -7.97186814733126514, -7.56422579138077733, -13.44850410552934150, -9.44073652843962563, NA, -11.74385624951894158, -9.82598689136206893, -11.24284106788937621, -7.77460118644292564, -11.16740326471694544, -11.15227176605118586, -11.75895997770249402, NA, -7.56460361961223970, -11.44083952606317744, -11.74274167567885208, -11.25530852059547726, -10.06685332947502332, -12.12679824106724169, -11.22932813363508586, -11.44342385857429178, -9.87577843595581939, -11.65160365076152615, NA, -6.49761299228058231,  -9.80418215078581845, -8.51968782135947933, -8.98289275442728474, -10.00116285411531436, NA,  -7.96983786135663408, -7.54713733602688475, -13.41155032871192176, -9.40650592301670585, NA, -11.71440939753789046, -9.80554799512001907, -11.19625314053545750, -7.73563984394461812, -11.12335518943913648, -11.10311969512201458, -11.58372805813183248, NA, -7.53248189581605310, -11.40253872098156407, -11.70812793525782425, -11.14957802999728997, -10.01138824861269860, -12.10737337066842123, -11.18886466853266981, -11.43117521958704863, -9.84741138262081606, -11.61297417838850876, NA, -6.46814459885247839,  -9.77836078344800796, -8.49839232207486717, -8.98296247588876540, -9.95883719800236378, NA,  -7.97675593136926242, -7.52790743423728959, -13.38771709527822829, -9.38658138528069053, NA, -11.69191887445637335, -9.80518835208583361, -11.16113633750882528, -7.72187708972487208, -11.09650600704310364, -11.07527705100899240, -11.45362064562696602, NA, -7.50262662432821603, -11.38042950339591464, -11.65153654875128098, -11.03011383040786342, -9.95307114464661069, -12.09589791863162489, -11.13544660159123190, -11.41337858223103829, -9.79870578984056806, -11.55236181565080322, NA, -6.42383017972230341,  -9.75216700213599452, -8.46889543165289993, -8.98574864738207602, -9.92178246476628090, NA,  -7.98429553256039615, -7.52333441119074742, -13.37049262345285605, -9.36428167818572277, NA, -11.65747157423039759, -9.80901241782723687, -11.12961823853395948, -7.70692737894871893, -11.07346008665009407, -11.05220812560888355, -11.13010891060728014, NA, -7.46049403392569488, -11.34317520798502876, -11.60951606475711273, -11.00923412540961976, -9.90822475367943589, -12.09307216332187096, -11.11288617188864869, -11.40283284168207878, -9.74641242502898564, -11.51653432837391833, NA, -6.38851784648419674,  -9.73749277888544462, -8.44447411846272189, -8.99365993676939368, -9.88011877046501041, NA,  -7.98807326222249081, -7.51686950771945650, -13.36517167201214917, -9.35900910567379185, NA, -11.61792836486810110, -9.81406942806210125, -11.11268982743948541, -7.70176815112936097, -11.05948553291809766, -11.03520678086139029, -10.97085908377301422, NA, -7.42559943659736632, -11.32130490362245823, -11.55802648080693373, -10.98516339658448437, -9.86729109677977512, -12.09512886997774750, -11.09869442343648416, -11.38143713118220290, -9.68661037004279635, -11.47562451044349530, NA, -6.34130540048146507,  -9.72775692237516587, -8.41985282662274948, -8.99617163551333654, -9.83144286427953062, NA,  -7.98198425319561800, -7.51131157715570730, -13.35977619131817207, -9.35054924947774246, NA, -11.57947003245631379, -9.81048734296826730, -11.08312533035481451, -7.68781986094284431, -11.03607372002989884, -11.01518345131343679, -10.84936486886942220, NA, -7.38210012236030799, -11.30681469093385161, -11.49953098688621722, -10.98993930302720656, -9.84130049053949030, -12.09271411943809760, -11.08724993783145152, -11.36435808609405562, -9.64660654088854486, -11.44321146504625020, NA, -6.28855593013248715,  -9.71702316498315000, -8.40227086917242083, -8.99821768940145716, -9.78142074089193692, NA,  -7.96440962461651836, -7.51122010108424210, -13.36162033024571016, -9.35421481670874932, NA, -11.55882006104612003, -9.80231239180474212, -11.07961859283865813, -7.68101915400192503,
                 -11.01341800824934758, -10.98372988524235261, -10.82084410707249233, NA, -7.34084229803062627, -11.27879560763079247, -11.42498816469566947, -10.96008725196166544, -9.80737323137308437, -12.07639805452960502, -11.06512325069927805, -11.33361004696488195, -9.59751425204342290, -11.38879243874843183, NA, -6.21094917606605446,  -9.70942439667684809, -8.38192479519786282, -8.98078702142385055, -9.72762369459863052, NA,  -7.93738519923342434, -7.48917191489489475, -13.35509541756494301, -9.34118459225702757, NA, -11.52144007958631455, -9.78058605186177488, -11.06970510442617872, -7.66827165428115354, -10.97779537780803771, -10.93722494786163679, -10.67691833126593082, NA, -7.30276417047747639, -11.24387223254735702, -11.35488856862232510, -10.91506768858571519, -9.76082574439956829, -12.05247363871388977, -11.03720094156433973, -11.30008162791257220, -9.55284188036994486, -11.32833273632481763, NA, -6.14447251595474242,  -9.67896299448614528, -8.35565225143678880, -8.96751380133857801, -9.66848258973697483, NA,  -7.90027785641969249, -7.45966915573446165, -13.34093017593963637, -9.33338684584526490, NA, -11.48881656059830014, -9.75115659923425682, -11.04949003789640116, -7.64884892038118913, -10.92622937327215382, -10.89547885991006559, -10.66241859020869320, NA, -7.25662646842009096, -11.20313662397422760, -11.28193746387865204, -10.86118473038274423, -9.72741125950440200, -12.03801598321307154, -11.00865530547728710, -11.26984290935565092, -9.52763222857550396, -11.26328609162233541, NA, -6.09937369710470811,  -9.65151834772744266, -8.32772930253895183, -8.94707349758742687, -9.63097507750320148, NA,  -7.86987249008322287, -7.42971346560430135, -13.32042421470168314, -9.31948250573863923, NA, -11.47005719442440430, -9.72809949580127764, -11.04770605598128164, -7.63165479069188812, -10.90140876410607440, -10.86547547116450119, -10.57094531221246925, NA, -7.22519556517690731, -11.16586763267583393, -11.19485655284967685, -10.79211937649225561, -9.68774162030090302, -12.01798434415109895, -10.96414208336503826, -11.23361082074462480, -9.49547483334396603, -11.18720223991863705, NA, -6.03051027703943809,  -9.61810021935745141, -8.28366348596115820, -8.91983179542284788, -9.58402068835984089, NA,  -7.82685996041639243, -7.38712173780206172, -13.29751578140789192, -9.29082695223834598, NA, -11.44244958351447927, -9.69452711398172795, -11.03656374384291539, -7.59990705900400432, -10.86981701699808589, -10.82869977344045509, -10.50888026687382748, NA, -7.18018408813019704, -11.12129293049388501, -11.16985187832304227, -10.75602162622842961, -9.68976233611604876, -12.01355639275456078, -10.94989362062308658, -11.22303661547703868, -9.49584481017817694, -11.18630949963799992, NA, -6.03874993719993114,  -9.61111952495402200, -8.28218271993970845, -8.90458977418582442, -9.56302888860181710, NA,  -7.82058884439131763, -7.36859101151219953, -13.30639085841252367, -9.28310524520221136, NA, -11.46090792639255085, -9.68704586289827496, -11.05549207923678345, -7.60113501893690113, -10.87540010557374970, -10.81009357969457341, -10.61308163333663401, NA, -7.18986335571544188, -11.11084386743935148, -11.13930840672657929, -10.70604842679164115, -9.68872377973134924, -11.99785392943054596, -10.93503598471890470, -11.20626014039417484, -9.49952838972162716, -11.18507346497628241, NA, -6.02973833041227358,  -9.58418056228205906, -8.27623593697155968, -8.88297014476859204, -9.53578492759062080, NA,  -7.81233593314567454, -7.34662794529009400, -13.30687610112843799, -9.27381456531383463, NA, -11.46646154003447293, -9.67681451092580858, -11.06565091635484777, -7.59637857131726335, -10.87247190019741794, -10.79014972684790230, -10.69367336310135741, NA, -7.18526550172531575, -11.08927420198377867, -11.13079536993497243, -10.69835095353311338, -9.69937538507978658, -11.99610010185898190, -10.92966588436965480, -11.20497926683065337, -9.52502728812765120, -11.20649965723871055, NA, -6.05622663189702681,  -9.57924815866326362, -8.28765001210061669, -8.88390697399901974, -9.53031750529647681, NA,  -7.81414337604189146, -7.33480164451525063, -13.32347097194282881, -9.27460996065291354, NA, -11.49002876067184431, -9.68164437135194511, -11.09365281895053812, -7.60433332714527754, -10.88913717037978834, -10.78946901005488890, -10.73810342697039566, NA, -7.19321643355546669, -11.08433223099974541, -11.09209604165655172, -10.66151635370649586, -9.69165310390557622, -11.97766558185033503, -10.91419156038341143, -11.18606715960006781, -9.53216829549251088, -11.21617555647628173, NA, -6.05324487157513325,  -9.56542415095045229, -8.28950513415230361, -8.87015484884404870, -9.51644576965883893, NA,  -7.79969094371833105, -7.30892133273725264, -13.32608225034745608, -9.26319045874854829, NA, -11.50192348889127025, -9.67880962166469772, -11.11458593085996682, -7.59939675925493763, -10.89242846918568652, -10.77796837590390133, -10.71303866159726681, NA, -7.19029641907224715, -11.06894432647511195, -11.09472801368404760, -10.65305194910554398, -9.68356861361410282, -11.97793357498023958, -10.91183540269016028, -11.17238875859048264, -9.54467347344792039, -11.22775992924707467, NA, -6.06863564795694099,  -9.55905516289088197, -8.29103187323294222, -8.87519845898536808, -9.51803925457311806, NA,  -7.79158879392417258, -7.29832178488480388, -13.32780082815583178, -9.26884203978898924, NA, -11.51660442176073218, -9.68893624739897241, -11.15102149814835997, -7.60440644020562040, -10.90747079054316870, -10.77942294422825498, -10.73968998120876606, NA, -7.20118828562328694, -11.06319089288804136, -11.08284616852960980, -10.59636009366385423, -9.65921715596053154, -11.96795554339535528, -10.89329977908488267, -11.14628821784848434, -9.55308635096785075, -11.20799830768076077, NA, -6.04854284567490375,  -9.54029707991267806, -8.27370242492159136, -8.86183283373808095, -9.51016324073189168, NA,  -7.76727763832112217, -7.26978218248913599, -13.31799562541129411, -9.25271821458067123, NA, -11.51052873028551815, -9.68181422134887093, -11.16901446594271441, -7.59093146439082300, -10.90425690221776556, -10.76751657763771597, -10.74062757036806204, NA, -7.18936328345082920, -11.04755357780824632, -11.07398909720632396, -10.57381830531733691, -9.64103001448217078, -11.96145424840098492, -10.86755055233836131, -11.12918895928876850, -9.55278642052632598, -11.18819362483995761, NA, -6.01667583532669603,  -9.51568279077048018, -8.25605838139046710, -8.85278158477918176, -9.49445833951411267, NA,  -7.73445066074922227, -7.23668848819408073, -13.30905665253152037, -9.23793336995324132, NA, -11.50389501871817188, -9.67606530076171367, -11.19010852024806368, -7.57840496450995449, -10.88889443405113866, -10.75385067573071396, -10.71839656724491618, NA, -7.17636199121622731, -11.03862061721783583, -11.05600711460712127, -10.52317535972627205, -9.63176012300208662, -11.95187730976141260, -10.85022484257875419, -11.12767130290064443, -9.55151641946190111, -11.17114079518711378, NA, -5.97112693744063350,  -9.50222651792935125, -8.23863251643754602, -8.84605710995306893, -9.48546339825264440, NA,  -7.71429707414801413, -7.21892315518284367, -13.30300974963660465,
                 -9.22212659009580626, NA, -11.48712277100626267, -9.68106648125753821, -11.20719786905283577, -7.57569711521468214, -10.88238521364780453, -10.74377706536142796, -10.64655815125608562, NA, -7.16422219220976331, -11.04258125762226150, -11.04981280643884389, -10.49629326686508612, -9.62996001492676434, -11.94222133706559319, -10.85031958936312080, -11.13574190040164602, -9.55016644207007204, -11.16409977616407545, NA, -5.95155016973242823,  -9.49518871678831289, -8.23301602636507823, -8.84234096450801665, -9.48159438275881072, NA,  -7.70236506968888524, -7.20975274438620595, -13.30456148050825149, -9.21107424517280826, NA, -11.48213056992173087, -9.68452939599093554, -11.21955740692589742, -7.57671591694074475, -10.88159787857479088, -10.74893164614004171, -10.66487628373087659, NA, -7.17242753806165734, -11.06349391892505629, -11.03801636499898109, -10.45538115828287928, -9.62483506388356425, -11.93050362959815835, -10.84547509178288571, -11.14162684045964724, -9.55322684666988664, -11.15339054151795573, NA, -5.92720054649391415,  -9.48282274101777212, -8.22393788516193780, -8.83348239318460315, -9.47568219738431594, NA,  -7.68396740659885413, -7.19675845573553197, -13.30400162848196288, -9.19448195220719633, NA, -11.47180378492709529, -9.68217060461162582, -11.23098839251052539, -7.57484356817101112, -10.87986771750883008, -10.75075561394181101, -10.68982364870047164, NA, -7.18102618942298054, -11.08021545080499237, -11.01621314405864460, -10.39066524071097319, -9.61551970842764092, -11.91282641428135491, -10.84032528536870821, -11.14464949596271204, -9.55360507726735619, -11.13394124683939523, NA, -5.88893589022554043,  -9.46592365263843938, -8.21052136941085742, -8.81700748368629839, -9.46154900427943168, NA,  -7.65907912597036855, -7.17897521487434442, -13.30357730308613995, -9.17109247978632780, NA, -11.45556989672814652, -9.67450754136231339, -11.23753363489095491, -7.57056644006879687, -10.87531642436330515, -10.74908737087914723, -10.70674343320914268, NA, -7.18607948625408177, -11.08790427403308954, -11.00290189567624743, -10.33750292320710429, -9.60550132441924553, -11.90239617363568314, -10.83425886737457233, -11.13828645845375753, -9.54616380700092115, -11.12123531879742089, NA, -5.85515780091480131,  -9.45759703914474770, -8.20343512147461595, -8.80987965177956589, -9.44587586729445050, NA,  -7.64637446390264852, -7.15236474661551203, -13.30185450920364687, -9.15817875725609376, NA, -11.43884897007172796, -9.67262381022973017, -11.23740443864993388, -7.56168327489148506, -10.86647259520498032, -10.74246613775286008, -10.62127933142599545, NA, -7.18580116016382409, -11.09157808690133074, -10.99722226115165746, -10.27386347574848813, -9.60922864989756853, -11.90031006670914238, -10.82870072585124532, -11.13692366926984789, -9.54199006431957741, -11.11178242085886758, NA, -5.82289398701642380,  -9.45768829568663350, -8.19384951288692598, -8.80943265400517284, -9.43961265349238410, NA,  -7.64078783926648786, -7.12535485789626311, -13.29731991254908863, -9.14523024465749756, NA, -11.43526640766178737, -9.67743850589088517, -11.23699792961309996, -7.55197993253902489, -10.86424603037749925, -10.74407432344131408, -10.56501871909092927, NA, -7.18528208428126636, -11.09818785854459122, -10.98640890401575199, -10.23167499445410478, -9.60943849659271621, -11.89225702332121770, -10.82350164001161019, -11.13140449700674672, -9.54177023015782311, -11.10776744222621737, NA, -5.80655271251956950,  -9.45880292924384669, -8.18562155369153466, -8.81140037449321056, -9.42801229010905040, NA,  -7.63609474786268816, -7.10477718057105445, -13.29933621905581731, -9.13181753199274482, NA, -11.43005842430649643, -9.68099814483116639, -11.23705117693563871, -7.54729023982404978, -10.86408014894836604, -10.74035814363934804, -10.55586586438337093, NA, -7.18211356793051348, -11.09895188225875273, -10.98535126264306427, -10.21550524251701297, -9.60892891713386277, -11.89190780263273872, -10.81648026825253694, -11.12583843147130302, -9.55031641413770771, -11.10464699864375682, NA, -5.79269727053398320,  -9.46254540680966194, -8.18336154069056221, -8.81836785194463957, -9.41822017987564664, NA,  -7.63265663954078555, -7.09281437305759699, -13.30089015863479673, -9.12583317544228656, NA, -11.42748254134182240, -9.68992521050584088, -11.24175833437927885, -7.54034548455642817, -10.86161928125076059, -10.74136609388880181, -10.58574955755049452, NA, -7.18096498410480599, -11.10823450357833941, -10.98061249089446711, -10.18257498460483923, -9.60562335771949982, -11.88907755207603678, -10.80536690731130278, -11.11629096512756831, -9.56298956322436844, -11.09996496628153473, NA, -5.77781307229402685,  -9.46249445192802519, -8.17732522701888875, -8.82212027659661580, -9.40549034665304440, NA,  -7.62710862529135181, -7.07794162604808541, -13.29812345174006261, -9.11458849629883261, NA, -11.41833608888713592, -9.69621036376932643, -11.24217024272415699, -7.53022562903686321, -10.85733654943813953, -10.74169561297259889, -10.61132020296415668, NA, -7.17699632605562687, -11.11342527813119752, -10.97558155912860656, -10.16002700862076047, -9.60362793223205813, -11.88698449607252883, -10.79402175629023297, -11.10898312703692703, -9.57939698247179727, -11.10698464483407299, NA, -5.77435049091470010,  -9.45834389999289193, -8.17129492604658303, -8.82394886869581185, -9.39896880424484493, NA,  -7.62160449453314026, -7.06246684534139835, -13.29876824129907575, -9.10448506136875224, NA, -11.41204776301476187, -9.70340402588071527, -11.24648152543623425, -7.51699851119061790, -10.85709488494032904, -10.74160699887319836, -10.67455426761050141, NA, -7.17773905783244714, -11.12020981619128435, -10.98082479804569900, -10.16343991474129460, -9.60052943437862183, -11.89354887931482452, -10.78554503502054018, -11.11023128346009514, -9.60277293291427547, -11.12198898815722004, NA, -5.78640803213106558,  -9.45281192533423820, -8.17139255613542481, -8.82810641095186810, -9.40558468955356908, NA,  -7.62436754037173792, -7.05813685085499554, -13.30012516049028193, -9.10324950307459169, NA, -11.41264265658769794, -9.71118786188262639, -11.25037223729800395, -7.51201663461405555, -10.86806739317214010, -10.74330491543065236, -10.73656209918899584, NA, -7.18575508413336372, -11.12990157939309555, -10.98226398963967831, -10.18031795390531258, -9.59696560440490742, -11.90115181057538862, -10.78352260689649533, -11.11300360682720445, -9.61433963324995311, -11.14084759314568807, NA, -5.80556240017870451,  -9.45342317448601754, -8.17486871723811781, -8.83481808393555390, -9.41289348749409349, NA,  -7.63202536018911104, -7.05731751447310351, -13.30503229120148667, -9.10502865389212168, NA, -11.41742489520668435, -9.71776365269688291, -11.25454956507704907, -7.51193443628673840, -10.87682465890512695, -10.74283262074707679, -10.74374893644987594, NA, -7.19143627758233528, -11.13959299247790824, -10.98720414338262330, -10.18710673812085155, -9.59218588028578800, -11.90699591143147096, -10.78156164866046041, -11.11563166006779468, -9.62077126471999833, -11.16076227920130748, NA, -5.82823582244350558,  -9.44911261863501650, -8.17566376659475047,
                 -8.83776020130054540, -9.41953757104874256, NA,  -7.63510561447364999, -7.05608088200646755, -13.30836465650135736, -9.10209966207759358, NA, -11.42049140527444884, -9.72253987115574070, -11.25654161058528757, -7.50950989675162006, -10.88627835184446724, -10.73878129296219441, -10.78100045109458449, NA, -7.20041319121988721, -11.14614454791598597, -10.99444460054917450, -10.17599890579769628, -9.59583838099909592, -11.91000504558494200, -10.78583103482346317, -11.11888009977022840, -9.62960103594372718, -11.16865433059911794, NA, -5.83057396626310798,  -9.45391616366557663, -8.17923228443066286, -8.84173009177032654, -9.42570371295198939, NA,  -7.63904421377250920, -7.05957203110689502, -13.31350715892086001, -9.09721745314260311, NA, -11.42470475378438799, -9.72715716438823286, -11.26241726677721111, -7.51524403719433920, -10.89297819423447322, -10.74302789612684172, -10.85944206135522982, NA, -7.21137538613022322, -11.15358940364675888,   0.05857342781893282, 0.05406159697401094, 0.05700582988753191, 0.06241086753294101, 0.06314760513789908,   0.05886156554013049, 0.04354528283281403, 0.06296333053667430, NA, 0.05880270709275377,   0.04820372282674491, 0.05165103417912762, 0.04732638603272275, 0.05509254027742023, NA,   0.06578316528656948, 0.06620670306117507, 0.05290909479807301, 0.05085411119476953, NA,   0.05924270033260785, 0.04864740527013311, 0.04424140309824949, 0.05303479170411466, 0.05353039708802453,   0.05833051266314326, 0.06069984263425739, NA, 0.05730189965460456, 0.07241486732122258,   0.05857342935031457, 0.05406159725640576, 0.05700583002413403, 0.06241086762672533, 0.06314760503249779,   0.05886156547262819, 0.04354528243269018, 0.06296333129547212, NA, 0.05880270730929527,   0.04820372178192191, 0.05165103406715404, 0.04732638595584512, 0.05509254062288317, NA,   0.06578316527413930, 0.06620670295687013, 0.05290909490017009, 0.05085411062745621, NA,   0.05924270045961635, 0.04864740488780542, 0.04424140295420679, 0.05303479132766197, 0.05353039721776032,   0.05833051239881803, 0.06069984321272152, NA, 0.05730189990724169, 0.07241486747839260,   0.05937525049217916, 0.05463645826156709, 0.05697041267911510, 0.06306730578761906, 0.06342634655937097,   0.06067462312973606, 0.04490204879224681, 0.06451497598143145, NA, 0.05948847239576064,   0.04924839695340116, 0.05186934420871676, 0.04914424214092059, 0.05637825978282292, NA,   0.06812062619883438, 0.06821347480567301, 0.05453285971653229, 0.05255156219815674, NA,   0.06013702834509619, 0.05078058059958215, 0.04488376716092500, 0.05307329230840912, 0.05582561620014979,   0.05961378974879356, 0.06161092784835855, NA, 0.06040168054859577, 0.07387164093134245,   0.05773700065886019, 0.05410328783466267, 0.05494186038279975, 0.06160307129778862, 0.06286332611036700,   0.05921321946343998, 0.04311539376827721, 0.06417451970837829, NA, 0.05699709412229945,   0.04862348173781741, 0.04994075067823783, 0.04781396308209920, 0.05577992614610183, NA,   0.06826403531672214, 0.06669563595276982, 0.05408418069732414, 0.05142588138202605, NA,   0.06079825894892027, 0.05084626145279861, 0.04387997730456077, 0.05030652211177550, 0.05537739261247161,   0.05872900768310107, 0.06169730238839877, NA, 0.05864854235429619, 0.07203061906972341,   0.05684521990317649, 0.05444125736999987, 0.05419879039654603, 0.06178357848302698, 0.06294903232372273,   0.05913116906098056, 0.04194600645822788, 0.06400975966075274, NA, 0.05435698380415916,   0.04872515500430116, 0.04855688767752555, 0.04756126451149813, 0.05722148742603855, NA,   0.06987769095090143, 0.06488542026155485, 0.05466654078411282, 0.05109168590013007, NA,   0.06206405483484762, 0.05188590499607573, 0.04500324260282201, 0.04915510706632097, 0.05776830865476149,   0.05880060133671611, 0.06213108418814083, NA, 0.05582973255886501, 0.07113069573841618,   0.05507747598506597, 0.05266960817938775, 0.05199287718917118, 0.05936279733400626, 0.05976627195998323,   0.05648435413378410, 0.04070791181557766, 0.06155168765068270, NA, 0.05153530583414405,   0.04661815534874331, 0.04674215053365292, 0.04638396705395629, 0.05391576699440981, NA,   0.06710870655492798, 0.05989206093022302, 0.05330067786449913, 0.04995566694476945, NA,   0.06084897896951398, 0.05077309455555161, 0.04361792962087432, 0.04712513295960979, 0.05411219591624186,   0.05604507450459004, 0.06109111159663844, NA, 0.05235431961411402, 0.06734924099695232,   0.05385663943418872, 0.05187953403225368, 0.05078584310850660, 0.05854408472480245, 0.05813209002429834,   0.05602282649030806, 0.04042064869309171, 0.06058836835780095, NA, 0.05046904042918483,   0.04612230963298405, 0.04638942347916213, 0.04694910539904578, 0.05275755215494590, NA,   0.06609470972903596, 0.05655990253626755, 0.05339716332792149, 0.05038591133401327, NA,   0.06064601528763413, 0.05156871095213629, 0.04417417296753539, 0.04645596022978286, 0.05241309825699664,   0.05543405945993250, 0.06067158806219657, NA, 0.05075662552590082, 0.06484390667779721,   0.05374616618063762, 0.05210839787103189, 0.05024264785826960, 0.05826003671247591, 0.05744416749564962,   0.05769318551043203, 0.04078939369517952, 0.06104955094465331, NA, 0.05058251919492548,   0.04775785641456461, 0.04716806843966939, 0.04871489758305798, 0.05347579171983538, NA,   0.06685844135137117, 0.05563327743578718, 0.05462270642528331, 0.05158574483027421, NA,   0.06146603415301651, 0.05334602147589720, 0.04596484989762080, 0.04620960540587067, 0.05151947314907692,   0.05573879084079469, 0.06091861347259467, NA, 0.05072156641468574, 0.06382465949070326,   0.05309082961228985, 0.05300648739918486, 0.04900039504319915, 0.05918592851924685, 0.05745448990401960,   0.05905317874558133, 0.04025741657815998, 0.06121955290324523, NA, 0.04975166333779693,   0.04771202482263427, 0.04773127588563378, 0.05001523453866106, 0.05394712312583070, NA,   0.06691247381166969, 0.05650790808141144, 0.05707509622719232, 0.05302229410919279, NA,   0.06256162882764095, 0.05536027411550262, 0.04776877016916093, 0.04566769695012477, 0.05044766927681649,   0.05667927389389274, 0.06123896140352893, NA, 0.05042787980419812, 0.06237152968491994,   0.05216979975901258, 0.05332749221166223, 0.04719336120822958, 0.05993857682362574, 0.05847389671561896,   0.05962183219825668, 0.04012907772349644, 0.06000552694909165, NA, 0.04899828877848275,   0.04723423527401777, 0.04804057434429061, 0.04973653322294808, 0.05224063092329640, NA,   0.06665166446712294, 0.05640422463004783, 0.05841141024745748, 0.05429810361141663, NA,   0.06285725683930887, 0.05596505644841922, 0.04933622143654606, 0.04516123305487284, 0.05017896019041695,   0.05596359819243454, 0.06107539384405192, NA, 0.04879889364258819, 0.06160867905708824,   0.05064347693758173, 0.05298957421783815, 0.04474408488614545, 0.06057420899485895, 0.05929166278027530,   0.05916989365041236,
                 0.03817583326021697, 0.05890513910924440, NA, 0.04762419525298049,   0.04766399424306465, 0.04692581812567638, 0.04897356682309800, 0.04967689369558391, NA,   0.06503949985801512, 0.05668698495482194, 0.05904954877409992, 0.05451271831680207, NA,   0.06282923233529991, 0.05709187057341761, 0.05001142255855399, 0.04367760242106897, 0.04965006332139067,   0.05518467639738098, 0.06119244844568930, NA, 0.04758998973078844, 0.06131666417514516,   0.04781247472142649, 0.05230889191645662, 0.04239641973974426, 0.05963287427314444, 0.05711426735541002,   0.05676034561055415, 0.03745516741828453, 0.05631968191413831, NA, 0.04373372509140640,   0.04699700160511822, 0.04517002831587905, 0.04713033118906621, 0.04646849619656716, NA,   0.06166250717124842, 0.05500295040827991, 0.05853225468076721, 0.05297089335781711, NA,   0.06164856595987086, 0.05539578823444835, 0.04832129625428005, 0.04172116180064996, 0.04727490297584629,   0.05120632081972624, 0.06024662186974417, NA, 0.04422319546872754, 0.05813461985380541,   0.04471500246924434, 0.05217523918836404, 0.04081171280829214, 0.05883652975275550, 0.05588806918238527,   0.05358503694285042, 0.03755242656497075, 0.05383685054436689, NA, 0.04068480043447019,   0.04747991728415287, 0.04514202866316475, 0.04515284461586707, 0.04395799393472440, NA,   0.06100015476813586, 0.05241861548334976, 0.05717810610989654, 0.05268438572546620, NA,   0.06041069400477721, 0.05380678307781944, 0.04706375112153934, 0.04094667193504985, 0.04418858202702704,   0.04849388431435966, 0.05891292177325026, NA, 0.04316311848877658, 0.05657827714625185,   0.04201192073349379, 0.05160818848264984, 0.03936252581733408, 0.05813420348469739, 0.05274111045619624,   0.05167870087612540, 0.03636823352188763, 0.05082737144186723, NA, 0.03682448559603343,   0.04815752495425906, 0.04373553758189196, 0.04523975475101726, 0.04081841379190698, NA,   0.05869133513540312, 0.04990061838401736, 0.05645592202675247, 0.05168371627597459, NA,   0.05825318870453364, 0.05140264643827296, 0.04388022721744071, 0.03946980564248457, 0.04232057809105522,   0.04527230242666781, 0.05756702955658519, NA, 0.04075846156819875, 0.05518062128477266,   0.03892729169949956, 0.05098250611726560, 0.03734399192549519, 0.05691723210220589, 0.05061388898585290,   0.04872956233179253, 0.03619756781282361, 0.04751040829023898, NA, 0.03355054688212560,   0.04843534799864409, 0.04282776301474974, 0.04448440894611289, 0.03814840483584996, NA,   0.05612404656938126, 0.04772180289493077, 0.05450653953896133, 0.04974617655671348, NA,   0.05689698768988992, 0.04931715296386436, 0.04092646515056957, 0.03863907325571081, 0.04047557069947369,   0.04259041544657696, 0.05653539744068797, NA, 0.03865609666166925, 0.05432968316769021,   0.03629880803981649, 0.04993616890877777, 0.03617121538017730, 0.05689467887158341, 0.04761515911840298,   0.04641782598461240, 0.03623554157102017, 0.04446730442749112, NA, 0.03100442479736753,   0.04931083920017973, 0.04173861794681804, 0.04478576374883690, 0.03618811158557605, NA,   0.05384022440641455, 0.04529010807160120, 0.05434782223171031, 0.04855613822449269, NA,   0.05577488307740282, 0.04696502454234711, 0.03910748445995409, 0.03788049575791491, 0.03901682054567444,   0.04031382293018769, 0.05537068425896427, NA, 0.03624718724220806, 0.05404605086516802,   0.03652545951076031, 0.04930409540474120, 0.03789820915402727, 0.05755803390402115, 0.04620927072804728,   0.04638595694457313, 0.03744641215251896, 0.04525571805602225, NA, 0.03316872315861567,   0.05027842095168382, 0.04244970701494236, 0.04621694900019034, 0.03492791896655469, NA,   0.05305588430375507, 0.04468749673274253, 0.05617278416247484, 0.04864748197750550, NA,   0.05648973717786293, 0.04572904072377326, 0.03826433045563245, 0.03812178116055305, 0.03921792459321240,   0.03886935484655208, 0.05463758527839892, NA, 0.03642319692074676, 0.05538042493914784,   0.03687595179450140, 0.04880743339231289, 0.04026801833204421, 0.05740254244440331, 0.04528613376594766,   0.04726874886146597, 0.03895641368723753, 0.04543589276685424, NA, 0.03455566790978289,   0.04939286757005466, 0.04253556215692414, 0.04638423050014012, 0.03299995736024006, NA,   0.05207244008380097, 0.04424491655297494, 0.05754030945381587, 0.04823921499233878, NA,   0.05672259556271707, 0.04568868611406193, 0.03731471439928100, 0.03789563823824482, 0.03979177544656732,   0.03843877358275857, 0.05413184632298276, NA, 0.03544461852363983, 0.05608378945653719,   0.03774761025484501, 0.04828130230706017, 0.04226335695548175, 0.05667862947149102, 0.04494058671562878,   0.04913979288612474, 0.04123949270609824, 0.04518003835716422, NA, 0.03497581011837406,   0.04752587589895037, 0.04225078222460422, 0.04667280674212024, 0.03109534197958748, NA,   0.05180390427924342, 0.04313694333543498, 0.05882803945394566, 0.04777853255827872, NA,   0.05747789833883928, 0.04622855661590488, 0.03707586324920369, 0.03672224953223697, 0.04000838840907384,   0.03866027399381181, 0.05359359732784757, NA, 0.03472939260080313, 0.05731835382061756,   0.03706638156757332, 0.04734759721361469, 0.04195124923526007, 0.05454595055287850, 0.04411564370028091,   0.04885611719751998, 0.04300539160748218, 0.04574453376495415, NA, 0.03546521449896858,   0.04636750674328547, 0.04200467923491082, 0.04598876267497976, 0.02980364542060275, NA,   0.04982965487963156, 0.04148404252324043, 0.05856981592682067, 0.04675780903895081, NA,   0.05748752785104327, 0.04680584899517237, 0.03711861423173751, 0.03516456615353014, 0.03913265588045442,   0.03830906354146302, 0.05260578010269058, NA, 0.03423568687386101, 0.05793649460554534,   0.03852990912532780, 0.04720023744292675, 0.04258816965180989, 0.05409887527163072, 0.04502276491387865,   0.04896252140448327, 0.04483100938072217, 0.04692265477973877, NA, 0.03764065119871894,   0.04577168392035626, 0.04280575873000687, 0.04689734435349731, 0.02994428356761650, NA,   0.04927338532044136, 0.04165253286107914, 0.05903908913746613, 0.04682447192929043, NA,   0.05790153590035754, 0.04830081266416536, 0.03866927532107550, 0.03532294691316994, 0.04061527659510072,   0.04013694547346991, 0.05210661988999109, NA, 0.03553898442826556, 0.05888457949500062,   0.04001900101753362, 0.04714833531882549, 0.04323170825172538, 0.05411223821109801, 0.04593705382389626,   0.04965334392105092, 0.04677586411873640, 0.04800565724648164, NA, 0.03962221842201974,   0.04560439081735818, 0.04371533418684365, 0.04811532918278225, 0.03032394349793893, NA,   0.04905678691629077, 0.04204604063421995, 0.06015446531718509, 0.04742035068370602, NA,   0.05835187624502796, 0.04990485675105221, 0.04015319471275743, 0.03571434995726212, 0.04214361546249871,   0.04228671010430929, 0.05155553537923425, NA, 0.03659955039172454, 0.06044550428250386,   0.04011605595637478,
                 0.04648277884780518, 0.04255179465994906, 0.05386331214720389, 0.04628664026781446,   0.04858869887274547, 0.04683776771794373, 0.04842867519806866, NA, 0.04029909365173879,   0.04413390869865768, 0.04357554193942469, 0.04726677010847001, 0.03029093899159134, NA,   0.04811420691317687, 0.04182956695962603, 0.06025763218896324, 0.04702738432905364, NA,   0.05848034870436972, 0.05104510510336227, 0.04044717164436996, 0.03514887756109050, 0.04224983584900815,   0.04299080293145015, 0.05092880419254062, NA, 0.03673773955047097, 0.05981244798683370,   0.04027430101799584, 0.04604234702702336, 0.04255388605822317, 0.05430117003236556, 0.04734955122400587,   0.04875622847255043, 0.04709062123015220, 0.04950389286348096, NA, 0.04094528438469955,   0.04362104865719324, 0.04399816999792507, 0.04713490135838805, 0.03083517730691128, NA,   0.04816043278267147, 0.04281386163692312, 0.06143793708635841, 0.04750706245147836, NA,   0.05881708307423790, 0.05318582548794575, 0.04125797090423494, 0.03545720454608101, 0.04314722760015349,   0.04446982651941399, 0.05047629661137489, NA, 0.03715233898531091, 0.06017935620984490,   0.04091153433962320, 0.04627643232678932, 0.04251006859030020, 0.05551191854299747, 0.04914974569456206,   0.04982869529236332, 0.04828054623676840, 0.05145377078781322, NA, 0.04278218389521903,   0.04334658223153341, 0.04551386117563297, 0.04824248765525954, 0.03156910856015609, NA,   0.04873437184299116, 0.04403952685209296, 0.06338511895658493, 0.04942541690133025, NA,   0.05909321989328124, 0.05548945769763447, 0.04277436486621380, 0.03576083239346162, 0.04501222889187730,   0.04573552276061294, 0.04995433466411032, NA, 0.03839523038546544, 0.06115550708027329,   0.04222579375567444, 0.04730352214543933, 0.04310201625352345, 0.05779072474155733, 0.05163385597954483,   0.05195463278876850, 0.05039527855378344, 0.05407410380551911, NA, 0.04587272692563794,   0.04394896438946021, 0.04774392807080467, 0.05044697339464944, 0.03314334398950201, NA,   0.04987701358625317, 0.04607601350098839, 0.06661989891077706, 0.05243509438258506, NA,   0.05962826963718523, 0.05846225590818205, 0.04529042379414290, 0.03726320729758922, 0.04838779250603618,   0.04736198765185996, 0.04959843468057368, NA, 0.04064775443458309, 0.06270946842205245,   0.04322915801093666, 0.04866178129756882, 0.04300342498240038, 0.05949201432049611, 0.05406487137958257,   0.05404965259729022, 0.05303736856264466, 0.05629610484539577, NA, 0.04936882664305196,   0.04410141707628478, 0.04984665329465549, 0.05294031004371724, 0.03427768603840829, NA,   0.05080658246316531, 0.04771021541603696, 0.06996722218501186, 0.05586646614099256, NA,   0.06031392320772605, 0.06144628552154727, 0.04801027442723030, 0.03905738328641425, 0.05189677380931353,   0.04835591381146376, 0.04902182013592891, NA, 0.04313559124274505, 0.06418372298342001,   0.04156861779918762, 0.04876429912734154, 0.04073055327949425, 0.05992136643729905, 0.05301105498668626,   0.05448462386988705, 0.05335453535744161, 0.05695688780010946, NA, 0.04958073224372004,   0.04224658128044415, 0.04868615665284936, 0.05289891984244207, 0.03299277227310866, NA,   0.04893553267021684, 0.04773332621279332, 0.06964336335047965, 0.05555429388815292, NA,   0.05940082708588793, 0.06132885960304383, 0.04827233633197597, 0.03907815298663284, 0.05328143042972305,   0.04850405319662284, 0.04832069366972497, NA, 0.04432829786172066, 0.06369587540767595,   0.04099450291571533, 0.04968885339923663, 0.03996869288282261, 0.06192357081843617, 0.05296950439973851,   0.05620597979299924, 0.05425093594067263, 0.05913976185336427, NA, 0.05158247460512481,   0.04199372862969929, 0.04821982889832067, 0.05448961424379810, 0.03313806240202569, NA,   0.04866759006697718, 0.04863691487341233, 0.07027547448842193, 0.05704631194411623, NA,   0.05959943242070646, 0.06291952941213942, 0.04943908162056915, 0.03995637340742680, 0.05621287469129781,   0.04984814479640808, 0.04797838582214906, NA, 0.04641394357347288, 0.06441543888335606,   0.04054918256839241, 0.04997537985449970, 0.03913923941346122, 0.06257802233774576, 0.05195717966738196,   0.05616293519819077, 0.05498556671030448, 0.06069386316656325, NA, 0.05245641640813747,   0.04060624910374604, 0.04721083442974321, 0.05469513116137195, 0.03320163859556848, NA,   0.04860653892626787, 0.04748891083942899, 0.07004521804705215, 0.05857423547838040, NA,   0.05880964743523996, 0.06279414375999587, 0.04899057182593798, 0.03928604721671452, 0.05727514915211782,   0.04963682141163463, 0.04717288065123969, NA, 0.04767976051339289, 0.06513899726057654,   0.04051655354572144, 0.05052267038802973, 0.03854024353774393, 0.06366471006901883, 0.05141933070953340,   0.05650345874279066, 0.05610510207909648, 0.06278310912201157, NA, 0.05390681619697632,   0.03961739566592570, 0.04652119443502313, 0.05531166501002788, 0.03364731030388344, NA,   0.04915848472550907, 0.04663786295804292, 0.07038016607097405, 0.06052777927987915, NA,   0.05819420010916146, 0.06312663065169521, 0.04889794755649608, 0.03913411931369526, 0.05898856363097189,   0.04971520559812950, 0.04666046329047777, NA, 0.04944262785882230, 0.06616986972490076,   0.04076271302425308, 0.05144060926457399, 0.03827242794129882, 0.06543009851529707, 0.05146276477992164,   0.05737604288034003, 0.05773466710052520, 0.06567810275958809, NA, 0.05615298477936316,   0.03922674885767768, 0.04633726058988927, 0.05663092249098427, 0.03465060199305162, NA,   0.05068225165996658, 0.04642172614976822, 0.07150584756300034, 0.06356810008844228, NA,   0.05776078024812761, 0.06437310440905708, 0.04941979808306107, 0.03965753562893302, 0.06155425706293206,   0.05038766473995299, 0.04631033422489932, NA, 0.05186828438944562, 0.06786327544171464,   0.04134830063216050, 0.05261603053352988, 0.03789313287812108, 0.06704443389381860, 0.05090123450008485,   0.05820383675486562, 0.05960424432055954, 0.06941117168492576, NA, 0.05896306346943384,   0.03862309959253163, 0.04575999543059146, 0.05834339597229449, 0.03679076346709082, NA,   0.05312069976462220, 0.04681178257768712, 0.07247605103354225, 0.06600491662078894, NA,   0.05749833791042407, 0.06624922603371727, 0.05061270181834304, 0.03956738073332960, 0.06544469424233290,   0.05089908312099008, 0.04587546908956859, NA, 0.05453927879545051, 0.07049548234336514,   0.04178841935020969, 0.05378349605569073, 0.03783995227788499, 0.06897017585945150, 0.05061144316743046,   0.05894737479255674, 0.06149781397650313, 0.07362801478688136, NA, 0.06190037653944659,   0.03851936757089292, 0.04516937617975620, 0.06055286076970452, 0.03885031519449973, NA,   0.05593368032248153, 0.04730908065714536, 0.07394250361006065, 0.06841509900006305, NA,   0.05727632081426096, 0.06903667689028796, 0.05242106805677962, 0.03950086150400183, 0.06950514522232755,
                 0.05191538289106130, 0.04581612238542444, NA, 0.05716634668329333, 0.07347815341082822,   0.04120940477246676, 0.05445942418154349, 0.03706681251349050, 0.06912310046275993, 0.05036798702789828,   0.05885633870017770, 0.06160247175881216, 0.07566422850372910, NA, 0.06448053730548954,   0.03923378307731162, 0.04543190182664410, 0.06113116096689112, 0.03989339096579613, NA,   0.05772321403597811, 0.04743115866931477, 0.07599363076317237, 0.06897949138950671, NA,   0.05758821149308251, 0.06977376291178289, 0.05451736544041227, 0.04027448035189997, 0.07129855222182130,   0.05161640415139980, 0.04621179437802429, NA, 0.05911620975125563, 0.07565712295448584,   0.04131439046051236, 0.05557197123775011, 0.03653643985700433, 0.06981982936377870, 0.05069948324261117,   0.05918239704349088, 0.06225274597176811, 0.07877442616179142, NA, 0.06809058267324156,   0.04018229283278331, 0.04603276689397842, 0.06207897244061963, 0.04149806650099852, NA,   0.06026832681223480, 0.04830155748760596, 0.07872147465131647, 0.07057678891264116, NA,   0.05804463528981733, 0.07111662374604270, 0.05673307636196820, 0.04171914202753751, 0.07425020906059911,   0.05149389475445476, 0.04667934537509000, NA, 0.06215479396432000, 0.07848479062254569,   0.04232758313115886, 0.05723997940996819, 0.03722263329901503, 0.07085441505896571, 0.05231223530934407,   0.06007530838430550, 0.06357366954569996, 0.08262106679698233, NA, 0.07224326933748168,   0.04267498425978716, 0.04740601041995451, 0.06385189038958901, 0.04367292532080622, NA,   0.06394954844848424, 0.05040979365869429, 0.08232250998948243, 0.07385744340570849, NA,   0.05879788575705681, 0.07310772573005100, 0.06011163621110492, 0.04483086876779990, 0.07784702741760266,   0.05256101788724066, 0.04761722309201944, NA, 0.06627975208284889, 0.08229852207237098,  -3.89728137893498516, -4.32498547813995060, -5.06130012563679976, -3.60526079517661113, -4.53712917236785618,  -4.32936770472647048, -4.82273846692511032, -5.13586365314866544, NA, -4.96480140544494830,  -4.50579863751381549, -4.38430302984659903, -4.54580165080617959, -4.68509654761264205, NA,  -4.84278226307245152, -4.24349570469782389, -4.06069317085453640, -4.72811128634380573, NA,  -3.76948745400201490, -4.29071903353531692, -4.32848517070262950, -3.99811077890107702, -4.19469857549849845,  -4.46757496598289716, -5.08999377019933430, NA, -3.97528519778345668, -4.33402192134756703,  -3.89728161482316526, -4.32498565814348623, -5.06130009033122441, -3.60526078277650353, -4.53712916968307045,  -4.32936770739337184, -4.82273860918732922, -5.13586397974692943, NA, -4.96480154403264429,  -4.50579853075354286, -4.38430305764586414, -4.54580170619463697, -4.68509659596580086, NA,  -4.84278231627069822, -4.24349572538584852, -4.06069322307117631, -4.72811139264249647, NA,  -3.76948748362334118, -4.29071906285522164, -4.32848522098764121, -3.99811075708802921, -4.19469858140699792,  -4.46757500731679613, -5.08999421847373057, NA, -3.97528523373604781, -4.33402195869379003,  -3.87629103250313856, -4.24386963117821914, -5.04137441633606276, -3.57808763988875000, -4.50994153647868146,  -4.31608308189886269, -4.82290152013813067, -5.09552374757635995, NA, -4.91249265309647143,  -4.46495941917008299, -4.34758541628272255, -4.52847719002899929, -4.68129169419290569, NA,  -4.83752110104628574, -4.23754851788530029, -4.04895653714017989, -4.66149886740421415, NA,  -3.72987164169752550, -4.28212919945027348, -4.29177284757989241, -3.96207001911910339, -4.20087098657400926,  -4.45082309159647593, -5.25978257955429207, NA, -3.93567055125453580, -4.30984959534829493,  -3.84169039396976419, -4.21206966031674757, -5.04620177015507565, -3.58365284640803505, -4.53207820917135873,  -4.29698975973729258, -4.81042814474754188, -5.02569155913940069, NA, -4.85709085937691931,  -4.47387111841667373, -4.33504651924554096, -4.51034182337092382, -4.63824446766033827, NA,  -4.86433403364971984, -4.23155278982828165, -4.07256710603298711, -4.67874021131716855, NA,  -3.74876959738912596, -4.30080408760864064, -4.22957200216415696, -3.93501539301455772, -4.20397045514117362,  -4.40607980560792889, -5.04582611565705808, NA, -3.91929632408179973, -4.29546256125913928,  -3.80047293362736749, -4.09667316419891225, -5.05084150176060032, -3.58882167059677570, -4.54522729079284016,  -4.29738337982942742, -4.83045971648602190, -4.92271407236043501, NA, -4.79113668842500129,  -4.48657471440901112, -4.32195484934114837, -4.50088885341869904, -4.59695908654368957, NA,  -4.91599746394808612, -4.26794632809761065, -4.07914560954132899, -4.66885352852585633, NA,  -3.76411694276617537, -4.32971294584707156, -4.18984644030220110, -3.90980785940207731, -4.20929400789177244,  -4.33691662974804260, -4.89344485235895732, NA, -3.95770739324717580, -4.27375853787428994,  -3.68201299477046273, -3.96581626014290656, -5.00892914684533341, -3.50783224703460128, -4.49114349460502638,  -4.25009684981168068, -4.72956943598747515, -4.80305520421623555, NA, -4.67635268940278603,  -4.43274010532913909, -4.26965311020786942, -4.44318743964828755, -4.51098432318071652, NA,  -4.89493012990422738, -4.26030426421173303, -4.02616762895134617, -4.57211851471981312, NA,  -3.74677568415306705, -4.28339434308224476, -4.10374106742736977, -3.83380343702890514, -4.13731542637696847,  -4.22278828459619504, -4.71912031115106334, NA, -3.95152630393838722, -4.19773334111481589,  -3.53320736205713892, -3.79426584760175079, -4.95599480427027572, -3.39693100156295680, -4.41464556600641345,  -4.19233826860183978, -4.62851464478694385, -4.64420099240596151, NA, -4.51808087861545182,  -4.36266274112430441, -4.19230199022887629, -4.36231021279432518, -4.41871919746748976, NA,  -4.85554999289154665, -4.23921083981306257, -3.95376379260823274, -4.42563302525589464, NA,  -3.69372227184819302, -4.21390980074947663, -4.00018960570668547, -3.73407788494759751, -4.04824982072335349,  -4.09820672579915968, -4.46086536287313962, NA, -3.91672369952091382, -4.10441589650081440,  -3.44398950379776814, -3.64125492369973092, -4.91583495107516821, -3.31869405846043630, -4.35664099739141975,  -4.15409789840082855, -4.59064796771721451, -4.51575506506850921, NA, -4.42445772421607941,  -4.28506162149470526, -4.14597357474520845, -4.31944977061077928, -4.35109408163564559, NA,  -4.81607376595548331, -4.19523169569106269, -3.88954718833980806, -4.30068054798689925, NA,  -3.64539121978964253, -4.17201010972441733, -3.92322241278285322, -3.69498746692068769, -4.00530726849394547,  -4.02755386139583482, -4.26709598178439009, NA, -3.88382722682527692, -4.03318552096998317,  -3.30992942062251938, -3.50029459214432981, -4.86939525877469315, -3.23864959847792599, -4.26288648107631118,  -4.08050237927977832, -4.55487694073732996, -4.30325635633721237, NA, -4.30504713439671338,  -4.18772626974577999, -4.06315266738498515, -4.26493707212814144, -4.28042273830901099, NA,  -4.76489665404930207, -4.18031024969215359, -3.83781520191404102, -4.17693738175922569,
                 NA,  -3.55129182959671708, -4.12674154063704268, -3.84154505715615446, -3.64371372356923073, -3.95690823091790289,  -3.93627826563573802, -3.69610416117353768, NA, -3.82926240385144245, -3.92607525673713909,  -3.22328211155319799, -3.54443586823634105, -4.84623284530050036, -3.22578156727703869, -4.22450272959220019,  -4.06868393184055144, -4.42750342608458158, -4.20143949846327835, NA, -4.22968488415975408,  -4.11891159577532573, -3.97863059644983741, -4.23257694478543911, -4.19341582290794168, NA,  -4.69926243154181567, -4.17026353087901924, -3.76942074592740850, -4.13650965096111545, NA,  -3.48456247865508884, -4.09727023863719175, -3.80824116625708875, -3.61014510140438283, -3.93457758471820807,  -3.91059581442152115, -3.37417233602518518, NA, -3.81313167877812509, -3.87791849962721225,  -3.12542325881914929, -3.60921663187339492, -4.86018572300910900, -3.23249130856819411, -4.21871052193186369,  -4.07189893137525427, -4.36630990948039610, -4.10216119604663820, NA, -4.15039150069260732,  -4.08139976401884397, -3.94783504259659779, -4.19793823417645928, -4.12086954287481433, NA,  -4.67087823740551222, -4.19008335314632685, -3.72325965435510131, -4.13533787739189673, NA,  -3.44443049258266321, -4.05411857335713677, -3.79747035874287242, -3.61542874761812971, -3.91705534312929782,  -3.93059887204736258, -3.13917338239042198, NA, -3.81473352589875914, -3.86129807794398294,  -3.03225319036725649, -3.66332600586927093, -4.84555796482932255, -3.21589866041482164, -4.20694678142501033,  -4.05563273205362851, -4.23879912427343442, -4.03641799701558401, NA, -4.07736143461537903,  -4.01891566377224052, -3.88992260461457695, -4.14958328365913687, -4.03033514938234472, NA,  -4.62715992365517348, -4.18601750889814994, -3.66800713524519173, -4.12848091260537586, NA,  -3.42533338678029908, -4.00537394847564077, -3.77358553197308755, -3.59426446600483906, -3.88561128566301761,  -3.92334636324282071, -3.07487089628614241, NA, -3.78881462293173499, -3.82330006005619616,  -2.91236442151375430, -3.64907482397523930, -4.80110073690667516, -3.15874880831217997, -4.15035885371561619,  -4.02421201242014615, -4.07342615677225872, -3.91418791748586825, NA, -3.93850022888903828,  -3.94088486748968769, -3.79446532973280526, -4.06513817635753139, -3.92806496558801754, NA,  -4.53882764126151983, -4.15790053939116611, -3.60349929310182260, -4.03059006073715942, NA,  -3.38644228665511582, -3.92339219588500221, -3.72644991434795436, -3.55015949205425452, -3.83786082932334116,  -3.87332989949211814, -2.81666274989293619, NA, -3.73132324144553662, -3.75764743333134899,  -2.81672862768506871, -3.62794747801083650, -4.74188827569266103, -3.14330279063539475, -4.13017640741120839,  -4.01194597977179246, -3.98877796512890725, -3.81809235178552342, NA, -3.88204842711135401,  -3.87353065687675002, -3.74304564411831375, -4.01932416783096524, -3.89064145703787823, NA,  -4.45605056660693055, -4.13609808861251516, -3.55559400206333009, -4.03616988326612081, NA,  -3.38508477873880631, -3.88310316350472240, -3.70709496608690792, -3.54789880866638541, -3.77946454888870997,  -3.85977442939228599, -2.82263111847037163, NA, -3.71727451083918226, -3.67688834036449474,  -2.67470942675995360, -3.58980776811260771, -4.67507724082629750, -3.09549559608717306, -4.09264724302610006,  -3.95896534644995235, -3.91979530328760939, -3.66960002447654698, NA, -3.84079572260613933,  -3.76930224984858508, -3.65865541263253213, -3.92104437807473882, -3.85628988347632795, NA,  -4.34334963344163683, -4.08758353820174669, -3.45661914737687015, -3.98572181730673769, NA,  -3.36810958792235127, -3.81662358929482126, -3.68726745786191445, -3.53133868139796325, -3.72721826405662426,  -3.84206296373616096, -2.67087291179451825, NA, -3.69518212441250160, -3.58152195947183349,  -2.50519245298769011, -3.52201754127325728, -4.58908765006693109, -3.02314187059479744, -4.04342865973729726,  -3.89184285143906816, -3.83959925262545898, -3.49387078669196072, NA, -3.73048060788948099,  -3.65244864322703933, -3.55006546143738433, -3.80428745179373706, -3.80705853666809357, NA,  -4.20801630855312947, -4.02142732300730543, -3.33507767683029677, -3.87946978880396198, NA,  -3.33561637558499946, -3.73541025106643643, -3.64529917055916153, -3.49578678335209814, -3.66537681087017253,  -3.80655343495041398, -2.59153134782063077, NA, -3.65008426664424812, -3.47663816337203579,  -2.39541103609018169, -3.45887253510780557, -4.57976634019273643, -2.96772073277101267, -4.01081489369500410,  -3.85539887981542062, -3.73582233878935277, -3.45617393294098196, NA, -3.71607281326529026,  -3.57462975676773764, -3.50628285693765029, -3.70809224002235949, -3.77498038232428224, NA,  -4.13348154121032074, -3.96969287822190520, -3.25996056100211273, -3.79326627085966273, NA,  -3.34548955888349253, -3.67155302173692366, -3.66122117370760858, -3.50998390420530715, -3.64572774990121706,  -3.78195510230896348, -2.78082458815749245, NA, -3.67053895174430345, -3.39870124009050389,  -2.29153880785006336, -3.37939449478796439, -4.56630006380257303, -2.92386548869606244, -3.97775461649374451,  -3.82252534047618608, -3.65627192151343028, -3.44570603617496207, NA, -3.68094580100907010,  -3.51580106907298662, -3.48238169564685007, -3.63925221585861047, -3.75530009489727590, NA,  -4.07908900201858682, -3.92474180687195906, -3.19626800090741803, -3.73915497580865663, NA,  -3.35422241366883345, -3.60570684235622263, -3.68058339480224239, -3.53780734787139384, -3.63017163913352059,  -3.75388430242195348, -2.93603137693629535, NA, -3.70003702368477239, -3.33022074952968650,  -2.21944449017586631, -3.35712198620588698, -4.56421103649032034, -2.90730513924143885, -3.94290258639845526,  -3.80185534880984521, -3.56972673110460459, -3.49329640503741734, NA, -3.73244418263757360,  -3.49037942063183415, -3.48790846685187450, -3.59686004876879384, -3.75366116435047736, NA,  -4.04355723744095030, -3.91518161283627508, -3.15429274510081470, -3.70138451104346533, NA,  -3.36728247475678710, -3.54755729983003931, -3.69791434626053839, -3.59205555820047895, -3.63878063088652448,  -3.73520350499669318, -3.00966555187987428, NA, -3.72451168530231858, -3.28845388303911346,  -2.07131997702802506, -3.27869352940616166, -4.52555203795163674, -2.83917192655538164, -3.89121376111292783,  -3.73722179481499950, -3.41773529155225697, -3.49793096511244350, NA, -3.69492332590863004,  -3.44641596814183115, -3.46345955252414273, -3.50864085540703341, -3.72464377073806308, NA,  -3.95927854955697844, -3.85999619221704870, -3.08907153264819900, -3.62857110528732685, NA,  -3.35003615804402388, -3.45921340784173870, -3.69569653600940651, -3.60991782971036734, -3.61364163629809632,  -3.69076165965527103, -2.96107004730469336, NA, -3.71899483983512980, -3.22136210560544889,  -2.02833600954021920, -3.26441977444642495, -4.49906315138768598, -2.83675912890868132, -3.87278193833127826,  -3.73117283976672454, -3.31204616439007360, -3.52346618595394379, NA, -3.68983231158800695,  -3.46827855954721231, -3.43958521458650779, -3.49912716546254066,
                 -3.74478789377356502, NA,  -3.90576274489704867, -3.86279399832284964, -3.04506648778603140, -3.62714721731141632, NA,  -3.35615946891506045, -3.43546328404835677, -3.73594306439877544, -3.66144707427066241, -3.61171695060782749,  -3.67110253295759392, -3.02056187178737812, NA, -3.73607718764886876, -3.22332251393408686,  -1.97655993145257280, -3.15882971148293290, -4.46115092310608841, -2.81925713449366899, -3.82950820113665369,  -3.71373015629909009, -3.21108636920611179, -3.46511032236184624, NA, -3.60009920583554166,  -3.48130854611160778, -3.36480382137732059, -3.44679971016409592, -3.75617431068138163, NA,  -3.81857491592460629, -3.83033214594990001, -2.98941137237972576, -3.54743204436974757, NA,  -3.31803594769134724, -3.37587259842316678, -3.73462778954609886, -3.66237288735595534, -3.57729306439446137,  -3.61907826145277101, -3.05909002241537920, NA, -3.70123014733096900, -3.21551948384843511,  -1.97676060080613292, -3.16950098479291098, -4.48382448226893793, -2.82228410712787925, -3.79416909343957265,  -3.75548333915479171, -3.17163976218782118, -3.43073834480220174, NA, -3.51258527749498350,  -3.51697077094701482, -3.32072064672368406, -3.44999526895810593, -3.76744056628719326, NA,  -3.74393672057012550, -3.81413955062887755, -2.96203199141125584, -3.51899395326670250, NA,  -3.28756920259935592, -3.34922436340751295, -3.75785923090621798, -3.68339185542267389, -3.54957231065686818,  -3.59536756175789041, -3.06857963266913769, NA, -3.68939707585398580, -3.25218847187759508,  -1.97322082984232794, -3.12987084749112610, -4.48704321912941317, -2.82535734121132132, -3.73632362362680315,  -3.77714927291231906, -3.13487102336241508, -3.35043166341359999, NA, -3.38276887265693782,  -3.52500291556216183, -3.25621932791859958, -3.43359656001624369, -3.75581103225692470, NA,  -3.65267632441239298, -3.77515850820754606, -2.93305710833787758, -3.44295299542921907, NA,  -3.23968975495232892, -3.29414809889863225, -3.77575235564425959, -3.68128071794856027, -3.50322719289358320,  -3.56305279451925783, -2.98796201944040796, NA, -3.65334567102266172, -3.26317926153595117,  -1.97844160404570135, -3.07966669491119349, -4.49972867022010536, -2.80693129589339874, -3.69392877505329320,  -3.80760363770353072, -3.04139233290892186, -3.26040443495097065, NA, -3.28435474496761959,  -3.52753164707533662, -3.18378880866198788, -3.38771572682976307, -3.74079871130982422, NA,  -3.55380770532507295, -3.73938055667317126, -2.89610408663612606, -3.32776739841380076, NA,  -3.20292004142357944, -3.22037783490931195, -3.79097291418703186, -3.67631594556416097, -3.46322091870047677,  -3.54578447863527479, -3.05848014947530844, NA, -3.62641796525228965, -3.28881830997909885,  -1.96127133204275816, -2.98111235901972593, -4.49979013556604901, -2.77201925267938698, -3.65355134501768930,  -3.82332223471856469, -2.92068556250900713, -3.16190435589777108, NA, -3.15097359710075642,  -3.51405208634985433, -3.10542613787697874, -3.31943156387414406, -3.70670106401736987, NA,  -3.44342752161601595, -3.69964819169934689, -2.84474424283432814, -3.17977206998419115, NA,  -3.15446741087341920, -3.13605327857365968, -3.79375405121086207, -3.65192130627452061, -3.40635419633689951,  -3.52996018062394956, -3.14428706974051808, NA, -3.57342467106148565, -3.30731588299612467,  -1.90060202001066392, -2.82367319658733829, -4.45746874339481280, -2.68865463676539562, -3.59269455273904459,  -3.81462234561892011, -2.73215249775421443, -3.01494063206337248, NA, -2.96523394720179390,  -3.46366780914294514, -2.99045340538733662, -3.22407588041543303, -3.63427718273853273, NA,  -3.29762224004222659, -3.62002112544514398, -2.77733781322561502, -2.97127308174139948, NA,  -3.08953242653752769, -3.02190166861014120, -3.77447499926047314, -3.61297123484467031, -3.32761657457024285,  -3.47966698483235026, -3.21661497402006802, NA, -3.50046110647576958, -3.28363886860280729,  -1.92925006171731073, -2.70422652417908127, -4.44892144471282425, -2.64150012912891308, -3.56891107211780589,  -3.77298095227654651, -2.66455932204337165, -2.92847492439937618, NA, -2.86284430053719952,  -3.46320163499803568, -2.94623947959003640, -3.19360448638984096, -3.58637197781118111, NA,  -3.24836989464275394, -3.51825366128630224, -2.74305449516177413, -2.90325682273752683, NA,  -3.06957809136174831, -2.98129324468446910, -3.74264702982537756, -3.56896093391308877, -3.24930280394418070,  -3.41720428475997284, -3.07535880350145829, NA, -3.44295759374431043, -3.27600276231752385,  -1.92591639472499287, -2.52662691641856973, -4.41629764709516781, -2.56890732864206228, -3.53219400841847486,  -3.71712380369634010, -2.59105571407209379, -2.80844650397738915, NA, -2.71483874490519117,  -3.43507675405690449, -2.88672139898275582, -3.12917863615317282, -3.52024963397525115, NA,  -3.16114320734215237, -3.40319285724224407, -2.69915912261595947, -2.75775906421442851, NA,  -3.02398794921030811, -2.91039071147932926, -3.70337011922284098, -3.51244121526586506, -3.15089038904729080,  -3.33856234987377931, -2.97955281904261771, NA, -3.37550437369301992, -3.25251837099867291,  -1.92154726600055636, -2.39980632694250051, -4.38960895867238676, -2.50240410730166518, -3.52898074393246164,  -3.67822137814418770, -2.53119250010264629, -2.72200902798345723, NA, -2.62437320497934534,  -3.43006619553125036, -2.84471324825139149, -3.08675996792852958, -3.47027127124445700, NA,  -3.08388655258494015, -3.31300866661629279, -2.67435234797770827, -2.61353046042410364, NA,  -2.99358309341252982, -2.86271534587658261, -3.69918869469012002, -3.47136790446789956, -3.08279325622667200,  -3.26523905858687069, -2.97633420891757616, NA, -3.31918422341312969, -3.23368318886543005,  -1.94954582373512997, -2.33481264488108931, -4.37089943896356026, -2.47117430899659585, -3.53073262694306678,  -3.65550114122124414, -2.49793959290196099, -2.64339962244963633, NA, -2.53694710836941351,  -3.44332687742545218, -2.83255570803272505, -3.06995276119204297, -3.43937376081636792, NA,  -3.01979826918162075, -3.26242303314352000, -2.66426135244417450, -2.50258292629460932, NA,  -2.98283735354796331, -2.84492079780286788, -3.72657544542636732, -3.42992726219690391, -3.01418819162679519,  -3.21470715212544178, -3.05813081435426426, NA, -3.27572430377794843, -3.25823506282782427,  -1.97709523823484101, -2.23676947784213986, -4.35832913521326049, -2.44650043783334858, -3.53376010985202704,  -3.64027106478344242, -2.47401566227065839, -2.57190783947877710, NA, -2.44477231326941080,  -3.46075682457088707, -2.82500755768231837, -3.05368848629094192, -3.41675906890839709, NA,  -2.96992626184653874, -3.21980414113275648, -2.66703065899313785, -2.36118282714477346, NA,  -2.97030186194842605, -2.83670063750254320, -3.75755057388701141, -3.39270504751297386, -2.95830355324458161,  -3.17718990085036346, -3.15441818882042169, NA, -3.23022335184798903, -3.28644975462770983,  -2.01241098066148449, -2.14023976757803203, -4.35473263206484429, -2.43951547990018680, -3.55461451440618470,  -3.63412432484830905, -2.46493854714883032,
                 -2.52516632995255463, NA, -2.38159156148345152,  -3.50052324475237464, -2.83907056201371599, -3.04941116167898274, -3.39560839570071549, NA,  -2.93523127455661781, -3.18884550977603531, -2.68514360429871024, -2.26388960148931950, NA,  -2.96495164735166039, -2.84020319603744298, -3.78609392721184257, -3.37988882814854863, -2.90963842428328601,  -3.16482030978688833, -3.31653942142439639, NA, -3.18864380330905650, -3.31511262574499987,  -2.09653249095353633, -2.11868975030439577, -4.36244668909282218, -2.47658675063650913, -3.60600036187167250,  -3.67317346132945355, -2.50375483957716760, -2.51827456768227709, NA, -2.37125395264670091,  -3.56524849690815993, -2.89555937762643323, -3.07088300033074635, -3.42652558914433403, NA,  -2.95744875908385918, -3.21364589345319107, -2.73350193197733482, -2.22870211968077214, NA,  -3.00034042786180377, -2.86673295524372485, -3.82671274122943128, -3.41418156741228884, -2.91624199516329519,  -3.18022133288563458, -3.47800454598138931, NA, -3.18211192107936558, -3.37140078070238980,  -2.16064006989503854, -2.13893148846261871, -4.38171449424455339, -2.50018410893231557, -3.61730686039610916,  -3.69757524009537208, -2.57521490182475654, -2.56800712729483749, NA, -2.38696930468326318,  -3.58103749146216099, -2.92806097011427280, -3.09344483983433882, -3.44168963793725924, NA,  -3.00106867832674418, -3.22789856433906230, -2.75537954219934766, -2.23470715470693460, NA,  -3.01556443997356682, -2.90609116912206877, -3.86112227568666500, -3.42515031347737509, -2.94509616544532138,  -3.20884619249417202, -3.51561490231996254, NA, -3.18049824299128092, -3.39475288310145107,  -2.21127361137913869, -2.13669375633083369, -4.39693754965304517, -2.52750021637196598, -3.62083345746970986,  -3.71284376396749005, -2.61820791943115916, -2.62435429008286425, NA, -2.40962412117482261,  -3.60382259396256543, -2.97068117723557057, -3.11422992899084949, -3.46020066630746692, NA,  -3.05198554310113668, -3.25775786005927870, -2.77725188077193286, -2.21743484862914908, NA,  -3.03055973673009538, -2.94185689861301913, -3.87502851208113164, -3.44590543562548124, -2.97106802998713526,  -3.25194621423449437, -3.60666394162082904, NA, -3.16914507512035204, -3.41045972408608922,  -2.21104437594965786, -2.07321682365249416, -4.40429674911373770, -2.52365685353512115, -3.61648668535901940,  -3.71661483323082509, -2.62489241927286310, -2.59577702014942924, NA, -2.33551765719388182,  -3.60113104441099141, -2.96136415138076625, -3.11731641473798327, -3.46739941543516794, NA,  -3.03507082009532647, -3.25337084664518583, -2.76193437520975316, -2.12518978543842652, NA,  -3.02585181341575771, -2.93792652179037761, -3.88331677030183231, -3.44793406857892881, -2.96705815844390619,  -3.25881994910238459, -3.77001834521386758, NA, -3.15338176846501428, -3.40897344713480477,   3.38428450591441443, 3.49329138450862686, 3.31541989344903154, 3.32506175907372414, 3.32981486738909327,   3.41777870218208513, 3.45798784029866990, 3.60739537020527923, NA, 1.53580223050310760,   3.12981777542395223, 2.33964455131055482, 2.68978976489208632, 3.03081074939223072, NA,   1.85791081957536464, 1.50148701233793713, 4.20937073396248351, 2.86287159772124289, NA,   3.32917130017162499, 2.92181419452291591, 3.63871982037239672, 1.88686407851577687, 3.30811537871968175,   3.40762520910219413, 4.23554492036372743, NA, 1.76567178442498518, 3.07234567876425047,   3.38428465202539375, 3.49329151973523722, 3.31541992035653710, 3.32506173696538454, 3.32981490787112477,   3.41777873607290239, 3.45798794751357885, 3.60739554076381053, NA, 1.53580231985951876,   3.12981773472745317, 2.33964457239635149, 2.68978982183437632, 3.03081082301395899, NA,   1.85791082230522386, 1.50148705518501235, 4.20937078053273339, 2.86287166024548867, NA,   3.32917137529594775, 2.92181422364876520, 3.63871990474583562, 1.88686404799125751, 3.30811543433692146,   3.40762524745826356, 4.23554533259354571, NA, 1.76567184115029030, 3.07234574401467420,   3.39059651561635622, 3.47413165248257449, 3.33701879481891650, 3.34335479893275522, 3.35058097411302347,   3.41632259193693244, 3.46891590863721966, 3.58631560340612232, NA, 1.53831656154795482,   3.13419367608026622, 2.35091092222562192, 2.67861311312662043, 3.04744170130429204, NA,   1.85683835162842925, 1.50042314978064972, 4.22051563912737571, 2.84151661357245811, NA,   3.31592317085990285, 2.91038688646351051, 3.63414528004501935, 1.91465623509152394, 3.31212307878242207,   3.41064157693415471, 4.42141581162544561, NA, 1.73801396869887803, 3.08509661480639252,   3.35410423267464530, 3.43873786219377209, 3.33656898894068554, 3.33478538596342000, 3.32875547227316915,   3.39754207756333448, 3.44759571221219385, 3.55701688113071945, NA, 1.53799354954440504,   3.11032501949703866, 2.33778641888791849, 2.64721432039346016, 3.04655504075007011, NA,   1.82285779351604482, 1.47480361675189720, 4.20816874316947143, 2.81442960127439612, NA,   3.27437316249397448, 2.87528221288737296, 3.60349759908480882, 1.90422103287149791, 3.30243947948590399,   3.38476593419874350, 4.18771570441489249, NA, 1.69853771468067594, 3.06148033310412870,   3.31299335295035391, 3.38814543708922766, 3.31533139084201478, 3.29844521974574567, 3.27581986533058966,   3.37929580200068047, 3.44679875037197103, 3.50709548108717284, NA, 1.55146577083240311,   3.08754263833444931, 2.32625370034072354, 2.63038520943392573, 3.03126227241986390, NA,   1.84587545903890948, 1.52385647202772434, 4.17838006827023367, 2.77161185449480474, NA,   3.24571535659654353, 2.83161286639476195, 3.56027231564522451, 1.91028979008624877, 3.26544907284925312,   3.35683587132857175, 4.02931236363767376, NA, 1.71725631729955852, 3.04347405705280183,   3.27500593407026397, 3.35144530736137014, 3.30940940599662348, 3.29765574168441544, 3.26431481092611886,   3.39679808002719286, 3.40851371624417254, 3.49722799522920269, NA, 1.55814373169428011,   3.08455102934532155, 2.32678039026804573, 2.64052290463103922, 3.04565744186099252, NA,   1.89487348377189191, 1.59845425811396957, 4.17034368839378100, 2.77299768073352570, NA,   3.23984842555661157, 2.83518796701414111, 3.54015260196025006, 1.92073804920698188, 3.29335685907241960,   3.36339226919469914, 3.88939277176170606, NA, 1.76246976067447303, 3.04114147969680371,   3.21410445996912797, 3.26355483042977479, 3.29088375864949256, 3.26542973668967873, 3.22143029760567057,   3.38213762772464754, 3.35955610120085435, 3.44664743720429279, NA, 1.51783699863322985,   3.05265510882957036, 2.29112031184449805, 2.62272502812227293, 3.03021750218366082, NA,   1.91960549442555939, 1.64686809803665257, 4.13438134671336854, 2.71939313866205401, NA,   3.20441426634968618, 2.80469710117610838, 3.48828160012786448, 1.90249061038183376, 3.29361762541265612,   3.32997006962748232, 3.67266301746433621, NA, 1.77545103692340667, 3.01538810529719381,   3.15993619304924245, 3.15592225045611485,
                 3.26082256725558883, 3.22618379407665579, 3.16367353544076746,   3.34451781360392442, 3.32297687744293668, 3.38617817316191605, NA, 1.47502762846756963,   2.99880437352800611, 2.23706742257258284, 2.59904320712489056, 2.99591878137212442, NA,   1.91982576324701881, 1.66502739263355881, 4.08463524155032509, 2.64838737264805912, NA,   3.15038905982623252, 2.76144181558213386, 3.41754304792974128, 1.88000713482150483, 3.28282010784988776,   3.28568363291834720, 3.50817878783806947, NA, 1.75529792469415780, 2.97063520505243028,   3.08345147170720146, 2.97089728635664985, 3.20036815659052021, 3.14151820958511419, 3.08519634167417411,   3.27607437164262816, 3.26583997902695877, 3.28098785573690588, NA, 1.40154590253217215,   2.89953378422487607, 2.13363901458869210, 2.53777818826017043, 2.91132637846469189, NA,   1.88350229377797529, 1.62276160897903776, 3.99186679810340950, 2.52722033403137969, NA,   3.04352595362600997, 2.66929105048133408, 3.28886841389241047, 1.80530134186071201, 3.22794481893411556,   3.17540100726215968, 2.98033099561459824, NA, 1.70019643836420586, 2.91526877285167174,   3.03250508241410266, 2.91301573775303790, 3.16784497493626294, 3.09361493470322912, 3.01926531312677504,   3.23966926478798856, 3.16027635675341534, 3.22787469804070071, NA, 1.34653704450802580,   2.80873686430786185, 2.02865183395189463, 2.49255599539927353, 2.83733188399340586, NA,   1.84049382093752567, 1.59721348026124210, 3.90341560735465798, 2.46300404596113687, NA,   2.95976893499115778, 2.60475050268518959, 3.16239795285832148, 1.74038341218399584, 3.17925207826883671,   3.10860813393876212, 2.68613466742926077, NA, 1.66607017032423954, 2.85955863336193250,   2.97007561833539935, 2.88204265537832915, 3.16979931600949971, 3.07684552239960141, 2.96052947838855962,   3.23683190407947130, 3.11033520909439343, 3.18627707492776580, NA, 1.29364147337153823,   2.73894995002052388, 1.98063165292558918, 2.44743818876543040, 2.78967769874128546, NA,   1.84193672328009250, 1.58045064286036774, 3.84966067483600582, 2.42457511668639381, NA,   2.90717526075087163, 2.54677954928958350, 3.07779357768186879, 1.70906515201560372, 3.15244112852949421,   3.08795839657808502, 2.46444585994894982, NA, 1.64118531447159777, 2.80342396896937895,   2.93189364336228486, 2.89990272391342740, 3.18029811204807000, 3.07259049225345304, 2.95841817873791602,   3.25730040714089109, 3.03689578151919726, 3.18711010466250499, NA, 1.29257231416065377,   2.69575324926373838, 1.95426393590938519, 2.43988468977399808, 2.77631805104166807, NA,   1.86063052495125580, 1.61472027030918008, 3.81868506358532223, 2.42058372258863574, NA,   2.91297546358320236, 2.53737575878671517, 3.06486962174051136, 1.69247075090755028, 3.16235370200167276,   3.10267423644853446, 2.44258330819116365, NA, 1.63990024428204340, 2.79987972562754805,   2.88958302690174884, 2.88047275924339541, 3.18129407083288163, 3.04797063865343398, 2.93260094713237152,   3.27469797023271791, 2.93634187362697618, 3.15665498802667566, NA, 1.25958050926162746,   2.65951601462845044, 1.90865850492342970, 2.41909431390453911, 2.75825690787802813, NA,   1.85465674500468447, 1.64124338492682287, 3.78890832221930562, 2.38011684967000248, NA,   2.91748059316676134, 2.51334433499045851, 3.04904257537888235, 1.68029064176740439, 3.17221794152896130,   3.09188562489545360, 2.24479804726131738, NA, 1.62308362418253438, 2.79571909676185015,   2.85533453596496090, 2.87384897884225321, 3.18136484734161495, 3.05065559987131740, 2.94892917527869702,   3.29397880426392486, 2.89444064365319731, 3.14517120496487435, NA, 1.27429066860199258,   2.63897924516456506, 1.90516460662475784, 2.39303565878765934, 2.78286998570366695, NA,   1.87669694579359558, 1.67609797302672048, 3.76953052225121965, 2.39479745854954995, NA,   2.96407503952473794, 2.52350642974859163, 3.07631643453291836, 1.70244832111115829, 3.16817091325630473,   3.10863136475463886, 2.27560958326344309, NA, 1.65096997365502807, 2.78598785195624998,   2.82368759930651247, 2.81335369272542257, 3.15980633326006899, 3.02833028757571698, 2.95549617268625919,   3.29025181253866350, 2.84330757857942773, 3.11124976119375685, NA, 1.28859418121275837,   2.57803159462509690, 1.88940168612425086, 2.36641133367514955, 2.78824554704748806, NA,   1.86864945571917285, 1.68603212323454832, 3.73953657829692920, 2.39480619626600255, NA,   2.96057008743317995, 2.52073383350651659, 3.09008509293705513, 1.71371219193501600, 3.13872010094704734,   3.10617057024053089, 2.15665877023200769, NA, 1.65814076662671450, 2.76980226233531335,   2.78180084330804078, 2.77001780026960631, 3.13565152282549731, 2.99376501900428815, 2.98226355275114141,   3.28512480484149672, 2.79122842976157770, 3.07273209255445634, NA, 1.27007036523511729,   2.52775936207904728, 1.87123619191457591, 2.34068265628217986, 2.79469134275714959, NA,   1.86458338983200411, 1.70361256194974242, 3.69165037673914620, 2.37684960633761122, NA,   2.96394165484945971, 2.52766405380052106, 3.08529553780829957, 1.73043508972118221, 3.12120208352761530,   3.10072781128356745, 2.12715703400136613, NA, 1.67356499402670589, 2.75954811222472607,   2.75331932886993291, 2.78085891208831004, 3.12461575712455719, 2.97371826739666378, 3.01305071644548406,   3.27311350950471169, 2.73533073756431566, 3.05707419635225630, NA, 1.24863427077974443,   2.50624811737017694, 1.88633536275554636, 2.32082680847007294, 2.80014635991588623, NA,   1.85736702454173641, 1.71392609936972495, 3.65173935526542559, 2.36983973559633654, NA,   2.97140643173541807, 2.53422769590255914, 3.12088833740481286, 1.76848059365824373, 3.11776335482912970,   3.08897617245478440, 2.34823219366003455, NA, 1.69428227285565058, 2.75924516157499644,   2.72252576719793238, 2.77542432991582899, 3.10881150201523804, 2.96290704104086666, 3.03518130532169739,   3.25724662220232952, 2.68635584829418983, 3.05390690713147928, NA, 1.22004773778392939,   2.50250207962137861, 1.90269023767729517, 2.31162403228608504, 2.80926268019331493, NA,   1.85246070171989619, 1.72364071977774058, 3.61729181198554528, 2.36822232891591344, NA,   2.97789553863036716, 2.52919462168463083, 3.15486621038667181, 1.81147410535583431, 3.11157716916267368,   3.06472972046237579, 2.53428046334509105, NA, 1.72166663439513767, 2.75727938362354275,   2.70584862575088625, 2.80142927031549727, 3.09441109902300804, 2.96035020104495272, 3.05810992370245360,   3.24159794102697463, 2.63073680105843266, 3.07524711025700137, NA, 1.23241611685681129,   2.51634027499007074, 1.92647856741888956, 2.31124543068383970, 2.82875518547884930, NA,   1.84670329312969184, 1.74026204481859148, 3.58140108704330906, 2.37122164220896581, NA,   2.99304098146461106, 2.52263132025816361, 3.19624107130045232, 1.86441627538522758, 3.11408508242493154,   3.04600913402015561,
                 2.61543331495635512, NA, 1.75194089661908547, 2.76155978202903318,   2.66732452640924800, 2.80909395781929128, 3.09437670723553637, 2.95269258938802448, 3.08215102564522381,   3.23247785611045080, 2.54529883532201717, 3.07231174710572796, NA, 1.20346793531883867,   2.52423106809543807, 1.93958940940298019, 2.30548999321089498, 2.83247284843819225, NA,   1.83884024156652104, 1.74541095062801177, 3.55727678705541761, 2.36715639800634081, NA,   3.00809849364511717, 2.50649917445886672, 3.22948991055265777, 1.90948587589972840, 3.11444661693811353,   3.02190504184323716, 2.60423194741324737, NA, 1.77004045352050787, 2.76293835424721879,   2.64904775186029617, 2.83293950289090768, 3.09838762024964165, 2.94460134951308339, 3.09824187394589190,   3.23709615506851867, 2.47890514884281110, 3.07144519178697006, NA, 1.16801555089043996,   2.54726811927770491, 1.94717984916615383, 2.30392365950941125, 2.83582382645726172, NA,   1.83025174017158010, 1.74785508309069892, 3.53648083482526188, 2.37622003873028254, NA,   3.02786709628617468, 2.49881689636360482, 3.26074661335606297, 1.95091411019222605, 3.09972399280364064,   2.98272263234255952, 2.66276355952677379, NA, 1.78162850250610427, 2.77565448063829301,   2.63186400654809916, 2.81250999506359545, 3.10501223567301610, 2.93014614410089047, 3.10027273454786956,   3.23651323854949258, 2.41770688129993339, 3.03254139662855593, NA, 1.09445882575720943,   2.57216414460683573, 1.92701877520682041, 2.28258368599562234, 2.83974335716154735, NA,   1.80825832906463457, 1.73221528177030315, 3.51012674119330814, 2.34532714257515051, NA,   3.02455017865012588, 2.47634476438410545, 3.26906416021093893, 1.96508255673379417, 3.06823659662425907,   2.92289560727381748, 2.71587740929431698, NA, 1.76822981965014581, 2.78445891612430385,   2.62868476189238409, 2.78578000097737588, 3.11680649108005703, 2.91811481783849391, 3.10765454519597295,   3.24616401909165875, 2.38477311066974451, 3.00781604947249948, NA, 1.03633237071171114,   2.61364544554407496, 1.91212624757898686, 2.27465090407882053, 2.85301540155446887, NA,   1.80666224743718140, 1.72883783326324680, 3.50001308480945594, 2.32680507138868142, NA,   3.02554300418021160, 2.45671128748088163, 3.28024199939546834, 1.98691001987406834, 3.05950666405779392,   2.87658139836910376, 2.74020802512291217, NA, 1.75638194387598379, 2.79860176784650250,   2.63000981362705666, 2.73285348319860777, 3.10427155250806130, 2.90701236647746741, 3.07996513065665889,   3.22172493770737223, 2.35597785707366647, 2.95128415516907472, NA, 0.95893514220311216,   2.62351747128036905, 1.87840601161260867, 2.24965580494537187, 2.84269423058670911, NA,   1.77918912186152300, 1.69077808790868156, 3.48200649566834075, 2.27511713875063437, NA,   3.01941521592577677, 2.40386582606698385, 3.28591125165170217, 1.98087279283940942, 3.02222805141798023,   2.81540430785932294, 2.68868761588129512, NA, 1.71892864954728997, 2.78080164696896004,   2.64598973757994527, 2.71505404755236457, 3.10779323879152125, 2.89051092269041909, 3.07880516188259845,   3.21088969503860966, 2.29828251578266718, 2.89345371839396659, NA, 0.90056380786255152,   2.64684107685153380, 1.85753947931679275, 2.22354748616315945, 2.84223152411455615, NA,   1.76666821610934321, 1.66425066973218039, 3.47290019336636613, 2.21833288082738678, NA,   3.03675384800902881, 2.35778461214959334, 3.28502498643047058, 1.98577877738234920, 2.98924947881720415,   2.77672663703017708, 2.79850976318083688, NA, 1.70520851268482443, 2.78787974453624754,   2.66058920289606826, 2.69065263662475207, 3.10437589470468378, 2.86695587302689647, 3.07943484901596820,   3.18971764679359771, 2.22963340351229533, 2.83192282227679071, NA, 0.82201791360598520,   2.66840488277317789, 1.83675185009267139, 2.19414410583693931, 2.83460337668535844, NA,   1.75870121770354060, 1.63402626004982299, 3.45699290609098986, 2.14859567810819740, NA,   3.05615337780885410, 2.31331790214879396, 3.27663028871176376, 1.98493482786755582, 2.94595557744343139,   2.73937951630382193, 2.93883104011128715, NA, 1.68099108708307110, 2.79447559951297819,   2.66723677473551257, 2.65394685139578357, 3.10243093102756839, 2.84459094614104524, 3.07526352354763244,   3.16619611344540131, 2.13850670294050715, 2.76109328113531483, NA, 0.72788458200346429,   2.68965287373041839, 1.81696976733669047, 2.16439937121105430, 2.83054827713240842, NA,   1.75172158653553756, 1.60320206752173156, 3.43713562673051776, 2.06763022015567888, NA,   3.07114165138138295, 2.26857303801059906, 3.26692103663630462, 1.98145958820610035, 2.89795147001080666,   2.70351465027747828, 3.08897229420938757, NA, 1.65454469349153088, 2.80488032956033528,   2.67572659980837413, 2.62013762597885869, 3.09938926663129877, 2.81402976600491916, 3.07898092398237466,   3.14183461599863900, 2.08995778746623229, 2.71624098098854061, NA, 0.68019923029375284,   2.70256395683236672, 1.80497523341666644, 2.13333868710818519, 2.82203375582320737, NA,   1.74592817846970161, 1.57875253073140809, 3.42463663325635004, 2.03708291176299738, NA,   3.07465915272123302, 2.22644636218504877, 3.25949026864919178, 1.97811050770033492, 2.85733619513761106,   2.66473026269789504, 2.97774034714354530, NA, 1.62835931161386527, 2.81128588984196304,   2.67468857618545686, 2.57749050838363081, 3.08204293210148661, 2.77057710350268582, 3.07875963241513428,   3.10908343077227256, 2.04122477716417894, 2.64932421800840201, NA, 0.60723938524392762,   2.70288541330568499, 1.79205298095410237, 2.08924693767011593, 2.80434611076802476, NA,   1.72786241292009679, 1.54933902286083081, 3.41172139032896915, 1.98035530720191755, NA,   3.07259768580464199, 2.17036636063809807, 3.24726241296439566, 1.97360554627538054, 2.80249698383043011,   2.61661001885913569, 2.92651637221573946, NA, 1.59759722736154663, 2.81238123386192518,   2.68130720967258895, 2.55363705117649875, 3.07243429297941528, 2.74993643576648283, 3.10317323696914915,   3.10262293935628053, 2.00328376140392050, 2.60274728460283544, NA, 0.56654941756678245,   2.72560973714665478, 1.79492097494580327, 2.06811227492053229, 2.80134068165049399, NA,   1.71706579182568486, 1.54526957260962372, 3.41027585007011513, 1.93037038917097115, NA,   3.09074005431767684, 2.14340656434027199, 3.26300838980107510, 1.98528947183050608, 2.77384222097958588,   2.59009013617943973, 2.95423541021270664, NA, 1.58141664045158237, 2.82273704396095759,   2.70140079587649939, 2.55295263106545889, 3.06593408505621534, 2.73922017687854513, 3.13006713716753415,   3.10173902717323147, 1.97908333617317367, 2.55697934463394994, NA, 0.52641540509931406,   2.75496813844578092, 1.80870248971482739, 2.05578901833305583, 2.80411880374733347, NA,   1.71242371716012753, 1.55158896852932648, 3.41545238265853079, 1.89325701928152323, NA,
                 3.11586754305647373, 2.12483392621814460, 3.28522566405490268, 1.99977479975070604, 2.74454756551396439,   2.57045040170346084, 3.04998567327768422, NA, 1.56758097208094194, 2.84365038011685112,   2.72743964697854357, 2.54652435041199565, 3.06303329328626717, 2.73368468778683660, 3.15966241400589842,   3.10439452701103580, 1.96135209422071677, 2.51064005571577109, NA, 0.48368041948856810,   2.78977082194618298, 1.82903827362590610, 2.04607726374186960, 2.81096662307886813, NA,   1.71708505753039509, 1.56197714668793730, 3.42873085734429139, 1.84299109577545184, NA,   3.14502312776348925, 2.10970414898301639, 3.30535543501377482, 2.01870833178050368, 2.71887738584041072,   2.55656476744775762, 3.16566427169455755, NA, 1.55198379255416152, 2.86776801329840936,   2.76446624291987852, 2.55705910329002206, 3.06627868888196664, 2.73935226399776388, 3.20310130743150356,   3.11352496022408198, 1.95260028331939250, 2.46874014903334649, NA, 0.45280665834192346,   2.84135980296371393, 1.86505795699976007, 2.04558291385422031, 2.81767723153923422, NA,   1.73229135386297317, 1.57856400212106673, 3.45166743464418513, 1.81323210036691562, NA,   3.18512149630555275, 2.10142300788213010, 3.32307162559685576, 2.05297747150438292, 2.69232832812917522,   2.55543801847365559, 3.34382136355652815, NA, 1.53832148899669008, 2.89342184664168345,   2.81296486879899010, 2.57174123409814737, 3.07771056830084699, 2.74970971679913534, 3.25295783568551666,   3.12504319800787878, 1.96277438403389870, 2.43734669645150870, NA, 0.43923936184810353,   2.90707139844511131, 1.90676753243811770, 2.05315615901452997, 2.82296720491169051, NA,   1.75779115515772522, 1.59883070497160884, 3.48578526020612234, 1.79280908134751304, NA,   3.22028788707002800, 2.10074637914851925, 3.34214039263050955, 2.08869986040461209, 2.66779566402184631,   2.56452485989428203, 3.49168207170800615, NA, 1.52182540347794881, 2.92224725472253288,   2.83158525721281551, 2.57300011117667005, 3.09106827227631387, 2.75055352716261492, 3.25899704621504194,   3.11991258863791021, 1.99358203629353414, 2.43561706135580058, NA, 0.42652273015054526,   2.92658714101776729, 1.92109873526970465, 2.05277255275986414, 2.81536961539597863, NA,   1.76994694505496475, 1.59925798241961670, 3.49078116714957565, 1.78915413283818436, NA,   3.22867763893313375, 2.10461949021485051, 3.35026372429928099, 2.09159395760507927, 2.65946388585479188,   2.58284631021161859, 3.52022770429717147, NA, 1.51094677114282394, 2.92429289653347446,   2.85443453940236447, 2.57379631444572965, 3.09757545868059081, 2.74971650906226106, 3.26441011919388968,   3.11729728098681180, 2.00102294833344363, 2.43243683015303436, NA, 0.41172904444413705,   2.94031001499163835, 1.93933261446134031, 2.04698032561426846, 2.80573483755191067, NA,   1.78582128659871442, 1.60704735369489615, 3.50075816833118569, 1.77694402166628418, NA,   3.23702272209025965, 2.10877284455201330, 3.34416405897206470, 2.09628475496087852, 2.64769515326449723,   2.59899971096258797, 3.60756225616440140, NA, 1.49196512187782870, 2.92569160867031108,   2.85685913583148254, 2.56956855331533962, 3.09234569486952360, 2.74469265537113882, 3.26248961137057547,   3.10975913776660917, 1.99112976997029101, 2.39296113867813354, NA, 0.36330286285574342,   2.93340534821547960, 1.93991484480701493, 2.03395208013037543, 2.79476450908630314, NA,   1.77494924660358544, 1.59475062729534089, 3.49287868380774658, 1.73703381283983815, NA,   3.24148412820650123, 2.09880435409558608, 3.32756063393821000, 2.08643552167401847, 2.62802380353169562,   2.59111145172208834, 3.76662928726379409, NA, 1.47205929579411166, 2.91971709214746600,   1.02157997562903935, 1.02695918894480820, 0.99237943138563467, 1.03441437697024807, 1.04014412894750752,   0.98892307694075254, 0.93132648555949105, 1.14289609186235541, NA, 1.20388177509659267,   0.86683280154570974, 1.06478769770346693, 1.05579909182186671, 1.03035619762878716, NA,   1.15936841711869087, 1.09661897652593754, 1.04055639705838332, 1.04875708470868512, NA,   0.88351510323036953, 0.89430839797028661, 0.96083001591464390, 1.09041076518946589, 1.04948112739644039,   0.95704311060519764, 0.54565689855972865, NA, 0.81509254996277203, 1.03861726094612239,   1.02157996791879180, 1.02695919639849720, 0.99237942346889785, 1.03441437696326233, 1.04014411968933751,   0.98892307551516934, 0.93132648557351305, 1.14289610036223710, NA, 1.20388178191101214,   0.86683280318390887, 1.06478769533723550, 1.05579909179695175, 1.03035619145239199, NA,   1.15936842055576883, 1.09661897728534541, 1.04055639738714922, 1.04875709248993543, NA,   0.88351509346062285, 0.89430839488940661, 0.96083000081395153, 1.09041076501263157, 1.04948112478267097,   0.95704310588728725, 0.54565672996962644, NA, 0.81509254777998641, 1.03861725051874609,   0.98746976777323769, 0.99536725694395434, 0.96496982384382335, 0.99048774196633727, 0.99553695986541491,   0.96278936371329427, 0.90115277955836981, 1.12809160521486218, NA, 1.13964908088969086,   0.82699045804054250, 1.03166039302830370, 1.04359367471831810, 0.97782478853041999, NA,   1.11593802846855761, 1.05780643502958505, 0.99339032733935551, 1.00869866508961348, NA,   0.87719796504098024, 0.87236668587602439, 0.95512186054455317, 1.02639171328626277, 1.00667808436278028,   0.94542145515760367, 0.43456946252977102, NA, 0.78525453776143206, 0.98267255065800518,   1.05081075886133357, 1.03711111110628207, 1.00949470548178288, 1.02950273358442024, 1.05315518132344454,   1.00180896396021679, 0.94651380318637723, 1.13090314639869738, NA, 1.14376429374644495,   0.87173040348830089, 1.06687626744466302, 1.09427138023074888, 0.95329015574788412, NA,   1.15898813737051243, 1.09474490338445274, 1.03091973650143220, 1.08910707820679709, NA,   0.92338433037668122, 0.94113431665425384, 0.99069517337819935, 1.06102527716716866, 1.01479878733352757,   0.97857636466632358, 0.56582307616080219, NA, 0.86042569586765349, 1.03356749781165425,   1.10243021664626206, 1.00477876931289622, 1.04160438332980920, 1.07582250040370675, 1.14051957004078242,   1.03422731153525183, 0.97862830715617943, 1.17869681557718042, NA, 1.15564972117610609,   0.91959976231344442, 1.09983422520258922, 1.12232074067812948, 0.92846279888123340, NA,   1.18179016227811329, 1.13057884719828561, 1.06312106330962042, 1.18804132446097332, NA,   0.93633203037237012, 1.02780919444031404, 0.99404649111354437, 1.04585821686181024, 1.01935799635897539,   0.99114054235215510, 0.65185455137280535, NA, 0.94637848105616418, 1.07728691198858595,   1.09967168876605381, 0.97684647218468479, 1.03711483474906530, 1.03969235850107466, 1.15567586302343939,   1.01893972332894611, 0.96778203247392303, 1.15966779289088939, NA, 1.13256294570141369,   0.91221681448473313, 1.08957689525922263, 1.09127653004479397,
                 0.89989698504245985, NA,   1.15554484122002998, 1.12374754140514632, 1.04337543821929457, 1.14566556899140992, NA,   0.94913631010794675, 1.00775952589465234, 0.96352351175621476, 1.01102007055720322, 0.97799619088965617,   0.95122320977920449, 0.71203366517521993, NA, 0.93521111840501903, 1.09654205149520401,   1.10839170568014600, 0.97487646894831914, 1.04916500072490870, 1.00946592145612724, 1.19314662364691482,   1.00852218558355378, 0.96397683924480770, 1.16019313820110481, NA, 1.11185941832754520,   0.92851825169764490, 1.10757899752584588, 1.06448772288981752, 0.87749774925823742, NA,   1.14400176023587563, 1.13888251139766483, 1.03730385128311808, 1.13130592731816293, NA,   0.97429669379725969, 1.00630884041474933, 0.94488245833556450, 0.99363945610892956, 0.93937928402401094,   0.93692902889806084, 0.80548298858646372, NA, 0.93416563260585039, 1.14102450931010169,   1.12454460696539504, 0.99811775375823641, 1.07230510455486883, 1.00989635179341253, 1.25984428224727529,   1.01336871373128523, 0.97311947404076093, 1.16442405978855135, NA, 1.11478402490057849,   0.93138105031671603, 1.14709317765921748, 1.05203416177424147, 0.87705456155199235, NA,   1.14693568228041265, 1.13556455809768386, 1.03474576752862846, 1.14335889312307337, NA,   1.00552628268076383, 1.01656871689795247, 0.96101888941036995, 1.02327592161467607, 0.93984701617285360,   0.95906909935553852, 0.86273821396203720, NA, 0.96230954012228875, 1.18733162099339440,   1.16491618934416219, 1.12965971974918222, 1.16667466526930941, 1.07051037324848819, 1.34722485778799150,   1.06453572322903200, 1.03414685111750337, 1.20316007733648433, NA, 1.17372880177106653,   1.03877908130831464, 1.25844051534274892, 1.10965076569563048, 0.96322356586350810, NA,   1.22699464887922627, 1.22141910085119254, 1.09826489632496482, 1.24337515979226687, NA,   1.08853839724728374, 1.10856043073548305, 1.07247078253649653, 1.14942976490405169, 1.02118247244599636,   1.07276246671406916, 1.11112355919924743, NA, 1.05528109053120289, 1.25681993947918924,   1.18284710672689997, 1.19936374752610453, 1.22780190795331579, 1.09756127277549687, 1.39149035249323960,   1.06815937690353691, 1.07103457098518229, 1.22821128480915354, NA, 1.21768073159665735,   1.10679085579145520, 1.35425303121750851, 1.15070654298634500, 1.04009338730715606, NA,   1.28938674408852672, 1.27144967314183943, 1.13119401997204250, 1.27957108250994978, NA,   1.16538559264786779, 1.16494614135834462, 1.19266804448095565, 1.22888383550503444, 1.07014727057827308,   1.13602577841409080, 1.24360141335605268, NA, 1.12538834990896897, 1.33022708189552730,   1.23381668017239754, 1.27310775112413932, 1.29063809870534785, 1.13394071915797157, 1.48627864668586307,   1.07975691752036473, 1.12300521061940084, 1.25995711848415048, NA, 1.27658230726381428,   1.17472474451589481, 1.43564785557451446, 1.22066593787105115, 1.12777718562298856, NA,   1.33614078817237614, 1.34412173633027443, 1.17343774993396521, 1.33698995537012499, NA,   1.23170389333105113, 1.22527058624017271, 1.31944501864018271, 1.32750874300581434, 1.13250113024809718,   1.17861217991974776, 1.34849804017247177, NA, 1.21904680949770028, 1.44607799762508415,   1.28415441234615080, 1.28142211848410237, 1.30343301415812385, 1.15914957303672783, 1.51823167486769872,   1.07606375672099253, 1.12847323007441802, 1.26734002925281075, NA, 1.31767058606224108,   1.21567066566204618, 1.47479352878433789, 1.23405863939907579, 1.18325189565971445, NA,   1.36046998596797497, 1.33308803392523290, 1.18994148135370192, 1.36733099135969693, NA,   1.24482973680551678, 1.24551511946666560, 1.34754939643118421, 1.39772664545924763, 1.16669514523186746,   1.20613907224448891, 1.34356073758582895, NA, 1.29819911324541115, 1.50598019143045692,   1.31978373657870285, 1.28268414938995501, 1.30513831932319979, 1.19120289419954806, 1.54554383066216472,   1.08955456622991487, 1.13210911154442773, 1.28383506243760781, NA, 1.33239942125834232,   1.21399872011855425, 1.48248207053978232, 1.24822571616462330, 1.20838495216512309, NA,   1.31220055532632918, 1.32284128090395292, 1.22482019645921625, 1.36598571177800165, NA,   1.23101948223589797, 1.25375572022108317, 1.36382426064484696, 1.40877873557447431, 1.19833427679792304,   1.20612994416446595, 1.42367143679236263, NA, 1.32357207150145229, 1.48513130491394385,   1.35405571452154794, 1.26548854641741482, 1.29866892505388787, 1.16921206454775728, 1.55607439669670145,   1.07708921826232373, 1.13306720768819225, 1.28675403729323090, NA, 1.34318177883010748,   1.17134020998061295, 1.46493371965612429, 1.25316950759521650, 1.19110365941319740, NA,   1.26192317924943254, 1.28244301872469513, 1.22601847266303210, 1.35118661302758913, NA,   1.19959016583129352, 1.23373426718338774, 1.36900939670971300, 1.39938581920609328, 1.21289663984014529,   1.19272160658977788, 1.39503549231814450, NA, 1.30646037075228572, 1.47305075122895390,   1.36627983027390987, 1.31976288749365778, 1.33771180292745728, 1.17459722973483705, 1.56682302095262771,   1.09876632408387009, 1.14471812662235406, 1.30506326902797531, NA, 1.36026863432917011,   1.19627072827665293, 1.44620959501797697, 1.25082007747554114, 1.20205707933436567, NA,   1.25433160529641419, 1.27420973099953883, 1.25323773339870459, 1.34889753723565686, NA,   1.22296441466060735, 1.21319493434073178, 1.39583055042845960, 1.40090228010429541, 1.27308055203431403,   1.20712777707952545, 1.44460157833351377, NA, 1.31409484464431969, 1.46871059119070857,   1.36276315964176908, 1.34074627020353976, 1.34641104857317284, 1.17196279007907456, 1.55112848776142287,   1.11236161775116527, 1.14889881112569259, 1.30258741748497986, NA, 1.36168333931245988,   1.17135846465563143, 1.43171458710154798, 1.20674446386906675, 1.18196683656772605, NA,   1.22319146996202677, 1.25249722507660355, 1.25736688201066982, 1.31987353446976918, NA,   1.22035546938530781, 1.18523335385632822, 1.43216221875175131, 1.38048284415274081, 1.28810925651496566,   1.22444589793187530, 1.43280817425903728, NA, 1.32119316569104051, 1.41753865145951985,   1.30526487618406795, 1.28063880921711548, 1.27023065136029101, 1.12485718183504879, 1.47962865975312985,   1.08457487998988533, 1.11374916259548318, 1.23649622313951779, NA, 1.31252020542333825,   1.08900550670951946, 1.32826990794547917, 1.13298152066198843, 1.14454561040380010, NA,   1.17015101570176228, 1.19008287859090767, 1.20017134576869378, 1.24802445308416110, NA,   1.18599238893947789, 1.13232639744564834, 1.38357117564105669, 1.30738600586301912, 1.23684635580621149,   1.23368914379756811, 1.28403833005673262, NA, 1.26309256270873860, 1.30177166657258581,   1.26225055302570066, 1.23403197676212328, 1.19851181168708099, 1.10346867690578510, 1.42174898481795764,   1.07060046274867804, 1.08776670173244172,
                 1.19545315681373188, NA, 1.28598582953760632,   1.05693714919658932, 1.26150036089095119, 1.09964148808530604, 1.13609993308524637, NA,   1.13798280929289608, 1.14193568001483481, 1.16151267415422721, 1.21373982985752193, NA,   1.16957865498381697, 1.09616674916017409, 1.36392012572226062, 1.25738570962503737, 1.21060151153635087,   1.25202161395122147, 1.15842378904435406, NA, 1.24741444516623612, 1.22432242203792585,   1.21424836293601612, 1.18642443873215520, 1.13942124875161821, 1.10135711360761301, 1.35161611465496234,   1.03472224458666995, 1.05211877534942122, 1.16651050974965953, NA, 1.28452936283067465,   1.02563795384738765, 1.21271360781379145, 1.06395819872652209, 1.11512056298506845, NA,   1.11555537861328191, 1.12896035556586938, 1.13393225831590372, 1.18578748983028337, NA,   1.13367712826605271, 1.05125030416084275, 1.31080795838450914, 1.23653610799550306, 1.19190243458135758,   1.24245124574048171, 1.10766058386326738, NA, 1.21926351209357775, 1.14064169943282478,   1.17840892682684895, 1.13546720504694276, 1.08749617135749066, 1.09976432090311693, 1.29197279583410407,   1.00171099659056329, 1.01973852917522700, 1.13748526740151878, NA, 1.28044912466745919,   0.99190686773302583, 1.16643711785338500, 1.03218955341273344, 1.09696507077555094, NA,   1.09795881301172815, 1.11403567562903372, 1.11268656710804326, 1.16365869858103066, NA,   1.09826066099492836, 1.00844504905195143, 1.26245677187235628, 1.21565182444640918, 1.18440002880888318,   1.24181143869581256, 1.09641497059715931, NA, 1.19594410065628720, 1.06516920614565569,   1.13066764711875245, 1.08910132210901844, 1.02304210538383455, 1.09361169211977227, 1.22142482338719205,   0.98170011031650939, 0.99687240144960976, 1.11562028723571327, NA, 1.26929663618853761,   0.96693537945487207, 1.11233380545126703, 1.00303395170213983, 1.07465834263153259, NA,   1.07842994151108917, 1.09088074491677989, 1.09699549755343795, 1.13800284925232487, NA,   1.06426693432352959, 0.97152589878853002, 1.20191963693757198, 1.18140852049749823, 1.16024675101857166,   1.22840614666603876, 1.06680858288300584, NA, 1.14761369375778943, 1.01311126645843319,   1.08299944269810444, 1.03758507016582602, 0.96215389198794654, 1.09179244342129533, 1.16730430101794935,   0.96250410690288257, 0.97500468570877297, 1.09714563499694306, NA, 1.26082524057087775,   0.94229814608522833, 1.06643286154998251, 0.97992469159417650, 1.05681693447167735, NA,   1.06027085158659617, 1.07330073071446175, 1.08424485521081371, 1.11283909971126160, NA,   1.03639259524793670, 0.93769948327353103, 1.15310777054398605, 1.15029906915401403, 1.14717958880561621,   1.22384464477293076, 1.03715946326964081, NA, 1.10781672003326892, 0.96579343313709232,   1.08385523584465959, 1.09155676470441665, 0.96865070364362671, 1.11463374463091958, 1.13370030736432703,   0.99612650194840047, 0.99034755421797471, 1.09472191879512915, NA, 1.26411891613528460,   0.94368339145127367, 1.06426422393602960, 1.02358661990384414, 1.04721131013023738, NA,   1.04763485805020773, 1.06926195538659785, 1.08892527274445117, 1.13054052331816668, NA,   1.01383701447296759, 0.93975024254998862, 1.13647472090060253, 1.14157792502197952, 1.13722951744226863,   1.25124909498053816, 1.02859529754425805, NA, 1.10670744948896105, 0.98393078403084810,   1.08662536308086577, 1.15069852538132200, 0.99816376543296337, 1.14451149850403100, 1.13723078131774180,   1.05148459872550237, 1.00600161758893458, 1.10702732174469132, NA, 1.27680537215485157,   0.96702774654683743, 1.08049932895807421, 1.08611135199983777, 1.05964148427239202, NA,   1.05898205328518724, 1.08665583032792767, 1.10435121549830351, 1.15922310317780464, NA,   0.99913989780752444, 0.96278267082846791, 1.12868277537381667, 1.15528145657140335, 1.15487871461684488,   1.29842467366757819, 1.05645086654945564, NA, 1.13715765907742394, 1.02901697179325091,   1.05454994917216016, 1.13655919178008502, 0.99164551257352274, 1.14522718109663946, 1.06471060468179424,   1.06759079039825000, 0.99830336467238134, 1.08708964141869213, NA, 1.24819080088752421,   0.95281855897866607, 1.03296027712423477, 1.08068025626245667, 1.04263095908171599, NA,   1.01145514289245519, 1.08072892544131460, 1.06388434436754142, 1.13696111275238398, NA,   0.95484175021080331, 0.93758696085512838, 1.12237958617338673, 1.14083114319758705, 1.14100508334221118,   1.31739693679040792, 0.98850123154040281, NA, 1.09795513297445235, 1.01288163318511959,   0.98991930467406397, 1.07969929888254290, 0.98387036377212711, 1.11471449115813259, 0.98339591211350141,   1.07351003330235906, 0.96267752697718989, 1.04879605492589278, NA, 1.19455756402640145,   0.90559794751344524, 0.96624594051296842, 1.02465684066295681, 1.00245685349824010, NA,   0.93252956502912920, 1.06879784249119436, 0.99176667213143999, 1.08962318020313975, NA,   0.89598051166832859, 0.88875815677015646, 1.09404585162925261, 1.08920100914208740, 1.08075423807495952,   1.34053252788204946, 0.89071449004745074, NA, 1.03089648969600045, 0.99190129345844802,   0.91432459650636688, 0.99519351649172050, 0.95504986879305354, 1.05593944682298968, 0.89871060237682998,   1.07004520877609433, 0.89794825140591450, 0.99421242582850833, NA, 1.11827108919578300,   0.84089186789742676, 0.87379254875831636, 0.95214194942408592, 0.93875599018721301, NA,   0.83360297036573550, 1.03533134739152710, 0.91649870739403161, 1.00894753986963637, NA,   0.83652905069669492, 0.82269955126971139, 1.05363711896422219, 1.03214748641817056, 1.01257524727227488,   1.34632350908196274, 0.77665243141727569, NA, 0.95295481234529134, 0.93539431956239938,   0.95402428065262757, 0.95309691071159697, 0.99272473324730715, 1.05765574667945605, 0.89026229219215247,   1.06623461000578512, 0.90418038790508193, 0.98060433799433955, NA, 1.10588586412867995,   0.86112019375484927, 0.88207288581527143, 0.97267310543146135, 0.93837484053211750, NA,   0.84769077442484786, 1.00219928195162322, 0.91406749104751595, 1.01280363551377128, NA,   0.84445625606530039, 0.85734961184614333, 1.03820086827295288, 0.99985072006175479, 0.97326799245696671,   1.36027360759270710, 0.83068842217921313, NA, 0.91216754310781201, 0.93187273633387635,   0.96129011110448292, 0.86844828891126435, 0.99604225568550919, 1.02735197458202099, 0.86404441184948855,   1.03941416072856030, 0.89833025922645615, 0.94346744947710215, NA, 1.05833323573670945,   0.84516062636842793, 0.87244601014129985, 0.95321748784209070, 0.90692779433535997, NA,   0.81813038785892611, 0.95420969144241485, 0.89581435564177780, 0.96645763497858006, NA,   0.82070855466490822, 0.85689133023076913, 1.01232408310566924, 0.94872476707302811, 0.90992808300749828,   1.35053693338283098, 0.83932706525531353, NA, 0.85852256388550885, 0.90100489825145247,   0.95803377794072997, 0.80756358877541989,
                 0.99989455370633284, 0.99689544090438131, 0.84417240672076888,   1.01945624390909950, 0.89204706799954236, 0.91439238225137265, NA, 1.03139630012911576,   0.83395490048741894, 0.86448043237284378, 0.93985408255224312, 0.87301924252275231, NA,   0.78709863793010537, 0.92022939610247056, 0.89212837218958896, 0.91252084579261572, NA,   0.79843200142022352, 0.86111024333482311, 0.99892641679291849, 0.91155127518838830, 0.86455599239286340,   1.34237126906444226, 0.81990716507382166, NA, 0.80970642463958797, 0.86586403357239095,   0.95001941291415126, 0.75214584062941481, 1.00640616626057255, 0.96662227098426778, 0.82371159496436108,   1.00086374729528771, 0.87962880911858810, 0.88314930405128234, NA, 0.99517972915121411,   0.82103799428938795, 0.85400048947032359, 0.92187809292395462, 0.84068479046706113, NA,   0.74912883391080021, 0.88855406547329119, 0.88488869376317159, 0.85173543268933738, NA,   0.77687630710417044, 0.86506535008176011, 0.99133631057778038, 0.86659902580818526, 0.81640569525755025,   1.33505494407860925, 0.77221560173704629, NA, 0.75940709665625938, 0.83435851715157427,   0.93171312606457490, 0.67780059010861549, 1.01401322009500294, 0.93035538957661246, 0.80024829368387151,   0.98253124247364820, 0.85893839923652116, 0.84832102631454709, NA, 0.94478580082662478,   0.80274773949276546, 0.83527631005634706, 0.89501518526779023, 0.80683496496458884, NA,   0.69934183448338993, 0.85353285461687545, 0.87099889137274233, 0.77353698369167989, NA,   0.75211551139201260, 0.86403814525235889, 0.98797041096217364, 0.81338658104430006, 0.76445218085157274,   1.32547302185867721, 0.71284628092346758, NA, 0.70530888699857486, 0.79731859000350092,   0.90020468575410373, 0.57099217653736511, 1.01686718688478317, 0.88810449156764570, 0.77556083385644881,   0.95109325190590976, 0.83265462156392867, 0.79088355140813582, NA, 0.87927356887598651,   0.78886432580810228, 0.81229645995677635, 0.86209025220920965, 0.74295195082841148, NA,   0.63067276790008908, 0.80916513186338790, 0.83912153166085035, 0.70480257278605352, NA,   0.71057212521171043, 0.84665114657692597, 0.96566301091723417, 0.76798959597651251, 0.68964438937915440,   1.31926663502945973, 0.61878782961556911, NA, 0.63555509834773083, 0.73454157728514491,   0.89363416821779007, 0.51242835254358532, 1.02043163434857020, 0.86922764701808142, 0.76938995677577859,   0.95350293891739868, 0.81668388489761690, 0.74022075802218967, NA, 0.82587375329956902,   0.77769627433435629, 0.81616622517747417, 0.83482005347881993, 0.72102784572189627, NA,   0.58619963175038681, 0.80035757201194213, 0.81167572116735864, 0.66800569997801096, NA,   0.70338009723867545, 0.82726271760539694, 0.94911443438070398, 0.75882663857949129, 0.65050776263701660,   1.31852337207319015, 0.55894604371104417, NA, 0.59940302338024953, 0.68658979756164051,   0.93093131060945566, 0.50674460949901412, 1.04878298543835635, 0.87915396785150779, 0.78026365226894245,   0.97740956354153363, 0.83046867930479329, 0.72840314643289017, NA, 0.79476650013117456,   0.76496493579270397, 0.81714095369286299, 0.84543184599904353, 0.71879906067822730, NA,   0.58451345399932486, 0.80833714940570200, 0.78407904324359912, 0.66612535656663652, NA,   0.70241738416635779, 0.84531614868618177, 0.94186040578474228, 0.75433241097794368, 0.65054784648763775,   1.34057362843139161, 0.55083673073934991, NA, 0.57703189724884574, 0.65801290722384220,   0.93393033212197762, 0.47648053222752312, 1.06298585281493185, 0.87376519801137131, 0.76438017922238088,   0.97878972125533659, 0.83487652813422553, 0.70496259173547537, NA, 0.74480382781746557,   0.75024792465107548, 0.81203565760675889, 0.84406180640617423, 0.70352707312563567, NA,   0.56022605748052157, 0.80852770815302344, 0.73257729016070405, 0.64254186828165238, NA,   0.68901613460288025, 0.83568039025214269, 0.91264810704412125, 0.73860668616679503, 0.61565696595556352,   1.36238467528031060, 0.49941643545039083, NA, 0.52077993599130556, 0.60684366500198039,   0.90539009562220829, 0.40265050891239351, 1.05671794331666735, 0.85125701041496871, 0.72544876405726566,   0.96952946532700213, 0.82545574610545580, 0.65690766924308630, NA, 0.66489371448116086,   0.70247060342842416, 0.77079820776519237, 0.82439102429571509, 0.67667772770234269, NA,   0.48863429077345499, 0.77694369984487632, 0.65610752785514426, 0.57017176586609075, NA,   0.66205968109056101, 0.80364509374604642, 0.87117116142290274, 0.68652878140811824, 0.56284816594987475,   1.35327023024426962, 0.40223612330288849, NA, 0.44706347769532362, 0.53176161881271466 ))
})

test_that("Boot gives previously computed values with timevarying covariates and fixed effects",{
  fit <- ddhazard(Surv(tstart, tstop, death == 2) ~ age +
                    log(albumin) + ddFixed(log(protime)) + log(bili), pbc2,
                  id = pbc2$id, by = 100, max_T = 3600,
                  Q_0 = diag(rep(10000, 4)), Q = diag(rep(0.001, 4)),
                  control = list(save_risk_set = T, save_data = T, eps = .1,
                                 fixed_terms_method = "M_step"))

  set.seed(747)
  suppressWarnings(tmp <- ddhazard_boot(fit, do_sample_weights = T, do_stratify_with_event = F, R = 30))

  # plot(fit)
  # get_expect_equal(tmp, file = "tmp.txt")

  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c( 1.00000088778563767, 1.00000174009561826, 1.02304753103683543, 1.01944404102942188, 1.00899290285153165, 1.04701662715267951,  1.06940710837511466, 1.05600873376315985, 1.01812776489103229, 0.97325913801785024, 0.95293336077703195, 0.95773174865017241,  0.96620647639292967, 0.99903704458071629, 1.00520639541201673, 1.03268980353765416, 1.02729242848519764, 1.02732896385795858,  1.01122609703073230, 1.01281913995027395, 1.00011655254244580, 1.00060281713075727, 1.00611081577129924, 0.99649311547138320,  0.99060170200508546, 0.99156157825457181, 1.00447654495826333, 1.00798744657310357, 1.00284571732886429, 1.01458649529204337,  1.02539862726250863, 1.04270331819876461, 1.06262671726664992, 1.07284909478848500, 1.06928074682467189, 1.06656389011133812,  1.05314672682059474, 0.05731815476167940, 0.05731814036688184, 0.07110830574865339, 0.06342398364352142, 0.05966557643891277,  0.05120077981569100, 0.05113808267629658, 0.05342563250785658, 0.05491871745693803, 0.05053739731791734, 0.05053960468479139,  0.04591178859456737, 0.04541290878746800, 0.04457130583886826, 0.04014239820973144, 0.04100879872716157, 0.04366389490036358,  0.04622534461029087, 0.04525593460221539, 0.04273532812784381, 0.04435181219092433, 0.04971280620709549, 0.04542758975016432,  0.04438245229360251, 0.04591085814238426, 0.05441539853097558, 0.06599708463146817, 0.05778742364900404, 0.05449694168285867,  0.04948646253153575, 0.04687109547861610, 0.05121471292200981, 0.05543193089176356, 0.06122190073890196, 0.05994362639165312,  0.06318971087312358, 0.07489102709573074, -4.74736294535845094, -4.74736315803805997, -4.68750150625701156, -4.71299931370267355, -4.73514854857563794, -4.63619812421150534, -4.54381636267413036, -4.50885021885178805, -4.51161030693639109, -4.52376700922874431, -4.56308616885795093, -4.53531377156277227, -4.45561202606929552, -4.38742672896691577, -4.33101566183158493, -4.23388485370226775, -4.15210833427189918, -4.10534690451752393, -4.08815964798977483, -4.02878153162040054, -4.02490303857841969, -3.98551529137580784, -4.00210638850729250, -4.02721453484295111, -4.00633005765977135, -3.94842117088446853, -3.83630078010985542, -3.81036121453092225, -3.76076071244999799, -3.72306923035553261, -3.70735462655223103, -3.69061259048338108, -3.67270231708632133, -3.70961721032744318, -3.75239145093765725, -3.78202023116616459, -3.76982830482328790, 1.14618427108415188, 1.14618435908348304, 0.92338539243408735,  1.11785249660020969, 1.25193236402235542, 1.16801850138528684, 1.12275869305729392, 1.10326747011519588, 1.36793491255470956,  1.43322266354394778, 1.58990124597570071, 1.65474051867337635, 1.60216121785625321, 1.49858846404766788, 1.58718397556157642,  1.58557549766583383, 1.33013476974486178, 1.21560076767575054, 1.14822078253437487, 1.11151116859161125, 1.03505845684244013,  0.95887501070529413, 1.08131390065832989, 1.29409992484946423, 1.23784532899573785, 1.03370000549887830, 0.74439785684607696,  0.83491491819621244, 0.83349068772845691, 0.85053207912729623, 0.85543521655900812, 0.83050546960032023, 0.74455694785065119,  0.75643370566809742, 0.85052411748459433, 0.82168067074154705, 0.60821758374810564, -1.73001250113853233 ))

  expect_equal(unname(c(tmp$t)),
               c(-0.190117518459368168, 1.157918563602441386, 0.877243403601014382, 0.874461864911239672, 1.106167560126770200,  1.151270942392870200, 1.668686382328767204, 1.478139060786484604, NA, 0.645673074660300439,  1.054147351532157773, 0.563021928450840314, 0.997329477999926439, 1.279940824392099774, 0.272893565127640791,  1.566065593412105050, 0.294201132325721426, 0.658423742106806986, 2.170813115822916917, 0.949683636880790050,  1.294621252557314195, 0.928345489823252268, 0.876459072619722557, 0.796734590253285746, 1.780811025244089407, -0.047193786324633530, -0.268201658023561940, 1.441604029999147274, 0.593399168110722020, 0.636575359681373798, -0.190116326191346552, 1.157919476789254887, 0.877245812821651594, 0.874462970415113894, 1.106169358794030444,  1.151271754775029521, 1.668686925995408910, 1.478141109941488729, NA, 0.645674010451995350,  1.054148646365139719, 0.563022374759681199, 0.997329062640073261, 1.279941844720784427, 0.272895660818363694,  1.566067076705961014, 0.294202881268384786, 0.658424224199169572, 2.170811289957489620, 0.949684708350742035,  1.294620514698504810, 0.928346655338213367, 0.876460365578177303, 0.796736186554333869, 1.780811542161726102, -0.047192583116926201, -0.268200092069023288, 1.441605428857571258, 0.593400507577617198, 0.636576864206733539, -0.178996729210362138, 1.182005531346535854, 0.912507512591118841, 0.915283914749084770, 1.112727882413147018,  1.188240935539833121, 1.701329204609304036, 1.501677688136792810, NA, 0.662201836393779497,  1.085054892588221076, 0.586100785314847217, 1.027020481410134689, 1.309407705844098269, 0.294040604093173896,  1.605543022122468244, 0.310595991491972701, 0.670348343006435865, 2.182228392415115348, 0.988625537778149877,  1.320707732844557114, 0.962660075857610931, 0.902005131019341322, 0.802298960836317132, 1.809338847054759558, -0.019716985975948509, -0.253068331602598007, 1.439398202038306263, 0.610154126406013186, 0.652593213316369880, -0.184974104750127905, 1.174657419310305428, 0.902264122022056303, 0.911544054902629242, 1.119797701981377402,  1.162121410842629876, 1.692684715351393754, 1.500444109219099875, NA, 0.647615953275044176,  1.087424540671998452, 0.581989128565814640, 0.990801699600063035, 1.319249701365547089, 0.265077101833075091,  1.616795015547972003, 0.316719679697416101, 0.667921680776752291, 2.181677351315447488, 0.971705264096772936,  1.301566169284449792, 0.951225564676553503, 0.907604482549297265, 0.815735004687913712, 1.806219265230304627, -0.001714212907754920, -0.242992250672427357, 1.430605657339378389, 0.595952585584119787, 0.638255787301042599, -0.196710180469094220, 1.131179958542432917, 0.875346250876575471, 0.903066751533053380, 1.131551201241347782,  1.111403628068821536, 1.675157004871882238, 1.475600614515860443, NA, 0.643158385257219245,  1.090839994840667071, 0.579070047762597295, 0.931230467981407628, 1.341155181714722477, 0.224532994460980961,  1.633736281352915487, 0.335297667784781539, 0.646650175441908837, 2.156907163263561245, 0.948179434101435681,  1.255855716280503742, 0.933696424760450006, 0.911638147766572926, 0.835204963444254211, 1.787151072976513877,  0.005783823020540230, -0.243414065382118960, 1.403698442179957651, 0.567278351223995614, 0.588300887832854613, -0.164943113659643759, 1.140153972717073083, 0.893901157790215506, 0.931983285186507304, 1.189554381748667122,  1.101532327612698037, 1.700280811507124401, 1.498565651474351057, NA, 0.705424411711623844,  1.143470335152590778, 0.628941481318415985, 0.923101871799253515, 1.401246861402660748, 0.251132191910359959,  1.684759489796618048, 0.398470580471013025, 0.689255147507282562, 2.158105990709679567, 0.977972382650916017,  1.234670607382440277, 0.952837129420238904, 0.948050371135237047, 0.876826797062442154, 1.814233605988236686,  0.068580368497099642, -0.188190585863230897, 1.443756562574039171, 0.595370442659037380, 0.610744692394079358, -0.154458268881620175, 1.142446674414920427, 0.902828410481591126, 0.953549235302246778, 1.229522626047797829,  1.084403571290783130, 1.716947834121235639, 1.523505554283832275, NA, 0.766695889561957888,  1.199947772955951564, 0.671600295006161474, 0.912872946017245779, 1.430702463071675590, 0.265816796893313612,  1.732191145506196017, 0.432142953085590031, 0.703849921763803454, 2.144430139463928242, 1.001078315546249042,  1.187343293957921819, 0.959694730803685170, 0.959767996484628405, 0.897611624585040069, 1.821953723899436328,  0.125236859610133811, -0.131133718292738788, 1.449553229846185953, 0.592963856817358081, 0.608103097092134393, -0.175158048717611647, 1.102919044819107297, 0.878697731897691514, 0.937676520749759712, 1.241188061173639001,  1.040556266895477533, 1.684143074624974368, 1.543167354172471972, NA, 0.780416295713144192,  1.219776568604476807, 0.696102137233017171, 0.888452286721027384, 1.414380704881233797, 0.250502542471048972,  1.741021478934290379, 0.425586691673415007, 0.685219054579307763, 2.093349370055642478, 1.000710372304008056,  1.098560946396920368, 0.922836249175546408, 0.939566376260627445, 0.883729407645567733, 1.801932813088552354,  0.140614777596011375, -0.099355376576601845, 1.414860567905505517, 0.554349299892693770, 0.564104583881802180, -0.195869847107855155, 1.077625565698636434, 0.831499194050621671, 0.906501800885785025, 1.218899325424783742,  0.986675666466504175, 1.639058680112653210, 1.541864017999790448, NA, 0.736120020987646839,  1.246045187087771122, 0.663162628429785128, 0.850432582134498682, 1.363235384128967942, 0.215629060855694332,  1.727418144154957869, 0.375559754010258207, 0.608573318851990841, 2.019817362373243164, 0.996173858196094830,  1.009389502030998198, 0.878303616110027274, 0.879031153298754031, 0.853069176768214588, 1.742466494715891656,  0.146300450244202351, -0.070194864560880532, 1.333219118771795220, 0.512331092419792467, 0.528839176175032177, -0.208990603121777085, 1.051201619335091930, 0.802255505652288980, 0.862481112596698130, 1.186034078732348895,  0.956381770285363864, 1.581774414741443868, 1.538472419190870744, NA, 0.680675544372284835,  1.254491664898838454, 0.606719657199498164, 0.816778990823769924, 1.298743812638365247, 0.181610644492023604,  1.698167874343689876, 0.310089964155512043, 0.530093760368112843, 1.940551989871819982, 0.986663631358682292,  0.915189053673886477, 0.809885087468861786, 0.805907687733232292, 0.838359941098628325, 1.683413185069506879,  0.152236070683158031, -0.081030331097954322, 1.259782824992737638, 0.478763722373010903, 0.483895928503392359, -0.233169372878168524, 1.062262678807542216, 0.760097897597009742, 0.823347261562736743, 1.166040417236662519,  0.974327499209473924, 1.545500131938806421, 1.578538343545414335, NA, 0.658263824871594405,  1.294539458231655171, 0.590165705318721923, 0.808583471034733048, 1.253344202868215129, 0.166014971210603324,  1.686340725419328779, 0.276905861944042653, 0.473152883093571119,
                 1.898206060149023378, 0.988722955457705832,  0.866910039327335458, 0.763760034606660532, 0.716653935201717607, 0.843111452323913158, 1.656599493927618960,  0.187108163649693460, -0.042593465969309352, 1.213003677168026817, 0.476864845797187575, 0.474037107844035910, -0.253101450495270552, 1.077846646798835240, 0.772061039028045348, 0.810370514067811354, 1.167148731977593057,  1.009505471299557922, 1.517461787619020175, 1.609085174439682575, NA, 0.631981358396198623,  1.342382362165136733, 0.601658463182440317, 0.823280689168314761, 1.233701359449832502, 0.188314730723350671,  1.706059231868043868, 0.276276898049456210, 0.476539018795169356, 1.877117737550580756, 1.009195664849560803,  0.872795629597127420, 0.758802723604216744, 0.659665572728293625, 0.865016370233033238, 1.662182850820403734,  0.236576563286173491, 0.001158426095363763, 1.201113906457367264, 0.499587188954064010, 0.500797307508539902, -0.268784219045544059, 1.096793981755149083, 0.793348107714917705, 0.804582006238225289, 1.180051485022321600,  1.050307376247922919, 1.490506144391131604, 1.639643681542189313, NA, 0.620544113480598569,  1.392268905305906745, 0.602987151460992910, 0.849282312813372409, 1.223708803171807968, 0.208843023899529578,  1.733967838179915066, 0.271052467616822168, 0.474111888536532022, 1.857340779458330537, 1.029225981239127119,  0.888251122720219999, 0.760127488610841562, 0.615108368609953660, 0.889858414949380228, 1.695011423588846267,  0.269526226449152639, 0.029660852103385427, 1.186623511584140367, 0.533637308764771778, 0.529972252324053095, -0.231482980110024911, 1.176207798733887033, 0.851444301557899630, 0.829512980854556048, 1.225299947655889898,  1.098294415873104501, 1.485912770144727890, 1.698857583911372204, NA, 0.657648872699347464,  1.426335870029420860, 0.631833478598702536, 0.894641972773967065, 1.236978482754483188, 0.266288851258667636,  1.769916566761297272, 0.289457663922520592, 0.521953003422703121, 1.854548418913642971, 1.060487340797428901,  0.926613056004859104, 0.790430343302932492, 0.615072123445945529, 0.923479154306517058, 1.751579278843311549,  0.311135181057007215, 0.089833246888712026, 1.215833143402696770, 0.580964492708075264, 0.556339218696502935, -0.228840799685637475, 1.206428869399238879, 0.872899570910692146, 0.810244598396554494, 1.248479853905376524,  1.086630902508181507, 1.480143752949508062, 1.716836661727481061, NA, 0.681277304332918332,  1.428539577711972486, 0.638506226255307285, 0.893085664549195912, 1.249331962954181252, 0.269719054069007258,  1.771145813988403317, 0.275441949388084417, 0.526422283977540428, 1.819351541168563680, 1.063116863241887255,  0.931673801607688756, 0.786368826224117301, 0.594238930789405151, 0.927699932338608702, 1.774223091096564842,  0.312820712274535340, 0.124145092614409647, 1.221075188856149074, 0.611708038452653202, 0.520941978283735896, -0.209086323660609641, 1.247789528755908872, 0.919466552505168067, 0.826081066181054702, 1.260804770877476999,  1.092822583753477295, 1.499670654434079431, 1.774894731009473592, NA, 0.727854369511223953,  1.447951369987803716, 0.661452775981641206, 0.905824214802991801, 1.301667092911569235, 0.295966983699144426,  1.803692103498556243, 0.269276041096227159, 0.552884984812476610, 1.806640130974593195, 1.089749860680645499,  0.964487709022076700, 0.810552159897401037, 0.570306143840602076, 0.954056398349270807, 1.810200205306549126,  0.346568662154865037, 0.203694641531791287, 1.245732541012152250, 0.675303345412560763, 0.517075072890956111, -0.194360337716214748, 1.264613539146195675, 0.948329472054299849, 0.819436521549846586, 1.236556299643323698,  1.075128334104775707, 1.510102188601746720, 1.808221382747417172, NA, 0.738441705873785215,  1.423779821014612867, 0.662503554799892425, 0.879558975342986660, 1.297750455696703220, 0.297335737260815469,  1.816876754957987616, 0.242646266820280898, 0.571380292176362214, 1.776091968263805887, 1.084036528919050335,  0.943632631144954992, 0.794802634386499562, 0.533733443181138578, 0.955962054511758863, 1.819336464164542422,  0.349861892306243483, 0.203615866987213512, 1.269791366820454570, 0.685448505908527350, 0.495105562417724854, -0.173939671288102504, 1.283654713979934137, 0.977396762139446706, 0.816575794268336264, 1.219080940377335498,  1.064218463731709363, 1.533607044681455811, 1.844297786023586960, NA, 0.762883185299695454,  1.411713671274712922, 0.664914682076892305, 0.851015838312725759, 1.290806919906188011, 0.307007318699474008,  1.832769708038899603, 0.239869797147500896, 0.598448453256438695, 1.753666435465671913, 1.095487744330456481,  0.921768679747273723, 0.777813143737336343, 0.507690827450941851, 0.955895356726995216, 1.821012228250601384,  0.363964935150809632, 0.214237809022488096, 1.290499010064816154, 0.689610258482539651, 0.479048176237173218, -0.169117057025033546, 1.296700546941922827, 0.981218516249175998, 0.786689051116158611, 1.184382733196344795,  1.035846674814139945, 1.537997229412682687, 1.848257640497977805, NA, 0.770839603026003850,  1.374786790488688570, 0.652640194847124544, 0.809397730581823738, 1.264468271374000530, 0.302974144746976648,  1.832897434072813247, 0.215772536330826492, 0.609862544627236880, 1.720206347608966402, 1.086222155282139390,  0.889056467165277198, 0.754840869025624972, 0.469876439923839317, 0.936323736837823439, 1.798659802608675085,  0.370365603483939543, 0.206862212275589263, 1.297745375372355126, 0.676635473310099544, 0.452744509753403079, -0.153209314846792538, 1.316280470603447217, 1.021432391518799054, 0.775792004310763228, 1.170972402848585503,  1.024339445809462568, 1.562523079228095524, 1.862108528157198295, NA, 0.813833067292654144,  1.361630486118169836, 0.643904876586962827, 0.796149779586986517, 1.260383984093790710, 0.318300077062424192,  1.850113216394319915, 0.203678250653665183, 0.635785353790424557, 1.692413628660149660, 1.103206704731504395,  0.872346986274943559, 0.756709868201488223, 0.432705778741072411, 0.947589124235614033, 1.794749062312210652,  0.380630712473347899, 0.219600876415230878, 1.306995013748267098, 0.671221561847056281, 0.444943830359866899, -0.152548776848035528, 1.319987015706069133, 1.045163410080877764, 0.762894169648366538, 1.137946173694541541,  0.989362659102846931, 1.569182654796231313, 1.886085110829104128, NA, 0.840211055949028229,  1.342859711910457499, 0.628015381919246218, 0.758057189796405573, 1.241087582237755171, 0.336239956603477719,  1.853497350453462866, 0.189496814947053727, 0.656595588868842062, 1.666463139255120351, 1.093903176120715859,  0.850293613276887239, 0.741938685150874910, 0.396622202806610202, 0.937488311845678046, 1.766853834457830619,  0.374663023510176663, 0.204253873065750047, 1.296405173085583895, 0.639033776237413309, 0.422705681080819851, -0.141968009782903581, 1.337017322232863403, 1.086381575834489244, 0.761703163128449212, 1.119266222651270626,
                 0.970393211126885147, 1.590201545053629761, 1.932980124104987318, NA, 0.880251345242638372,  1.338307887359866388, 0.623010300839559328, 0.738436324418205836, 1.246274354836955700, 0.370265735968404119,  1.875752232409209741, 0.187556571200211936, 0.688344115060867345, 1.652125303636954312, 1.104115331353981988,  0.838200550473015116, 0.746340636525840040, 0.369784974466763727, 0.940627656272057022, 1.751110758925595867,  0.380128016672043267, 0.195487508754569161, 1.293899067711476114, 0.617002663744017799, 0.414920743577265994, -0.142030580042575127, 1.352203643464931382, 1.159295995332923379, 0.777442370047206843, 1.115855714687343569,  0.956286174945751188, 1.618485635599255623, 2.007364381440746470, NA, 0.936324030999959644,  1.358025266653305696, 0.644840851060298981, 0.745894335179927870, 1.263190850073319993, 0.381147248346989542,  1.925594733554788274, 0.195210635470463079, 0.723427471685769041, 1.656095749641620207, 1.118247673191683944,  0.859407884718117465, 0.754280154949652171, 0.350345147601828333, 0.959836329751718953, 1.722953821015175357,  0.373069834490621255, 0.208068775529656136, 1.278364451227689269, 0.619195046458739906, 0.407966610646686267, -0.139087401613489037, 1.345767105781095818, 1.196174146116317782, 0.768138185109724203, 1.080965520362529464,  0.929912663039869525, 1.614359530218046013, 2.050379218958103156, NA, 0.954439648839647381,  1.361937373942369911, 0.632784335360717876, 0.751559517981083225, 1.260353585140392418, 0.367714295151062576,  1.994393158163143553, 0.189120126079970363, 0.736371792929557456, 1.660676971248472800, 1.134602097376407048,  0.849808941035561705, 0.750901104084323423, 0.307296099063651185, 0.963897075526082614, 1.683604281490823862,  0.359325285621859614, 0.228520496137937013, 1.254887748281476201, 0.583300084812771624, 0.401622914034178102, -0.127473801351232818, 1.346962152410688507, 1.244944300806285487, 0.760465279542760442, 1.045981695945579215,  0.908713427397507600, 1.615583697431485666, 2.085683968390201315, NA, 0.980219482525110042,  1.368641285733474877, 0.625798631056851185, 0.765278652779681989, 1.264909879346083121, 0.358810072175460792,  2.065787952438334685, 0.187195625927753351, 0.753696615543319570, 1.675443401893208284, 1.156526396188225281,  0.843549423283180833, 0.754219220332216111, 0.272110788107457702, 0.975580695547143550, 1.653665567501442446,  0.349530846954004881, 0.246006089594547728, 1.237952674436722855, 0.552839685093581501, 0.398619643025108772, -0.108317850554930151, 1.364074872805436600, 1.299850358452670829, 0.750606180287308078, 1.011620468111146698,  0.900319362739844919, 1.621169392767065176, 2.121576688176571857, NA, 1.013760404189011810,  1.388274052618735155, 0.625114432607091652, 0.782279428171096414, 1.277611570975501820, 0.364782091373990258,  2.147078985538544149, 0.199403219265621517, 0.782402531116848321, 1.697894800500928447, 1.184843648259743754,  0.840482733643175717, 0.763704832409949463, 0.247950871319519539, 0.990809956200558561, 1.636572930769429668,  0.348666988961311208, 0.266310696704150307, 1.218923026297166201, 0.538436137930042857, 0.390621151024933333, -0.069485670355359519, 1.398630261931207652, 1.376015538694142393, 0.757506474798134377, 0.991685124361462167,  0.896163013848945700, 1.641921765145997192, 2.174206669166684591, NA, 1.051246074488284110,  1.414380975617178349, 0.638137890201509128, 0.814720020098163356, 1.297563115888727081, 0.378492816529069298,  2.247269533016949872, 0.229454950715488493, 0.815198917837148596, 1.731142949816266263, 1.224993581429916656,  0.849605693813958784, 0.782258720791460971, 0.229716090179521715, 1.013054473671664013, 1.630951739069477213,  0.354438429752687079, 0.298465046968672487, 1.205673452763346365, 0.534355765132835114, 0.404777238780171800, -0.048890468355661854, 1.399646275273767948, 1.423133784626256171, 0.746744815483747049, 0.969961617325334813,  0.901085443040460565, 1.637253266943445418, 2.181824690173603365, NA, 1.078319185317751838,  1.433169631689466650, 0.634950827450357802, 0.821094479577408221, 1.315292545038835526, 0.390346453174603647,  2.312211373449613028, 0.259496507793468723, 0.842495331713900231, 1.725796044301339149, 1.236960777328520722,  0.861102136841879195, 0.803221677919077282, 0.208607575556031710, 1.020788373523912718, 1.633944616335018640,  0.370326371619718375, 0.328045560411637960, 1.206879840455816755, 0.527815926755714204, 0.393459886577561369, -0.030442556183501557, 1.390179983745322145, 1.454643715663786585, 0.723185699273434901, 0.943216369161534840,  0.883309084926044319, 1.622587610624013132, 2.175272024645728219, NA, 1.111728601106171554,  1.439105132191775160, 0.625728935175105416, 0.817132348164888889, 1.316806019268976335, 0.392169501173505630,  2.323191506142717078, 0.256280815658794014, 0.853897021510569787, 1.715683756884372357, 1.228957027640705579,  0.862194002232340884, 0.805834994862353549, 0.165894180797055729, 1.031470993490269272, 1.620980302103816495,  0.393646238736872045, 0.337431487023162335, 1.201465737726904015, 0.501372517456203814, 0.362444071023806436, -0.004972329060424996, 1.394045559792942868, 1.492062826738557835, 0.716765168108961293, 0.934917402941275233,  0.892489740132829779, 1.631073133592008118, 2.187736767963679441, NA, 1.141118220293847463,  1.469052987724442927, 0.632189375410524779, 0.844670798334584028, 1.333311880082185930, 0.405114391064504309,  2.346549839344823418, 0.273956216838791955, 0.889275627220208587, 1.718993366498800635, 1.235601397027973336,  0.871984150645609191, 0.827928534872362065, 0.150123350828214430, 1.065899579533133634, 1.632341363278651203,  0.428044754261863170, 0.353761612450684770, 1.204070526517905915, 0.485994300089820308, 0.346660305116960665,  0.022274135652185056, 1.401690037447182569, 1.520876714633980775, 0.704307823911267228, 0.921729249844483456,  0.897284886674792781, 1.634581919446069342, 2.198401814447474401, NA, 1.173735579639228588,  1.491417400888395139, 0.637699038148266117, 0.871841823089535839, 1.351714196848911254, 0.415489515425715217,  2.369055157028532754, 0.288428445188212645, 0.921083516610018860, 1.718881495691846784, 1.240828258321535760,  0.872736595226395795, 0.843123435540982791, 0.139895117863910207, 1.100386223069524050, 1.644886218277382905,  0.454490746310454385, 0.366524312288851351, 1.208073146136150466, 0.467896466716897386, 0.330466696494702727,  0.055110598715304326, 1.418405593777153140, 1.559277098226035774, 0.697591054719727044, 0.910816808645943388,  0.908968994016830734, 1.644151108063007349, 2.223439397062371370, NA, 1.211329416980476692,  1.521736426831337585, 0.647386132858636554, 0.910860959159823014, 1.375096438989727998, 0.431323003479746281,  2.403441222898422325, 0.310023972152414951, 0.959488355379290114, 1.727361965229273544, 1.250266520220540523,  0.878947784169169744, 0.862603846350342818, 0.136561331900499294,
                 1.140801579280875888, 1.665361781051653445,  0.488114427019081587, 0.383864053766230118, 1.213483042671818835, 0.451596810668794424, 0.318791442725120833,  0.083658396535181256, 1.438669684764160461, 1.597649599173885715, 0.698929855438114811, 0.899932142081651509,  0.920105631410400693, 1.661429975893251143, 2.258919116218316425, NA, 1.255817276812501326,  1.548782830138065059, 0.657696276561681503, 0.951553682938505618, 1.399545321318325319, 0.447614682453529977,  2.448428868513339296, 0.341728507801616788, 0.994836208979716141, 1.733376760262890448, 1.256566580713171177,  0.886797683802782011, 0.887310024136409359, 0.133369589478775474, 1.177861551989576316, 1.686990087214347733,  0.528839645508148437, 0.406374895704958994, 1.228165765107002150, 0.440253145008285995, 0.310933777176515991,  0.102492948081240154, 1.460120457389864068, 1.632326211328856580, 0.700939018102044842, 0.890139249689149481,  0.927216354582238056, 1.684053539800787203, 2.282610361790640141, NA, 1.275016208679673024,  1.568013421642128202, 0.657129291977462504, 0.971257310952780228, 1.427647346298566600, 0.454675480275329114,  2.473869597030171175, 0.367672745821698732, 1.004053391967059117, 1.717913772575563769, 1.260958802644935872,  0.895375098101281686, 0.894134606390307618, 0.123399987539411476, 1.196594483986846535, 1.708724710775342936,  0.551230119440185184, 0.414968468428880521, 1.242998340067897223, 0.416642120811067274, 0.287766555921285916,  0.116607141392433666, 1.471466573158438873, 1.626294474069685636, 0.708298177672490370, 0.879655802008668886,  0.922783654463120828, 1.697096963241915502, 2.277452668932904878, NA, 1.281102220522060797,  1.577666218352240080, 0.657549504314116406, 0.957870621322072791, 1.418963634430923992, 0.451464731736476999,  2.473244638965328868, 0.375316758529359584, 1.003870291833610651, 1.690977503181728236, 1.246662631947749400,  0.899302254066477458, 0.881042388770007179, 0.113796059728269242, 1.184346097970047529, 1.709531787327152941,  0.548519386598291470, 0.427074360449752777, 1.225144213725131603, 0.407304591425760920, 0.275107528796407208,  0.133882324184896345, 1.485150669880412888, 1.620295801938424152, 0.716332573117920535, 0.867292482310338619,  0.922750529944706188, 1.705956659053287749, 2.278216024987461008, NA, 1.292368804534264948,  1.580539093871315970, 0.660416350011103770, 0.948564863375485956, 1.401938397334037578, 0.452241971217460348,  2.477109993295027568, 0.386348171022350695, 1.005731221019850796, 1.664700633227830462, 1.232904813978129344,  0.902224428636043796, 0.866630221208169127, 0.109362404505986166, 1.169445698151799995, 1.715663923028933002,  0.546580392001896342, 0.441873949497823881, 1.208891501646122801, 0.395710531915624053, 0.256030579446179185,  0.130042229528757380, 1.481332774183261591, 1.605172558067525124, 0.715169052733365507, 0.843227091374973536,  0.905902097870491696, 1.692005855423025285, 2.270516820326189400, NA, 1.281109281873191286,  1.557683790626557707, 0.647401360918228863, 0.943317413596651111, 1.378487671890516575, 0.435048248887182742,  2.462439207173764633, 0.372999671440418457, 0.995535677451289125, 1.642799125784110803, 1.215333231009053883,  0.890534850240304432, 0.846063995173420236, 0.104045550906148421, 1.148639511965744209, 1.709903151885347805,  0.530716926846326986, 0.430981112707850167, 1.192556363692963917, 0.379333070467427258, 0.230863735163320150,  0.069053885813441893, 0.060275980829665031, 0.067079125377652968, 0.061947052814434939, 0.062180949559397840,  0.048962188195342468, 0.046599012205015521, 0.054068387971404598, NA, 0.055190904330706114,  0.067195716437897152, 0.048793995530132868, 0.056857127622694202, 0.059915679421702318, 0.059374202700266700,  0.057706368148604185, 0.078719522288371646, 0.055163901192144128, 0.055044140210587851, 0.052623507319415633,  0.047818012979328223, 0.049363285571433206, 0.061211005533910595, 0.076232839219432205, 0.047974726110084638,  0.077313006202323523, 0.081824339210309002, 0.046768142018955669, 0.046262423693696708, 0.061875562071633479,  0.069053871858866689, 0.060275945968134313, 0.067079042409212281, 0.061947034580769944, 0.062180912558879975,  0.048962171666221091, 0.046598952610616078, 0.054068363996527077, NA, 0.055190899335849551,  0.067195708127289505, 0.048793981367056580, 0.056857116595559518, 0.059915657901569490, 0.059374179489792928,  0.057706358345481906, 0.078719506827001282, 0.055163891923406393, 0.055044104373276274, 0.052623505479053569,  0.047818001241988969, 0.049363282432105346, 0.061210993070482361, 0.076232822555130447, 0.047974716624371264,  0.077312996084957661, 0.081824304014620791, 0.046768111106261889, 0.046262411175193127, 0.061875544732004545,  0.091930581199707956, 0.084393081706054418, 0.086331763552463692, 0.071555214286554730, 0.083938375908183496,  0.065458230026490635, 0.069148922166171051, 0.069837134323643885, NA, 0.065547148297823460,  0.077331268141338977, 0.057687890746222950, 0.074275737605438166, 0.072939855522538982, 0.069333207536647679,  0.067766733180584321, 0.092171400067769554, 0.064305793876184597, 0.076657490484223420, 0.059281333597428522,  0.065004085481904422, 0.061807141250822845, 0.079751226674003209, 0.087624165667502418, 0.061626889305565140,  0.089601394580651750, 0.098110088449170393, 0.061378296989368537, 0.055267165154556576, 0.078166051436612177,  0.085187614810651777, 0.075381450937692218, 0.077558301530529256, 0.063824309713944691, 0.074526047116468624,  0.060427280206098985, 0.059128678627956045, 0.064262046253390673, NA, 0.062102552028315924,  0.070497862546288065, 0.054285472507269859, 0.069513215871234810, 0.062940737463638416, 0.066782725379958260,  0.058798414417777792, 0.083893384295794335, 0.059648680786131059, 0.064072605762108742, 0.054261160414432463,  0.056507528242288982, 0.056289018730931194, 0.070216500647548599, 0.077828730695070542, 0.051658434268184818,  0.082081390264468646, 0.088299219064417012, 0.055094811702942892, 0.050828701726567491, 0.071847999238585175,  0.074700717409451195, 0.067736375528290080, 0.070459715964744291, 0.060559784602270489, 0.067069529959182889,  0.053534422362126292, 0.048264583871448788, 0.057790432445417442, NA, 0.067278995141361070,  0.067926805182244365, 0.060206829999555816, 0.067328487136876183, 0.060420294722230698, 0.066507556170381635,  0.054700574460407665, 0.086089285613268285, 0.070060671645426037, 0.053362970537605484, 0.053540083851660522,  0.048659641749298946, 0.052153419000965096, 0.064093382650511413, 0.069233883150797121, 0.044911535531834232,  0.087002786816487945, 0.084400558323448630, 0.054560626908422102, 0.050151878400607738, 0.072788784486931837,  0.063492387186887253, 0.057938736161360964, 0.058128262452385782, 0.051577453608328910, 0.054959948440716125,  0.042572767367691416, 0.037221145236005865, 0.048371231398238679, NA, 0.057352111890090561,  0.059928383262701390,
                 0.051722066241283733, 0.057476418981052171, 0.050643513359800191, 0.056372773993698985,  0.047810496822260927, 0.077705905673152442, 0.060649036660970011, 0.045811147287795867, 0.046321115860352396,  0.043068516919316908, 0.043483341989598759, 0.055133186745805682, 0.064800867109738106, 0.038544870715203600,  0.075643282424398481, 0.072626543879487665, 0.046208133681661226, 0.046496882858754092, 0.062854477074510764,  0.060514085565518211, 0.058513451937723213, 0.055214469236626068, 0.052291920806813406, 0.051427191144917048,  0.039553882099928966, 0.037299948288495872, 0.044624696742786699, NA, 0.057042370735951795,  0.060928329677077592, 0.051592870617875472, 0.056228614597013914, 0.048238983841181436, 0.053343457058928588,  0.051188173429833138, 0.077896992310822213, 0.060473078726681298, 0.050031792284203305, 0.046573631858241170,  0.046100823787479900, 0.044181600086563418, 0.053730517895683498, 0.066027125816269364, 0.041091920070479612,  0.073598381128124099, 0.067017013702075107, 0.045072640969480368, 0.049490750430315505, 0.061539174884746708,  0.065704821802576349, 0.056592600972194158, 0.058894754451725019, 0.052400329743628613, 0.055764734629180940,  0.041598432683125922, 0.037089300440965450, 0.048198413217964080, NA, 0.061440618601173710,  0.064602290171644192, 0.060693975085760597, 0.060181712267688856, 0.044955403751797948, 0.052158894749285506,  0.053342786710068445, 0.080740831678493580, 0.067912946469940561, 0.055363912819129002, 0.052816238262594871,  0.050614713408352476, 0.050076987434244033, 0.054907720402075408, 0.071681381620349024, 0.049018805825022167,  0.072878618640653689, 0.067171693896597828, 0.045362163205662484, 0.055634661025566960, 0.059648018299882748,  0.079767071513563353, 0.058909381854858438, 0.067446105945525622, 0.057457493254236471, 0.057455665966939989,  0.054305095536501577, 0.043793571587548291, 0.053898506104454166, NA, 0.060472221135582621,  0.070245784988753737, 0.057788877791665812, 0.063176546808176318, 0.045498053958732652, 0.061991860443210464,  0.055078827895528010, 0.076232037303445258, 0.066794117822233504, 0.055300852060448942, 0.055803217489240844,  0.052713920130556056, 0.056595629861243814, 0.056242356416344459, 0.076459727993767629, 0.053461943079033798,  0.072696332467829611, 0.069320895519507930, 0.050653686579199643, 0.065981469239494508, 0.062804587162821446,  0.070092256274688167, 0.058000416267537165, 0.059309740019203243, 0.053516408493572747, 0.052837653093134943,  0.052401258156191913, 0.037447778962966922, 0.049868132211146693, NA, 0.057029698555062826,  0.060395004376586535, 0.054953874893109309, 0.054802875915278958, 0.041823074202825689, 0.054577945088263954,  0.053825559917747803, 0.074823160126014032, 0.060086039021730875, 0.050824423840591618, 0.049032212439898124,  0.049142281305896915, 0.054689359802902800, 0.049077015976337637, 0.065182394739868948, 0.048046287108608358,  0.064800720769290179, 0.063268086372844742, 0.046908557527404991, 0.061176003625675823, 0.054618175800317109,  0.065911269691229554, 0.059271612149700267, 0.059800335122438444, 0.054403153139672242, 0.052381768607930500,  0.053764646518953421, 0.037940339336543345, 0.052252625618712821, NA, 0.054921440373688679,  0.056211496209518479, 0.056803206059406502, 0.055177104154058361, 0.039394072328751965, 0.053810920303010781,  0.055534101996747645, 0.080069566650190938, 0.065803477744443561, 0.048338866719092738, 0.045692151653778476,  0.051075430520559198, 0.056410683030516828, 0.051245443428147533, 0.062815688690371602, 0.045852704786406621,  0.062635409163889905, 0.056861402037990200, 0.050455678378463792, 0.059063270665217937, 0.052140174864733956,  0.063612069230913448, 0.053657078086921092, 0.054327297932455401, 0.049828238613090194, 0.049197466111074714,  0.048467691166282224, 0.031212692472538054, 0.048424277337631771, NA, 0.054617393378102688,  0.051916179936187412, 0.052413686742894330, 0.048956846530258190, 0.034244915390900993, 0.052589541364412684,  0.046527764728913519, 0.070601537320836211, 0.057767149845779353, 0.038022885595362446, 0.039737916972372500,  0.048260320799383187, 0.050667651534965462, 0.053688782923595080, 0.051134414085606678, 0.044108608436739277,  0.050543685505319194, 0.048760022172889675, 0.046096484807814829, 0.052065662740371901, 0.042822697053314054,  0.062642942533175539, 0.052314537349318277, 0.055453438011802893, 0.050141692080525935, 0.051128196114322004,  0.050917913368191160, 0.031487689002542542, 0.050729374359160626, NA, 0.053903576127299187,  0.049085363206237027, 0.050893293091108044, 0.043438352207106561, 0.037961399376791660, 0.060113567326292958,  0.043468123784704819, 0.071069884517370382, 0.059781527440890089, 0.037195499440736114, 0.041424910142778194,  0.047975211448934074, 0.049886701395340627, 0.055300944049080411, 0.044648926068652596, 0.039733368604127337,  0.049508106438972695, 0.050807472242803513, 0.045170058228077049, 0.046401451668302632, 0.037984760026984490,  0.059389586472241788, 0.047713632441716142, 0.055132487037241863, 0.054376068910648820, 0.051334392880376167,  0.050561267168338718, 0.033662501256347704, 0.052627770012998565, NA, 0.048116401037685157,  0.053115652262942820, 0.047354851490716762, 0.047103995490356446, 0.036523912251404472, 0.059857858409546078,  0.046835823733663316, 0.071031665164832183, 0.057279191451871908, 0.038227056736714560, 0.041045383294566889,  0.046207047122906950, 0.049181365337348695, 0.048185043605675139, 0.043957407188871089, 0.041067037337794828,  0.053904237191129897, 0.050748067615175939, 0.041410531171134246, 0.042313521629471398, 0.039881904447810472,  0.055700815708315053, 0.043367719399049966, 0.049858695270732306, 0.050304509411490880, 0.039652773131030099,  0.042680657547733716, 0.031409138757362287, 0.046903367790151390, NA, 0.046064954224411855,  0.046930496845052079, 0.039529197951151243, 0.040151619496063128, 0.033256488650170397, 0.053286917447824531,  0.044145365323574187, 0.067471499361911824, 0.050927033161084725, 0.032586422898451491, 0.038700417570378613,  0.040082059054489416, 0.043725803233098159, 0.041298252230894721, 0.042286989580907638, 0.033653869091168812,  0.048887531656558132, 0.051458177507425013, 0.037393503572666374, 0.041301187272899852, 0.041928277018844294,  0.057897937900765729, 0.048849783123117739, 0.051215445592259032, 0.049990055225378964, 0.038409124209111731,  0.039038177908632031, 0.033198357170818538, 0.045418645337441083, NA, 0.050875213193203914,  0.048060916584681437, 0.036705734142378349, 0.040195666185810879, 0.032149126525332195, 0.052125053462739997,  0.045231461728503841, 0.072750263244624955, 0.048617262621062182, 0.030126855965302516, 0.042482947343783800,  0.036707770507663530, 0.040992948710539476, 0.043301509580232773, 0.044422039386442076, 0.033519406970229246,  0.049223227260370664, 0.049421467735388129, 0.037262644271552771,
                 0.040763114963532951, 0.045464164044076799,  0.059123775462837272, 0.045656127438746891, 0.055779971868352964, 0.051479678833439638, 0.038682115045248724,  0.038500112924650512, 0.038515558166835037, 0.048724446559671238, NA, 0.056240975691140282,  0.046973885150842484, 0.036562565357873632, 0.042306537634879918, 0.037822786719482006, 0.058199692701517251,  0.047388755878996804, 0.078466038417252526, 0.051447766738111966, 0.029116636701887414, 0.048261934200188095,  0.040589729772493199, 0.042540397344029958, 0.048108907088231567, 0.049683955049347632, 0.033658159971495752,  0.052312531502947873, 0.050864419124952501, 0.037002183224836928, 0.039473149377741414, 0.048391773048888545,  0.063483171207465869, 0.045715707005707734, 0.060629853423605114, 0.057623642786684456, 0.044201398868175173,  0.042595101061595163, 0.040009176657660905, 0.054841176265677562, NA, 0.058650952911899590,  0.048175918571315243, 0.041880147798835941, 0.048743049485917039, 0.048309487687430722, 0.061598034811982352,  0.049740587329092263, 0.080527710658553742, 0.050519592690965974, 0.032242788818306806, 0.053512569161058496,  0.044740780619386557, 0.045998157434660056, 0.044077146422912314, 0.055711569783225397, 0.040485043688095093,  0.047224938656277277, 0.054430904214724588, 0.041674631400224246, 0.042355693599046271, 0.048828202887922736,  0.066641363770276268, 0.043538778430914346, 0.059148137751705659, 0.053034587500437567, 0.043352354442740479,  0.044619396000384476, 0.041318375170874617, 0.048077011614805397, NA, 0.055001755277292001,  0.048068487603578548, 0.039742615030548392, 0.048295592787079421, 0.046161595365121461, 0.056849469584787607,  0.043120202823533293, 0.070914492088022987, 0.047089788806417066, 0.029302173650707773, 0.051937411803149709,  0.045609887602653928, 0.051188397424810093, 0.043304162249879014, 0.056799302136809449, 0.041284568956883497,  0.043614231197072736, 0.052748213001566777, 0.037768087521621800, 0.041857508761990410, 0.051909659625847863,  0.067279586490869889, 0.046961382869231079, 0.048301877209231306, 0.047170704508938670, 0.040593930641634378,  0.042155525764668349, 0.037359346698538215, 0.042218039620617959, NA, 0.046699656483554883,  0.043272715511576132, 0.043421534713697502, 0.047371267343590637, 0.042402540803304620, 0.054483001071785109,  0.036355499658947528, 0.063910812682328635, 0.044653275368888530, 0.028782102452764882, 0.046686950790455449,  0.040883469917870577, 0.050171219753746964, 0.047192796428820842, 0.051054034582011992, 0.040469174130629629,  0.041101810490553961, 0.048955681720110930, 0.037260772547068714, 0.041020914289759070, 0.050727933320108616,  0.070783137729898027, 0.048195559763287807, 0.044008377936115110, 0.045790280998269595, 0.038199711381495707,  0.050157320049042997, 0.040949803450734470, 0.040076711314870304, NA, 0.042371706801258964,  0.046235638966405368, 0.046296323560356969, 0.047438606314875842, 0.049797632531111587, 0.052859257797423398,  0.037673029838387301, 0.058775644134214371, 0.042567025946528739, 0.034471242009334796, 0.046809896654269184,  0.042475503530376366, 0.054904366205398715, 0.058897942541192523, 0.051055630321631691, 0.043165876359771468,  0.042598642524095717, 0.052803843224506805, 0.037563388869507122, 0.043456893296975913, 0.051407129431053224,  0.077501329958872198, 0.053499829961718863, 0.045386897696137511, 0.048600160111214694, 0.044421712945568140,  0.059871573540195337, 0.051263524661595894, 0.043994109935233686, NA, 0.040952391938749153,  0.054080291240260023, 0.050532687407355129, 0.051493787402030464, 0.061662096717235777, 0.058666974006161492,  0.043937678012093653, 0.059273659505530894, 0.048029897114206507, 0.041237707867257355, 0.049091645763970550,  0.044900561466470271, 0.063586561426896296, 0.071639274022662278, 0.055404349086093957, 0.048224350628202572,  0.048655874051346720, 0.056917115842991250, 0.042944924545962500, 0.050367870072507440, 0.051806133061991708,  0.072420451748830217, 0.048457145531523076, 0.042479784974177114, 0.046872702846717372, 0.041123772901795863,  0.051937084093555547, 0.047477600706940307, 0.040660217033295698, NA, 0.039457101304495770,  0.049805255539250318, 0.049271089753248530, 0.047171307734065589, 0.053095455370538495, 0.052259164514246483,  0.038770864718773480, 0.056821735677863201, 0.042549936275450340, 0.036450004134350761, 0.043290396616035925,  0.040873098395311391, 0.055892015504664015, 0.066225625209781397, 0.050237585442376338, 0.044560467906641081,  0.044539436977251053, 0.053492250757474880, 0.039470744867667364, 0.050412465768304590, 0.046769395319223722,  0.072546574293690991, 0.046911483682538897, 0.040221944064524118, 0.047132188614250940, 0.040838435867557137,  0.045501834138311409, 0.048009831028272054, 0.039105056510523893, NA, 0.037970137825795697,  0.048166418813396950, 0.050337064213640551, 0.048983126617268474, 0.043653036385284151, 0.045405321459964919,  0.041864668787611997, 0.058375309742857422, 0.038552630248409880, 0.037641022101717186, 0.041739719479445658,  0.037985474623919289, 0.048316950842469834, 0.064472629723783176, 0.048530519000261778, 0.045246857987207192,  0.043792586453056698, 0.055498435803320499, 0.039379016790623445, 0.052172273937046286, 0.047692403300117918,  0.070221594737739465, 0.047093186751423322, 0.041150171970289667, 0.047929151010264917, 0.040225998485562110,  0.047418841880841438, 0.045880045148553372, 0.040895262807745233, NA, 0.037510804683881002,  0.046598445258475944, 0.052360520025345379, 0.050191224446792748, 0.038643992168252976, 0.044574701233121468,  0.044328539498594874, 0.064225524509744281, 0.041627721290683074, 0.039376185150536952, 0.044068102131240593,  0.038371859670300691, 0.042926633868094415, 0.063177677598485540, 0.048951079554191854, 0.046928719348572089,  0.044285442421548520, 0.057516550736662619, 0.043125917538519999, 0.054784559192409869, 0.051934782220826731,  0.073989776397749507, 0.055815451697001535, 0.047327988653304121, 0.054124451169700064, 0.046898945444651070,  0.057083522641842410, 0.054057839751766375, 0.053747780601474657, NA, 0.041725338448520582,  0.048679422526955306, 0.062724612922278850, 0.062063145040634815, 0.039720157870480582, 0.049483937439354031,  0.056922656714581021, 0.077144553320389886, 0.050447746273305040, 0.049029291638651031, 0.053286575484249164,  0.043009701033199885, 0.041713800791456077, 0.071121951428502264, 0.056449917296269615, 0.055705490914682479,  0.051477397215702783, 0.063790501060694102, 0.055917770791062020, 0.062398118761871131, 0.065386869814877735,  0.078441743182997459, 0.066603081030384773, 0.051739023264263148, 0.062715306947454710, 0.060057447968335689,  0.072901890661169810, 0.068952851667626963, 0.073419814488142165, NA, 0.049071906267225178,  0.054685031387729857, 0.079187329042712792, 0.078535182089748801, 0.042507383642978402, 0.059551568873224776,  0.078631059989938662,
                 0.090410350509578991, 0.061026335569404774, 0.060925694464100288, 0.061493025852447369,  0.046612619544843574, 0.044905507231989578, 0.082228413033069617, 0.063132845803416687, 0.070223334288168848,  0.059193810231313218, 0.068229859473582316, 0.070222135550885550, 0.077403936394837045, 0.082970501320123027,  0.074992597618553550, 0.057217903680963009, 0.042445940537255991, 0.056838020269268277, 0.052184226705578075,  0.061878213178398075, 0.058978484174572389, 0.063313043846615583, NA, 0.043888341828354022,  0.050363057170841781, 0.069718965421121229, 0.069066559224921720, 0.036959846804593376, 0.052317672003437711,  0.064429015805937212, 0.078229750982828430, 0.053296472247686077, 0.051373758829768856, 0.052365600134611147,  0.046933978045535084, 0.042580252993691170, 0.074126921366622495, 0.054512263850284043, 0.062733745387498749,  0.053362321880655830, 0.061444064122242820, 0.061870470054826689, 0.069741160616853828, 0.073298576269907309,  0.079643039032384483, 0.054217530454418933, 0.038593373337272988, 0.059363189206742134, 0.048174895831334372,  0.057038697408660063, 0.052825590689898394, 0.055440493806849053, NA, 0.043526337767470627,  0.054390451451144364, 0.061665256241783754, 0.062654098292257487, 0.040814391289939814, 0.053201725856211968,  0.054146187134606991, 0.067434085068729499, 0.052663212573962087, 0.046078947678846266, 0.044408000867278218,  0.055444628394456484, 0.049692551427082898, 0.071202417455823161, 0.050436272011057728, 0.063681112554454772,  0.055449516237508212, 0.062271436818607820, 0.057505637857400344, 0.066594497644149112, 0.063689740656995411,  0.074657299040113131, 0.046399564738551460, 0.036341061613816714, 0.055393845370686333, 0.043516794139464174,  0.050331842980803586, 0.047238298423103148, 0.049108113293293273, NA, 0.039070380379546563,  0.052516877342964639, 0.055176453688559744, 0.054608809297749181, 0.035465647945876415, 0.048925183324340080,  0.044761281001645538, 0.060970282405079498, 0.048398617874193667, 0.040793431801108442, 0.040660936818294600,  0.051042746651204597, 0.048291228420357324, 0.064799432155589004, 0.045820917051427872, 0.057596230681683347,  0.051226295380763745, 0.059883639380965817, 0.051612673231001792, 0.061524219447290272, 0.055939673234900192,  0.072507728089711704, 0.042386462626165390, 0.032116682670252854, 0.052200420015126855, 0.039975213900934919,  0.044769230339879909, 0.043183592682005355, 0.045831310716954912, NA, 0.037988200497624768,  0.050762475303635590, 0.052037951948903649, 0.048441480431390641, 0.033878767392814702, 0.046712261744458063,  0.037358404047659270, 0.057600455341639913, 0.044506927171760441, 0.037215280146958768, 0.038775893729230945,  0.045969778563424213, 0.048050723624307082, 0.063870212102834029, 0.043603114619347086, 0.053706433493682273,  0.046959720859023620, 0.059713968191324317, 0.050511633437498538, 0.060159437337397273, 0.052559106345435605,  0.076395705288705526, 0.046545597296051289, 0.035241714954838646, 0.055557033849784727, 0.043160299429553156,  0.048152569538224152, 0.047151937599728022, 0.051801717153856362, NA, 0.042379228602172615,  0.056448635436555268, 0.056345712521490648, 0.049535807650649771, 0.040129734318506395, 0.053259340446912787,  0.037314972819067573, 0.064099895678770391, 0.046908560747475264, 0.043116136335481774, 0.042598438905650765,  0.046788937247521478, 0.056078901361639975, 0.070419430372314989, 0.048386080961845115, 0.056036958208243476,  0.049407376575651668, 0.066897157876130098, 0.056295573401618150, 0.066275516434374709, 0.058363564205654118,  0.078824834368333110, 0.043674244437300377, 0.037245530022956129, 0.056494807488324694, 0.048144032228535374,  0.047185986297680095, 0.046518542152083259, 0.054977659607779172, NA, 0.048885023064937534,  0.063401944661752252, 0.056276824283519683, 0.058369368010799394, 0.039467329607831607, 0.053958942224417626,  0.044488099408858874, 0.065680471927218112, 0.051817667612569125, 0.051782875398665967, 0.048018310571486172,  0.051570220609940570, 0.059450196633753914, 0.072579484513034420, 0.057764682402078826, 0.058175277495310304,  0.057994795346288104, 0.066953679026827029, 0.056351091458805583, 0.072925808708314679, 0.063774650006065928,  0.081888391095641916, 0.045720329745238215, 0.042998219450258411, 0.062663677462961878, 0.061936964627637162,  0.051706862927753290, 0.054956116777698079, 0.059067597891022311, NA, 0.048429045567140352,  0.074033928021101827, 0.057845646577072848, 0.061494402108865365, 0.048829855289287349, 0.058923989460646642,  0.046649111582124952, 0.072196241976723877, 0.050263015831214425, 0.060098021106493246, 0.058154133248874001,  0.061896700315327111, 0.063948802683990516, 0.076819350580969420, 0.065738491034334712, 0.064934264689461257,  0.063410651745885671, 0.067126826550064633, 0.063226251968936956, 0.081148480063634149, 0.071097091331736056,  0.082872865406791493, 0.050817106142030791, 0.043341606438532167, 0.069592672594779095, 0.062187793415787798,  0.053540294900723526, 0.055134628119762191, 0.054791119843333937, NA, 0.048575602114734627,  0.068592527907721718, 0.058168550004027188, 0.062050403439751267, 0.047435420241709832, 0.058006500756462173,  0.043257891931262733, 0.073110389699887493, 0.047238371448745972, 0.056436011177783560, 0.061196200390409607,  0.059907567997739733, 0.061403242012929841, 0.078632066066817302, 0.063921381221592036, 0.063169339110901446,  0.061731463885426316, 0.065317881486733884, 0.064552943379112104, 0.077968490149628922, 0.070355342494970649,  0.088104873403490450, 0.057565009163319962, 0.043958622779552044, 0.079181070692417851, 0.072605043102677569,  0.067081966793477832, 0.057612710983598360, 0.058698039276427111, NA, 0.053852283499795497,  0.065364495230782318, 0.066142298066302341, 0.067318718567822691, 0.049159908091379614, 0.066993206140282174,  0.043354269364940023, 0.079665948111198939, 0.046831931961473054, 0.062201250771807909, 0.071390900501720089,  0.060434895718726379, 0.066580788843435601, 0.084906766970099967, 0.067315135997219069, 0.067436150232277489,  0.065126468702228504, 0.070291877866057270, 0.074242145360705308, 0.080378203423150829, 0.070617143115609238,  0.101381068717944750, 0.072716850182467513, 0.049399663065894342, 0.097637850905149259, 0.095450641961058430,  0.089751144596268831, 0.072242182088867485, 0.075615856629800790, NA, 0.066432939414292652,  0.073014293611196962, 0.083303479246792433, 0.085035686594151505, 0.058912755657499816, 0.086359908088818063,  0.052817236501791029, 0.098002408415253678, 0.049317509719802990, 0.080867521568789916, 0.092483319483437354,  0.067906556518779276, 0.083712883749521350, 0.098618297549017409, 0.083572745517167618, 0.080301180233012526,  0.078572149064297908, 0.081874396622967263, 0.090915320589439033, 0.092515816088690445, 0.078546965676694516, -3.913220962243279644, -4.763619267145426939, -4.590126390898944031,
                 -4.757619621424368894, -4.875387499005624292, -4.238026259084287872, -4.293816172494828720, -4.225109339434756528, NA, -4.833243421632585246, -5.499101562012784328, -4.579526660447919006, -4.387822139883613914, -5.013861910354064477, -4.217696734014592330, -5.258967228868244526, -5.468401756477135223, -4.988660220064788042, -5.144515758607536071, -4.787288965558587961, -4.295037800524511340, -4.402487701779431717, -4.734424707502040697, -5.791124671989683392, -4.909395369393330100, -4.990252821984061171, -4.806302079101049785, -4.566178585057744321, -4.194724007508189523, -4.541656487926445607, -3.913221055098049739, -4.763620717912793623, -4.590128299498082320, -4.757619728886752775, -4.875388085990020492, -4.238026864744409039, -4.293820017402887323, -4.225109529663329866, NA, -4.833243351631923090, -5.499101791807422757, -4.579527894793484322, -4.387823286345792440, -5.013862253844926897, -4.217697171326592276, -5.258967200627687077, -5.468401782626794549, -4.988661498898681046, -5.144519003702568760, -4.787289368007698442, -4.295039273595009810, -4.402488181987404658, -4.734424937635782982, -5.791124747351088509, -4.909395631596645160, -4.990253048027054383, -4.806303066450574590, -4.566179173197493668, -4.194724279789600274, -4.541656902446701949, -3.854884006401545271, -4.706541979390959085, -4.482306955344278698, -4.651861302635174056, -4.825011733696501715, -4.138343581979645691, -4.219345521123176290, -4.160665304663201169, NA, -4.795798478867711268, -5.444254010595081184, -4.555387695590013664, -4.306064619152302875, -4.931605995786526364, -4.175057177655880913, -5.175686198273843353, -5.428965305785913742, -4.963622323099622591, -5.091740589371850589, -4.715356458428894726, -4.228399802927434337, -4.316325744506590567, -4.660588754968665981, -5.762012550492795704, -4.840108919454698011, -4.927390200187240055, -4.745669941188656260, -4.548151585402731456, -4.151349905756628544, -4.476994443550120550, -3.899297623039155614, -4.746129412977624362, -4.540193572490033525, -4.677394828407173577, -4.842878460829568077, -4.227378442178594931, -4.259037059752120769, -4.207854178987153304, NA, -4.845920056566844103, -5.473444508368333850, -4.600396906875493741, -4.387597942929632566, -4.922114065581767228, -4.257445313070777004, -5.185440009547605200, -5.422110727368213645, -4.979242227567159951, -5.081445296508173115, -4.772752416810997822, -4.283144784825837270, -4.386704648057270006, -4.664870392198453786, -5.721725455837987973, -4.844257101135605481, -4.926054189462234945, -4.751603118188740815, -4.571661267225322511, -4.203041147460242932, -4.496670117233001029, -3.955017787644296678, -4.886566107980306661, -4.634401862425260532, -4.678892422715885679, -4.856376330701144717, -4.325544665280678380, -4.317364204533138405, -4.326535748603549614, NA, -4.844844088968517148, -5.485005532722661670, -4.584743571691757502, -4.479171703667460314, -4.861567355695921577, -4.343588677895938943, -5.168413391367342236, -5.376191383083508413, -4.983405148306938592, -5.098435745528229646, -4.828705398430593654, -4.394915700314713014, -4.471083513325020853, -4.647010951295952097, -5.621107156977457109, -4.871683349304066546, -4.895023441683515664, -4.811141839315102864, -4.613692089635151206, -4.292203027207280641, -4.540954355092827299, -3.911527162155126636, -4.862905818948028980, -4.601244778617057385, -4.614944004048567194, -4.743109711923455762, -4.326075395467633200, -4.259534566368998298, -4.316730032134209694, NA, -4.712338769542903094, -5.379376334279379890, -4.435592025033646024, -4.468653427295024017, -4.698031763743045985, -4.256395803926945831, -5.069139163678746840, -5.218843313612437029, -4.851400294898461141, -4.989915025571916019, -4.741991554024957978, -4.352472285137592323, -4.433024053192522906, -4.543602356834238520, -5.444888161154935347, -4.755203644349174752, -4.747037328595292038, -4.730120170810508284, -4.478467326968919693, -4.220490781213015019, -4.423466631028748708, -3.889806240099362711, -4.848543419412200350, -4.576447983991343449, -4.557150772910184777, -4.643567598517471140, -4.325784047487232442, -4.199763523453697189, -4.302791261359524277, NA, -4.582328635066814648, -5.268971508963641526, -4.286168604887016009, -4.446473191330738928, -4.545548246399476966, -4.180191292189014973, -4.969058180250983447, -5.090176446518639430, -4.750412016527317149, -4.872274442294417973, -4.657802180928631408, -4.322005815936723394, -4.406496158844221966, -4.441586316598358231, -5.279590928233812264, -4.649272240952861068, -4.588362977570419332, -4.651081206519275923, -4.375536162477438040, -4.167553354925999543, -4.302317081063231186, -3.884067952312432848, -4.915404341628157425, -4.599936922353109026, -4.552183882529465819, -4.574422213730281683, -4.356591528488984011, -4.227350162402585987, -4.298146685319313320, NA, -4.500163080526268899, -5.207514644192354325, -4.134401404874163966, -4.423668392471434174, -4.448191577995886270, -4.157491440546772665, -4.944508271776573594, -4.979314691734501253, -4.673389262589003224, -4.808621398810068115, -4.586609140921706462, -4.388645134427685335, -4.428664090988593927, -4.398837479439775322, -5.174538253623284234, -4.592123373536828090, -4.513406251896341814, -4.567284156947080120, -4.341783256134090330, -4.175955194566122408, -4.270462261356406053, -3.874556658577355872, -4.960886767278993581, -4.631724953007759105, -4.551728317252572431, -4.571114119708959755, -4.394203790282287692, -4.246127666006806578, -4.341068441463523442, NA, -4.522391218531174317, -5.145620137688561080, -4.088842244201822851, -4.411095966524031908, -4.410413589579192717, -4.162539623424341251, -4.965678522576246756, -4.954480941641214997, -4.699504343070595169, -4.754658675904550158, -4.520832765096106876, -4.434174632098516255, -4.440108360083637074, -4.420786170048248920, -5.102152927142634020, -4.605579566784275158, -4.455805411266450911, -4.476454654858252447, -4.412022935827652148, -4.167848908373960271, -4.215446646341462156, -3.897133946279244032, -4.974882281067189105, -4.646652971151834066, -4.576232072018626695, -4.547413768128691913, -4.428093196980856128, -4.315268141832221183, -4.373722020069429561, NA, -4.519848833540430100, -5.133948599453279193, -4.121919315702765374, -4.412164359355596588, -4.430387748920166402, -4.157487734965644499, -4.959145804643209310, -4.974462728287029734, -4.721803624002840039, -4.751389243508587157, -4.491549295474513315, -4.504414739206806573, -4.474218877815299145, -4.432484095448229411, -5.056094764042124545, -4.677859346091007176, -4.415057020907582341, -4.492211064918234520, -4.414060703247812967, -4.194310975044603929, -4.168145092736875057, -4.011343999330907906, -4.991467501117814010, -4.710589900134488417, -4.664814110608226372, -4.559870053538545243, -4.450476911451842099, -4.440391368155175655, -4.405782827710075011, NA, -4.546764984662072351, -5.138772570478808532, -4.169629402713169064, -4.445017123562264416, -4.483518458050559374, -4.188842914889280955, -5.001148179631234925, -5.016564807174029639, -4.773298704202810328, -4.764015687179206537, -4.505783556904957798, -4.579249275718938250,
                 -4.528535194954390164, -4.508890501320966138, -5.063202158357037241, -4.761246012773127312, -4.399481537672711440, -4.527111532102112257, -4.462852907477187436, -4.254677255526906343, -4.128836418275994902, -4.023297331354486417, -4.954973860285818432, -4.687287678936113089, -4.671892383500688695, -4.535542461107362477, -4.412350574564113614, -4.461244480583677507, -4.418512537641416493, NA, -4.541204670668529175, -5.069305032586640891, -4.127368392841162859, -4.384376360104218051, -4.462385178916191464, -4.185663943029053868, -4.989703981416770517, -4.982189025969113949, -4.757603960738048698, -4.710117094544544081, -4.470329522695841717, -4.564462429942487631, -4.509622003625784359, -4.475941501342127538, -5.014602658287478754, -4.764270228377747607, -4.336638378884287270, -4.508034063856635854, -4.435149158677379688, -4.266938116395226466, -4.055662412087924551, -4.012435641665225816, -4.848982332501493353, -4.621832440374783246, -4.617459356130036774, -4.445343657895132417, -4.325146981508327215, -4.460720930069693502, -4.372915104076660953, NA, -4.473955870096774312, -4.940122903843449542, -4.037771160373812229, -4.284657712800404639, -4.377365709517894388, -4.082199751726355963, -4.907676819934921220, -4.872582750473601010, -4.667978862854386790, -4.614820606941469450, -4.356611633913632708, -4.472088684395314928, -4.433371152162441042, -4.401268778127762182, -4.906524180532522372, -4.697735298577243590, -4.216302978537150636, -4.442523013613022265, -4.345680088726489032, -4.229987748028161576, -3.959927904026529610, -3.970153120887448761, -4.685011417018335678, -4.559659149981373893, -4.537448387983133102, -4.366770936094847677, -4.284291001839182655, -4.457720514334481798, -4.348178802978900137, NA, -4.394667162927067494, -4.874550899864932951, -3.997663557938900603, -4.175196434512126409, -4.351003457051779044, -3.989549938112672667, -4.868608442782884183, -4.764457087009825287, -4.540973375384850463, -4.553689099828273079, -4.269837014085683613, -4.406252987432853097, -4.376680002030754579, -4.338043047026362231, -4.810671957162600876, -4.617491176431300026, -4.136642358376573902, -4.358839202307574823, -4.239932480652110591, -4.195979133515273851, -3.877912930173001538, -3.955087844248239115, -4.593961155830895926, -4.519054129727060776, -4.504099175804336497, -4.281689497389435495, -4.285769644032175307, -4.426560254327657873, -4.347684862609897394, NA, -4.317939974468247399, -4.831086146414507176, -3.967953133628602469, -4.124827568806650469, -4.299336276088993714, -3.930933015690112331, -4.848045995330351410, -4.679846818530884356, -4.451513757487513878, -4.512020299495649311, -4.198405645217735405, -4.366113392601780774, -4.341700049946207507, -4.296513485737046167, -4.759778177266428045, -4.552005510977695124, -4.089555344047733243, -4.303138012533809231, -4.154955186148384882, -4.171384430974808488, -3.887890323652748226, -3.906348345633438690, -4.483487846401283505, -4.421169621823239027, -4.435436380710747528, -4.179597484169377708, -4.241006049355928731, -4.341579088373372919, -4.302229001844462886, NA, -4.203652199090519126, -4.763192818925682026, -3.901105226463472686, -4.022376701156926693, -4.202260680146436833, -3.835597328888502133, -4.785520301895468620, -4.543548697159010885, -4.330271791463941256, -4.441404100041138392, -4.098008728254650457, -4.272596252624073720, -4.269798591780184616, -4.218397866273220664, -4.652000839364658802, -4.445547278938810187, -4.006788862649314886, -4.203127731218630814, -4.043091689221966156, -4.100413409990393276, -3.825933570214712365, -3.857308280501877018, -4.373710544770540665, -4.334793817761418921, -4.359706083271142063, -4.074897091130446292, -4.215569351282110411, -4.257232722865026631, -4.205071098196823343, NA, -4.110296275504980557, -4.697741136159029551, -3.823313612738135081, -3.960584212314889996, -4.149993654069136895, -3.731397436998917350, -4.723456489532561697, -4.409961675598896846, -4.172628709728492424, -4.372741422671952982, -4.019055961950835254, -4.216568584715971291, -4.234507342559196807, -4.133188378847969524, -4.550802101925172316, -4.376198131144857228, -3.922104995049685794, -4.177347176960748421, -3.923797912442498426, -4.042408875820951764, -3.747672888647158640, -3.837641288444058496, -4.288248212647432389, -4.284942792488147134, -4.310521829458951792, -3.980774157256928181, -4.203436350141097400, -4.228266064263246804, -4.126776838489224275, NA, -4.067213671929191143, -4.673981444211286806, -3.775283195169714112, -3.896910841784490209, -4.113649980879343637, -3.645362462278501248, -4.665744637373648196, -4.305162946569327964, -4.070665236389634956, -4.331154882059221833, -3.983195206176783465, -4.211564590027509603, -4.219094160065647081, -4.101586759213387445, -4.485421204309481880, -4.325737141921230489, -3.881742131038937949, -4.186677318078831256, -3.841841066258476545, -4.004498832110023443, -3.699558880151439411, -3.834457814902559658, -4.215211010350179421, -4.288344378968389492, -4.304852783728222221, -3.913058013993319761, -4.192335772883726719, -4.219840386149620315, -4.132850150778963005, NA, -4.072734282638004188, -4.653390262311821957, -3.760408010369418186, -3.866805566620545420, -4.126976906596759420, -3.608448462187494599, -4.646108088471329722, -4.247089101933241295, -3.986289483026706826, -4.314431903695196802, -4.010181489672580035, -4.259892586319918628, -4.198201447405002895, -4.056165820422562795, -4.470127821364396681, -4.325791600163843675, -3.850898093494727448, -4.216408248112748502, -3.796389475849534900, -3.985959470645929592, -3.669957771922636613, -3.775538510914893564, -4.104911873270493672, -4.220841650501725439, -4.282874905514435504, -3.800813014212415020, -4.121678220282156246, -4.174958487919780303, -4.096456158569195871, NA, -4.027140302715335629, -4.609839210566329726, -3.730855956300303866, -3.792299471392679866, -4.068594117994035919, -3.512400718051896575, -4.593503641478257293, -4.159058750191976905, -3.880872656528996600, -4.275432105923844261, -3.964006126976247657, -4.258525861967446424, -4.122529334575472326, -3.994126016661170731, -4.390245421471611564, -4.271261769970331024, -3.769428358125841427, -4.234915969043356654, -3.716661487374079087, -3.918288276160950545, -3.584464206752473903, -3.791103776608974130, -4.089541510920171596, -4.191411539798864716, -4.282889963597941829, -3.786252124078135051, -4.070369367002098393, -4.161178972650425578, -4.092913688790512872, NA, -4.027164564070028874, -4.576509458592049384, -3.786782631310762870, -3.811169063260797962, -4.029895732942707198, -3.476966668346041622, -4.556907350776405075, -4.113510701161780858, -3.849807474509160610, -4.251186538673688453, -4.016115864585130701, -4.275020576746864975, -4.083051695833190387, -3.941545709770118755, -4.368904320182493528, -4.295405673766256704, -3.732069947285626910, -4.284991801553868385, -3.727691911056080443, -3.919140409614111142, -3.585956679194303831, -3.750096571676079460, -4.020196637799609007, -4.151936008158070734, -4.269137030957010026, -3.762096858467690375, -3.959837250781432783, -4.125928812181232530, -4.067303432824200371,
                 NA, -4.004682730752217701, -4.530013948471257379, -3.787326729206695219, -3.774149972787653073, -3.919210090826779957, -3.400835619448910219, -4.490538807789873132, -4.054325898352505675, -3.830568513872900205, -4.184408653497675878, -3.997814033980184423, -4.264828285333015501, -3.993401088570710744, -3.845224618806849293, -4.310843293948813937, -4.260072062853905273, -3.677876181885774365, -4.292219037671961779, -3.723295240627426672, -3.900804010297445945, -3.506195926428396614, -3.806053742137973206, -4.034517643896515793, -4.119510107232835061, -4.264112592550366010, -3.765045777750143152, -3.955186233902014070, -4.138467161276403239, -4.070888560008200763, NA, -4.000972880346049365, -4.522215220123080748, -3.773523253851243986, -3.747355165987438674, -3.919350121620830407, -3.438323379594360407, -4.427296315338640653, -4.013681248889256814, -3.863433058226953598, -4.156293289397244628, -4.036320470099653335, -4.271052043605506832, -3.999959107147053494, -3.838136915836858343, -4.305756648157445632, -4.309870126059616169, -3.703049360445834992, -4.300663717036039557, -3.810624239585413076, -3.904925861488589334, -3.484326557291239190, -3.878265308189358773, -4.059736688360555235, -4.105594658059890101, -4.272553463750178970, -3.775951619694124961, -3.969018772383543059, -4.150528420912404926, -4.085061237454484129, NA, -4.013364400834666057, -4.521061204173552284, -3.762239548630754715, -3.711500889206322373, -3.943832933761099113, -3.495993080769135730, -4.351659462587939231, -3.977358784541386250, -3.930487807694731384, -4.127050713430341133, -4.075705429401430102, -4.315976886230902920, -4.031865994431381495, -3.834077740539234558, -4.308150736893384902, -4.363253352338656299, -3.739566137499955367, -4.313232210460204108, -3.934369700696826122, -3.914868420600835730, -3.461187706952146836, -3.932529194387193172, -4.010864574504868330, -4.058491167018901002, -4.258363400799323628, -3.748673967173626664, -3.930657332986355446, -4.134468835452419277, -4.017036645884245694, NA, -3.957826030449643095, -4.499507097769265940, -3.767350574177646294, -3.655418815837444324, -3.933985789120495458, -3.488126887263314213, -4.290738358449273093, -3.938266426937086173, -3.919103583539211400, -4.061141925266259278, -4.060317135550159051, -4.278163861579731275, -4.025876673873368006, -3.830429829992722635, -4.215183790206617154, -4.383674085737224502, -3.700100933836575567, -4.298682387054438614, -3.945470348540657479, -3.915236569328365945, -3.406849725405775242, -3.960363984892955269, -3.906875142613285057, -3.962367270041406098, -4.202634958421660016, -3.675402667561362691, -3.843038259782858734, -4.082170812085109901, -3.906398034543932862, NA, -3.860591842336327240, -4.434292611937959094, -3.713500696854925565, -3.531890180111083133, -3.866253860715397561, -3.448380411470288287, -4.172286272688523034, -3.875254118924582603, -3.888910985285667810, -3.949767353460808650, -4.022761781864521602, -4.213162477250603999, -3.961493661715235426, -3.799736471578822883, -4.097181485932284595, -4.366615658202233519, -3.642667946600803841, -4.245967770368065608, -3.929507039397147850, -3.877348893805299124, -3.298137481839571450, -3.901972181576523369, -3.725085088763061325, -3.790623178598383713, -4.084552481864700546, -3.542926539605071756, -3.729588688417517783, -3.960380255865629362, -3.720033625326478521, NA, -3.748015253480140530, -4.349058929075662050, -3.592998663670861959, -3.334619236812501164, -3.765889977947535883, -3.383877865402266316, -3.984315932795255311, -3.723230941097695812, -3.831360135287533719, -3.788317503810526787, -3.921486820974733778, -4.100132584504171618, -3.877316849092025564, -3.721585251112603743, -3.933716229216126514, -4.306849392298612855, -3.546419937144340562, -4.133978508356190851, -3.862612276216511820, -3.797668313662737827, -3.087347059829563367, -3.904124733936074509, -3.709369732204971903, -3.747848083158122456, -4.022557165085030917, -3.518818113079883680, -3.688135445809943125, -3.937376016030440962, -3.697162499113134437, NA, -3.702271358461675987, -4.303957721106678846, -3.577625539325115511, -3.317601864742212303, -3.724330743879899153, -3.351902230008182215, -3.953645043061271735, -3.660355776937613381, -3.796599889970840191, -3.743907711305552333, -3.911766688165867922, -4.014482620581896199, -3.811174947490538756, -3.724586725702116574, -3.882114215007952396, -4.301009883474902651, -3.476969879167933986, -4.084376993616106510, -3.838177829408373931, -3.786744544534380363, -3.054329256413686799, -3.879912648178430956, -3.662719826580480564, -3.660259948049071621, -3.914735812197210230, -3.486173770685852880, -3.620949905984297335, -3.902505739931681816, -3.677832476911737203, NA, -3.639634977995145348, -4.215516597669307863, -3.573365753644998755, -3.304024382211255606, -3.639776254505520114, -3.277735033501388617, -3.970519369832456125, -3.580770672559583367, -3.725865081612369067, -3.682759425167764888, -3.880544453951070505, -3.892864870338283545, -3.690305938900252158, -3.701200960377213800, -3.814457304723079201, -4.257075378088497786, -3.380374366548016596, -3.991849697938702146, -3.806314538845509521, -3.750030326860172014, -3.045811421439064137, -3.889283319465905997, -3.638505282329302037, -3.590116414378878851, -3.828296339700395734, -3.452308553861941132, -3.543927294222468660, -3.865557984608043451, -3.644333959363861286, NA, -3.606227175848824906, -4.150115876370486845, -3.570433784890786555, -3.301272423402934031, -3.594317594690422446, -3.232217804975817899, -3.975296322277644734, -3.513266601483783003, -3.651966526029598370, -3.631071759483052830, -3.857857160475524516, -3.825062824496145097, -3.600791776034912850, -3.701136034672721564, -3.760313938406984136, -4.247888915408171151, -3.316716475218967730, -3.907461555200394177, -3.811239506049438841, -3.730051733529923474, -3.043984499572927049, -3.915035526580560532, -3.621757247553370807, -3.585749587040523156, -3.772744195733451988, -3.447437830874923836, -3.513843444670252669, -3.856707596587413800, -3.633854908537977657, NA, -3.567977620013879303, -4.133828136677449194, -3.584822091182361614, -3.332610205821606275, -3.559015793979023901, -3.221456524308451197, -4.014187528629815205, -3.474668287982519033, -3.623400568252371112, -3.619894798164586813, -3.853693819503587026, -3.808100392446679638, -3.552492736936741391, -3.697035428690434333, -3.725108009151141708, -4.266962958142704210, -3.299287897543619508, -3.844140358847303318, -3.816778679193494117, -3.719802695419351934, -3.052565787371591899, -3.941077276316905742, -3.607345274598453777, -3.593228205500942352, -3.715440321280950364, -3.448748022068430341, -3.490906701871470563, -3.837539702997038393, -3.597121676747105479, NA, -3.525600818177877738, -4.112922510771766405, -3.598489327293197704, -3.347308396161345456, -3.532611980893153980, -3.218174631196595303, -4.032093020780417980, -3.433942360114575099, -3.611003778383707008, -3.606996719323438683, -3.850603051309684943, -3.798194750092791949, -3.497758834561632391, -3.695553768809170858, -3.690984491879774065, -4.284736733054191227, -3.278151469211091129,
                 -3.782113336389874014, -3.830441065551101865, -3.698981871946333211, -3.056504506002161836, -3.945915287661835968, -3.598197185010829635, -3.597558695851004185, -3.683493248734609260, -3.455513099320215886, -3.476456778826258365, -3.844194899674914900, -3.571885065280600458, NA, -3.492004489856662452, -4.077956097559389370, -3.608901333452608107, -3.318368755117547142, -3.528994823166463846, -3.205151310088862449, -4.024412233508505743, -3.417137047953901163, -3.572060980455165513, -3.572994603469438868, -3.838992657240754003, -3.790071536700486909, -3.464307473149682792, -3.683869828478884223, -3.637786323336579031, -4.282404331972058387, -3.262724905064905645, -3.753997779448243310, -3.842790572214392242, -3.678754768807984021, -3.053307910341857223, -4.001002253511965812, -3.617305800174882791, -3.661444942609961561, -3.675107396663356507, -3.476388064786865684, -3.502675421976114034, -3.854613723190048802, -3.608131797612795122, NA, -3.556444455161475560, -4.088252714998866111, -3.670273833970619570, -3.375652261003357246, -3.542399719453285023, -3.243223742081086325, -4.093932064563722051, -3.457588727210830548, -3.646118087556486920, -3.624945917760807212, -3.848452868197937526, -3.811665087732237556, -3.502180834926496455, -3.721024231246296665, -3.672177926860291208, -4.300429996323331672, -3.338312488378939680, -3.794099572500635453, -3.871217696883199277, -3.699876173677931757, -3.117641613911190213, -4.029258834992459626, -3.647996871517099393, -3.734940325426525654, -3.678799810364427270, -3.520142701021503395, -3.539114567020594571, -3.898939921436766376, -3.644705362189329279, NA, -3.619287529600337727, -4.131211648945710913, -3.715814411158322450, -3.394694998237287820, -3.598253461504501427, -3.299495936481140035, -4.142000003698028010, -3.515619581385589942, -3.716022876995643376, -3.686628206240480488, -3.870614810439373166, -3.870659216242194223, -3.552988954487301676, -3.732681609354140839, -3.709575511979958939, -4.315685717872705496, -3.394492946007046541, -3.828569024642732632, -3.893885073480182424, -3.754550763533146984, -3.218308621833333660, -4.013403996254142569, -3.670495871992033976, -3.801090841190729819, -3.672084512788809096, -3.567340232708307024, -3.565568994079189036, -3.917957371477317263, -3.654422336478355238, NA, -3.656215030636007057, -4.182167925315464885, -3.722545704293872326, -3.365194485270758484, -3.674302153044089359, -3.336992540213580938, -4.175132891650084055, -3.543971209240851294, -3.782551449296603252, -3.746420261250857031, -3.890841294023246189, -3.916586035518292430, -3.568371440949546169, -3.730537615465033596, -3.748155116254612285, -4.307879217778106451, -3.433368785216737606, -3.870571556153954429, -3.913288468441349366, -3.772125825367553364, -3.310680761586521470, -3.969366298776712654, -3.627007467553946540, -3.814493077921665698, -3.603682877615379088, -3.549838897629400414, -3.532278011648688487, -3.911452598812121373, -3.615543206612954030, NA, -3.637450542815188648, -4.203486666168613439, -3.697494841964198642, -3.303640674981099146, -3.704583757368088648, -3.315381216365982375, -4.174861016575058592, -3.510849041397643422, -3.798928876989323467, -3.745932307827731123, -3.869394875746396512, -3.913800558360143178, -3.553455924755485107, -3.682385116036381412, -3.733459474876949180, -4.268128349120815734, -3.417135134501664062, -3.852837800166903914, -3.895107876961122706, -3.760451949627305446, -3.345592396491724685,  1.055766853294694219, 1.092746749788775951, 0.898475838346793121, 1.118169389838209238, 1.058721581174988380,  1.157063676763250726, 0.850018861713837115, 0.763119571206350877, NA, 1.355982087333043484,  1.298049923105032422, 1.383418309387927536, 1.177634648217851909, 1.112364512546645923, 1.010274195822145726,  1.281561179318754240, 1.396193492182971507, 1.315404056075728079, 0.956846220847364592, 1.398248036310460130,  1.118684467986390763, 1.249160002863307861, 1.272873000867790649, 1.383265559297313541, 1.207742429316000399,  1.278998175159148820, 1.083635565070466544, 1.215172280971973162, 1.259339064245692308, 1.212038221255148818,  1.055766656054725638, 1.092748201636295491, 0.898478166934181588, 1.118169390083540549, 1.058722049636096951,  1.157064061754901729, 0.850022628744309894, 0.763119299539753326, NA, 1.355981631489269423,  1.298049575030880032, 1.383419268707560601, 1.177635906177380232, 1.112364795685870433, 1.010273960149009387,  1.281560650456750050, 1.396192976456575696, 1.315404906885828895, 0.956850218978247180, 1.398247722369176937,  1.118686202314510325, 1.249159711845409415, 1.272872771684790294, 1.383265189851683896, 1.207742586390931017,  1.278997931140411382, 1.083636361575350593, 1.215172849015240697, 1.259338835045981542, 1.212038137801688453,  0.710618609996370720, 0.797534840303233628, 0.539966589374866746, 0.900514539139461112, 0.715773154977085047,  0.796804881654316510, 0.597955230489712664, 0.562046849449673980, NA, 1.183027556817232151,  1.112597739186139911, 1.293802823925992218, 0.886155206864240874, 0.920380415071198055, 0.940207580042788482,  1.049846188270157299, 1.208853576976508304, 1.228657542749288512, 0.688034345632559496, 1.262701434161556113,  0.841129305451125076, 0.923431623332048535, 0.935745853106738634, 1.245531714125759803, 1.003315530220594054,  0.985941690151125316, 0.870445637603839306, 0.972714847635719915, 1.101248203940334403, 0.900844804043975156,  0.936385601863886530, 0.934043969148378728, 0.759453217268593050, 1.116831788984546048, 0.990319303564978326,  1.073333289715203698, 0.832781673779027787, 0.814334609978500934, NA, 1.387231946089513057,  1.292420601090774612, 1.523267572744745779, 1.047166252165161682, 1.164648175904829497, 1.244952419666836008,  1.167279669338805892, 1.399674069977493884, 1.453518804765112638, 0.834190679078090747, 1.461010600049699404,  1.031917836554139711, 1.136183878040272877, 1.070410313865049590, 1.388727259076741438, 1.200116886027802421,  1.199748628039468024, 1.124493738857536229, 1.156576046304110594, 1.308080626395417889, 0.952686511472403019,  1.183283410467937946, 1.267727543536047108, 1.019055906605000672, 1.134979786210511543, 1.226295772190481248,  1.239575919625559797, 1.096313514170433212, 1.153957082232161424, NA, 1.297847341536936483,  1.360296846470019139, 1.418863627039627318, 1.094368646234807896, 1.201001894574363016, 1.417991841154486288,  1.264922130150842872, 1.374823647488061962, 1.391531929320310113, 1.169504989373057358, 1.553770006755150757,  1.327402220848812542, 1.294789795414016398, 1.107935283529128734, 1.467390814587164805, 1.398926093676510218,  1.111466769638073782, 1.358378707005428065, 1.378970987007599902, 1.530060106405113363, 1.138085666639886417,  1.178265896068660501, 1.235237554775750146, 1.035803085350970099, 1.102511439185931641, 1.184675415521982300,  1.239121080572844980, 1.069557610447057883, 1.134893362996490795, NA, 1.186019126646147503,  1.258589454187025058, 1.226751868241596766, 1.092476185507767594, 1.122376956013021010,
                 1.274839609974384169,  1.197524891616407849, 1.265881888943277689, 1.238382129360981132, 1.070084849234100854, 1.403097610835293496,  1.193227387976510956, 1.269563637608265960, 1.106685355706271467, 1.310208052682571900, 1.213975140057953039,  1.061792651139833676, 1.312839067591488496, 1.199868792949860152, 1.363794979513576866, 1.081335288379608794,  1.207072900005992766, 1.250425575068112360, 1.067711609162747211, 1.142007293245391519, 1.211052352230176821,  1.264661745646386448, 1.070810700683067607, 1.133828934178417125, NA, 1.152423976005658268,  1.195453147792616244, 1.084075883610001290, 1.136070654166326355, 1.101728117004439511, 1.216611824718506485,  1.129365490591236254, 1.230828978825306841, 1.162449336523143417, 0.951926147398163836, 1.306999726705796228,  1.059615887465295669, 1.318447489714046128, 1.110414911800862825, 1.200386893530817511, 1.069688217792337248,  0.991678578533453869, 1.367054154248784181, 1.116792606369545027, 1.252584296763326499, 1.014225713647383120,  1.164809823135410927, 1.287024917294591564, 1.078498434737016565, 1.189856903976708624, 1.195997821950869344,  1.270949409545411024, 1.130710694839171415, 1.166166480520052007, NA, 1.060564864474576208,  1.134609946551978110, 0.848567063906777941, 1.112200388914917015, 1.099807875111444844, 1.310764826857093723,  1.066581224565035679, 1.063109257059754897, 0.975904053543894978, 0.885136118882377865, 1.193311256343159599,  1.063826816605775383, 1.285307228079158426, 1.180444603370367940, 1.170603794172115464, 0.976460841347721820,  0.981563291864742005, 1.331512589772116728, 1.151864802254128017, 1.182799805652923419, 1.121411900406063422,  1.200626017412988578, 1.449136427258984838, 1.246716983641884546, 1.462075108825605207, 1.595885443907651746,  1.326870359025004609, 1.381599358956727253, 1.497444898826878035, NA, 1.462588172383855856,  1.252017403956096731, 1.126331848298596672, 1.375528644937003664, 1.370815228796014917, 1.618548403725140750,  1.249281487931711565, 1.301151842350675825, 1.238187182550262389, 1.108969828653288081, 1.334683358052191737,  1.259833678765645226, 1.493410693334335626, 1.583636969322120391, 1.410473078459480778, 1.273031588591473362,  1.160641322573550127, 1.513229749980034455, 1.550143311882740349, 1.288058899811258762, 1.353032017213221305,  1.331099515000489619, 1.490796023416925742, 1.290034433268374414, 1.487166201710518720, 1.510587037063411575,  1.422500490744371016, 1.452112917680037585, 1.439656202391272455, NA, 1.500369578341204102,  1.332795500681423384, 1.225609788366981867, 1.444663507950736836, 1.462786447603107609, 1.620862348994469571,  1.307661687314271903, 1.367483235163306610, 1.489305463288652653, 1.203316817898201219, 1.415077621437448752,  1.454863013430466978, 1.454520958529364805, 1.579563628167839351, 1.522286437632164757, 1.339411456272351142,  1.367301147557257179, 1.626895636487312169, 1.508752472551290191, 1.351290976853536607, 1.471726760730947658,  1.664037202796484483, 1.665092925729386497, 1.456616074718694920, 1.677162533478576023, 1.514962555227253205,  1.479389836177407380, 1.639260967823917836, 1.435966102996853122, NA, 1.674822251946539220,  1.595399694344231900, 1.454844312769960624, 1.634312549758032418, 1.616573197137783691, 1.712370520177535882,  1.444642425069736724, 1.626256857900143959, 1.605812805733042081, 1.355510450885417217, 1.632052221007492809,  1.614759681223517651, 1.544138166514285926, 1.761882288054779755, 1.730662649269167241, 1.490252879119149609,  1.563366300748048054, 1.862100102149449654, 1.609628047587350563, 1.422121975567295582, 1.541735281136156654,  1.733817088242877835, 1.765440786624415725, 1.541785770896896057, 1.731038546600138783, 1.508652335305507464,  1.500443608392727279, 1.645234655397335155, 1.473882994647455646, NA, 1.798176375099762581,  1.625076983088540317, 1.549906856437125224, 1.631963986541492106, 1.684193298919647086, 1.879215657721944321,  1.586563016173270002, 1.744904009465593875, 1.770565667366420293, 1.529278908849782459, 1.792147519203886707,  1.780978204356857164, 1.614391322073404300, 1.683810540882277218, 1.825079514219539734, 1.631912524776041318,  1.731515410996472548, 1.979218037261214480, 1.673224151756274924, 1.549512936419564557, 1.573167813127509307,  1.799933966889096348, 1.716911842999728766, 1.524366564635546872, 1.664836469549558773, 1.322182189907721739,  1.386968070758755056, 1.655507699921369591, 1.380158087717372695, NA, 1.773754751791617679,  1.457603108884169441, 1.543918302709588852, 1.626116606755946714, 1.602512595456694777, 1.685156779136276484,  1.520957165089389385, 1.558592095639369512, 1.718488532103291222, 1.515794556664953108, 1.582799791285485425,  1.726809764379684031, 1.624962298355316470, 1.662589285314287846, 1.812777591505547736, 1.609637350434866221,  1.663206984446626180, 1.967028276881325866, 1.549947107267464208, 1.647925782182583676, 1.534755154771882291,  1.719035390549158215, 1.509284773654717915, 1.474186671691775974, 1.410687488832141678, 1.225942812568932005,  1.358503945902550747, 1.594278885205806073, 1.220134539452238265, NA, 1.674264951014059255,  1.267864930454008832, 1.479822373370866195, 1.442710329727950436, 1.542603481447106128, 1.414681708860489939,  1.354089731293131704, 1.392690059267840175, 1.529683836661253515, 1.347644796375598419, 1.430218423419107543,  1.591622254898884314, 1.497372149529989205, 1.601126785922910578, 1.729969995024457585, 1.419628785067676979,  1.478324207124801504, 1.760085418929893786, 1.431536911324257311, 1.609687892919436925, 1.481946906478268211,  1.774385519718318704, 1.498649616552213626, 1.568720685170418516, 1.481130849539834315, 1.350261202303648655,  1.575179571528225297, 1.730863202394972600, 1.335100995010537250, NA, 1.699278044749806815,  1.407865594281031862, 1.709298629991943486, 1.709102084146699063, 1.619367276423512747, 1.383757482851550424,  1.361832151196747498, 1.516111578429230722, 1.633612091592682614, 1.439651055388182499, 1.433110649687709737,  1.716026674943686547, 1.631620430642141795, 1.724577496165344614, 1.887384558277064528, 1.539200742674840638,  1.598852753674172344, 1.773325328137411727, 1.482928836672939399, 1.687372835851705499, 1.693463170470083767,  1.735924745535664560, 1.510024541521074237, 1.475784407658617514, 1.580613618413481136, 1.370126145899462244,  1.639422548071564067, 1.690928548110352203, 1.411403668553186108, NA, 1.656458843653269009,  1.532243459745799896, 1.813384571800940703, 1.808134144282389943, 1.679464323158625039, 1.341011884202030213,  1.312441579561334448, 1.431127198738458173, 1.714870560146918077, 1.546761569082174326, 1.443443162230854115,  1.720167124677791426, 1.739756839667454980, 1.674827703610071605, 1.783922320260352246, 1.579538455498268990,  1.694831507571660900, 1.760433802628854849, 1.517291968667914492, 1.681316919111875530, 1.575731989422971102,  1.528212024066830654,
                 1.323546180910876835, 1.219762338380623090, 1.339533351720727072, 1.183317232578744882,  1.467236382938118977, 1.343834709365182789, 1.112490994180625581, NA, 1.327563439074256557,  1.339677917605245128, 1.501865273316751015, 1.554387411762937710, 1.422889521109577338, 1.064491070835254627,  1.176155817910342849, 1.118305983604465315, 1.393563053993670753, 1.346244845625776287, 1.149697604947247598,  1.443434192618304746, 1.554678530866881347, 1.455118837979196167, 1.493660239830817860, 1.343707869658692600,  1.476448606117198636, 1.542147335353537985, 1.273226485861291835, 1.533352280155700909, 1.308029791130671748,  1.420485257403682366, 1.216859314506502088, 1.063386846263609131, 1.192897024576119680, 1.074064777153423078,  1.393440208378009970, 1.268211247278941967, 0.881427373574242390, NA, 1.220250809818456572,  1.342041515747763869, 1.288123056366028418, 1.326659162294697047, 1.244002219787039509, 0.878111169843311168,  1.095325893717153942, 1.021587913885286580, 1.294167285333839956, 1.248616507768661066, 1.057464912716806760,  1.298052013746813005, 1.477607725874099165, 1.466450433548508014, 1.311725268886303475, 1.113786597342711238,  1.474131414024450581, 1.504548449378860120, 1.139733575123395015, 1.454087820576795087, 1.164882130229820456,  1.303223692962441227, 1.101674263898658301, 1.058854718900709369, 1.143895411378790916, 1.024204860443943321,  1.308127825430161817, 1.146575150695867462, 0.908598429135139107, NA, 1.241618355555312858,  1.199755316603128286, 1.177263124470333544, 1.239386650403186962, 1.195052411232628842, 0.867243526314424584,  1.133983612686365827, 1.005567151505759371, 1.169673422278231012, 1.188498891691194048, 1.082583018808561670,  1.274121556004033096, 1.342711029530746503, 1.331263001105092769, 1.248143767009804561, 1.052174013280962939,  1.430973733685157123, 1.440709511976548285, 1.085007883382385963, 1.377919928537391758, 1.065566961466162477,  1.181167802889738550, 0.946501033630344990, 1.128041030256188737, 1.159280000554360424, 1.011493259801321987,  1.276798033467658744, 1.115194420293595945, 0.924183379173123321, NA, 1.317885602160098202,  1.149401895433569409, 1.053964002190835858, 1.220566455682971174, 1.167140775107597062, 0.793846275640201382,  1.209722701261854771, 0.985798379195690022, 1.074042761452902894, 1.122542771964974540, 1.120741808925549865,  1.285182153588893872, 1.289489102384204999, 1.172061746048066455, 1.244261277065195959, 1.002768050934256472,  1.356122916350124497, 1.485178967691538121, 1.018018249940259690, 1.312581331651370808, 1.000677774919223051,  1.038748140598126923, 0.859583996454848398, 1.091846778941117924, 1.156472199683073354, 1.021076238081326082,  1.069106028283157706, 0.981094646704926854, 0.962040261086566595, NA, 1.292772602989924691,  1.038880563404829038, 1.046688866496476633, 1.164556814939738949, 0.981426605147511566, 0.763660876829135749,  1.137276651069096589, 0.988767448485117773, 1.037685683831562233, 1.028399549318356865, 1.069150700402690113,  1.262379160417999691, 1.122003951770427888, 0.968269505697203159, 1.146458392666972470, 0.879134065344794546,  1.237017383386002756, 1.400913213213941022, 0.947321339666773454, 1.183305796600315762, 0.948660710817802899,  0.851299503947806291, 0.753154950462086137, 1.076421510129419445, 1.179571413688762282, 1.027459635624550671,  0.853416383337568751, 0.863256276519855703, 0.995629982864913443, NA, 1.289936437396959734,  0.944740718541894786, 1.038306018593231128, 1.114041888991518015, 0.762871043706958574, 0.672452535539696861,  1.065414801100889708, 0.984648289352171968, 1.026088595644857326, 0.931568078574880887, 0.970068299558391578,  1.307102684922356595, 0.952298316095404607, 0.755240132808563724, 1.046207530081175285, 0.733450269114555464,  1.118998477250494394, 1.357181204654193341, 0.890108579487275398, 1.067670216039533715, 0.860139698969060729,  0.949496560576515991, 0.949015572636080718, 1.111657220283980951, 1.240212080207335976, 1.141845028939303841,  0.976733569155239900, 0.929215050136968146, 1.170322790448971695, NA, 1.380059014463030476,  1.022684561064354014, 1.016804972394345752, 1.162837251080713008, 0.873406157642171888, 0.908606809326781173,  1.049280090998790049, 1.037221638190820583, 1.194934279579474179, 1.007376551922269048, 1.037171830271923056,  1.475540416241901021, 1.078205268558911190, 0.813866284505479376, 1.162023742289583161, 0.860405349893716198,  1.256998408618878127, 1.381832848018006654, 1.161565691298255931, 1.073432370552757753, 0.958664687427501438,  1.091888992518174284, 1.256283526960488395, 1.262569028781480585, 1.404378288257628338, 1.379177197706826119,  1.227807880349560898, 1.057158688911991495, 1.478465314353040849, NA, 1.603770482672234987,  1.187094291423543391, 1.069769574722108540, 1.262746886516958877, 1.157253014848425909, 1.289065972207374733,  1.015255271220201649, 1.129659178248436202, 1.543634031548349883, 1.114526380146838802, 1.128811389605216764,  1.853813307237298869, 1.379110245912437804, 0.906820070462200256, 1.367683570322748121, 1.042144888571408190,  1.474868719728597677, 1.486586904254415531, 1.604952784567410973, 1.164112769219826093, 1.087041337476662983,  1.153644321636733583, 1.180334231808367385, 1.221714326326373889, 1.416384648359950260, 1.317172437751848113,  1.204506679476243791, 1.043270621316432889, 1.271364220513870880, NA, 1.513008439677036021,  1.209351810026258800, 1.120787563535459164, 1.199089327542643524, 1.209826726609227299, 1.248050638420429959,  1.011861071789144528, 1.142592448879070632, 1.432422087085255535, 1.026384954189812460, 1.056561096451415827,  1.741707848523554869, 1.466615262190632540, 0.952654470447648061, 1.181532667153232685, 1.047080543679940634,  1.362682694413913653, 1.468567118165988061, 1.474458715876266979, 1.162370301809263262, 1.022089282291990475,  1.119792768516298276, 0.915256183091371245, 1.041337307617950403, 1.285317160168298001, 1.077875771732930366,  0.965813012679173077, 0.897669140849270075, 0.923320341146739665, NA, 1.242369518034201370,  1.042987455204113179, 0.888404789651101368, 0.883329298194693635, 1.067799373272248387, 1.098005659930815892,  0.719564901880431762, 0.998521243326038199, 1.221159510213305044, 0.775976213662832315, 0.922556706756382239,  1.559919983771814955, 1.349726655431649824, 0.853604222310302641, 0.928611842058099435, 0.889149375714586188,  1.191860707202177849, 1.299482805238993510, 1.187337530182158485, 0.987047187177248375, 0.749019820638219480,  0.939803858144574678, 0.528546200437186697, 0.772866407965527302, 1.073930671145644000, 0.707272130411329081,  0.691400596604120787, 0.603759950776516008, 0.420930864118333914, NA, 0.954719039947612758,  0.877465262752397224, 0.503297884469082857, 0.420056891606683913, 0.914581541626819372, 0.940244787211612976,  0.265775702274971659, 0.705030058059415987, 1.014512275602594382, 0.444701958483756821,
                 0.696729516963266682,  1.354487293094640110, 1.240085044168242057, 0.659404029434510264, 0.623075782326849348, 0.665942170014258150,  0.985404283101467549, 1.089020974133643405, 0.849223618425054783, 0.732157693492809347, 0.242624731474764016,  0.974525109697824909, 0.662071041778158786, 0.926079868755180780, 1.124782425500472804, 0.776972706818683689,  0.842936060357098449, 0.724567235753396099, 0.511164438604181037, NA, 0.932679886295640914,  0.930559697304714994, 0.560199914359022166, 0.579868630211845759, 1.006848641667964728, 1.030115708651973883,  0.445196573435629517, 0.856372171740110044, 1.076144627270402720, 0.555290626901075446, 0.815421878905730813,  1.298933351852715967, 1.285632915021191813, 0.839359655668150073, 0.697111627872111006, 0.789667748442719031,  0.998052600920760979, 1.152696166687659396, 0.855694569600980781, 0.836421466401123914, 0.333849395109488079,  0.898310199995706826, 0.673717877335486937, 0.901354669394298758, 1.006238381217748223, 0.808730485991750681,  0.854266415489932540, 0.797212707654919805, 0.610063262074398760, NA, 0.873354540304455695,  0.810143164640152680, 0.651254359124684434, 0.733854539899006619, 0.894774230935609816, 0.954230948657168732,  0.587481769742517645, 0.907947001313487378, 0.986095377485264546, 0.606398099475562780, 0.844552322633995267,  1.101938748602430795, 1.094001150755143481, 0.874518399562266069, 0.725110683011723878, 0.740887343872203119,  0.926838697076689533, 1.027880132561620030, 0.832617756445860224, 0.829691195993414898, 0.477528680475460543,  0.948070607929619058, 0.732449029021470799, 0.913284990358493642, 0.958328464221246401, 0.834546147676434646,  0.885023356326831268, 0.839783131201410571, 0.664117733920716269, NA, 0.859450099340956508,  0.766297984789621522, 0.731316588325387040, 0.846136378139044076, 0.895744952704006003, 0.948383007721300464,  0.685937126256621843, 0.928482756613907001, 0.970176558234121300, 0.658721001374729775, 0.854711008258244309,  1.078458341168495815, 1.005895267100416435, 0.940057320791280793, 0.754930511976671159, 0.771577626867959387,  0.920199685726159400, 0.954851773531024017, 0.838446420339353349, 0.823189828841465077, 0.586119572285415225,  1.011591801889366593, 0.790432564300803664, 0.971051011464697522, 0.906130594281362778, 0.848500622739133958,  0.924751919672966105, 0.847847435503489288, 0.649766920637349199, NA, 0.819156781397914258,  0.730437290017997620, 0.798303635684716850, 0.934081002468225230, 0.897032891136454347, 0.951807844004454395,  0.740219963829571004, 0.913424617002436490, 0.995454220953967406, 0.691222877979012384, 0.859599859453127224,  1.076798628896156407, 0.914268550118673362, 0.999321243923586566, 0.775564555757430774, 0.814955955705642721,  0.900461392618984391, 0.878395875252740543, 0.835719918690116081, 0.775107994038439774, 0.649339127120815318,  1.069389460257124114, 0.822569915132477458, 1.033585054035639494, 0.816391449935999103, 0.833843485998288836,  0.937255395976419270, 0.803620582333727684, 0.546249503704746675, NA, 0.742457115660595468,  0.674710048100384929, 0.840044201189238393, 0.984586324186034090, 0.883738358511939159, 0.961086270950997967,  0.726374845101454447, 0.836269268754698203, 1.041421792603302032, 0.674764424242762950, 0.842101889357162015,  1.067488043015811572, 0.776782013616571176, 1.036729345590677864, 0.773622234236146289, 0.860344236839498455,  0.841156750039579904, 0.765784365792052446, 0.810991396772781314, 0.665321421456139594, 0.650671610835624969,  1.013977925144233572, 0.822068944865349427, 0.995223406146353518, 0.774921879667599511, 0.764868264885490312,  0.894262574557504797, 0.845855781750442293, 0.461972534575716631, NA, 0.673053805959302975,  0.507243562261159853, 0.816300465192051461, 0.830191634379543464, 0.869341464576427914, 0.868483479120046864,  0.610578039052428245, 0.804034625777307665, 0.896294978407966125, 0.470816129571775210, 0.727133664936740365,  0.994628535739574660, 0.672919759615831614, 0.946375564140062742, 0.608760393232793628, 0.784066775747596312,  0.748680768038035849, 0.718893432552698219, 0.768755123206201429, 0.536576995910345023, 0.558437473043575161,  1.073761520690490467, 0.857142661104862924, 1.062411544769204408, 0.742763056329802551, 0.653063118577064117,  0.883709463828723285, 0.855272941834630540, 0.502748808031658001, NA, 0.841763409923151684,  0.419990237958787815, 0.895816987416561794, 0.901641059225191333, 0.817665815761249970, 0.859411175947648154,  0.694188687786544634, 0.852671738071890672, 1.023100679788283784, 0.416478666776057715, 0.620667756783213398,  0.952719607245553157, 0.698929699308348917, 0.936129981727504501, 0.619944428127149672, 0.739908854084722734,  0.863236313856391546, 0.794743715567435727, 0.708562308137838270, 0.469862955578352615, 0.572391362492299161,  1.137333191468432370, 0.903020375281703203, 1.178217068169594928, 0.741428151845726302, 0.718642339700332977,  0.920007081178969899, 1.030764397920481246, 0.608003245036107476, NA, 1.000658953642366900,  0.579811065600736364, 0.987775172864674422, 0.905003918309621458, 0.923978890035518696, 0.960436451219840781,  0.796101303293663021, 0.991193297661697725, 1.168518626736280597, 0.497814314345899012, 0.596550181498332943,  1.120369318632415290, 0.822760141204464723, 0.897795488935619113, 0.695938536819561193, 0.773856102207991725,  0.993063357115673617, 0.889964200292827190, 0.659015339632780583, 0.612836113957110795, 0.769361179430798514,  1.040436942433507816, 0.854006301150693758, 1.187018245994956089, 0.589095273294720223, 0.631629871845607704,  0.799327262185624332, 1.034875196927947716, 0.563005040982531790, NA, 1.015057047179275695,  0.710019106056504090, 0.879830774493319367, 0.702581076461076925, 0.946622201132994778, 0.913849600314010080,  0.857617381199666795, 0.931871015715760587, 1.257636932302037547, 0.437392880881510182, 0.460933428995552119,  1.148476794576355875, 0.722403893959474130, 0.755678234184738229, 0.641341024712956731, 0.699466017551650610,  0.958369494140185463, 0.941259216534239718, 0.502732695269977103, 0.563840808709643815, 0.827923589201013277,  0.754256974705460603, 0.548773957484139485, 1.075671114600478484, 0.209728463318596958, 0.284791051461897982,  0.439037144454146100, 0.814850971869963692, 0.244117738536303253, NA, 0.764452375179721533,  0.601023542764633989, 0.584647298196172116, 0.317497304404957825, 0.846497166071687013, 0.603997866867798550,  0.694564366691977408, 0.564412864713239371, 1.205207966815099452, 0.169058464674651021, 0.139421156090502096,  0.994658643751725413, 0.446897431463291528, 0.455094221191189252, 0.363408354916349707, 0.427067370125662427,  0.690251282525917054, 0.726923863795380321, 0.220622688700271463, 0.366215195054604148, 0.746072635662521355, -1.856901367232044020, -1.806026935985030502, -1.786542466796166240, -1.740546662044292647, -1.807278754467173121, -1.777995338404200654,
                 -1.824692734498160274, -2.034534258994030509, NA, -1.727664469103285860, -1.732777752334838439, -1.743210273198977367, -1.804307572569330720, -1.733505293537862757, -1.793317497233932745, -1.786378339742048516, -1.777480540977414147, -1.725253716325601872, -1.832233398234561328, -1.734569109125067676, -1.768866086757280298, -1.739366138801384354, -1.787510132245943240, -1.767778716788760196, -1.758439905827106964, -1.729045596499570570, -1.748598158167564032, -1.883992289217879756, -1.741735683127336953, -1.800268808710526303 ))

  set.seed(66526)
  suppressWarnings(tmp <- ddhazard_boot(fit, do_sample_weights = F, do_stratify_with_event = F, R = 30))

  # plot(fit)
  # get_expect_equal(tmp, file = "tmp.txt")

  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c( 1.00000088778563767, 1.00000174009561826, 1.02304753103683543, 1.01944404102942188, 1.00899290285153165, 1.04701662715267951,  1.06940710837511466, 1.05600873376315985, 1.01812776489103229, 0.97325913801785024, 0.95293336077703195, 0.95773174865017241,  0.96620647639292967, 0.99903704458071629, 1.00520639541201673, 1.03268980353765416, 1.02729242848519764, 1.02732896385795858,  1.01122609703073230, 1.01281913995027395, 1.00011655254244580, 1.00060281713075727, 1.00611081577129924, 0.99649311547138320,  0.99060170200508546, 0.99156157825457181, 1.00447654495826333, 1.00798744657310357, 1.00284571732886429, 1.01458649529204337,  1.02539862726250863, 1.04270331819876461, 1.06262671726664992, 1.07284909478848500, 1.06928074682467189, 1.06656389011133812,  1.05314672682059474, 0.05731815476167940, 0.05731814036688184, 0.07110830574865339, 0.06342398364352142, 0.05966557643891277,  0.05120077981569100, 0.05113808267629658, 0.05342563250785658, 0.05491871745693803, 0.05053739731791734, 0.05053960468479139,  0.04591178859456737, 0.04541290878746800, 0.04457130583886826, 0.04014239820973144, 0.04100879872716157, 0.04366389490036358,  0.04622534461029087, 0.04525593460221539, 0.04273532812784381, 0.04435181219092433, 0.04971280620709549, 0.04542758975016432,  0.04438245229360251, 0.04591085814238426, 0.05441539853097558, 0.06599708463146817, 0.05778742364900404, 0.05449694168285867,  0.04948646253153575, 0.04687109547861610, 0.05121471292200981, 0.05543193089176356, 0.06122190073890196, 0.05994362639165312,  0.06318971087312358, 0.07489102709573074, -4.74736294535845094, -4.74736315803805997, -4.68750150625701156, -4.71299931370267355, -4.73514854857563794, -4.63619812421150534, -4.54381636267413036, -4.50885021885178805, -4.51161030693639109, -4.52376700922874431, -4.56308616885795093, -4.53531377156277227, -4.45561202606929552, -4.38742672896691577, -4.33101566183158493, -4.23388485370226775, -4.15210833427189918, -4.10534690451752393, -4.08815964798977483, -4.02878153162040054, -4.02490303857841969, -3.98551529137580784, -4.00210638850729250, -4.02721453484295111, -4.00633005765977135, -3.94842117088446853, -3.83630078010985542, -3.81036121453092225, -3.76076071244999799, -3.72306923035553261, -3.70735462655223103, -3.69061259048338108, -3.67270231708632133, -3.70961721032744318, -3.75239145093765725, -3.78202023116616459, -3.76982830482328790, 1.14618427108415188, 1.14618435908348304, 0.92338539243408735,  1.11785249660020969, 1.25193236402235542, 1.16801850138528684, 1.12275869305729392, 1.10326747011519588, 1.36793491255470956,  1.43322266354394778, 1.58990124597570071, 1.65474051867337635, 1.60216121785625321, 1.49858846404766788, 1.58718397556157642,  1.58557549766583383, 1.33013476974486178, 1.21560076767575054, 1.14822078253437487, 1.11151116859161125, 1.03505845684244013,  0.95887501070529413, 1.08131390065832989, 1.29409992484946423, 1.23784532899573785, 1.03370000549887830, 0.74439785684607696,  0.83491491819621244, 0.83349068772845691, 0.85053207912729623, 0.85543521655900812, 0.83050546960032023, 0.74455694785065119,  0.75643370566809742, 0.85052411748459433, 0.82168067074154705, 0.60821758374810564, -1.73001250113853233 ))

  expect_equal(unname(c(tmp$t)),
               c( 2.47370421502620763, NA, NA, NA, 2.07918577203342991, NA,  2.16224168204778477, NA, 1.82948533644019440, NA, NA, 3.26620445081869137,  3.39757812460505626, 0.98536930548741430, 10.19309142480878450, -1.68003025505384929, 6.26065664632605756, -0.74354314615985506,  5.74409797920253151, 2.21296324704403391, NA, 0.98606723706909905, 2.38715542978373207, 1.84070427684200144,  8.47413386124463486, NA, 3.82498238067379503, NA, 0.37034489883237670, 1.21420709422568396,  2.47370632426337078, NA, NA, NA, 2.07918857909408317, NA,  2.16224290892047666, NA, 1.82948694132775413, NA, NA, 3.26620296187978898,  3.39758076952819632, 0.98537102272412147, 10.19309722427041365, -1.68002954962847806, 6.26066067238816082, -0.74354102257981936,  5.74410118007989556, 2.21296446398046331, NA, 0.98606764668598323, 2.38715760191452953, 1.84070445963593698,  8.47414016180158569, NA, 3.82498485377020003, NA, 0.37034669405120635, 1.21420925618468356,  2.49320420156030309, NA, NA, NA, 1.95665748453704125, NA,  2.18193638637439236, NA, 1.81672481712121625, NA, NA, 3.31092273294803086,  3.42333915665827071, 1.00092187309954461, 10.29832466202667263, -1.65702993671101906, 6.27784782792535090, -0.70501943986757443,  5.76658202533663378, 2.26077152907679224, NA, 1.01425028301048692, 2.38380439697973445, 1.88095571888578506,  8.50535230056186897, NA, 3.96935661890279956, NA, 0.41410466146260760, 1.24396542602655780,  2.46870331494651607, NA, NA, NA, 2.01122345720834605, NA,  2.17047965693370504, NA, 1.81301154210764071, NA, NA, 3.32039155070017511,  3.38425256016103937, 1.00314181003204528, 10.30484340692709821, -1.66417793316524310, 6.30731977363424257, -0.68820140162874166,  5.76315952985009794, 2.23409070272043486, NA, 1.03521352059424787, 2.39058480994306688, 1.88207119798252354,  8.19383339634243058, NA, 3.91152713093107174, NA, 0.42460085059432273, 1.24559853858661884,  2.40559452616372749, NA, NA, NA, 2.03280774305992651, NA,  2.10476042333394542, NA, 1.76864884608320705, NA, NA, 3.29928055387490105,  3.29140249786318240, 0.97795775692077425, 10.23668911026795492, -1.68220075974173677, 6.19393502888127934, -0.65531087831100931,  5.73542426289363139, 2.15148612597380762, NA, 1.06268512264045345, 2.43855357280509022, 1.89063186687614393,  8.41481677587434973, NA, 3.80365225185468736, NA, 0.41256244123605135, 1.26433022195348288,  2.44286750482902004, NA, NA, NA, 2.06904343313753980, NA,  2.11968199154368087, NA, 1.77919096313149749, NA, NA, 3.33050390745646352,  3.25478064784877663, 1.01468073435740491, 10.31786783952881414, -1.67615235852986277, 6.27739125325576097, -0.56637857632001642,  5.70202063385657443, 2.09917854558538064, NA, 1.11916035361569266, 2.52811358406127029, 1.95155665144863955,  8.62427047833709715, NA, 3.76418377201161380, NA, 0.46145108273112001, 1.33818904532580651,  2.43441614997455913, NA, NA, NA, 2.04396466036040581, NA,  2.13445134597773967, NA, 1.77285283371724134, NA, NA, 3.33503569232728525,  3.22017533130857236, 1.04891020408587288, 10.31238501179600320, -1.67464451323597596, 6.23249404853707567, -0.48200551942575443,  5.70571879553992556, 2.02361388172415246, NA, 1.13859867135118620, 2.59331848235436269, 1.97837130393622118,  8.75745170815859808, NA, 3.75587546978939191, NA, 0.51491722019551389, 1.38098830182295851,  2.36372490210866415, NA, NA, NA, 1.99494486350540079, NA,  2.12015189158138506, NA, 1.72252178320768246, NA, NA, 3.32131457412072217,  3.16840614480672489, 1.03755324146410821, 10.29026600042209338, -1.68667121704715672, 6.13276386329992729, -0.41030546124785483,  5.72617302701024844, 1.91872097721922552, NA, 1.14309871833189702, 2.60046734574932747, 2.00162374612113059,  8.83641879936842400, NA, 3.65062562981319250, NA, 0.54414106991306144, 1.37356210277840507,  2.27226513175086309, NA, NA, NA, 1.90104726820010161, NA,  2.05890384059976350, NA, 1.63490650517119551, NA, NA, 3.31253552897045633,  3.10480678814322886, 1.01375088926919954, 10.02859275407508122, -1.70539712437745594, 5.89544468388352438, -0.40121970757696990,  5.79253332860188053, 1.80094427623304654, NA, 1.14039147178627731, 2.56587414297462013, 2.01990455157192317,  8.67067986578609684, NA, 3.50211815158901363, NA, 0.54577185745891810, 1.24132042408026022,  2.20689227811448907, NA, NA, NA, 1.83183034366093045, NA,  1.99754002459747682, NA, 1.59077948729721119, NA, NA, 3.30029931597778248,  3.03413779531694106, 0.99470696516054558, 10.12691487280364555, -1.73224422509313314, 5.87988980588582244, -0.39841022456732844,  5.75629638974209268, 1.65047845822374351, NA, 1.13621634903839652, 2.47711445116789530, 1.93733837127122888,  8.62488116889942447, NA, 3.36292885927156515, NA, 0.52580535311126164, 1.09009183146012756,  2.18800795951363725, NA, NA, NA, 1.78455528233261829, NA,  1.91173378958083373, NA, 1.57258029727288351, NA, NA, 3.27069396312495053,  2.98012344125787187, 1.04434627026759319, 10.05524088454200360, -1.75509965458621098, 5.74788890470867386, -0.33402638262890993,  5.78912468485626341, 1.55700433998346188, NA, 1.13824292051373588, 2.37116125031138214, 1.89539340505670761,  8.21885706276687777, NA, 3.25631478994404588, NA, 0.51664771543475840, 0.98509056219848945,  2.20694468356568052, NA, NA, NA, 1.71260413520701316, NA,  1.90785393063364972, NA, 1.60905316331328230, NA, NA, 3.31613939701555571,  2.90765022992330513, 1.10078703061533489, 10.17098615371323334, -1.73749805718893402, 5.75643866143508021, -0.29633300009906210,  5.73231839817021260, 1.49493912619616376, NA, 1.11710852957764484, 2.29895398451521205, 1.88810073935901235,  8.38933601784338556, NA, 3.18229889157544177, NA, 0.54939478536434461, 0.94233335101211491,  2.20105219230317051, NA, NA, NA, 1.63914602039386614, NA,  1.90199950855426381, NA, 1.64322933837239815, NA, NA, 3.37832068388259232,  2.85091049147714726, 1.16090927052776305, 10.17636657498715635, -1.73054409029476264, 5.82876583269565618, -0.26249160253977688,  5.72429195985051553, 1.43262656507010755, NA, 1.08040096883661851, 2.23383365888819219, 1.88117208992497864,  8.50212146133254265, NA, 3.17232341180319288, NA, 0.59170989516417372, 0.92970398745531724,  2.23198471288768863, NA, NA, NA, 1.62002085865008838, NA,  1.93893900803087882, NA, 1.65825431525090305, NA, NA, 3.44033965164340483,  2.82616709685455048, 1.22931265755020602, 10.35820447564415936, -1.68143055768891547, 5.92705713133426304, -0.20344463459438317,  5.75263147170652900, 1.40481441957238862, NA, 1.10008282488854370, 2.22551813760410777, 1.94287717053512421,  8.65913741951817073, NA, 3.16067840013663748, NA, 0.63974292037248948, 0.94391319761542614,  2.22107126512366060, NA, NA, NA, 1.58811145834745293, NA,  1.95786933133205809, NA, 1.60975413177468774, NA, NA, 3.52461237460561705,  2.79731118416537727, 1.24987510044786565, 10.46016776106000101, -1.62758564559163554, 5.96475633395493166, -0.15375685037235221,  5.75513570893576443, 1.36174120708506452, NA, 1.11297733786630548, 2.21036113176681948, 1.91535127475931333,  8.74019801291776233, NA, 3.19647507645462525, NA, 0.63812137279686931, 0.96109085657605065,  2.26251007332334542, NA, NA, NA, 1.60441176907251370, NA,  1.96903249702274730, NA, 1.60832283908634022, NA, NA, 3.63681851806046019,  2.78887834428154147, 1.29623805204376308,
                  10.60292533756160616, -1.55889690461015662, 6.03000995690897490, -0.09519715127274297,  5.70340504073632903, 1.35093750350507880, NA, 1.15364694796126743, 2.20807915557969503, 1.88700362608027716,  8.81586323985126796, NA, 3.26842667021870925, NA, 0.67372393598937930, 1.00318833510692818,  2.29221557034547763, NA, NA, NA, 1.51910296659635691, NA,  1.92579587622615289, NA, 1.60571108902988091, NA, NA, 3.69188514462447825,  2.81264611172715595, 1.31035975112963987, 10.77491491177479865, -1.55176475570830785, 6.07841855562725275, -0.10118443110935205,  5.65897999751380443, 1.33510525639422406, NA, 1.17039917845224917, 2.18595682159588200, 1.83499254769337661,  8.89091918138911552, NA, 3.25099387466259326, NA, 0.69229349292661424, 1.04572032647408020,  2.31720303802896721, NA, NA, NA, 1.49657377971303163, NA,  1.88230799501629398, NA, 1.60875716345039033, NA, NA, 3.73107010286788565,  2.84430650725013390, 1.32091624312998768, 10.90473258880454388, -1.54250011568580492, 6.07246071647467556, -0.10064431528868201,  5.64621111524320796, 1.31167270160848948, NA, 1.17339444802809978, 2.14612336400895298, 1.78823288859196561,  8.81108448708225467, NA, 3.29190817740905706, NA, 0.73744251648636594, 1.08217056991396632,  2.32488100474039960, NA, NA, NA, 1.44250918885315205, NA,  1.82792712993226547, NA, 1.60122886810498577, NA, NA, 3.77422781857261302,  2.85622706133350412, 1.27760300237014524, 10.92846943784030422, -1.54252152783648677, 6.07961253143265701, -0.14522560867301226,  5.59167495181066521, 1.26965387185166945, NA, 1.15314694956238672, 2.06755225665720221, 1.73465342691106761,  8.91076509315189114, NA, 3.23470977024433592, NA, 0.76965103109728750, 1.12566448385818263,  2.33345414129081474, NA, NA, NA, 1.37621601026793172, NA,  1.78878898348631221, NA, 1.59244290532341992, NA, NA, 3.82147100427334596,  2.88188458334726860, 1.26116772263326360, 10.90022990413864790, -1.52010398557284354, 6.00115837412832320, -0.15675491114368056,  5.55465344538987704, 1.25081562256921819, NA, 1.15263749135138704, 2.03229978607010064, 1.68076404944617686,  8.87695978271285746, NA, 3.17730702026943401, NA, 0.83195038057491522, 1.17986823742966007,  2.30551553546869492, NA, NA, NA, 1.28212171931200936, NA,  1.75460083094827257, NA, 1.58622017836405438, NA, NA, 3.84814830700393973,  2.93475631169629958, 1.22472575995947830, 10.83227038615737747, -1.52483568678694725, 5.93622999684782737, -0.19779640032333912,  5.54072816669212642, 1.23237328641121779, NA, 1.15881751455385884, 1.98458515436948968, 1.62164259948207246,  8.94325246669946594, NA, 3.15676391609702645, NA, 0.87956408363028504, 1.24765371700926275,  2.30559683487251199, NA, NA, NA, 1.19222384783901725, NA,  1.73629888217665940, NA, 1.58549230116011808, NA, NA, 3.88171535908455434,  3.01114270487772995, 1.19363511398944322, 10.73823388282068692, -1.52540821793219905, 5.80894677754346933, -0.22226038311512802,  5.54559011199765362, 1.22355227600205962, NA, 1.17119484475336244, 1.95434014647645626, 1.58124532021008202,  9.08297973964202043, NA, 3.20402147508893842, NA, 0.93818684861872148, 1.32087592447227964,  2.32670808437100307, NA, NA, NA, 1.12457052233193466, NA,  1.73173178589156285, NA, 1.57419150780818273, NA, NA, 3.92431709638649062,  3.08940559145177263, 1.23112977498601284, 10.83338459865464110, -1.49863545827301903, 5.73613496996673167, -0.25527041534409528,  5.59559895085497949, 1.19432954631652022, NA, 1.19156687064329048, 1.93819317475791575, 1.57152944075640599,  8.99746472417894338, NA, 3.18787294945728705, NA, 1.01749465332197420, 1.35241457375683671,  2.33626393383628228, NA, NA, NA, 1.03321135951535847, NA,  1.72037279793930997, NA, 1.56440274685472147, NA, NA, 3.94530675673502707,  3.13693263560922375, 1.22582741779247950, 10.83382252992685224, -1.46461473408285592, 5.59554301835730517, -0.27917725874263555,  5.69515863986526227, 1.16969757294496635, NA, 1.22065659671510662, 1.93359261881349465, 1.57600598459776742,  8.75000733341794934, NA, 3.09170553118372071, NA, 1.05003714078831267, 1.37383041520451865,  2.36333844555799333, NA, NA, NA, 0.93696289756788209, NA,  1.71423029287235762, NA, 1.56027427642149163, NA, NA, 3.96484774079013746,  3.19494740689065759, 1.21679687418326354, 10.83315408774296351, -1.42930590695283244, 5.59642935094932703, -0.29474869494367706,  5.75454308729977804, 1.14984421779086210, NA, 1.23146375723553159, 1.92379927491126335, 1.58080365241645415,  8.47103788055176388, NA, 3.01429243414123782, NA, 1.08476355402884916, 1.37849637606354603,  2.39955751314886134, NA, NA, NA, 0.82955782820780288, NA,  1.70653992609398064, NA, 1.53902413451490405, NA, NA, 3.98436842609781161,  3.25371461149653474, 1.20944322680048044, 10.80658320686026919, -1.38117282739532587, 5.55751305242980997, -0.30232620991309739,  5.80966844888960754, 1.15319609073833451, NA, 1.23712666574887198, 1.91982854659600699, 1.58889617759563295,  8.44815670527920304, NA, 2.98509457540382916, NA, 1.12963715504195195, 1.40381089233778433,  2.44789985514160691, NA, NA, NA, 0.72356766132639638, NA,  1.70522899113330695, NA, 1.51045808053770214, NA, NA, 4.00535349066225699,  3.32151811055964696, 1.19741675225332500, 10.82928214343039031, -1.33702155311792792, 5.45802131493970233, -0.29955065428956673,  5.85369093143380681, 1.16039746490815299, NA, 1.24934459216688065, 1.92398924808612604, 1.61762030713264404,  8.51794825360834551, NA, 3.02679323469767247, NA, 1.18444050721681204, 1.46703465081651174,  2.47807747328791939, NA, NA, NA, 0.63997234341852516, NA,  1.72120054572857617, NA, 1.50634881204114390, NA, NA, 4.04706694142666468,  3.40960427171013025, 1.21084201291244087, 10.87608609036221630, -1.31604010634186563, 5.47730209080738550, -0.28974244857759512,  5.92059223035167825, 1.13561296609766060, NA, 1.28248438667946951, 1.90642244753214163, 1.63701155969434442,  8.45980656796190900, NA, 2.95541917209445604, NA, 1.21800946753105199, 1.45752238786693078,  2.50896713255331605, NA, NA, NA, 0.54285852826270498, NA,  1.74222519906515561, NA, 1.49203150049467026, NA, NA, 4.07958465393455594,  3.48892080797417847, 1.23321805043713417, 10.94429160252522770, -1.32626463871764333, 5.47650376446812626, -0.26894588999281660,  5.92009569724318396, 1.10647744859474129, NA, 1.32326298177452939, 1.87044773793251817, 1.65077268518393749,  8.38365895665550553, NA, 2.89859157329646289, NA, 1.26172963657454984, 1.48477508650255774,  2.52979041986870623, NA, NA, NA, 0.48039162491018927, NA,  1.77209069100934125, NA, 1.51908071229075814, NA, NA, 4.12977585193098307,  3.56443430036777986, 1.26501781471371544, 10.98421602081938175, -1.32405441254586531, 5.46661313028935236, -0.23592551967328773,  5.96693575575320967, 1.06282042177420410, NA, 1.37066583102138351, 1.85934371772551166, 1.67253034048825233,  8.30413904289447480, NA, 2.84719895362846742, NA, 1.29454954866023231, 1.47574133620599057,  2.54527821516815411, NA, NA, NA, 0.43077434260124758, NA,  1.79809859510705405, NA, 1.54030360235968233, NA, NA, 4.18733611172793641,  3.64922739326493595, 1.29510118238999672, 11.03737374055405773, -1.33083455761491143, 5.43203753201945272, -0.21441221922791709,  6.02846983431817574, 1.02576581258914179, NA,
                  1.39037801589234600, 1.84414169804414629, 1.67474313833656341,  8.24046309618466921, NA, 2.82435332829605557, NA, 1.33281328106081354, 1.46638494358817217,  2.55380572276553863, NA, NA, NA, 0.36073505869145894, NA,  1.82814049740740403, NA, 1.57052427280849338, NA, NA, 4.25052780837515076,  3.74259228374826014, 1.33253485523600901, 11.05946691207841859, -1.33477494717109901, 5.37260646754288729, -0.18330738776289013,  6.11448916455442770, 0.99734539798339905, NA, 1.41685290284792997, 1.82938085489838742, 1.68103590330549979,  8.13922228213998800, NA, 2.84257847372481276, NA, 1.37638814808279131, 1.47251966900879938,  2.55559652068176257, NA, NA, NA, 0.31436346124467274, NA,  1.84389982972248578, NA, 1.61709281658844128, NA, NA, 4.31228676318300330,  3.84465090278261101, 1.38301176392208047, 11.07847834593717806, -1.33260759781600968, 5.31899734250855083, -0.14917337040585077,  6.18312766769793232, 0.97569576924109491, NA, 1.40259110564869394, 1.81648213258587510, 1.69801946854127594,  8.14942032395542171, NA, 2.82053697236537904, NA, 1.42665592484371895, 1.47896866503226532,  2.54699605996761624, NA, NA, NA, 0.25261751806544758, NA,  1.84715275579999116, NA, 1.66926147622454391, NA, NA, 4.36225585878148614,  3.94284899434822478, 1.39463484240839231, 11.05120392394811724, -1.32495738044556544, 5.24039393927958930, -0.14001664494950933,  6.28275869731476266, 0.94457568814407611, NA, 1.38550700896759005, 1.77679581445311685, 1.70568607107734183,  8.07145749313648686, NA, 2.84732720811153373, NA, 1.45081817885383479, 1.49927211874107513,  2.54118497900467055, NA, NA, NA, 0.24539490544368142, NA,  1.85131932064233196, NA, 1.67988371722167829, NA, NA, 4.38722485023779551,  3.95630272902820623, 1.37384059072349185, 11.00551969877772152, -1.33774100920802219, 5.23212024229574801, -0.12082738398744292,  6.31285389252954943, 0.92710398611755385, NA, 1.36363732624063339, 1.73748527222719229, 1.69015322704074045,  7.96541635538237980, NA, 2.82949410621423869, NA, 1.45258485663071313, 1.51961972103677523,  2.51935828751027113, NA, NA, NA, 0.22764219503599264, NA,  1.85382500810909367, NA, 1.69187138078122579, NA, NA, 4.41403891343473997,  3.95369832857354098, 1.35697742707403735, 10.86805406328278245, -1.34155775848848036, 5.24848206841359577, -0.10219504757995682,  6.30900499699992690, 0.90308372637674128, NA, 1.33326407123368673, 1.69472416901908973, 1.67287170051580114,  7.91005397368538965, NA, 2.80608513498351275, NA, 1.45880423430071260, 1.55546761729984295,  2.44119639757453699, NA, NA, NA, 0.17299823028314665, NA,  1.84027101725585807, NA, 1.69152795889103835, NA, NA, 4.41499413549298225,  3.94153337705175311, 1.34836003780798253, 10.63545527131451962, -1.33790730609344588, 5.23692997754088108, -0.10894083434650303,  6.29251177871386869, 0.88640439416473393, NA, 1.29707231592241490, 1.65385054876470861, 1.65789485401881986,  8.05396750086645774, NA, 2.79208210378880306, NA, 1.45706726206378345, 1.59011914576747593,  0.03731826125724894, NA, NA, NA, 0.07830502044416139, NA,  0.04530010501282205, NA, 0.06096265335618177, NA, NA, 0.05120223153606863,  0.02563378642889205, 0.07556372422647700, 0.06362877290251390, 0.08443437684909647, 0.06424799616929543, 0.10149111202735604,  0.04643192210867285, 0.03486949469320832, NA, 0.06519617221384787, 0.02420582755836560, 0.04726234302209704,  0.03656805623571387, NA, 0.03174816638895331, NA, 0.05983940253494553, 0.06309962945043116,  0.03731822666517703, NA, NA, NA, 0.07830495821677566, NA,  0.04530008376786400, NA, 0.06096262832328837, NA, NA, 0.05120219695209510,  0.02563376233317680, 0.07556368983541528, 0.06362868162297752, 0.08443436623639769, 0.06424790070174846, 0.10149107278417459,  0.04643189273663492, 0.03486946573352923, NA, 0.06519617909331492, 0.02420579830031158, 0.04726234935585282,  0.03656794919544440, NA, 0.03174814548827511, NA, 0.05983938309752296, 0.06309960858832546,  0.04712224753353612, NA, NA, NA, 0.11868837881917518, NA,  0.05522187889503279, NA, 0.08485957962919595, NA, NA, 0.08516148547956534,  0.03815016684104041, 0.09443829893656486, 0.10888695803749963, 0.09684665190840216, 0.09048892007085751, 0.11170068026301472,  0.09561637609993588, 0.04982258382706347, NA, 0.08221069572874166, 0.03483471353575119, 0.05421886085361367,  0.03834669555173880, NA, 0.05409224565779798, NA, 0.06918681353577948, 0.07025469764416772,  0.04860603134748585, NA, NA, NA, 0.10302044122244723, NA,  0.05129887425923424, NA, 0.07373961183230199, NA, NA, 0.07139167582443988,  0.05299968360205658, 0.08495712872298948, 0.08640548728730262, 0.09079247905127838, 0.08030448315150682, 0.10227182841859647,  0.08285750022078196, 0.04386761594293111, NA, 0.07132217527971822, 0.02837750892175014, 0.04482457393348940,  0.03355271394598745, NA, 0.05015658712300525, NA, 0.06057736767877315, 0.06456331496082539,  0.06622580926558751, NA, NA, NA, 0.08812386515494153, NA,  0.05673039508123617, NA, 0.07435640979875108, NA, NA, 0.06207650810997967,  0.06550081872415635, 0.08670040215039104, 0.07546940399290909, 0.08226407719395180, 0.06828791715728194, 0.10108743428351201,  0.07176485082940355, 0.03601558296593875, NA, 0.06489291598344071, 0.03018997713359241, 0.04086950682461774,  0.03059655211288798, NA, 0.04247862363681632, NA, 0.05442664454409502, 0.06171628937673829,  0.05799067623022060, NA, NA, NA, 0.07535409404143463, NA,  0.04625251134893145, NA, 0.05956439748878411, NA, NA, 0.04984213009431485,  0.04802051313834809, 0.06611560586037506, 0.05834164810945364, 0.07334010553205597, 0.05704972481921870, 0.08825595793350979,  0.05333015364933554, 0.03096893481457925, NA, 0.05371893661568750, 0.03205571249600362, 0.03390796242635709,  0.02431113881001406, NA, 0.03449121269422389, NA, 0.04397571633576133, 0.05772137632859081,  0.06352281275210510, NA, NA, NA, 0.07601318113977597, NA,  0.04444425634132115, NA, 0.05420675268677248, NA, NA, 0.04241813122905931,  0.04046055719493669, 0.05837763903773439, 0.05516237480534690, 0.07039907697724979, 0.05482678745564430, 0.08312768441249693,  0.04005528460350200, 0.03583779385709455, NA, 0.04744037700380985, 0.04111845414294370, 0.02203592931001248,  0.02458826188559129, NA, 0.03794239530923535, NA, 0.04020421502334617, 0.06169130545829452,  0.07047292969376387, NA, NA, NA, 0.07452141950086547, NA,  0.04377886067519660, NA, 0.06036352502196711, NA, NA, 0.03274609364873578,  0.04148768512789543, 0.05409399966145428, 0.04763639157853933, 0.06849495987595410, 0.05988521375369890, 0.09292456190584038,  0.02670055151999818, 0.04210175753161427, NA, 0.05560615144944185, 0.04045923821518624, 0.01482667782613038,  0.02994574148878891, NA, 0.03576033388461528, NA, 0.03772397168460386, 0.06928010618285703,  0.07273968134372535, NA, NA, NA, 0.07302165596314558, NA,  0.04494917135045354, NA, 0.07866709426906963, NA, NA, 0.03620143197981697,  0.05101092924709261, 0.06902421548170086, 0.06108191537563640, 0.07551821598955168, 0.08422632896180193, 0.08970911074511058,  0.01191479448500728, 0.04203551585455210, NA, 0.06854024221127819, 0.03441550677731735, 0.01463101924088215,  0.02980695807583707, NA, 0.03374489864773101, NA, 0.03819734177951382, 0.06783788594216769,
                  0.06424103156473543, NA, NA, NA, 0.06999781705477300, NA,  0.04635916073962466, NA, 0.06803820578954968, NA, NA, 0.03502789411432564,  0.04897899217764863, 0.06758546307622552, 0.04567692166561119, 0.08399209630259592, 0.07323725490782651, 0.07715556672169913,  0.02878723668862340, 0.04236514711994795, NA, 0.06430973703286635, 0.03315057861645446, 0.03267663034586068,  0.03054144667867637, NA, 0.03035364213730646, NA, 0.03672579819882336, 0.06160579103890334,  0.05875116751473085, NA, NA, NA, 0.06928239197020497, NA,  0.05225475211587466, NA, 0.06851571524158970, NA, NA, 0.04391961698086180,  0.03822957069042052, 0.06263605354820555, 0.04482686319543740, 0.08510855569046256, 0.08077154940137921, 0.07176436571026289,  0.04232588560794031, 0.04292016037294599, NA, 0.06531591248395210, 0.04065696566634008, 0.05002418686125463,  0.04258773441123594, NA, 0.02689557438263912, NA, 0.04812077920638503, 0.06259687460719202,  0.05025058667254678, NA, NA, NA, 0.07050410450223937, NA,  0.04003033260146865, NA, 0.06118615084873207, NA, NA, 0.04185584360499960,  0.04075968897721630, 0.05972900544681156, 0.03351084720843120, 0.07850152364382616, 0.06750190706351014, 0.06311673590032632,  0.04412997476291959, 0.04099669246953459, NA, 0.06324937341622142, 0.04126330490637559, 0.04861829744987695,  0.04030662786975156, NA, 0.02218571507996192, NA, 0.04890685330941789, 0.05860281964449011,  0.06099480939917836, NA, NA, NA, 0.07431489363742858, NA,  0.03236018690860835, NA, 0.05355858203476471, NA, NA, 0.04227743783043424,  0.04105764598684742, 0.05266308528843271, 0.05326506657606879, 0.07925080200355107, 0.06356048527643895, 0.06601065562778938,  0.04124384393790535, 0.04638693374024463, NA, 0.06871456053702242, 0.04523049216726319, 0.05618417148001162,  0.03472072420357616, NA, 0.02782469113892760, NA, 0.05179562902920429, 0.05138852729017745,  0.05510853163185365, NA, NA, NA, 0.07696188223617569, NA,  0.02832636803560899, NA, 0.05113392296207691, NA, NA, 0.03978434198814369,  0.04084536596921991, 0.04843374811383211, 0.04782617544934796, 0.07813756759537226, 0.05675824898320157, 0.06892352487404234,  0.04872537746237143, 0.04336394257743487, NA, 0.06457810864590201, 0.03850716668655747, 0.05344218826742341,  0.03027943765964676, NA, 0.03168185088742460, NA, 0.05984979499070544, 0.04567869389210830,  0.05109534305197257, NA, NA, NA, 0.06890917905684278, NA,  0.01854686560351311, NA, 0.05430091872531335, NA, NA, 0.03353018864345760,  0.04230496549282314, 0.04793074711835101, 0.04297560377919079, 0.07521827878379761, 0.04840550538606228, 0.06075053814131752,  0.03282362622375899, 0.04352261922972318, NA, 0.06329595186485153, 0.03274228393111323, 0.04832464312140230,  0.02052494654582234, NA, 0.03888596338119343, NA, 0.05781595986164188, 0.04596336318566627,  0.04506576643270246, NA, NA, NA, 0.05714214277794804, NA,  0.02015414642160482, NA, 0.05458040491731622, NA, NA, 0.03132368696299315,  0.05248829105405773, 0.05346305147342724, 0.04103586916431973, 0.07768768270847186, 0.04893490480134566, 0.06340324445833041,  0.04579830201466874, 0.04608667500341211, NA, 0.06677854487978303, 0.03439696091375044, 0.05596841490439758,  0.01217483276168953, NA, 0.05348780416309055, NA, 0.05889454859854075, 0.04804619753426823,  0.03872846172809970, NA, NA, NA, 0.06886908598248917, NA,  0.02135078438369486, NA, 0.05278627201691369, NA, NA, 0.02719415901902086,  0.04741630955729875, 0.05527735284141021, 0.04827233511923165, 0.07361194411076062, 0.05789593294872909, 0.06553206594937022,  0.07771202413316818, 0.04805409641636243, NA, 0.06588108690047924, 0.03885296388608996, 0.05284922058356127,  0.01714054195015038, NA, 0.05835704256155277, NA, 0.07215543510666747, 0.04917606700971286,  0.04294612149931695, NA, NA, NA, 0.06356440292462874, NA,  0.02507596680090843, NA, 0.05196353188645462, NA, NA, 0.02152923589641387,  0.04659461449483282, 0.06879224975399591, 0.07925638694322125, 0.07236621620803681, 0.08358627458365114, 0.07406497505706414,  0.07097090216995422, 0.05894903427551447, NA, 0.07448833026431473, 0.05545262716210046, 0.05076429731392006,  0.01894429370241588, NA, 0.07113700186699538, NA, 0.06718548532702201, 0.04938275265689138,  0.04136203427978803, NA, NA, NA, 0.06192149437428554, NA,  0.02753598162362703, NA, 0.04653401608193083, NA, NA, 0.01840460245713724,  0.03950838353194259, 0.06846843450066674, 0.07462038917458709, 0.07850861705090470, 0.07355538999116223, 0.06897140171318242,  0.06011247610623355, 0.05380157881374312, NA, 0.06940101660856519, 0.05908514572577279, 0.04603825470495358,  0.01288699803109714, NA, 0.06352669600429886, NA, 0.06263297308880754, 0.04954195171726290,  0.04205497784642986, NA, NA, NA, 0.06058958765118526, NA,  0.03113114862718662, NA, 0.04634168913832213, NA, NA, 0.02073432722649953,  0.04119420415751818, 0.06366271033124135, 0.06615293294717613, 0.07487317582780845, 0.06595034061041957, 0.06256074106683913,  0.04919706705503792, 0.04911035461937033, NA, 0.06112757055692809, 0.05314888295267170, 0.04464780524915095,  0.01113930156115615, NA, 0.05826753944945934, NA, 0.05319690922796395, 0.04842438394305330,  0.04438540568122337, NA, NA, NA, 0.06673987991778695, NA,  0.03313201981935555, NA, 0.05085000781814275, NA, NA, 0.02163134005791855,  0.05441284650388190, 0.05909226948440131, 0.06811620279557255, 0.07574282887621535, 0.05857617109103276, 0.05986158078439463,  0.06065025859457202, 0.04574274861786349, NA, 0.05821197995189546, 0.05935196896997537, 0.05101577417604596,  0.01837252622662422, NA, 0.05945642647840819, NA, 0.05084469901698641, 0.05200652317926024,  0.04258251892921706, NA, NA, NA, 0.06805928812957626, NA,  0.03200294937128347, NA, 0.05372147525719077, NA, NA, 0.02099877082074321,  0.07479605940334541, 0.06523743300086270, 0.08280191524688861, 0.08048846130611201, 0.06034406623493755, 0.06843085480598855,  0.08503810346737564, 0.04853186149266587, NA, 0.05753184335456629, 0.06902504858341751, 0.04596249663364618,  0.03348484160626355, NA, 0.07099431129660465, NA, 0.05356369519960682, 0.05698083833747403,  0.03554579780596070, NA, NA, NA, 0.06815924057695562, NA,  0.03338256116677525, NA, 0.05163130528092345, NA, NA, 0.02498083529200598,  0.06993581746822368, 0.06065752493136313, 0.06864363816410599, 0.07798937370891121, 0.05690084621806485, 0.06392475922151360,  0.07656266219641676, 0.04352929307420980, NA, 0.05052642779618188, 0.05698547456887325, 0.04161608877720373,  0.03151639755173745, NA, 0.06720614020270312, NA, 0.05176890840898024, 0.05893715760702135,  0.03519178979448998, NA, NA, NA, 0.06785661107175603, NA,  0.03785853176291266, NA, 0.05558922004658243, NA, NA, 0.02859111714056734,  0.06437066738725321, 0.05765345266712842, 0.05231753786102770, 0.08441082582300291, 0.05711817463816389, 0.06492479945177436,  0.06686419146868647, 0.04197316969041300, NA, 0.04975341290365271, 0.04654130396284327, 0.04416532490802844,  0.03298906480612583, NA, 0.06219133077416475, NA, 0.04784134023646494, 0.06672505145427027,  0.03450657926774572, NA, NA, NA, 0.06998003877165201, NA,  0.03755852296689445, NA, 0.05185516201151325, NA, NA, 0.03318855538691730,  0.05745275283762512, 0.05474924462572199,
                  0.04625371327211857, 0.08524247398597888, 0.06095369176002231, 0.06444544759727711,  0.06132596605806877, 0.04790967412699312, NA, 0.04701609750709045, 0.04745063924293460, 0.04668798726411967,  0.03493087006388408, NA, 0.05968997570544978, NA, 0.04660839052824966, 0.06086631950829909,  0.04134544563036575, NA, NA, NA, 0.07688778166989563, NA,  0.04098611125354239, NA, 0.05473069734293574, NA, NA, 0.04597225753930180,  0.05492146127939116, 0.06146213713203712, 0.05227920839368299, 0.09419211244315209, 0.07242030307786979, 0.07145966047164055,  0.07033839115437306, 0.05872869819348896, NA, 0.05362974275234780, 0.05944695466808514, 0.05492547095405090,  0.03661541529111368, NA, 0.06403159895378835, NA, 0.05290955167949292, 0.06077869710859712,  0.05717197262191968, NA, NA, NA, 0.08350426665067065, NA,  0.03945134116154973, NA, 0.06883740775002012, NA, NA, 0.07001394903911454,  0.06048285720086455, 0.06941011856149694, 0.06050871900191445, 0.09233157791569327, 0.09430767611799896, 0.07195550317406779,  0.07825574586954456, 0.06886533948126507, NA, 0.05759482227573281, 0.08129886235599257, 0.06625749314077864,  0.04051357401374437, NA, 0.07676320042383929, NA, 0.06230768691705038, 0.06885247811667380,  0.04651458294214705, NA, NA, NA, 0.08163845764285231, NA,  0.03591623584782284, NA, 0.06350427688158015, NA, NA, 0.05972349120955846,  0.05744530296619595, 0.06278208475599009, 0.04680633660435897, 0.08484594955011812, 0.08823645829695942, 0.06864613921215636,  0.06515681538951500, 0.06171957351216486, NA, 0.04877125438397741, 0.07049524140466364, 0.05703040239664701,  0.03676990148417554, NA, 0.07183318545990081, NA, 0.05507596788564672, 0.06563375473536937,  0.03907993759025952, NA, NA, NA, 0.08793182064442771, NA,  0.03470347900435452, NA, 0.07104316529427052, NA, NA, 0.05168075968250900,  0.05693580367716519, 0.06183304209730663, 0.03911941972939396, 0.08430871024407820, 0.08259774504525870, 0.07252952496922650,  0.06138802678692802, 0.06279220058678146, NA, 0.04509445954273212, 0.06240125080310826, 0.04964897087019235,  0.03485552195940534, NA, 0.07042630856875189, NA, 0.05327880576617294, 0.07184587857455775,  0.03045575478637225, NA, NA, NA, 0.09021627499082359, NA,  0.03690178444635165, NA, 0.06788693205012711, NA, NA, 0.04261102870309078,  0.05478638984487944, 0.06381102637554911, 0.03926470792041796, 0.08370777060086690, 0.07788276323535839, 0.07233592702280696,  0.06941239281576855, 0.05555996391704850, NA, 0.04808454017168663, 0.05413320366382801, 0.04736314892841758,  0.03595051262558237, NA, 0.06745415851680067, NA, 0.04708420898693057, 0.06511491053410121,  0.02451503879770282, NA, NA, NA, 0.08739495994918829, NA,  0.03755509748389883, NA, 0.06674999551407827, NA, NA, 0.03875848433123877,  0.05733341295300288, 0.06753228733806499, 0.03819810175566917, 0.08448097947041193, 0.07993512935752600, 0.07129845549586221,  0.07768145606091434, 0.05416475921820731, NA, 0.04003264451134093, 0.04971631610210620, 0.04024436435646382,  0.03832330574825228, NA, 0.07039890008944050, NA, 0.04582693157414381, 0.05973732101013653,  0.02708663056193908, NA, NA, NA, 0.09561142921755818, NA,  0.04182701319073501, NA, 0.08092315216862972, NA, NA, 0.04061448504202025,  0.06580730624664238, 0.07888420785714094, 0.04592982066051900, 0.09221317930063087, 0.08936382501450980, 0.07750798778828422,  0.09751218915842950, 0.06160814389947976, NA, 0.03754841930373542, 0.05280892651303101, 0.03994518562154857,  0.04415750173992512, NA, 0.08127493438458636, NA, 0.05030115736693659, 0.05940066815349425,  0.02646999157679061, NA, NA, NA, 0.09966609238415432, NA,  0.04763543081042852, NA, 0.06990070602820618, NA, NA, 0.05005814201863430,  0.07266787455823150, 0.09038020133223079, 0.04731810025933358, 0.09764763556107113, 0.10811912835974015, 0.08494097244059076,  0.09728908106713745, 0.06299940343346842, NA, 0.03222287848684388, 0.06529038760988917, 0.03972741259433168,  0.05328909091852996, NA, 0.07956121928271469, NA, 0.06342474664455125, 0.05868328730678633,  0.03700180919870989, NA, NA, NA, 0.10965733410055510, NA,  0.05170484860256742, NA, 0.06515366332976709, NA, NA, 0.05924590281812789,  0.08240796767346331, 0.08881173469831462, 0.06124200711656095, 0.11088352735827986, 0.12225218070123189, 0.08917310807702543,  0.11934946777537885, 0.06792472487861534, NA, 0.03377145967397854, 0.06786681594069299, 0.03962192511637683,  0.06166191264559377, NA, 0.08739857832006723, NA, 0.06881993526654356, 0.06322702239617310,  0.03906429325639675, NA, NA, NA, 0.10962576506164200, NA,  0.05696971151168712, NA, 0.06002723725618878, NA, NA, 0.05873918753190988,  0.08479522738395948, 0.08191547548015962, 0.05864344064999773, 0.11135421103074956, 0.11556047773365842, 0.08273432402966593,  0.11163794864752263, 0.07391032376193944, NA, 0.03861383229674997, 0.07339179746216171, 0.03995752759160304,  0.06758594593333898, NA, 0.08144256211058115, NA, 0.06622246280121714, 0.06541787995115832,  0.05682929989611955, NA, NA, NA, 0.11517460231805643, NA,  0.06427295829485485, NA, 0.05529901159970458, NA, NA, 0.06319012623675510,  0.07628838776942568, 0.07944370900319628, 0.06845666547667553, 0.12230863527613869, 0.11091021072041497, 0.07938626530116491,  0.09695410629932855, 0.07304208759525610, NA, 0.04693524486850616, 0.08325223158547823, 0.04112845927079411,  0.07652920173365883, NA, 0.07448941111774530, NA, 0.07178186210994898, 0.07368331003363482,  0.10000761892223259, NA, NA, NA, 0.13402878328156048, NA,  0.07664765429841383, NA, 0.05221378280878547, NA, NA, 0.08611130957039401,  0.07494099049952570, 0.08649947354555079, 0.10192325961865813, 0.14110776714584927, 0.12085250602431938, 0.08668067166345123,  0.08936849373113108, 0.08235074739474252, NA, 0.06414487637371080, 0.11298254039056434, 0.04452436532393794,  0.08802347347283407, NA, 0.07158675443159165, NA, 0.09235784208106104, 0.08456757008228344, -5.52824120125565699, NA, NA, NA, -4.83523545389908804, NA, -5.13612027558903428, NA, -5.21280431704613800, NA, NA, -5.76760992768417147, -3.63568098072667878, -5.54135661664118295, -4.55595747932758144, -3.89926789289919373, -4.03564121886069493, -5.27809453291388930, -4.25002603018384040, -4.26947142334605534, NA, -5.50863142343318835, -5.25628910358222434, -5.63457880706324499, -4.03108478769752310, NA, -2.87262262575088156, NA, -4.28033896274704517, -4.42382816434162418, -5.52824165258151012, NA, NA, NA, -4.83523675240144435, NA, -5.13612043924939510, NA, -5.21280486968603984, NA, NA, -5.76761127035347521, -3.63568169099737215, -5.54135671609095048, -4.55595745417098996, -3.89927007304504691, -4.03564196584866863, -5.27809482346300562, -4.25002724121200082, -4.26947071828913849, NA, -5.50863099205764062, -5.25628936683551284, -5.63458095526970659, -4.03108062960049729, NA, -2.87262282669291036, NA, -4.28033908335469793, -4.42382835025165555, -5.47522338733311376, NA, NA, NA, -4.81528813471305828, NA, -5.08936653031789277, NA, -5.16645824701742384, NA, NA, -5.63889260333214004, -3.66241066104936897, -5.48044967025781737, -4.23033789764828150, -3.84813528446726227, -3.54495871791129602, -5.19579424093630404, -4.18565027581885563, -4.12678862393222978, NA,
                  -5.45991196084861130, -5.30193693905512831, -5.57405708563789837, -3.90335821617729106, NA, -2.62866805350355115, NA, -4.18950058952852089, -4.34933407929630533, -5.57826745833707704, NA, NA, NA, -4.78870087342191209, NA, -5.06336357545199700, NA, -5.16732173384253457, NA, NA, -5.64804476594961180, -3.93884634781862397, -5.47637066395963767, -4.23403289188805854, -3.89205269020001765, -3.61808140578076509, -5.19467859530230669, -4.21588249569063045, -4.19498275049220659, NA, -5.45784222393092122, -5.30899875478002059, -5.58423845252986961, -4.75865873701632669, NA, -2.73253120998697874, NA, -4.20393894901391452, -4.32826328122142190, -5.76571401514281234, NA, NA, NA, -4.89626884250423622, NA, -5.08447557824273044, NA, -5.23433358217266864, NA, NA, -5.75432127627494605, -4.15287710778514629, -5.50222957466043106, -4.38762570091461068, -3.95542876146328837, -3.99484844505065517, -5.15471892763916983, -4.31352121513268649, -4.37146741203662170, NA, -5.46207789012689471, -5.20616447596689547, -5.59303745640560379, -4.22615651707992157, NA, -2.89667834206904695, NA, -4.27873810029605206, -4.19782545174871302, -5.63203475225861361, NA, NA, NA, -4.83376529922564568, NA, -4.95439930277345830, NA, -5.19309892298891729, NA, NA, -5.76091800278287725, -3.96966323713491676, -5.33498099768386336, -4.32775467706928563, -3.92968275067484374, -3.85925166934791131, -4.98435745937665509, -4.28699632967300381, -4.38875697780550311, NA, -5.40362007442897507, -4.97523981013103000, -5.46082387594071506, -3.79397516938276924, NA, -2.91398836676373785, NA, -4.21113562446899881, -3.95322646599762395, -5.54797451104403105, NA, NA, NA, -4.77002569258552267, NA, -4.81225995469492318, NA, -5.17619792318031990, NA, NA, -5.77880717366020402, -3.86205006126262607, -5.21398021247719967, -4.34572500830740882, -3.91340867967968720, -3.81500767086592640, -4.82413327170944761, -4.23919686964717801, -4.42411465122662850, NA, -5.39784003472963825, -4.81899232906358499, -5.37861566028774440, -3.49508643609977465, NA, -2.84125962175815072, NA, -4.14563192399137570, -3.69544980687361724, -5.56349600035319902, NA, NA, NA, -4.82329427773917807, NA, -4.69569954142064283, NA, -5.17523291698169441, NA, NA, -5.82866848034020890, -3.88366639965317040, -5.13279343239373098, -4.44191016648640780, -3.95481786023131265, -3.75229222749099378, -4.68010584873484703, -4.19143824967089884, -4.52423287383100803, NA, -5.38601955912517205, -4.72132951351229835, -5.25708176426824281, -3.25836095302139084, NA, -2.92275971408837121, NA, -4.09044616146560891, -3.53967413005958864, -5.63142369031485046, NA, NA, NA, -4.93458406701348107, NA, -4.70471953086530359, NA, -5.22783669867443912, NA, NA, -5.87182892015371838, -4.03507248160313026, -5.07021417774586425, -4.81811665063210004, -4.01760118688446699, -3.67550340518813279, -4.64748757625360298, -4.09022881667966320, -4.61762231121712574, NA, -5.37355524596069856, -4.77006833056070967, -5.14682752993851889, -3.71049744246158042, NA, -3.05714092238604085, NA, -4.09840617710582755, -3.59015165378262902, -5.62806603051673715, NA, NA, NA, -4.98402292413669112, NA, -4.62063023660835892, NA, -5.21651678686740272, NA, NA, -5.92859043954151588, -4.05904906661827081, -4.98616129300123667, -4.70004804998422632, -4.06398971404470011, -3.77476436073045329, -4.59470597398523406, -4.09840351716439599, -4.70659949259461996, NA, -5.43639156061373363, -4.84569723329582569, -5.14288617313569141, -3.77861464292434635, NA, -3.15965427525071174, NA, -4.13769746946670303, -3.75160924217347036, -5.64318848591747901, NA, NA, NA, -5.01881935391544154, NA, -4.58072176409830867, NA, -5.24780238063293325, NA, NA, -6.05925526416991200, -4.06954507369406571, -4.94059993169471934, -4.91029106720995934, -4.13909876834332202, -3.90807187694470271, -4.56014942468784845, -4.13077087769058160, -4.80542116596631086, NA, -5.53184754362965236, -5.02800081046324010, -5.17781142633936931, -4.66756148886943389, NA, -3.29213812548085860, NA, -4.22258168578154525, -3.93213847766749591, -5.57198633938739718, NA, NA, NA, -5.04846990538827534, NA, -4.51964468866527547, NA, -5.16232592667950385, NA, NA, -6.08312193057372852, -4.21131076644369351, -4.84506698049589790, -4.86025031876219415, -4.17026943674439821, -4.10902556960325693, -4.46000961153514908, -4.12918167761290000, -4.84396353650235678, NA, -5.56690678863123090, -5.02701841518699588, -5.16305026866730898, -4.28757393909870022, NA, -3.39212015379553433, NA, -4.24880960513714445, -4.01316111647682838, -5.46633024068088957, NA, NA, NA, -4.98057425192318259, NA, -4.40818851763803377, NA, -5.07946338729506142, NA, NA, -6.06879241847965289, -4.26207748001459841, -4.72722359324195551, -4.85696033142496830, -4.09987148207308660, -3.94311550841197489, -4.30222666781996743, -4.11331536986350610, -4.79060780606914172, NA, -5.57171569457358729, -5.01239868398662392, -5.09867942474553981, -3.99084221379219173, NA, -3.37077032572485846, NA, -4.26793057257442587, -4.03896596376966421, -5.37552447359541752, NA, NA, NA, -4.91551807220964054, NA, -4.29379052148486640, NA, -5.03314179600268652, NA, NA, -6.12254878528761992, -4.27176434129641969, -4.62885965935168286, -4.77138419952661863, -4.04376676636474919, -3.76643272297982890, -4.13384350663673139, -4.12620603755714566, -4.74077671418459978, NA, -5.57883050852592355, -4.97888525454768605, -5.02868778169205122, -3.62294391942253435, NA, -3.38382890494202337, NA, -4.28759573023909546, -4.06051564224547246, -5.34055966579733976, NA, NA, NA, -4.87038314257265714, NA, -4.20266429459785495, NA, -5.02913828941677288, NA, NA, -6.11021730193110368, -4.22420189678937241, -4.57891257796119433, -4.76505152392753839, -3.95926232333391903, -3.69978711301329977, -3.94256273142383895, -4.13700955022440375, -4.69415301100743854, NA, -5.56376088196147389, -4.95181087270754894, -5.03736418015834087, -3.44709262552028006, NA, -3.31448152859209433, NA, -4.35606714330094658, -4.05714539917756056, -5.27486573419398130, NA, NA, NA, -4.79018631848883381, NA, -4.12256662744710489, NA, -4.96819887105903923, NA, NA, -6.06734296529482187, -4.19300970212736956, -4.48077747268083648, -4.70556289019719465, -3.84657804451472884, -3.42314190751812220, -3.73863349440376336, -4.15439909219299164, -4.58308661421629004, NA, -5.49762310342287019, -4.85488793470243341, -4.98814095439848249, -3.28483995954091190, NA, -3.18629795048356312, NA, -4.36868656837189917, -3.99872391793703263, -5.14883758958224114, NA, NA, NA, -4.68644589645415532, NA, -4.04763326633539133, NA, -4.93249603846806028, NA, NA, -6.05715543549970903, -4.02030663869676630, -4.37890624792248229, -4.56319161260267236, -3.72773973571704875, -2.99220073031924727, -3.59071932035961749, -4.16557077685214328, -4.45582796828344740, NA, -5.45567981779633193, -4.69654046003998271, -4.98127953486353814, -2.89894634275783147, NA, -3.17829683337910973, NA, -4.35989282841482950, -3.89021412547447021, -5.04140295099185565, NA, NA, NA, -4.70327977676911857, NA, -4.02872694211960081, NA, -4.91310578091534378, NA, NA, -6.09205271301768558, -3.92632357410789368, -4.30921791785003272, -4.44119706523042979, -3.73848811717059837, -2.42154294010187376, -3.47758078020544481, -4.20756776272293020, -4.32697405310178329, NA, -5.41026975776494812, -4.59078291576232633, -4.99299359287012567, -2.96916402947263691, NA, -3.08962053147394888, NA, -4.41981142910630354, -3.85037239362171846,
                  -4.93915187365491981, NA, NA, NA, -4.69834493954967236, NA, -4.05258280081123878, NA, -4.92635049753497078, NA, NA, -6.10206484988791953, -3.82687250587849404, -4.37417871011149728, -4.57526487986852182, -3.77199356782599304, -2.43915506580183683, -3.43045163354148386, -4.29594849873034157, -4.27453143282555992, NA, -5.41866113579979114, -4.58705286151155889, -5.04607825912670638, -2.62393178554300333, NA, -3.17971777201129724, NA, -4.47363277565412787, -3.80583066847410478, -4.78274014849346596, NA, NA, NA, -4.64961700021660995, NA, -4.07371753731127662, NA, -4.93643876015147676, NA, NA, -6.10856314637541065, -3.77622383929650152, -4.34275297086603373, -4.79201505939607486, -3.77519736283735607, -2.60470341758806878, -3.31603635703603228, -4.36088625132203589, -4.19473688368608233, NA, -5.37972629234461142, -4.49256464398820743, -5.06645888567476899, -2.58332924375427098, NA, -3.28111953802290657, NA, -4.49839393113912056, -3.76048319977943724, -4.75434109333419475, NA, NA, NA, -4.61133570131137294, NA, -4.21112389625255457, NA, -4.94298002225578603, NA, NA, -6.17487689876236878, -3.80648051489156192, -4.45327524156900711, -5.09770442948144087, -3.85715772046646332, -2.85472263375690316, -3.30589191006994954, -4.46759387799504815, -4.15415895782721112, NA, -5.33263378713900504, -4.44335861243012609, -5.08605592963051301, -2.20215036072900006, NA, -3.36334027819840564, NA, -4.53725955525077929, -3.69371396118032536, -4.62813178444928219, NA, NA, NA, -4.52556394563761888, NA, -4.28367397562626628, NA, -4.90792997024857947, NA, NA, -6.21591687565168449, -3.86881368733784381, -4.55780041571018835, -5.30713981971343873, -3.92743864955302735, -3.13267988666780983, -3.28666502558172047, -4.51576073839548897, -4.09693580060470186, NA, -5.25550961985919507, -4.27050813038530563, -4.98929925522583062, -1.55260171223289878, NA, -3.31896832203997505, NA, -4.57282704821327091, -3.58237226419977484, -4.57449156789600941, NA, NA, NA, -4.44582153332342944, NA, -4.37431941684936909, NA, -4.93530325222847122, NA, NA, -6.26495371767639142, -3.94416812836660124, -4.56842238186981486, -5.29535769968140002, -3.98315590993948421, -3.38252668603024098, -3.39209188446667476, -4.62850500954553912, -4.10351192904704565, NA, -5.17817738527018534, -4.24280909529085015, -4.89121257373397711, -1.78548675104824683, NA, -3.44081855617328003, NA, -4.59358975602618891, -3.56238306943470340, -4.54083155854022547, NA, NA, NA, -4.38891880063292916, NA, -4.48450339982288870, NA, -4.96476589332526963, NA, NA, -6.32635666046631950, -4.14634139158925130, -4.56424959459190127, -5.38318335462838338, -4.07608920623129478, -3.75448619971394182, -3.52593926928259593, -4.80279815978843239, -4.10634731744296566, NA, -5.11436426865529725, -4.28938335474829469, -4.81143883849979392, -2.36070708505789595, NA, -3.58857235850413003, NA, -4.61457719385029463, -3.50693688293913564, -4.42333592111328677, NA, NA, NA, -4.28802472382740874, NA, -4.50125079976805775, NA, -4.98082382833619075, NA, NA, -6.38570874447980596, -4.31030204333224631, -4.55527359564684176, -5.45310782341605638, -4.09223076081235781, -3.70923329255876899, -3.58818630518790949, -4.97713846691467587, -4.07330667690193238, NA, -5.01246112597550919, -4.18099745604763040, -4.67036996402559712, -2.96767617408516937, NA, -3.66451400362958690, NA, -4.58365149168374142, -3.47572609419475231, -4.23670783452281619, NA, NA, NA, -4.15960754145561218, NA, -4.48961190118099918, NA, -4.90484293139046645, NA, NA, -6.43397084919640516, -4.48141391457999738, -4.51987492839409732, -5.52960616790267867, -4.10550739949114085, -3.60661440801088418, -3.59903666111965093, -5.12435485499271781, -3.99058768889176907, NA, -4.94079454268506169, -3.96241753487272597, -4.53125754343832554, -2.89565794500371387, NA, -3.64408174101459759, NA, -4.55022903512624044, -3.40917635106388861, -3.98848470176606407, NA, NA, NA, -3.98843697474782211, NA, -4.42943147026383954, NA, -4.83882017378701601, NA, NA, -6.47868459549275144, -4.71345585865127070, -4.46642964870119386, -5.47247538012257007, -4.05936773893847658, -3.47825911188448389, -3.51202844958192095, -5.21197835323516756, -3.86992610092540934, NA, -4.78907546046090982, -3.64046259336417233, -4.31986963009411529, -2.52352705608844285, NA, -3.48022712504185483, NA, -4.47654736780874440, -3.24475607813545119, -3.95704118422704498, NA, NA, NA, -3.83834975627629538, NA, -4.33003356555941199, NA, -4.76573855702568583, NA, NA, -6.45453927969935304, -4.80498464594495189, -4.40661650202751698, -5.43151568365942250, -4.06687929074029242, -3.52648152019369121, -3.47187147132895291, -5.29183996338904006, -3.85024978163327791, NA, -4.60277419911133290, -3.65376866289483715, -4.22489895120929759, -2.65819822958972418, NA, -3.53772061101673341, NA, -4.43633653811823336, -3.23811913196684520, -3.94396679804290740, NA, NA, NA, -3.67239535188784139, NA, -4.23427353266138606, NA, -4.67278182276472709, NA, NA, -6.43886706870730485, -4.84491630150989039, -4.32877999611638575, -5.31352935792780023, -4.02268836482105741, -3.61870405996819233, -3.41584172300885580, -5.33278277550180846, -3.78758948951123298, NA, -4.42802019660497592, -3.72880805746301514, -4.10271898001869850, -2.82168469651237652, NA, -3.57231394978072681, NA, -4.38162567917877865, -3.14857948746976080, -3.95052067504399274, NA, NA, NA, -3.54145670723950001, NA, -4.15520056224859502, NA, -4.58216909630947722, NA, NA, -6.42092208324410851, -4.88312981590359296, -4.26626246991672264, -5.22568387983193361, -3.97980532933703701, -3.71058163777683658, -3.37075173495832736, -5.40503315863532130, -3.78232021479975300, NA, -4.27434082329376519, -3.76191067276223379, -3.99570699239490423, -2.95931881829327104, NA, -3.61658240938165232, NA, -4.34653633867359801, -3.15652836128124603, -3.99316185625247932, NA, NA, NA, -3.50169150733404955, NA, -4.11550572331282627, NA, -4.55878376603479296, NA, NA, -6.40266353456935278, -4.94025553195214062, -4.21890762729510715, -5.14759515477372975, -3.96527495091110316, -3.75087663957318407, -3.38125801968247597, -5.50524528031064708, -3.76186083937218374, NA, -4.28908323285987603, -3.80201273636809578, -3.97965224834038267, -3.06310425963955613, NA, -3.60855569029215317, NA, -4.30038285472274584, -3.18241843115668166, -4.06039403489033024, NA, NA, NA, -3.45551593339984198, NA, -4.09323234453512441, NA, -4.55432523839041359, NA, NA, -6.38629638594608817, -5.03049119273704015, -4.15071341081946166, -5.10456972759883154, -3.94969261664527505, -3.72776605281456597, -3.37719692565554030, -5.60128717377445717, -3.73062630471406731, NA, -4.36404303401131344, -3.81925140027995580, -3.97152539488896750, -3.24200758920523757, NA, -3.53424793606007004, NA, -4.24288874489894852, -3.18048646525314593, -4.09108705904018954, NA, NA, NA, -3.41766501592924721, NA, -4.04055136655944569, NA, -4.55945024447012592, NA, NA, -6.35712514974507847, -5.14378253471552860, -4.08399058642679691, -5.09635574500347666, -3.95535784681024571, -3.51442959987021686, -3.36047619941003362, -5.68500077059914588, -3.70376620229344722, NA, -4.39769207495968839, -3.81356472453334883, -3.99453857514467359, -3.07180989044326491, NA, -3.57662647697545566, NA, -4.19029315941274305, -3.20903629742466023, -4.15001444506931882, NA, NA, NA, -3.42415206482094714, NA, -4.05054146083803612, NA, -4.60224962196488097, NA, NA, -6.39485189149120359, -5.35931584333229072, -4.14726923130706648,
                  -5.14118377448364861, -3.96600713542273908, -3.44683517948783713, -3.45236720244488238, -5.75999797745949937, -3.71847040065816614, NA, -4.45244267270606464, -3.93073309658586512, -4.08754312940058995, -3.16509903008093785, NA, -3.54911216022594989, NA, -4.24821897824904937, -3.22345344016577817, -4.23632352423538183, NA, NA, NA, -3.50414250051036102, NA, -4.05034767030199028, NA, -4.62501694214351478, NA, NA, -6.37384031042276078, -5.44042083401901699, -4.16913596090093641, -5.22360817403865862, -3.98078440308715864, -3.56061528476809652, -3.49649239333880235, -5.79725576657526709, -3.75602009735973663, NA, -4.50779135804251574, -3.96735636843247663, -4.17905617019967046, -3.39972405790660437, NA, -3.63783413548115986, NA, -4.20361745306572931, -3.21262362154382153, -4.29790007307716060, NA, NA, NA, -3.53515516051148371, NA, -4.07226277317451757, NA, -4.66325108863807003, NA, NA, -6.32813360314669282, -5.41594877544569187, -4.17584982971236318, -5.43786096808888075, -3.94997780082925720, -3.56810510415082849, -3.49803325383304475, -5.80500181451997577, -3.80048646046823579, NA, -4.56922624757075724, -3.97513334481656777, -4.24581271402411886, -3.47760820824702499, NA, -3.72695671641461601, NA, -4.15233840203628635, -3.18643870198964896, -4.39398521877860748, NA, NA, NA, -3.44395889242976683, NA, -4.06188823235735974, NA, -4.66551270391372519, NA, NA, -6.25065917399216264, -5.42423527709996733, -4.16297310081532856, -5.76942004120063423, -3.87154518851342599, -3.42050947113051151, -3.46799880614655054, -5.80830283653268431, -3.79509489219416452, NA, -4.63580437592621575, -3.92850729452050240, -4.27149561271748901, -2.92638860410439916, NA, -3.75087989444979897, NA, -4.07749678811763161, -3.09617448023564812,  1.27848529414332712, NA, NA, NA, 0.82848067512050660, NA,  1.19018244539051277, NA, 0.80222646641824358, NA, NA, 1.16323233188522979,  1.12451339743768464, 1.07072525400963614, 0.80800657874863857, 1.10017494645685376, 1.20463147314849106, 1.12236040863820197,  1.20754476524094012, 0.94212989034968098, NA, 1.49042656196248546, 1.59196579318679787, 1.55555113825794789,  1.09526734168519391, NA, 0.94283138370730601, NA, 1.17806951674604088, 1.12664092156216067,  1.27848537696207964, NA, NA, NA, 0.82848152200330172, NA,  1.19018254242790245, NA, 0.80222664875646588, NA, NA, 1.16323497762195327,  1.12451318181327897, 1.07072540705333141, 0.80800366710330773, 1.10017634545090970, 1.20463160985995543, 1.12236062890402311,  1.20754487692558010, 0.94212966403006326, NA, 1.49042583070420376, 1.59196558058969595, 1.55555214845288470,  1.09526185903212925, NA, 0.94283088784731928, NA, 1.17806915623666941, 1.12664051717741343,  1.05706583996062786, NA, NA, NA, 0.26285042958685123, NA,  1.22743585159398916, NA, 0.48747847187020155, NA, NA, 0.45669590788491887,  1.05748668708221993, 0.93786643062735342, -0.41407513018465847, 0.87807226943854078, 0.35516521400571405, 0.99534788486436643,  0.13435528561513693, 0.59926597351925981, NA, 1.17814135317649593, 1.74125720777277504, 1.44482830572375520,  1.03997814603424232, NA, 0.39196541906216675, NA, 0.94578213889953178, 1.13304781734588578,  1.19296616986282156, NA, NA, NA, 0.49136609120843272, NA,  1.29492458637051544, NA, 0.78952965763244898, NA, NA, 0.73700398391339739,  1.23268701461736185, 1.17624222460484584, 0.01724199793397561, 1.09390291445421650, 0.56253345383840081, 1.19966969344170415,  0.38412277729873828, 0.71713195569773858, NA, 1.31274063779676364, 1.90107278534829804, 1.60388701788427901,  2.55614749504465477, NA, 0.61423548299885022, NA, 1.09832695676705372, 1.35072578626046558,  1.38291684732606357, NA, NA, NA, 1.06959865025720258, NA,  1.19875760275783838, NA, 0.87693427329835372, NA, NA, 0.94340537325956109,  1.22147002318594677, 1.35970104096600797, 0.47900844107280921, 1.29194269754549995, 1.29296198288499276, 1.32868068487278523,  0.66589974964797338, 1.15909825210623629, NA, 1.51473863188339286, 1.78183916463362513, 1.85200450155582952,  1.75561065832916929, NA, 0.93396609098140282, NA, 1.48146856986345377, 1.51407219223792633,  1.14079975778095766, NA, NA, NA, 1.12889980470702977, NA,  1.05764372262182027, NA, 0.87125085665653001, NA, NA, 1.04895647233458900,  1.12342734882452255, 1.34106133826083940, 0.61837918346462240, 1.22489962377869710, 1.22420528470105294, 1.26641054267592024,  0.93451783344033024, 1.07395651907862932, NA, 1.48528674096804303, 1.35502418672742819, 1.71337719142076605,  1.22161104832564105, NA, 1.00395993154693564, NA, 1.41765170300771648, 1.16867064793032416,  0.88762009951709953, NA, NA, NA, 1.10708273153606873, NA,  1.01048508180818697, NA, 0.93941549356417442, NA, NA, 1.19727932960271866,  1.06541861939538962, 1.36468606921817148, 0.76146789127561976, 1.19290916963682858, 1.27566366964047107, 1.20497430460423005,  1.17231239338829063, 1.04450649312344868, NA, 1.49969355224570089, 1.18853967372949243, 1.81296761872068179,  0.85970275105684335, NA, 0.94676247960423188, NA, 1.42803940810536689, 0.97515342887737755,  0.73616934216656005, NA, NA, NA, 1.24354228540046896, NA,  0.91696915028635440, NA, 0.86880406618209916, NA, NA, 1.39690877460729568,  1.07423051258941871, 1.37153435144025293, 1.04448993897352116, 1.32084397936972753, 1.21972882775913161, 1.06968056732044792,  1.48762625752555055, 1.05386806576819270, NA, 1.37890181514730314, 0.95359506921456005, 1.88699756443932709,  0.49979356346033976, NA, 1.09756834071190768, NA, 1.40686252541449042, 0.86871025303238836,  0.95315562973622348, NA, NA, NA, 1.68497408205451626, NA,  1.24744371978136126, NA, 0.98400776879154050, NA, NA, 1.66657754501598721,  1.24157967915385403, 1.42953166950219024, 1.75410381276102112, 1.55046593204527250, 1.10737376830933854, 1.36468425383720748,  1.90003509680525107, 1.42836167627355248, NA, 1.34896318221498168, 1.21601996024946413, 2.16849567831301338,  1.28448631214539910, NA, 1.49820730861079499, NA, 1.63636934732534733, 1.32500072200176522,  1.15772338981540512, NA, NA, NA, 1.76100217242854096, NA,  1.18064356160171879, NA, 1.16282414856306282, NA, NA, 1.51786840270063217,  1.32142168887099420, 1.40342392231013391, 1.69653238700263387, 1.67143024425980502, 1.38336301188775135, 1.48010196402084349,  1.56351612557668807, 1.46427524248630503, NA, 1.50702168713793627, 1.31554239859513067, 2.03354613423571484,  1.24180931201154188, NA, 1.60907795545775167, NA, 1.62513458971078695, 1.53532828397255172,  1.39499399128760215, NA, NA, NA, 1.76597719126400521, NA,  1.30702803713504490, NA, 1.33522683687670773, NA, NA, 1.65541958700544534,  1.42528860489348452, 1.38819268046010591, 2.14583038717587149, 1.84391916827648306, 1.52083243656800660, 1.73662321346772219,  1.38378026661162057, 1.55354755216909668, NA, 1.78928059135504292, 1.54966113136939043, 1.86698539763066074,  2.36743914628513519, NA, 1.72049872297655604, NA, 1.64738500928001885, 1.73208870667910286,  1.39985450672616096, NA, NA, NA, 1.83206436474730006, NA,  1.43402074993836526, NA, 1.28670156461739493, NA, NA, 1.53933002271239561,  1.64622408740155390, 1.38834059332403204, 2.14745602537652625, 2.12629864340876518, 1.97006309887983111, 1.72653124476230269,  1.25801458265648858, 1.75973576022507672, NA,
                  1.71170536514537086, 1.43845754882937316, 1.66869533725406183,  1.82766213176001080, NA, 1.89689361812462254, NA, 1.54181815395431876, 1.79535824004560096,  1.16543821994314389, NA, NA, NA, 1.70118700774606268, NA,  1.49068250367454924, NA, 1.34005787061338921, NA, NA, 1.42526251089913059,  1.75403710899225151, 1.39793211304225862, 1.89610197921283730, 1.92456586468410595, 1.78685115568980413, 1.51482331063077713,  1.30396700790896780, 1.55086606546540606, NA, 1.60065267537258826, 1.40079402359918914, 1.41954924759528778,  1.63981241808195000, NA, 1.73222009431565649, NA, 1.51115969113378190, 1.84604689912348863,  1.11177081658537769, NA, NA, NA, 1.57796939563687078, NA,  1.37785167373684181, NA, 1.32029515046686874, NA, NA, 1.62604593557539134,  1.76839472548413834, 1.41435473151215452, 1.61344927253786441, 1.83714211008953709, 1.70331069580078043, 1.25659891174501803,  1.10645299947499676, 1.55923807566931028, NA, 1.59409212097529762, 1.50835128287882858, 1.37139198767851811,  1.29539197678321516, NA, 1.57489167675212038, NA, 1.26884065276878966, 1.81939743772172036,  1.22358499658734399, NA, NA, NA, 1.86216063316806602, NA,  1.64128153013865208, NA, 1.33818915158470220, NA, NA, 1.80532050143304512,  1.69531775899667414, 1.56744023491625972, 1.51038681386541396, 2.02054645896718243, 1.90053163130028024, 1.20707215476676000,  1.45594005697937900, 1.64399048966505101, NA, 1.65330627370579464, 1.79976337500926942, 1.50945656220787150,  1.41790770428620450, NA, 1.29951576550679526, NA, 1.42348424467516410, 1.98340296359796397,  1.36051186890613396, NA, NA, NA, 2.16248736938836661, NA,  2.12795116638304105, NA, 1.31565999414271184, NA, NA, 1.96559378086375380,  1.59129243299225709, 1.50838988455698608, 1.28095985271227075, 2.13458009896421697, 1.75845109525086007, 1.18425212863236462,  1.16300508161313787, 1.53246253041085279, NA, 1.50341572540437740, 1.81052242261687413, 1.38192643188683184,  1.67783197416451801, NA, 0.93571260397845113, NA, 1.38568378775925183, 1.97681616918903824,  1.23959221649302598, NA, NA, NA, 1.76309467389521357, NA,  1.69973058027144441, NA, 1.11989548772342240, NA, NA, 1.75583793622690476,  1.38657867138613700, 1.13918068082974333, 0.72297821784837835, 1.73352175922981888, 1.14301890198237066, 1.01493152681066645,  0.50910456748874688, 1.28670421253202005, NA, 1.40329419203912353, 1.37385930469756934, 1.23656521347744053,  1.13085468893284169, NA, 0.81340006795707365, NA, 1.14256021901386928, 1.68229160725554783,  1.16630968886102093, NA, NA, NA, 1.84692160287430696, NA,  1.48473458138856418, NA, 1.03313531210496468, NA, NA, 1.69151031998892387,  1.24398073283962596, 0.81959002206467169, -0.13432038025970014, 1.63455188341274615, 0.21067627142961076, 0.94206888994060456,  0.64849261441411654, 1.00570706849883873, NA, 1.24578848025381972, 0.99789699291942324, 1.15096890340164770,  1.38881465246639557, NA, 0.53219793993106268, NA, 1.32710842215495828, 1.63160771778353109,  1.12323695176611404, NA, NA, NA, 1.76781865941892202, NA,  1.26831121447209494, NA, 0.96617321599108630, NA, NA, 1.54359576099753792,  1.11098622746811149, 0.86055229155479329, 0.05825514013207456, 1.49649526565979696, 0.35602156231622017, 0.91550883093189550,  0.84897334304178784, 0.93124745923632224, NA, 1.21975256683292277, 0.91434006517440281, 1.21517609866276999,  1.09359923922417623, NA, 0.64839319630125836, NA, 1.26583860969180995, 1.44610036043199308,  1.05750306060047561, NA, NA, NA, 1.68775899003667496, NA,  1.14264688278052562, NA, 0.89747887805039528, NA, NA, 1.41287519973831799,  0.97252715087020802, 0.87332986111064603, 0.48599744140946738, 1.47869148498500369, 0.72671070906890267, 0.88482091521085249,  1.07655341461688314, 0.86856430653521899, NA, 1.21241426252091733, 0.91052422605981786, 1.26240787268057186,  1.38037702361119874, NA, 0.80234538775677888, NA, 1.30628026674727726, 1.30999720669242348,  0.95251523549118045, NA, NA, NA, 1.52447080847044525, NA,  1.17167454369403545, NA, 0.85454311884432455, NA, NA, 1.29686468274599398,  0.83702209117429160, 1.02292788191571438, 0.85798655643118860, 1.46552103689640334, 0.98215687234810645, 0.80403069921999282,  0.90339052694922284, 0.85181722098498536, NA, 1.22263584696586047, 0.78796291245249195, 1.20875544094381548,  0.68941794091519792, NA, 0.85854081653659087, NA, 1.30208617322461762, 1.19148727302186730,  0.85458575723909069, NA, NA, NA, 1.46447129398330866, NA,  1.22930592918877180, NA, 0.84771960625525788, NA, NA, 1.23705039474568945,  0.70877636665517385, 1.18882420539568545, 0.98255780150628280, 1.53410738252762280, 1.23731121428460455, 0.64511622660919965,  0.46290542909439503, 0.80646686281718438, NA, 1.27964448999154490, 0.56530149910041139, 1.20728086215793895, -0.42013171952967687, NA, 0.67881357448758783, NA, 1.37401333665570857, 1.08039937282273479,  0.98719254369032527, NA, NA, NA, 1.48380271714830725, NA,  1.38491006110065440, NA, 1.10643288127890949, NA, NA, 1.22539403035979899,  0.72802583195979120, 1.24185249075161330, 1.03380309506248946, 1.64896030173412056, 1.49134295766188885, 0.83992493913319555,  0.67626946047679115, 0.95884918942250463, NA, 1.35293493966013401, 0.82882274883819962, 1.27787329062757893, -0.17538932398559587, NA, 0.86806758626037572, NA, 1.52432244161265862, 1.14662245755077774,  1.14762747719691416, NA, NA, NA, 1.64739550547444424, NA,  1.74538410994419002, NA, 1.42288126707995577, NA, NA, 1.32589316539933799,  0.87958953383039518, 1.38930244658806057, 1.39611111172626456, 1.98263767478696940, 1.95169516509450314, 1.11276539188478973,  0.99746146158308757, 1.14472366173469142, NA, 1.54723459735206648, 1.42552449334169684, 1.46111466530102452,  0.52104061703321614, NA, 1.17722335110556320, NA, 1.83871526230715721, 1.25718542954793677,  1.04493752425382702, NA, NA, NA, 1.58842796852419199, NA,  1.63867850016588013, NA, 1.56586355075080474, NA, NA, 1.30650669553086018,  0.97035772486409344, 1.35818058077519299, 1.54726478461429484, 1.77949907265912177, 1.70096624076859393, 1.22269035758219879,  1.15096131615787245, 1.13933670857194436, NA, 1.39595108411606650, 1.25648184234379134, 1.20881840048064637,  1.31263317280488812, NA, 1.31132164090453163, NA, 1.75211124172469601, 1.18692423295610960,  0.66509562846343817, NA, NA, NA, 1.41482475821109288, NA,  1.46827177148153520, NA, 1.28466628546504014, NA, NA, 1.15459953145765204,  1.04450405666910839, 1.17096739965712882, 1.55090077399634030, 1.60989144065841683, 1.34281206602566128, 1.13743437151006144,  1.01138843013032642, 0.90112413441840777, NA, 1.36239819424238529, 0.71079660259152466, 1.01781000591482274,  1.03697138161261071, NA, 1.18463628589866588, NA, 1.63125576388203997, 1.09162908293646699,  0.08329280553426249, NA, NA, NA, 1.19950318273099521, NA,  1.30646009643547667, NA, 1.01969909035355877, NA, NA, 0.96930360420056749,  1.14912206389803462, 0.99673498745173361, 1.30463801041141436, 1.46911309047333449, 0.90048194166012208, 0.93204077005214048,  0.85880194160238277, 0.63894941003885819, NA, 1.18313192610423501, -0.12810677899499137, 0.68366613600755022,  0.36343160686253217, NA, 0.76802334441478659, NA, 1.41306354614725960, 0.89311673085491039,
                  0.18705389483057153, NA, NA, NA, 1.16051991064865123, NA,  1.13781004058158741, NA, 0.96926959062923645, NA, NA, 0.93917804022542861,  1.18379840714204398, 0.97135376057996670, 1.43034377619575692, 1.55563103660427204, 0.93541437975533048, 0.89746837540034097,  1.14398828355799842, 0.74213289903231705, NA, 1.08380427952991898, -0.01812673066092099, 0.78465653638964938,  0.51439639490265865, NA, 0.86790675454507205, NA, 1.38704126278894968, 0.95631306070340250,  0.33646936080408374, NA, NA, NA, 1.02780631213538598, NA,  1.01399843825052760, NA, 0.80768632434973586, NA, NA, 0.90473738572752271,  1.14151526483239851, 0.89905117788194810, 1.31303885672989873, 1.41986634656703403, 1.04868949375090059, 0.83379411178976093,  1.18751113996159807, 0.68922897630159397, NA, 1.03636013921376824, 0.19165359715751384, 0.82787833318313953,  0.71728480012516205, NA, 0.92685007689163068, NA, 1.31439902424760580, 0.94516499874239579,  0.50541735857403858, NA, NA, NA, 0.96156642569455042, NA,  0.94072793836752466, NA, 0.78321682276514104, NA, NA, 0.90935397946915097,  1.11615728922003465, 0.79580445975256564, 1.15762626894008203, 1.30499358926747488, 1.16232337918364537, 0.81825382411780812,  1.04659386177657643, 0.72176247784488134, NA, 1.05626532316698185, 0.30437360815811298, 0.84833404992314665,  0.88504436940019238, NA, 0.99920875042595192, NA, 1.28198048615510540, 0.95730139959184901,  0.66615473920064150, NA, NA, NA, 0.95764161094499689, NA,  0.91945471921699895, NA, 0.81693211953552514, NA, NA, 0.91022892661756816,  1.09091783243429075, 0.65369066715351765, 0.98784981652722870, 1.18755341633591538, 1.17428685640614261, 0.80946418458306257,  0.93141799037263318, 0.71585224896918465, NA, 1.23389772350757787, 0.33280024083505910, 0.87227322573240496,  0.94139412684528589, NA, 0.95994490819489964, NA, 1.20388482526838425, 0.95335068195160833,  0.82655277684464112, NA, NA, NA, 0.82652271107714115, NA,  0.97117881164670905, NA, 0.86152069584460012, NA, NA, 0.90824969140124523,  1.07534798627423633, 0.43764858213912783, 0.76752276761041849, 1.02497138050535663, 1.06575063881670884, 0.78028160800120538,  0.61603597740353200, 0.64813349848442525, NA, 1.60051499227721106, 0.26091982080101611, 0.87331562361966608,  1.09491455786340630, NA, 0.76018551053965633, NA, 1.07389165144023391, 0.93503671820651313,  0.81071769742122723, NA, NA, NA, 0.68198531960448661, NA,  0.78968803418646250, NA, 0.87378714942781577, NA, NA, 0.77025963160470934,  1.10919822764778986, 0.23812500263577879, 0.69550986110498969, 0.92189989175972098, 0.61406339791250997, 0.67974769569899141,  0.65825228661843393, 0.55440879593904924, NA, 1.56179360149849855, 0.07342417660151568, 0.93644818819854814,  0.67188921858001893, NA, 0.83188384754581923, NA, 0.90285342077201758, 0.97209410519333983,  0.77342152092947614, NA, NA, NA, 0.52312481745782813, NA,  0.74601638945527537, NA, 0.94501303189873120, NA, NA, 0.80655218578823284,  1.24659037852502985, 0.36430781024826342, 0.55993463606104876, 0.78348364957474403, 0.38788626505672386, 0.83073005861783511,  0.28543534676323379, 0.50470404791382517, NA, 1.52790300170771132, 0.20511380781600236, 1.13545919517661309,  0.60956629782599636, NA, 0.72074954058875229, NA, 1.01800011243811861, 1.02979925743086498,  0.92989302046375455, NA, NA, NA, 0.64769855130716181, NA,  0.68854454375712448, NA, 0.96094828792498210, NA, NA, 0.79635124034613591,  1.30287666892364351, 0.42111621479365341, 0.74386544992228976, 0.73984056930111741, 0.60559082095637384, 1.04277865169697970,  0.47178091506764086, 0.53104150802903338, NA, 1.50204056252153895, 0.15225480417175979, 1.29730848702263990,  0.81180471775758245, NA, 0.93799419624758229, NA, 0.94682098721832519, 1.03884815834221600,  0.81996253798262630, NA, NA, NA, 0.51825973864616626, NA,  0.65080929658860720, NA, 1.02621899814492590, NA, NA, 0.67482937088061667,  1.31489971197155620, 0.38937967105479110, 1.02806219213402295, 0.45199065466788901, 0.62786441668634740, 1.08276047848331958,  0.73675840796934944, 0.51220340866768976, NA, 1.41514628645146501, -0.04740731011205895, 1.27986939509102804,  0.72827665335308656, NA, 1.13384288711450387, NA, 0.76095501180362191, 1.07204746469403189,  0.43055880316216144, NA, NA, NA, 0.05480458308444347, NA,  0.45610317108821896, NA, 1.00194427940270159, NA, NA, 0.18421624286878058,  1.30820838199585898, 0.24335673061708563, 1.21724372030615413, 0.02546561411128700, 0.31922431274089380, 0.86565102572839747,  0.86199709987571116, 0.34403210999390388, NA, 1.31996385440850639, -0.54590640454242134, 1.24061865804755778, -0.22770691870417958, NA, 1.17845912164297806, NA, 0.30100686467105398, 0.86141685287879832, -1.81562305835720506, NA, NA, NA, -2.54307412799816079, NA, -1.91952746160938803, NA, -1.83958705352087337, NA, NA, -1.90365583661223559, -2.71538677124348604, -1.79726749218641735, -5.71473804377745864, -1.81656334578221879, -4.56848345020020297, -1.78135055874982884, -3.61915179925795982, -1.77977157301279276, NA, -1.76589753515936110, -1.91908475698102099, -1.75833252625911629, -4.94764205513953303, NA, -3.08030902414023666, NA, -1.78657638571721322, -2.38307941551021374 ))




  fit <- ddhazard(Surv(tstart, tstop, death == 2) ~ age +
                    log(albumin) + ddFixed(log(protime)) + log(bili), pbc2,
                  id = pbc2$id, by = 100, max_T = 3600,
                  Q_0 = diag(rep(10000, 4)), Q = diag(rep(0.001, 4)),
                  control = list(save_risk_set = T, save_data = T, eps = .1,
                                 fixed_terms_method = "E_step"))

  set.seed(2017)
  tmp <- ddhazard_boot(fit, do_sample_weights = T, do_stratify_with_event = F, R = 30,
                       print_errors = T)

  # plot(fit)
  # get_expect_equal(tmp, file = "tmp.txt")

  expect_no_error(plot(fit, ddhazard_boot = tmp))

  expect_equal(unname(c(tmp$t0)),
               c(-9.71427794596020000, -9.71426863405032748, -9.66429675480953065, -9.67915761917839212, -9.73403049213075633, -9.66335711130821906, -9.61877410517220532, -9.62387310858449574, -9.68996238438255375, -9.73081241629477667, -9.77496786613567004, -9.76212742669888556, -9.73737739932531454, -9.68287097244324890, -9.67522956508486409, -9.63597070762502028, -9.60193341462220218, -9.58577658961859491, -9.59082653578726507, -9.56890909920737798, -9.57153661723432059, -9.56139212922683690, -9.57101034280360352, -9.61964594738461010, -9.62723584190070980, -9.61730461616751775, -9.57983980315896844, -9.58442737960460001, -9.58934202760303478, -9.57923977845451979, -9.57528935286363847, -9.56982991465971544, -9.55013513315960161, -9.56408247804876588, -9.59083617544932210, -9.60577743866973854, -9.61217728956017048, 0.05707023225184612, 0.05707009728904124, 0.07000163445718838, 0.06234178999257182, 0.06267819826696713,  0.05270503009874744, 0.05261645067973238, 0.05469016347588252, 0.05875554303413838, 0.05312454709827244, 0.05471897759760462,  0.04933273555667553, 0.04787088750921523, 0.04498919768431698, 0.04026614166681948, 0.04126425750354708, 0.04063532804796410,  0.04192048652003801, 0.03975591622572050, 0.03608974563459055, 0.03622088099251320, 0.04120239032896506, 0.03729746141318582,  0.03892161974347889, 0.03961237985284240, 0.04745629385463788, 0.05802333474247689, 0.04986402507731781, 0.04643304295969896,  0.04046903751875554, 0.03712852683076195, 0.04161213443389424, 0.04485179200324753, 0.05190302257118647, 0.05131571278543696,  0.05507876543323656, 0.06705493908295046, -4.19316035484239968, -4.19316041135647755, -4.08491653675794630, -4.12042858667208112, -4.24765885498851681, -4.09204798616696852, -3.96614811331380324, -3.92386316038442340, -3.97967880065180291, -3.98379760719253806, -4.06857349041381422, -4.01781252437845815, -3.92287138941047075, -3.82483735478484421, -3.77091090086097491, -3.65696704340467171, -3.50715266262089465, -3.44385237304692726, -3.41673514069495488, -3.32504048647217765, -3.30788362779748812, -3.25248520376567951, -3.28672764749959656, -3.36621942515345074, -3.33346989231375446, -3.25105810323073952, -3.08529201844371448, -3.05747679597121458, -2.99004239250518733, -2.92571371905870503, -2.89609589930691147, -2.87779211451528116, -2.84134584100959486, -2.91278266162065469, -2.98986129732850259, -3.04272953786446321, -3.02535135958150647, 0.98493670456071603, 0.98493525974407170, 0.75946682649811148,  0.89512785671943551, 1.12779904880385917, 1.00340478878604400, 0.93859402119159097, 0.91036914378935763, 1.18406400510857335,  1.20627277680977985, 1.36860564358572301, 1.39979316839532064, 1.40437919952739643, 1.33604489143911009, 1.45257214912918964,  1.48812837529106101, 1.21377975965025064, 1.11658590137993863, 1.04916712002905843, 0.99385163169306934, 0.91668814957664724,  0.84732017057955200, 0.96492030755171443, 1.20045697357863301, 1.14363909175140144, 0.96661997107741149, 0.69847347229837931,  0.77002292619710344, 0.75013659784306919, 0.74148733939666822, 0.73096254551217521, 0.71674368386874954, 0.62901198018686655,  0.67324836487899153, 0.77782077859174270, 0.76711629495806788, 0.58054519330920606, 2.56939954302423912 ))

  expect_equal(unname(c(tmp$t)),
               c( -8.371111987353447859, -6.848957554297902206, -9.840493036347698563, -11.894169475886377185, -10.220223475776132105,  -8.851848125780943377, -12.301788689054523829, -8.813247837376186666, -6.924243417591849692, -7.358988672087002136,  -6.710972674982070174, -11.275408321045954807, -10.149345088629205591, -7.607382978228278247, -8.660213506002419237,  -9.436645898984741265, -8.696446864530480880, -8.558450792341895763, -13.909631856208084955, -9.479298713733793846,  -9.940683217262090920, -11.045553183271819009, -7.011549009410757982, -9.754645109675637471, -8.890524669981839878,  -7.686823053319581156, -13.157074266476328717, -10.947588578420221594, -8.253581493683947556, -11.073931874065459979,  -8.371113223716731611, -6.848955218668512757, -9.840481884080805486, -11.894171027344400571, -10.220216693565662425,  -8.851848234403822957, -12.301784777750484068, -8.813245833479662394, -6.924239961361573492, -7.358987307520797216,  -6.710974329358205814, -11.275407485503079030, -10.149338416230440529, -7.607382101007724273, -8.660214918547991303,  -9.436642852046967178, -8.696440361761986537, -8.558447607331586937, -13.909631264607060785, -9.479296455223931162,  -9.940683513007439132, -11.045552976982012439, -7.011546467344384581, -9.754636752864044524, -8.890520968639975763,  -7.686820980448123564, -13.157070954219298287, -10.947585856452938913, -8.253577991450089968, -11.073925625150705443,  -8.341235375718653700, -6.822170241283526337, -9.791548060962620426, -11.820276286849258796, -10.161514214529574929,  -8.760861499758625826, -12.266973191803813847, -8.766191706683194695, -6.861585376787285639, -7.310587183380150123,  -6.602175235510678419, -11.201684180662208945, -10.074681684833286965, -7.583904937933461099, -8.620298172964425021,  -9.377222222866020473, -8.659429285649089536, -8.528512233021150024, -13.864763571236515816, -9.409200672857434000,  -9.878712284475986039, -10.965493189568459087, -6.997115839783013769, -9.724048629873067284, -8.836647396625334494,  -7.638324821060720637, -13.082259646744295978, -10.891685942106867913, -8.227394126761524262, -10.998830072768861044,  -8.352852201539825217, -6.842567090063390367, -9.833864955194581370, -11.834730810980566673, -10.187145745112406559,  -8.785775029954352888, -12.310246945039693856, -8.731912208027853950, -6.872292737293726361, -7.321910461968457184,  -6.599807389165242455, -11.188396507375157896, -10.073237845049423100, -7.578399432446318862, -8.621830286891484008,  -9.421283280515410752, -8.639412191484472814, -8.553041220737330974, -13.887780348476727710, -9.407231508835097955,  -9.920330568533854532, -10.975639714079193254, -6.997857630133008655, -9.721820903373201261, -8.880091177245368783,  -7.626235469214141638, -13.065039395565840152, -10.916967897601677961, -8.258037958624726116, -10.999045481449252648,  -8.358021841422045739, -6.887996984333291550, -9.909587362680374767, -11.855931164426500857, -10.276203485968995466,  -8.854147783365061031, -12.404980360669494388, -8.738544019327552093, -6.943868924052967628, -7.363581803615593557,  -6.643470719665544699, -11.183581266531115617, -10.131523336818251124, -7.590986479039161239, -8.657523534482356453,  -9.450220461492406798, -8.630646315095704679, -8.584577630889942412, -13.963030216858618004, -9.434891545174144412,  -9.967777976235081994, -10.995071016688786614, -7.009863392176981733, -9.752927872780650986, -8.970284021018787968,  -7.658194081591422453, -13.059287037989461666, -10.959400777970952134, -8.332718917202422304, -11.025560264498480478,  -8.287761648209311716, -6.804465866265613450, -9.893853192438886168, -11.760549669696459318, -10.217203627654107834,  -8.820984472841484347, -12.365707184831286725, -8.649543882553071938, -6.890642535824615500, -7.306456845671335465,  -6.562562856060331740, -11.103356985648872524, -10.085329673443322207, -7.507118026728761606, -8.608492442478837603,  -9.306139599365494064, -8.523962236726836394, -8.495775023696175765, -13.860343565180388126, -9.335092302858544855,  -9.897839455363714123, -10.887665988671930251, -6.909308349996066134, -9.624549582675864912, -8.911475260318793445,  -7.539440025168427262, -12.965100216094123553, -10.850777737137242696, -8.286678057453677582, -10.952994696207946745,  -8.251999356506427574, -6.754736718966057651, -9.902422323991761033, -11.703450350878256359, -10.170421789750719199,  -8.784052259057482104, -12.345154102620497838, -8.569074289981006842, -6.846924583429244038, -7.296995026284744057,  -6.463585301880064193, -11.044221248059244544, -10.052232595893007883, -7.456213429978856055, -8.573155701189119071,  -9.206440739613025670, -8.423520741719517702, -8.452720973756120415, -13.778610777679393351, -9.265848044364190628,  -9.868586303870243270, -10.824081160390697676, -6.833004940765657231, -9.524932380626358608, -8.869079196026081746,  -7.445004902698055815, -12.915829192441641737, -10.783044083688368531, -8.253582147074462583, -10.916996888582863434,  -8.251365655495316531, -6.756256082119983297, -9.939297575502990867, -11.743742213509483108, -10.158106879063101502,  -8.820671863948289726, -12.348030072222908160, -8.550106122849006951, -6.862332803157999983, -7.338512843309271894,  -6.423188910700194931, -11.058511977239586344, -10.006975062045547986, -7.433831106892712981, -8.554143810094748446,  -9.167525715827691712, -8.370714142236383992, -8.473847007860666025, -13.723788893813367196, -9.241453364876974774,  -9.908357678789919021, -10.867469916005399355, -6.843034761900555374, -9.517970953482429053, -8.863596276531042761,  -7.385080883844597821, -12.902561739455158829, -10.772745804387600543, -8.271749276686403363, -10.910189856532502617,  -8.262410876618782041, -6.850935954498377889, -9.946477497506638343, -11.834158447289468086, -10.216006656625683746,  -8.900967254622807801, -12.398238128659729185, -8.575704470132473389, -6.889252038415592416, -7.408514963355793626,  -6.499054466012873732, -11.164568841653874642, -10.079930608189590302, -7.492449300230514808, -8.555215623607342224,  -9.228727138692624621, -8.320103278868373309, -8.515733687212090786, -13.714490288350573266, -9.319592681723804617,  -9.978924366593727058, -10.930675377991985187, -6.861032836508350385, -9.530525795355957897, -8.945673306395473645,  -7.427389739847829908, -13.015972202360778454, -10.850628290886696448, -8.280078617373140659, -11.000596167964733496,  -8.283621340784188547, -6.894434957269321984, -9.968122542911807216, -11.867392207478015820, -10.248095083518776960,  -8.955565692288173452, -12.393016854282805639, -8.579190246255899766, -6.866831914438868978, -7.469290469664866983,  -6.548017082977785286, -11.214025341896485699, -10.115790437341884100, -7.536886248153082057, -8.555918692720339891,  -9.264004246705741963, -8.279998071658415171, -8.566195346657208276, -13.697498783425643865, -9.336574244236496511, -10.025995930420025459, -10.980190517110036907, -6.865517879058158357, -9.477472530847524723, -8.963116539664387261,  -7.447604889680556006, -13.073801175106531502, -10.899791811354798909, -8.256503719618871528, -11.043568745124252573,  -8.253160631260016089, -6.965978155650152992, -10.015855309841318999, -11.873237636637032466, -10.230913787783030955,  -8.965084512124469995, -12.429594807649168686,
                  -8.562861623962842117, -6.859376308428908331, -7.545403547529536681,  -6.564316161283805684, -11.270253486768746498, -10.159996537589647048, -7.572518824548532201, -8.584624350600451592,  -9.249818858998576587, -8.256300282424527737, -8.637686247511881632, -13.722698197185188462, -9.439915831532189827, -10.053530177228712006, -11.054384806492144833, -6.868406159039801118, -9.489197655653386576, -8.969220095633877676,  -7.501034830041279911, -13.167411083016082429, -10.932259060475860935, -8.253575547633310805, -11.052542364208340686,  -8.197948624540376983, -6.938531727092685841, -10.018018524478868869, -11.817152536143934682, -10.199363576477315618,  -8.924426429554477735, -12.397307614925221486, -8.541207588389060135, -6.813076272001084455, -7.545655099923902931,  -6.555176744918425946, -11.273903123338641308, -10.188937126636700370, -7.615269062686813584, -8.545586232584444275,  -9.203221380215316216, -8.194081247800284018, -8.619527393398286108, -13.660334963530170072, -9.454198055528376443, -10.070392527453737586, -11.016659979067339847, -6.843760753790732565, -9.409827076307344385, -8.901231104711200715,  -7.454875942786304499, -13.175939911416174155, -10.904827476104694384, -8.164726094630918141, -11.018490268910824170,  -8.144539186628989569, -6.910841049366768019, -10.005836944611779415, -11.753838018202475979, -10.132318941548954072,  -8.876016183029035389, -12.358120089399861286, -8.535168021310843400, -6.742631943246507831, -7.548379856729070525,  -6.534351861837582653, -11.255878387873698543, -10.182322258838381046, -7.652265074821777979, -8.487343236129975210,  -9.129393822903679734, -8.132949423102981967, -8.590209469878745452, -13.592577260799640015, -9.499704129831807009, -10.076738121243918656, -11.003121273107010936, -6.811460293902386276, -9.319732191214251671, -8.801905042896391862,  -7.380249896379730501, -13.174545651400448421, -10.842671869564497911, -8.073534829941111823, -10.953327730788339878,  -8.092164633966831389, -6.841438357203215581, -9.947599266838466292, -11.650545743769344043, -10.055907755801680281,  -8.816700160882767534, -12.300900615229029000, -8.490767242110242918, -6.602428826106010895, -7.544432993156067191,  -6.474025525975325657, -11.224496973915062625, -10.136981897947707765, -7.663083961539331135, -8.412209553236847270,  -9.010081048319610986, -8.046013544743452428, -8.560708941653153659, -13.528832299285959095, -9.482795550422171260, -10.064916707160259435, -10.903812556978524029, -6.734890475474466065, -9.226197275917995100, -8.702807080575786003,  -7.277315849742523213, -13.124202132031959067, -10.791096001529282944, -7.982814865056449882, -10.877477645027298081,  -8.079033805282184844, -6.847274037131857227, -9.988022437422667821, -11.610769062086609082, -10.037282733598884121,  -8.767487476310760641, -12.292868621828931808, -8.515290098629034077, -6.604603627044535408, -7.577005608821907323,  -6.480889115421642366, -11.227286753116123563, -10.172647331982869190, -7.679430812803108353, -8.411840150443346431,  -8.973555427863745493, -8.030874226227135537, -8.564325730852990759, -13.474218868248854619, -9.512493659159090953, -10.075097771971179839, -10.838219382369562283, -6.700703619804083466, -9.204708130879232897, -8.684684550922915136,  -7.221463501684472597, -13.110580373721640868, -10.814595154445582281, -7.963178957294737792, -10.845252020509061452,  -8.020002420534794396, -6.798731434557738318, -9.995314858847757833, -11.573801750445317893, -9.979595259360319659,  -8.694894375006715137, -12.245435165511757702, -8.496683913013532319, -6.560104147509240313, -7.570284355533136278,  -6.440691307258903819, -11.183048603151846123, -10.176208969444216379, -7.657255323719949658, -8.396105173923052334,  -8.889712627228874453, -8.022469235672046395, -8.515868744054577277, -13.425806399392305579, -9.497509725744620823, -10.084849459467806199, -10.755768863043721595, -6.642710073273481619, -9.147933467468838842, -8.677313121235826543,  -7.144671805877830906, -13.083787562597063214, -10.798229125520832028, -7.917026395397760652, -10.819579307044458361,  -7.994298071399570382, -6.740305926527588198, -9.988979974317372168, -11.544385435911589610, -9.952920887527717753,  -8.608084581662424739, -12.243260081550632279, -8.504966782473221798, -6.546838966543949034, -7.565715774481178713,  -6.417848702461116162, -11.105366433053868391, -10.103956450103719789, -7.627914467611331517, -8.414248703786800121,  -8.815923863539818939, -8.009709104337289887, -8.487648686993694014, -13.401189019230143273, -9.433275302613685298, -10.103339028703210545, -10.714615276012303013, -6.630956495824720243, -9.130891852483497217, -8.660980808907609685,  -7.062485428060923276, -13.039422452894225302, -10.763046967653670194, -7.877808417355473658, -10.801569799084234091,  -7.989798031042306903, -6.709883654228267602, -9.988311470045227125, -11.530869321930383720, -9.946377199773277056,  -8.541389488236031724, -12.251639445093116976, -8.509817591990083585, -6.541978673145840695, -7.585450741693533061,  -6.407832125427699310, -11.060012378890849760, -10.072837017980432961, -7.620680461507888914, -8.460309513742569010,  -8.745609054207916344, -8.016100951714840050, -8.466039415465218099, -13.397592683978819750, -9.406502194868780364, -10.141899605642592519, -10.678822354826086993, -6.617391076522228133, -9.135715021259464663, -8.662990062276850978,  -7.011504563477699570, -13.020614242823157269, -10.750316256202893328, -7.834966793145136776, -10.808912820405755184,  -8.008779437311602578, -6.707624100784515875, -9.981452885841658684, -11.517301809562475512, -9.958589262718030355,  -8.547763242077369483, -12.280384189711110565, -8.523836022260487866, -6.563988384270149190, -7.616653743378407171,  -6.431003796353000901, -11.040302199113360970, -10.051997219079634505, -7.625153878905870819, -8.529945563060113756,  -8.715810508917854804, -8.054890688861870984, -8.458969284214672868, -13.404352017947072540, -9.419560708848264241, -10.189606303089325579, -10.657873545127364068, -6.635914861027745637, -9.188416227454291985, -8.676008452441749341,  -6.981954170880625199, -13.003875834671681844, -10.780080684098019006, -7.846432960208271190, -10.836246988708317929,  -8.014051993766068804, -6.692561999063217471, -9.959526818228331635, -11.484545656910981393, -9.945485312817922363,  -8.538475716673451288, -12.283218409360562973, -8.525506881159648387, -6.545173301838328683, -7.614587979721996192,  -6.430796443302043919, -11.004110799561527045, -9.989074163428458775, -7.602604990947977370, -8.574381517986152446,  -8.649880727653579626, -8.066282057007782669, -8.430812492435006078, -13.375561792456723609, -9.387879200329447116, -10.218244116410469147, -10.621567262970135204, -6.626600037641706464, -9.217256671874046958, -8.668857254605256912,  -6.952854307167153358, -12.948207951866265830, -10.762752967423255157, -7.847115617445359348, -10.826861781951127028,  -8.053343004020396378, -6.694560632414400025, -9.975309504310210684, -11.469537196285687131, -9.940774857195391334,  -8.549949642883357015, -12.312333431740267287, -8.557263440764600659, -6.511095077482813132, -7.642577101229156789,  -6.447978647991718049, -11.016135360399674781, -9.963336109355353187, -7.597889101767384190,
                  -8.640981344900151129,  -8.604415201581890216, -8.077584283782098140, -8.424405250346957530, -13.373542845177205152, -9.387858601694544447, -10.272349633694815907, -10.618048946265945531, -6.623584882095938653, -9.274637311098581804, -8.689460915248325534,  -6.964904897633626746, -12.897009264274299056, -10.760656150679057674, -7.864695307198501517, -10.826320886294579893,  -8.059301926232345537, -6.701042758614017103, -9.977942165243417705, -11.442932986568735032, -9.925295864187603456,  -8.527096920754116738, -12.337416514701317283, -8.571161899871558632, -6.452347897213079442, -7.668910218182809935,  -6.449068667219433948, -11.017303958988966528, -9.924405872300656739, -7.572651945908612348, -8.704994182718062135,  -8.525750188030055909, -8.070794614995469018, -8.404293219700315731, -13.364117777071415460, -9.370846551711537131, -10.324873127303268561, -10.583592017503644200, -6.597261970637497086, -9.318600073646845061, -8.704245440956418278,  -6.974150543884805487, -12.839125738427991763, -10.745901597942822292, -7.855951166340312319, -10.802466233074953905,  -8.105624936684153070, -6.693267365579021799, -9.989392671921912381, -11.481017530130225524, -9.936650589742896500,  -8.544611968195129137, -12.336583662609015732, -8.616926461030113060, -6.445092520085640153, -7.706010366719588944,  -6.470835011843965034, -11.053043725938010766, -9.942527116149314281, -7.584049806302790131, -8.749647498110640242,  -8.525485279976424025, -8.051643093892916880, -8.417660567288963946, -13.386205217998675465, -9.419056178200566620, -10.358037574573010886, -10.594486452914054553, -6.591003934063486724, -9.352328582100303933, -8.737353464594734831,  -6.987571184369734567, -12.831385062179073486, -10.754784541541708265, -7.885642162555007495, -10.832116773914753693,  -8.176508698678786047, -6.752081737533080386, -10.037117752026757600, -11.545327012834748714, -9.979519779741124808,  -8.618801729788991395, -12.372897642825932962, -8.678830340882106142, -6.486621115818654637, -7.763043220053873483,  -6.525395473175254146, -11.113592166370294123, -9.980840275214378110, -7.634120556679533820, -8.812744367093902298,  -8.570845458945258954, -8.079086024021075119, -8.509617026508017545, -13.452587622900020747, -9.526719964751515946, -10.435377783506190497, -10.652363933547269070, -6.597449644328466256, -9.403276176155285881, -8.823345825769955297,  -7.056200229062945084, -12.880835322117791364, -10.783162781116676143, -7.968538149070315768, -10.905637604687814957,  -8.228350080449686743, -6.776790892111625553, -10.021844592997988244, -11.554233152485995362, -10.002243546034012667,  -8.669494260430926857, -12.354042778070594366, -8.720572334207433940, -6.435807951257605630, -7.765663830694585990,  -6.535409460741439247, -11.136658095325454809, -9.929701782852891512, -7.625246834180343214, -8.848013224974046409,  -8.560982480494489621, -8.087088838870647223, -8.557881440063708212, -13.492263112382250512, -9.574400463668617078, -10.475924827782192139, -10.679786918139567931, -6.576017219339360231, -9.434025720397796633, -8.864000995677754347,  -7.061828505093842523, -12.900622876272709760, -10.725857153102460018, -8.007983585964520756, -10.936728544803987262,  -8.240323142668470879, -6.772775938244224747, -9.997138573697201380, -11.566030440465354445, -10.011088700952784691,  -8.664328409739223247, -12.345142401545528443, -8.740574372034290107, -6.325358822099078182, -7.758282766745902670,  -6.540154663234042154, -11.131712440857310042, -9.845522669060667198, -7.597905274290083000, -8.877553827728414504,  -8.505535153325462616, -8.067874329407151990, -8.586930886821720676, -13.541736323159510036, -9.625383064233204777, -10.522238618626699846, -10.696967668984102673, -6.542785477410657968, -9.458187964955676819, -8.904387950827258535,  -7.075362277006631118, -12.871387080212070231, -10.637452845075831220, -8.036198669402187633, -10.951593571028112706,  -8.189387041249027277, -6.783153087341374565, -9.963613377025430040, -11.546142453038795495, -9.995209247773724215,  -8.625336176628414009, -12.314939407023116402, -8.735877181975407524, -6.177415251972740329, -7.736521277896240001,  -6.505524436630919283, -11.117319384201170607, -9.738916792997068583, -7.540779799718189125, -8.895157220763811878,  -8.378122139853418915, -8.000694717708469739, -8.593020943399903189, -13.560912243930262733, -9.631453953889884900, -10.543724944819345524, -10.673592796186721188, -6.465724313559535474, -9.456840382472611495, -8.911887977099198110,  -7.074516265511642921, -12.787112107368598046, -10.518765637029433790, -8.030053578558710825, -10.928066500275017248,  -8.220283021763174602, -6.802854048019127298, -9.958953466864526760, -11.553995135592359134, -9.973869763504662700,  -8.646991347323377042, -12.306306019068518509, -8.731976235803397657, -6.173616067775421712, -7.738849185498086314,  -6.525460628148112896, -11.134999050668261233, -9.717266186912834058, -7.555412707069045553, -8.909655926956400407,  -8.371678066221395653, -8.006102987825990169, -8.628166492234312202, -13.556926981639715990, -9.631311203418983169, -10.573731766065353810, -10.667212081146558944, -6.447165806922575015, -9.441617874969361779, -8.931768708622477959,  -7.074209306880042014, -12.797216209996928882, -10.511776429262880583, -8.052141950764164946, -10.961510205609503288,  -8.272034656209752868, -6.825986967050609699, -9.965562479394113993, -11.584783052196064190, -9.954060939648519124,  -8.696278483174145535, -12.297594999124465076, -8.727279030052409681, -6.192926199586326064, -7.736954639240486209,  -6.553934046541662006, -11.157012533006939847, -9.716276766889020422, -7.594716632718058413, -8.923446735438336219,  -8.383449344801519487, -8.017683587118623834, -8.648607935849133455, -13.557032303673810958, -9.624714334262085558, -10.600648756711716914, -10.654240724872927615, -6.460355634650690071, -9.419073589796292723, -8.956357670595888720,  -7.088992573078534143, -12.840146021839146329, -10.558197642430076968, -8.097755593216758641, -11.016920852928716812,  -8.303235946573526149, -6.829728684907532710, -9.958307130537273011, -11.583245874885726323, -9.905473886631607883,  -8.701510338854237858, -12.258701418748898959, -8.700222992343350725, -6.183468420844504543, -7.728105108892303399,  -6.544987259318400419, -11.153068712269059048, -9.705812318432956332, -7.613936145700952451, -8.925267924671022257,  -8.375203043576668449, -8.020090307123272311, -8.648966884712164571, -13.556087684626357159, -9.588709517478916311, -10.610585746627274162, -10.638509423948262267, -6.464794609139104864, -9.401513895805234000, -8.950398410158342699,  -7.098776440636615170, -12.839607875798247250, -10.590904679114656162, -8.114142877879906735, -11.035840281869777613,  -8.344417108940145766, -6.843606125694522646, -9.965772168509651863, -11.586455479557930204, -9.873205508881085279,  -8.704787966781807995, -12.237868607091145989, -8.686141260412689746, -6.184400621023726785, -7.716811960675387994,  -6.540806574819526098, -11.161773424944641420, -9.700226402418691052, -7.643408173056771560, -8.945608787983251986,  -8.372079239922671334, -8.016854375147897116, -8.652911412150018577, -13.565940812626875456, -9.569077981113874642, -10.627474113638948339,
                  -10.621511220536122622, -6.482149146259216010, -9.404345514683381424, -8.944180493280947886,  -7.111707850586107327, -12.841107993930339859, -10.616518520557265859, -8.130086550071091267, -11.053589338626204963,  -8.374783139357289130, -6.860615897218557357, -9.968399877371117768, -11.591064307192521454, -9.847904741064246181,  -8.692925049957713313, -12.219904411483469175, -8.669005402808160454, -6.186199256277768121, -7.706686319795205797,  -6.528122992334974839, -11.172920479831677909, -9.691228698671013220, -7.665679267048173706, -8.965469854365315072,  -8.367627958083911466, -8.008602846250287399, -8.648715613582249162, -13.582824036088895880, -9.553555842318020908, -10.643562830162187183, -10.601971348924791982, -6.495697738438539481, -9.411159649624231349, -8.936248971482335435,  -7.122915586439154900, -12.834901762572775041, -10.626084179548419328, -8.145453879822873944, -11.064794065199658135,  -8.394791179950939153, -6.867226043559129245, -9.949825601285137822, -11.578222805583461508, -9.809140728198096681,  -8.670993766177774731, -12.186500973287307303, -8.643762555599687758, -6.162928913861605906, -7.679355569418573779,  -6.497602916344145108, -11.160611481267716627, -9.650492355337741301, -7.679729560978792335, -8.970811940656691519,  -8.351714864877171962, -7.988530488401931251, -8.633542106299962882, -13.588588992152908830, -9.516029478584700385, -10.644147814379575578, -10.579086423167982645, -6.502627254867258344, -9.409753529874397060, -8.908987595084616018,  -7.112045475402462458, -12.794100856491123608, -10.608790189439844198, -8.143762961207389850, -11.044836414750553999,  -8.425682834875283689, -6.892008528716969451, -9.990078174458572491, -11.582510941711547048, -9.783116237159269701,  -8.668898267834334703, -12.188483357583288935, -8.648320428901268286, -6.189903781992677878, -7.706603596662707112,  -6.511229134503105698, -11.168971468110477474, -9.641503237936777992, -7.708185596478391410, -9.000094653392734401,  -8.388543107651848629, -8.009833883655078424, -8.653127546548287086, -13.615083028566370871, -9.535377394517910332, -10.681455169384443948, -10.621738688787543126, -6.532519003979638583, -9.438640456073912333, -8.917097904809070386,  -7.120881975837130007, -12.811551257217159261, -10.637468575260973225, -8.166669450942670494, -11.075862877139780238,  -8.448650060464039768, -6.901785734245253323, -10.022813296143937123, -11.604945898548379546, -9.779890716803425477,  -8.677994343723149484, -12.215174257719636941, -8.653195304186041881, -6.231081055634367871, -7.739276764616994875,  -6.539657083900481283, -11.198114746103046357, -9.661419460042859342, -7.751218267692322250, -9.019184143074216564,  -8.422783800091318440, -8.028004633586384386, -8.689690502442417142, -13.646957110810973290, -9.584838049261794168, -10.716175740123214055, -10.681055480969419591, -6.559737150562916952, -9.474622261530175393, -8.919852366519346987,  -7.124591197643400875, -12.842797690425847179, -10.663152850929094484, -8.199746120925803794, -11.116343424701158327,  -8.453544986144828854, -6.892770006869158195, -10.052808124548816338, -11.615708179378994913, -9.760056271228480185,  -8.661130855159351327, -12.226774175306090342, -8.647856505933749105, -6.245339316181760303, -7.769889059085330452,  -6.548614067551233653, -11.216124783460935888, -9.682953163507500349, -7.794185581347098157, -9.016561098868571378,  -8.435007275397566673, -8.047496776288516784, -8.706292767651246933, -13.664396888763231885, -9.615037826554164724, -10.735407632828136926, -10.717452825594623533, -6.575206791594420785, -9.516934501749082287, -8.911109970015818504,  -7.129586175625791888, -12.855545656746695471, -10.686521377179992953, -8.209779039855179761, -11.148302986305326456,  -8.442292348744924624, -6.886246698051775361, -10.061091987906602085, -11.603568554950063785, -9.751125694892877505,  -8.632630293863991611, -12.227333319012339885, -8.651152361116061229, -6.263971083338531365, -7.789750559400050456,  -6.534949762625217318, -11.224465772477657310, -9.709489427757342028, -7.807263839045068821, -9.005710223970412542,  -8.453208493254752653, -8.068326060922192511, -8.712321071566881514, -13.677033388465103414, -9.628901182220715782, -10.730157813956175517, -10.742017582032543643, -6.590413101386449313, -9.561358366080591864, -8.882636386368007564,  -7.133614598068945156, -12.831485314058191705, -10.686659384445215792, -8.199720446457808620, -11.147500287325350143,   0.064944574036336161, 0.055468989439886096, 0.053073381389885654, 0.058610710751819957, 0.063260006725601864,   0.053610343413658043, 0.059616727360604874, 0.065178936514281513, 0.054820559507496952, 0.057438125662161346,   0.052729033239091179, 0.059622631716981705, 0.060954120930683704, 0.067249788082575632, 0.070201695478340445,   0.076261807182388425, 0.060710421528301292, 0.054804134267166675, 0.060185668903671517, 0.045129922444795939,   0.057306431650943679, 0.059224140982002793, 0.072689496320843205, 0.071980301149962331, 0.051077309143529273,   0.057652700379270852, 0.077880115766981700, 0.062242776232138280, 0.046747034222913764, 0.074238260366210132,   0.064944583465758488, 0.055468949811361523, 0.053073213544367268, 0.058610741481077033, 0.063259914131034642,   0.053610340640855667, 0.059616670721743606, 0.065178909291513931, 0.054820507933684759, 0.057438105573692451,   0.052729061603387375, 0.059622628879029943, 0.060954028089121717, 0.067249771250939810, 0.070201721592870747,   0.076261761273402007, 0.060710325730132519, 0.054804092461216002, 0.060185661915098526, 0.045129896199559261,   0.057306452005984061, 0.059224150820250378, 0.072689453962474920, 0.071980176060616630, 0.051077266048906847,   0.057652671344350807, 0.077880065152738154, 0.062242741386824466, 0.046746978440190851, 0.074238171593866378,   0.081827602420535786, 0.070774712605519347, 0.064541329011080589, 0.065176614952358450, 0.078324674555472151,   0.074436934102852015, 0.077727773914863948, 0.072802813166601324, 0.067051400349745313, 0.069650722181916869,   0.067109223269171514, 0.070685139242944964, 0.072448133688896535, 0.087512110630151407, 0.088541463060800449,   0.085783240697459726, 0.069397234031609620, 0.064564398732739073, 0.071445242819676819, 0.051099688945865598,   0.068854650669281287, 0.062964464389201821, 0.095893080926704816, 0.094058168002076670, 0.066409089058238419,   0.072007348676406124, 0.091809848738574479, 0.074386455861366538, 0.063571806244801146, 0.085369582862228816,   0.072855953495116227, 0.063989450645936266, 0.058847355569829840, 0.059431419093720392, 0.069945595022538998,   0.064152640614277959, 0.072204023267218601, 0.065014648138439032, 0.060389684453240430, 0.060351276499128623,   0.059993534404888096, 0.060787490027510785, 0.064320715472182433, 0.075831438491557143, 0.078774166197590145,   0.079267019331710362, 0.063470784709267730, 0.060284523603442407, 0.066857528565371055, 0.047691683549959007,   0.066690656224376227, 0.058811698883930924, 0.082971842590450012, 0.079982658135818646, 0.060759718495921475,   0.062120373687856889, 0.078025763330524467, 0.069459407582020086,
                  0.056709262995946211, 0.076016443323750474,   0.072137379659940104, 0.065770217282393878, 0.056980471015737340, 0.057610918620965471, 0.074615971360126015,   0.059477820376772204, 0.075712605410806946, 0.071636742289465344, 0.059326985486581982, 0.061217223284274695,   0.060636990921219401, 0.053753183199145932, 0.065955621630338399, 0.071978622178345220, 0.087967139335267772,   0.084936438973549855, 0.071455965347806696, 0.060516955035580855, 0.077469866353905070, 0.052319640059840376,   0.069555709393398391, 0.062627668797099037, 0.078983324868095200, 0.078216202940824112, 0.062831590576969493,   0.060743707795103026, 0.074689124958741721, 0.072228080625248270, 0.052539089573957656, 0.073841250982846676,   0.062083739426310167, 0.053787515815335078, 0.047602951513067752, 0.052514729155225051, 0.061846063990876783,   0.045916748485754458, 0.063113061435403378, 0.060155886541783055, 0.051079728882912297, 0.051774073758487062,   0.049562358564478701, 0.048551738674392997, 0.053958964702992024, 0.060224766368243088, 0.072965154405123675,   0.072129013224485811, 0.056541137675695463, 0.053012930351106595, 0.062876007597970171, 0.042545080611324768,   0.058910917002117300, 0.057015403843268753, 0.068524085724837577, 0.064511958098402225, 0.047998419853130428,   0.048459974942087279, 0.064376013542957466, 0.060420195437461535, 0.041474484773210654, 0.064811223444899285,   0.057044330395645694, 0.051350942151721604, 0.044556289713084590, 0.057913176413764499, 0.058606633759397689,   0.046070325596639761, 0.061986896617540636, 0.057137872894485478, 0.051206677186201671, 0.048408779505659041,   0.049848833392989955, 0.054590910416164345, 0.051384425533613151, 0.056264565641753042, 0.065066964579628808,   0.065612718777692522, 0.050104072085220733, 0.054139596294539287, 0.056205644660206028, 0.042584050083617389,   0.058643949866775326, 0.062076574106302775, 0.067653950441228378, 0.064035849045759818, 0.044825236261748891,   0.044688491134811950, 0.062535917420517270, 0.061313963287337098, 0.038258907180688598, 0.063027527304792719,   0.060056689660142090, 0.052541300808944674, 0.051425786844018474, 0.061255678366014490, 0.056753416276703383,   0.046071015591056988, 0.068150491382795664, 0.052548323876768686, 0.054746502958810284, 0.047872921459944907,   0.051472169103340334, 0.059456207669162514, 0.058424626457081477, 0.060670868546191817, 0.066217754866293166,   0.063639920885112583, 0.050224077091525640, 0.055721113545803037, 0.058892146852962644, 0.047713766688016149,   0.058735557765081520, 0.059935311135744726, 0.063808332604414375, 0.064061174556933498, 0.044709150113906880,   0.046141885124789978, 0.062658165175035807, 0.059330135514331257, 0.040073216223396810, 0.071882824125577546,   0.062177839290195561, 0.060119454909133174, 0.057056894293128303, 0.066984865808265598, 0.062855866031690411,   0.047080567132678733, 0.068375083350950563, 0.058186348761042971, 0.059559313656826804, 0.054114656982163320,   0.050176571635722117, 0.067605621036634436, 0.066647742029730589, 0.065629092026563710, 0.073234664877155436,   0.062041403730269701, 0.053685527356017206, 0.055591398524331516, 0.065399302822631122, 0.051648705745546183,   0.064600126374907030, 0.065574887016912123, 0.064439792502090740, 0.067773759401964923, 0.050077258708510923,   0.051688372689737229, 0.067536719316352273, 0.063589679373768790, 0.047828363474941439, 0.076346646836931986,   0.052659899147133302, 0.050863251323282017, 0.053741752793587103, 0.057372028144863342, 0.056434227408430480,   0.040311041285914957, 0.063186727909731982, 0.053478597326610817, 0.054570999902056543, 0.053705319225909126,   0.046864528776030784, 0.062090431195863161, 0.060439283278170788, 0.052725033695769460, 0.065949173125359639,   0.055846402642814749, 0.049799152044464862, 0.051582836109057395, 0.058140213799698207, 0.041406669961621706,   0.058837432482532918, 0.057385985814372514, 0.055775251576796192, 0.056437530915185094, 0.039702577883904508,   0.044139988809582510, 0.059554525976766234, 0.056440370937446409, 0.044527330124097958, 0.068624308259858935,   0.048034239307752663, 0.055266486296422238, 0.050738242583799970, 0.059520696010060936, 0.055000952105845588,   0.047661696555858203, 0.062961587760811941, 0.052575178090571557, 0.052654335451789797, 0.057403470505276236,   0.054046999787702457, 0.063405894428893952, 0.056936950566794528, 0.046894163720388155, 0.067080060106692319,   0.056055042177235392, 0.048277233958566548, 0.055537268433661698, 0.066755124932936255, 0.045257020707951401,   0.057919187920429244, 0.056786678904423453, 0.052738632535936171, 0.059564721458575963, 0.042915280602010705,   0.050500859133160519, 0.064715495596179409, 0.055656712432645208, 0.041216566267789308, 0.070559285847409475,   0.041695933959242537, 0.047251252820947220, 0.048232554019053776, 0.052563666157531240, 0.048135737801915129,   0.039004807206725278, 0.056906297993287967, 0.046905557356737203, 0.048116780161592175, 0.045389190832153044,   0.050091318462525763, 0.055906813041184236, 0.048381165258049932, 0.042426086933415534, 0.062613731901867226,   0.047763085938088652, 0.043193037317334447, 0.048348616598006364, 0.056041421376961789, 0.039623088858170334,   0.053713478137095122, 0.050267716643953284, 0.052114914940945133, 0.050858135042574359, 0.034489769408084042,   0.043052653183781361, 0.058759089262337104, 0.048011567122563006, 0.034457550020164165, 0.067735721623641193,   0.041920513597565899, 0.049498695045516336, 0.055184455894192054, 0.049480188885466569, 0.048366873632134301,   0.033445205585735223, 0.049436946087583866, 0.047391182150686675, 0.045308512447294808, 0.039567515328978620,   0.047228993939582006, 0.056069268740084763, 0.046178187730982652, 0.041127489515217941, 0.062938344493982140,   0.047849081266350658, 0.041327509776860928, 0.045746058908570073, 0.049225223611189250, 0.036317489232115849,   0.054555126009104879, 0.049258713632106516, 0.052484541278527599, 0.043107166643420236, 0.036494465448529051,   0.043922542992190908, 0.063370880565892213, 0.049063091131991171, 0.032234500159691218, 0.061907080684288664,   0.039610890004096851, 0.041748500548874551, 0.052991297586801087, 0.040282158800520537, 0.045749642412108132,   0.034852245486309719, 0.049882820832220273, 0.047896722088822813, 0.045613331360417411, 0.038485196253130974,   0.044189215460369191, 0.051263302908765213, 0.044601157268021732, 0.038816677529832616, 0.061648960642506405,   0.041312309768561276, 0.038012888793935731, 0.047570367585226592, 0.047580646206837100, 0.030791262782326558,   0.051342570891894437, 0.038990838925691121, 0.049098988251021124, 0.042197443491792198, 0.036736893322907818,   0.036045649338398364, 0.063690402428801571, 0.054766757722823481, 0.032491369021554674, 0.058203582881771174,   0.034509846339998575, 0.039240476621132427, 0.047131899684334511, 0.031053146654844987, 0.042658954116230594,
                  0.028062184496135116, 0.043191777025835279, 0.040555992672236169, 0.040346452961342788, 0.037298460739547241,   0.041709834890818968, 0.047792670493461704, 0.042370496673301318, 0.037632563019044288, 0.054605410803152399,   0.039112163874551187, 0.033511956983148600, 0.042342111569908733, 0.036210769307936125, 0.029889063295735135,   0.041088707525980650, 0.031642039002657295, 0.043179825334704373, 0.035691618722336539, 0.026275007946211085,   0.031071732117139234, 0.053290552740671744, 0.048463777886893211, 0.027566442983987801, 0.048436814247661011,   0.034628734079715187, 0.037322434664644098, 0.044727812259616972, 0.033447785070575102, 0.043863867732382794,   0.031482414259143089, 0.040475919134114663, 0.036385762764411807, 0.038050262710367704, 0.038703181870346930,   0.041565853147297160, 0.047882259755776709, 0.046679863252975802, 0.041668298746180908, 0.058609624458828696,   0.040831966663307365, 0.040792796713727299, 0.040416746428172036, 0.034864477174726724, 0.034234450762689148,   0.038907779497976823, 0.032201417294091955, 0.047523925665023670, 0.035377774390575878, 0.029056837225576729,   0.036271438873333575, 0.049711633231271410, 0.049775279152479951, 0.034370722606266939, 0.053144201264950397,   0.037544695888031762, 0.038668449401225741, 0.044508615282142991, 0.029005594424557569, 0.041097239398647706,   0.029978595153169380, 0.039971807182634836, 0.038177022649190262, 0.038414666063690518, 0.040066094109970259,   0.036694805998622848, 0.043325151994130971, 0.043109410664899865, 0.045184208567333094, 0.058152306164878050,   0.041730983756170195, 0.037653582374503031, 0.040562296673433237, 0.028499689207684774, 0.034427730587916802,   0.034203603309645191, 0.035077935835303858, 0.046977679111024906, 0.034905155841632021, 0.029110613720317265,   0.029117703346768976, 0.047116433714443601, 0.046061698890161901, 0.033396399895408047, 0.047045497741904321,   0.035888691677114240, 0.044173487727240661, 0.041790250978764022, 0.025569835224550239, 0.045456756786982987,   0.037489578452596733, 0.038865981556041214, 0.039195097367162222, 0.040003804796216784, 0.039143278578463801,   0.042027298325531481, 0.042731074047662479, 0.040604285584679108, 0.044498612974527604, 0.056177243087378759,   0.043049618624318532, 0.044083417512688447, 0.039364766491708017, 0.028624662738860447, 0.040082677356821500,   0.037635186857969113, 0.036873386532137488, 0.051481662851322357, 0.043970465594934172, 0.030432613788993781,   0.031475149865815052, 0.050275486156833277, 0.052202136055848135, 0.037718498875138312, 0.051817809490509958,   0.032413299255027055, 0.038552939910551712, 0.038405118774823034, 0.025212331449832214, 0.045318621662045283,   0.028829133655537523, 0.038675352290788965, 0.033037449725512119, 0.041093129401144117, 0.043261114820304390,   0.040581327438460504, 0.040496540995172449, 0.042114163279910549, 0.042537159114426591, 0.055645113575532298,   0.038043630398815628, 0.043388742660195531, 0.031679213238290066, 0.031314001530240429, 0.038274618226689990,   0.038550177664627802, 0.028118134105687000, 0.048161831501141165, 0.041370303114525470, 0.027227302318744956,   0.024419676652037280, 0.052332060957253533, 0.047335482801425577, 0.030268545728780464, 0.050086637089429335,   0.037387891470743506, 0.032049634593522579, 0.037472475055582648, 0.022398866578834935, 0.042496736047603630,   0.028415753206849997, 0.033590606441562948, 0.032661782386649026, 0.038938006717515780, 0.039254876858809740,   0.036176143649100920, 0.038337902905982345, 0.035923052236547834, 0.037493196968913518, 0.051068950387672521,   0.033502605101777279, 0.039907401288392028, 0.028059783841126357, 0.026054754661362010, 0.033143564618509862,   0.036933835569818628, 0.027642781664912373, 0.044123854421935470, 0.035138098576585383, 0.025991666917932033,   0.024205679351009698, 0.046293314028675182, 0.038235330243555507, 0.023479289908727618, 0.044136122333415094,   0.042505071939138273, 0.030175815402251109, 0.034570754773551093, 0.026073356775572223, 0.046016027512173027,   0.033914237292657468, 0.032092348450947029, 0.033235834734025145, 0.043713289143261051, 0.035563898353736439,   0.037707652556380609, 0.036610885719327753, 0.031096501726928228, 0.041390741605587006, 0.053157472802315073,   0.034345765211283102, 0.043272773274114162, 0.028402308634599853, 0.022815162995991792, 0.033187367307259526,   0.035678821846511860, 0.027997333849682303, 0.047591186443044241, 0.034769297320544360, 0.026796063246700138,   0.026447171515579272, 0.046608286628939739, 0.031797964269005212, 0.022543722153404694, 0.047268111912850609,   0.059151947910995678, 0.035039228207308301, 0.035754817095959839, 0.030476345916753855, 0.054626207943819012,   0.054370271339545112, 0.034831325775986674, 0.039285781507419633, 0.052919627780937978, 0.034203502226326954,   0.039678145048957079, 0.036542651153342626, 0.031608785003851059, 0.046886701097611554, 0.058832240877427021,   0.039894469493840429, 0.049449717977545167, 0.038786371522083488, 0.023012935563203156, 0.040693645353976619,   0.038967226378950490, 0.034024237503046437, 0.054624278183852050, 0.033831360922803956, 0.037131109144296286,   0.034762979004443681, 0.051306335565336966, 0.029539883635667671, 0.022543746891512853, 0.050879444193117830,   0.052646631996470197, 0.034355274401330342, 0.034907489894587855, 0.027686990008224721, 0.049001645662809047,   0.048276118018197638, 0.030597188896274136, 0.035227130944327015, 0.046104875463601126, 0.032942824157282724,   0.038768779992737104, 0.034152917890952122, 0.028137107843836787, 0.039743671440807891, 0.056150854929150958,   0.036147237792867798, 0.048100771652418856, 0.033398138428113643, 0.023887844220463354, 0.036072021497398167,   0.041167072647343081, 0.029699519280497074, 0.047116433627633250, 0.028777350071344847, 0.033761480947172719,   0.034286290202018607, 0.045340095476982173, 0.028249155738853372, 0.021779648301983000, 0.048256966127048745,   0.045880311886240532, 0.034536402440444709, 0.040752117464834224, 0.032461123756639079, 0.047583336938725171,   0.043749786838268169, 0.033903040274587265, 0.035844881519849929, 0.041053361374653451, 0.039372014442207906,   0.043342540282721276, 0.039314519135655326, 0.034976792629516251, 0.038952054461607002, 0.060884501651016949,   0.037030822626934627, 0.050438221943440886, 0.029084730205191062, 0.028115989113484355, 0.034979372084096724,   0.047243511366806656, 0.028329262968344419, 0.045547473010870788, 0.029302307242141785, 0.036105165084711711,   0.040645354496635908, 0.043301741134484178, 0.035902980641046967, 0.022905834381648324, 0.051912439901730653,   0.046715697453822563, 0.036345474899057553, 0.041256914037060756, 0.038869817396982170, 0.048909190193713814,   0.041639126454231588, 0.033886606125275817, 0.035269323715315500, 0.037736371485327318, 0.039880327657970754,   0.048816851498241637, 0.042397630942704545,
                  0.038261952612111488, 0.038061977619741692, 0.063755113957017523,   0.036523666529953036, 0.050013942804928310, 0.027951577558751269, 0.032855166059136992, 0.033167732345161530,   0.045002586771829958, 0.025958604386114054, 0.047044028604471634, 0.032767477907546999, 0.035890788458895699,   0.041205583310995872, 0.047347470068506273, 0.041196527424360682, 0.027159902361850903, 0.052384165992902576,   0.058563227278004233, 0.043081544351818074, 0.050429910466322787, 0.054619029518469718, 0.055180866244263188,   0.048212658270599035, 0.043047947656253360, 0.039338362889211595, 0.043399528241679391, 0.046657398878258247,   0.061240271807428734, 0.049996533888958701, 0.050672775565891531, 0.048237609896174177, 0.071596733743874055,   0.043409848182529055, 0.054876756691646482, 0.032312966852504697, 0.045513530524133705, 0.039789263696191020,   0.050702891679414469, 0.029950611230845650, 0.059109823462363584, 0.040421288550785119, 0.048076834305969901,   0.053559101281005758, 0.059603122602429282, 0.055338731917757469, 0.041130810802847362, 0.063696201946940645,   0.078400179923261176, 0.058878084353450030, 0.063277584365065323, 0.073089503114659948, 0.065915121470130844,   0.069925969108598524, 0.051758771222207345, 0.048750813642875535, 0.055043185444629189, 0.052526791482911742,   0.070975030857967414, 0.064367580943101840, 0.064579613013620252, 0.066269196619808651, 0.084909142326701284,   0.054228737995449645, 0.058523368105890677, 0.038717227654038025, 0.059926362237687235, 0.047719877370394140,   0.059821206440201030, 0.038452888740489070, 0.077719252054686591, 0.050562912765806406, 0.062655358475219095,   0.076332612005467065, 0.076441709115627643, 0.069943132108445660, 0.061912361558225062, 0.082011127071376075,   0.068467870099361491, 0.052670810000071247, 0.053931461637183653, 0.063820822210931716, 0.060433787793837561,   0.061307279603216837, 0.042975253865978939, 0.044358109735870482, 0.045270133587310045, 0.045280193700355763,   0.063490362083400961, 0.057596717224732932, 0.054589601143932125, 0.055450451312534878, 0.078473731895435120,   0.045599887938414671, 0.050620118977475828, 0.036307717925491950, 0.050783157241665357, 0.040628341173495508,   0.052935938443644290, 0.031890304039691848, 0.065834284388218298, 0.043308286737361118, 0.053069265042266708,   0.066716821772300483, 0.065661151502287310, 0.056120673892745755, 0.051592668400097547, 0.072130819208917368,   0.058837497398500992, 0.055747866300907403, 0.044773190004677561, 0.055173887721310856, 0.062177864721658066,   0.055763228837663222, 0.039826464402511773, 0.048405916741961520, 0.035820789533216491, 0.043800005409068858,   0.058956247446336874, 0.056001464802227181, 0.046246407732949364, 0.046128403915002635, 0.077931171921805806,   0.037356410149329562, 0.044966057137129090, 0.041605849110991536, 0.042646416116497429, 0.040578999102114002,   0.051705707795575588, 0.030649343429962322, 0.056685365965639424, 0.042069668379702788, 0.049735586115118656,   0.057606662666700982, 0.056306364011277567, 0.041753046409853049, 0.042714626127343100, 0.064104168876456397,   0.052969656034577012, 0.049557159477504206, 0.041889510072161608, 0.046157357967069816, 0.057591367489850352,   0.048514422857567650, 0.033083654552747963, 0.042817740718611129, 0.029235533074652917, 0.039038271479615354,   0.053148782855892682, 0.053376048136487955, 0.038627692889177848, 0.039265386249876397, 0.077856299272884483,   0.030953727732175182, 0.040396810932580414, 0.036170076176808209, 0.036908346107737015, 0.036192057226679578,   0.048044416525416488, 0.025053900037147331, 0.049165581493501267, 0.041876148572341530, 0.041232052352778117,   0.053788907821624259, 0.047208791308462256, 0.034051866829100767, 0.032119134356230421, 0.056942251972976643,   0.048262495929892110, 0.046227081470398862, 0.039559000910414838, 0.040546216539457475, 0.053512741252474377,   0.044903940467728096, 0.026371596945500928, 0.037733361207862748, 0.023657848559534771, 0.039949307551742447,   0.048713528246681820, 0.053347302422941695, 0.035328517469467031, 0.033495521739984283, 0.078301987628769221,   0.026999913283591897, 0.041133289938847580, 0.031872436498906129, 0.031550556713108080, 0.031924957755097939,   0.046919996502796393, 0.022521074144193003, 0.043646856836510248, 0.041107471454002592, 0.037840974813635478,   0.054456212966321089, 0.042581169055457863, 0.029417661684575807, 0.026613932119842906, 0.054758588495045162,   0.050176699522517604, 0.051449743168816314, 0.046382487369486747, 0.042313908622791663, 0.055681605929926478,   0.049848512305034939, 0.026284495939554238, 0.038501139974895487, 0.023911410610826713, 0.048723909102645624,   0.051560395583229349, 0.062463218177714462, 0.042380222566636067, 0.035521159745124042, 0.088946493891464945,   0.029162149267545456, 0.048810719895942017, 0.033756443860882140, 0.033469088605416572, 0.035808189154258721,   0.054190222325966403, 0.026225789781155525, 0.047491230023935183, 0.049927873827275940, 0.042934932001749769,   0.063704331565917746, 0.048781255068279766, 0.034500006634840145, 0.029799572604102199, 0.063675248262190259,   0.052713018274386673, 0.052360290705436699, 0.051502762158574029, 0.045909682905948046, 0.058013678937605617,   0.052073572198438736, 0.027745293496601823, 0.039246964521158244, 0.025786343063103512, 0.050739639951542043,   0.060680266086028042, 0.067907125643977120, 0.046077873046457989, 0.041515699770249770, 0.092015123604913734,   0.035977542636517083, 0.053792166615758891, 0.035052036969469000, 0.034676725926915036, 0.038891183402801413,   0.057853143998133343, 0.036192621374397882, 0.048017212649708256, 0.055674505431252061, 0.046564046070774322,   0.068845174310082871, 0.051846924390449023, 0.036503417553864548, 0.028260034652418685, 0.065496810942661479,   0.057130878063193738, 0.061638469799366047, 0.054562952265368558, 0.056230802024512014, 0.068542696267603717,   0.056224112311092374, 0.031324038900615324, 0.041535257021791824, 0.027726474389488942, 0.054060953711832241,   0.070756089094484262, 0.083967372725099687, 0.059304140904990532, 0.053940888426099308, 0.105483157265245306,   0.042733937949375550, 0.061175387192742715, 0.037134738543338049, 0.043116230179054055, 0.042533488297378436,   0.064588629851264082, 0.041751037508363821, 0.053617368974661905, 0.071372871720052286, 0.053492043573694602,   0.081891046803073145, 0.056118233518634777, 0.038627132491546423, 0.030837825235129391, 0.069590751781594018,   0.066474373162033737, 0.065353392505190566, 0.050961397977573764, 0.055438360347797394, 0.069683546902394686,   0.060799407467563071, 0.033522987461532747, 0.043537505567293751, 0.028491662567726814, 0.053827883722046388,   0.069765455346230881, 0.080125656463467948, 0.054448984463489745, 0.055977068149887066, 0.105853639402274505,   0.043082437597911624, 0.056748702629156780, 0.039570305073415836, 0.049134549110807585,
                  0.046425362619651189,   0.064285162368478324, 0.042743456788282494, 0.057203245166070713, 0.067701979388775577, 0.051965881052746257,   0.075529974537325364, 0.053324993444184107, 0.038912494850958455, 0.037240389016360197, 0.067419955694558553,   0.083111749387523012, 0.071154920258967497, 0.050187607367291547, 0.058267415005257811, 0.076254814356061851,   0.071854530604595374, 0.034515329449935135, 0.045982856247691981, 0.034702498627455738, 0.055699175724775896,   0.070192499805709935, 0.078780106274032854, 0.047329750495316014, 0.062521133263192391, 0.106683202912326358,   0.049448194839024184, 0.055551609682310002, 0.041227705752478840, 0.053973814130385510, 0.049845828603707662,   0.072155861743805366, 0.046108890628391713, 0.064505195095291248, 0.064302041469722460, 0.058740216315219439,   0.074452367466043168, 0.059717313709775753, 0.042140721463571285, 0.046131932394762848, 0.073837260785529046,   0.111614746389441591, 0.090457758594305188, 0.052005019044807810, 0.066660087954308389, 0.091672794448468986,   0.092545837207763909, 0.039897039411427485, 0.054994364810843778, 0.049264756923091638, 0.062780507629757468,   0.076805734296398337, 0.093813270680351127, 0.049122033536173032, 0.078779902335731899, 0.116959682610985688,   0.066777838074911611, 0.063530845992403898, 0.044466279874521966, 0.064808523618883687, 0.059995339492158739,   0.088107357358490263, 0.058361246992199342, 0.084819914973837632, 0.072440744462079626, 0.072939579738397334,   0.080683978388700367, 0.072962464441177960, 0.050684390978377784, 0.061223072971632368, 0.088595563765189261,  -4.156547678737422480, -4.545909236799782427, -3.839489800283417242, -3.970796339072161896, -3.933854322685995442,  -3.745423170024539061, -4.312841868705550752, -4.575894277529727106, -4.096421801029848986, -4.623198554277585970,  -3.667762237893384736, -4.442806340411779686, -4.587592183875862872, -4.469273028218855082, -3.317550066099983486,  -4.505944351311461205, -5.008339761921895672, -4.503378779356657979, -4.349690452462246171, -4.050616019121323852,  -4.073484053539020877, -4.362437423698007244, -4.507227648088896643, -4.378734114089544605, -4.640988881703212776,  -4.349056698192836024, -3.518762260472581005, -4.666309836858219207, -4.195167478101256897, -4.359935475889177070,  -4.156548308092668620, -4.545908605984485540, -3.839488537920173972, -3.970799776807237613, -3.933854502593048252,  -3.745423139830196479, -4.312842030933651927, -4.575894256514128955, -4.096420708771271180, -4.623198361139946755,  -3.667763608136412223, -4.442808760076302654, -4.587592940544265474, -4.469272937278170765, -3.317550750498826417,  -4.505943818211814111, -5.008339715471967324, -4.503378989228631113, -4.349691286502585186, -4.050618053219120718,  -4.073486061670202218, -4.362440131630139994, -4.507227240382867706, -4.378732242832302113, -4.640989253156259764,  -4.349056668181882301, -3.518761635004365118, -4.666311024608864599, -4.195166281865495783, -4.359937247863370224,  -4.064636172929979807, -4.455191276876519169, -3.716593325462508979, -3.825602278520411126, -3.822434972954139365,  -3.537505649276735387, -4.235310551313733463, -4.463067039845164174, -3.955650822678440370, -4.515889439810668371,  -3.428771727954502246, -4.289026924051963618, -4.434642128898320657, -4.392865155181850945, -3.231751939365725690,  -4.367803598144514687, -4.955992805154906478, -4.436156460829422166, -4.258789497320868200, -3.914059757394615424,  -3.937122676890868700, -4.197308467623385830, -4.462276478103392563, -4.278822833294375627, -4.509199862057353236,  -4.256547243920947743, -3.342307621464713829, -4.546520889522149389, -4.096152339519021801, -4.189353695262223454,  -4.078909449542375221, -4.482015897363252677, -3.822209083493721682, -3.870433479246147890, -3.878151211986775149,  -3.566821628179955628, -4.336942693165224405, -4.399792249122098653, -3.987787196171017623, -4.540501719259772528,  -3.464824911475465896, -4.276167302459448649, -4.431798158996405057, -4.364105859511330898, -3.250145449555303934,  -4.462850910884537114, -4.933346360922458373, -4.494366065844157987, -4.324046493906106292, -3.922972270288509478,  -4.052307074078988158, -4.237523966107632489, -4.453233907618361798, -4.290013673108010295, -4.588607319176611909,  -4.240162514603121657, -3.323803504647907747, -4.631586708904242400, -4.152931354161516708, -4.202009179586522869,  -4.112743270956266350, -4.584094989318747260, -3.994079664825574305, -3.928670295369446563, -4.059377835380795219,  -3.704293868886924468, -4.539475828165167037, -4.427676920563778040, -4.196949783145466739, -4.632506939907813148,  -3.657368452158753502, -4.306186277322837874, -4.553066846759622699, -4.428263109397473585, -3.432099839632640759,  -4.540330476359243583, -4.934421110339504146, -4.585415429263341558, -4.478567407425853908, -3.990729626948321229,  -4.159727273957545712, -4.301629011600997643, -4.552636712654178908, -4.440883664414096543, -4.775970908225945166,  -4.375196202217859209, -3.320268579065105641, -4.771024454013948812, -4.342680118990745797, -4.271404794224931578,  -3.969112029261637531, -4.411979670993611435, -3.949692807473226708, -3.723062485301576796, -3.909064969231267117,  -3.596469814644978058, -4.453697496497846409, -4.237602586289815676, -4.102398689430391876, -4.490512638803298096,  -3.592618161805857913, -4.153410920978861220, -4.443017001790782849, -4.225482923116789635, -3.382276519213320398,  -4.259291662492180386, -4.736948714239931135, -4.359024128132273468, -4.295183015151875949, -3.779017359665672871,  -4.001733372470718919, -4.085556397664186434, -4.375425787489587215, -4.202878226288566665, -4.639291895573358637,  -4.148687290335995925, -3.096378148224347537, -4.541491726321211786, -4.254084154707035381, -4.098099978615162620,  -3.849594923309697769, -4.261186713610446652, -3.924744161795664432, -3.550771455497710338, -3.777248987125443769,  -3.465663926272517958, -4.404470513539454402, -4.057804991628373159, -4.020415698163285256, -4.409116845782065752,  -3.498575405304841457, -4.029908678278359524, -4.361291301506151186, -4.051616028509062772, -3.363006874399997237,  -4.023403997169887170, -4.565360926512930462, -4.194478814164522618, -4.144399536017006724, -3.598161079882848945,  -3.879561208204198586, -3.909633871160155927, -4.205192650684082878, -4.012515683021636193, -4.538364040449076597,  -3.973936611831791588, -2.920341617570098691, -4.394489862921451717, -4.177707402878772847, -3.960494121227121411,  -3.778773753877042729, -4.203149345869503506, -3.941393331936399846, -3.566957380727198590, -3.700745631021141957,  -3.465280478120788210, -4.375234573991491516, -3.991848617912005626, -4.068272817456898061, -4.406747577682216921,  -3.520764116359420104, -4.071362947194700155, -4.217432254734275077, -3.874820323824786250, -3.347491040979491128,  -3.894367089871435539, -4.480467518147401407, -4.170785443583802987, -4.033018121511839915, -3.490911046141084384,  -3.890272718783659212, -3.955220010253404084, -4.212000696091993568, -4.006406149326455335, -4.494563925920505021,  -3.860024004073913328,
                  -2.766501615181256035, -4.352122933664301030, -4.204913973358371138, -3.847234221988770742,  -3.709330553617389725, -4.293051853482417357, -3.890931457765351098, -3.643481188234900969, -3.767226519830088716,  -3.522547402939994576, -4.403366856821277153, -3.995581121948115655, -4.138694239739093561, -4.454790014260213482,  -3.745188409125231654, -4.289466836376121250, -4.283537310571423617, -3.825398331983468125, -3.357903698792277236,  -3.931041439221758971, -4.403158628293576626, -4.145340729337235075, -3.973334371745021354, -3.565376087269096494,  -3.945542412478530103, -3.982550287984719883, -4.188831850390613454, -4.013358447354218761, -4.598267531531144314,  -3.927521257155299761, -2.855468337055021344, -4.486886677515078858, -4.208759606727101144, -3.928477368560521654,  -3.634886313953284542, -4.307173442776811356, -3.808421985258670261, -3.630988874262868649, -3.786828895622636360,  -3.546797787779715883, -4.348069838555590927, -3.929025597603242659, -4.071672489299070108, -4.469506443584213784,  -3.854114207968491801, -4.312095849055815222, -4.268571268040352251, -3.793705328694696011, -3.336552836939514766,  -3.883517865420658577, -4.318448445519486256, -4.121155129544481710, -3.965147613792626480, -3.546116445433350250,  -3.935874638388096258, -3.971980070773896809, -4.156504519183219060, -3.931541130736258438, -4.522613606107379169,  -3.898263290424902827, -2.914765059313873419, -4.457040425910871129, -4.154141288309691760, -3.951662507032548177,  -3.549884421834451409, -4.432246398910635321, -3.852284782268918573, -3.643855087940679915, -3.817495010972893787,  -3.571276884882977232, -4.415646049889978109, -3.906323086698937352, -4.115882734372839913, -4.589947248165413107,  -3.974029293387034478, -4.424166962149614690, -4.332359770643880559, -3.803552911246049550, -3.416820204430914831,  -3.844116090765396176, -4.362155637448497991, -4.206167317273843409, -4.104953028774150248, -3.734878225211415881,  -3.953929367669769768, -4.074094426097543575, -4.172579550902965728, -4.012104283918255021, -4.524808518563827597,  -4.044792658743008751, -3.123486596345022903, -4.481821323683045755, -4.192500575974895227, -3.982029928440323108,  -3.390319200879429395, -4.392466879132969737, -3.788999105301563475, -3.541782736417437683, -3.809387850733410907,  -3.493208110030582958, -4.323122832039679508, -3.801833406081624567, -4.075028737045151672, -4.500123569799875156,  -3.970302136068273313, -4.396817196708417974, -4.330922272564527553, -3.777607335655170040, -3.342070869970420688,  -3.769683375712406193, -4.282499430951877351, -4.090198837409786847, -4.024509830092515728, -3.663098039666901506,  -3.917505587980506299, -3.950723020187467682, -4.099264887410029701, -3.881040636939387944, -4.406264464786102231,  -3.985108084984377630, -3.138496123533044191, -4.380479560928471905, -4.062958799599342363, -3.891022424536844326,  -3.193014827755344953, -4.308406394799806804, -3.657965848102073814, -3.421432871024832512, -3.677905749028450799,  -3.400910284399437877, -4.202901489277151903, -3.688362268148952516, -3.962446586589870812, -4.366963742769756074,  -3.915335708750363519, -4.290069621572881964, -4.209871865972900196, -3.716290234794378033, -3.202260253522962063,  -3.587086123223492962, -4.196557100162924492, -3.921252336913260805, -3.927010680194015890, -3.650851918479490088,  -3.816733683908503316, -3.859263365282489033, -3.969787413824056532, -3.716722910318457362, -4.207876268278917031,  -3.853974157325443350, -3.076849020154675429, -4.187055621138276784, -3.929086913676963722, -3.730836662485970123,  -3.069480677452482098, -4.215875754008563625, -3.525892420528105475, -3.301581202369676227, -3.614727483021799248,  -3.334172030049607827, -4.089161187606341841, -3.556060960208466071, -3.731705961752019629, -4.279938875087817785,  -3.820064180044840274, -4.217076033002934210, -4.041642265702158454, -3.685621620580431745, -3.072311895111735858,  -3.391364287663473309, -4.074586895333036196, -3.816344661360318913, -3.880035897691251012, -3.553223021040664342,  -3.759880962685191896, -3.653488256717163818, -3.847193719740457318, -3.583240144250745995, -4.070975199923323018,  -3.728068517911350632, -2.993991441998261394, -4.048097133542186477, -3.816842870173613456, -3.582965050085709535,  -2.970609612731116123, -4.222351353063677593, -3.517878392899691509, -3.282918942283597818, -3.622925987846269180,  -3.224032723642687337, -4.019560704366256232, -3.472508327143062345, -3.735541829393660862, -4.231476072317821746,  -3.804178078985315015, -4.172527527042076301, -3.987338160421503019, -3.632359707106025404, -3.012985269264222943,  -3.331810039653782418, -4.055490402890901791, -3.706495078933489840, -3.810704845352436365, -3.517081079258642440,  -3.678968235121425678, -3.489227653624175751, -3.751441489487161718, -3.527623113850316905, -4.025973259231754753,  -3.686175073231943600, -2.932918078395259975, -4.007967425920757165, -3.797450152359047149, -3.460935177834794985,  -2.776706840943757193, -4.129006400811464417, -3.458443621490054021, -3.238795774817717632, -3.586351473431607761,  -3.082572957072902131, -3.854792346764005728, -3.278300982262511098, -3.669947276874439446, -4.101936884004298456,  -3.737508124818469835, -4.042321203213761649, -3.886075985479370054, -3.498576097360127868, -2.924127114396212423,  -3.177754630790118995, -4.012620001469100472, -3.489609914097636434, -3.723705674166068569, -3.391897822533116891,  -3.592890504737299651, -3.295578800208273762, -3.584522307978101896, -3.433320281982902955, -3.972465110079852479,  -3.603148819134680103, -2.832094180498485159, -3.865114161355344802, -3.690252994705740530, -3.326442482740521456,  -2.563422785594254183, -3.994515893812661034, -3.320589329585814209, -3.149562609378700806, -3.530804720311090605,  -2.880774634914016197, -3.705287287350336634, -3.088422581984821669, -3.584918633095793616, -3.921080511243968836,  -3.641413473583009708, -3.833131526466035943, -3.626733204805630884, -3.326598532102873484, -2.815012428506726394,  -3.005443575206725004, -3.913085868648232601, -3.238311443846519921, -3.594783827978480506, -3.161538575900635806,  -3.493340083839198140, -3.131458696244663997, -3.490645935631426333, -3.357806696249643785, -3.841723306083461242,  -3.425051347622239284, -2.666656236371595767, -3.621179982057787683, -3.540698819307478207, -3.177064912978139422,  -2.484743640992302716, -3.944608268731178757, -3.263861078493769163, -3.129124034106155694, -3.528437009327384910,  -2.720529139056793611, -3.626820466330096249, -2.985072072910702978, -3.535722063459297626, -3.844013293476939985,  -3.595519134530512861, -3.723284661011910579, -3.555924704520218693, -3.265457325577266978, -2.858587221471339479,  -2.875427297106502245, -3.921849906504127681, -3.080047753307701086, -3.554274019813480567, -3.081605867970055534,  -3.450918515752950722, -3.004189402965404820, -3.458417196338254396, -3.328995647821002990, -3.787715023332055786,  -3.387401721982476133, -2.589089008768719946, -3.488740799251127012, -3.410240385263247020, -3.117047112584549584,  -2.425074650065610182, -3.964086282396162453, -3.171703074707119541,
                  -3.107090986899413654, -3.544298238715092531,  -2.694531058881662666, -3.568890263711990585, -2.891719354631235639, -3.540397049219906833, -3.786308690219354567,  -3.632786229462919181, -3.660709920327178235, -3.481812239091685246, -3.225772483350612951, -2.921022588403840547,  -2.852764552568514578, -3.959409598344780790, -2.946445597400329852, -3.528623614953044552, -3.074153375241563069,  -3.403558523576237782, -2.904267236527605434, -3.496912704578442721, -3.408746974541198682, -3.740010501248234309,  -3.372137237176033420, -2.500471927337202249, -3.501224494684771926, -3.441138366688094585, -3.099157645224464019,  -2.289148798765573645, -3.926684315079525422, -3.015282641177312595, -3.022425674369814796, -3.513106063574831683,  -2.637678327446396409, -3.480642343504057656, -2.768549200069876370, -3.442453452577924899, -3.656665865498609058,  -3.637826544726450173, -3.563873204816951024, -3.305800975767019700, -3.112793212899610751, -2.944143291252986128,  -2.737942903524741567, -3.962311192389678549, -2.776201183552976737, -3.415033420390006569, -2.954264054171859222,  -3.304212094248208942, -2.771698839769790812, -3.443313205617153372, -3.413224024559440295, -3.639229150743906516,  -3.368212383776357122, -2.336258563084871120, -3.387961635659166859, -3.406779890439024072, -3.021703600386090649,  -2.288177282843788074, -3.930397325374924034, -3.022900313613000822, -2.950455483039473137, -3.520584826904863185,  -2.646981340332575261, -3.456254385671213125, -2.756760166820178881, -3.316620406855250280, -3.650392961419039128,  -3.662292763548843588, -3.585857794613815486, -3.240558265582226483, -3.041520478934987004, -3.005795218844522942,  -2.694408449021408636, -3.983187522119089330, -2.676865533784770435, -3.383622085379140554, -2.903812117851447550,  -3.297395803624574295, -2.753477640072124366, -3.425452582668929047, -3.481171845133331111, -3.633048374020586113,  -3.453166504184965291, -2.185008374913947726, -3.341107308569299938, -3.407956287581034083, -2.950606943816229233,  -2.175647985974975551, -3.950765900456731483, -2.991991639211238230, -2.826883125535855523, -3.503584806665437590,  -2.592326784016492969, -3.418316660843511379, -2.699479599573788580, -3.121710001132695034, -3.627751286491255112,  -3.625999051119946248, -3.559862082374849024, -3.141114265499217684, -2.871458170094098428, -3.047966515116778741,  -2.563230863592535158, -3.948146764850261281, -2.578221657470269879, -3.329063230894271364, -2.798752357314935768,  -3.284593058526879883, -2.655865900114116940, -3.318516067402408343, -3.490535086392995368, -3.619394371461239324,  -3.527277855061926815, -1.997511167433114743, -3.250804190422605267, -3.322535750856715264, -2.780288183558866866,  -2.248486003254370758, -3.975505056647393509, -3.019267600722201905, -2.876964222737496524, -3.602372484799110985,  -2.678944936272860122, -3.377259382247330244, -2.774975481701952074, -3.114364812853524178, -3.648838898546438614,  -3.657373606023540002, -3.634106238548606704, -3.212470727992237762, -2.878722808585418846, -3.096985535996022776,  -2.641536340981267106, -3.941291100139705073, -2.642836210265486319, -3.365485010282918132, -2.884884957919852777,  -3.263683685247410260, -2.738816108253367521, -3.328368600338837080, -3.510941186442413375, -3.708642836056659675,  -3.657682309906763685, -1.968159826684758640, -3.227728761990313444, -3.370607590819304722, -2.774311411501614089,  -2.410114795273358368, -4.042070106037459531, -3.086231399359843586, -2.977318028340087075, -3.803313044733356296,  -2.842191950789859423, -3.400107512168180968, -2.892783267654955637, -3.219798115088218449, -3.730047729203237239,  -3.747458389766797904, -3.762025152478187806, -3.351278463705439137, -2.938489803132565914, -3.196143527283221708,  -2.773706585708360262, -3.945809383244737845, -2.817678999295515574, -3.435641817962626465, -3.063588047931476321,  -3.277135499419384423, -2.872175817495723926, -3.366325304027830256, -3.574073272443444083, -3.860433525394343146,  -3.855704944046957738, -1.990214028147766712, -3.235684154164724990, -3.448529811648719967, -2.780146156077215824,  -2.477646087803397901, -4.026681249896542525, -2.985209354376517688, -2.911761495609732187, -3.939315061401761131,  -2.946896452408315881, -3.263941920217887649, -2.962964116245295543, -3.078716814273603219, -3.638037737376698555,  -3.699955373301661243, -3.774084872377959421, -3.230528579487079277, -2.798193617651401510, -3.184002112426848718,  -2.777888108243619758, -3.890753774713390012, -2.872154965588306297, -3.415081182391118908, -3.081315447014362441,  -3.194107917678207009, -2.939082858679858745, -3.306464582338674685, -3.577909002826879181, -3.885769553920274788,  -3.851628930908012283, -1.926641552320437878, -2.977832425764000313, -3.393388139066732201, -2.691766588298778906,  -2.405843206974945581, -3.955525065617492508, -2.836739752181423491, -2.847123591347906046, -4.008629006899300684,  -2.895011688497560520, -3.143609718319401392, -2.983102556822411877, -2.768178267723200303, -3.527826667620824530,  -3.606344914444363781, -3.693004024817785336, -3.044691009741884002, -2.593861932336977372, -3.161429671785031204,  -2.680713665477956198, -3.788802675998153013, -2.852991631010399765, -3.411781575869638150, -3.123108648064947701,  -3.128933639230280228, -2.957755298049358217, -3.220536877682850907, -3.577479795124166806, -3.913028751614367362,  -3.852815083590126477, -1.722864203384638859, -2.657635807719435750, -3.292726037858124677, -2.563204613512051644,  -2.120214367503366226, -3.904863240257430590, -2.649361944388603174, -2.689799559494594039, -4.010334507120096426,  -2.757263031948114840, -2.951515879898576333, -2.932071009677659656, -2.349501260103841638, -3.352120464422875479,  -3.390869141172997203, -3.577100036907817415, -2.779576345936852544, -2.287802598455438208, -3.104540870226323968,  -2.396618028926071897, -3.543847145663811649, -2.760787362520485289, -3.325796268864852578, -3.037611909593561421,  -2.987321902043846844, -2.868924136888447940, -3.004599282872135291, -3.508653823334630495, -3.833934505715427488,  -3.814844566919973978, -1.363289225405424077, -2.238407279507298231, -3.086885939591620609, -2.325169467610252116,  -2.142772771411274757, -3.894537086182385810, -2.625142644775221967, -2.651182649938003344, -4.004799749711517087,  -2.819110074049324233, -2.872568716088588658, -2.897623150995993324, -2.363900581000184964, -3.268427833927324500,  -3.385559556868968922, -3.556207587588571251, -2.774293989980414654, -2.259071650547961774, -3.044746029827245515,  -2.440018526244343455, -3.535978407412365687, -2.773528537221664703, -3.242038567366857915, -2.975783871258162350,  -2.919803673055661797, -2.839614870880676989, -2.963395086636024800, -3.422625261242400629, -3.809792280647808838,  -3.803346837093406929, -1.322115167203011143, -2.177899981211987246, -3.041903913074628374, -2.296963059422874487,  -2.220872023230936687, -3.810160097869616180, -2.543296348836795673, -2.661297286151433461, -3.958540110302311010,  -2.883220391731452281, -2.749357761367894337, -2.799911647461940767, -2.408928493475238852, -3.102529149536998432,
                  -3.329877730825601123, -3.531910265391156045, -2.756996662917584828, -2.242258878680611023, -2.970209191167850982,  -2.457096966590614606, -3.484694244757088999, -2.685672108959499926, -3.154573279534243380, -2.852504597459472713,  -2.822336089543824755, -2.748701849588891744, -2.936647458777921216, -3.246882373485247442, -3.746787729903388264,  -3.834345685260361680, -1.361073243997076965, -2.239869805338051911, -3.050346799505892381, -2.323977447231565296,  -2.254513446102317076, -3.750787265254604375, -2.447132719264911493, -2.650610520900778422, -3.935676122975718449,  -2.913621480153351317, -2.620608591317824310, -2.707513784183240180, -2.443241172459133637, -2.964924935913119342,  -3.247826966066557741, -3.495447293173745695, -2.768443120987773565, -2.197486041144948388, -2.899689406875256559,  -2.471773674635699969, -3.437936511351902702, -2.591112504637804737, -3.088077468747448595, -2.707788261486616221,  -2.712949850835642440, -2.670986157278069317, -2.898598419822160821, -3.102229904687734852, -3.680168794432938206,  -3.875155009630301350, -1.347448142415637484, -2.301791400621370887, -3.040291515889087304, -2.309956729647148599,  -2.337951007777519408, -3.736849665983807967, -2.415625407327191443, -2.666138624515857813, -3.984004751770546804,  -2.955016998686088137, -2.567335384120617459, -2.675788587923797035, -2.537921874066714878, -2.827178325110836443,  -3.206059797386542343, -3.512648365041813658, -2.818085811268152607, -2.207208262178450564, -2.900278968986174988,  -2.517626024428020415, -3.384540802255154190, -2.535324769699788927, -3.069331253204849119, -2.636576738631720929,  -2.638994308157763147, -2.598149992297733579, -2.916463791528743688, -3.035577674080569999, -3.626539379596704471,  -3.940462535894658558, -1.364162044469035218, -2.367384510133909270, -3.037223510256866987, -2.308702056174361417,  -2.387953954966750114, -3.730047989494373439, -2.384051956104722425, -2.699938219657988192, -4.069633457886466132,  -2.969943155911950239, -2.544750448900044315, -2.648259297966002013, -2.650951994987885918, -2.714405998196621539,  -3.169378165363040623, -3.543195828863523911, -2.874645187428789939, -2.204904895998990089, -2.916189568435077817,  -2.566889605064974766, -3.331024833959574494, -2.482622382153174101, -3.085779671235868715, -2.596875707161084623,  -2.577341026690084025, -2.531880172259875383, -2.930613221459099194, -2.988837926514553711, -3.588205087588099751,  -4.011288770844661400, -1.377554147431298226, -2.405918548207232721, -3.052760359594318462, -2.305174988393425828,  -2.414353000517877934, -3.724211424058661013, -2.298843881142671464, -2.686598481245920933, -4.098174826382152247,  -2.995146904190016723, -2.480815077273259561, -2.620911700526220134, -2.667932256733938079, -2.580471096420002830,  -3.106942184727824241, -3.504577042499703143, -2.854158439007496550, -2.205400278112621404, -2.908585652725633075,  -2.582461113611416170, -3.258814780881015505, -2.460277905722011216, -3.088605432777790316, -2.510155581812941517,  -2.495334639685021560, -2.475024926757988464, -2.939989710115628707, -2.950476425442459139, -3.520920434169846125,  -4.040790551428783850, -1.331163138689514192, -2.423940699535662269, -3.042703826609086892, -2.262941593999130419,  -2.472086548021134256, -3.768564536374229501, -2.379755885530562498, -2.739111855871308787, -4.188337573691768512,  -3.085454366520160363, -2.534178825406974589, -2.689993893458038077, -2.830055017678930973, -2.621454198763446097,  -3.199007293428454357, -3.533588697933042422, -2.943266055937527437, -2.260693061579827301, -2.989434791903637922,  -2.733953771867054350, -3.312659518482714116, -2.570288517444767162, -3.169993881934331181, -2.589961761620262592,  -2.527956997402728945, -2.598943685137396820, -3.015095278264353862, -3.011150808003503432, -3.571510378221325777,  -4.144603911929360507, -1.468698061235916619, -2.588376143534715457, -3.121364737061997907, -2.386714683306252471,  -2.480866819437431481, -3.769977046014355349, -2.491212553498968862, -2.794953999956230284, -4.215573243022156191,  -3.152996974077778347, -2.620484417751923978, -2.763915777741537294, -2.948578771432937007, -2.745143632042865711,  -3.290185580216409722, -3.587310397418955965, -3.027578321584720733, -2.390429696402210702, -3.057261506910338156,  -2.862050002680648131, -3.372052448344295961, -2.711422958413025608, -3.251714774904575833, -2.707185063015277393,  -2.627775940101356067, -2.760061712575964776, -3.079128464735157333, -3.105745759263924821, -3.587759464850593272,  -4.169670947738516986, -1.570843773083353545, -2.690305448130386079, -3.213930303203991645, -2.456500346336549345,  -2.423376763304847525, -3.715770714211409587, -2.601418875266307218, -2.829818597421219550, -4.190121658563334073,  -3.155530140669715600, -2.669405122093905458, -2.804411535636746944, -2.974414895999754282, -2.883411884492891097,  -3.351908594264751962, -3.593156903836883576, -3.095504363213524623, -2.536015336928311381, -3.052923155971790870,  -2.924593124848077430, -3.434147957203647294, -2.814101264343200626, -3.308887629100223382, -2.783476419924861833,  -2.696424960730547937, -2.857071140654500407, -3.097884562281897125, -3.202299198630973009, -3.597568496125303383,  -4.202302445140943199, -1.637351081191139368, -2.806954926945383288, -3.246820759097478692, -2.523153363597652188,  -2.307031034768292077, -3.646494006678214106, -2.609461994602098489, -2.772059430582977768, -4.134761194306652676,  -3.058229273280355542, -2.655944327170220109, -2.781901668763138602, -2.987680006274549438, -2.904498440420967942,  -3.281071003132859154, -3.562172000385922743, -3.147966233876116515, -2.518097921263937167, -2.985183339131443514,  -2.935786089725621384, -3.449798188049355563, -2.814079303475710780, -3.308871299975590485, -2.789272974130875937,  -2.628190657417902809, -2.889382408650030598, -3.098048985310921655, -3.278621753766816305, -3.487496143615631095,  -4.187055319886876958, -1.534945539400520698, -2.778077765343489158, -3.174414086013101599, -2.467111246944073244,   1.051604430279504232, 0.809391350047680591, 0.966196263727004734, 1.134869633109036879, 0.972603305027433196,   0.830604452329502752, 0.979249003103297544, 1.199474456002184164, 1.068111612139342581, 1.077005123634550809,   0.901553809654226268, 1.115129189001011323, 1.189022324984245316, 1.011521644811130605, 1.063361535799106417,   0.868877558476145895, 1.320297484068821525, 0.965154867854108311, 1.092141948547320940, 1.040079060014033718,   0.863295067768180102, 1.000902346273548238, 0.919880552990245737, 0.772954864168783007, 0.808533831707076156,   1.013401852397425307, 1.081237861094180630, 0.958478278377522863, 0.839968560621724758, 1.157764634256350789,   1.051605196822854582, 0.809390679299843541, 0.966193824961277814, 1.134872106111185941, 0.972602191918104930,   0.830604578668630356, 0.979248499470021128, 1.199474049748030735, 1.068110404470083097, 1.077004769526966887,   0.901554897818139911, 1.115130545335369039, 1.189021740128762827, 1.011521587561197588, 1.063362070975563123,   0.868876736070779643, 1.320296569073589055,
                  0.965154369539052759, 1.092142448593776738, 1.040080056113009910,   0.863296027140208033, 1.000903890137130325, 0.919880105033826334, 0.772952175997521640, 0.808533116213484115,   1.013401425621024199, 1.081236966246511999, 0.958478568013938714, 0.839967370001823865, 1.157764816258179064,   0.770600766207763632, 0.613521893784818251, 0.729595951276566401, 0.926242524014050606, 0.747800240944529238,   0.420647551819010923, 0.656383816341794102, 0.956158061408621363, 0.833995153303062509, 0.865070394194093817,   0.430483851319837907, 0.786773359245176218, 0.877660450446758000, 0.831847077102608967, 0.839347595608000763,   0.641170795497490231, 1.172276450626122068, 0.891516309267101459, 0.867198856615535640, 0.884810704114756819,   0.621995422712255275, 0.842814054777438426, 0.721218341578329869, 0.432362109991072829, 0.525201601412642716,   0.776118051452216573, 0.764126429620053504, 0.761495395195702307, 0.680114582914688959, 0.820331426385188744,   0.901421540338019245, 0.745081487153462230, 0.963189050838919925, 1.127525927015222695, 0.874074184477371063,   0.584213233566611922, 0.777669659651580969, 0.939708230297005342, 0.895181513557067099, 0.946819402671765231,   0.507155638722480484, 0.913536155677981299, 0.840709724467153752, 0.979399049906236940, 0.874862520821695688,   0.965555624890064745, 1.211961407934617929, 1.024504279090503012, 1.005799332637670851, 0.914588659706156770,   0.829589872747434676, 0.956033038141770986, 0.902530893868819795, 0.643845692299572225, 0.739148534694698545,   0.865494550274512386, 0.979899192611352943, 1.001326615166299172, 0.892004986974045844, 0.983880211897905199,   1.057244392801346411, 0.973261078191939744, 1.163513855923661078, 1.324415472267368665, 1.125234333470145565,   0.800662428526958792, 0.981358172985168475, 0.995470016748876807, 1.234608443548041778, 1.132917499664603644,   0.803928067450430550, 1.079884099904938610, 1.022163330142076942, 1.168726746333189404, 1.189174692352533480,   1.165385173716688705, 1.303859074379478455, 1.254270949585878148, 1.151275496534400089, 1.067396551551938799,   0.967303074800040186, 1.097141936145954233, 1.199469089727138948, 1.169224229156320405, 0.989177424840022357,   1.291745516300195185, 1.110097311383865248, 1.246636548044816273, 1.231891418694194273, 1.223207741191622899,   1.020470368963760421, 0.897244107521801948, 1.102800539566689775, 1.062999693355786368, 1.012202098079718171,   0.810007563807968789, 0.947293794745382334, 0.930893771330451480, 1.102082188640661542, 1.032746030081785671,   0.743061173044561540, 0.895231869269147440, 0.973454402642800809, 1.087901094644210920, 1.117236457124944016,   1.017965939250985441, 1.191438730023290260, 1.062910225080188731, 1.005543224839188898, 0.901876904226031950,   0.868248982336316422, 0.873104888428914316, 1.101320562140593928, 0.994650525873832958, 0.911409612145592130,   1.107163400012860510, 0.994262751820175872, 1.051956724322296699, 1.127460126780729821, 1.079386569212992919,   1.024456406416238030, 0.830155873744217176, 1.079130038174909378, 0.838567784061751698, 0.997544374012972246,   0.813827814473216504, 1.006017662979855398, 0.926411258385862912, 1.006537818956459329, 1.015861400977126738,   0.644176808301423875, 0.743118790633833792, 1.041048858769981988, 1.116263280431466054, 1.105952391661463308,   0.921107373990242451, 1.166351968699514075, 0.964609736285703678, 0.915858708305638869, 0.788268213278150665,   0.817092732077986827, 0.679557953037383200, 1.004759133954721273, 0.883321704623412751, 0.931661502539424147,   1.049350178594444571, 0.953828791579398372, 1.005666934981891636, 1.057137192820049254, 0.982623430520084495,   1.054807566836659438, 0.800505595564891648, 1.149517831579222493, 0.867497031458086765, 0.936491034609454931,   0.909775567469392366, 0.947811514275614719, 1.008251353039090414, 1.069437408586066285, 1.056416877271660892,   0.648619303464672003, 0.821306518093123716, 0.934796744600051799, 1.000125506245512552, 1.066809611867348462,   0.877123599881293958, 1.229931483198098752, 0.990299992910821492, 0.858511078722907328, 0.683897162547554505,   0.860256936522536209, 0.714716163427313034, 1.099847638544233952, 0.898759865331781871, 0.958048582736356513,   0.994599764204442405, 0.826809835942177496, 0.972640905449953408, 1.138392742349430220, 0.857608457968905391,   1.297900561553018139, 1.050295405740939048, 1.305308974888104157, 1.195581733858221973, 1.166342952453300219,   1.289179297311166428, 1.204841848032432328, 1.284085489332957852, 1.356679509265525718, 1.255832926686030415,   1.098369300724146846, 1.276931135832396480, 1.316729756604522183, 1.281228287028103541, 1.182724109703300108,   1.187609486701524331, 1.433930617729697765, 1.234339422877382519, 1.027819335348657104, 1.019289847820186612,   1.122325090058422736, 0.876679724604548016, 1.331184405665948089, 1.124313475244753890, 1.335687012326665757,   1.430717367870034895, 1.141379811816873646, 1.329957361408013794, 1.326565811329135425, 1.193829981240099425,   1.392840382338169425, 1.178322352637785730, 1.240558346458130234, 1.272072215283719832, 1.208174320044673289,   1.324674165310257390, 1.151374601804230569, 1.229568784965023420, 1.315223298528785278, 1.307574976224515018,   1.215169480064567598, 1.275879965713576647, 1.248677389160423923, 1.336719859537751853, 1.179808979593646168,   1.274349003531405344, 1.313223860061130566, 1.275811479906096002, 1.002320847788288782, 0.981132260804153056,   1.194611501035941714, 0.919756660541767523, 1.359103337877084483, 1.071860930864664718, 1.394292958302954855,   1.309923861367672249, 1.238362163796565385, 1.324039799634387782, 1.212714966726016197, 1.285358676834061020,   1.480348487221958331, 1.378081557184540173, 1.400048068378910715, 1.373161435893444970, 1.335163880167349326,   1.279646611585551597, 1.431573592381720106, 1.326798642025242891, 1.435688574330908462, 1.583274410807040677,   1.349813404934105510, 1.389771571718136922, 1.373429146192490347, 1.416004410280190751, 1.426439905552019383,   1.339756176656843989, 1.376622244382925020, 1.446967577730368326, 1.288604414244287177, 1.344354749215149081,   1.318199861867115796, 1.178168889887015958, 1.544519483428919759, 1.341105494961676303, 1.486628234304391727,   1.482700463923462886, 1.520806173129709116, 1.437796061665856051, 1.380798843509305707, 1.444868932840148945,   1.523434772092440159, 1.429825802289057313, 1.410027665421241139, 1.339441469076402935, 1.496514911970221462,   1.279853898189087680, 1.418327817363269627, 1.316010603000788315, 1.539530180306088791, 1.628413848149084631,   1.397282959487692544, 1.452132550255936128, 1.527581305389499322, 1.496843585552027900, 1.461255062969766039,   1.486366508654196394, 1.321889578199926696, 1.409461493231078943, 1.255440597142031933, 1.268279114023979082,   1.435896446244120828, 1.220318676242474476, 1.605755231747763023, 1.394589536745836300,
                  1.466906811058824589,   1.420888595926655507, 1.568950218892812387, 1.451802378676130711, 1.305364487667658668, 1.421889321646443438,   1.532374974107302723, 1.432587519127653586, 1.302882504912963313, 1.403835944713675676, 1.435283114669479332,   1.336670100931621086, 1.456608824182555884, 1.347665508917589294, 1.532273875571223520, 1.644447268805020235,   1.430150141943185682, 1.413261680274669141, 1.490381834925922577, 1.596967822467934450, 1.398894167098883923,   1.411571041244868097, 1.340686409585511107, 1.338843153545847597, 1.250292412622483695, 1.456221005355680109,   1.427868853239919522, 1.470381457674084569, 1.627206993787431388, 1.448687993456416079, 1.301488251304258936,   1.285486921519640058, 1.482101791401363888, 1.312333708514755948, 1.294040616257691401, 1.387066073216902229,   1.437506501720088092, 1.422360966069190269, 1.130147408521108288, 1.356366093361986769, 1.357629161992988998,   1.297077656654879840, 1.373990617911920786, 1.256616258036752365, 1.225337261472747352, 1.627164126770645591,   1.284752228173815514, 1.384846746096609627, 1.372819261086129616, 1.619701366170701817, 1.319788384044441987,   1.241681514699810140, 1.324794977125345996, 1.248251797922818040, 1.238696714298417323, 1.364913186043299476,   1.379662328113854031, 1.380128844671470567, 1.478246632143374040, 1.296358030281279161, 1.113720003728312591,   1.197945858049875634, 1.404622805097608396, 1.188417206680608018, 1.226219244242540585, 1.341935782630711227,   1.588989104720589296, 1.487696509107897969, 1.287670295973507129, 1.595190946781376296, 1.432300819293213401,   1.346293700248482050, 1.579760940693686555, 1.396889021208677528, 1.372350380869245656, 1.685746597277197889,   1.369679329512993871, 1.538753603048722329, 1.524569638690865592, 1.676530682595678723, 1.514783380727246831,   1.233532744195594777, 1.476979306017863536, 1.367976558860211300, 1.471183689256446714, 1.480770645155379039,   1.531216180101284463, 1.355837787059009703, 1.517298333639011343, 1.369372501010812648, 1.246016814869544520,   1.459208717426241142, 1.581954718419860573, 1.388044717386177629, 1.372742877756125690, 1.463144604796200676,   1.618376384211527963, 1.377020077859459679, 1.412525092048207531, 1.775249574549666143, 1.488505607263037822,   1.422031559467825890, 1.537112420000639412, 1.349430028498284528, 1.465380844237975211, 1.601695553956430373,   1.457128618349051496, 1.549121170230546074, 1.661351175294198512, 1.593954012706524370, 1.753615851594237940,   1.045796157314792918, 1.543296901585071712, 1.337688256370643813, 1.622647639964225785, 1.453011781534458979,   1.742212494954770374, 1.358523891269338257, 1.458691818118799333, 1.497750767709301156, 1.324737731804249474,   1.637325909866474749, 1.695789750705578802, 1.411471957059663440, 1.321753831441087712, 1.561171017685781148,   1.185484118562297740, 1.134114994499907159, 1.201743904250061501, 1.561751234426096646, 1.336643507684945664,   1.101807783336624658, 1.308798349453303089, 1.026921242978102056, 1.238655999638200500, 1.322270952699460045,   1.260596311259546631, 1.263062393401644279, 1.275172907239292552, 1.306282927183425091, 1.429999280115156424,   0.800353055036243655, 1.267857929700970887, 1.040415725616067100, 1.398705313557245589, 1.051499555627606330,   1.564200399998096458, 1.117636919079890934, 1.193131294465034964, 1.259559780280624874, 1.095890793254755247,   1.261288965533377660, 1.439784621984930180, 1.039034766195268356, 1.009208022807224436, 1.323161636585715506,   1.069375866941943309, 1.040230859991482681, 1.228232684071385306, 1.489210710795333625, 1.237071342949509045,   0.804939620663527422, 1.263698406525002316, 0.954708546461607965, 1.066294092948566252, 1.214242038884324915,   1.125970340181459584, 1.127707614783794110, 1.234633352995305300, 1.267078311335129603, 1.376881477291086675,   0.654251839176706484, 1.198973931166687157, 0.980256710703238610, 1.339307785435479126, 0.938993645034100877,   1.502863724535548506, 1.000535236820830010, 1.073489525283737756, 1.043881070183124704, 1.039858573759247840,   1.105468398791455753, 1.341705952841699023, 0.829518096176756536, 0.754685398967501420, 1.230074839392873765,   0.970589686573169907, 1.047899576190485460, 1.096419573342632292, 1.387898365160713077, 1.109443390070141922,   0.770465034432894935, 1.188216953761689787, 0.880767666242326719, 0.999435353275409222, 1.101685882859899213,   1.130772294607173212, 1.060888479939354978, 1.131539024920954040, 1.167727539548359283, 1.277678800345860344,   0.725750849664789977, 1.068986347299468287, 0.910198632638385963, 1.228004950827745523, 0.903098031352520203,   1.356959169093720607, 0.924689118462150295, 1.048762424162153817, 1.049869445948060642, 0.927918652562520041,   1.022271055616164048, 1.162543471275469864, 0.880805631893443630, 0.821507850056614419, 1.137905814235566027,   0.771505062116684459, 1.047104769433707672, 0.932588442775013959, 1.304639252950396955, 1.003431034804298516,   0.714226592659970128, 1.206777692379743527, 0.797864649078443455, 0.887565377150006274, 1.024826383594885204,   1.210868722062335090, 1.051914277973020839, 1.059349582991142569, 1.090625828994962410, 1.241421164329207905,   0.737212848443539626, 0.977102596306620108, 0.865617033118032730, 1.126356863532694819, 0.871769959772701664,   1.233761523865539056, 0.850205797401225927, 0.973772689001085046, 1.064220472324315603, 0.808803992586045739,   1.016699553022718305, 1.022690020617519124, 0.866479609758391556, 0.872002888126911957, 1.083327573976683222,   0.643617377602282548, 1.042187306600282692, 0.928502913595488977, 1.083008759385193809, 0.877358671722577643,   0.577378610267900494, 1.180795105386698296, 0.714049742832761569, 0.752342049010042979, 1.037159301574281711,   1.137258505667996467, 0.972714915954617787, 0.930153677998920125, 0.966762333193899037, 1.188932422189189264,   0.659656543796525718, 0.909211921020695146, 0.739019073165447127, 1.024013381486017282, 0.715185150595188612,   1.175463208958145600, 0.713191089220998831, 0.867050363890360387, 1.054171772295770859, 0.704533448563296805,   0.918910093773397052, 0.861067225558907801, 0.871951471284914237, 0.828864098844255603, 0.933954793697437280,   0.357971299709871182, 1.085504713383047770, 0.938122223275794642, 0.871455807113316494, 0.763024000068868613,   0.334796646801549691, 1.209392050613838920, 0.596568444594444025, 0.599010795736536705, 1.098074391960134122,   1.078002354596015788, 0.907990847660164246, 0.804681835323760586, 0.822432960786837586, 1.181528144234006206,   0.513353697693843869, 0.845861104126891750, 0.624194443359611273, 0.920462291280707667, 0.537583223167096058,   1.159100742203432954, 0.520370388625681857, 0.711159728718274042, 1.075728685599476542, 0.607310386944797775,   0.876578185325437076, 0.711000282551342844, 0.875359460616932150, 0.757091446998178319, 0.737602683269314885,   0.519986383077626235,
                  1.190101602161771410, 1.083159627126741409, 0.975693270796545109, 0.881488112289717018,   0.459377304365628680, 1.287445889493385787, 0.745630026870163221, 0.783058363892327414, 1.245914206035615646,   1.134339261483813477, 1.045485872210204725, 1.007405548173635257, 1.035141221814709578, 1.249937699031158589,   0.708787144273090197, 0.855705948577560993, 0.773531487337236534, 0.963849360596192306, 0.718858241677109433,   1.173585705750941122, 0.666380169576386683, 0.805905857497079992, 1.159346716772627905, 0.774309029435458984,   1.044101140644660752, 0.860850600919218278, 1.007771269692763649, 0.935736658858548620, 0.840713933042612327,   0.932998885541780565, 1.427224552210626829, 1.346367019106702090, 1.189458542615627890, 1.182139986717664071,   0.797207661465522355, 1.517433978289171259, 0.988504894660716116, 1.269572264250398153, 1.518723346089864590,   1.328007669877688590, 1.307640653707007994, 1.367912939826005259, 1.392651926667652029, 1.456185063994115847,   1.016941405320487046, 0.924317652407350310, 1.151494341521267284, 1.072339088303179144, 1.124154947362269841,   1.281908438695244179, 0.928823310340551545, 0.974038573461386781, 1.344785547915750001, 1.088670489676333730,   1.419914536855923171, 1.145371047065461001, 1.226295745630443079, 1.208970241431417625, 0.987487618529231392,   1.030248349655890738, 1.396396859034795712, 1.152956868603191243, 0.984006100555759766, 1.299050939086291434,   0.950110317258604287, 1.303440630529137945, 1.095297660248901739, 1.133644624834986736, 1.391528227899703207,   1.204028229751220369, 1.286423970075768031, 1.138703471984649607, 1.226258745766347413, 1.374626389909677782,   1.017780584365154617, 0.922867065382831364, 1.194079731938074262, 0.977084216727391652, 1.073608060687712973,   1.151961245263128042, 0.999907751487348118, 0.918073642165158743, 1.353422002004971869, 1.034787073949267944,   1.232469266066271274, 1.150639125235735083, 0.877728543520688409, 1.075637170395147502, 0.835761924744654161,   0.761489987443020655, 1.207431641651349752, 0.864903401005407968, 0.751540478657290567, 1.249679292663973484,   0.712487469324515121, 1.109042425972445400, 1.073912900206597465, 0.682473258217345613, 1.198435874123296108,   0.986034261028023162, 1.030650333723694523, 0.757095895926857110, 0.894221224153531002, 1.251194296859265975,   0.803544662602910020, 0.808099418630352839, 1.110530510151914063, 0.900323783177031878, 1.066505074492481109,   1.049035242323586248, 0.966287953905768937, 0.772576491643748864, 1.331083738671050565, 0.921561592475181213,   0.975886935924592525, 0.795952188115152892, 0.382631829310557281, 0.801912354492974933, 0.587291126355232729,   0.147654261772061923, 1.046282333267948950, 0.569269542427049968, 0.394593201063015597, 1.130114369122105611,   0.301415592657598652, 0.857389987388972097, 0.978759394986568498, 0.078155737947780357, 0.979296218414379971,   0.657583205404231652, 0.722020528843348997, 0.308462471700379659, 0.410945197637059279, 1.106245329029999613,   0.348008702581956486, 0.509335579872277400, 0.974492725278377714, 0.719639919012811435, 0.928214521416495075,   0.845935049704515718, 0.799143570254912117, 0.437743522443734823, 1.243250365588908224, 0.698125733419455941,   0.663950112469765652, 0.212420484018967043, -0.235197039941282660, 0.369932014114470631, 0.177655362013502321,   0.234048405374791740, 1.137546313741258475, 0.691898409202436282, 0.400058536096782547, 1.140114055551595618,   0.432805517545295060, 0.930224129569587777, 1.036769810984893958, 0.228644508910045202, 1.002201336533717901,   0.780899432006810112, 0.711761397076928870, 0.411748305588408248, 0.514411873485199234, 1.068217699320720193,   0.503805436127964357, 0.622542080121431751, 1.083913473970593344, 0.716322396995347610, 0.964977758946856223,   0.877275322105209532, 0.840055029494332861, 0.549714133103138769, 1.281125326138140341, 0.757655412391786021,   0.598437595787244203, 0.288330490428739206, -0.178015254017640068, 0.385625261231110461, 0.241542465965755127,   0.422142790983337401, 1.031970856891536448, 0.685864055736832401, 0.491956295270992439, 1.040691665045120562,   0.526207817208749984, 0.900557712362336726, 0.947517454807493609, 0.427204216085955601, 0.860927349348084237,   0.776501691592679344, 0.682781604277816645, 0.464470109233101669, 0.605435573021699924, 1.005059055227737863,   0.580582875554690236, 0.618644950341430855, 0.997223650647262039, 0.721242806035458872, 0.866888456432737042,   0.838791213696997384, 0.750776410626504354, 0.635842953108796127, 1.104410655623415449, 0.709208489556238608,   0.613971485239314774, 0.502031541306265483, 0.020789947035633166, 0.481934396724909564, 0.401759777023586451,   0.533873697021945426, 0.993294471135937940, 0.659489081600782789, 0.562346528854113514, 1.019635130353772290,   0.568468539091702252, 0.903048229975879080, 0.906049971013392463, 0.596053816102596778, 0.767046054145436473,   0.750794479506993628, 0.654658187258942093, 0.527446923728283723, 0.642896500295219719, 0.961517379697295560,   0.629324581322170951, 0.607445143223286688, 0.955331618609147504, 0.748776905033901707, 0.793627422344567623,   0.800878055830862690, 0.688713641945689403, 0.698269020427275122, 0.966929989175038918, 0.677346409559776652,   0.632868783891855968, 0.651577609067717223, 0.204638378897258971, 0.572066206698282342, 0.507009788000074901,   0.594545189924297168, 0.953544937528757797, 0.632999348083856939, 0.639735492280188556, 1.062019429628887623,   0.570252436454102396, 0.943871912007145264, 0.862712588870811370, 0.759253451413443470, 0.673691524055099578,   0.722554903875277654, 0.635725467623316498, 0.592120118161156395, 0.644244001612972350, 0.934204344534443898,   0.665290930645521361, 0.584389974030000481, 0.900903639305551174, 0.789219449485903946, 0.749494928291946083,   0.748904620073477689, 0.613309014655107121, 0.740095530664033663, 0.846820861359773969, 0.645529784036669163,   0.640387793334963096, 0.770828864728818064, 0.311080528440759341, 0.670390153805251021, 0.579180416001923715,   0.576966760641376752, 0.896813840952416785, 0.591698404019693536, 0.724034026363315042, 1.166330311213934046,   0.498644925023376206, 1.018442981932729419, 0.820231715320538735, 0.966134426156696646, 0.591384880334990370,   0.674323641972599530, 0.623785567564994259, 0.666197514102662924, 0.591335108695194345, 0.927280406770220278,   0.689473146754987165, 0.553740648009392267, 0.835016500762456371, 0.856270434938964820, 0.747113630783822869,   0.683395320251224514, 0.528107934726028239, 0.743044816435124211, 0.722106632889938371, 0.609411313483649408,   0.648525179343675262, 0.844886250977440079, 0.349571516957603190, 0.764431929324958293, 0.615187620095945520,   0.504270752025533398, 0.837502940459557199, 0.401034875467698115, 0.647013970148715978, 1.113162485096047316,   0.468326374469129170, 0.939700697260589468, 0.774145902788306306,
                  0.936728074882608786, 0.424864691289864482,   0.527234300005559930, 0.448397319440923869, 0.548134271260080408, 0.509029073317453951, 0.831351088294552953,   0.614221473621605396, 0.438261569326269196, 0.818805557561708142, 0.840718423884149746, 0.600853103121853138,   0.523490783683138061, 0.422252052912855169, 0.708798275416423262, 0.571570744503181682, 0.477487109916337693,   0.567367998261169237, 0.747232495977631417, 0.356000837544128090, 0.737223893472898406, 0.522003981516685478,   0.463552816076785668, 0.804280760456347554, 0.429156808988682137, 0.636145251328270489, 1.139915070455665980,   0.531628225900952267, 1.015785592234887069, 0.861490576191962454, 1.128082894661864177, 0.459383463766356637,   0.584405126824920629, 0.329833322714336630, 0.583340876728702495, 0.459566094418079829, 0.830270619036511115,   0.732997303832097269, 0.482154862829445252, 0.973572858476594916, 0.902885827328276380, 0.675169735316506703,   0.486458865244447203, 0.562681397159449537, 0.735375331246299835, 0.516609137729931778, 0.495326974697398170,   0.593912418040224321, 0.891488894257877806, 0.561002703663513791, 0.811140110755862342, 0.631772146855427263,   0.320356096105449017, 0.779758595873769389, 0.576525823976070217, 0.681752692795351667, 1.134151552060198576,   0.564609645612460409, 1.141882824926027551, 0.992216917678707566, 1.263813772721216866, 0.658460571451185972,   0.718354624280962528, 0.411082923579169368, 0.726579264945781622, 0.599307338543493606, 0.923371401339522446,   0.907307451867791381, 0.619095579492662251, 1.170850598815310395, 0.972921616513803267, 0.791966723925494165,   0.623821003543688635, 0.777937326316033739, 0.783604540562321428, 0.663381182346682086, 0.525413748654159796,   0.621426659006713233, 1.004399665590256729, 0.680396991380199045, 0.880024629478140197, 0.654589928682266509,  -0.003356503065873762, 0.599574757552877147, 0.669719823194019348, 0.612189758397620243, 1.012752440692055167,   0.447367264905836004, 1.116143106068329338, 1.028454115013166525, 1.187318951695155000, 0.794155847437459816,   0.761915254778999129, 0.349097715091089777, 0.808233386939533727, 0.651903890196309455, 0.828091718140316901,   0.910001094125418852, 0.701001596410978700, 1.264130941297534116, 0.925346378630874877, 0.759847927031674919,   0.601482700583739649, 0.763101621965832644, 0.675214173847554555, 0.729818158986803245, 0.487555908888080092,   0.655337979365113399, 0.969984395869748495, 0.800186013286825815, 0.750469599560590339, 0.591216078311033844,  -0.511741298747917961, 0.275043510923585699, 0.624599904454924637, 0.405750670897994659, 0.739979908196746305,   0.046652733256579815, 0.999591902697514190, 0.846524381886554100, 1.035765767488057820, 0.698766663764570528,   0.542003190310543781, 0.109053151431616702, 0.817740625290732437, 0.423758826607079975, 0.580330639182732977,   0.683950448925485843, 0.613481304512613845, 1.179823580283142714, 0.770256879309193287, 0.628502371049922681,   0.295679852690024725, 0.642999087770074418, 0.455567587398746698, 0.723933184924200890, 0.142709839552383144,   0.509949982109287658, 0.642577287458449664, 0.619745339990394051, 0.460120748553102921, 0.329723797113126160,   1.663782504651734051, 1.636035806380044200, 2.559726420911143663, 3.229415369652231860, 2.588381363456075945,   2.240322658559840363, 3.639469910177072887, 2.104714669449029074, 1.368433096709945618, 1.772333417384705179,   1.312351572889226548, 3.208803933127438945, 2.794236531203623475, 1.565029652176951114, 1.428912543132843771,   2.151468261330010012, 2.117727026366615029, 2.203538759409934933, 4.228561311941552603, 2.600398600217474154,   2.630190232584548760, 3.129228341115585010, 1.247586088927168158, 2.419303837756912046, 2.668965741023429317,   1.743971327504775015, 3.235094653509045237, 3.167789729768629936, 2.224562675801824874, 2.709383244899799870 ))
})

# Had issues with win builder. Thus, these lines
cat("\nFinished", test_name, "\n")
