if(interactive()){
  library(survival); library(dynamichazard); library(testthat)

  if(grepl("testthat$", getwd()))
    source("../../R/test_utils.R") else
      source("./R/test_utils.R")
}

# Had issues with win builder. Thus, these lines
test_name <- "ddhazard"
cat("\nRunning", test_name, "\n")

# Test on data set that is one of Farhmiers papers
result = ddhazard(
  formula = survival::Surv(start, stop, event) ~ group,
  data = head_neck_cancer,
  by = 1, # Use by month intervals
  control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-4,
                 save_data = F, save_risk_set = F),
  a_0 = rep(0, 2), Q_0 = diag(1, 2), # Initial value
  max_T = 45,
  id = head_neck_cancer$id, order = 1
)

# plot(result, cov_index = 2)
# get_expect_equal(result, file = "tmp.txt")

expect_equal(c(result$state_vecs),
             c(-3.5025281961134787, -3.5025877576033477, -3.1321088612085983, -2.7757440091832057, -2.4919226130185796, -2.0481629213672718, -2.1192831215763093, -2.5081974639847040, -2.7397385003032992, -2.8845741297267971, -3.0249425773489134, -3.2400528773087220, -3.3649953385374434, -3.4209971757442106, -3.2738906455544505, -3.3169512577607794, -3.4382520761838764, -3.5423227286382666, -3.5025280610662426, -3.5881855240853717, -3.5581765201478746, -3.6816191615343223, -3.8597566904517313, -3.9786676622646073, -4.0361038571766708, -4.1732583956028995, -4.2656563848289011, -4.3141579768727913, -4.4604842005904972, -4.5793523090954995, -4.6702239867590185, -4.7332575267101360, -4.7685792576318100, -4.7761302924183315, -4.7556545004572994, -4.7067217620380086, -4.6287569806218709, -4.5210666088772840, -4.5211246420955638, -4.6290434142170565, -4.7225212272260517, -4.7987715172838055, -4.8571588039855502, -4.9005507089961524, -4.9291008412739341, -4.9428800599987603, 0.4611869044292034, 0.4611973957569191, 0.4644812524045190, 0.4736355394554351,  0.4669827742862612, 0.4837779729722258, 0.4898265943380803, 0.4860728861187437, 0.5052218028384193,  0.5135282631906429, 0.5190721140735243, 0.5202520241841503, 0.5286518809011893, 0.5439737247040330,  0.5619043578043230, 0.5619249621124215, 0.5611736349659849, 0.5660506677350730, 0.5722505331289359,  0.5773467910015616, 0.5834634039203218, 0.5769074684671019, 0.5749777881080025, 0.5734079048915875,  0.5722037907454223, 0.5757739716021596, 0.5797523773232870, 0.5841753632079369, 0.5934274294860569,  0.6031478029556855, 0.6132990871420009, 0.6238518360306121, 0.6347839458711073, 0.6460791936890807,  0.6577259525203066, 0.6697161933542817, 0.6820447299834685, 0.6947086479136694, 0.7009643574480928,  0.7006518516294673, 0.7004158564901843, 0.7002594597824986, 0.7001824345106531, 0.7001175765838670,  0.7000642351810415, 0.7000221623978419 ))

expect_equal(c(result$state_vars),
             c( 0.24966773368218786, -0.04852029000361392, -0.04852029000361389, 0.10685530511125829,  0.16324645041995289, -0.05271757666862437, -0.05271757666862434, 0.09828781852130458,  0.15325430938443149, -0.05162795076841659, -0.05162795076841659, 0.09136058704366490,  0.13495516427714296, -0.04929835651868680, -0.04929835651868683, 0.08496667085337734,  0.11565049223272644, -0.04726320825775959, -0.04726320825775961, 0.07932121905215284,  0.10178753695006632, -0.04274365026883695, -0.04274365026883695, 0.07487249696497597,  0.08289059384724072, -0.03967528470467367, -0.03967528470467367, 0.07163527330767552,  0.08578092764648139, -0.03846953031547166, -0.03846953031547165, 0.07084598469140357,  0.09989891251830449, -0.03828003928875170, -0.03828003928875170, 0.07199997499224448,  0.11295375016727069, -0.03890277688215408, -0.03890277688215407, 0.07418835553444229,  0.12373842571401289, -0.03974565037309754, -0.03974565037309753, 0.07705264985807921,  0.13473520084838175, -0.04057675152505523, -0.04057675152505523, 0.08045959690347994,  0.15058105305824362, -0.04169144974851560, -0.04169144974851560, 0.08433515349608449,  0.16475487689604817, -0.04313298868294341, -0.04313298868294340, 0.08844909148216468,  0.17649960812428095, -0.04469608555201694, -0.04469608555201694, 0.09271710753685869,  0.17457340811056918, -0.04624984779405965, -0.04624984779405965, 0.09698071771152830,  0.17729143834732247, -0.04793202830663135, -0.04793202830663135, 0.10153069332180298,  0.18781431644426710, -0.04988371630787453, -0.04988371630787453, 0.10645602554255014,  0.20226968890371799, -0.05200709477099941, -0.05200709477099941, 0.11169506602767454,  0.21057571962387894, -0.05417688326323991, -0.05417688326323991, 0.11709140194506812,  0.22152250807520027, -0.05647512985785519, -0.05647512985785519, 0.12275524191816919,  0.22514073774398694, -0.05881421064946086, -0.05881421064946086, 0.12856556534008104,  0.23575065936263351, -0.06169553642911084, -0.06169553642911083, 0.13471898476829020,  0.25340692731452963, -0.06480570637453512, -0.06480570637453510, 0.14123036954284510,  0.27107977399326072, -0.06808713214761060, -0.06808713214761059, 0.14798506451979404,  0.28472014812725088, -0.07152287576324623, -0.07152287576324623, 0.15491381998439313,  0.30339422994056781, -0.07518177129671275, -0.07518177129671275, 0.16212017559769024,  0.32068511056675808, -0.07881456491481259, -0.07881456491481259, 0.16951965225692336,  0.33378230195915271, -0.08244341708455906, -0.08244341708455906, 0.17707023560687726,  0.35358556972543320, -0.08605941482097960, -0.08605941482097960, 0.18490296127670985,  0.37617971793236965, -0.08966669605211727, -0.08966669605211727, 0.19295926473176217,  0.39964313796565276, -0.09325935310968274, -0.09325935310968275, 0.20120153675928507,  0.42279983339880933, -0.09682742127231320, -0.09682742127231321, 0.20960319710208880,  0.44474826795908584, -0.10035929363122134, -0.10035929363122136, 0.21814439918955283,  0.46465242556313213, -0.10384280859001971, -0.10384280859001972, 0.22680979489598740,  0.48163515765033976, -0.10726579703353845, -0.10726579703353846, 0.23558722226836393,  0.49471515025978507, -0.11061639909706367, -0.11061639909706368, 0.24446686862305730,  0.50276354516355681, -0.11388328248736546, -0.11388328248736547, 0.25344070457840195,  0.50362840905683237, -0.11694708051804112, -0.11694708051804113, 0.26249616065860049,  0.50886077519654149, -0.12000602290451649, -0.12000602290451649, 0.27177305917769967,  0.53753940894196228, -0.12331671148414861, -0.12331671148414861, 0.28150057273930468,  0.58422478156386226, -0.12675788251307221, -0.12675788251307221, 0.29159871019358652,  0.64854604358809853, -0.13024962589839847, -0.13024962589839847, 0.30202284254748069,  0.72726312575007357, -0.13408461161855237, -0.13408461161855237, 0.31268054143506063,  0.82372145328515756, -0.13821320036111659, -0.13821320036111659, 0.32355970964820718,  0.94204781019899575, -0.14259877676047242, -0.14259877676047242, 0.33465057109659269 ))

expect_equal(c(result$lag_one_cov),
             c( 0.14263116443415219, -0.04574920743260352, -0.05150543127174809, 0.09699079285706530,  0.10452193073617830, -0.04874739256245561, -0.05088688773815839, 0.08928873888007144,  0.09205816511743103, -0.04756288533170949, -0.04850365075455021, 0.08264748691502014,  0.07619687084467806, -0.04535505584257066, -0.04622466673349493, 0.07665184030229086,  0.06287176863202983, -0.04325886709103238, -0.04160716987308177, 0.07163427585559659,  0.04986944205987141, -0.03926405432290643, -0.03806053191486011, 0.06781573294293351,  0.04420678758208781, -0.03685812925848052, -0.03626291638490246, 0.06582468960959215,  0.04965011164211783, -0.03619367549364379, -0.03566682153220575, 0.06600412552478148,  0.05984037617699758, -0.03636192416054163, -0.03603336054933529, 0.06766354299450550,  0.06941035250370327, -0.03719088862092337, -0.03674084832627796, 0.07017664132977827,  0.07843493570034743, -0.03816550473187627, -0.03750684008628748, 0.07329844218824491,  0.08981493623946754, -0.03911819040254304, -0.03855808469632750, 0.07692587692407565,  0.10310658635478691, -0.04033010070998009, -0.03993869643713704, 0.08090961285689185,  0.11484463103360977, -0.04182722190206258, -0.04147133842862829, 0.08509110857921251,  0.11943889701282755, -0.04342857676366355, -0.04299835576581806, 0.08934994645705957,  0.11985243543009538, -0.04503133111582946, -0.04463580013043104, 0.09374691854979829,  0.12588538775084249, -0.04678858788297480, -0.04651894024076669, 0.09847389307447472,  0.13738385412440385, -0.04880681824779191, -0.04858628426379726, 0.10354633878610373,  0.14810467051698023, -0.05097881767359873, -0.05071611232481709, 0.10885655043099464,  0.15713971417898470, -0.05320244372983142, -0.05296561856819847, 0.11437825391674580,  0.16409749051278422, -0.05554564899105975, -0.05526366996309488, 0.12010875527332561,  0.17082169764926447, -0.05798517239621716, -0.05803435922163328, 0.12608210458780958,  0.18419244224951820, -0.06094157095036142, -0.06106650077427613, 0.13240606956799864,  0.20111915566759409, -0.06411857444864097, -0.06428014648816466, 0.13903231308527825,  0.21626339706997472, -0.06746230932018556, -0.06765132467092075, 0.14586836604869613,  0.23182744393716970, -0.07096532642848453, -0.07123809751737308, 0.15292935143336489,  0.24930432312366094, -0.07466552730666579, -0.07483158079442649, 0.16022691389911348,  0.26415101326279677, -0.07833917344388450, -0.07842064208601737, 0.16769729388739543,  0.28012526849294306, -0.08200216252035231, -0.08200453448237387, 0.17538307791434882,  0.30081469440241881, -0.08565137931089720, -0.08558233025367115, 0.18332254453471691,  0.32337781059751103, -0.08928760593720944, -0.08914990301383557, 0.19146736372042583,  0.34628498339489699, -0.09290425309985131, -0.09269742679066141, 0.19978529664717548,  0.36849989976218822, -0.09649155120250313, -0.09621288382965747, 0.20825303230026262,  0.38915143680505826, -0.10003839047880873, -0.09968351196260299, 0.21685290223635492,  0.40737973041049202, -0.10353321248541911, -0.10309647246244749, 0.22557109735345471,  0.42225260062760034, -0.10696449380634662, -0.10643920190335200, 0.23439658806889352,  0.43271270640033899, -0.11032104844718801, -0.10969963395979600, 0.24332041966466161,  0.43712323548021881, -0.11357939358846696, -0.11277225427960132, 0.25233231529368649,  0.44011649003356479, -0.11665841117598591, -0.11581344390893213, 0.26149456293695200,  0.45665574959249472, -0.11976869591380095, -0.11906857542687402, 0.27099088451026349,  0.49359720139669694, -0.12311415858361298, -0.12247186720389780, 0.28089856452764828,  0.54817221315200193, -0.12657419894355940, -0.12594165166472532, 0.29115499561373626,  0.61879172548337624, -0.13011247751412444, -0.12971638317819814, 0.30169208152706284,  0.70537058299019650, -0.13398820658801849, -0.13378869915457109, 0.31245682036970446,  0.81162044555582791, -0.13815236427340125, -0.13812122259985932, 0.32343824571520668 ))

expect_equal(c(result$fixed_effects),
             c(numeric(0) ))

expect_equal(c(result$n_iter),
             c(155 ))

expect_equal(c(result$Q),
             c( 0.142998595679639462, -0.004474914532711241, -0.004474914532711241, 0.011173790883017135 ))

expect_equal(c(result$Q_0),
             c(1, 0, 0, 1 ))

expect_equal(c(result$n_risk),
             c(96, 95, 91, 84, 78, 65, 53, 49, 46, 43, 39, 37, 36, 36, 33, 31, 30, 30, 25, 25, 23, 21, 21, 21, 20, 19, 19, 18, 18, 18, 18, 18, 18, 18, 18, 18, 17, 15, 14, 14, 14, 12, 12, 12, 11 ))

expect_equal(c(result$times),
             c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45 ))

expect_equal(c(result$risk_set),
             c(NULL ))

expect_equal(c(result$data),
             c(NULL ))

expect_equal(c(result$order),
             c(1 ))

expect_equal(c(result$F_),
             c(1, 0, 0, 1 ))

expect_equal(c(result$method),
             c("EKF" ))

expect_equal(c(result$model),
             c("logit" ))

expect_equal(c(result$est_Q_0),
             c(FALSE ))

expect_equal(c(result$LR),
             c(1 ))

test_that("Testing names of output from ddhazard on head and neck cancer dataset", {
  expect_equal(colnames(result$state_vecs), c("(Intercept)", "group1"))
  expect_equal(unlist(dimnames(result$state_vars)), unlist(list(c("(Intercept)", "group1"), c("(Intercept)", "group1"), NULL)))
  expect_equal(unlist(dimnames(result$Q)), rep(c("(Intercept)", "group1"), 2))
  expect_equal(unlist(dimnames(result$Q_0)), rep(c("(Intercept)", "group1"), 2))
})

test_that("Invalid penalty terms throw error", {
  expect_error(
    ddhazard(
      formula = survival::Surv(start, stop, event) ~ group,
      data = head_neck_cancer,
      by = 1, # Use by month intervals
      control = list(ridge_eps = 0)),
    regexp = "Method not implemented with penalty term \\(control\\$ridge_eps\\) equal to 0")

  expect_error(
    ddhazard(
      formula = survival::Surv(start, stop, event) ~ group,
      data = head_neck_cancer,
      by = 1, # Use by month intervals
      control = list(ridge_eps = -1)),
    regexp = "Method not implemented with penalty term \\(control\\$ridge_eps\\) equal to -1")
})

# tmp <- file("tmp.txt")
# sink(tmp)
suppressMessages(
  result_exp <- ddhazard(
    formula = survival::Surv(start, stop, event) ~ group,
    data = head_neck_cancer,
    by = 1, Q_0 = diag(.1, 2),
    Q = diag(1e-3, 2),
    control = list(est_Q_0 = F, n_max = 10^3, eps = 10^-4,
                   ridge_eps = 6e-2, save_data = F, save_risk_set = F),
    max_T = 30,
    id = head_neck_cancer$id, order = 1,
    verbose = F,
    model = "exp_combined"))
# sink()
# close(tmp)

# matplot(result_exp$state_vecs, type = "l")
# plot(result_exp, cov_index = 1)
# get_expect_equal(result_exp, file = "tmp.txt")

test_that("Result of exponential model on head_neck_data match previous results", {
  expect_equal(c(result_exp$state_vecs),
               c(-3.394447244746940928, -3.395153862315309024, -3.196934232144033938, -2.971370346786660566, -2.822363247966916955, -2.590374297594293385, -2.845451418921749021, -3.208754849655392150, -3.195992238503036376, -3.240424933427576981, -3.326744568839408167, -3.486460861415831669, -3.541928860019987191, -3.555787834370727651, -3.500120792413564885, -3.537496720642140602, -3.619653771189799585, -3.674506083482546259, -3.674651986577258000, -3.721577918162045862, -3.728201675120116576, -3.805010414189156975, -3.888055436948069321, -3.941960186742282879, -3.974466929102445700, -4.020706260103636076, -4.050110181610269144, -4.066153562166032742, -4.097505411945125964, -4.116975796142062549, -4.126152028734782817, 0.122916801171788451, 0.122533990313387006, 0.229527436797392376, 0.351370734874934643,  0.431297003639860010, 0.556782716408514511, 0.418632585638882948, 0.222172318828795290, 0.230645591942916117,  0.207498087114898033, 0.161454926414956679, 0.075452660198855256, 0.046037773572253332, 0.039263985561146436,  0.070000816752319173, 0.050007468669361621, 0.005752906900555035, -0.023585512732629672, -0.023428508348726215, -0.048575868452773931, -0.051999554661921105, -0.093668183155480020, -0.138558245876324398, -0.167751472331311990, -0.185423032115404118, -0.210398017154723921, -0.226308766328840061, -0.235025858701956525, -0.251906418694424872, -0.262388738913313824, -0.267328586639344778 ))

  expect_equal(c(result_exp$state_vars),
               c( 0.0769109729673901760, 0.0024536580489280015, 0.0024536580489280032, 0.0698992385820926820, 0.1042941119147692275,  0.0150384675508058330, 0.0150384675508058382, 0.0759448389094937448, 0.1219728793991194521, 0.0231903058451575182,  0.0231903058451575286, 0.0800679656176722099, 0.1230432906455513065, 0.0223322095596818343, 0.0223322095596818412,  0.0793223844019501079, 0.1103004950688061120, 0.0138735447522551902, 0.0138735447522552041, 0.0744169746041761365,  0.0970627617237544010, 0.0052158068860564261, 0.0052158068860564365, 0.0694800066653862014, 0.0684019434883764849, -0.0120319071185752517, -0.0120319071185752483, 0.0598041160592471394, 0.0882051195996330223, -0.0003465937743108031, -0.0003465937743108031, 0.0674182539353717220, 0.1154779469923206681, 0.0156772204485779477, 0.0156772204485779477,  0.0776181230783223042, 0.1272127282062879561, 0.0229150724776038574, 0.0229150724776038608, 0.0828882130290883035,  0.1355465325022736645, 0.0281392010628513725, 0.0281392010628513725, 0.0870069181623930876, 0.1468915885519631037,  0.0350130485236963274, 0.0350130485236963204, 0.0920617412760395548, 0.1692065682033239149, 0.0479053005015653496,  0.0479053005015653635, 0.1004510102245208536, 0.1876054679883882303, 0.0584981337118647762, 0.0584981337118647832,  0.1075157569113651956, 0.2009721122650755254, 0.0662366601197965782, 0.0662366601197965921, 0.1129782227697642139,  0.2043995550809924733, 0.0684504302554253402, 0.0684504302554253541, 0.1153808161034944602, 0.2105592801998395214,  0.0720417136418841086, 0.0720417136418841225, 0.1184903098172603031, 0.2259541153201423125, 0.0806342874444740398,  0.0806342874444740398, 0.1243269906938002933, 0.2440338533475068794, 0.0906345387485875398, 0.0906345387485875537,  0.1309141887455235465, 0.2586345465007773292, 0.0986441193013575524, 0.0986441193013575524, 0.1363743609270683943,  0.2754463345565064669, 0.1078279486353412309, 0.1078279486353412447, 0.1424681063093525546, 0.2895308626534787155,  0.1154620255376587745, 0.1154620255376587329, 0.1476902588984260190, 0.3126473663631990640, 0.1278971261909256363,  0.1278971261909255808, 0.1554753949527016288, 0.3445473415315708499, 0.1450772192704306018, 0.1450772192704305463,  0.1658336656178109880, 0.3809290561832989752, 0.1646133235319788557, 0.1646133235319788279, 0.1774362201678598150,  0.4205881332338599066, 0.1858346074048432273, 0.1858346074048431995, 0.1899082289433333681, 0.4678716938627417443,  0.2111260537034579388, 0.2111260537034579110, 0.2045582772708172403, 0.5214745058633968311, 0.2397885695312690912,  0.2397885695312690912, 0.2210105524792665976, 0.5816098706683554553, 0.2719243960554488804, 0.2719243960554488804,  0.2393125890989960258, 0.6537036914529315013, 0.3104984222982312136, 0.3104984222982312136, 0.2610848294124529234,  0.7380428629117394834, 0.3556472659006939541, 0.3556472659006939541, 0.2863904657529356967 ))

  expect_equal(c(result_exp$lag_one_cov),
               c(NULL ))

  expect_equal(c(result_exp$fixed_effects),
               c(numeric(0) ))

  expect_equal(c(result_exp$n_iter),
               c(131 ))

  expect_equal(c(result_exp$Q),
               c(0.09771736591399401, 0.05231684402165718, 0.05231684402165718, 0.02914800107904403 ))

  expect_equal(c(result_exp$Q_0),
               c(0.1, 0.0, 0.0, 0.1 ))

  expect_equal(c(result_exp$n_risk),
               c(96, 95, 91, 84, 78, 65, 53, 49, 46, 43, 39, 37, 36, 36, 33, 31, 30, 30, 25, 25, 23, 21, 21, 21, 20, 19, 19, 18, 18, 18 ))

  expect_equal(c(result_exp$times),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30 ))

  expect_equal(c(result_exp$risk_set),
               c(NULL ))

  expect_equal(c(result_exp$data),
               c(NULL ))

  expect_equal(c(result_exp$order),
               c(1 ))

  expect_equal(c(result_exp$F_),
               c(1, 0, 0, 1 ))

  expect_equal(c(result_exp$method),
               c("EKF" ))

  expect_equal(c(result_exp$model),
               c("exp_combined" ))

  expect_equal(c(result_exp$est_Q_0),
               c(FALSE ))

  expect_equal(c(result_exp$LR),
               c(1 ))
})

test_that("exponential model and logit moels hazzard functions differs", {
  expect_true(result_exp$model != result$model)
  expect_true(toString(body(result_exp$hazard_func)) !=
                toString(body(result$hazard_func)))
  expect_true(toString(body(result_exp$hazard_first_deriv)) !=
                toString(body(result$hazard_first_deriv)))
})

test_that("Testing names of output from ddhazard on head and neck cancer dataset", {
  expect_equal(colnames(result_exp$state_vecs), c("(Intercept)", "group1"))
  expect_equal(unlist(dimnames(result_exp$state_vars)), unlist(list(c("(Intercept)", "group1"), c("(Intercept)", "group1"), NULL)))
  expect_equal(unlist(dimnames(result_exp$Q)), rep(c("(Intercept)", "group1"), 2))
  expect_equal(unlist(dimnames(result_exp$Q_0)), rep(c("(Intercept)", "group1"), 2))
})

test_that("You can ommit the first entry when covariates are not time-varying",{
  r1 <- ddhazard(
    formula = survival::Surv(start, stop, event) ~ group,
    data = head_neck_cancer,
    by = 1, # Use by month intervals
    control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-4,
                   save_data = F, save_risk_set = F),
    a_0 = rep(0, 2), Q_0 = diag(1, 2), # Initial value
    max_T = 45,
    id = head_neck_cancer$id, order = 1
  )

  r2 <- ddhazard(
    formula = survival::Surv(stop, event) ~ group,
    data = head_neck_cancer,
    by = 1, # Use by month intervals
    control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-4,
                   save_data = F, save_risk_set = F),
    a_0 = rep(0, 2), Q_0 = diag(1, 2), # Initial value
    max_T = 45,
    id = head_neck_cancer$id, order = 1
  )

  expect_equal(r1$state_vecs, r2$state_vecs)
  expect_equal(r1$state_vars, r2$state_vars)
})


# Change by argument
suppressMessages(result_exp <- ddhazard(
  formula = survival::Surv(start, stop, event) ~ group,
  data = head_neck_cancer,
  by = 2, Q_0 = diag(1, 2),
  Q = diag(1e-1, 2),
  control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-3,
                 save_data = F, save_risk_set = F),
  max_T = 30,
  id = head_neck_cancer$id, order = 1,
  verbose = F,
  model = "exp_combined"
))

suppressMessages(result_exp <- ddhazard(
  formula = survival::Surv(start, stop, event) ~ group,
  data = head_neck_cancer,
  by = 2, Q_0 = diag(1, 2),
  Q = diag(1e-1, 2),
  control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-3,
                 save_data = F, save_risk_set = F),
  max_T = 30,
  id = head_neck_cancer$id, order = 1,
  verbose = F,
  model = "exp_combined"
))

# matplot(result_exp$state_vecs, type = "l")
# plot(result_exp, cov_index = 1)
# get_expect_equal(result_exp, file = "tmp.txt", eps = 1e-3)

test_that("Result of exponential model on head_neck_data match previous results with altered by argument", {
  expect_equal(unname(c(result_exp$state_vecs)),
               c(-3.852220556152348596, -3.852292033224423484, -1.963364349806266729, -2.451248922041040501, -3.221968319054067109, -3.291529236826010951, -3.877406281623367867, -3.641273563134608704, -3.532511423111788229, -3.920539412674798552, -3.844817210918701100, -4.084680881338860203, -4.383221748147751562, -4.587664579566934187, -4.307412523490374845, -4.674200919118026931, 0.240382670645352570, 0.240043347135763974,  1.294054252249459136, 0.958437551269521348, 0.495319017350084878, 0.521371068834693530, 0.192531090142120831, 0.414664932839877043,  0.443759331480920560, 0.221498990518044947, 0.244024183549669760, 0.001796395959245023, -0.224201433245315274, -0.375654552101357009, -0.264576579408945134, -0.460413519154605821 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$state_vars)),
               c( 0.4361530128706400111, 0.1079469454476254531, 0.1079469454476254253, 0.2919489473176493100, 0.1549413557971896505, -0.0329812963480873855, -0.0329812963480873855, 0.1596645600350908301, 0.1743114638221781310, -0.0001278314933700064, -0.0001278314933700064, 0.1425999558799868061, 0.0577393483183587936, -0.0484452504734892014, -0.0484452504734891945,  0.0701935067745317609, 0.0780898181812915426, -0.0397771651253612454, -0.0397771651253612524, 0.0891819915914530414,  0.1347809401246817806, -0.0167233091237283145, -0.0167233091237283110, 0.1327588080104361024, 0.1557362962094638870, -0.0185566511144944525, -0.0185566511144944456, 0.1602565818788772856, 0.2513269219578416758, 0.0263062534547130501,  0.0263062534547130467, 0.2230369870261285636, 0.2411182673118905662, 0.0005383113437213965, 0.0005383113437214034,  0.2357981395945896175, 0.2273234009086583518, -0.0267159927832882407, -0.0267159927832882511, 0.2510516830431762658,  0.3309714055512086861, 0.0143917899388671874, 0.0143917899388671944, 0.3203789421163170670, 0.3417727167941560507, -0.0051532069897200078, -0.0051532069897200113, 0.3545045782289721892, 0.4132282280797076601, 0.0172651281539309767,  0.0172651281539309490, 0.4264890135164262075, 0.5177726933790286479, 0.0550705837912595919, 0.0550705837912595780,  0.5139963825119391316, 0.6387386749245280715, 0.0963741173096935855, 0.0963741173096935855, 0.6066733626770471899,  0.6473576921349208924, 0.0660602010078818608, 0.0660602010078818608, 0.6570028088492781970 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$lag_one_cov)),
               c(NULL )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$fixed_effects)),
               c(numeric(0) )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$n_iter)),
               c(15 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$Q)),
               c(0.3247712710662991, 0.1648413409026548, 0.1648413409026548, 0.1329308997806527 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$Q_0)),
               c(1, 0, 0, 1 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$n_risk)),
               c(96, 91, 78, 53, 46, 39, 36, 33, 30, 25, 23, 21, 20, 19, 18 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$times)),
               c( 0, 2, 4, 6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$risk_set)),
               c(NULL )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$data)),
               c(NULL )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$order)),
               c(1 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$F_)),
               c(1, 0, 0, 1 )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$method)),
               c("EKF" )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$model)),
               c("exp_combined" )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$est_Q_0)),
               c(FALSE )
               , tolerance = 0.001)

  expect_equal(unname(c(result_exp$LR)),
               c(0.1316872427983539 )
               , tolerance = 0.001)
})



########
# Test on simulated data
set.seed(599479)
sims <- test_sim_func_exp(n_series = 1e3, n_vars = 10, t_0 = 0, t_max = 10,
                          x_range = 1, x_mean = 0, re_draw = T, beta_start = 0,
                          intercept_start = -5, sds = c(.1, rep(1, 10)))
# tmp_file <- file("exponential.log")
# sink(tmp_file)
suppressMessages(result_exp <- ddhazard(
  formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
  data = sims$res,
  by = (by_ <- 1),
  Q_0 = diag(10, 11),
  Q = diag(1e-2, 11),
  control = list(est_Q_0 = F, eps = 10^-2, n_max = 10^3,
                 save_data = F, save_risk_set = F, ridge_eps = 1e-2),
  max_T = 10,
  id = sims$res$id, order = 1,
  verbose = F,
  model = "exp_combined"))
# sink()
# close(tmp_file)

# matplot(seq_len(nrow(sims$betas)) - 1, sims$betas, type = "l", lty = 1)
# matplot(result_exp$times, result_exp$state_vecs, result_exp$state_vecs, type = "l", lty = 2, add = T)
# get_expect_equal(result_exp, eps = 1e-2, file = "tmp.txt")

test_that("Result of exponential model on simulated data match previous results", {
  expect_equal(c(result_exp$state_vecs),
               c(-5.07872608596536868, -5.08272698899318698, -4.47821907152965704, -4.55711659033740712, -3.77355134524433256, -4.06807825761109143, -4.10233637256440531, -3.87028726351097907, -3.77189696736534241, -3.39327887972167952, -3.11298344129774485, 0.09832895506657863, 0.08842932643230686, 0.29293821545864146, -0.49228884010482354, -0.09673940655896374, -0.48033783898571170, -1.76590367362402367, -2.67008699805099470, -2.79245863709676234, -1.25779412552592929, -0.28060384121361703, -0.09913940754070860, -0.07977674960540226, -1.36580212286798619, -2.21469972485730260, -5.78920205300872404, -4.92230002770736164, -4.48536048630945938, -4.18407698657516303, -3.38972532689482176, -2.37049335529395044, -1.65821621720172718, 0.15342598673345686, 0.13645283848644685,  0.85432915108018448, 0.58680129783290158, 2.51301138246105715, 1.85892927253342677, 0.64552402279561893, -0.27394732555356277, -0.77638808078283583, -0.03120250244389102, 0.48466348259473513, -0.05178808930154528, -0.05940044397653893, 0.87836326669539344, 1.51977058802354259, 3.55876442978964835, 3.08054357976342441,  3.33356043908029953, 3.73382639556066476, 3.50461526289647995, 2.72754538353266529, 2.25630002041405708, -0.58108959502201640, -0.58277191049523291, -0.60686690368182949, -0.56800521751008359, -0.39796337672278692, -0.44211363125774072, -0.58245770348258530, -0.77513821636115332, -0.85907973207449029, -0.92964384040818093, -0.99120712465291105, -0.21763066984878715, -0.21553566881376884, -0.27580825845487167, -0.39941281984911814, -0.73137819680074501, -0.65895771117443425, -0.57468762932909567, -0.47384850766949382, -0.35855258028620235, -0.15143288203016259, 0.03413374976609035, 3.13096112531667448, 3.13248769595813137, 3.40206711242696791,  2.81263905651351465, 2.39143202120144904, 2.32378282764119826, 2.22581358929696904, 2.46317321262473055,  2.94775944515515231, 4.10813515657577888, 4.89574111910982701, 1.24552513638519002, 1.24117630766322740,  1.64268050123873199, 1.82780276850869416, 2.72629076516208890, 2.46570378761898601, 2.41536183653684056,  2.42803585115145593, 2.37863655466326884, 2.18694502637768329, 1.94584354703589413, 0.52431550182494813,  0.51088678813673405, 0.87284814976485792, 0.45129048067034738, 1.60774171501283436, 1.15634364127584544, -0.05107404044334063, -1.03919004001507997, -1.50655158107339116, -0.61519088244460485, 0.08390072235628476, -3.39046668036432841, -3.38623972718172617, -4.18641504303866530, -4.06227442141247863, -5.01330966600714678, -4.62892011892447020, -4.63264508121882557, -4.96183567267715908, -5.17367083474895040, -5.72790447863891572, -6.10806942236412276 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$state_vars),
               c( 1.947996460792680e-01, 1.006538878904288e-01, -2.224904531086914e-01, 1.749817309404552e-01, 1.775134858357247e-01, -1.121746047626193e-02, 3.848178716829885e-03, 9.673392043484362e-02, 6.096107358684216e-02, 1.058880006580275e-01, -1.795339557991456e-01, 1.006538878904288e-01, 9.614091985596644e-01, 1.909904520999474e-01, 6.379900803905373e-01, -3.576609862936822e-01, 3.834832503320491e-02, 3.542635107839503e-02, 3.500803257361436e-01, -6.393224756538578e-02,  6.938762050807364e-01, -1.081789123615612e-01, -2.224904531086912e-01, 1.909904520999472e-01, 1.895609490698387e+00, -6.792124524550022e-01, -1.042364726296933e+00, -9.933894748494240e-02, 2.185955672252405e-01, 5.112295371389645e-01, -4.573262118496514e-01, -3.032761163235430e-01, 2.433141767978794e-01, 1.749817309404552e-01, 6.379900803905374e-01, -6.792124524550025e-01, 9.035161872970772e-01, 1.790212360155597e-01, 8.000374759679423e-02, -6.886239147532340e-02,  3.807565334261310e-02, 1.436176608244936e-01, 7.286815674306885e-01, -1.880585713692722e-01, 1.775134858357246e-01, -3.576609862936823e-01, -1.042364726296933e+00, 1.790212360155596e-01, 8.021950154011517e-01, 1.741831762028856e-02, -1.024038974267314e-01, -2.612130042566878e-01, 2.816964779880219e-01, -5.849264849450232e-02, -2.281240187496364e-01, -1.121746047626194e-02, 3.834832503320484e-02, -9.933894748494250e-02, 8.000374759679427e-02, 1.741831762028852e-02,  8.198186010097608e-02, -1.997448145549525e-02, -5.451657932586830e-02, 2.541775300945849e-02, 6.548387279517778e-02,  2.917038095294602e-02, 3.848178716829887e-03, 3.542635107839503e-02, 2.185955672252405e-01, -6.886239147532343e-02, -1.024038974267314e-01, -1.997448145549525e-02, 9.643115254212908e-02, 1.023037323583601e-01, -5.897913111462962e-02, -2.403345889286568e-02, -1.552379524892951e-02, 9.673392043484344e-02, 3.500803257361436e-01, 5.112295371389648e-01,  3.807565334261317e-02, -2.612130042566878e-01, -5.451657932586828e-02, 1.023037323583601e-01, 5.240456672818343e-01, -9.053514686843282e-02, 1.099828676638576e-01, -2.160297203451227e-01, 6.096107358684220e-02, -6.393224756538575e-02, -4.573262118496516e-01, 1.436176608244936e-01, 2.816964779880219e-01, 2.541775300945850e-02, -5.897913111462966e-02, -9.053514686843278e-02, 2.135753011198407e-01, 2.349279704223147e-02, -1.039861433471812e-01, 1.058880006580275e-01,  6.938762050807366e-01, -3.032761163235429e-01, 7.286815674306887e-01, -5.849264849450238e-02, 6.548387279517776e-02, -2.403345889286569e-02, 1.099828676638576e-01, 2.349279704223149e-02, 7.675676940134792e-01, -9.726723475725110e-02, -1.795339557991455e-01, -1.081789123615611e-01, 2.433141767978795e-01, -1.880585713692722e-01, -2.281240187496364e-01,  2.917038095294601e-02, -1.552379524892955e-02, -2.160297203451229e-01, -1.039861433471813e-01, -9.726723475725120e-02,  3.992800933614848e-01, 4.776614918536282e-02, -9.859076279355122e-03, -1.945608644964762e-02, -6.470554762103287e-04,  2.577743364748444e-02, -1.257385956878306e-03, 2.854853237555523e-03, -1.349721781431024e-02, -6.528241793225614e-03, -6.711251766116699e-03, 5.378638026257265e-03, -9.859076279355145e-03, 3.190251259616833e-01, 1.091561424354665e-01,  1.621021926452313e-01, -1.546111096464488e-01, 2.054858790399572e-02, 9.747860720263929e-03, 8.199788758996979e-02, -3.237201156444531e-02, 1.908145128293041e-01, 1.816821787670688e-02, -1.945608644964762e-02, 1.091561424354665e-01,  5.310555113252464e-01, -1.277321726118375e-01, -2.639642749283430e-01, -2.973114618029122e-02, 6.491898035600342e-02,  1.951006156190433e-01, -1.125406614391958e-01, -3.675134984143790e-02, 1.218125583555536e-02, -6.470554762103356e-04,  1.621021926452313e-01, -1.277321726118376e-01, 2.568875149156002e-01, -5.557906010505168e-03, 3.212820708637149e-02, -1.958458057946682e-02, -2.705586065223760e-02, 1.639160870019063e-02, 1.929895689146358e-01, 1.053440438170054e-02,  2.577743364748444e-02, -1.546111096464489e-01, -2.639642749283429e-01, -5.557906010505098e-03, 2.592819595580137e-01,  1.510812432618663e-03, -2.603314851007480e-02, -1.014775916732638e-01, 6.734176173580936e-02, -6.079394707026277e-02, -3.634823016288587e-02, -1.257385956878312e-03, 2.054858790399564e-02, -2.973114618029124e-02, 3.212820708637150e-02,  1.510812432618622e-03, 6.284339097685687e-02, -6.798888610109646e-03, -1.617946763815565e-02, 1.225685589172330e-02,  2.609106402722532e-02, 1.092304031048160e-02, 2.854853237555523e-03, 9.747860720263929e-03, 6.491898035600344e-02, -1.958458057946681e-02, -2.603314851007477e-02, -6.798888610109646e-03, 6.602877807907237e-02, 3.415970963979637e-02, -2.233777627702219e-02, -3.233218578252443e-03, -8.452383641533417e-03, -1.349721781431046e-02, 8.199788758996972e-02,  1.951006156190435e-01, -2.705586065223752e-02, -1.014775916732638e-01, -1.617946763815564e-02, 3.415970963979637e-02,  2.005828142591097e-01, -3.341878020182382e-02, 5.004506923065380e-03, -5.268433168274088e-02, -6.528241793225559e-03, -3.237201156444525e-02, -1.125406614391959e-01, 1.639160870019061e-02, 6.734176173580927e-02, 1.225685589172332e-02, -2.233777627702224e-02, -3.341878020182384e-02, 1.077839358095787e-01, -2.156470394955288e-02, -1.749705833448877e-02, -6.711251766116685e-03, 1.908145128293041e-01, -3.675134984143787e-02, 1.929895689146359e-01, -6.079394707026284e-02,  2.609106402722530e-02, -3.233218578252457e-03, 5.004506923065394e-03, -2.156470394955284e-02, 2.601645502986117e-01,  2.132742254159579e-02, 5.378638026257265e-03, 1.816821787670696e-02, 1.218125583555547e-02, 1.053440438170059e-02, -3.634823016288579e-02, 1.092304031048158e-02, -8.452383641533459e-03, -5.268433168274107e-02, -1.749705833448884e-02,  2.132742254159568e-02, 1.381922110611515e-01, 4.541587568977776e-02, -7.916478849482184e-03, -3.267977356772404e-02,  4.990800042136131e-03, 3.266072030382207e-02, -5.551423798434638e-04, -2.402181542698195e-04, -1.392413023689398e-02, -2.097886201281499e-03, -2.787344990664770e-03, 9.315207586216845e-04, -7.916478849482204e-03, 2.762240847599179e-01,  8.673555317282468e-02, 1.424797461478947e-01, -1.292063783090094e-01, 1.612586803305378e-02, 7.788567619987224e-03,  7.047112712826434e-02, -2.743249924185427e-02, 1.664670958776922e-01, 1.404411081459333e-02, -3.267977356772403e-02,  8.673555317282472e-02, 4.946449221527687e-01, -1.317499029387821e-01, -2.549622475186046e-01, -2.262287875232748e-02,  5.999375590672641e-02, 1.605176751968333e-01, -1.074420364076972e-01, -4.600939325762538e-02, 3.176055571594261e-02,  4.990800042136111e-03, 1.424797461478948e-01, -1.317499029387821e-01, 2.326826867447472e-01, 1.029119935523503e-02,  2.564525162249013e-02, -2.030925145640895e-02, -2.308868175624201e-02, 1.977290808222393e-02, 1.724015198840753e-01,  1.077331606416342e-03, 3.266072030382204e-02, -1.292063783090095e-01, -2.549622475186045e-01, 1.029119935523511e-02,  2.445076965463702e-01, 9.709127864565154e-04, -2.607798777255044e-02, -8.458654945561903e-02, 6.677923404542177e-02, -4.286214077951560e-02, -4.511335937858822e-02, -5.551423798434534e-04, 1.612586803305374e-02, -2.262287875232750e-02,  2.564525162249012e-02, 9.709127864564945e-04, 5.388486648370028e-02, -5.274276421457631e-03, -1.313163537478839e-02,  9.718729361262959e-03, 2.082175025012991e-02,
                  9.460617116275064e-03, -2.402181542698265e-04, 7.788567619987197e-03,  5.999375590672643e-02, -2.030925145640894e-02, -2.607798777255045e-02, -5.274276421457638e-03, 5.765450107849757e-02,  2.790865205649986e-02, -2.026290800784354e-02, -5.216057243441446e-03, -3.752552600220491e-03, -1.392413023689398e-02,  7.047112712826419e-02, 1.605176751968334e-01, -2.308868175624190e-02, -8.458654945561901e-02, -1.313163537478839e-02,  2.790865205649989e-02, 1.709477891135769e-01, -2.842081963790814e-02, 5.330278160626777e-03, -4.539557646168527e-02, -2.097886201281443e-03, -2.743249924185421e-02, -1.074420364076972e-01, 1.977290808222384e-02, 6.677923404542180e-02,  9.718729361262959e-03, -2.026290800784357e-02, -2.842081963790818e-02, 9.514574229908812e-02, -1.331528344651297e-02, -2.005355524194229e-02, -2.787344990664742e-03, 1.664670958776921e-01, -4.600939325762536e-02, 1.724015198840753e-01, -4.286214077951568e-02, 2.082175025012990e-02, -5.216057243441443e-03, 5.330278160626784e-03, -1.331528344651293e-02,  2.268212299788816e-01, 1.297317043206007e-02, 9.315207586217955e-04, 1.404411081459343e-02, 3.176055571594259e-02,  1.077331606416432e-03, -4.511335937858806e-02, 9.460617116275053e-03, -3.752552600220505e-03, -4.539557646168524e-02, -2.005355524194238e-02, 1.297317043206002e-02, 1.294590169416141e-01, 2.916131544811300e-02, -5.929415964629267e-03,  2.095746663323532e-03, -9.003173417279638e-03, 6.897121161055423e-03, -2.876797575800671e-04, 3.300588875816995e-03, -1.362685330045271e-02, -1.153264020571854e-02, -8.294443171563526e-03, 1.485886327408778e-02, -5.929415964629292e-03,  1.622765217893624e-01, 3.564063743019548e-02, 7.489763462844770e-02, -6.244377621311301e-02, 6.945989623264374e-03,  2.418791711016980e-03, 3.513815446506709e-02, -9.774128528630452e-03, 8.448252652388752e-02, 6.239604543667452e-03,  2.095746663323541e-03, 3.564063743019550e-02, 2.275850174346325e-01, -5.404201943285543e-02, -9.549849144273023e-02, -1.112128886263418e-02, 2.936824414129576e-02, 8.413658466982857e-02, -4.125868950968876e-02, -2.030254225464612e-02, -9.864593970273401e-04, -9.003173417279610e-03, 7.489763462844776e-02, -5.404201943285546e-02, 1.303725734557836e-01, -5.974332107795965e-03, 1.256330499641679e-02, -1.157968131253749e-02, -1.725591638594654e-02, 6.125229335074641e-03,  8.618993516331101e-02, 9.242917648844405e-03, 6.897121161055430e-03, -6.244377621311308e-02, -9.549849144273018e-02, -5.974332107795916e-03, 1.220337003668194e-01, 1.406431521558536e-04, -8.530802344348257e-03, -3.970514792662581e-02,  2.344556169779140e-02, -2.439144167723560e-02, -1.459245208883014e-02, -2.876797575800741e-04, 6.945989623264358e-03, -1.112128886263420e-02, 1.256330499641678e-02, 1.406431521558302e-04, 4.468851434232055e-02, -3.614974610497763e-03, -9.090359298481603e-03, 6.821844572959796e-03, 9.196230763794186e-03, 7.330317636774962e-03, 3.300588875816995e-03,  2.418791711016973e-03, 2.936824414129579e-02, -1.157968131253747e-02, -8.530802344348229e-03, -3.614974610497771e-03,  4.778059448911232e-02, 1.817573901770486e-02, -1.148635373027702e-02, -3.303647350133069e-03, -6.260307442104280e-03, -1.362685330045274e-02, 3.513815446506700e-02, 8.413658466982862e-02, -1.725591638594645e-02, -3.970514792662573e-02, -9.090359298481598e-03, 1.817573901770487e-02, 1.194892675648856e-01, -1.308516384018019e-02, -7.562587598733406e-04, -3.597402418538213e-02, -1.153264020571848e-02, -9.774128528630398e-03, -4.125868950968875e-02, 6.125229335074586e-03,  2.344556169779143e-02, 6.821844572959810e-03, -1.148635373027704e-02, -1.308516384018019e-02, 6.627707360085083e-02, -9.342786131843826e-03, -6.648679901046086e-03, -8.294443171563519e-03, 8.448252652388737e-02, -2.030254225464605e-02,  8.618993516331111e-02, -2.439144167723559e-02, 9.196230763794154e-03, -3.303647350133061e-03, -7.562587598733406e-04, -9.342786131843814e-03, 1.315392026620609e-01, 1.048671931216865e-02, 1.485886327408772e-02, 6.239604543667494e-03, -9.864593970273401e-04, 9.242917648844447e-03, -1.459245208883011e-02, 7.330317636774962e-03, -6.260307442104283e-03, -3.597402418538216e-02, -6.648679901046149e-03, 1.048671931216860e-02, 9.694071280696104e-02, 2.915510841369867e-02,  9.863273672870394e-03, 1.737035853703059e-02, -8.155131116281585e-04, -5.078608618326156e-03, -8.476938465910092e-04,  5.722505340451425e-03, 1.030154943947009e-03, -1.208533120517916e-02, 4.414435639029622e-04, 9.173100723787284e-03,  9.863273672870387e-03, 1.769504685771388e-01, 5.278648619003606e-02, 8.667053020838843e-02, -7.062343902539452e-02,  4.083847844199718e-03, 8.731763335488059e-03, 5.658973366501200e-02, -1.528612610537850e-02, 9.915133123622034e-02, -9.845611527521642e-03, 1.737035853703060e-02, 5.278648619003608e-02, 2.616207111573915e-01, -5.680924139090170e-02, -1.103511772482419e-01, -1.845866491351776e-02, 3.482453519497142e-02, 1.041920371429850e-01, -5.216637575052926e-02, -2.274471617311700e-02, -1.153297456688800e-02, -8.155131116281342e-04, 8.667053020838844e-02, -5.680924139090165e-02,  1.363340980813926e-01, -5.688281014623426e-03, 1.366355336209960e-02, -8.968534591144346e-03, -5.744065932312538e-03,  7.757510675202850e-03, 9.845761530438604e-02, -1.464204541516110e-03, -5.078608618326139e-03, -7.062343902539456e-02, -1.103511772482419e-01, -5.688281014623400e-03, 1.196380297910419e-01, 4.211363147815162e-03, -1.250053807425504e-02, -5.080924385656380e-02, 2.996931536297901e-02, -2.528391644166989e-02, -4.628126957365086e-03, -8.476938465910161e-04,  4.083847844199704e-03, -1.845866491351777e-02, 1.366355336209959e-02, 4.211363147815140e-03, 3.979802138459598e-02, -4.179284216329342e-03, -1.236589785839315e-02, 7.087999165873665e-03, 1.027951107117152e-02, 7.250041580160474e-03,  5.722505340451428e-03, 8.731763335488054e-03, 3.482453519497144e-02, -8.968534591144332e-03, -1.250053807425504e-02, -4.179284216329349e-03, 4.325544686878793e-02, 2.208398474921369e-02, -1.175053370541233e-02, -1.704118943820131e-03, -7.604991830851553e-03, 1.030154943947009e-03, 5.658973366501195e-02, 1.041920371429849e-01, -5.744065932312492e-03, -5.080924385656381e-02, -1.236589785839316e-02, 2.208398474921370e-02, 1.304999176767682e-01, -1.759232117452640e-02,  9.615638085725695e-03, -4.532350265684615e-02, -1.208533120517915e-02, -1.528612610537847e-02, -5.216637575052928e-02,  7.757510675202806e-03, 2.996931536297907e-02, 7.087999165873672e-03, -1.175053370541235e-02, -1.759232117452639e-02,  6.064158717889934e-02, -6.473975022139763e-03, -5.442479187205796e-03, 4.414435639029553e-04, 9.915133123622033e-02, -2.274471617311695e-02, 9.845761530438617e-02, -2.528391644166985e-02, 1.027951107117150e-02, -1.704118943820138e-03,  9.615638085725657e-03, -6.473975022139740e-03, 1.388624331640014e-01, 4.003977325208116e-04, 9.173100723787284e-03, -9.845611527521638e-03, -1.153297456688801e-02, -1.464204541516105e-03, -4.628126957365065e-03, 7.250041580160467e-03, -7.604991830851565e-03, -4.532350265684616e-02, -5.442479187205834e-03, 4.003977325207873e-04, 9.203294407850482e-02,  2.059585062207707e-02, 2.414473861172756e-03, 3.056984917633808e-02, -1.116465889900533e-02, -1.387769768177675e-02, -8.578475926878106e-04, 4.776371591143261e-03, -8.970260728317116e-04, -1.241549399879087e-02,
                  -8.118231303223648e-03,  1.561973706862997e-02, 2.414473861172747e-03, 6.263425408411924e-02, 9.925236782225016e-03, 1.750350290593276e-02, -1.668862760118341e-02, 8.832111106470721e-04, 3.376163343329839e-03, 9.373756917758443e-03, -3.385090991784640e-03,  2.075569120600253e-02, -7.391473099931629e-04, 3.056984917633808e-02, 9.925236782225066e-03, 1.117799208319561e-01, -2.123882050820617e-02, -3.122202088678844e-02, -5.275907815577805e-03, 1.474509986245957e-02, 2.452019326583954e-02, -2.109629650394977e-02, -1.485063714184732e-02, 7.690584380110249e-03, -1.116465889900530e-02, 1.750350290593275e-02, -2.123882050820611e-02, 5.585650952517116e-02, 1.675863480318495e-03, 4.614721078282275e-03, -3.800051169334189e-03, -4.316612904694823e-03, 3.884989039943813e-03, 2.683632001267790e-02, -4.990124112957350e-03, -1.387769768177675e-02, -1.668862760118344e-02, -3.122202088678844e-02, 1.675863480318538e-03, 5.817795558397708e-02, 9.100293866551664e-04, -4.281379864157074e-03, -6.391638493477612e-03, 1.332743713977281e-02, -2.516420484367603e-03, -8.545352771991289e-03, -8.578475926878117e-04, 8.832111106470589e-04, -5.275907815577813e-03, 4.614721078282279e-03, 9.100293866551494e-04,  3.250413484872455e-02, -1.371820055873442e-03, -4.781001835831885e-03, 3.378169255056386e-03, 3.702977308429434e-03,  2.899697583771327e-03, 4.776371591143264e-03, 3.376163343329839e-03, 1.474509986245958e-02, -3.800051169334191e-03, -4.281379864157080e-03, -1.371820055873443e-03, 3.478357792767890e-02, 7.590290625160303e-03, -6.567559590954778e-03, -6.365830418027627e-04, -4.499965748021029e-04, -8.970260728317081e-04, 9.373756917758387e-03, 2.452019326583958e-02, -4.316612904694782e-03, -6.391638493477572e-03, -4.781001835831888e-03, 7.590290625160300e-03, 6.537125984123543e-02,  5.858051650567729e-04, -3.724465411482544e-03, -2.237520533391391e-02, -1.241549399879086e-02, -3.385090991784616e-03, -2.109629650394978e-02, 3.884989039943784e-03, 1.332743713977279e-02, 3.378169255056386e-03, -6.567559590954794e-03,  5.858051650567617e-04, 4.598742864687834e-02, -2.621826029556346e-03, -1.020472987760974e-02, -8.118231303223651e-03,  2.075569120600257e-02, -1.485063714184730e-02, 2.683632001267787e-02, -2.516420484367569e-03, 3.702977308429432e-03, -6.365830418027737e-04, -3.724465411482589e-03, -2.621826029556312e-03, 6.138882660442036e-02, -2.380569258120593e-05,  1.561973706862998e-02, -7.391473099931609e-04, 7.690584380110284e-03, -4.990124112957378e-03, -8.545352771991291e-03,  2.899697583771331e-03, -4.499965748021193e-04, -2.237520533391394e-02, -1.020472987760976e-02, -2.380569258120073e-05,  6.981048730049502e-02, 2.463627407685796e-02, 9.405099247302037e-03, 3.808118577264658e-02, -9.211020160282236e-03, -1.551033362842426e-02, -2.316542138001418e-03, 5.933862025884433e-03, 8.413485684145121e-03, -1.244722809313040e-02, -6.101615289918852e-03, 8.129984675781170e-03, 9.405099247302021e-03, 1.074764723707951e-01, 3.772053199647793e-02,  4.155025480238153e-02, -4.175736528289767e-02, 2.827721107383142e-03, 3.625149454485042e-03, 3.139346439633814e-02, -9.619910493426231e-03, 5.103645516449964e-02, -5.931452547075098e-03, 3.808118577264658e-02, 3.772053199647796e-02,  1.633038928044040e-01, -2.694129443887470e-02, -5.611073595717973e-02, -8.467151854751923e-03, 2.295627229785794e-02,  5.876005447087978e-02, -2.973523837939215e-02, -9.116193739503318e-03, -1.152796754367310e-02, -9.211020160282210e-03,  4.155025480238147e-02, -2.694129443887467e-02, 7.792250563874585e-02, -6.359699927203396e-03, 8.342491421477100e-03, -7.685399380617823e-03, -3.839262801761913e-03, 3.640667928802683e-03, 4.935248762479269e-02, 4.733928228836193e-04, -1.551033362842427e-02, -4.175736528289768e-02, -5.611073595717973e-02, -6.359699927203343e-03, 7.679089898866051e-02, -4.005618378413456e-04, -6.698146293494504e-03, -2.090532332546218e-02, 1.856580625677334e-02, -1.740728871571235e-02, -4.843161520259485e-03, -2.316542138001422e-03, 2.827721107383126e-03, -8.467151854751940e-03, 8.342491421477104e-03, -4.005618378413495e-04, 3.283472599992149e-02, -2.303610262250743e-03, -8.084234634516123e-03, 3.304143126453176e-03,  7.403881414138423e-03, 6.721732432417960e-03, 5.933862025884432e-03, 3.625149454485047e-03, 2.295627229785792e-02, -7.685399380617824e-03, -6.698146293494525e-03, -2.303610262250741e-03, 3.514489591478989e-02, 1.202003785614975e-02, -7.310945393439119e-03, -3.214833097843872e-03, -3.823714689568449e-03, 8.413485684145149e-03, 3.139346439633810e-02,  5.876005447087982e-02, -3.839262801761887e-03, -2.090532332546220e-02, -8.084234634516125e-03, 1.202003785614974e-02,  9.094002821999791e-02, -4.285169137280142e-03, 1.996484951677865e-03, -3.892242721716995e-02, -1.244722809313038e-02, -9.619910493426226e-03, -2.973523837939215e-02, 3.640667928802700e-03, 1.856580625677333e-02, 3.304143126453175e-03, -7.310945393439126e-03, -4.285169137280144e-03, 4.584622384814465e-02, -5.140910955518811e-03, -7.689581250511883e-03, -6.101615289918866e-03, 5.103645516449968e-02, -9.116193739503333e-03, 4.935248762479260e-02, -1.740728871571234e-02,  7.403881414138425e-03, -3.214833097843875e-03, 1.996484951677852e-03, -5.140910955518795e-03, 8.517747927810299e-02,  4.503232390863189e-03, 8.129984675781167e-03, -5.931452547075109e-03, -1.152796754367308e-02, 4.733928228836158e-04, -4.843161520259483e-03, 6.721732432417963e-03, -3.823714689568456e-03, -3.892242721717000e-02, -7.689581250511881e-03,  4.503232390863252e-03, 7.936175355535764e-02, 2.507149855812582e-02, 1.421791362968356e-02, 3.319998680800504e-02, -2.709951923073640e-03, -1.674884960389012e-02, -5.301115279596690e-04, 6.108336683172011e-03, 6.413373570339971e-03, -1.288157211997375e-02, 1.070083262899305e-03, 9.423482245194459e-03, 1.421791362968354e-02, 1.002686979210702e-01,  2.120612227843072e-02, 4.332073906659888e-02, -3.218216587750572e-02, 4.117013832166330e-03, 3.359292711276672e-03,  1.871815554655922e-02, -7.444773058059225e-03, 4.973251146254078e-02, 4.375730936587094e-04, 3.319998680800504e-02,  2.120612227843071e-02, 1.463582737616592e-01, -3.117391967494091e-02, -4.597635710176762e-02, -9.522084587534311e-03,  1.784443539510782e-02, 4.615673530416377e-02, -2.707391344014232e-02, -1.425049853839065e-02, -7.671717719481923e-03, -2.709951923073634e-03, 4.332073906659890e-02, -3.117391967494095e-02, 8.075685164573720e-02, -4.381576043903563e-03,  9.780173442248880e-03, -4.921872992992394e-03, -6.431224088050034e-03, 4.289395276655803e-03, 5.004994354682127e-02,  3.287218383538984e-03, -1.674884960389011e-02, -3.218216587750571e-02, -4.597635710176764e-02, -4.381576043903530e-03,  6.930376762565035e-02, 1.059468133601241e-03, -4.706596093075528e-03, -1.518197134217063e-02, 1.567911117542494e-02, -1.289939420675715e-02, -5.705782808688112e-03, -5.301115279596747e-04, 4.117013832166320e-03, -9.522084587534323e-03,  9.780173442248878e-03, 1.059468133601236e-03, 3.327762625299904e-02, -2.071823678696639e-03, -7.109313266708916e-03,  4.142251705017478e-03, 7.670947428803067e-03, 5.362432975503536e-03, 6.108336683172005e-03, 3.359292711276674e-03,  1.784443539510782e-02, -4.921872992992395e-03, -4.706596093075541e-03, -2.071823678696638e-03, 3.478607853906523e-02,  1.058803047733213e-02,
                  -6.285183726928982e-03, -7.561684625962009e-04, -4.820442086224333e-03, 6.413373570339987e-03,  1.871815554655922e-02, 4.615673530416382e-02, -6.431224088050020e-03, -1.518197134217063e-02, -7.109313266708924e-03,  1.058803047733213e-02, 8.033440327955120e-02, -4.386657028076805e-03, -2.945089100443088e-04, -3.420558891771885e-02, -1.288157211997375e-02, -7.444773058059235e-03, -2.707391344014232e-02, 4.289395276655790e-03, 1.567911117542492e-02,  4.142251705017481e-03, -6.285183726928999e-03, -4.386657028076791e-03, 4.453497388449202e-02, -3.731008479022555e-03, -6.370315489851520e-03, 1.070083262899307e-03, 4.973251146254084e-02, -1.425049853839068e-02, 5.004994354682118e-02, -1.289939420675713e-02, 7.670947428803058e-03, -7.561684625962036e-04, -2.945089100443053e-04, -3.731008479022569e-03,  8.344398129003952e-02, 6.471549701205212e-03, 9.423482245194464e-03, 4.375730936587224e-04, -7.671717719481911e-03,  3.287218383538974e-03, -5.705782808688155e-03, 5.362432975503541e-03, -4.820442086224337e-03, -3.420558891771888e-02, -6.370315489851524e-03, 6.471549701205215e-03, 7.643556089089684e-02, 2.315583256205988e-02, 2.052515253425456e-02,  2.397027472831581e-02, 5.925181382039212e-03, -1.642620059028780e-02, 1.021802840421347e-03, 3.872108090627764e-03,  2.937718065271367e-03, -1.040954053050528e-02, 8.752990916877573e-03, 1.105709532770334e-02, 2.052515253425456e-02,  9.695182716618872e-02, 1.268993087930249e-02, 4.029058340670533e-02, -2.482087478330892e-02, 3.928004697257391e-03,  3.930355018320186e-03, 1.724574746941825e-02, -5.256275926445163e-03, 4.432486183747391e-02, 1.773861468268162e-04,  2.397027472831582e-02, 1.268993087930246e-02, 1.182102634225652e-01, -2.366843211873513e-02, -3.766728660490918e-02, -3.290503061085933e-03, 9.948520405308308e-03, 2.694822692206911e-02, -1.997665557445230e-02, -1.089272875760882e-02,  3.557145460010442e-03, 5.925181382039216e-03, 4.029058340670529e-02, -2.366843211873517e-02, 7.354680212389961e-02, -2.666246107282342e-03, 6.713741012279293e-03, -4.996300697585145e-04, 1.372910038436267e-04, 2.250511137851243e-03,  4.191391333988759e-02, -8.068216525855003e-04, -1.642620059028780e-02, -2.482087478330892e-02, -3.766728660490923e-02, -2.666246107282274e-03, 6.710035888291692e-02, -8.620249416438986e-04, -3.095939328844262e-03, -8.573833588935299e-03,  1.412426959038158e-02, -9.545939685677933e-03, -1.087047192960886e-02, 1.021802840421347e-03, 3.928004697257386e-03, -3.290503061085947e-03, 6.713741012279293e-03, -8.620249416439021e-04, 3.380806597607135e-02, -3.980539577969109e-04, -4.642040077619614e-03, 2.167599432021127e-03, 5.578480525871163e-03, 5.114652498547917e-03, 3.872108090627759e-03,  3.930355018320176e-03, 9.948520405308305e-03, -4.996300697585178e-04, -3.095939328844257e-03, -3.980539577969110e-04,  3.423014067931257e-02, 5.980897914483385e-03, -4.169093281653996e-03, 2.877954928350883e-03, -2.945062054333825e-03,  2.937718065271376e-03, 1.724574746941827e-02, 2.694822692206906e-02, 1.372910038436182e-04, -8.573833588935360e-03, -4.642040077619618e-03, 5.980897914483385e-03, 7.618031652923682e-02, 2.388423595703854e-03, 2.690372383650027e-03, -3.407871115924564e-02, -1.040954053050530e-02, -5.256275926445167e-03, -1.997665557445229e-02, 2.250511137851256e-03,  1.412426959038162e-02, 2.167599432021130e-03, -4.169093281654013e-03, 2.388423595703862e-03, 4.438192258896546e-02, -5.198197636848803e-03, -1.136621313976364e-02, 8.752990916877569e-03, 4.432486183747394e-02, -1.089272875760883e-02,  4.191391333988762e-02, -9.545939685677910e-03, 5.578480525871159e-03, 2.877954928350878e-03, 2.690372383650040e-03, -5.198197636848813e-03, 7.619172681805739e-02, 5.357571551481880e-03, 1.105709532770338e-02, 1.773861468268396e-04,  3.557145460010459e-03, -8.068216525855118e-04, -1.087047192960894e-02, 5.114652498547922e-03, -2.945062054333837e-03, -3.407871115924570e-02, -1.136621313976363e-02, 5.357571551481893e-03, 8.113829794566130e-02, 2.344217536802330e-02,  2.217505360251924e-02, 1.933914677043619e-02, 1.044468308092495e-02, -1.608803643637435e-02, 2.148086643070818e-03,  3.068731490151549e-03, -5.976562413195431e-04, -1.110302181872119e-02, 1.416263278408889e-02, 1.489890904872850e-02,  2.217505360251924e-02, 1.085181791599103e-01, 2.203029619860142e-02, 4.421607170510058e-02, -3.118428618692221e-02,  2.516198584153257e-03, 2.990488786206115e-03, 2.498955449051245e-02, -6.930277457239858e-03, 4.911803134707848e-02,  1.807428841732617e-03, 1.933914677043617e-02, 2.203029619860142e-02, 1.190409898776075e-01, -1.731555921422943e-02, -3.908528313377827e-02, -3.224172026797462e-03, 1.155206326248705e-02, 2.927469282293395e-02, -2.029085819575271e-02, -4.463222349005406e-03, 2.980928450791788e-03, 1.044468308092495e-02, 4.421607170510054e-02, -1.731555921422948e-02,  7.894063540926602e-02, -6.811601924140466e-03, 5.359179743613629e-03, -1.806038702276159e-03, 5.739116816215218e-03,  1.108159935501736e-03, 4.504026510163073e-02, -5.830487720013384e-05, -1.608803643637432e-02, -3.118428618692219e-02, -3.908528313377824e-02, -6.811601924140426e-03, 7.204672243250251e-02, -1.492649824423207e-04, -3.189914054322758e-03, -1.236782703119162e-02, 1.347018105005502e-02, -1.219085802715927e-02, -9.793284981383769e-03, 2.148086643070818e-03,  2.516198584153249e-03, -3.224172026797476e-03, 5.359179743613631e-03, -1.492649824423299e-04, 3.648214095512006e-02, -5.701908083633037e-04, -4.574303330603515e-03, 2.591797890112699e-03, 3.598098544194159e-03, 4.585840772327409e-03,  3.068731490151547e-03, 2.990488786206111e-03, 1.155206326248704e-02, -1.806038702276167e-03, -3.189914054322760e-03, -5.701908083633057e-04, 3.777870410662437e-02, 6.377120229017875e-03, -4.789491390713878e-03, 2.498951856647650e-03, -3.358033369491926e-03, -5.976562413195293e-04, 2.498955449051244e-02, 2.927469282293396e-02, 5.739116816215238e-03, -1.236782703119161e-02, -4.574303330603519e-03, 6.377120229017865e-03, 8.389926924662458e-02, 1.990226005113623e-03,  9.915287298125000e-03, -3.356833324798969e-02, -1.110302181872120e-02, -6.930277457239850e-03, -2.029085819575269e-02,  1.108159935501726e-03, 1.347018105005503e-02, 2.591797890112703e-03, -4.789491390713888e-03, 1.990226005113605e-03,  4.810187519981989e-02, -6.922014696907338e-03, -1.058025296618448e-02, 1.416263278408888e-02, 4.911803134707853e-02, -4.463222349005421e-03, 4.504026510163075e-02, -1.219085802715931e-02, 3.598098544194156e-03, 2.498951856647651e-03,  9.915287298125026e-03, -6.922014696907355e-03, 8.211251218058896e-02, 3.888302670222800e-03, 1.489890904872850e-02,  1.807428841732603e-03, 2.980928450791805e-03, -5.830487720007876e-05, -9.793284981383791e-03, 4.585840772327406e-03, -3.358033369491931e-03, -3.356833324798968e-02, -1.058025296618446e-02, 3.888302670222792e-03, 8.194848885564000e-02,  2.091731646891761e-02, 1.038525213377589e-02, 1.243606325033677e-02, 3.418459014905585e-03, -1.107402201800049e-02,  2.165964126720222e-03, 1.240872167211510e-03, -1.095553920507656e-02, -1.004280809592643e-02, 6.351618814768306e-03,  2.074600027338205e-02, 1.038525213377589e-02, 9.077841165704746e-02, 1.027214180214973e-02, 3.092494888548669e-02, -2.107561427685398e-02, 3.054818173397137e-03, -2.574505998782103e-03,
                  1.385861349271759e-02, -1.151701927809804e-03,  3.275405793066540e-02, 1.937918353018940e-03, 1.243606325033677e-02, 1.027214180214973e-02, 1.255444584852529e-01, -3.034498068737904e-02, -3.922106689267021e-02, -2.892051583756624e-03, 1.117443241051188e-02, 2.559237922401282e-02, -2.035007720867598e-02, -1.434420050950242e-02, 5.213926546803594e-03, 3.418459014905585e-03, 3.092494888548669e-02, -3.034498068737904e-02, 8.099506833085349e-02, 1.071539303141814e-03, 6.132646191119579e-03, -6.048689134751926e-03, -4.131060474209837e-03, 4.808727112410356e-03, 3.976911297430290e-02, 6.934274796726427e-04, -1.107402201800049e-02, -2.107561427685398e-02, -3.922106689267021e-02, 1.071539303141814e-03, 7.635179314768359e-02, -8.572588860098708e-04, -1.466194312313571e-03, -9.563783797862197e-03, 1.225135566130793e-02, -4.872906234463498e-03, -9.920405031825064e-03,  2.165964126720222e-03, 3.054818173397137e-03, -2.892051583756624e-03, 6.132646191119579e-03, -8.572588860098708e-04,  4.153437448541653e-02, -8.380325158228064e-04, -4.885344537955377e-03, 2.433695213688286e-03, 4.275588260015189e-03,  5.296176003724820e-03, 1.240872167211510e-03, -2.574505998782103e-03, 1.117443241051188e-02, -6.048689134751926e-03, -1.466194312313571e-03, -8.380325158228064e-04, 4.269636440940176e-02, 3.901003837489495e-03, -5.130095000966299e-03, -6.226364508819487e-04, -2.599459983664355e-03, -1.095553920507656e-02, 1.385861349271759e-02, 2.559237922401282e-02, -4.131060474209837e-03, -9.563783797862197e-03, -4.885344537955377e-03, 3.901003837489495e-03, 8.498835520148953e-02,  6.273864345683090e-03, -7.200386309267005e-04, -3.358382884597645e-02, -1.004280809592643e-02, -1.151701927809804e-03, -2.035007720867598e-02, 4.808727112410356e-03, 1.225135566130793e-02, 2.433695213688286e-03, -5.130095000966299e-03,  6.273864345683090e-03, 5.578086696904813e-02, -6.448679766086251e-03, -1.343881441913017e-02, 6.351618814768306e-03,  3.275405793066540e-02, -1.434420050950242e-02, 3.976911297430290e-02, -4.872906234463498e-03, 4.275588260015189e-03, -6.226364508819487e-04, -7.200386309267005e-04, -6.448679766086251e-03, 8.082054629417132e-02, 5.546495256122505e-03,  2.074600027338205e-02, 1.937918353018940e-03, 5.213926546803594e-03, 6.934274796726427e-04, -9.920405031825064e-03,  5.296176003724820e-03, -2.599459983664355e-03, -3.358382884597645e-02, -1.343881441913017e-02, 5.546495256122505e-03,  8.858761592720060e-02 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$lag_one_cov),
               c(NULL )
               , tolerance = 0.01)

  expect_equal(c(result_exp$fixed_effects),
               c(numeric(0) )
               , tolerance = 0.01)

  expect_equal(c(result_exp$n_iter),
               c(10 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$Q),
               c( 0.170774175745269374, 0.150601245426881192, -0.324411439259318701, 0.261310535974126901, 0.212502084268907404, -0.001182976527371735, -0.010389657982083290, 0.099007971300100819, 0.094983978103075795, 0.174758252120500557, -0.216664083930412893, 0.150601245426881192, 0.928275125179445437, 0.070594965324080675, 0.712115115254548847, -0.274281600040261753, 0.033720850171021707, 0.031953146249362824, 0.372905255726187268, -0.032040140332981450,  0.745412600799243874, -0.189591411756892259, -0.324411439259318701, 0.070594965324080675, 2.292785050164081184, -0.971372595071613665, -1.294483409880044178, -0.126641890728242223, 0.254291536552086472, 0.545447688280203957, -0.560527225642866167, -0.505308689053857618, 0.369159740943566961, 0.261310535974126901, 0.712115115254548847, -0.971372595071613665, 1.023911499066027186, 0.355073247570799055, 0.086050052388584042, -0.088464408027687805,  0.047589500949222797, 0.221358686523219966, 0.838654154522284201, -0.310018198159935698, 0.212502084268907404, -0.274281600040261753, -1.294483409880044178, 0.355073247570799055, 0.854499152624077718, 0.042468644652391072, -0.133521498502737035, -0.311227489311219574, 0.335318509535591613, 0.068725736841120447, -0.257899905112093430, -0.001182976527371735, 0.033720850171021707, -0.126641890728242223, 0.086050052388584042, 0.042468644652391072,  0.023624838919816250, -0.019213378078115177, -0.047755075807346350, 0.026572284816633970, 0.067000431509695452,  0.007880684854614295, -0.010389657982083290, 0.031953146249362824, 0.254291536552086472, -0.088464408027687805, -0.133521498502737035, -0.019213378078115177, 0.041661909603711911, 0.096690649350824498, -0.060612397704672048, -0.040064787711579369, 0.005156495699132680, 0.099007971300100819, 0.372905255726187268, 0.545447688280203957,  0.047589500949222797, -0.311227489311219574, -0.047755075807346350, 0.096690649350824498, 0.426880173675462271, -0.111253366696134573, 0.133491754049718248, -0.160159978793164626, 0.094983978103075795, -0.032040140332981450, -0.560527225642866167, 0.221358686523219966, 0.335318509535591613, 0.026572284816633970, -0.060612397704672048, -0.111253366696134573, 0.157130794994277234, 0.095932972757506535, -0.118251309692078754, 0.174758252120500557,  0.745412600799243874, -0.505308689053857618, 0.838654154522284201, 0.068725736841120447, 0.067000431509695452, -0.040064787711579369, 0.133491754049718248, 0.095932972757506535, 0.772445362062836871, -0.202786963224768246, -0.216664083930412893, -0.189591411756892259, 0.369159740943566961, -0.310018198159935698, -0.257899905112093430,  0.007880684854614295, 0.005156495699132680, -0.160159978793164626, -0.118251309692078754, -0.202786963224768246,  0.304165146762391925 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$Q_0),
               c(10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0,  0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0,  0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$n_risk),
               c(2041, 1968, 1876, 1813, 1684, 1539, 1448, 1262, 1189, 1042 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$times),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$risk_set),
               c(NULL )
               , tolerance = 0.01)

  expect_equal(c(result_exp$data),
               c(NULL )
               , tolerance = 0.01)

  expect_equal(c(result_exp$order),
               c(1 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$F_),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 0.01)

  expect_equal(c(result_exp$method),
               c("EKF" )
               , tolerance = 0.01)

  expect_equal(c(result_exp$model),
               c("exp_combined" )
               , tolerance = 0.01)

  expect_equal(c(result_exp$est_Q_0),
               c(FALSE )
               , tolerance = 0.01)

  expect_equal(c(result_exp$LR),
               c(0.1975308641975309 )
               , tolerance = 0.01)
})

test_that("Unmacthed control variable throw error",
          expect_error({
            result = ddhazard(
              formula = survival::Surv(start, stop, event) ~ group,
              data = head_neck_cancer,
              by = 1, # Use by month intervals
              a_0 = rep(0, 2), Q_0 = diag(1, 2), # Initial value
              max_T = 45,
              id = head_neck_cancer$id, order = 1,
              control = list("None_existing_parem" = 1)
            )}, regexp = "These control parameters are not recognized"))


test_that("Result of exponential model with only binary or right clipped time yield previous results", {
  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
    data = sims$res,
    by = (by_ <- 1),
    Q_0 = diag(10, 11),
    Q = diag(1e-2, 11),
    control = list(est_Q_0 = F, eps = 10^-2, n_max = 10^3,
                   save_data = F, save_risk_set = F),
    max_T = 10,
    id = sims$res$id, order = 1,
    verbose = F,
    model = "exp_bin"))

  # matplot(sims$betas, type = "l", lty = 1)
  # matplot(result_exp$state_vecs, lty = 2, type = "l", add = T)
  # get_expect_equal(result_exp, file = "tmp.txt")

  expect_equal(unname(c(result_exp$state_vecs)),
               c(-5.16309486548459429, -5.17333824066297687, -3.86741436934948490, -2.83866993395222922, -3.14595536541853127, -4.12033487638510909, -4.68784807454975283, -3.24058792193489742, -3.75665139927300773, -3.80301132579245937, -3.78546179084658174, 0.05714064782511509, 0.06839859215585706, 0.31973732520596876, -0.38263448352647572, -0.10090066376977758, -0.89055056102515329, -2.08893030351857689, -3.29928951437810980, -2.88091148836416666, -1.46421068992520831, -1.11214849122303949, -0.21203645837417406, -0.24537193770833349, -3.03737793395045941, -0.73186215768448182, -4.24242202734049201, -4.94543590436122749, -5.33690760648949691, -4.01573125618214277, -3.86468986163999517, -3.18927570240072766, -1.94983344759280031, 0.64701350694861792, 0.68258088478118428,  1.85345330186649093, -0.83361699846728932, 1.42787362259208916, 1.70049385262866593, 1.14799372674331379, -1.41348871841725265, -0.93158447936862210, 0.23538659959847047, 0.53065210451731715, -0.72744195996645811, -0.70878149255934386, 1.97106944706242126, 0.68922844328566069, 2.82626172918876861, 3.08561143722881237,  3.57132420387449301, 3.88185354375973590, 3.73355322277308588, 3.46505699146572521, 3.14777379138134172,  0.16307045737014955, 0.16453582848547610, -0.16833320793955697, -0.23406526925267160, -0.26276518219106376, -0.44398561893932603, -0.77525693932409934, -1.36129439291064092, -1.18072216777097405, -0.98011556525764087, -1.25939216139519683, -0.39884874027886130, -0.40000312497894103, -0.47244085792197266, -0.45244966760591243, -0.69801606585846943, -0.64677820002329789, -0.56349159634845380, -0.33463646546652215, -0.37366245299206519, -0.20943590378421656, 0.33099684937876861, 2.06233075352039785, 2.09296374719694933, 3.54611603459741609,  1.01436415975699190, 2.42565086400088470, 2.81274131408786987, 3.04137908835325099, 2.10277388042633362,  3.03346563517705015, 4.89830880686261239, 6.41081053456501682, 1.63034181087901708, 1.65724800647435866,  2.65181834503938818, 0.61997173485011281, 2.41990291047613892, 2.74928196552285709, 2.76731329888418154,  1.15758590173247455, 1.91997487284200719, 2.73491169193022987, 2.39021721657893771, 0.52652327433983892,  0.54348553170008862, 1.03563486785772052, -0.29095373358121540, 0.64265869060387404, 0.72908363408299914,  0.17529130259199788, -1.20697494241118752, -1.33747159219720713, -0.64409910586082764, 0.27786538589516119, -2.50958318138691272, -2.55445243134253719, -5.21610411330674495, -1.66354962842364640, -4.33067072730240810, -4.83483896861901385, -5.10484049651423355, -3.69435681930601634, -4.66980901775327339, -6.71524163332084711, -8.13094371201584387 ))

  expect_equal(unname(c(result_exp$state_vars)),
               c( 7.837230007260114e-01, 4.797333635186708e-02, 7.430659066889773e-02, -4.070245391211649e-01,  2.834176909079384e-01, -8.276266378671709e-02, 2.893268039026279e-03, -2.667000877015999e-01, -3.394262471048303e-01, -1.696531352673870e-01, 2.491539984006457e-01, 4.797333635186711e-02,  9.048693640314696e-01, 1.998736356701177e-01, 6.492161246260058e-01, -3.115528590731408e-01,  2.414706258614723e-01, -3.199071502265910e-02, 3.146883750439627e-01, 3.038462504495706e-01,  4.327027202138315e-01, -3.603979004566885e-01, 7.430659066889776e-02, 1.998736356701175e-01,  2.266836244839353e+00, -9.225479642552828e-01, -1.297851729478053e+00, 3.497094867274171e-02,  2.337830765240100e-01, -1.774328599928633e-01, -8.131879091686862e-01, -2.596738950301137e-01,  8.472341778424446e-01, -4.070245391211650e-01, 6.492161246260055e-01, -9.225479642552828e-01,  1.397171701893930e+00, 2.431122932559854e-01, 1.926191666942122e-01, -7.932458465305368e-02,  7.267922750269563e-01, 8.390316352189516e-01, 7.709039882049399e-01, -1.110415871197468e+00,  2.834176909079384e-01, -3.115528590731412e-01, -1.297851729478052e+00, 2.431122932559855e-01,  1.154713423698022e+00, -1.833073561516208e-01, -6.269218332579246e-02, 2.983894493075057e-01,  3.410289913498266e-01, -1.777524201476053e-02, -7.436380760672375e-01, -8.276266378671711e-02,  2.414706258614723e-01, 3.497094867274175e-02, 1.926191666942121e-01, -1.833073561516207e-01,  1.683757325130149e-01, -5.721119541548784e-02, -6.271036439273098e-02, 1.226071316968627e-01,  7.683232981712190e-02, 7.775548215238504e-02, 2.893268039026268e-03, -3.199071502265908e-02,  2.337830765240100e-01, -7.932458465305380e-02, -6.269218332579241e-02, -5.721119541548778e-02,  1.206355956517111e-01, 1.466533956262968e-01, -1.070320864960745e-01, 4.103593273081793e-02, -9.765986701410206e-02, -2.667000877015999e-01, 3.146883750439624e-01, -1.774328599928633e-01,  7.267922750269564e-01, 2.983894493075057e-01, -6.271036439273085e-02, 1.466533956262968e-01,  1.357581778134138e+00, 5.616123436854809e-01, 4.259393700127981e-01, -1.554935229433093e+00, -3.394262471048303e-01, 3.038462504495707e-01, -8.131879091686867e-01, 8.390316352189511e-01,  3.410289913498266e-01, 1.226071316968628e-01, -1.070320864960745e-01, 5.616123436854815e-01,  7.793943010282192e-01, 2.700537398716718e-01, -8.699472932229275e-01, -1.696531352673871e-01,  4.327027202138315e-01, -2.596738950301139e-01, 7.709039882049399e-01, -1.777524201476057e-02,  7.683232981712189e-02, 4.103593273081791e-02, 4.259393700127984e-01, 2.700537398716720e-01,  6.954005815042024e-01, -5.807010621047282e-01, 2.491539984006457e-01, -3.603979004566885e-01,  8.472341778424449e-01, -1.110415871197467e+00, -7.436380760672374e-01, 7.775548215238512e-02, -9.765986701410209e-02, -1.554935229433093e+00, -8.699472932229277e-01, -5.807010621047282e-01,  2.094135775774953e+00, 1.660846966202624e-01, -9.507848169651965e-03, 1.958985403911415e-02, -1.184039112400004e-01, 2.881612633584539e-02, -9.793416848477985e-03, -9.662347464869783e-03, -1.564520410179993e-01, -1.096234894567353e-01, -5.001949100777938e-02, 1.717131893852962e-01, -9.507848169651910e-03, 3.312980464275775e-01, 1.116998572586786e-01, 1.795146318399461e-01, -1.706671555270417e-01, 1.057913527647220e-01, -2.588107229685603e-02, 1.230613223448699e-02,  8.373090961856866e-02, 1.058316950105698e-01, 2.436858579416038e-03, 1.958985403911412e-02,  1.116998572586784e-01, 5.744704671021229e-01, -1.383891768933988e-01, -2.828467851675898e-01,  5.445201759165040e-03, 7.324296281820963e-02, 1.085590639037839e-01, -1.375351205162988e-01, -9.663000582036656e-03, 2.780057530403519e-02, -1.184039112400005e-01, 1.795146318399458e-01, -1.383891768933987e-01, 3.524754626050938e-01, -3.684021169022557e-02, 6.947261756907717e-02, -1.999243094350762e-02, 1.037680381085074e-01, 1.646737344553072e-01, 2.023349451871042e-01, -1.606673915755535e-01, 2.881612633584534e-02, -1.706671555270421e-01, -2.828467851675894e-01, -3.684021169022549e-02, 2.867613092169307e-01, -6.102114481434614e-02, -1.167078672397322e-02, -4.438698699253172e-03, 2.849998168204832e-02, -6.301000528295164e-02, -7.557391565051066e-02, -9.793416848477995e-03, 1.057913527647220e-01, 5.445201759165082e-03, 6.947261756907719e-02, -6.102114481434609e-02, 9.352231905445985e-02, -3.051845256086575e-02, -3.531581929846735e-02,  5.608168227570368e-02, 1.322268219131356e-02, 3.827748657523507e-02, -9.662347464869783e-03, -2.588107229685602e-02, 7.324296281820955e-02, -1.999243094350771e-02, -1.167078672397316e-02, -3.051845256086570e-02, 7.360072718995958e-02, 7.030820656963085e-02, -4.394343980965409e-02,  3.301028863255347e-02, -5.624253552533658e-02, -1.564520410179993e-01, 1.230613223448701e-02,  1.085590639037838e-01, 1.037680381085074e-01, -4.438698699253334e-03, -3.531581929846722e-02,  7.030820656963077e-02, 4.289737863363813e-01, 1.056806525880525e-01, 6.671320489238615e-02, -4.001955109326328e-01, -1.096234894567354e-01, 8.373090961856886e-02, -1.375351205162990e-01,  1.646737344553068e-01, 2.849998168204812e-02, 5.608168227570369e-02, -4.394343980965405e-02,  1.056806525880529e-01, 2.385682120104778e-01, -5.821792914561044e-03, -1.552824377515029e-01, -5.001949100777937e-02, 1.058316950105699e-01, -9.663000582036761e-03, 2.023349451871041e-01, -6.301000528295164e-02, 1.322268219131352e-02, 3.301028863255345e-02, 6.671320489238633e-02, -5.821792914561127e-03, 2.822765589525692e-01, -8.363132811516712e-02, 1.717131893852962e-01,  2.436858579415823e-03, 2.780057530403508e-02, -1.606673915755535e-01, -7.557391565051080e-02,  3.827748657523516e-02, -5.624253552533660e-02, -4.001955109326326e-01, -1.552824377515029e-01, -8.363132811516716e-02, 4.999719563122561e-01, 1.601964478490424e-01, -1.728224212046307e-02,  1.455533409192518e-02, -1.175471347703901e-01, 4.331116436654012e-02, -1.450848014952415e-02, -6.439807088407056e-03, -1.319461817026952e-01, -1.030807288497183e-01, -5.308643164082101e-02,  1.415381583765663e-01, -1.728224212046303e-02, 2.465366089518742e-01, 1.024222746130521e-01,  1.309948517483665e-01, -1.370659903737120e-01, 7.589816088212556e-02, -1.311762530706768e-02,  2.070143187515769e-02, 5.666804746470133e-02, 8.322559030285732e-02, -3.658851729951156e-03,  1.455533409192519e-02, 1.024222746130520e-01, 5.214338165696305e-01, -1.339918998019173e-01, -2.796745704295514e-01, 1.316485432898869e-02, 6.017167935091443e-02, 5.399406539041297e-02, -1.353639174052332e-01, -1.304964108370376e-02, 7.776297483714520e-02, -1.175471347703901e-01,  1.309948517483662e-01, -1.339918998019173e-01, 2.989742498754995e-01, -1.046035037076182e-02,  5.163590461571869e-02, -1.526092435491213e-02, 1.117742605047106e-01, 1.539846345613171e-01,  1.598826153201806e-01, -1.652494569382138e-01, 4.331116436654013e-02, -1.370659903737124e-01, -2.796745704295512e-01, -1.046035037076161e-02, 2.811623927417866e-01, -5.322045906056838e-02, -1.356577453536899e-02, 1.081627407691361e-02, 3.886942096405703e-02, -4.411227295066777e-02, -9.632882686334869e-02, -1.450848014952414e-02, 7.589816088212560e-02, 1.316485432898873e-02,  5.163590461571868e-02, -5.322045906056828e-02, 7.262648046929376e-02, -1.962321367605767e-02, -2.325239286102376e-02, 3.805218138503446e-02, 1.454952619543018e-02,
                  2.947481600951479e-02, -6.439807088407058e-03, -1.311762530706765e-02, 6.017167935091435e-02, -1.526092435491223e-02, -1.356577453536895e-02, -1.962321367605763e-02, 5.805721093778160e-02, 5.136458399478017e-02, -3.069717097221050e-02, 2.113131626327275e-02, -3.942252493417155e-02, -1.319461817026953e-01,  2.070143187515778e-02, 5.399406539041302e-02, 1.117742605047106e-01, 1.081627407691349e-02, -2.325239286102364e-02, 5.136458399478012e-02, 3.524874583657790e-01, 1.034407724775102e-01,  6.787698148264557e-02, -3.384421693270996e-01, -1.030807288497183e-01, 5.666804746470139e-02, -1.353639174052333e-01, 1.539846345613170e-01, 3.886942096405674e-02, 3.805218138503447e-02, -3.069717097221046e-02, 1.034407724775104e-01, 2.004691894525154e-01, 1.865380394373797e-02, -1.516829559927326e-01, -5.308643164082100e-02, 8.322559030285735e-02, -1.304964108370381e-02,  1.598826153201807e-01, -4.411227295066782e-02, 1.454952619543015e-02, 2.113131626327273e-02,  6.787698148264593e-02, 1.865380394373797e-02, 2.028115765975405e-01, -8.310994496203020e-02,  1.415381583765663e-01, -3.658851729951142e-03, 7.776297483714528e-02, -1.652494569382136e-01, -9.632882686334858e-02, 2.947481600951490e-02, -3.942252493417157e-02, -3.384421693270993e-01, -1.516829559927331e-01, -8.310994496203046e-02, 4.417093472254114e-01, 5.268400258572594e-02,  7.824433525792145e-03, 3.358168331699876e-02, -4.390556290397050e-02, -1.450929120031760e-02,  1.178318588335087e-03, 3.248269539373854e-03, -4.931064176974173e-02, -4.295843114346824e-02, -1.690276442535195e-02, 6.806388919665839e-02, 7.824433525792172e-03, 7.397240136977710e-02,  1.958980168872901e-02, 1.411595894617457e-02, -2.390669107911173e-02, 1.475471392518819e-02, -5.309454178562884e-03, -6.463051162994064e-03, 9.229359763901981e-03, 3.983007706636789e-03,  1.090929136524026e-02, 3.358168331699875e-02, 1.958980168872902e-02, 1.067247943992857e-01, -4.077994279726240e-02, -3.516074696382602e-02, 3.261106102233605e-03, 1.075580507440220e-02, -8.188526441346027e-03, -3.270035251662148e-02, -1.393513643266845e-02, 3.216918520986987e-02, -4.390556290397051e-02, 1.411595894617447e-02, -4.077994279726242e-02, 9.315651753631218e-02,  4.897568895876071e-03, 5.783194719680508e-03, -6.180904025465030e-03, 3.205197809586226e-02,  4.248215965942961e-02, 3.590523811255348e-02, -5.106146272143831e-02, -1.450929120031760e-02, -2.390669107911185e-02, -3.516074696382592e-02, 4.897568895876362e-03, 5.640465975600719e-02, -8.192323903571190e-03, 5.028618901220689e-04, 8.468082383994630e-03, 7.486900982743761e-03,  3.287054620587041e-05, -1.978800457214452e-02, 1.178318588335073e-03, 1.475471392518822e-02,  3.261106102233638e-03, 5.783194719680564e-03, -8.192323903571100e-03, 3.642077771949549e-02, -6.235048926412379e-03, -6.840840937650266e-03, 9.648990754189395e-03, -3.894890305913925e-03,  9.212641320634041e-03, 3.248269539373863e-03, -5.309454178562842e-03, 1.075580507440217e-02, -6.180904025465191e-03, 5.028618901220824e-04, -6.235048926412343e-03, 3.488203997547298e-02,  7.692727841096286e-03, -1.290029464432093e-02, 6.564034655423035e-03, -5.221635300531487e-03, -4.931064176974179e-02, -6.463051162994101e-03, -8.188526441346008e-03, 3.205197809586230e-02,  8.468082383994588e-03, -6.840840937650220e-03, 7.692727841096243e-03, 1.014091973614094e-01,  3.218678545809099e-02, 1.705102055969413e-02, -8.481663047683380e-02, -4.295843114346820e-02,  9.229359763902080e-03, -3.270035251662153e-02, 4.248215965942961e-02, 7.486900982743612e-03,  9.648990754189419e-03, -1.290029464432091e-02, 3.218678545809102e-02, 8.221690693056417e-02,  1.560716654667657e-05, -4.695904084153675e-02, -1.690276442535190e-02, 3.983007706636949e-03, -1.393513643266854e-02, 3.590523811255342e-02, 3.287054620581837e-05, -3.894890305913933e-03,  6.564034655422993e-03, 1.705102055969427e-02, 1.560716654669912e-05, 6.972259171100720e-02, -2.513440248830888e-02, 6.806388919665839e-02, 1.090929136524022e-02, 3.216918520986993e-02, -5.106146272143803e-02, -1.978800457214460e-02, 9.212641320634068e-03, -5.221635300531442e-03, -8.481663047683353e-02, -4.695904084153700e-02, -2.513440248830916e-02, 1.409965352044604e-01,  1.789339623281499e-02, 1.072401307751488e-02, 5.225838012531254e-03, -2.286475465068513e-04, -2.582635420893185e-03, 1.989367765873429e-03, -6.409780788336084e-06, -1.228221523580656e-02, -7.206418504142065e-03, 2.471602228737349e-03, 1.462272793564895e-02, 1.072401307751488e-02,  8.444288287686780e-02, 1.763567770367015e-02, 3.025002155407043e-02, -2.488777561605195e-02,  1.855005361845569e-02, -4.672985747486568e-03, 2.421370438981205e-03, 1.439445916807833e-02,  1.765966702279766e-02, -1.518337722109197e-03, 5.225838012531273e-03, 1.763567770367003e-02,  1.352700667067895e-01, -3.470640006620439e-02, -5.712872218517641e-02, 2.242173887566374e-03,  1.374601736283894e-02, 1.143180058822103e-02, -3.194969146472153e-02, -8.788912972890892e-03,  1.776717718974788e-02, -2.286475465068305e-04, 3.025002155407036e-02, -3.470640006620443e-02,  7.714986488354690e-02, 7.930891576388537e-03, 9.158847225192246e-03, -3.677134791508386e-03,  1.304569227018759e-02, 2.353028383986769e-02, 3.686852397125801e-02, -2.782694963375022e-02, -2.582635420893162e-03, -2.488777561605209e-02, -5.712872218517631e-02, 7.930891576388668e-03,  7.263850847211827e-02, -9.704719258211206e-03, -2.142184115607770e-03, 9.419064653246250e-03,  1.580355751044531e-02, -3.476198626852100e-03, -2.664678597082988e-02, 1.989367765873419e-03,  1.855005361845573e-02, 2.242173887566444e-03, 9.158847225192262e-03, -9.704719258211111e-03,  3.504944256575309e-02, -5.713076085976917e-03, -7.461583213269555e-03, 9.091031474944569e-03,  1.769977385495477e-04, 8.987260003341234e-03, -6.409780788347794e-06, -4.672985747486513e-03,  1.374601736283884e-02, -3.677134791508541e-03, -2.142184115607836e-03, -5.713076085976885e-03,  3.243397083230876e-02, 1.091469698608276e-02, -1.098057366443987e-02, 7.295930905345162e-03, -7.570875141081318e-03, -1.228221523580654e-02, 2.421370438981277e-03, 1.143180058822097e-02,  1.304569227018771e-02, 9.419064653246300e-03, -7.461583213269512e-03, 1.091469698608275e-02,  9.181781157915152e-02, 1.538851240694201e-02, 7.606836222841277e-03, -6.975641239381036e-02, -7.206418504142068e-03, 1.439445916807835e-02, -3.194969146472165e-02, 2.353028383986763e-02,  1.580355751044523e-02, 9.091031474944611e-03, -1.098057366443987e-02, 1.538851240694215e-02,  6.239397786391087e-02, -7.311460708495098e-03, -2.691898049763687e-02, 2.471602228737334e-03,  1.765966702279773e-02, -8.788912972890917e-03, 3.686852397125800e-02, -3.476198626852072e-03,  1.769977385495435e-04, 7.295930905345131e-03, 7.606836222841376e-03, -7.311460708495065e-03,  7.473134001466269e-02, -1.325303220575915e-02, 1.462272793564899e-02, -1.518337722109246e-03,  1.776717718974777e-02, -2.782694963375009e-02, -2.664678597083000e-02, 8.987260003341225e-03, -7.570875141081266e-03, -6.975641239381035e-02, -2.691898049763688e-02, -1.325303220575922e-02,  1.141543381245022e-01, 2.808329359396309e-02, 7.743984034620669e-03, 3.337276749288699e-02, -1.942389094725916e-02, -1.759596995328497e-02, 1.131421448092163e-03, 2.447694797244453e-03, -2.195725622606897e-02, -2.272752641167344e-02,
                  -4.932031464472237e-03, 3.418322358895984e-02,  7.743984034620672e-03, 5.948463897143503e-02, 1.726384662941868e-02, 1.251229481882404e-02, -1.997915765297820e-02, 1.119976518765933e-02, -2.218440538344908e-03, -1.502724048681065e-03,  5.602849533442480e-03, 7.288280395193750e-03, 3.950700641664678e-03, 3.337276749288701e-02,  1.726384662941862e-02, 1.103691230942912e-01, -3.000109592265297e-02, -3.800863026661448e-02,  3.747340862232483e-04, 1.234768466345347e-02, 4.347313379299157e-04, -3.250253234642865e-02, -6.564386747397435e-03, 2.298643750786442e-02, -1.942389094725916e-02, 1.251229481882398e-02, -3.000109592265298e-02, 6.055909654623270e-02, 6.360328436161737e-03, 5.882792715726902e-03, -3.194694306471152e-03, 1.368594409297450e-02, 2.093550793364305e-02, 2.440795960831826e-02, -2.693131530664843e-02, -1.759596995328494e-02, -1.997915765297829e-02, -3.800863026661438e-02,  6.360328436161956e-03, 5.424429133954758e-02, -5.915660409863650e-03, -2.054966494387138e-03,  1.119692090717599e-02, 1.557467248728412e-02, -3.543273444525410e-03, -2.244989856121758e-02,  1.131421448092149e-03, 1.119976518765935e-02, 3.747340862232583e-04, 5.882792715726920e-03, -5.915660409863618e-03, 2.992861796186086e-02, -3.922788268682876e-03, -4.510560950799613e-03,  6.815203097472693e-03, -6.644550317239553e-04, 5.245289695233311e-03, 2.447694797244433e-03, -2.218440538344866e-03, 1.234768466345336e-02, -3.194694306471255e-03, -2.054966494387150e-03, -3.922788268682850e-03, 2.937531680059275e-02, 6.807585608240161e-03, -9.535362503324204e-03,  5.696193259486581e-03, -3.718494780083267e-03, -2.195725622606896e-02, -1.502724048681091e-03,  4.347313379299382e-04, 1.368594409297466e-02, 1.119692090717601e-02, -4.510560950799575e-03,  6.807585608240113e-03, 7.740640267303950e-02, 1.923957091553471e-02, 3.643325448018495e-03, -5.913076416187918e-02, -2.272752641167343e-02, 5.602849533442504e-03, -3.250253234642876e-02,  2.093550793364279e-02, 1.557467248728407e-02, 6.815203097472678e-03, -9.535362503324187e-03,  1.923957091553482e-02, 5.979376453632854e-02, -7.618323702255776e-03, -3.101738535323043e-02, -4.932031464472257e-03, 7.288280395193820e-03, -6.564386747397480e-03, 2.440795960831820e-02, -3.543273444525332e-03, -6.644550317239596e-04, 5.696193259486552e-03, 3.643325448018527e-03, -7.618323702255814e-03, 5.985179132872748e-02, -8.178232039548258e-03, 3.418322358895985e-02,  3.950700641664668e-03, 2.298643750786424e-02, -2.693131530664843e-02, -2.244989856121767e-02,  5.245289695233397e-03, -3.718494780083251e-03, -5.913076416187900e-02, -3.101738535323053e-02, -8.178232039548202e-03, 1.016362656745730e-01, 6.117126065589116e-02, 2.586375151098383e-02,  7.933901057493462e-02, -4.351601098708110e-02, -3.842080007197699e-02, 4.567914655178046e-03,  4.227557635469743e-03, -4.391877438357455e-02, -4.918197678533376e-02, -9.493043055074396e-03,  6.760874785526040e-02, 2.586375151098383e-02, 1.083591237072460e-01, 5.790305619137068e-02,  2.633689313721891e-02, -5.038534293891912e-02, 2.520055491126139e-02, -4.391120380783333e-03, -8.161043790824067e-06, 5.157106680793238e-03, 2.158900831527822e-02, 9.270738244472946e-03,  7.933901057493466e-02, 5.790305619137060e-02, 2.193034528322315e-01, -6.727442834235454e-02, -8.821123351941938e-02, 5.870463113893125e-03, 2.231211229295844e-02, -5.292889818923063e-03, -7.372097061831627e-02, -7.399814630168478e-03, 5.085665731418929e-02, -4.351601098708105e-02,  2.633689313721878e-02, -6.727442834235445e-02, 1.055047234767080e-01, 1.363944374936946e-02,  1.232995725985438e-02, -8.532530617563629e-03, 3.579748361227815e-02, 5.262392650181089e-02,  4.517259781301135e-02, -5.973877659069731e-02, -3.842080007197693e-02, -5.038534293891927e-02, -8.821123351941919e-02, 1.363944374936957e-02, 8.912182168576870e-02, -1.520785134870331e-02, -3.438501184720454e-03, 2.465443916536757e-02, 3.302494496042653e-02, -1.103204632629187e-02, -4.827969840480130e-02, 4.567914655178033e-03, 2.520055491126142e-02, 5.870463113893117e-03,  1.232995725985435e-02, -1.520785134870328e-02, 3.661029184327765e-02, -6.761291369548406e-03, -1.032439015822583e-02, 9.522027780393221e-03, 2.503097949461137e-03, 1.280279506620114e-02,  4.227557635469716e-03, -4.391120380783279e-03, 2.231211229295837e-02, -8.532530617563714e-03, -3.438501184720430e-03, -6.761291369548390e-03, 3.274138452869450e-02, 1.179146544161019e-02, -1.483153900794940e-02, 6.446233476822622e-03, -7.729368506347215e-03, -4.391877438357456e-02, -8.161043790870254e-06, -5.292889818923051e-03, 3.579748361227835e-02, 2.465443916536767e-02, -1.032439015822582e-02, 1.179146544161014e-02, 1.347429374721417e-01, 4.449330392252157e-02,  1.098706873503277e-02, -1.265430760717100e-01, -4.918197678533375e-02, 5.157106680793293e-03, -7.372097061831638e-02, 5.262392650181100e-02, 3.302494496042654e-02, 9.522027780393188e-03, -1.483153900794940e-02, 4.449330392252197e-02, 9.437230473598869e-02, -4.451507017895652e-03, -6.807181574423429e-02, -9.493043055074405e-03, 2.158900831527824e-02, -7.399814630168432e-03,  4.517259781301141e-02, -1.103204632629190e-02, 2.503097949461141e-03, 6.446233476822567e-03,  1.098706873503266e-02, -4.451507017895689e-03, 8.149510873001239e-02, -1.653564387235409e-02,  6.760874785526041e-02, 9.270738244472916e-03, 5.085665731418919e-02, -5.973877659069705e-02, -4.827969840480131e-02, 1.280279506620110e-02, -7.729368506347126e-03, -1.265430760717098e-01, -6.807181574423472e-02, -1.653564387235401e-02, 1.849944177834729e-01, 9.153477524048576e-02,  4.812081644540211e-02, 1.115907327026091e-01, -5.661481282019357e-02, -5.213588063510607e-02,  7.511849824505926e-03, 4.508161089659066e-03, -5.756467288232602e-02, -6.620173029827961e-02, -1.046197081461656e-02, 8.775955334939568e-02, 4.812081644540211e-02, 1.408886891096023e-01,  7.406242195009716e-02, 3.202371611905709e-02, -6.831765284654631e-02, 3.651265669349418e-02, -8.360141480364326e-03, -2.566695212680046e-02, -1.267419236664410e-03, 2.940391315849959e-02,  3.680219549045299e-02, 1.115907327026092e-01, 7.406242195009710e-02, 2.642952390572796e-01, -9.008794419531659e-02, -1.023288402722684e-01, 3.546682762343370e-03, 2.387326577055116e-02, -1.477685675858067e-02, -9.601930647387001e-02, -1.114182040219678e-02, 6.811799131921883e-02, -5.661481282019357e-02, 3.202371611905710e-02, -9.008794419531671e-02, 1.298995306082338e-01,  1.164451756878204e-02, 2.036971191554144e-02, -1.092470230278017e-02, 3.146863362021980e-02,  6.648965530431683e-02, 5.693519773794916e-02, -5.990762794212623e-02, -5.213588063510603e-02, -6.831765284654637e-02, -1.023288402722684e-01, 1.164451756878217e-02, 9.882271843424284e-02, -1.842823257818335e-02, -1.029506042017223e-03, 3.061114837339569e-02, 3.536158713793855e-02, -1.437287215255200e-02, -5.558124056222201e-02, 7.511849824505930e-03, 3.651265669349420e-02,  3.546682762343379e-03, 2.036971191554143e-02, -1.842823257818337e-02, 4.278531777972311e-02, -9.564222447235673e-03, -1.642877095332721e-02, 1.383787937952611e-02, 5.570681188579490e-03,  1.706079381728470e-02, 4.508161089659042e-03, -8.360141480364305e-03, 2.387326577055115e-02, -1.092470230278026e-02, -1.029506042017199e-03, -9.564222447235661e-03, 3.550714813524103e-02,  1.721416820052925e-02,
                  -1.646269778134364e-02, 7.125581500726418e-03, -1.334966621012470e-02, -5.756467288232603e-02, -2.566695212680048e-02, -1.477685675858053e-02, 3.146863362022000e-02,  3.061114837339582e-02, -1.642877095332722e-02, 1.721416820052927e-02, 1.429572922918941e-01,  4.144032552910207e-02, 1.318145606198610e-02, -1.350287080939237e-01, -6.620173029827964e-02, -1.267419236664371e-03, -9.601930647387007e-02, 6.648965530431666e-02, 3.536158713793858e-02,  1.383787937952609e-02, -1.646269778134365e-02, 4.144032552910228e-02, 1.059266279120374e-01, -1.968890956459447e-04, -6.966410746570982e-02, -1.046197081461657e-02, 2.940391315849960e-02, -1.114182040219682e-02, 5.693519773794910e-02, -1.437287215255195e-02, 5.570681188579521e-03,  7.125581500726360e-03, 1.318145606198614e-02, -1.968890956460297e-04, 9.149510871177322e-02, -1.882763352693015e-02, 8.775955334939566e-02, 3.680219549045297e-02, 6.811799131921870e-02, -5.990762794212611e-02, -5.558124056222191e-02, 1.706079381728468e-02, -1.334966621012461e-02, -1.350287080939236e-01, -6.966410746571003e-02, -1.882763352693005e-02, 1.966235383791935e-01,  3.315946776693457e-02, 4.042997031809790e-02, 3.737700554617814e-02, 2.943789995369457e-03, -3.004684219075429e-02, 1.230269908440501e-02, -1.709176871402551e-03, -2.255771525228744e-02, -1.281796090058826e-02, 9.543652924336730e-03, 3.220352814956523e-02, 4.042997031809790e-02,  1.104364133562063e-01, 4.094342666065755e-02, 2.760359267586972e-02, -3.874028656042425e-02,  2.405631516022370e-02, -4.968594004536296e-03, -1.494957950075762e-02, 1.291068416054789e-03,  2.195302821997095e-02, 2.075159039436047e-02, 3.737700554617813e-02, 4.094342666065761e-02,  1.211578713384951e-01, -1.854153874069387e-02, -5.524973567802308e-02, 1.079028550129402e-02,  4.936646255131800e-03, -1.524776480272381e-02, -3.140670044337426e-02, 5.749164173142229e-03,  3.825548970330171e-02, 2.943789995369454e-03, 2.760359267586975e-02, -1.854153874069386e-02,  6.437086394828632e-02, 2.773487182052924e-03, 7.557135373802631e-03, -6.840485219722211e-04,  1.427549308451290e-02, 1.594141741828602e-02, 2.667403503917490e-02, -2.210036263509149e-02, -3.004684219075428e-02, -3.874028656042422e-02, -5.524973567802303e-02, 2.773487182052989e-03,  7.676368843641215e-02, -1.364830979859276e-02, 1.430459444135003e-03, 2.960044619920225e-02,  2.388486731354018e-02, -1.227307380868807e-02, -4.673212665663373e-02, 1.230269908440501e-02,  2.405631516022369e-02, 1.079028550129400e-02, 7.557135373802627e-03, -1.364830979859277e-02,  3.728165228100981e-02, -4.220852769352180e-03, -1.207059265527682e-02, 3.243235835363587e-03,  2.117341708573356e-03, 1.506176783637213e-02, -1.709176871402558e-03, -4.968594004536314e-03,  4.936646255131800e-03, -6.840485219722779e-04, 1.430459444134992e-03, -4.220852769352166e-03,  3.048284624459465e-02, 7.785104116455786e-03, -5.501100095516857e-03, 6.446828588914618e-03, -7.657954864660171e-03, -2.255771525228745e-02, -1.494957950075761e-02, -1.524776480272389e-02,  1.427549308451294e-02, 2.960044619920224e-02, -1.207059265527681e-02, 7.785104116455800e-03,  9.939669619042998e-02, 2.729900457074657e-02, 6.483573075538097e-04, -8.481530850179075e-02, -1.281796090058827e-02, 1.291068416054831e-03, -3.140670044337426e-02, 1.594141741828597e-02,  2.388486731354015e-02, 3.243235835363574e-03, -5.501100095516839e-03, 2.729900457074673e-02,  5.944179126561006e-02, -1.264108704759640e-02, -3.914123509144062e-02, 9.543652924336725e-03,  2.195302821997096e-02, 5.749164173142231e-03, 2.667403503917493e-02, -1.227307380868800e-02,  2.117341708573360e-03, 6.446828588914580e-03, 6.483573075538418e-04, -1.264108704759640e-02,  6.633505083560902e-02, 1.925086618196009e-03, 3.220352814956524e-02, 2.075159039436050e-02,  3.825548970330175e-02, -2.210036263509141e-02, -4.673212665663370e-02, 1.506176783637212e-02, -7.657954864660101e-03, -8.481530850179078e-02, -3.914123509144064e-02, 1.925086618195970e-03,  1.349526542888319e-01, 5.337749993835923e-02, 4.937575903314968e-02, 5.584817125953473e-02, -7.641094880837686e-03, -4.219411695244771e-02, 1.511850266904931e-02, -1.699963203035401e-03, -4.545046601110448e-02, -3.173130095046604e-02, 1.136365067131804e-02, 6.200294490838529e-02,  4.937575903314968e-02, 1.381712695614205e-01, 5.702527367145657e-02, 3.311146500429857e-02, -5.560236803295424e-02, 3.246596227714741e-02, -7.803498824974140e-03, -2.271340551145790e-02, -1.762106229104326e-03, 2.839402427284683e-02, 3.514690649675189e-02, 5.584817125953474e-02,  5.702527367145657e-02, 1.784591264650978e-01, -4.231533297962461e-02, -7.871540165538024e-02,  1.154321137192586e-02, 1.270506447206168e-02, -1.551083514168872e-02, -5.201439635617474e-02, -1.493143414296175e-03, 5.080661197567716e-02, -7.641094880837682e-03, 3.311146500429856e-02, -4.231533297962470e-02, 9.251009284831431e-02, 8.525703565094914e-03, 1.017556841641250e-02, -4.336567565431924e-03, 2.549443180880695e-02, 3.116207061393618e-02, 4.281587748116691e-02, -3.795459920327525e-02, -4.219411695244771e-02, -5.560236803295425e-02, -7.871540165538030e-02,  8.525703565094897e-03, 9.619740009329419e-02, -1.808470664903953e-02, 1.224401743894503e-03,  3.858012331150938e-02, 3.153280838733688e-02, -1.052620497478499e-02, -6.264271592902136e-02,  1.511850266904931e-02, 3.246596227714742e-02, 1.154321137192587e-02, 1.017556841641252e-02, -1.808470664903954e-02, 4.418456368601505e-02, -7.108006634637912e-03, -1.744745534822483e-02,  5.079963453880164e-03, 1.650168081252360e-03, 2.175599782731099e-02, -1.699963203035407e-03, -7.803498824974150e-03, 1.270506447206169e-02, -4.336567565431952e-03, 1.224401743894498e-03, -7.108006634637908e-03, 3.718687747666651e-02, 1.407318830385197e-02, -9.702196076042946e-03,  8.556558741827903e-03, -1.353852829451922e-02, -4.545046601110449e-02, -2.271340551145786e-02, -1.551083514168876e-02, 2.549443180880688e-02, 3.858012331150948e-02, -1.744745534822484e-02,  1.407318830385198e-02, 1.504097595170893e-01, 4.476588307599237e-02, 6.509662922199400e-03, -1.366454738705094e-01, -3.173130095046602e-02, -1.762106229104320e-03, -5.201439635617475e-02,  3.116207061393625e-02, 3.153280838733680e-02, 5.079963453880168e-03, -9.702196076042953e-03,  4.476588307599236e-02, 8.325896840821836e-02, -1.349612316786436e-02, -6.089716433851138e-02,  1.136365067131805e-02, 2.839402427284686e-02, -1.493143414296197e-03, 4.281587748116695e-02, -1.052620497478497e-02, 1.650168081252366e-03, 8.556558741827897e-03, 6.509662922199405e-03, -1.349612316786436e-02, 8.969142483185694e-02, -6.209307272920737e-03, 6.200294490838530e-02,  3.514690649675184e-02, 5.080661197567708e-02, -3.795459920327522e-02, -6.264271592902131e-02,  2.175599782731098e-02, -1.353852829451919e-02, -1.366454738705095e-01, -6.089716433851148e-02, -6.209307272920836e-03, 1.969741183453589e-01, 6.644528523689923e-02, 3.353828196045756e-02,  5.131206275770830e-02, -2.380223974013841e-02, -3.970279006153062e-02, 1.194127185334503e-02, -2.102820258996391e-03, -7.675398984152554e-02, -5.353137295844689e-02, 1.075615646840089e-02,  9.811134271227734e-02, 3.353828196045756e-02, 1.176277119306107e-01, 3.611786775496241e-02,  2.461073220150114e-02, -4.168883867163507e-02, 2.822760523137995e-02, -1.199679859273503e-02,
                  -2.615789475281158e-02, 3.613797883614971e-03, 1.314259723419428e-02, 3.435590578738370e-02,  5.131206275770830e-02, 3.611786775496241e-02, 1.704863560167747e-01, -4.934227990274379e-02, -6.883949293840558e-02, 7.829011607834030e-03, 1.298281066059364e-02, -1.770584862827597e-02, -5.733705605158852e-02, -2.085630625970895e-03, 5.229810186494524e-02, -2.380223974013841e-02,  2.461073220150114e-02, -4.934227990274379e-02, 9.950055219702715e-02, 9.935928939353791e-03,  7.851614993825948e-03, -5.550242654409999e-03, 3.358196982204593e-02, 4.061046682848987e-02,  3.943768082572607e-02, -5.016650109111830e-02, -3.970279006153062e-02, -4.168883867163507e-02, -6.883949293840558e-02, 9.935928939353791e-03, 9.249568824205559e-02, -1.632367337365854e-02,  2.724053642401495e-03, 3.789918348736034e-02, 3.130570424333728e-02, -8.899288746378117e-03, -6.065357746120395e-02, 1.194127185334503e-02, 2.822760523137995e-02, 7.829011607834030e-03,  7.851614993825948e-03, -1.632367337365854e-02, 4.902652511948347e-02, -9.975988309378483e-03, -2.012447334678656e-02, 8.309515707907621e-03, -5.137147994401834e-03, 2.418037863287812e-02, -2.102820258996391e-03, -1.199679859273503e-02, 1.298281066059364e-02, -5.550242654409999e-03,  2.724053642401495e-03, -9.975988309378483e-03, 4.396467926042273e-02, 1.573632403374913e-02, -1.451009951173014e-02, 1.398346559335680e-02, -1.581299230548473e-02, -7.675398984152554e-02, -2.615789475281158e-02, -1.770584862827597e-02, 3.358196982204593e-02, 3.789918348736034e-02, -2.012447334678656e-02, 1.573632403374913e-02, 1.869715022861864e-01, 6.726681215168456e-02,  1.660578268047069e-03, -1.747995893206164e-01, -5.353137295844689e-02, 3.613797883614971e-03, -5.733705605158852e-02, 4.061046682848987e-02, 3.130570424333728e-02, 8.309515707907621e-03, -1.451009951173014e-02, 6.726681215168456e-02, 1.150434628921192e-01, -2.894950539421561e-02, -8.773758348327104e-02, 1.075615646840089e-02, 1.314259723419428e-02, -2.085630625970895e-03,  3.943768082572607e-02, -8.899288746378117e-03, -5.137147994401834e-03, 1.398346559335680e-02,  1.660578268047069e-03, -2.894950539421561e-02, 1.069898381186380e-01, -2.344478318663864e-03,  9.811134271227734e-02, 3.435590578738370e-02, 5.229810186494524e-02, -5.016650109111830e-02, -6.065357746120395e-02, 2.418037863287812e-02, -1.581299230548473e-02, -1.747995893206164e-01, -8.773758348327104e-02, -2.344478318663864e-03, 2.471546526445470e-01 ))

  expect_equal(unname(c(result_exp$lag_one_cov)),
               c(NULL ))

  expect_equal(unname(c(result_exp$fixed_effects)),
               c(numeric(0) ))

  expect_equal(unname(c(result_exp$n_iter)),
               c(21 ))

  expect_equal(unname(c(result_exp$Q)),
               c( 0.819621921069278114, -0.076633907204574109, 0.337044266513460311, -0.699195664180150223, 0.154413575982610074, -0.114437104911859738, 0.026039075092820746, -0.454286356430655403, -0.546316490795121901, -0.311935669027262774,  0.546756707705478462, -0.076633907204574109, 0.879035174669110297, -0.149477955388155614, 0.969019643425764077, -0.064076823790412921, 0.213199331003754100, -0.021116927757207998, 0.708694455541440949, 0.571657609268234279,  0.582156781400191781, -0.908835340620173793, 0.337044266513460311, -0.149477955388155614, 3.527404425816398348, -2.165822714531850934, -2.142917404245006097, 0.028313498655752938, 0.268404071848330816, -1.349219589143979992, -1.791436856691492663, -0.826325448960591791, 2.585051300216389514, -0.699195664180150223, 0.969019643425764077, -2.165822714531850934, 2.544540157847527961, 1.042807024122708315, 0.203270193573661745, -0.116906489614251513,  1.874397965679741596, 1.803159215259719117, 1.287162759932121592, -2.800871055164192214, 0.154413575982610074, -0.064076823790412921, -2.142917404245006097, 1.042807024122708315, 1.636086445311627058, -0.168665487312464363, -0.096174875545221222, 0.961647565032418883, 0.932910528952014761, 0.371502645393316455, -1.764302933966887865, -0.114437104911859738, 0.213199331003754100, 0.028313498655752938, 0.203270193573661745, -0.168665487312464363,  0.112669489811701690, -0.042238232923053870, -0.005916982160182448, 0.128940345338819679, 0.090159340743069361,  0.014921894994289843, 0.026039075092820746, -0.021116927757207998, 0.268404071848330816, -0.116906489614251513, -0.096174875545221222, -0.042238232923053870, 0.070308650536177297, 0.080809716179047086, -0.125038451305420822,  0.008578150461200810, -0.015814891243669381, -0.454286356430655403, 0.708694455541440949, -1.349219589143979992,  1.874397965679741596, 0.961647565032418883, -0.005916982160182448, 0.080809716179047086, 2.158997725040637139,  1.385317061064939015, 1.002176310503482437, -2.890092892493813626, -0.546316490795121901, 0.571657609268234279, -1.791436856691492663, 1.803159215259719117, 0.932910528952014761, 0.128940345338819679, -0.125038451305420822,  1.385317061064939015, 1.429038184502237741, 0.774805445105931412, -2.112924294377452217, -0.311935669027262774,  0.582156781400191781, -0.826325448960591791, 1.287162759932121592, 0.371502645393316455, 0.090159340743069361,  0.008578150461200810, 1.002176310503482437, 0.774805445105931412, 0.807802699220983689, -1.411486779874163044,  0.546756707705478462, -0.908835340620173793, 2.585051300216389514, -2.800871055164192214, -1.764302933966887865,  0.014921894994289843, -0.015814891243669381, -2.890092892493813626, -2.112924294377452217, -1.411486779874163044,  4.120311957824151605 ))

  expect_equal(unname(c(result_exp$Q_0)),
               c(10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 10 ))

  expect_equal(unname(c(result_exp$n_risk)),
               c(2041, 1968, 1876, 1813, 1684, 1539, 1448, 1262, 1189, 1042 ))

  expect_equal(unname(c(result_exp$times)),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 ))

  expect_equal(unname(c(result_exp$risk_set)),
               c(NULL ))

  expect_equal(unname(c(result_exp$data)),
               c(NULL ))

  expect_equal(unname(c(result_exp$order)),
               c(1 ))

  expect_equal(unname(c(result_exp$F_)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 ))

  expect_equal(unname(c(result_exp$method)),
               c("EKF" ))

  expect_equal(unname(c(result_exp$model)),
               c("exp_bin" ))

  expect_equal(unname(c(result_exp$est_Q_0)),
               c(FALSE ))

  expect_equal(unname(c(result_exp$LR)),
               c(1 ))

  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
    data = sims$res,
    by = (by_ <- 1),
    Q_0 = diag(10, 11),
    Q = diag(1e-2, 11),
    control = list(est_Q_0 = F, eps = 10^-2, n_max = 10^3,
                   save_data = F, save_risk_set = F, ridge_eps = 1e-4),
    max_T = 10,
    id = sims$res$id, order = 1,
    verbose = F,
    model = "exp_clip_time"))
  # sink()
  # close(tmp)

  # matplot(sims$betas, type = "l", lty = 1)
  # matplot(result_exp$state_vecs, lty = 2, type = "l", add = T)
  # get_expect_equal(result_exp, file = "tmp.txt", eps = 1e-4)

  expect_equal(unname(c(result_exp$state_vecs)),
               c(-4.83496631066747540, -4.83827657128822253, -3.60104420539053915, -3.92472537377249475, -3.84261230457305825, -3.98848891583527054, -3.98261420292011126, -3.20774772035663425, -3.32704256445864255, -3.18881713139187095, -2.84934220855557152, 0.50248605882036335, 0.50112413473985196, -0.44243243011193467, -0.61481561855225686, -0.03557656801026665, -0.58573102426325374, -1.24366294864792204, -3.32116639491237908, -2.40812382675990078, -0.93110116593594572, -0.33883679377078946, -0.10570533239245440, -0.08728595457141919, -2.89176481450374645, -1.64647083757056567, -4.84722271051326548, -4.11981475631735883, -4.22388670853599724, -3.24088942409996950, -3.15646510399733193, -2.37693277529079783, -1.47568107734313481, 0.78781961613123153, 0.78037979027337667,  0.96674700253174972, 0.36797074741987806, 1.93193348844114654, 1.25751900256890758, 0.77865380113381089, -1.28601460075800555, -0.57142771134210013, 0.32306761705157061, 0.46776185938348924, -0.04318982940565610, -0.05352587681895304, 2.26841761981651935, 1.53142270736535635, 2.93921596936607887, 2.62222211087362966,  2.82237223257756442, 3.37043185620091146, 3.09984113021527108, 2.59308837387206115, 2.35174504778521110, -0.01496927335405050, -0.01743845271427158, -0.11044204022905912, -0.30299534575977494, 0.27856525420953321,  0.01184740369567214, -0.16605642026132439, -1.10152092463820139, -0.81341322528647997, -0.60440432350110784, -0.71853163284994159, -0.23648822498303176, -0.23481743598188440, -0.43761701229136463, -0.34229414358215127, -0.67410864520680880, -0.62782954619519504, -0.68306414546951877, -0.56200229249430378, -0.50486654538339359, -0.22668100926900536, 0.06411827495756994, 2.19738089927888280, 2.19686406450889748, 2.59663752706145656,  2.43006974500895989, 2.19898235860984714, 2.06641690769811870, 1.88433036809802656, 2.14561666663507467,  2.40205598362805439, 3.20226290581567063, 3.94415319868755443, 1.01809275815945566, 1.00950104278848984,  2.05807336356373050, 1.41473378091235724, 2.93681849984709986, 2.42761476053307801, 2.26197964518364092,  1.28765275788747546, 1.60238314320217046, 1.78580837956156868, 1.59593224716980586, 0.71962615634994320,  0.71682267565947988, 0.26623974459571531, 0.04521802526873658, 0.86325212673820839, 0.50377501225697618,  0.16016827642781836, -1.43658189979056927, -0.97431220240600236, -0.37542276885955256, -0.27256397104620922, -2.89798921568332091, -2.88975565229916898, -4.46308799788953348, -3.75718316990657275, -4.82970461189994715, -4.33684759124135510, -4.16423812971313723, -3.96885343242957678, -4.24366408670328088, -4.85600626632587851, -5.31331110481035029 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$state_vars)),
               c( 3.121219020665862e-01, -2.329417813203776e-01, -1.648301215750859e-01, -1.424023540695966e-01,  3.187342259737159e-01, -1.031071304957204e-01, 1.830830440654304e-02, 1.536862306942879e-01,  2.797349203493293e-02, -1.934709458188047e-01, -2.259929889462741e-01, -2.329417813203776e-01,  1.223891999775164e+00, 1.843017928490470e-01, 8.753347755835711e-01, -4.759886159733369e-01,  3.366121292608998e-01, 8.081549380522859e-02, 1.843526571168002e-01, 2.554328082254913e-01,  6.853894484792592e-01, -1.654219301430441e-01, -1.648301215750860e-01, 1.843017928490470e-01,  2.129718768163152e+00, -5.686560599712277e-01, -1.068627592840221e+00, -2.415304199056048e-01,  2.680912660465497e-01, 2.797519719424539e-01, -8.218789120115663e-01, -2.012015748935407e-01,  5.459132393190331e-01, -1.424023540695966e-01, 8.753347755835709e-01, -5.686560599712277e-01,  9.667593398054581e-01, -1.982938367741119e-02, 3.634579777497544e-01, -3.155971402894950e-02,  4.739408271310036e-02, 4.937403715964215e-01, 6.331044037045442e-01, -3.202750180101099e-01,  3.187342259737159e-01, -4.759886159733371e-01, -1.068627592840221e+00, -1.982938367741111e-02,  8.650436542484332e-01, -3.803530886295028e-02, -1.141223414938790e-01, -2.138589380633664e-02,  3.436938465523707e-01, -1.891598072931060e-01, -4.009389584191862e-01, -1.031071304957204e-01,  3.366121292608997e-01, -2.415304199056048e-01, 3.634579777497544e-01, -3.803530886295026e-02,  2.205674529615216e-01, -3.250028411811316e-02, -4.210854113046242e-02, 2.025159713562346e-01,  2.606808905886269e-01, -6.626895125898880e-02, 1.830830440654304e-02, 8.081549380522858e-02,  2.680912660465498e-01, -3.155971402894947e-02, -1.141223414938789e-01, -3.250028411811315e-02,  1.076187701230911e-01, 1.017823386012377e-01, -8.929477107065163e-02, -5.133072341569336e-03, -7.860689280502504e-04, 1.536862306942877e-01, 1.843526571168002e-01, 2.797519719424538e-01,  4.739408271310025e-02, -2.138589380633667e-02, -4.210854113046241e-02, 1.017823386012376e-01,  3.716600278968532e-01, -2.506267067953106e-02, -6.107391120557040e-03, -2.321037818948964e-01,  2.797349203493292e-02, 2.554328082254914e-01, -8.218789120115663e-01, 4.937403715964216e-01,  3.436938465523706e-01, 2.025159713562346e-01, -8.929477107065165e-02, -2.506267067953110e-02,  5.052949121932855e-01, 2.555493125549570e-01, -3.232425552020554e-01, -1.934709458188047e-01,  6.853894484792595e-01, -2.012015748935408e-01, 6.331044037045441e-01, -1.891598072931061e-01,  2.606808905886269e-01, -5.133072341569343e-03, -6.107391120557139e-03, 2.555493125549571e-01,  5.489828450083767e-01, -9.951367339933617e-02, -2.259929889462740e-01, -1.654219301430441e-01,  5.459132393190332e-01, -3.202750180101100e-01, -4.009389584191861e-01, -6.626895125898878e-02, -7.860689280503146e-04, -2.321037818948965e-01, -3.232425552020552e-01, -9.951367339933603e-02,  5.520331621499874e-01, 6.777735799106263e-02, -7.133639258830497e-02, 1.601402631154302e-02, -6.841444637889316e-02, 3.706565688552711e-02, -3.182089431076486e-02, 5.708554570355491e-03,  7.742620767358543e-03, -3.060633681332371e-02, -5.748553001371993e-02, 3.424581077159206e-03, -7.133639258830500e-02, 4.090381607933451e-01, 9.976578322468668e-02, 2.532512409149376e-01, -1.629672899787572e-01, 9.858562135838929e-02, 3.348921367606686e-02, 6.647859335893538e-02,  6.182188580711497e-02, 1.999022264946561e-01, -4.271817386487356e-02, 1.601402631154300e-02,  9.976578322468661e-02, 5.805783300311113e-01, -1.026126022577954e-01, -2.420404152663464e-01, -5.994329105541465e-02, 8.481773905621864e-02, 1.483086766125450e-01, -1.811702573219360e-01, -4.200544074497317e-02, 5.684971122817056e-02, -6.841444637889316e-02, 2.532512409149373e-01, -1.026126022577954e-01, 2.946558717845986e-01, -4.840475177491806e-02, 1.055594320308784e-01, -4.643870026819708e-03, 2.832543380413523e-03, 1.155877644456273e-01, 1.826438616899628e-01, -5.737809175020673e-02, 3.706565688552706e-02, -1.629672899787573e-01, -2.420404152663463e-01, -4.840475177491799e-02, 2.272087464076424e-01, -1.848987359713690e-02, -3.419061683697004e-02, -4.876247138868985e-02, 5.603101949242417e-02, -6.367772271268859e-02, -4.103449971327744e-02, -3.182089431076486e-02, 9.858562135838923e-02, -5.994329105541461e-02, 1.055594320308784e-01, -1.848987359713688e-02, 1.012798785503404e-01, -1.067754879839962e-02, -1.470595535534078e-02,  6.051641310700975e-02, 7.442001991950017e-02, -1.256921275410581e-02, 5.708554570355485e-03,  3.348921367606683e-02, 8.481773905621871e-02, -4.643870026819673e-03, -3.419061683696997e-02, -1.067754879839961e-02, 6.712242049431705e-02, 3.889336908892854e-02, -2.863187527667375e-02,  3.698282639045019e-03, -6.090805434649828e-03, 7.742620767358516e-03, 6.647859335893531e-02,  1.483086766125450e-01, 2.832543380413411e-03, -4.876247138868988e-02, -1.470595535534078e-02,  3.889336908892851e-02, 1.513055698462129e-01, -3.119554700633061e-02, 7.459246607563613e-04, -4.866710136174734e-02, -3.060633681332373e-02, 6.182188580711497e-02, -1.811702573219361e-01,  1.155877644456271e-01, 5.603101949242398e-02, 6.051641310700975e-02, -2.863187527667378e-02, -3.119554700633064e-02, 1.578596001866123e-01, 6.146090633613449e-02, -4.550902048980177e-02, -5.748553001371992e-02, 1.999022264946560e-01, -4.200544074497319e-02, 1.826438616899629e-01, -6.367772271268866e-02, 7.442001991950015e-02, 3.698282639045012e-03, 7.459246607562642e-04,  6.146090633613449e-02, 1.962523910529954e-01, -2.438896779319570e-02, 3.424581077159317e-03, -4.271817386487349e-02, 5.684971122817069e-02, -5.737809175020685e-02, -4.103449971327741e-02, -1.256921275410582e-02, -6.090805434649883e-03, -4.866710136174738e-02, -4.550902048980174e-02, -2.438896779319566e-02, 1.376474631340765e-01, 6.620258319671030e-02, -7.356831052518702e-02, -1.649902239988918e-02, -5.906918062091756e-02, 5.605959120400117e-02, -2.685555828528752e-02,  1.443912501303289e-03, 4.990987018159226e-03, -1.772356608187441e-02, -5.499919541008087e-02, -8.269486820014743e-03, -7.356831052518704e-02, 3.485774894189250e-01, 1.056138703626663e-01,  2.093534837146555e-01, -1.571092372593400e-01, 8.356390256135765e-02, 2.962214855994955e-02,  4.706899305060058e-02, 4.185344752182554e-02, 1.717108046437585e-01, -1.617601114444904e-02, -1.649902239988919e-02, 1.056138703626663e-01, 5.176619426817282e-01, -7.939336901887860e-02, -2.465865895928807e-01, -4.298072593051447e-02, 7.048281362890536e-02, 1.055905958189919e-01, -1.627672330797096e-01, -1.902729879649480e-02, 7.999259960608081e-02, -5.906918062091759e-02,  2.093534837146553e-01, -7.939336901887853e-02, 2.447588456832487e-01, -4.128146254333098e-02,  8.685707817309300e-02, -2.636247372008668e-03, 1.158491782067491e-03, 9.317354527154359e-02,  1.514212613642159e-01, -4.359251424831413e-02, 5.605959120400113e-02, -1.571092372593401e-01, -2.465865895928806e-01, -4.128146254333092e-02, 2.362380211751530e-01, -1.771865604702780e-02, -3.185274046906085e-02, -3.183574784231289e-02, 6.199561608535829e-02, -6.436266119417346e-02, -6.313434684265265e-02, -2.685555828528752e-02, 8.356390256135769e-02, -4.298072593051443e-02,  8.685707817309307e-02, -1.771865604702775e-02, 8.394740614590712e-02, -6.822000944039803e-03, -9.904077575764874e-03, 4.775800640990158e-02, 6.224985079955995e-02,
                  -9.156815361863340e-03,  1.443912501303293e-03, 2.962214855994953e-02, 7.048281362890539e-02, -2.636247372008627e-03, -3.185274046906080e-02, -6.822000944039803e-03, 5.651376051041268e-02, 2.842237792742504e-02, -2.416684069492224e-02, 4.945597574573218e-03, -9.838592428135576e-04, 4.990987018159282e-03,  4.706899305060056e-02, 1.055905958189919e-01, 1.158491782067431e-03, -3.183574784231294e-02, -9.904077575764874e-03, 2.842237792742503e-02, 1.183305036338221e-01, -2.387198267583122e-02, -2.083122160533510e-04, -3.843345917098605e-02, -1.772356608187439e-02, 4.185344752182557e-02, -1.627672330797096e-01, 9.317354527154351e-02, 6.199561608535822e-02, 4.775800640990160e-02, -2.416684069492227e-02, -2.387198267583125e-02, 1.354657990563894e-01, 4.735307450499930e-02, -4.519452833759373e-02, -5.499919541008087e-02, 1.717108046437586e-01, -1.902729879649481e-02,  1.514212613642159e-01, -6.436266119417350e-02, 6.224985079955990e-02, 4.945597574573204e-03, -2.083122160534065e-04, 4.735307450499929e-02, 1.662155586189531e-01, -1.178411785560223e-02, -8.269486820014771e-03, -1.617601114444897e-02, 7.999259960608079e-02, -4.359251424831430e-02, -6.313434684265262e-02, -9.156815361863316e-03, -9.838592428135853e-04, -3.843345917098605e-02, -4.519452833759370e-02, -1.178411785560225e-02, 1.296921101434202e-01, 2.171830471852956e-02, -1.471674316503784e-03, 2.064309796521692e-02, -1.402966665087271e-02, -6.557181938541153e-03, -5.855880579706373e-03, 5.203062506105778e-03, -4.869979855250679e-03, -1.569465085461526e-02, -7.394848324216267e-03, 1.686493075582655e-02, -1.471674316503784e-03, 9.313926114293263e-02,  1.507153173486031e-02, 3.554104153289330e-02, -2.119437081853560e-02, 9.550680527942022e-03,  6.211413187228764e-03, 1.560555500080688e-02, 1.067264147632672e-02, 2.489481640767202e-02, -1.291647855249673e-02, 2.064309796521691e-02, 1.507153173486033e-02, 1.156076215118068e-01, -1.544749819251926e-02, -2.680764658595431e-02, -6.908641823529162e-03, 1.638069090167619e-02,  2.832533798127865e-02, -2.323549846605227e-02, -7.087434977119506e-03, 1.945296623429692e-03, -1.402966665087272e-02, 3.554104153289329e-02, -1.544749819251925e-02, 7.152659988051950e-02, -6.158728627851861e-03, 1.256278068269506e-02, -3.056460056707950e-03, 8.716490447969278e-05,  1.757610773292058e-02, 2.502767177097435e-02, -1.124084467836774e-02, -6.557181938541179e-03, -2.119437081853567e-02, -2.680764658595428e-02, -6.158728627851847e-03, 5.750509860909883e-02, -3.029976584966646e-03, -3.047735875070946e-03, -7.598171992562465e-03, 4.587593487424636e-03, -6.229619184243950e-03, -1.519743963336745e-03, -5.855880579706376e-03, 9.550680527942050e-03, -6.908641823529143e-03, 1.256278068269507e-02, -3.029976584966601e-03, 4.290845982962001e-02, -3.843166464169885e-03, -4.240785003647874e-03, 1.046610572110044e-02, 6.616036506065378e-03,  1.998924905739588e-03, 5.203062506105787e-03, 6.211413187228752e-03, 1.638069090167620e-02, -3.056460056707924e-03, -3.047735875070939e-03, -3.843166464169888e-03, 4.075875122579697e-02,  1.026937281182977e-02, -8.076610278973396e-03, 7.211305112839724e-04, -3.145065206693348e-03, -4.869979855250672e-03, 1.560555500080687e-02, 2.832533798127863e-02, 8.716490447965635e-05, -7.598171992562512e-03, -4.240785003647864e-03, 1.026937281182975e-02, 6.448211882877865e-02, -6.059405528576338e-03, 2.386004268815075e-03, -1.771848733213728e-02, -1.569465085461526e-02,  1.067264147632678e-02, -2.323549846605225e-02, 1.757610773292057e-02, 4.587593487424532e-03,  1.046610572110041e-02, -8.076610278973414e-03, -6.059405528576379e-03, 5.801066873447286e-02,  6.215509335282631e-03, -5.767464486462217e-03, -7.394848324216282e-03, 2.489481640767194e-02, -7.087434977119477e-03, 2.502767177097444e-02, -6.229619184243957e-03, 6.616036506065405e-03,  7.211305112839689e-04, 2.386004268815042e-03, 6.215509335282646e-03, 5.492514005766665e-02, -5.909757308403398e-03, 1.686493075582653e-02, -1.291647855249665e-02, 1.945296623429633e-03, -1.124084467836781e-02, -1.519743963336690e-03, 1.998924905739587e-03, -3.145065206693388e-03, -1.771848733213723e-02, -5.767464486462220e-03, -5.909757308403358e-03, 6.751392220134381e-02,  2.267199631890625e-02, -4.028025651378201e-03, 2.084612793362112e-02, -1.281175972649431e-02, -4.090059442553665e-03, -7.163806612816515e-03, 5.257100332601468e-03, 1.594891704583207e-03, -1.307400686155973e-02, -9.799336139886585e-03, 9.846783770822022e-03, -4.028025651378218e-03,  1.367459475996803e-01, 2.571744294780940e-02, 7.061984440306969e-02, -3.796833748268273e-02,  2.385709167024918e-02, 1.293876540191718e-02, 2.969881503385724e-02, 1.907248106134219e-02,  5.356637218564674e-02, -2.475835612168412e-02, 2.084612793362111e-02, 2.571744294780936e-02,  2.019971627370479e-01, -3.719616813138388e-02, -6.637975970387275e-02, -2.214106062708696e-02,  2.897616644631326e-02, 5.016905607059752e-02, -5.907383875238198e-02, -2.172394453833108e-02,  1.241152772153731e-02, -1.281175972649432e-02, 7.061984440306962e-02, -3.719616813138386e-02,  1.044451129975394e-01, -4.741516265390626e-03, 2.903765910002839e-02, -1.139057058547028e-03,  6.183947290991528e-03, 3.670466495414235e-02, 5.268206517552042e-02, -2.681755297827403e-02, -4.090059442553700e-03, -3.796833748268272e-02, -6.637975970387272e-02, -4.741516265390662e-03,  7.662622419769397e-02, -7.340104208002304e-04, -1.018572397691502e-02, -1.686987775575008e-02,  1.903011150181297e-02, -8.345806636336488e-03, -6.327934459770175e-03, -7.163806612816512e-03,  2.385709167024918e-02, -2.214106062708694e-02, 2.903765910002837e-02, -7.340104208001987e-04,  4.708059028694469e-02, -3.742013941177697e-03, -4.787206746326725e-03, 1.950699524758510e-02,  2.002643312817511e-02, -3.712832741564018e-03, 5.257100332601477e-03, 1.293876540191716e-02,  2.897616644631327e-02, -1.139057058547003e-03, -1.018572397691501e-02, -3.742013941177691e-03,  3.982470404426871e-02, 1.510708611930808e-02, -1.077706965662745e-02, 1.598785000375527e-03, -3.798611038841958e-03, 1.594891704583197e-03, 2.969881503385725e-02, 5.016905607059755e-02,  6.183947290991483e-03, -1.686987775575009e-02, -4.787206746326707e-03, 1.510708611930807e-02,  7.553626747625065e-02, -8.724965760417043e-03, 4.106786216342326e-03, -2.549241468490602e-02, -1.307400686155972e-02, 1.907248106134217e-02, -5.907383875238204e-02, 3.670466495414226e-02,  1.903011150181284e-02, 1.950699524758504e-02, -1.077706965662745e-02, -8.724965760417067e-03,  7.112824426726025e-02, 2.035531386044614e-02, -1.644806460677811e-02, -9.799336139886577e-03,  5.356637218564666e-02, -2.172394453833107e-02, 5.268206517552047e-02, -8.345806636336496e-03,  2.002643312817510e-02, 1.598785000375520e-03, 4.106786216342291e-03, 2.035531386044613e-02,  7.395656678384738e-02, -1.425362477548906e-02, 9.846783770822001e-03, -2.475835612168403e-02,  1.241152772153734e-02, -2.681755297827410e-02, -6.327934459770067e-03, -3.712832741564038e-03, -3.798611038841991e-03, -2.549241468490598e-02, -1.644806460677808e-02, -1.425362477548906e-02,  7.624342058699407e-02, 2.476826270481240e-02, -3.264321371508873e-03, 3.715626825892823e-02, -1.735371090430141e-02, -9.890666841032699e-03, -7.977940040457312e-03, 6.659246378090296e-03,  5.507627932740971e-03, -1.801415915964642e-02,
                  -1.235164699388927e-02, 1.163809849176014e-02, -3.264321371508876e-03, 7.869367467064503e-02, 1.136955987850807e-02, 3.256173149295345e-02, -1.761948927243287e-02, 9.724843205149793e-03, 7.370827974465558e-03, 1.452532506356457e-02,  9.327533196969148e-03, 2.368718117052631e-02, -1.353886850397537e-02, 3.715626825892820e-02,  1.136955987850811e-02, 1.358598608989376e-01, -2.270840737642001e-02, -2.968275874873018e-02, -1.502834202389454e-02, 2.052044746833690e-02, 3.865224063185539e-02, -3.356304029136498e-02, -1.976219333491958e-02, -2.491055340739703e-04, -1.735371090430140e-02, 3.256173149295352e-02, -2.270840737641999e-02, 6.641615051291262e-02, -3.284080249408147e-03, 1.525926412903749e-02, -2.032418425488178e-03, -1.338276291556422e-03, 1.880380722729732e-02, 2.800163042721900e-02, -1.278744197492393e-02, -9.890666841032729e-03, -1.761948927243288e-02, -2.968275874873013e-02, -3.284080249408197e-03, 5.237472625246602e-02, -1.569626681093917e-05, -5.248825810513971e-03, -7.497299297770202e-03, 9.226985115672454e-03, -2.783956005308553e-03, -2.301168992064444e-03, -7.977940040457317e-03, 9.724843205149755e-03, -1.502834202389450e-02, 1.525926412903749e-02, -1.569626681093179e-05, 3.792638907359591e-02, -2.792520358675802e-03, -5.177760121344030e-03,  1.152594471597282e-02, 1.063413512367586e-02, -4.136056315461331e-04, 6.659246378090304e-03,  7.370827974465541e-03, 2.052044746833690e-02, -2.032418425488160e-03, -5.248825810513962e-03, -2.792520358675802e-03, 3.545576005455056e-02, 1.085616759593332e-02, -8.034920160216853e-03, -1.546658959179435e-04, -2.636619216900333e-03, 5.507627932740967e-03, 1.452532506356458e-02,  3.865224063185537e-02, -1.338276291556457e-03, -7.497299297770204e-03, -5.177760121344022e-03,  1.085616759593332e-02, 6.358767665877786e-02, -6.906244341277719e-03, -3.097289572415495e-03, -2.068439176392748e-02, -1.801415915964642e-02, 9.327533196969185e-03, -3.356304029136509e-02,  1.880380722729729e-02, 9.226985115672335e-03, 1.152594471597283e-02, -8.034920160216839e-03, -6.906244341277725e-03, 5.286602758164272e-02, 1.075044672392071e-02, -8.064369608893873e-03, -1.235164699388926e-02, 2.368718117052626e-02, -1.976219333491954e-02, 2.800163042721897e-02, -2.783956005308564e-03, 1.063413512367577e-02, -1.546658959179450e-04, -3.097289572415527e-03,  1.075044672392069e-02, 5.380779209216746e-02, -5.435250170852267e-03, 1.163809849176010e-02, -1.353886850397533e-02, -2.491055340739495e-04, -1.278744197492399e-02, -2.301168992064349e-03, -4.136056315461321e-04, -2.636619216900358e-03, -2.068439176392746e-02, -8.064369608893833e-03, -5.435250170852288e-03, 6.355709117825137e-02, 2.908423718885458e-02, -1.087080271609451e-03,  4.584310436518948e-02, -1.822907636059566e-02, -9.716166693816617e-03, -1.061155962574999e-02,  9.096129887276760e-03, 1.341407799399938e-02, -1.966408937428313e-02, -1.452965163964578e-02,  5.497082317007620e-03, -1.087080271609460e-03, 1.338154661092531e-01, 2.572323894127029e-02,  7.018881609132931e-02, -3.502798201466206e-02, 2.409224868354470e-02, 1.040911443780104e-02,  3.255863635833275e-02, 2.122501294592015e-02, 5.247183653145676e-02, -2.905680901204894e-02,  4.584310436518949e-02, 2.572323894127029e-02, 1.988682012373580e-01, -3.561504634347437e-02, -5.431724341344654e-02, -2.233083105254836e-02, 3.288662360284899e-02, 6.154559035161462e-02, -5.362026407978351e-02, -2.436055655450581e-02, -4.501540319475544e-03, -1.822907636059565e-02,  7.018881609132921e-02, -3.561504634347428e-02, 1.000491812895766e-01, -8.001164175012293e-03,  2.942949099369306e-02, -3.965126188613260e-03, 4.501108594364579e-03, 3.561081369473008e-02,  5.220183606855257e-02, -2.344198343439343e-02, -9.716166693816641e-03, -3.502798201466209e-02, -5.431724341344638e-02, -8.001164175012284e-03, 6.905798101680978e-02, -3.043754526734614e-03, -9.342470803604440e-03, -1.438814203594600e-02, 1.410006347154035e-02, -1.030343025462422e-02, -2.938598962454712e-03, -1.061155962574998e-02, 2.409224868354464e-02, -2.233083105254831e-02,  2.942949099369311e-02, -3.043754526734603e-03, 4.469279953173057e-02, -4.542708795325723e-03, -6.613709711498047e-03, 1.855197761837861e-02, 2.164341689799513e-02, -1.589806796590507e-03,  9.096129887276762e-03, 1.040911443780103e-02, 3.288662360284898e-02, -3.965126188613237e-03, -9.342470803604437e-03, -4.542708795325717e-03, 3.775013299861342e-02, 1.637516640561668e-02, -1.124911797254195e-02, -1.258435246448424e-03, -4.775807121574348e-03, 1.341407799399938e-02,  3.255863635833274e-02, 6.154559035161462e-02, 4.501108594364590e-03, -1.438814203594596e-02, -6.613709711498023e-03, 1.637516640561668e-02, 8.120452495610482e-02, -8.664751986782852e-03,  2.474048336070529e-07, -3.321273288196908e-02, -1.966408937428314e-02, 2.122501294592021e-02, -5.362026407978358e-02, 3.561081369473004e-02, 1.410006347154022e-02, 1.855197761837862e-02, -1.124911797254195e-02, -8.664751986782855e-03, 6.360830966367348e-02, 2.115305614524611e-02, -1.257848791972696e-02, -1.452965163964578e-02, 5.247183653145679e-02, -2.436055655450580e-02,  5.220183606855253e-02, -1.030343025462425e-02, 2.164341689799509e-02, -1.258435246448417e-03,  2.474048336056434e-07, 2.115305614524623e-02, 7.225330316404389e-02, -1.013816702531134e-02,  5.497082317007608e-03, -2.905680901204890e-02, -4.501540319475615e-03, -2.344198343439352e-02, -2.938598962454739e-03, -1.589806796590487e-03, -4.775807121574362e-03, -3.321273288196906e-02, -1.257848791972692e-02, -1.013816702531133e-02, 7.467855436707484e-02, 3.114728889143981e-02, -2.629194178445865e-03, 4.895266095185604e-02, -2.035899262625107e-02, -1.099000263561400e-02, -1.082475833179784e-02, 1.019601658447922e-02, 1.427193082379400e-02, -2.188469848344236e-02, -1.526411322516712e-02, 5.657112577095263e-03, -2.629194178445874e-03, 1.310143681451270e-01,  1.150157937901334e-02, 7.284044089111814e-02, -3.105973333054347e-02, 2.763163819456860e-02,  8.416798894518952e-03, 1.978541473389513e-02, 2.467322766525814e-02, 5.430647357261525e-02, -2.270351738478042e-02, 4.895266095185606e-02, 1.150157937901337e-02, 1.952766696634124e-01, -4.651728853951066e-02, -4.765384552870543e-02, -2.868466908740767e-02, 2.969707612120446e-02,  5.678828284799550e-02, -5.780272452945460e-02, -3.069674806269639e-02, 5.382468346121903e-04, -2.035899262625105e-02, 7.284044089111819e-02, -4.651728853951068e-02, 1.071158435011789e-01, -7.386065848081009e-03, 3.494484618219713e-02, -4.153403684402542e-03, -3.316872531863102e-03,  4.063632415055846e-02, 5.617895340999987e-02, -2.075132775684403e-02, -1.099000263561401e-02, -3.105973333054349e-02, -4.765384552870542e-02, -7.386065848080986e-03, 6.538751577979540e-02, -1.585029969548916e-03, -7.136289834818043e-03, -1.086410883043045e-02, 1.282299438685531e-02, -9.613981325514793e-03, -4.666463688795554e-03, -1.082475833179783e-02, 2.763163819456856e-02, -2.868466908740764e-02, 3.494484618219713e-02, -1.585029969548933e-03, 4.833961174466098e-02, -4.859932449739504e-03, -8.613256835949504e-03, 2.257921296531695e-02, 2.461979211796601e-02, -2.949443022615645e-03, 1.019601658447922e-02, 8.416798894518948e-03, 2.969707612120446e-02, -4.153403684402538e-03, -7.136289834818053e-03, -4.859932449739507e-03, 3.794513344574246e-02,  1.530668846824722e-02,
                  -1.057166202326009e-02, -1.388889809211859e-03, -5.335289881749965e-03,  1.427193082379400e-02, 1.978541473389512e-02, 5.678828284799554e-02, -3.316872531863114e-03, -1.086410883043044e-02, -8.613256835949500e-03, 1.530668846824723e-02, 7.536902116289912e-02, -1.162533604985135e-02, -4.713796682247686e-03, -2.841204312210785e-02, -2.188469848344236e-02,  2.467322766525818e-02, -5.780272452945469e-02, 4.063632415055850e-02, 1.282299438685527e-02,  2.257921296531696e-02, -1.057166202326010e-02, -1.162533604985135e-02, 6.621515832444562e-02,  2.435710755171928e-02, -1.219031093041058e-02, -1.526411322516712e-02, 5.430647357261522e-02, -3.069674806269638e-02, 5.617895340999981e-02, -9.613981325514812e-03, 2.461979211796594e-02, -1.388889809211859e-03, -4.713796682247690e-03, 2.435710755171936e-02, 7.479825894645153e-02, -8.016196525565665e-03, 5.657112577095299e-03, -2.270351738478037e-02, 5.382468346121695e-04, -2.075132775684405e-02, -4.666463688795473e-03, -2.949443022615648e-03, -5.335289881749976e-03, -2.841204312210782e-02, -1.219031093041053e-02, -8.016196525565684e-03, 7.202853420814653e-02,  1.770771211630441e-02, 1.831290732844219e-02, 1.746255143561659e-02, 7.376815780008224e-03, -1.102613705793878e-02, 3.118471247917355e-03, 4.026489097227999e-03, 2.954178813385481e-03, -3.029993050868967e-03, 5.218742476314610e-03, 4.464806746597504e-03, 1.831290732844218e-02,  8.822573706047593e-02, 9.889565207571369e-03, 3.498905860794184e-02, -1.057305651418887e-02,  1.198725393047724e-02, 8.771571848977165e-03, 1.825291632187300e-02, 1.137360571406687e-02,  2.294898320190811e-02, -1.673861600128904e-02, 1.746255143561659e-02, 9.889565207571345e-03,  1.037129303144714e-01, -1.469399899334257e-02, -2.965439216061266e-02, -4.869680175828032e-03,  8.233027600566545e-03, 1.242516314032473e-02, -2.335722582134586e-02, -6.994007277627817e-03,  1.323786892120508e-02, 7.376815780008230e-03, 3.498905860794183e-02, -1.469399899334262e-02,  6.438401029348650e-02, 2.415258101904980e-03, 1.336911743274609e-02, 3.735609682902811e-03,  1.037393235527360e-02, 1.668999984339908e-02, 2.121739748311221e-02, -1.906776118733260e-02, -1.102613705793878e-02, -1.057305651418887e-02, -2.965439216061265e-02, 2.415258101905005e-03,  5.671130389664315e-02, -4.407676300083737e-04, -2.383573117108180e-03, 3.956022329455852e-04,  1.298164124986806e-02, -1.929579814695395e-03, -1.213304904861674e-02, 3.118471247917353e-03,  1.198725393047718e-02, -4.869680175828029e-03, 1.336911743274612e-02, -4.407676300083800e-04,  3.842685164344772e-02, 5.879883363785300e-04, 9.083022314320380e-04, 8.274312383711171e-03,  8.076732272519576e-03, -2.439101496417666e-03, 4.026489097228001e-03, 8.771571848977172e-03,  8.233027600566522e-03, 3.735609682902800e-03, -2.383573117108203e-03, 5.879883363785271e-04,  3.446018408844344e-02, 5.649972765293450e-03, -1.876953454439950e-03, 4.199996885210524e-03, -3.522552213049501e-03, 2.954178813385474e-03, 1.825291632187299e-02, 1.242516314032472e-02,  1.037393235527359e-02, 3.956022329455874e-04, 9.083022314320323e-04, 5.649972765293452e-03,  5.852530069909069e-02, 5.875255485907533e-03, 4.162716692141721e-03, -2.537110508557541e-02, -3.029993050868970e-03, 1.137360571406685e-02, -2.335722582134582e-02, 1.668999984339916e-02,  1.298164124986806e-02, 8.274312383711189e-03, -1.876953454439956e-03, 5.875255485907526e-03,  4.955218005712608e-02, 5.322919494061301e-03, -1.631847812848056e-02, 5.218742476314599e-03,  2.294898320190803e-02, -6.994007277627777e-03, 2.121739748311229e-02, -1.929579814695356e-03,  8.076732272519508e-03, 4.199996885210527e-03, 4.162716692141737e-03, 5.322919494061363e-03,  4.824645736583033e-02, -4.667059351743241e-03, 4.464806746597510e-03, -1.673861600128900e-02,  1.323786892120508e-02, -1.906776118733260e-02, -1.213304904861674e-02, -2.439101496417665e-03, -3.522552213049510e-03, -2.537110508557537e-02, -1.631847812848058e-02, -4.667059351743240e-03,  7.302313806158031e-02, 2.106685749094099e-02, 1.204994544473955e-02, 2.532980966416545e-02,  3.911848231287133e-04, -1.234171173615314e-02, 4.628546601878015e-04, 5.001002543399014e-03,  2.549778301837826e-03, -9.059890831467111e-03, 1.729793450877977e-03, 8.519502005171143e-03,  1.204994544473957e-02, 1.140583114919021e-01, 1.999567898254831e-02, 4.922201624479207e-02, -2.331955239391577e-02, 1.614347700926091e-02, 9.828885784571325e-03, 2.583624755745699e-02,  1.503294170790730e-02, 3.546619385468353e-02, -1.867226005393430e-02, 2.532980966416544e-02,  1.999567898254825e-02, 1.435139076183736e-01, -1.904881220012290e-02, -4.015435670434046e-02, -1.068075053919845e-02, 1.690942654528458e-02, 3.023513566344335e-02, -3.421328478473287e-02, -1.104026707332660e-02, 7.041060385803862e-03, 3.911848231287084e-04, 4.922201624479208e-02, -1.904881220012297e-02, 8.066056899309158e-02, -4.450626816070331e-03, 1.869261340087044e-02,  1.703733739845338e-03, 1.115969669835241e-02, 2.340833684999542e-02, 3.268540040444241e-02, -1.907368104637948e-02, -1.234171173615316e-02, -2.331955239391573e-02, -4.015435670434053e-02, -4.450626816070349e-03, 6.684546602065532e-02, -1.636167487203754e-03, -4.734267706857035e-03, -6.567054686070095e-03, 1.259875435723186e-02, -6.126597674451943e-03, -8.348423961492510e-03,  4.628546601878053e-04, 1.614347700926088e-02, -1.068075053919841e-02, 1.869261340087047e-02, -1.636167487203744e-03, 4.503301299422566e-02, -1.397209392579546e-03, -1.288939013641037e-03,  1.260737895330299e-02, 1.183763409874459e-02, -2.069699579907254e-03, 5.001002543399014e-03,  9.828885784571313e-03, 1.690942654528458e-02, 1.703733739845337e-03, -4.734267706857034e-03, -1.397209392579547e-03, 3.980801536465021e-02, 9.554341327790445e-03, -4.910248621136081e-03,  3.455659013207991e-03, -4.483665278528203e-03, 2.549778301837826e-03, 2.583624755745699e-02,  3.023513566344334e-02, 1.115969669835241e-02, -6.567054686070122e-03, -1.288939013641037e-03,  9.554341327790462e-03, 7.222986118828575e-02, 2.158450684549465e-03, 5.866660821605460e-03, -2.843438686502389e-02, -9.059890831467117e-03, 1.503294170790728e-02, -3.421328478473291e-02,  2.340833684999542e-02, 1.259875435723184e-02, 1.260737895330298e-02, -4.910248621136097e-03,  2.158450684549504e-03, 5.922007546115453e-02, 1.013292033421719e-02, -1.477496360902690e-02,  1.729793450877970e-03, 3.546619385468357e-02, -1.104026707332661e-02, 3.268540040444237e-02, -6.126597674451925e-03, 1.183763409874457e-02, 3.455659013207994e-03, 5.866660821605460e-03,  1.013292033421724e-02, 6.107409000031086e-02, -6.749222867156754e-03, 8.519502005171160e-03, -1.867226005393432e-02, 7.041060385803917e-03, -1.907368104637947e-02, -8.348423961492504e-03, -2.069699579907251e-03, -4.483665278528203e-03, -2.843438686502387e-02, -1.477496360902684e-02, -6.749222867156734e-03, 7.569869888109049e-02, 2.299346219160090e-02, 3.847509551997805e-03,  2.510037762960216e-02, -7.997833465472613e-03, -1.220209712305485e-02, -2.340386890602055e-03,  3.846702876083402e-03, -3.511967503419066e-03, -1.325618235376554e-02, -3.044257026934646e-03,  1.596597875150126e-02, 3.847509551997805e-03, 1.056849519188822e-01, 1.539809056863618e-02,  4.049779627923087e-02, -2.006358448514170e-02, 1.260426646270894e-02, 6.016280381794463e-03,
                  1.975546112504373e-02, 1.477053222401129e-02, 2.799570480205035e-02, -1.820771685613701e-02,  2.510037762960216e-02, 1.539809056863618e-02, 1.596909890143651e-01, -3.166824994688033e-02, -4.406297846310112e-02, -1.153475871366816e-02, 1.832989311676964e-02, 3.073013891709081e-02, -3.893974066330769e-02, -1.828791094452018e-02, 9.884951287633543e-03, -7.997833465472613e-03,  4.049779627923087e-02, -3.166824994688033e-02, 8.598948517714333e-02, -7.216751117786239e-04,  1.834736071740499e-02, -3.150961205640490e-03, 3.690418308204342e-03, 2.702781502063162e-02,  3.213981185938886e-02, -1.907996565134858e-02, -1.220209712305485e-02, -2.006358448514170e-02, -4.406297846310112e-02, -7.216751117786239e-04, 7.399378613464276e-02, -2.840382411657704e-03, -3.942228955667768e-03, -5.487344309783263e-03, 1.372855518205424e-02, -4.257725890120872e-03, -8.990490623141500e-03, -2.340386890602055e-03, 1.260426646270894e-02, -1.153475871366816e-02,  1.834736071740499e-02, -2.840382411657704e-03, 5.083391796051433e-02, -3.655983034992778e-03, -5.035699270430829e-03, 1.302690363951874e-02, 1.104848179757223e-02, 2.592371273958727e-04,  3.846702876083402e-03, 6.016280381794463e-03, 1.832989311676964e-02, -3.150961205640490e-03, -3.942228955667768e-03, -3.655983034992778e-03, 4.530370899710441e-02, 9.206300361257694e-03, -6.818439517736739e-03, 8.236683179268078e-04, -3.911514057078817e-03, -3.511967503419066e-03,  1.975546112504373e-02, 3.073013891709081e-02, 3.690418308204342e-03, -5.487344309783263e-03, -5.035699270430829e-03, 9.206300361257694e-03, 7.708373362206018e-02, 1.223005204082060e-03, -8.629033565614669e-05, -2.848884110118283e-02, -1.325618235376554e-02, 1.477053222401129e-02, -3.893974066330769e-02, 2.702781502063162e-02, 1.372855518205424e-02, 1.302690363951874e-02, -6.818439517736739e-03, 1.223005204082060e-03, 6.815861635319259e-02, 9.467293696769344e-03, -1.620555696569845e-02, -3.044257026934646e-03, 2.799570480205035e-02, -1.828791094452018e-02,  3.213981185938886e-02, -4.257725890120872e-03, 1.104848179757223e-02, 8.236683179268078e-04, -8.629033565614669e-05, 9.467293696769344e-03, 6.678210610348938e-02, -5.698864554892610e-03,  1.596597875150126e-02, -1.820771685613701e-02, 9.884951287633543e-03, -1.907996565134858e-02, -8.990490623141500e-03, 2.592371273958727e-04, -3.911514057078817e-03, -2.848884110118283e-02, -1.620555696569845e-02, -5.698864554892610e-03, 8.281365440199308e-02 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$lag_one_cov)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$fixed_effects)),
               c(numeric(0) )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_iter)),
               c(12 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q)),
               c( 3.011373301565367e-01, -2.732071701593596e-01, -2.999598282633814e-01, -1.257743042545124e-01,  4.021147452339077e-01, -9.953381659749683e-02, -2.475904425529724e-05, 1.475138002187206e-01,  7.737088178904440e-02, -1.972208983228809e-01, -2.776272345349033e-01, -2.732071701593596e-01,  1.271366247975332e+00, 5.046756513475633e-02, 1.006444252423627e+00, -4.615110295719209e-01,  3.970130652745434e-01, 6.207499731553250e-02, 1.653290144387048e-01, 3.504793120544634e-01,  7.730229129024571e-01, -2.050307069258388e-01, -2.999598282633814e-01, 5.046756513475633e-02,  2.757085310969182e+00, -9.139210443530403e-01, -1.452623094138808e+00, -3.500827871870797e-01,  3.129976242808997e-01, 2.016466197879042e-01, -1.178284180900774e+00, -3.389235275148647e-01,  9.130985924811374e-01, -1.257743042545124e-01, 1.006444252423627e+00, -9.139210443530403e-01,  1.141535691299220e+00, 1.232762681382284e-01, 4.469665355787976e-01, -6.059665927970481e-02,  5.649288255396151e-02, 6.928097536305053e-01, 7.484534142585957e-01, -4.748868159102941e-01,  4.021147452339077e-01, -4.615110295719209e-01, -1.452623094138808e+00, 1.232762681382284e-01,  1.038645754481653e+00, 5.556233285716680e-03, -1.491386239025320e-01, -6.612844911565862e-03,  5.161664417082612e-01, -1.425550241928261e-01, -5.771811736230141e-01, -9.953381659749683e-02,  3.970130652745434e-01, -3.500827871870797e-01, 4.469665355787976e-01, 5.556233285716680e-03,  1.976408276039046e-01, -3.583095931053813e-02, -2.844535988965987e-02, 2.646109249325559e-01,  3.094400595534376e-01, -1.277965849356620e-01, -2.475904425529724e-05, 6.207499731553250e-02,  3.129976242808997e-01, -6.059665927970481e-02, -1.491386239025320e-01, -3.583095931053813e-02,  5.662693160964904e-02, 7.861780377908131e-02, -1.125597512690173e-01, -1.859594710550099e-02,  4.105887697094333e-02, 1.475138002187206e-01, 1.653290144387048e-01, 2.016466197879042e-01,  5.649288255396151e-02, -6.612844911565862e-03, -2.844535988965987e-02, 7.861780377908131e-02,  2.560619866332757e-01, -7.969919056407549e-03, 4.288844558552822e-03, -1.963591879872242e-01,  7.737088178904440e-02, 3.504793120544634e-01, -1.178284180900774e+00, 6.928097536305053e-01,  5.161664417082612e-01, 2.646109249325559e-01, -1.125597512690173e-01, -7.969919056407549e-03,  6.308807835023461e-01, 3.608640153013691e-01, -4.904859127408917e-01, -1.972208983228809e-01,  7.730229129024571e-01, -3.389235275148647e-01, 7.484534142585957e-01, -1.425550241928261e-01,  3.094400595534376e-01, -1.859594710550099e-02, 4.288844558552822e-03, 3.608640153013691e-01,  5.581193134797733e-01, -1.714539833392357e-01, -2.776272345349033e-01, -2.050307069258388e-01,  9.130985924811374e-01, -4.748868159102941e-01, -5.771811736230141e-01, -1.277965849356620e-01,  4.105887697094333e-02, -1.963591879872242e-01, -4.904859127408917e-01, -1.714539833392357e-01,  6.079486136375902e-01 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q_0)),
               c(10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 10, 0, 0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 10 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_risk)),
               c(2041, 1968, 1876, 1813, 1684, 1539, 1448, 1262, 1189, 1042 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$times)),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$risk_set)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$data)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$order)),
               c(1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$F_)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$method)),
               c("EKF" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$model)),
               c("exp_clip_time" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$est_Q_0)),
               c(FALSE )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$LR)),
               c(1 )
               , tolerance = 1e-04)

  # tmp <- file("tmp.txt")
  # sink(tmp)
  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
    data = sims$res,
    by = (by_ <- 1),
    Q_0 = diag(1, 11),
    Q = diag(1, 11),
    control = list(est_Q_0 = F, eps = 10^-2, n_max = 10^3,
                   save_data = F, save_risk_set = F, ridge_eps = 1e-2),
    max_T = 10,
    id = sims$res$id, order = 1,
    verbose = F,
    model = "exp_clip_time_w_jump"))
  # sink()
  # close(tmp)

  # matplot(sims$betas, type = "l", lty = 1)
  # matplot(result_exp$state_vecs, lty = 2, type = "l", add = T)
  # get_expect_equal(result_exp, file = "tmp.txt", eps = 1e-4)

  expect_equal(unname(c(result_exp$state_vecs)),
               c(-4.31402525412808924, -4.43734205669995774, -4.41769875882428398, -3.70452180544693643, -4.03308319013330774, -3.99657743856493131, -4.06007633866015816, -2.64851112503273312, -2.98702116941458762, -3.17466087292495613, -2.98805478184462814, -0.44961059430247086, -0.49947187831584461, -0.41150795051254169, -0.18480562115783666,  0.27634763023888398, -0.84521839906995333, -1.95103407759708625, -1.69967766872656156, -1.91162750139675119, -1.01164440157729918, -0.69950858123247306, -0.49334210247035176, -0.30722021965799629, -1.67359463022880228, -1.63548956944678481, -4.88932934531754260, -4.24612187904331062, -4.11986824385506978, -3.41888866683800519, -2.90359211771388770, -2.40192311863148689, -1.65650930926378770, 0.13225322579856499, 0.34186408449582284,  1.82023259919423408, 0.19432085606134092, 1.74655861952587421, 1.28844450895279472, 1.13103916515958369, -1.44585609889575739, -0.52779379538621352, 0.20835973360594201, 0.29542197837307138, -0.58121910822178369, -0.89920070982587075, 0.46742140161353607, 1.16165161120964555, 3.44245807251627323, 2.65975657124752152,  2.54933239179081239, 3.77242071391651912, 2.85506696826477402, 2.37862192081196744, 2.73823348362669128, -0.24284825333314708, 0.04518743273219306, 0.95421598396152185, -0.46891240542093904, -0.22563889199510317, -0.36442348749516995, -0.03441146098187847, -2.43293485875791582, -1.02395059028211288, -0.54048954668372928, -1.03735139406667543, -0.65934802888690369, -0.69245738362228593, -0.46846139119321634, -0.48556954402643054, -0.36275021017160575, -0.50211076763428608, -0.52599083411843550, -0.31851555600295584, -0.49109137242988038, -0.40099020839158139, 0.32860425211587851, 1.90412178313307279, 2.06706014696978269, 3.23362414243317708,  2.22191779535861444, 2.14719596579494798, 2.27121343238562678, 2.29646792490297313, 1.47259414543947420,  2.31703309673068514, 3.23637201312216405, 4.25019427061172728, 1.36240325429935116, 1.68260167561409446,  3.27492562354912220, 1.46485782970488265, 2.64592639743246538, 2.37663057219467166, 2.59206175969738695, -0.35394856678098741, 1.40873079108676191, 2.38592849430314891, 1.27059550397564913, 0.53596067015816062,  0.63334789576558159, 0.99162154004688108, 0.05747640709893130, 0.66738825168582827, 0.63986636072724001,  0.46748791838579756, -1.03737684709839528, -1.10249853245871088, -0.99437191888856946, 0.27601783313029449, -1.94856375804392590, -2.08287738331746919, -4.71635270019066688, -3.12643683828605656, -5.06067591473922018, -4.29744723545038276, -4.45139212505888793, -2.60537779669231728, -3.47354684390075352, -4.76205146866823004, -5.82279296878147878 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$state_vars)),
               c( 1.391007045008298e-01, 1.957412154209669e-02, 4.000138189550997e-02, -6.483803878915563e-02,  7.773221319514617e-02, -5.439157282596144e-02, 2.200989437037107e-02, 2.818813365231248e-03, -8.641927302339818e-02, -6.775402370791180e-02, 3.749857518430277e-02, 1.957412154209669e-02,  5.092824574934263e-01, 4.953992031138772e-04, 4.141728766751474e-02, 3.743587333452641e-02, -4.697294697348432e-02, 2.020812304995841e-02, -1.632808506310585e-03, -6.239462500404888e-03, -1.759894542596869e-02, -4.257815402455695e-02, 4.000138189550999e-02, 4.953992031138857e-04,  5.415798834588939e-01, -8.690058187082728e-02, -2.168561208965235e-01, 2.026074135726359e-02,  3.413674295277119e-02, 1.045131446054413e-01, -8.002882258170992e-02, -4.846973605668422e-03,  4.183611960755165e-02, -6.483803878915557e-02, 4.141728766751471e-02, -8.690058187082714e-02,  3.441305470311233e-01, -1.294302012505753e-02, 1.142544269253300e-01, 4.232522950479978e-03,  4.809527435728517e-02, 1.645778380044908e-01, 1.227939942716674e-01, -1.228987761695301e-01,  7.773221319514619e-02, 3.743587333452644e-02, -2.168561208965234e-01, -1.294302012505759e-02,  4.723362636155289e-01, -8.373781589794359e-02, 5.338668012557226e-02, -3.199035094492151e-02, -7.928003406669670e-02, -2.286669078073172e-02, -1.024309235558921e-01, -5.439157282596145e-02, -4.697294697348430e-02, 2.026074135726353e-02, 1.142544269253302e-01, -8.373781589794363e-02,  3.639755430228685e-01, -1.665617237615295e-02, 2.220382916902683e-02, 1.972936674602351e-01,  2.346413201990583e-02, -5.678211252935229e-02, 2.200989437037106e-02, 2.020812304995842e-02,  3.413674295277119e-02, 4.232522950480015e-03, 5.338668012557223e-02, -1.665617237615297e-02,  1.964794298071291e-01, 3.730914908667048e-02, -6.714824316516391e-02, 6.362716165511727e-02, -6.289794165583532e-02, 2.818813365231302e-03, -1.632808506310570e-03, 1.045131446054413e-01,  4.809527435728515e-02, -3.199035094492152e-02, 2.220382916902673e-02, 3.730914908667050e-02,  3.620165448940671e-01, 3.505260646367918e-02, 7.169519868480181e-02, -1.500361722698487e-01, -8.641927302339820e-02, -6.239462500404877e-03, -8.002882258170994e-02, 1.645778380044907e-01, -7.928003406669686e-02, 1.972936674602351e-01, -6.714824316516389e-02, 3.505260646367900e-02,  4.755201971463463e-01, -6.856650484623185e-02, -9.418250552444388e-02, -6.775402370791184e-02, -1.759894542596869e-02, -4.846973605668397e-03, 1.227939942716675e-01, -2.286669078073175e-02,  2.346413201990580e-02, 6.362716165511728e-02, 7.169519868480176e-02, -6.856650484623185e-02,  4.231114716581974e-01, -7.988681668676738e-02, 3.749857518430279e-02, -4.257815402455694e-02,  4.183611960755168e-02, -1.228987761695299e-01, -1.024309235558922e-01, -5.678211252935261e-02, -6.289794165583530e-02, -1.500361722698489e-01, -9.418250552444371e-02, -7.988681668676738e-02,  5.120551009296987e-01, 8.722074727003010e-02, 1.270238001364568e-02, 2.986665031565114e-02, -7.187585232877670e-02, 3.928080278733127e-02, -5.657205552003246e-02, 9.060040313699025e-03, -4.570221337755241e-02, -9.797952600569064e-02, -5.202102495365612e-02, 9.312501497402818e-02,  1.270238001364569e-02, 3.600452488178455e-01, 2.338231316925290e-03, 2.137340930694227e-02,  2.588103425848880e-02, -2.716666424643747e-02, 1.613508353133341e-02, -7.183241825778255e-03, -8.418863740265653e-03, -1.784295805770906e-02, -2.090143892448263e-02, 2.986665031565111e-02,  2.338231316925293e-03, 4.322624405509687e-01, -5.828500853825874e-02, -1.799584143287276e-01,  1.603473986100964e-02, 2.468528714091962e-02, 9.099929159179729e-02, -5.975520036621999e-02,  4.882092131311723e-03, 4.070638015424765e-02, -7.187585232877675e-02, 2.137340930694227e-02, -5.828500853825861e-02, 2.932273313879143e-01, -2.695620041304319e-02, 1.246682337455404e-01,  6.006470728605304e-03, 5.186251460954111e-02, 1.634102716208642e-01, 1.093893984249238e-01, -1.290533629231159e-01, 3.928080278733129e-02, 2.588103425848884e-02, -1.799584143287277e-01, -2.695620041304328e-02, 3.642414213501625e-01, -7.935165144906213e-02, 3.975003907116270e-02, -6.023043053797222e-02, -8.675141514748867e-02, -1.612731277002162e-02, -4.834459453282287e-02, -5.657205552003238e-02, -2.716666424643749e-02, 1.603473986100963e-02, 1.246682337455406e-01, -7.935165144906223e-02, 3.141505315832929e-01, -1.015206734659150e-02, 4.296012449687535e-02,  1.980500886940579e-01, 3.835270376212331e-02, -8.620704999171477e-02, 9.060040313699020e-03,  1.613508353133341e-02, 2.468528714091963e-02, 6.006470728605315e-03, 3.975003907116270e-02, -1.015206734659150e-02, 1.551442519200631e-01, 2.856652756357689e-02, -5.151618937811730e-02,  5.300637594495562e-02, -5.074799605877806e-02, -4.570221337755234e-02, -7.183241825778237e-03,  9.099929159179726e-02, 5.186251460954101e-02, -6.023043053797228e-02, 4.296012449687536e-02,  2.856652756357688e-02, 2.737719334480734e-01, 3.819880431052010e-02, 8.309775042154162e-02, -1.097181291417215e-01, -9.797952600569064e-02, -8.418863740265629e-03, -5.975520036622005e-02,  1.634102716208641e-01, -8.675141514748871e-02, 1.980500886940580e-01, -5.151618937811728e-02,  3.819880431051997e-02, 4.268058243501322e-01, -2.951504277994809e-02, -1.065237914003187e-01, -5.202102495365611e-02, -1.784295805770906e-02, 4.882092131311723e-03, 1.093893984249239e-01, -1.612731277002165e-02, 3.835270376212330e-02, 5.300637594495562e-02, 8.309775042154156e-02, -2.951504277994805e-02, 3.276036504964595e-01, -1.038468119158547e-01, 9.312501497402816e-02, -2.090143892448264e-02, 4.070638015424766e-02, -1.290533629231159e-01, -4.834459453282289e-02, -8.620704999171491e-02, -5.074799605877806e-02, -1.097181291417215e-01, -1.065237914003185e-01, -1.038468119158547e-01, 4.254920637752174e-01, 1.011611965303901e-01, 1.692183437811833e-02,  1.995918831593646e-02, -1.021923746734567e-01, 7.119695608987521e-02, -9.151251444007547e-02,  1.257768969158938e-02, -5.349205990004088e-02, -1.358608523649207e-01, -6.756982304517610e-02,  9.924425257803622e-02, 1.692183437811833e-02, 3.391917922444547e-01, -4.257788573956589e-03,  1.914117988123058e-02, 4.559059601396213e-02, -4.343789008174710e-02, 2.354707972948761e-02, -3.064865031821188e-03, -2.513115592100517e-02, -1.034591873419920e-02, -3.079843764448453e-02,  1.995918831593644e-02, -4.257788573956575e-03, 4.761781759662422e-01, -8.347129257649964e-02, -2.579304780102021e-01, 2.294153681169682e-02, 1.782090329982408e-02, 9.143990466159729e-02, -7.077858053099162e-02, 6.793588317998388e-05, 8.794524564965533e-02, -1.021923746734567e-01,  1.914117988123058e-02, -8.347129257649957e-02, 3.193579562387784e-01, -3.021262147337522e-02,  1.674777863407356e-01, 3.877554038169416e-03, 7.204922143886040e-02, 2.245947824593857e-01,  1.330028103204481e-01, -1.790583117388297e-01, 7.119695608987520e-02, 4.559059601396213e-02, -2.579304780102021e-01, -3.021262147337527e-02, 4.490141600213608e-01, -1.294523559017556e-01,  5.157981565356388e-02, -6.035642390309418e-02, -1.209031471465787e-01, -1.985148451964156e-02, -9.228943996210294e-02, -9.151251444007540e-02, -4.343789008174711e-02, 2.294153681169679e-02,  1.674777863407356e-01, -1.294523559017556e-01, 3.366346486095669e-01, -2.285545659576797e-02,  6.423333168128267e-02, 2.684588535323629e-01, 5.281589214215586e-02,
                  -1.072129838651435e-01,  1.257768969158937e-02, 2.354707972948761e-02, 1.782090329982407e-02, 3.877554038169428e-03,  5.157981565356386e-02, -2.285545659576797e-02, 1.346137187078361e-01, 3.418748544119875e-02, -6.228415585230886e-02, 5.709432002699541e-02, -5.848940611897584e-02, -5.349205990004091e-02, -3.064865031821176e-03, 9.143990466159718e-02, 7.204922143886031e-02, -6.035642390309422e-02,  6.423333168128267e-02, 3.418748544119876e-02, 2.623026058688850e-01, 5.847573358145940e-02,  9.711996964823158e-02, -1.430053813072437e-01, -1.358608523649208e-01, -2.513115592100517e-02, -7.077858053099165e-02, 2.245947824593856e-01, -1.209031471465788e-01, 2.684588535323629e-01, -6.228415585230882e-02, 5.847573358145926e-02, 4.964083401253773e-01, -5.591185441986529e-03, -1.481434214357466e-01, -6.756982304517610e-02, -1.034591873419922e-02, 6.793588317996480e-05,  1.330028103204482e-01, -1.985148451964159e-02, 5.281589214215576e-02, 5.709432002699544e-02,  9.711996964823150e-02, -5.591185441986508e-03, 3.203572107955949e-01, -1.330636357776206e-01,  9.924425257803626e-02, -3.079843764448449e-02, 8.794524564965531e-02, -1.790583117388299e-01, -9.228943996210293e-02, -1.072129838651435e-01, -5.848940611897583e-02, -1.430053813072439e-01, -1.481434214357465e-01, -1.330636357776205e-01, 4.682990010866837e-01, 6.136570218628715e-02,  5.127313788920802e-04, 2.083961030659010e-02, -4.632451369038707e-02, 1.755682102063956e-02, -2.909741525231246e-02, 7.840219059365643e-03, -3.078012420166724e-02, -6.388877013915498e-02, -2.797959315646978e-02, 6.556736196150509e-02, 5.127313788920755e-04, 1.740678056784189e-01,  7.977778603331986e-03, 1.735745994574937e-02, -4.074543311483259e-03, -8.422694590370546e-03,  7.276969639969540e-03, 4.124167618290115e-03, 2.946442516102072e-03, 3.087158522094422e-03, -1.169871344896493e-02, 2.083961030659008e-02, 7.977778603331986e-03, 1.738410369210779e-01, -2.413527086715012e-02, -4.410352043293819e-02, 8.724687461577060e-03, 1.636305237157967e-02,  3.373733625163736e-02, -1.874338423661244e-02, -2.866349067043108e-04, 1.346422159085020e-02, -4.632451369038702e-02, 1.735745994574940e-02, -2.413527086715010e-02, 1.328175073354571e-01, -1.643657926887727e-02, 2.923711260435468e-02, -7.875206712656138e-03, 2.443011019875870e-03,  5.602952288740004e-02, 3.769903013989423e-02, -3.278953339884727e-02, 1.755682102063953e-02, -4.074543311483259e-03, -4.410352043293821e-02, -1.643657926887727e-02, 1.443478082403696e-01, -3.101907327465004e-02, 1.667247914004701e-02, -2.367664377958837e-02, -4.616319613029304e-02, -6.359911028916452e-03, -7.419936817104279e-03, -2.909741525231248e-02, -8.422694590370533e-03,  8.724687461577056e-03, 2.923711260435463e-02, -3.101907327465004e-02, 1.322200166083892e-01, -1.320393780169949e-02, -1.985246170900485e-03, 6.859148042067137e-02, 5.769937203031943e-04, -2.209849317418518e-03, 7.840219059365640e-03, 7.276969639969539e-03, 1.636305237157966e-02, -7.875206712656103e-03, 1.667247914004700e-02, -1.320393780169953e-02, 8.764994771779092e-02,  1.233001066671332e-02, -3.614322396315887e-02, 1.961456341023637e-02, -1.370422180277250e-02, -3.078012420166721e-02, 4.124167618290108e-03, 3.373733625163734e-02, 2.443011019875926e-03, -2.367664377958843e-02, -1.985246170900551e-03, 1.233001066671332e-02, 1.408094340498167e-01, -6.888542369908736e-04, 3.322058953913716e-02, -4.081865618387505e-02, -6.388877013915503e-02,  2.946442516102061e-03, -1.874338423661247e-02, 5.602952288740007e-02, -4.616319613029306e-02,  6.859148042067133e-02, -3.614322396315884e-02, -6.888542369907955e-04, 2.062547310057165e-01, -1.904652468056403e-02, -2.465171435084215e-02, -2.797959315646979e-02, 3.087158522094417e-03, -2.866349067043215e-04, 3.769903013989417e-02, -6.359911028916469e-03, 5.769937203031926e-04,  1.961456341023638e-02, 3.322058953913715e-02, -1.904652468056404e-02, 1.413578534566248e-01, -3.884717986712945e-02, 6.556736196150512e-02, -1.169871344896490e-02, 1.346422159085021e-02, -3.278953339884737e-02, -7.419936817104237e-03, -2.209849317418463e-03, -1.370422180277246e-02, -4.081865618387510e-02, -2.465171435084205e-02, -3.884717986712943e-02, 2.060838999532166e-01,  3.976578138679812e-02, 3.599703550773348e-03, 2.815719863037637e-02, -3.075619993001066e-02, -2.475121161099028e-03, -1.878160179541827e-02, 1.558980020740580e-03, -2.430138164700923e-02, -3.977325466130725e-02, -2.393489966892842e-02, 5.616553016496699e-02, 3.599703550773341e-03,  1.812634741944260e-01, 1.225228329896505e-02, 5.234740496362278e-03, -8.041422573631220e-04, -2.007555741503437e-02, 6.887412595191851e-03, -3.685911429433569e-03, -6.938308202706717e-03, -9.248687855085924e-03, -3.574732943360418e-03, 2.815719863037636e-02, 1.225228329896504e-02,  2.081090363620813e-01, -3.417220646413331e-02, -7.054279974761050e-02, -1.826010071381683e-03,  1.530213956508462e-02, 3.850067803547671e-02, -3.824753316058924e-02, -9.856165314065281e-03,  2.046536390670111e-02, -3.075619993001066e-02, 5.234740496362284e-03, -3.417220646413332e-02,  1.322199840928497e-01, -9.623198093666949e-04, 4.516298057902465e-02, 4.184124886914366e-03,  2.659391636005348e-02, 6.161160926858007e-02, 5.479020542344090e-02, -6.562556550872126e-02, -2.475121161099068e-03, -8.041422573631220e-04, -7.054279974761057e-02, -9.623198093667244e-04,  1.425814557129584e-01, -2.246207005135024e-02, 1.010801156862586e-02, -3.047958836518579e-02, -2.324836968955840e-02, 3.516601207416283e-03, -9.337721367467713e-03, -1.878160179541829e-02, -2.007555741503436e-02, -1.826010071381686e-03, 4.516298057902457e-02, -2.246207005135023e-02,  1.321140758545527e-01, -4.372037072947289e-03, 1.735745339360072e-02, 7.462195616652892e-02,  1.356396268421666e-02, -3.281696366493760e-02, 1.558980020740580e-03, 6.887412595191854e-03,  1.530213956508463e-02, 4.184124886914395e-03, 1.010801156862585e-02, -4.372037072947274e-03,  7.670864767901017e-02, 1.557064294046328e-02, -2.641684282686338e-02, 2.920088678668547e-02, -2.077371729975946e-02, -2.430138164700921e-02, -3.685911429433567e-03, 3.850067803547669e-02,  2.659391636005348e-02, -3.047958836518586e-02, 1.735745339360079e-02, 1.557064294046327e-02,  1.422087090877970e-01, 2.083198638757161e-02, 4.186925677441881e-02, -6.569989645079224e-02, -3.977325466130727e-02, -6.938308202706715e-03, -3.824753316058922e-02, 6.161160926858007e-02, -2.324836968955842e-02, 7.462195616652890e-02, -2.641684282686335e-02, 2.083198638757165e-02,  1.890385274322822e-01, -1.446113556249585e-02, -5.259761235465873e-02, -2.393489966892843e-02, -9.248687855085921e-03, -9.856165314065284e-03, 5.479020542344086e-02, 3.516601207416275e-03,  1.356396268421670e-02, 2.920088678668547e-02, 4.186925677441881e-02, -1.446113556249590e-02,  1.615851114051853e-01, -5.942836250156990e-02, 5.616553016496695e-02, -3.574732943360390e-03,  2.046536390670112e-02, -6.562556550872126e-02, -9.337721367467628e-03, -3.281696366493751e-02, -2.077371729975944e-02, -6.569989645079222e-02, -5.259761235465865e-02, -5.942836250156985e-02,  2.220200810457604e-01, 3.982969019412343e-02, -2.407614402318421e-03, 4.867949818860572e-02, -2.813396703780903e-02, -1.971178748627085e-02, -1.025784373831229e-02, 2.259740414580839e-03, -9.099699308294993e-03, -3.063962303416253e-02,
                  -2.032270493869707e-02, 4.927231375283636e-02, -2.407614402318417e-03, 1.146189541330974e-01, -4.811228533325954e-03, 5.272433861376308e-03, -4.309722098548934e-04, -1.123869272080879e-02, 7.405189112620449e-03, 3.054886702520035e-03,  1.590134296381457e-03, -2.302366735169445e-03, -9.075281887753811e-03, 4.867949818860569e-02, -4.811228533325943e-03, 1.684728318210923e-01, -2.397195361701879e-02, -4.234190402576785e-02,  6.151925876662505e-04, 1.273336658026821e-02, 2.863000215663405e-02, -2.761578237346231e-02, -2.003337111210794e-02, 1.891775697281040e-02, -2.813396703780901e-02, 5.272433861376327e-03, -2.397195361701879e-02, 9.137152960384562e-02, 2.865990158171387e-03, 2.497738845368321e-02,  2.365561775982783e-04, 9.639391930705609e-03, 3.370761902973463e-02, 3.373956728314772e-02, -4.066399113031888e-02, -1.971178748627086e-02, -4.309722098548795e-04, -4.234190402576788e-02,  2.865990158171347e-03, 1.002081828175957e-01, -6.558132446220772e-03, 6.783785014320466e-03, -9.040367850931624e-03, -4.070017146608622e-03, 1.186091414968031e-02, -1.407217337318711e-02, -1.025784373831232e-02, -1.123869272080880e-02, 6.151925876662248e-04, 2.497738845368319e-02, -6.558132446220770e-03, 8.700486391535089e-02, -1.946569384268954e-03, 6.510465043309013e-03,  3.703004483901823e-02, 6.029040102221658e-03, -1.940127905740537e-02, 2.259740414580841e-03,  7.405189112620452e-03, 1.273336658026821e-02, 2.365561775982588e-04, 6.783785014320465e-03, -1.946569384268942e-03, 5.993051167862824e-02, 9.304154560581836e-03, -1.772286932348023e-02,  1.624759226611980e-02, -9.064230176644227e-03, -9.099699308294984e-03, 3.054886702520026e-03,  2.863000215663404e-02, 9.639391930705624e-03, -9.040367850931642e-03, 6.510465043308986e-03,  9.304154560581838e-03, 1.009342199450176e-01, 1.292580112499068e-02, 1.174569439974613e-02, -4.331697501002467e-02, -3.063962303416252e-02, 1.590134296381452e-03, -2.761578237346226e-02,  3.370761902973463e-02, -4.070017146608588e-03, 3.703004483901821e-02, -1.772286932348022e-02,  1.292580112499067e-02, 1.217912977137832e-01, -1.515013929801334e-02, -3.879697173733733e-02, -2.032270493869707e-02, -2.302366735169445e-03, -2.003337111210792e-02, 3.373956728314768e-02,  1.186091414968031e-02, 6.029040102221671e-03, 1.624759226611980e-02, 1.174569439974614e-02, -1.515013929801334e-02, 1.190324066811159e-01, -3.074965996531451e-02, 4.927231375283630e-02, -9.075281887753788e-03, 1.891775697281039e-02, -4.066399113031898e-02, -1.407217337318704e-02, -1.940127905740526e-02, -9.064230176644220e-03, -4.331697501002464e-02, -3.879697173733738e-02, -3.074965996531454e-02, 1.697515508233993e-01, 4.373444368755321e-02, 8.751290425626179e-03,  5.781148727164878e-02, -3.417708035252151e-02, -1.463955745739399e-02, -1.616458220981579e-02,  5.362124842334889e-03, -9.734573604317589e-03, -3.917343801953045e-02, -2.533658915932913e-02,  4.618456555255122e-02, 8.751290425626182e-03, 1.637898528103289e-01, 1.389684086186655e-02,  1.130085689545605e-02, 2.464126584178886e-03, -1.563088618743865e-02, 4.964987251249424e-03,  1.159852065003590e-02, 2.309024038003130e-03, -7.814747525365498e-03, -2.125817210820391e-02,  5.781148727164876e-02, 1.389684086186655e-02, 2.260964545115390e-01, -4.115903555901837e-02, -6.654853738730059e-02, 2.944873972963055e-03, 2.313920357455017e-02, 4.456107229925808e-02, -4.559968406178109e-02, -1.287465188119217e-02, 1.217214732121430e-02, -3.417708035252150e-02,  1.130085689545605e-02, -4.115903555901839e-02, 1.135595549712679e-01, 7.970321219344899e-05,  4.006461212118323e-02, -3.881978672832230e-03, 2.046887191300241e-02, 5.896602210465912e-02,  4.521099819829794e-02, -5.434726894260793e-02, -1.463955745739399e-02, 2.464126584178888e-03, -6.654853738730064e-02, 7.970321219339955e-05, 1.284544194330498e-01, -2.282096506358031e-02,  6.208477332805851e-03, -1.677721344412779e-02, -1.341487806786627e-02, 1.433668489880009e-03, -1.723878467811100e-02, -1.616458220981579e-02, -1.563088618743865e-02, 2.944873972963078e-03,  4.006461212118324e-02, -2.282096506358029e-02, 1.127064528785566e-01, -6.251097327438684e-03,  1.397809130005242e-02, 6.553926500645535e-02, 1.264608109674380e-02, -2.548938058012092e-02,  5.362124842334893e-03, 4.964987251249425e-03, 2.313920357455016e-02, -3.881978672832253e-03,  6.208477332805835e-03, -6.251097327438671e-03, 6.249552537892136e-02, 1.062753263350243e-02, -2.646800324167619e-02, 1.815116360047443e-02, -1.056006925774737e-02, -9.734573604317566e-03,  1.159852065003591e-02, 4.456107229925810e-02, 2.046887191300237e-02, -1.677721344412782e-02,  1.397809130005246e-02, 1.062753263350243e-02, 1.219882426731433e-01, 2.327537465548805e-02,  2.316870977870565e-02, -6.707521927316408e-02, -3.917343801953045e-02, 2.309024038003128e-03, -4.559968406178110e-02, 5.896602210465907e-02, -1.341487806786630e-02, 6.553926500645543e-02, -2.646800324167620e-02, 2.327537465548807e-02, 1.671281570718213e-01, -1.765627167111924e-02, -5.070103170819557e-02, -2.533658915932915e-02, -7.814747525365501e-03, -1.287465188119217e-02,  4.521099819829794e-02, 1.433668489880011e-03, 1.264608109674379e-02, 1.815116360047443e-02,  2.316870977870567e-02, -1.765627167111924e-02, 1.440992399829698e-01, -3.772461063407620e-02,  4.618456555255123e-02, -2.125817210820392e-02, 1.217214732121436e-02, -5.434726894260795e-02, -1.723878467811098e-02, -2.548938058012093e-02, -1.056006925774737e-02, -6.707521927316408e-02, -5.070103170819550e-02, -3.772461063407614e-02, 1.963861502553106e-01, 4.856510164475948e-02,  1.850200479162224e-02, 5.820973627084561e-02, -4.166389662880518e-02, -1.005460443792049e-02, -2.686038153766302e-02, 5.841092004678410e-03, -1.200246802814706e-02, -5.164021112155932e-02, -2.678451410637516e-02, 5.244169481379335e-02, 1.850200479162224e-02, 1.737235871665321e-01,  5.498388738934488e-03, -2.436399804896048e-04, 1.886038879031674e-02, -2.838580710830312e-02,  6.339135595780171e-03, -1.866209321374992e-02, -2.047093832164832e-02, -1.316919269549021e-02,  3.793565114627378e-03, 5.820973627084559e-02, 5.498388738934479e-03, 2.090942404128263e-01, -4.778420477420366e-02, -5.704245389012997e-02, -7.224522635455827e-03, 1.754919536913267e-02,  3.530932542808575e-02, -5.315643882830316e-02, -1.320099014746974e-02, 2.238796690895760e-02, -4.166389662880520e-02, -2.436399804896083e-04, -4.778420477420369e-02, 1.268052743439960e-01, -8.646712985962009e-03, 6.150303718840151e-02, -1.279232264801174e-03, 1.935900252716466e-02,  8.102887322362061e-02, 4.880418088434824e-02, -6.317319907793041e-02, -1.005460443792047e-02,  1.886038879031674e-02, -5.704245389012996e-02, -8.646712985962045e-03, 1.255300689115489e-01, -2.596582549539397e-02, 1.143586708517418e-02, -1.828825048467640e-02, -2.421876432295277e-02, -3.305354684300054e-03, -1.242071297754487e-02, -2.686038153766298e-02, -2.838580710830312e-02, -7.224522635455736e-03, 6.150303718840149e-02, -2.596582549539400e-02, 1.362572198547996e-01, -7.080887503228207e-03, 1.779231691749533e-02, 9.338467186877876e-02, 1.951388626227130e-02, -4.707157704924352e-02, 5.841092004678406e-03, 6.339135595780173e-03, 1.754919536913266e-02, -1.279232264801208e-03, 1.143586708517417e-02, -7.080887503228181e-03, 6.398227755373323e-02,  1.248220961966576e-02,
                  -2.584835741887626e-02, 2.090638646788600e-02, -1.761491131391367e-02, -1.200246802814704e-02, -1.866209321374992e-02, 3.530932542808576e-02, 1.935900252716466e-02, -1.828825048467642e-02, 1.779231691749540e-02, 1.248220961966575e-02, 1.149101839855844e-01,  1.898591651706144e-02, 3.040648559645387e-02, -6.287201103891554e-02, -5.164021112155936e-02, -2.047093832164833e-02, -5.315643882830316e-02, 8.102887322362060e-02, -2.421876432295276e-02,  9.338467186877869e-02, -2.584835741887625e-02, 1.898591651706144e-02, 1.889498181659976e-01, -4.435797158262063e-03, -6.342075077226993e-02, -2.678451410637517e-02, -1.316919269549021e-02, -1.320099014746975e-02, 4.880418088434824e-02, -3.305354684300041e-03, 1.951388626227126e-02,  2.090638646788600e-02, 3.040648559645390e-02, -4.435797158262022e-03, 1.386916850406155e-01, -4.408639293006339e-02, 5.244169481379338e-02, 3.793565114627383e-03, 2.238796690895767e-02, -6.317319907793049e-02, -1.242071297754495e-02, -4.707157704924351e-02, -1.761491131391366e-02, -6.287201103891554e-02, -6.342075077226982e-02, -4.408639293006335e-02, 1.975591810690250e-01,  1.701314040747431e-02, 9.954287689831431e-03, 2.315826172564472e-02, 3.472038001197251e-03, -1.739460152179892e-02, 1.061639479737099e-02, 8.693086811012497e-04, -4.232696589852464e-03,  7.152671670324681e-04, 6.662847707127658e-05, 1.423941323444664e-02, 9.954287689831431e-03,  1.035536821671401e-01, 5.880064959143976e-03, 7.570357159906191e-03, 6.810078579586681e-03, -6.163472039144142e-03, 5.649246818565427e-03, -1.034214525068987e-02, -3.368043261699574e-03, -1.604205657394363e-03, 9.057978551343764e-04, 2.315826172564473e-02, 5.880064959143972e-03,  1.071652562159943e-01, -9.005013969167020e-03, -3.505527224762620e-02, 1.405673742337716e-02,  1.084985321228388e-03, 2.878436679263352e-04, -9.709396758089187e-03, -1.133258559682778e-03,  2.328845847012126e-02, 3.472038001197272e-03, 7.570357159906194e-03, -9.005013969167003e-03,  6.439501382933567e-02, 5.104842930470041e-03, 1.344953417683852e-02, 5.591532661064800e-03,  8.720176192574317e-03, 1.510903977477251e-02, 1.065772237226059e-02, -1.751333435463773e-02, -1.739460152179893e-02, 6.810078579586676e-03, -3.505527224762618e-02, 5.104842930470069e-03,  8.956323659344580e-02, -7.759723143199455e-03, 6.611275765816908e-03, 3.999039128500643e-04,  6.097039885638450e-03, -4.247117955566192e-04, -2.691957647286648e-02, 1.061639479737098e-02, -6.163472039144134e-03, 1.405673742337715e-02, 1.344953417683852e-02, -7.759723143199438e-03,  7.508318701289483e-02, 2.621331682834824e-03, 5.089540336364691e-03, 2.087488107729520e-02,  8.928971899977009e-05, -2.637609266845158e-03, 8.693086811012477e-04, 5.649246818565426e-03,  1.084985321228394e-03, 5.591532661064807e-03, 6.611275765816908e-03, 2.621331682834820e-03,  4.854031668900008e-02, 5.297345596638656e-04, -2.965391285142632e-03, 7.938586484301239e-03, -9.001284361350007e-03, -4.232696589852466e-03, -1.034214525068987e-02, 2.878436679263439e-04,  8.720176192574352e-03, 3.999039128500478e-04, 5.089540336364705e-03, 5.297345596638613e-04,  7.718448696858327e-02, 1.394324269300695e-02, 8.437973122975779e-03, -3.208394680140478e-02,  7.152671670324746e-04, -3.368043261699579e-03, -9.709396758089139e-03, 1.510903977477247e-02,  6.097039885638476e-03, 2.087488107729513e-02, -2.965391285142609e-03, 1.394324269300697e-02,  7.639159164882441e-02, -1.467363885065054e-02, -2.120458935928655e-02, 6.662847707128178e-05, -1.604205657394363e-03, -1.133258559682756e-03, 1.065772237226058e-02, -4.247117955566140e-04,  8.928971899975317e-05, 7.938586484301246e-03, 8.437973122975782e-03, -1.467363885065054e-02,  7.352595248391749e-02, -1.441869556145423e-03, 1.423941323444662e-02, 9.057978551343781e-04,  2.328845847012124e-02, -1.751333435463780e-02, -2.691957647286648e-02, -2.637609266845144e-03, -9.001284361350009e-03, -3.208394680140475e-02, -2.120458935928657e-02, -1.441869556145425e-03,  1.120260958768600e-01, 2.386217756275334e-02, 1.305911096765746e-02, 2.642455275488278e-02, -4.747815125032838e-03, -1.511119692062633e-02, 4.592866714222383e-04, 2.304511589138096e-03, -1.278881583995413e-02, -1.702783019309990e-02, -2.762066086929968e-04, 2.874202315628934e-02,  1.305911096765745e-02, 1.344174830352540e-01, 1.040804284741141e-02, 2.834731232461768e-03,  1.736974245326187e-03, -1.016735888038684e-02, 2.851092134172039e-03, -8.588241569662758e-03, -9.055007902283640e-03, -1.829424162549602e-03, 8.469390896848658e-03, 2.642455275488278e-02,  1.040804284741141e-02, 1.382093211268902e-01, -1.174341727890479e-02, -3.829031851313872e-02,  1.021843536617096e-02, 8.156801194486961e-03, 1.347877130325748e-02, -1.723444085005846e-02, -2.103383209907141e-03, 1.470743522772383e-02, -4.747815125032850e-03, 2.834731232461743e-03, -1.174341727890481e-02, 8.224057281614362e-02, -1.328212250341578e-03, 1.819795672575887e-02,  3.571807260622367e-03, 1.650989682744213e-02, 2.662286690399475e-02, 1.985222354414193e-02, -2.287156843441957e-02, -1.511119692062633e-02, 1.736974245326183e-03, -3.829031851313874e-02, -1.328212250341575e-03, 1.080212544324932e-01, -1.116153532338621e-02, 8.762762818419288e-03, -6.509375480429548e-03, -8.018022570451299e-03, 6.428316502925748e-03, -2.403209153959746e-02,  4.592866714222266e-04, -1.016735888038683e-02, 1.021843536617094e-02, 1.819795672575886e-02, -1.116153532338620e-02, 9.632186790551617e-02, 6.737519711673386e-04, 7.768277063407667e-03,  3.752724784814666e-02, -2.209866685621569e-03, -1.265124151776705e-02, 2.304511589138097e-03,  2.851092134172037e-03, 8.156801194486956e-03, 3.571807260622385e-03, 8.762762818419279e-03,  6.737519711673423e-04, 6.335963033960991e-02, 3.846912548172856e-03, -1.072395357942699e-02,  1.588808745993021e-02, -1.400679783395740e-02, -1.278881583995413e-02, -8.588241569662747e-03,  1.347877130325745e-02, 1.650989682744214e-02, -6.509375480429569e-03, 7.768277063407684e-03,  3.846912548172856e-03, 1.087478329181043e-01, 2.267889003573655e-02, 1.952355892320098e-02, -4.789718259362363e-02, -1.702783019309989e-02, -9.055007902283640e-03, -1.723444085005844e-02,  2.662286690399483e-02, -8.018022570451311e-03, 3.752724784814665e-02, -1.072395357942699e-02,  2.267889003573655e-02, 1.161473118838560e-01, -1.817106173575409e-02, -2.904929759594474e-02, -2.762066086929920e-04, -1.829424162549596e-03, -2.103383209907141e-03, 1.985222354414198e-02,  6.428316502925752e-03, -2.209866685621547e-03, 1.588808745993021e-02, 1.952355892320098e-02, -1.817106173575411e-02, 1.001153606244444e-01, -1.591224477073870e-02, 2.874202315628934e-02,  8.469390896848660e-03, 1.470743522772382e-02, -2.287156843441952e-02, -2.403209153959746e-02, -1.265124151776702e-02, -1.400679783395741e-02, -4.789718259362364e-02, -2.904929759594476e-02, -1.591224477073871e-02, 1.486514512462358e-01, 3.596447862086136e-02, 9.996318313741467e-03,  3.239927465185523e-02, -1.495832434049028e-02, -1.465233418904743e-02, -3.347263784974150e-03,  2.923966451680061e-03, -2.384846062711299e-02, -3.338592381359286e-02, -4.164584546425703e-04,  5.342080423102794e-02, 9.996318313741467e-03, 1.586290206854839e-01, 6.292221466613651e-03, -1.312904535240721e-03, 8.025444434842550e-03, -1.148554771869483e-02, -1.840526566621705e-03,
                  -3.021254815689899e-03, 2.680695457990276e-03, -1.958567158816680e-02, -2.247670317934288e-03,  3.239927465185523e-02, 6.292221466613651e-03, 1.690370319738496e-01, -3.062090876312928e-02, -4.134935654953310e-02, 1.171513626637497e-02, 8.720629803287579e-03, 2.166520554018957e-03, -2.986508603200463e-02, -8.929698511550115e-03, 2.661764833233941e-02, -1.495832434049028e-02, -1.312904535240721e-03, -3.062090876312928e-02, 1.100988668543387e-01, -9.006903809189348e-04,  1.611059103947837e-02, -2.762542549436532e-03, 1.248604730181230e-02, 3.833481233343240e-02,  2.291113023466116e-02, -2.988584689288261e-02, -1.465233418904743e-02, 8.025444434842550e-03, -4.134935654953310e-02, -9.006903809189348e-04, 1.412327560853690e-01, -1.533341650452055e-02,  1.574291519550666e-02, -9.715503612055486e-03, -1.531760821873935e-02, 7.546394916743642e-03, -1.993943942386996e-02, -3.347263784974150e-03, -1.148554771869483e-02, 1.171513626637497e-02,  1.611059103947837e-02, -1.533341650452055e-02, 1.140129536864358e-01, 4.106618747696167e-05,  1.198943049644964e-03, 2.875684996295427e-02, 3.158464316113752e-03, -5.505217578858872e-03,  2.923966451680061e-03, -1.840526566621705e-03, 8.720629803287579e-03, -2.762542549436532e-03,  1.574291519550666e-02, 4.106618747696167e-05, 8.829335354754346e-02, -4.488873185168017e-03, -1.648909655807400e-02, 1.727225618612907e-02, -1.294166822980827e-02, -2.384846062711299e-02, -3.021254815689899e-03, 2.166520554018957e-03, 1.248604730181230e-02, -9.715503612055486e-03,  1.198943049644964e-03, -4.488873185168017e-03, 1.300105700071443e-01, 3.141960249160024e-02,  1.205268962128401e-02, -5.276811303626095e-02, -3.338592381359286e-02, 2.680695457990276e-03, -2.986508603200463e-02, 3.833481233343240e-02, -1.531760821873935e-02, 2.875684996295427e-02, -1.648909655807400e-02, 3.141960249160024e-02, 1.539265763582731e-01, -2.462932049899615e-02, -4.535402991341685e-02, -4.164584546425703e-04, -1.958567158816680e-02, -8.929698511550115e-03,  2.291113023466116e-02, 7.546394916743642e-03, 3.158464316113752e-03, 1.727225618612907e-02,  1.205268962128401e-02, -2.462932049899615e-02, 1.310221744227308e-01, -1.263781591979387e-02,  5.342080423102794e-02, -2.247670317934288e-03, 2.661764833233941e-02, -2.988584689288261e-02, -1.993943942386996e-02, -5.505217578858872e-03, -1.294166822980827e-02, -5.276811303626095e-02, -4.535402991341685e-02, -1.263781591979387e-02, 2.042444129561639e-01 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$lag_one_cov)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$fixed_effects)),
               c(numeric(0) )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_iter)),
               c(9 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q)),
               c( 0.349107024262949928, 0.048237805055287380, 0.240799462588214858, -0.652277584966313939, 0.227215213888580514, -0.582373341586375437, 0.051370658050987228, -0.226810184647718655, -0.776167881962441797, -0.325889201211273627,  0.534165655479053259, 0.048237805055287380, 0.648614055229271846, -0.175988500680124643, 0.122637442868192281,  0.288111345365017812, -0.141532881503842134, 0.081116162785229123, 0.040025232220657667, -0.041659207814644314,  0.035536500033946136, -0.262966353970167654, 0.240799462588214858, -0.175988500680124643, 1.814491894381718540, -0.920620026222993659, -1.126075411654097191, -0.315262237231973885, 0.007301296951680169, 0.071308786608678648, -0.849618291682344817, -0.276421175437300826, 1.035119129057143539, -0.652277584966313939, 0.122637442868192281, -0.920620026222993659, 1.772338543199488026, 0.027071548585058780, 1.316296535548109103, 0.005511506054244838,  0.750471889725512264, 1.891359823103741622, 0.813545476374856080, -1.797575364509969686, 0.227215213888580514,  0.288111345365017812, -1.126075411654097191, 0.027071548585058780, 1.359802885434355568, -0.466241045008483013,  0.163617167525834467, -0.182821267966829959, -0.315302117599304987, -0.018398751974653918, -0.466114035105392122, -0.582373341586375437, -0.141532881503842134, -0.315262237231973885, 1.316296535548109103, -0.466241045008483013,  1.349930869260141275, -0.099775971438187561, 0.610873581876046901, 1.709308211208269812, 0.504479410764583358, -1.199549743020140058, 0.051370658050987228, 0.081116162785229123, 0.007301296951680169, 0.005511506054244838,  0.163617167525834467, -0.099775971438187561, 0.138287483288981078, 0.102652695337275862, -0.160828267919056467,  0.129069596408492859, -0.174083019944997086, -0.226810184647718655, 0.040025232220657667, 0.071308786608678648,  0.750471889725512264, -0.182821267966829959, 0.610873581876046901, 0.102652695337275862, 0.749922773400209630,  0.782162298954788060, 0.462404416466704493, -1.029188034506257843, -0.776167881962441797, -0.041659207814644314, -0.849618291682344817, 1.891359823103741622, -0.315302117599304987, 1.709308211208269812, -0.160828267919056467,  0.782162298954788060, 2.484584608645461934, 0.585727106404809406, -1.800083824998018445, -0.325889201211273627,  0.035536500033946136, -0.276421175437300826, 0.813545476374856080, -0.018398751974653918, 0.504479410764583358,  0.129069596408492859, 0.462404416466704493, 0.585727106404809406, 0.761842331993397015, -0.889518280029498287,  0.534165655479053259, -0.262966353970167654, 1.035119129057143539, -1.797575364509969686, -0.466114035105392122, -1.199549743020140058, -0.174083019944997086, -1.029188034506257843, -1.800083824998018445, -0.889518280029498287,  2.406433656063439841 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$Q_0)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$n_risk)),
               c(2041, 1968, 1876, 1813, 1684, 1539, 1448, 1262, 1189, 1042 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$times)),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$risk_set)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$data)),
               c(NULL )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$order)),
               c(1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$F_)),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$method)),
               c("EKF" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$model)),
               c("exp_clip_time_w_jump" )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$est_Q_0)),
               c(FALSE )
               , tolerance = 1e-04)

  expect_equal(unname(c(result_exp$LR)),
               c(1 )
               , tolerance = 1e-04)
})

#############
# Test ukf for exponential model
test_that("UKF and exp models give previous computed results with exponential estimation method",{
  suppressMessages(result_exp <- ddhazard(
    formula = survival::Surv(start, stop, event) ~ group,
    data = head_neck_cancer,
    by = 1,
    Q_0 = diag(1, 2),
    Q = diag(1e-1, 2),
    control = list(est_Q_0 = F, n_max = 10^4, eps = 10^-4,
                   method = "UKF", beta = 0, alpha = 1, save_risk_set = F,
                   save_data = F),
    max_T = 30,
    id = head_neck_cancer$id, order = 1,
    verbose = F,
    model = "exp_combined"))

  # plot(result_exp, cov_index = 1)
  # get_expect_equal(result_exp, file = "tmp.txt")

  expect_equal(c(result_exp$state_vecs),
               c(-3.69824910038536681, -3.69826065307419105, -3.26859707903190611, -2.95268233222984611, -2.77543499709954977, -2.30794811268055700, -2.42439356837259012, -2.81149408862572647, -2.96604835642004883, -3.07858919668809561, -3.20063414968332038, -3.41926978204197374, -3.52994455186449541, -3.55641905123767721, -3.37491862878595850, -3.44499480899945176, -3.58227233617389818, -3.68103585148111367, -3.63065354518771377, -3.71200873204608328, -3.67032845288164422, -3.81395837158664808, -3.99381412724516816, -4.11150917245706715, -4.16695822942270855, -4.28767616986713485, -4.36004225410994461, -4.38670995220066917, -4.49278911604918907, -4.56290234223730184, -4.59769109320577751, 0.11807985586911025, 0.11808530336174596, 0.27632089931661302, 0.39441596880613600,  0.44475387808013933, 0.62553592624579213, 0.58700138241384914, 0.44698311793234968, 0.41573489470184921,  0.38651047757107021, 0.35020473931936630, 0.27393947697249116, 0.24291147143516253, 0.24814219767729603,  0.32890209135425325, 0.30181476068198732, 0.24962479163596751, 0.21627433894170778, 0.23651753893562372,  0.20830070281157384, 0.22414450177235412, 0.16171554778759148, 0.09051806014666031, 0.04117246851472023,  0.01374834998307983, -0.03268537836081864, -0.06211552824670100, -0.07542661687135926, -0.11279018769181048, -0.13746377408854771, -0.14969203785384111 ))

  expect_equal(c(result_exp$state_vars),
               c( 0.2564700426806703071, 0.0193186468689041067, 0.0193186468689036800, 0.1460679280286585335, 0.1728246489568892597, -0.0103853264130547573, -0.0103853264130552569, 0.1245967614459552730, 0.1636068616711931767, -0.0111959308819632730, -0.0111959308819639114, 0.1157155704057426515, 0.1365216670876577831, -0.0195022107404757561, -0.0195022107404763667,  0.1040987073854811595, 0.1088375823506339268, -0.0295253088367425226, -0.0295253088367430361, 0.0922122755274398243,  0.0999342564870863564, -0.0282906762882705470, -0.0282906762882710328, 0.0878067097506801952, 0.0761398448221971558, -0.0355081971451097628, -0.0355081971451101652, 0.0807934996775148823, 0.0812534168716248228, -0.0315515210821908920, -0.0315515210821913500, 0.0824053001504273380, 0.0979061630803494398, -0.0233180542765690779, -0.0233180542765695636,  0.0887772943842919937, 0.1091170148401460760, -0.0190664945155813240, -0.0190664945155818374, 0.0944788549118243948,  0.1182914827457968998, -0.0162006846878412492, -0.0162006846878417522, 0.1005302060185183921, 0.1282544104358352977, -0.0131465079714294733, -0.0131465079714299625, 0.1075358609811898902, 0.1460849851380808251, -0.0071849005684081965, -0.0071849005684086944, 0.1164908309144385978, 0.1610515286206129770, -0.0029014446928423386, -0.0029014446928428174,  0.1250620445695320437, 0.1714332977747100861, -0.0009241896977249184, -0.0009241896977254145, 0.1328601799079411339,  0.1633019047119910749, -0.0065095597550550131, -0.0065095597550555214, 0.1376802972770761468, 0.1665792881782757751, -0.0079532327386078842, -0.0079532327386084081, 0.1446630721132587949, 0.1800091224566259807, -0.0054513004701847341, -0.0054513004701852476, 0.1539260354743817172, 0.1959445847411625530, -0.0021116460115749511, -0.0021116460115754680,  0.1640443628399728127, 0.2029690571334170079, -0.0024857963115429149, -0.0024857963115434596, 0.1729761367209960432,  0.2137256119087934947, -0.0015423366061816310, -0.0015423366061821930, 0.1828936731151880046, 0.2146132972968490260, -0.0045835080219415875, -0.0045835080219421738, 0.1915432375761303707, 0.2282835894504103291, -0.0030645417715430660, -0.0030645417715436454, 0.2025428104461669543, 0.2509516423207453495, 0.0020564607669640231, 0.0020564607669634610,  0.2156260536991032595, 0.2749757908163666942, 0.0075651377915518583, 0.0075651377915513171, 0.2292765668001862933,  0.2973751834460302002, 0.0121401558254466482, 0.0121401558254461209, 0.2428204459950295269, 0.3300729581246119038,  0.0201865451705297780, 0.0201865451705292437, 0.2579818829645606493, 0.3688951390670381913, 0.0302760385673999377,  0.0302760385673994034, 0.2741578304716998926, 0.4144867036031239516, 0.0424556108013083577, 0.0424556108013077957,  0.2912416011837608565, 0.4868508931419497920, 0.0640569472414180274, 0.0640569472414174584, 0.3120384092986653313,  0.5895922959581253453, 0.0961793387860182258, 0.0961793387860176568, 0.3368047533002774263 ))

  expect_equal(c(result_exp$lag_one_cov),
               c(NULL ))

  expect_equal(c(result_exp$fixed_effects),
               c(numeric(0) ))

  expect_equal(c(result_exp$n_iter),
               c(92 ))

  expect_equal(c(result_exp$Q),
               c(0.14161964589735906, 0.04550611673022104, 0.04550611673022104, 0.02948981054928508 ))

  expect_equal(c(result_exp$Q_0),
               c(1, 0, 0, 1 ))

  expect_equal(c(result_exp$n_risk),
               c(96, 95, 91, 84, 78, 65, 53, 49, 46, 43, 39, 37, 36, 36, 33, 31, 30, 30, 25, 25, 23, 21, 21, 21, 20, 19, 19, 18, 18, 18 ))

  expect_equal(c(result_exp$times),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30 ))

  expect_equal(c(result_exp$risk_set),
               c(NULL ))

  expect_equal(c(result_exp$data),
               c(NULL ))

  expect_equal(c(result_exp$order),
               c(1 ))

  expect_equal(c(result_exp$F_),
               c(1, 0, 0, 1 ))

  expect_equal(c(result_exp$method),
               c("UKF" ))

  expect_equal(c(result_exp$model),
               c("exp_combined" ))

  expect_equal(c(result_exp$est_Q_0),
               c(FALSE ))

  expect_equal(c(result_exp$LR),
               c(1 ))




  set.seed(44334)
  sims <- test_sim_func_exp(n_series = 1e3, n_vars = 10, t_0 = 0, t_max = 10,
                            x_range = 1, x_mean = 0, re_draw = T, beta_start = 0,
                            intercept_start = -5, sds = c(.1, rep(1, 10)))
  # sum(sims$res$event)

  arg_list <- list(
    formula = survival::Surv(tstart, tstop, event) ~ . - id - tstart - tstop - event,
    data = sims$res,
    by = (by_ <- 1),
    Q_0 = diag(1, 11),
    Q = diag(1e-2, 11),
    control = list(est_Q_0 = F, n_max = 10^2, eps = 10^-2,
                   method = "UKF", beta = 0, save_risk_set = F,
                   save_data = F, alpha = 1),
    max_T = 10,
    id = sims$res$id, order = 1,
    model = "exp_combined")

  suppressWarnings(suppressMessages(fit <- do.call(ddhazard, arg_list)))

  # matplot(fit$state_vecs, lty = 2, type = "l", ylim = range(fit$state_vecs, sims$betas))
  # matplot(sims$betas, lty = 1, add = T, type = "l")
  # get_expect_equal(fit, eps = 1e-6, file = "tmp.txt")

  expect_equal(c(fit$state_vecs),
               c(-5.55596071795522750, -5.56994330281473271, -5.47689448436752269, -5.38793541512054119, -5.21429346433077257, -5.06707911210309891, -5.03707944211337111, -4.94352150162690318, -4.87185144619147437, -4.86554726374850777, -4.78894763492490494, 1.29999868485253467, 1.30307585289973016, 1.47774739447913062, 1.33884940920273410,  1.29707437066499454, 1.24904735969331737, 1.12087916547118005, 1.63410742259599084, 2.15868281733513578,  2.42562338875773031, 2.63219402999858909, 1.49272721176549950, 1.43732150945292103, 1.72574580636434649,  2.15842563026728129, 2.90562207503666059, 3.57025966532525452, 3.76575479011061898, 3.94747184486900915,  4.03901196640563853, 3.95270655282910433, 4.13522182208106681, -0.27885010346086636, -0.24963726399557681,  0.07534231340038378, -0.45691074923146047, -0.84266278212089285, -1.18094164598099649, -1.53395504619747669, -0.24637184968808046, 1.10284050170390602, 1.83803505477622275, 2.33426813385907694, 0.02078099042395620,  0.10490281129282830, 0.07327038873602221, -0.82862833912243894, -1.96976471657602792, -2.99134311646635531, -3.52306165646384040, -2.65833749262610253, -1.64506209976289530, -0.91795885637259400, -0.65939667597762130,  1.18860394548060389, 1.22984682609031171, 1.40323632764705253, 0.83364203354611432, 0.27811017192791843, -0.22178654491102698, -0.56703848145922586, 0.42682093645585750, 1.46078847592952110, 2.11292303913317392,  2.54619430283935211, -0.30754840690248714, -0.29237306830484600, 0.02962773282481754, -0.33245032016104681, -0.52944038582821307, -0.70471047175291368, -0.97018899884246079, 0.10156840797995847, 1.23920607011676243,  1.84101202188925206, 2.29811150793390384, -1.12993179598458360, -1.07887994235466800, -1.44561856197917216, -1.78091443275995243, -2.47421402057239881, -3.09494273063691816, -3.23934312525480594, -3.70636758094830343, -4.10009293033647904, -4.15587715343927488, -4.42952011092997733, -0.06459103533476443, -0.04401720840447931, -0.47211147149827803, -0.43925151731062023, -0.73156516235836899, -0.99448352804730078, -0.88495315379438499, -1.84733009153873340, -2.84393035292687912, -3.29287633230710197, -3.75086202746032482, 0.30714960728769941,  0.29294040939087085, 0.58292828629070526, 0.57293039644299237, 0.77612980457228653, 0.93658710064529993,  0.86677028650819965, 1.51850334264821307, 2.15767936167917096, 2.43755378789837707, 2.73838952830004523,  0.28554674147999470, 0.28776449657512970, 0.38399364275267411, 0.30432007178652010, 0.26776597527882151,  0.23986911746485004, 0.19099545575716717, 0.49895596751496624, 0.78111752069757578, 0.94152809289256278,  1.16196370779521274 )
               , tolerance = 1e-06)

  expect_equal(c(fit$state_vars),
               c( 5.428323918659683e-02, 6.828319106633629e-03, 3.562055678772259e-02, 7.867252268149756e-03, -3.953699535271971e-02, -1.400447441693450e-02, 1.256156463997895e-02, -3.632649937805758e-02, -3.281466285267567e-02, 2.228672048744447e-02,  7.372224908608878e-03, 6.828319106633301e-03, 9.965498388274285e-02, 2.027485247422458e-02, 7.374666550411850e-02,  4.050843487279526e-02, 4.840215970483507e-02, 6.183444775230796e-02, -3.662551383813264e-02, -6.217858636386067e-02,  4.521863079618784e-02, 1.112576847873031e-02, 3.562055678771938e-02, 2.027485247422460e-02, 2.391130965208168e-01, -8.620385190704965e-03, -1.535594143363059e-01, -4.287219639285533e-02, 2.070874405464775e-02, -1.409695427415606e-01, -9.944626168031730e-02, 6.774072318027202e-02, 1.759201599224404e-02, 7.867252268151041e-03, 7.374666550411793e-02, -8.620385190704898e-03, 2.691325534834618e-01, 1.544917981080607e-01, 1.423296487988132e-01, 1.506961892217125e-01, -3.818859316254582e-02, -1.155365383625339e-01, 7.751934222834458e-02, 3.607735330491409e-02, -3.953699535271872e-02,  4.050843487279588e-02, -1.535594143363061e-01, 1.544917981080606e-01, 3.948680070316337e-01, 1.714951849130901e-01,  1.020854665879710e-01, 1.261358897834986e-01, 6.468549727754451e-03, -8.004187750250382e-03, 1.875309386223769e-02, -1.400447441693874e-02, 4.840215970483466e-02, -4.287219639285505e-02, 1.423296487988133e-01, 1.714951849130902e-01,  2.529337911036064e-01, 1.152435278914924e-01, 1.660910239209631e-02, -6.458320670591815e-02, 3.993783237940832e-02,  4.585033002533691e-02, 1.256156463997769e-02, 6.183444775230812e-02, 2.070874405464773e-02, 1.506961892217126e-01,  1.020854665879714e-01, 1.152435278914918e-01, 2.280719455898476e-01, -5.107369132091402e-02, -1.152799248336780e-01,  7.285366940619832e-02, 3.458967360632654e-02, -3.632649937805540e-02, -3.662551383813349e-02, -1.409695427415599e-01, -3.818859316254560e-02, 1.261358897834988e-01, 1.660910239209602e-02, -5.107369132091400e-02, 2.696096362101996e-01,  1.234187259587724e-01, -8.227150577631705e-02, -1.641014809965517e-02, -3.281466285267579e-02, -6.217858636386046e-02, -9.944626168031695e-02, -1.155365383625339e-01, 6.468549727754417e-03, -6.458320670591805e-02, -1.152799248336783e-01,  1.234187259587723e-01, 2.611862021997762e-01, -9.852589086182106e-02, -3.049172285766424e-02, 2.228672048744487e-02,  4.521863079618853e-02, 6.774072318027226e-02, 7.751934222834425e-02, -8.004187750250573e-03, 3.993783237940863e-02,  7.285366940619868e-02, -8.227150577631774e-02, -9.852589086182106e-02, 1.740454641986210e-01, 2.128721158863927e-02,  7.372224908609572e-03, 1.112576847872982e-02, 1.759201599224437e-02, 3.607735330491448e-02, 1.875309386223752e-02,  4.585033002533717e-02, 3.458967360632630e-02, -1.641014809965440e-02, -3.049172285766418e-02, 2.128721158863884e-02,  1.364980431492050e-01, 4.799789819037709e-02, 5.441452227615710e-03, 3.544338256334856e-02, 5.903256245629858e-03, -4.216848550437481e-02, -1.648270878415659e-02, 1.093784362062969e-02, -3.694717409541864e-02, -3.206112979761090e-02,  2.165965488993177e-02, 6.185109241825424e-03, 5.441452227614531e-03, 1.013764853459888e-01, 1.921728422520904e-02,  9.028329843615308e-02, 5.787356845455780e-02, 6.708026778424259e-02, 7.975987589883135e-02, -4.041561625019539e-02, -7.667242309814384e-02, 5.369724980208937e-02, 1.701245039219794e-02, 3.544338256334417e-02, 1.921728422520927e-02,  2.471259890837695e-01, -1.478201344338423e-02, -1.766500994496679e-01, -5.582804828553284e-02, 1.753888965371743e-02, -1.570163765818481e-01, -1.065218563664263e-01, 7.288646569227457e-02, 1.588048348922941e-02, 5.903256245629502e-03,  9.028329843615233e-02, -1.478201344338419e-02, 2.951124270026151e-01, 1.968917863008016e-01, 1.834871069053798e-01,  1.876479122856238e-01, -4.021509821688596e-02, -1.428165323690476e-01, 9.398843953154976e-02, 4.758130693685855e-02, -4.216848550437403e-02, 5.787356845455814e-02, -1.766500994496683e-01, 1.968917863008015e-01, 4.574190656535225e-01,  2.252122395678915e-01, 1.424443203902462e-01, 1.413120411182909e-01, -1.197475874665821e-02, 2.321536158950926e-03,  2.992335659902632e-02, -1.648270878416205e-02, 6.708026778424209e-02, -5.582804828553306e-02, 1.834871069053802e-01,  2.252122395678924e-01, 2.917436509495464e-01, 1.559981479844745e-01, 1.909074942796399e-02, -9.133256685343344e-02,  5.595737559202858e-02, 5.412331709956784e-02, 1.093784362062678e-02, 7.975987589883135e-02, 1.753888965371708e-02,  1.876479122856239e-01, 1.424443203902468e-01, 1.559981479844738e-01, 2.612109613694222e-01, -5.799377664710675e-02, -1.454692405605157e-01, 9.204944499890885e-02, 4.573291934584014e-02, -3.694717409541483e-02, -4.041561625019647e-02, -1.570163765818471e-01, -4.021509821688590e-02, 1.413120411182906e-01, 1.909074942796304e-02, -5.799377664710720e-02,  2.836348408906617e-01, 1.401542168026184e-01, -9.339068108699461e-02, -1.877641863702911e-02, -3.206112979760906e-02, -7.667242309814375e-02, -1.065218563664258e-01, -1.428165323690477e-01, -1.197475874665863e-02, -9.133256685343367e-02, -1.454692405605164e-01, 1.401542168026181e-01, 2.844334950838674e-01, -1.180649239645828e-01, -4.024146019393968e-02,  2.165965488993100e-02, 5.369724980209028e-02, 7.288646569227508e-02, 9.398843953154973e-02, 2.321536158950773e-03,  5.595737559202918e-02, 9.204944499890977e-02, -9.339068108699575e-02, -1.180649239645832e-01, 1.802937468172428e-01,  2.716822867466095e-02, 6.185109241825580e-03, 1.701245039219735e-02, 1.588048348922960e-02, 4.758130693685876e-02,  2.992335659902617e-02, 5.412331709956800e-02, 4.573291934583976e-02, -1.877641863702813e-02, -4.024146019393938e-02,  2.716822867466026e-02, 1.296179270117577e-01, 4.036417484609059e-02, 3.011790801388393e-03, 2.877232859743313e-02,  3.728831451802854e-03, -3.812857391721963e-02, -1.709523415542760e-02, 8.103628285156858e-03, -3.068392949997178e-02, -2.656572389826149e-02, 1.776288968425226e-02, 4.389242817967175e-03, 3.011790801387224e-03, 9.492127879232448e-02,  1.614567486212890e-02, 8.782991824901268e-02, 6.038148379786979e-02, 6.835017863842018e-02, 7.931582118664195e-02, -3.760630583446757e-02, -7.536956560118532e-02, 5.228289734131707e-02, 1.760721450320955e-02, 2.877232859742878e-02,  1.614567486212910e-02, 2.316204196291864e-01, -1.736101075831219e-02, -1.712661690974228e-01, -5.740059709004307e-02,  1.435158829547043e-02, -1.490330804013401e-01, -9.974901978279503e-02, 6.810966765111537e-02, 1.299750412765511e-02,  3.728831451802452e-03, 8.782991824901180e-02, -1.736101075831215e-02, 2.765443537543624e-01, 1.943883580625409e-01,  1.796046747530529e-01, 1.814225498523945e-01, -3.529961802105821e-02, -1.377644686398671e-01, 9.072610067716398e-02,  4.706405969330638e-02, -3.812857391721892e-02, 6.038148379787005e-02, -1.712661690974230e-01, 1.943883580625407e-01,  4.500427653148210e-01, 2.266815197039402e-01, 1.440466100035826e-01, 1.375173081046702e-01, -1.611604926828770e-02,  5.368925027955734e-03, 3.163570299757492e-02, -1.709523415543307e-02, 6.835017863841970e-02, -5.740059709004319e-02,  1.796046747530536e-01, 2.266815197039414e-01, 2.847276312793682e-01, 1.558134296937111e-01, 2.039720975664346e-02, -9.200784193859284e-02, 5.631378885957406e-02,
                  5.157254516447842e-02, 8.103628285153940e-03, 7.931582118664195e-02,  1.435158829547019e-02, 1.814225498523949e-01, 1.440466100035833e-01, 1.558134296937102e-01, 2.537158043795482e-01, -5.471022417057865e-02, -1.430030353685322e-01, 9.088841240099471e-02, 4.535687887676954e-02, -3.068392949996798e-02, -3.760630583446863e-02, -1.490330804013389e-01, -3.529961802105827e-02, 1.375173081046698e-01, 2.039720975664253e-02, -5.471022417057904e-02, 2.713275261107813e-01, 1.347526847505134e-01, -8.945960725515190e-02, -1.702324915546247e-02, -2.656572389825965e-02, -7.536956560118512e-02, -9.974901978279450e-02, -1.377644686398674e-01, -1.611604926828821e-02, -9.200784193859284e-02, -1.430030353685328e-01, 1.347526847505132e-01, 2.739862121071220e-01, -1.156059437310415e-01, -4.006425259314273e-02, 1.776288968425151e-02, 5.228289734131796e-02, 6.810966765111597e-02, 9.072610067716395e-02,  5.368925027955533e-03, 5.631378885957438e-02, 9.088841240099550e-02, -8.945960725515308e-02, -1.156059437310419e-01,  1.726752863974074e-01, 2.679340199137069e-02, 4.389242817967331e-03, 1.760721450320896e-02, 1.299750412765531e-02,  4.706405969330669e-02, 3.163570299757483e-02, 5.157254516447854e-02, 4.535687887676918e-02, -1.702324915546150e-02, -4.006425259314243e-02, 2.679340199137004e-02, 1.199223378168219e-01, 3.206847580086782e-02, -1.548404229998322e-03,  1.518733028374152e-02, -8.919949541708015e-04, -2.859093671855872e-02, -1.759152971836864e-02, 2.000283128727050e-03, -1.638098754474370e-02, -1.435192846556775e-02, 9.253543706064998e-03, 1.785516695721505e-03, -1.548404229999460e-03,  7.531777287410363e-02, 8.257992345620002e-03, 5.609847875953178e-02, 4.199846067222682e-02, 4.538668894855559e-02,  5.233109938971870e-02, -2.287268323825339e-02, -4.962488663914665e-02, 3.483607886783503e-02, 1.060837589387442e-02,  1.518733028373708e-02, 8.257992345620170e-03, 1.853401523601706e-01, -1.946165169586594e-02, -1.314729964645545e-01, -4.905381776833138e-02, 6.842696566051639e-03, -1.081370477091877e-01, -7.111467081391823e-02, 4.824814214650125e-02,  8.480352930945125e-03, -8.919949541711450e-04, 5.609847875953054e-02, -1.946165169586596e-02, 1.916094119757109e-01,  1.308580087468096e-01, 1.161322659870077e-01, 1.157659109439662e-01, -1.518614183255382e-02, -8.450529022394553e-02,  5.611139152295759e-02, 2.971188999209162e-02, -2.859093671855793e-02, 4.199846067222684e-02, -1.314729964645549e-01,  1.308580087468092e-01, 3.507172262365826e-01, 1.639030094224727e-01, 9.685987567684812e-02, 1.116051673239599e-01, -8.412698240777353e-04, -3.324469189231580e-03, 2.016992928749079e-02, -1.759152971837407e-02, 4.538668894855494e-02, -4.905381776833154e-02, 1.161322659870086e-01, 1.639030094224743e-01, 2.224774356913670e-01, 1.043585904053795e-01,  2.524628138437109e-02, -5.675223493863209e-02, 3.354344985919725e-02, 3.483930965805750e-02, 2.000283128724160e-03,  5.233109938971854e-02, 6.842696566051423e-03, 1.157659109439667e-01, 9.685987567684907e-02, 1.043585904053786e-01,  1.931223323787202e-01, -3.288680931739516e-02, -9.450388266047402e-02, 5.964200042471774e-02, 2.983608536020855e-02, -1.638098754473984e-02, -2.287268323825440e-02, -1.081370477091866e-01, -1.518614183255406e-02, 1.116051673239592e-01,  2.524628138437007e-02, -3.288680931739561e-02, 2.209740278264548e-01, 9.480449802312788e-02, -6.217300914288462e-02, -9.443015843824492e-03, -1.435192846556592e-02, -4.962488663914626e-02, -7.111467081391779e-02, -8.450529022394612e-02, -8.412698240784638e-04, -5.675223493863202e-02, -9.450388266047466e-02, 9.480449802312760e-02, 2.134027569830098e-01, -7.971241612495389e-02, -2.655495146596925e-02, 9.253543706064252e-03, 3.483607886783575e-02, 4.824814214650192e-02,  5.611139152295756e-02, -3.324469189231769e-03, 3.354344985919741e-02, 5.964200042471846e-02, -6.217300914288579e-02, -7.971241612495410e-02, 1.431474877678496e-01, 1.768206182575079e-02, 1.785516695721677e-03, 1.060837589387375e-02,  8.480352930945335e-03, 2.971188999209195e-02, 2.016992928749074e-02, 3.483930965805760e-02, 2.983608536020816e-02, -9.443015843823507e-03, -2.655495146596897e-02, 1.768206182575013e-02, 1.063726446585493e-01, 2.803025384255298e-02, -8.793181275190409e-04, 1.073053463514007e-02, 3.468147486588202e-03, -2.039025292457087e-02, -1.259298537581270e-02,  4.404906123274793e-03, -1.272277861568344e-02, -1.372272645930150e-02, 8.366169789825562e-03, 1.924041089290991e-03, -8.793181275201685e-04, 6.923462302147906e-02, 6.886265846395858e-03, 5.478005122659609e-02, 4.250472780478729e-02,  4.569195439119639e-02, 5.120980245575694e-02, -2.143108509879223e-02, -4.755795324071779e-02, 3.303442896131664e-02,  1.060392302710838e-02, 1.073053463513563e-02, 6.886265846395976e-03, 1.755104860343868e-01, -1.927511541821129e-02, -1.249802106407780e-01, -4.735433966066568e-02, 5.893976476715667e-03, -1.022702627561686e-01, -6.713844453579332e-02,  4.473792294943070e-02, 7.174366361759527e-03, 3.468147486587830e-03, 5.478005122659500e-02, -1.927511541821151e-02,  1.857375802004353e-01, 1.294848294384809e-01, 1.152868053110318e-01, 1.143919338786248e-01, -1.529830643143281e-02, -8.239161149325464e-02, 5.504978754127321e-02, 2.927181593411540e-02, -2.039025292457014e-02, 4.250472780478758e-02, -1.249802106407786e-01, 1.294848294384801e-01, 3.365637924604047e-01, 1.596966069340465e-01, 9.633006627372055e-02,  1.037048717053566e-01, -3.629124894126274e-03, -2.323801159329932e-04, 1.979965610773287e-02, -1.259298537581816e-02,  4.569195439119596e-02, -4.735433966066588e-02, 1.152868053110326e-01, 1.596966069340480e-01, 2.137367956865923e-01,  1.034682866806209e-01, 2.233404631831018e-02, -5.693713027227676e-02, 3.413029546637233e-02, 3.164919120284454e-02,  4.404906123271889e-03, 5.120980245575688e-02, 5.893976476715398e-03, 1.143919338786250e-01, 9.633006627372138e-02,  1.034682866806199e-01, 1.866279405892745e-01, -3.203344469974207e-02, -9.110701781593702e-02, 5.797459844579871e-02,  2.870139432873263e-02, -1.272277861567967e-02, -2.143108509879322e-02, -1.022702627561676e-01, -1.529830643143320e-02,  1.037048717053558e-01, 2.233404631830921e-02, -3.203344469974258e-02, 2.099198643765041e-01, 9.033819507096091e-02, -5.824811004252133e-02, -8.661027244706011e-03, -1.372272645929969e-02, -4.755795324071752e-02, -6.713844453579296e-02, -8.239161149325533e-02, -3.629124894127086e-03, -5.693713027227672e-02, -9.110701781593773e-02, 9.033819507096065e-02,  2.020635423866806e-01, -7.564822871606018e-02, -2.575445359901398e-02, 8.366169789824832e-03, 3.303442896131743e-02,  4.473792294943149e-02, 5.504978754127322e-02, -2.323801159332291e-04, 3.413029546637247e-02, 5.797459844579950e-02, -5.824811004252258e-02, -7.564822871606039e-02, 1.352473629203306e-01, 1.694606321367134e-02, 1.924041089291161e-03,  1.060392302710772e-02, 7.174366361759761e-03, 2.927181593411562e-02, 1.979965610773274e-02, 3.164919120284458e-02,  2.870139432873220e-02, -8.661027244705020e-03, -2.575445359901362e-02, 1.694606321367065e-02, 9.779568467890826e-02,  2.379250550141343e-02, -1.150027786211014e-03, 2.335067441615480e-03, 6.450909229905735e-03, -8.778234329574298e-03, -7.384622626984689e-03, 5.030415473222489e-03, -4.444337612437186e-03, -8.999895718499427e-03,
                  4.466940852149900e-03,  1.857000029455910e-03, -1.150027786212229e-03, 5.875585055214633e-02, 6.083417068662601e-03, 4.278671858963570e-02,  3.358485112580163e-02, 3.645285124306060e-02, 4.044310059365233e-02, -1.758804888657369e-02, -3.817765766952677e-02,  2.625978071768810e-02, 8.531735647881875e-03, 2.335067441610973e-03, 6.083417068662877e-03, 1.496752126668880e-01, -1.142312187333393e-02, -9.384392979037129e-02, -3.363167131509268e-02, 8.850984928014061e-03, -8.050160173444555e-02, -5.562162844746796e-02, 3.485188080224673e-02, 7.304595837388780e-03, 6.450909229905167e-03, 4.278671858963475e-02, -1.142312187333472e-02, 1.567963328532487e-01, 9.957306640022003e-02, 9.030279902334540e-02, 9.182188141012325e-02, -1.667388487354323e-02, -6.854220279525831e-02, 4.595701254006827e-02, 2.444773066003875e-02, -8.778234329573744e-03,  3.358485112580170e-02, -9.384392979037248e-02, 9.957306640021833e-02, 2.719630280875677e-01, 1.220446481521309e-01,  7.359380362187697e-02, 7.757050371546155e-02, -4.664698771368517e-03, 3.161033473316185e-03, 1.422888901312352e-02, -7.384622626990333e-03, 3.645285124306015e-02, -3.363167131509319e-02, 9.030279902334587e-02, 1.220446481521326e-01,  1.801461351085511e-01, 8.224907723583333e-02, 1.346238806531214e-02, -4.869618761528287e-02, 2.946856751815838e-02,  2.409971968786763e-02, 5.030415473219432e-03, 4.044310059365236e-02, 8.850984928013481e-03, 9.182188141012342e-02,  7.359380362187830e-02, 8.224907723583244e-02, 1.613530811029753e-01, -2.918055673359446e-02, -7.501678428945992e-02,  4.754152788929795e-02, 2.359351994030717e-02, -4.444337612433463e-03, -1.758804888657487e-02, -8.050160173444484e-02, -1.667388487354404e-02, 7.757050371546061e-02, 1.346238806531095e-02, -2.918055673359528e-02, 1.824941348972613e-01,  7.408369439652630e-02, -4.502184079857958e-02, -7.844049957421150e-03, -8.999895718497535e-03, -3.817765766952670e-02, -5.562162844746762e-02, -6.854220279525947e-02, -4.664698771369751e-03, -4.869618761528316e-02, -7.501678428946103e-02,  7.408369439652593e-02, 1.764428138304692e-01, -6.061106246772131e-02, -2.261252625038358e-02, 4.466940852149105e-03,  2.625978071768899e-02, 3.485188080224760e-02, 4.595701254006845e-02, 3.161033473316146e-03, 2.946856751815868e-02,  4.754152788929893e-02, -4.502184079858080e-02, -6.061106246772150e-02, 1.198072826536249e-01, 1.452563338145894e-02,  1.857000029456023e-03, 8.531735647881246e-03, 7.304595837388933e-03, 2.444773066003899e-02, 1.422888901312354e-02,  2.409971968786774e-02, 2.359351994030676e-02, -7.844049957420064e-03, -2.261252625038315e-02, 1.452563338145821e-02,  8.909598925263865e-02, 2.101533190310689e-02, -1.138334620542587e-03, -4.177560084311614e-03, 8.732056145654095e-03,  1.461093900622113e-03, -2.545837772746731e-03, 5.341483434808109e-03, 2.693180938647776e-03, -4.382043417120176e-03,  1.492321346868445e-03, 1.807746713173752e-03, -1.138334620543940e-03, 4.864805022200422e-02, 5.110709062479370e-03,  3.138825659833371e-02, 2.571121843702391e-02, 2.708038228962917e-02, 2.988263425688566e-02, -1.336236737396834e-02, -2.890794487749887e-02, 1.997173777616028e-02, 6.435267352784119e-03, -4.177560084316266e-03, 5.110709062479661e-03,  1.294026074827695e-01, -6.922424810402849e-03, -6.653403714743510e-02, -2.105902226774943e-02, 9.401478897615247e-03, -6.064219496620974e-02, -4.256103110196145e-02, 2.595362203282254e-02, 6.900226529722359e-03, 8.732056145653169e-03,  3.138825659833315e-02, -6.922424810404299e-03, 1.375311351012076e-01, 7.947746880303352e-02, 7.033383799570435e-02,  7.493159025736516e-02, -1.624274846102531e-02, -5.806784636048026e-02, 3.913229340133340e-02, 2.123791975349211e-02,  1.461093900622386e-03, 2.571121843702431e-02, -6.653403714743683e-02, 7.947746880303105e-02, 2.151356222977870e-01,  8.779700806334682e-02, 5.778155681558834e-02, 5.114281552973651e-02, -1.227527344441899e-02, 9.003798743808938e-03,  1.081794786718009e-02, -2.545837772752707e-03, 2.708038228962883e-02, -2.105902226775038e-02, 7.033383799570425e-02,  8.779700806334849e-02, 1.485246179809321e-01, 6.414974019197672e-02, 4.329146405082952e-03, -4.345457663697533e-02,  2.644935364876264e-02, 1.791685893810505e-02, 5.341483434804727e-03, 2.988263425688596e-02, 9.401478897614344e-03,  7.493159025736543e-02, 5.778155681559014e-02, 6.414974019197626e-02, 1.400124838056145e-01, -2.437097853061022e-02, -6.046203435895912e-02, 3.849833121214443e-02, 1.950138668544300e-02, 2.693180938651443e-03, -1.336236737396968e-02, -6.064219496620961e-02, -1.624274846102672e-02, 5.114281552973565e-02, 4.329146405081428e-03, -2.437097853061142e-02,  1.548704108466773e-01, 5.496736392323182e-02, -3.193507802429082e-02, -6.341109250054811e-03, -4.382043417118096e-03, -2.890794487749906e-02, -4.256103110196126e-02, -5.806784636048162e-02, -1.227527344442030e-02, -4.345457663697601e-02, -6.046203435896038e-02, 5.496736392323118e-02, 1.507665700430140e-01, -4.632762440341543e-02, -1.966539421792585e-02,  1.492321346867542e-03, 1.997173777616132e-02, 2.595362203282361e-02, 3.913229340133378e-02, 9.003798743808892e-03,  2.644935364876318e-02, 3.849833121214553e-02, -3.193507802429200e-02, -4.632762440341573e-02, 1.066136390817806e-01,  1.254032637467094e-02, 1.807746713173771e-03, 6.435267352783596e-03, 6.900226529722434e-03, 2.123791975349244e-02,  1.081794786718030e-02, 1.791685893810532e-02, 1.950138668544263e-02, -6.341109250053607e-03, -1.966539421792540e-02,  1.254032637467020e-02, 8.168736323502687e-02, 2.060974508222378e-02, -2.290846198886847e-05, -6.637547026952505e-03,  1.173689773914268e-02, 7.982565990597131e-03, 9.470043064709202e-04, 7.066205823625500e-03, 6.159985366848272e-03, -3.679714051768366e-03, 1.006865992587409e-03, 1.160246182459149e-03, -2.290846199011747e-05, 4.227714787024963e-02,  3.449807081078541e-03, 2.529698051347544e-02, 2.203189550191487e-02, 2.250689524744676e-02, 2.481971234032501e-02, -1.057000085049651e-02, -2.296887091554520e-02, 1.589629825653492e-02, 5.128357224034333e-03, -6.637547026956568e-03,  3.449807081078293e-03, 1.188772295120567e-01, -7.528841651649455e-03, -5.186084423161388e-02, -1.719073785357055e-02,  7.050595566273771e-03, -4.615127105179175e-02, -3.317263801685223e-02, 1.923915278948841e-02, 2.127619966788486e-03,  1.173689773914181e-02, 2.529698051347578e-02, -7.528841651649865e-03, 1.309521650280075e-01, 7.150684649468647e-02,  6.275388468881671e-02, 6.949613840041582e-02, -1.649505341120486e-02, -5.133564285750819e-02, 3.470452470399710e-02,  2.090167989629882e-02, 7.982565990596914e-03, 2.203189550191634e-02, -5.186084423161383e-02, 7.150684649468432e-02,  1.800980631025815e-01, 7.087254848075678e-02, 5.245559040920825e-02, 3.000392424948638e-02, -1.761459831485154e-02,  1.361970162951274e-02, 1.411717481761710e-02, 9.470043064648626e-04, 2.250689524744722e-02, -1.719073785357019e-02,  6.275388468881665e-02, 7.087254848075805e-02, 1.337003011546112e-01, 5.729621030891822e-02, -1.683499061644384e-03, -3.934019430644706e-02, 2.424505410734291e-02, 1.626012735909505e-02, 7.066205823622188e-03, 2.481971234032578e-02,  7.050595566273690e-03, 6.949613840041545e-02, 5.245559040920938e-02, 5.729621030891720e-02, 1.313729634535650e-01, -2.125796894654926e-02,
                  -5.137015476927791e-02, 3.291062601621698e-02, 1.766832126355153e-02, 6.159985366851327e-03, -1.057000085049761e-02, -4.615127105179146e-02, -1.649505341120581e-02, 3.000392424948646e-02, -1.683499061645135e-03, -2.125796894655020e-02, 1.327304940447324e-01, 4.076722951478842e-02, -2.189845365328855e-02, -1.239491654328408e-03, -3.679714051766657e-03, -2.296887091554546e-02, -3.317263801685236e-02, -5.133564285750877e-02, -1.761459831485176e-02, -3.934019430644674e-02, -5.137015476927883e-02, 4.076722951478791e-02, 1.322160304023869e-01, -3.561321243857993e-02, -1.502383506231199e-02, 1.006865992586758e-03, 1.589629825653605e-02, 1.923915278948970e-02, 3.470452470399697e-02,  1.361970162951200e-02, 2.424505410734283e-02, 3.291062601621790e-02, -2.189845365328974e-02, -3.561321243858026e-02,  9.700371667822991e-02, 9.551863630964252e-03, 1.160246182459193e-03, 5.128357224033923e-03, 2.127619966788729e-03,  2.090167989629901e-02, 1.411717481761711e-02, 1.626012735909513e-02, 1.766832126355117e-02, -1.239491654327235e-03, -1.502383506231161e-02, 9.551863630963536e-03, 7.661859617647032e-02, 2.065983214100071e-02, -8.718113714007436e-04, -8.639318253181110e-03, 6.679350335606291e-03, 6.110882143818124e-03, -2.688545863770244e-03, 2.394736130713085e-03,  9.808063938386817e-03, 5.946193476487160e-04, -2.503844434679151e-03, -8.756445431861795e-04, -8.718113714021496e-04,  3.949111323163727e-02, 1.913261929544722e-05, 1.781573904448141e-02, 2.066331140755954e-02, 1.876328886960250e-02,  1.880439388106926e-02, -5.080984484728683e-03, -1.635065432655240e-02, 1.188224828751969e-02, 3.152258945592943e-03, -8.639318253185478e-03, 1.913261929538694e-05, 1.115869029984338e-01, -1.999813966283418e-02, -5.270371083547130e-02, -2.423593439768761e-02, -4.350396986855637e-03, -3.514616374852337e-02, -2.187683064978797e-02, 9.216828101972789e-03, -3.303704542306914e-03, 6.679350335605153e-03, 1.781573904448230e-02, -1.999813966283510e-02, 1.086112216817587e-01,  6.735277273282560e-02, 4.863857157785137e-02, 4.850427775533299e-02, 2.765251073158592e-03, -2.635552697406059e-02,  1.981217602821696e-02, 1.451336238526825e-02, 6.110882143817909e-03, 2.066331140756114e-02, -5.270371083547185e-02,  6.735277273282261e-02, 1.744107998841834e-01, 6.575479022961761e-02, 4.763832845468190e-02, 3.228265588851602e-02, -1.072983383953302e-02, 1.345133663755599e-02, 1.425505190594880e-02, -2.688545863776455e-03, 1.876328886960327e-02, -2.423593439768770e-02, 4.863857157785090e-02, 6.575479022961926e-02, 1.227626600660603e-01, 4.320505994859815e-02,  9.796434963026641e-03, -2.295427150840091e-02, 1.565461595628198e-02, 1.185900139061946e-02, 2.394736130709540e-03,  1.880439388107051e-02, -4.350396986855902e-03, 4.850427775533267e-02, 4.763832845468349e-02, 4.320505994859734e-02,  1.109008626027499e-01, -3.301817792564440e-03, -2.810337721866652e-02, 1.886135556225788e-02, 1.075349843303573e-02,  9.808063938390146e-03, -5.080984484729980e-03, -3.514616374852307e-02, 2.765251073157380e-03, 3.228265588851568e-02,  9.796434963025628e-03, -3.301817792565367e-03, 1.161346443305122e-01, 2.155720905160667e-02, -6.984253441661561e-03,  6.756882798605783e-03, 5.946193476507817e-04, -1.635065432655302e-02, -2.187683064978797e-02, -2.635552697406127e-02, -1.072983383953380e-02, -2.295427150840081e-02, -2.810337721866740e-02, 2.155720905160617e-02, 1.072061914809437e-01, -1.897226703794403e-02, -6.189268358314516e-03, -2.503844434680058e-03, 1.188224828752110e-02, 9.216828101973822e-03,  1.981217602821708e-02, 1.345133663755593e-02, 1.565461595628225e-02, 1.886135556225884e-02, -6.984253441662579e-03, -1.897226703794435e-02, 8.480874565414076e-02, 4.198366149124307e-03, -8.756445431862159e-04, 3.152258945592643e-03, -3.303704542306685e-03, 1.451336238526841e-02, 1.425505190594889e-02, 1.185900139061956e-02, 1.075349843303530e-02,  6.756882798606959e-03, -6.189268358314137e-03, 4.198366149123567e-03, 7.311809148390068e-02, 2.138453042851975e-02, -3.921476424153277e-03, -9.999328172476037e-03, -4.772304547439268e-03, -2.504440417633179e-03, -1.175390691247253e-02, -7.464982479540464e-03, 1.315972405710519e-02, 9.131500934633207e-03, -6.779974824439885e-03, -2.509896959282681e-03, -3.921476424154613e-03, 4.239743596031619e-02, -2.750575588236780e-03, 2.299519439846057e-02, 2.789329483672434e-02,  2.291306815946659e-02, 2.086688699106802e-02, -4.023321261654864e-03, -1.696460476585217e-02, 1.290062658146775e-02,  5.419399267423436e-03, -9.999328172480573e-03, -2.750575588237291e-03, 1.074823592034171e-01, -3.153589041435804e-02, -5.614631880289653e-02, -2.980013020433600e-02, -1.339344557825111e-02, -2.971417526304706e-02, -1.140104839531224e-02,  7.302783537961896e-03, -1.832918749942045e-03, -4.772304547440281e-03, 2.299519439846030e-02, -3.153589041435809e-02,  1.276711936611526e-01, 9.180536974728949e-02, 6.031791193553969e-02, 5.277734432908347e-02, 1.131783425479166e-02, -2.410008448315309e-02, 1.702187348965702e-02, 1.860771634443023e-02, -2.504440417633192e-03, 2.789329483672548e-02, -5.614631880289652e-02, 9.180536974728740e-02, 1.952156838798405e-01, 7.801756241461796e-02, 5.800468296401484e-02,  3.423329060652575e-02, -1.870787320818221e-02, 1.197061126274942e-02, 1.439013651494101e-02, -1.175390691247852e-02,  2.291306815946696e-02, -2.980013020433522e-02, 6.031791193554009e-02, 7.801756241461959e-02, 1.293313803699249e-01,  4.371680392086591e-02, 1.440795476269768e-02, -2.173859189166563e-02, 1.236969068354921e-02, 1.326164502511749e-02, -7.464982479543850e-03, 2.086688699106860e-02, -1.339344557825037e-02, 5.277734432908347e-02, 5.800468296401573e-02,  4.371680392086436e-02, 1.072718864231636e-01, 5.075156316693887e-03, -1.981428591960441e-02, 1.358127781395969e-02,  1.216589382535338e-02, 1.315972405710839e-02, -4.023321261655740e-03, -2.971417526304754e-02, 1.131783425479082e-02,  3.423329060652641e-02, 1.440795476269753e-02, 5.075156316693540e-03, 1.136094289167991e-01, 1.157055047011840e-02, -5.804891352640396e-03, 4.686584769976056e-03, 9.131500934635120e-03, -1.696460476585218e-02, -1.140104839531330e-02, -2.410008448315402e-02, -1.870787320818221e-02, -2.173859189166481e-02, -1.981428591960514e-02, 1.157055047011821e-02,  9.670620415320116e-02, -1.390531647360911e-02, -8.365413822059103e-03, -6.779974824440711e-03, 1.290062658146876e-02,  7.302783537963558e-03, 1.702187348965749e-02, 1.197061126274909e-02, 1.236969068354918e-02, 1.358127781396067e-02, -5.804891352641556e-03, -1.390531647360945e-02, 8.482311303802527e-02, 6.183148733142592e-03, -2.509896959282687e-03,  5.419399267422928e-03, -1.832918749941573e-03, 1.860771634443055e-02, 1.439013651494096e-02, 1.326164502511748e-02,  1.216589382535302e-02, 4.686584769977124e-03, -8.365413822058777e-03, 6.183148733141857e-03, 7.597461971544209e-02,  2.456504826491050e-02, -5.520276351994414e-03, -7.851909450056350e-03, -1.303398396789118e-02, -1.213755065627969e-02, -1.888017320077412e-02, -1.379230639596984e-02, 1.396995566429146e-02, 1.392036888737180e-02, -1.000831559231787e-02, -3.995594050692027e-03, -5.520276351997786e-03, 5.032406658409677e-02, -2.306170883797538e-03, 3.330388708432208e-02,  3.530296522626319e-02, 3.104461234193429e-02, 2.894586069880514e-02,
                  -4.599595839864232e-03, -2.301876478254286e-02,  1.566980567894849e-02, 6.700830388263551e-03, -7.851909450060382e-03, -2.306170883791630e-03, 1.237481603361494e-01, -3.688046308761547e-02, -7.453985361413651e-02, -3.554027310198028e-02, -1.464661894043396e-02, -3.784962256958713e-02, -1.433725823843565e-02, 9.315843053762457e-03, -1.913345889563626e-03, -1.303398396789689e-02, 3.330388708432142e-02, -3.688046308763025e-02, 1.565904011342188e-01, 1.150249667518732e-01, 8.026228557193726e-02, 6.937278891946563e-02,  1.575242596462026e-02, -3.337875996498907e-02, 1.776321795522529e-02, 2.083433232912249e-02, -1.213755065628416e-02,  3.530296522625731e-02, -7.453985361414872e-02, 1.150249667518558e-01, 2.346588263716883e-01, 9.853791068009565e-02,  6.982497683373634e-02, 4.833777490400437e-02, -2.131146741387560e-02, 8.678216315978426e-03, 1.542638125350537e-02, -1.888017320078409e-02, 3.104461234193251e-02, -3.554027310199148e-02, 8.026228557193360e-02, 9.853791068010631e-02,  1.514397666611699e-01, 5.653393291434566e-02, 1.909005454137064e-02, -2.877702949106559e-02, 1.287329142838278e-02,  1.708798680239348e-02, -1.379230639597697e-02, 2.894586069880661e-02, -1.464661894044523e-02, 6.937278891946841e-02,  6.982497683375211e-02, 5.653393291434972e-02, 1.219142939418878e-01, 7.381422590613668e-03, -2.688704556369964e-02,  1.385910242470767e-02, 1.346091688138082e-02, 1.396995566429522e-02, -4.599595839871476e-03, -3.784962256958421e-02,  1.575242596460455e-02, 4.833777490398919e-02, 1.909005454135743e-02, 7.381422590601830e-03, 1.212556683191295e-01,  1.026647893413637e-02, -4.234636902812783e-03, 5.879477479045790e-03, 1.392036888737655e-02, -2.301876478254719e-02, -1.433725823842671e-02, -3.337875996500084e-02, -2.131146741389495e-02, -2.877702949107636e-02, -2.688704556370730e-02,  1.026647893412860e-02, 1.057874011368111e-01, -1.271288958074412e-02, -7.759681279506214e-03, -1.000831559232054e-02,  1.566980567895243e-02, 9.315843053757517e-03, 1.776321795523311e-02, 8.678216315991055e-03, 1.287329142839053e-02,  1.385910242471339e-02, -4.234636902808953e-03, -1.271288958074462e-02, 8.623901792893357e-02, 4.897086092398098e-03, -3.995594050693035e-03, 6.700830388263704e-03, -1.913345889566436e-03, 2.083433232912449e-02, 1.542638125351015e-02,  1.708798680239579e-02, 1.346091688138124e-02, 5.879477479049801e-03, -7.759681279504618e-03, 4.897086092396488e-03,  8.193496156512459e-02 )
               , tolerance = 1e-06)

  expect_equal(c(fit$lag_one_cov),
               c(NULL )
               , tolerance = 1e-06)

  expect_equal(c(fit$fixed_effects),
               c(numeric(0) )
               , tolerance = 1e-06)

  expect_equal(c(fit$n_iter),
               c(19 )
               , tolerance = 1e-06)

  expect_equal(c(fit$Q),
               c( 0.021964034863330231, 0.014316760756824767, 0.063024660014879968, 0.013023668264957084, -0.056719851171780536, -0.008490777919755974, 0.021369191311888087, -0.067589475849311742, -0.054532204665104111, 0.036582603561075740,  0.009200579263664372, 0.014316760756824767, 0.112920387233807512, 0.014478637176649899, 0.272105630469700654,  0.212612476001798389, 0.217670653577722251, 0.227224330663046037, -0.072706595536605578, -0.192733982560595762,  0.126813206830725778, 0.062175392561057870, 0.063024660014879968, 0.014478637176649899, 0.261076048419547180, -0.054786303790622472, -0.320579145543623045, -0.122873446348084975, -0.003733231353859226, -0.251093133003368718, -0.147834566043921367, 0.099585813186606489, 0.010470476160406878, 0.013023668264957084, 0.272105630469700654, -0.054786303790622472, 0.740342065104288460, 0.672372678076431862, 0.616919325106298100, 0.597349584500389330, -0.100021104852921816, -0.451400362536735988, 0.294351610995536805, 0.161641591961863146, -0.056719851171780536,  0.212612476001798389, -0.320579145543623045, 0.672372678076431862, 0.919920225998842489, 0.653101990260101184,  0.504799576492812485, 0.190354151045869113, -0.218346556732260361, 0.138845038868360665, 0.127009585206176057, -0.008490777919755974, 0.217670653577722251, -0.122873446348084975, 0.616919325106298100, 0.653101990260101184,  0.557537701435354305, 0.491129437482700770, -0.002018396264815337, -0.323280536519813588, 0.209919857780930252,  0.136025637605303812, 0.021369191311888087, 0.227224330663046037, -0.003733231353859226, 0.597349584500389330,  0.504799576492812485, 0.491129437482700770, 0.501134843458191481, -0.122914415519579184, -0.398684140576798962,  0.259715474910659716, 0.136409157060167596, -0.067589475849311742, -0.072706595536605578, -0.251093133003368718, -0.100021104852921816, 0.190354151045869113, -0.002018396264815337, -0.122914415519579184, 0.289533938530543045,  0.248817970149291690, -0.165659327227622166, -0.043572117926406570, -0.054532204665104111, -0.192733982560595762, -0.147834566043921367, -0.451400362536735988, -0.218346556732260361, -0.323280536519813588, -0.398684140576798962,  0.248817970149291690, 0.419800467487968731, -0.269806404487484663, -0.114541107670625836, 0.036582603561075740,  0.126813206830725778, 0.099585813186606489, 0.294351610995536805, 0.138845038868360665, 0.209919857780930252,  0.259715474910659716, -0.165659327227622166, -0.269806404487484663, 0.185217445917811918, 0.075259294377460179,  0.009200579263664372, 0.062175392561057870, 0.010470476160406878, 0.161641591961863146, 0.127009585206176057,  0.136025637605303812, 0.136409157060167596, -0.043572117926406570, -0.114541107670625836, 0.075259294377460179,  0.052043980638921440 )
               , tolerance = 1e-06)

  expect_equal(c(fit$Q_0),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-06)

  expect_equal(c(fit$n_risk),
               c(2012, 1938, 1929, 1935, 1824, 1832, 1789, 1707, 1600, 1507 )
               , tolerance = 1e-06)

  expect_equal(c(fit$times),
               c( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 )
               , tolerance = 1e-06)

  expect_equal(c(fit$risk_set),
               c(NULL )
               , tolerance = 1e-06)

  expect_equal(c(fit$data),
               c(NULL )
               , tolerance = 1e-06)

  expect_equal(c(fit$order),
               c(1 )
               , tolerance = 1e-06)

  expect_equal(c(fit$F_),
               c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1 )
               , tolerance = 1e-06)

  expect_equal(c(fit$method),
               c("UKF" )
               , tolerance = 1e-06)

  expect_equal(c(fit$model),
               c("exp_combined" )
               , tolerance = 1e-06)

  expect_equal(c(fit$est_Q_0),
               c(FALSE )
               , tolerance = 1e-06)

  expect_equal(c(fit$LR),
               c(1 )
               , tolerance = 1e-06)
})

# Had issues with win builder. Thus, these lines
cat("\nFinished", test_name, "\n")


