% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/docs_bulk.r
\name{docs_bulk}
\alias{docs_bulk}
\title{Use the bulk API to create, index, update, or delete documents.}
\usage{
docs_bulk(x, index = NULL, type = NULL, chunk_size = 1000,
  doc_ids = NULL, raw = FALSE, ...)
}
\arguments{
\item{x}{A data.frame or path to a file to load in the bulk API}

\item{index}{(character) The index name to use. Required for data.frame input, but
optional for file inputs.}

\item{type}{(character) The type name to use. If left as NULL, will be same name as index.}

\item{chunk_size}{(integer) Size of each chunk. If your data.frame is smaller
thank \code{chunk_size}, this parameter is essentially ignored. We write in chunks because
at some point, depending on size of each document, and Elasticsearch setup, writing a very
large number of documents in one go becomes slow, so chunking can help. This parameter
is ignored if you pass a file name. Default: 1000}

\item{doc_ids}{An optional vector (character or numeric/integer) of document ids to use.
This vector has to equal the size of the documents you are passing in, and will error
if not. If you pass a factor we convert to character. Default: not passed}

\item{raw}{(logical) Get raw JSON back or not.}

\item{...}{Pass on curl options to \code{\link[httr]{POST}}}
}
\description{
Use the bulk API to create, index, update, or delete documents.
}
\details{
More on the Bulk API:
\url{https://www.elastic.co/guide/en/elasticsearch/reference/current/docs-bulk.html}.

This function dispatches on data.frame or character input. Character input has
to be a file name or the function stops with an error message.

If you pass a data.frame to this function, we by default to an index operation,
that is, create the record in the index and type given by those parameters to the
function. Down the road perhaps we will try to support other operations on the
bulk API. if you pass a file, of course in that file, you can specify any
operations you want.

Row names are dropped from data.frame, and top level names for a list are dropped
as well.

A progress bar gives the progress for data.frames and lists
}
\section{Large numbers for document IDs}{

Until recently, if you had very large integers for document IDs, \code{docs_bulk}
failed. It should be fixed now. Let us know if not.
}
\examples{
\dontrun{
plosdat <- system.file("examples", "plos_data.json", package = "elastic")
docs_bulk(plosdat)
aliases_get()
index_delete(index='plos')
aliases_get()

# Curl options
library("httr")
plosdat <- system.file("examples", "plos_data.json", package = "elastic")
docs_bulk(plosdat, config=verbose())

# From a data.frame
docs_bulk(mtcars, index = "hello", type = "world")
docs_bulk(iris, "iris", "flowers")
## type can be missing, but index can not
docs_bulk(iris, "flowers")
## big data.frame, 53K rows, load ggplot2 package first
# res <- docs_bulk(diamonds, "diam")
# Search("diam")$hits$total

# From a list
docs_bulk(apply(iris, 1, as.list), index="iris", type="flowers")
docs_bulk(apply(USArrests, 1, as.list), index="arrests")
# dim_list <- apply(diamonds, 1, as.list)
# out <- docs_bulk(dim_list, index="diamfromlist")

# When using in a loop
## We internally get last _id counter to know where to start on next bulk insert
## but you need to sleep in between docs_bulk calls, longer the bigger the data is
files <- c(system.file("examples", "test1.csv", package = "elastic"),
           system.file("examples", "test2.csv", package = "elastic"),
           system.file("examples", "test3.csv", package = "elastic"))
for (i in seq_along(files)) {
  d <- read.csv(files[[i]])
  docs_bulk(d, index = "testes", type = "docs")
  Sys.sleep(1)
}
count("testes", "docs")
index_delete("testes")

# You can include your own document id numbers
## Either pass in as an argument
index_create("testes")
files <- c(system.file("examples", "test1.csv", package = "elastic"),
           system.file("examples", "test2.csv", package = "elastic"),
           system.file("examples", "test3.csv", package = "elastic"))
tt <- vapply(files, function(z) NROW(read.csv(z)), numeric(1))
ids <- list(1:tt[1],
           (tt[1] + 1):(tt[1] + tt[2]),
           (tt[1] + tt[2] + 1):sum(tt))
for (i in seq_along(files)) {
  d <- read.csv(files[[i]])
  docs_bulk(d, index = "testes", type = "docs", doc_ids = ids[[i]])
}
count("testes", "docs")
index_delete("testes")

## or include in the input data
### from data.frame's
index_create("testes")
files <- c(system.file("examples", "test1_id.csv", package = "elastic"),
           system.file("examples", "test2_id.csv", package = "elastic"),
           system.file("examples", "test3_id.csv", package = "elastic"))
readLines(files[[1]])
for (i in seq_along(files)) {
  d <- read.csv(files[[i]])
  docs_bulk(d, index = "testes", type = "docs")
}
count("testes", "docs")
index_delete("testes")

### from lists via file inputs
index_create("testes")
for (i in seq_along(files)) {
  d <- read.csv(files[[i]])
  d <- apply(d, 1, as.list)
  docs_bulk(d, index = "testes", type = "docs")
}
count("testes", "docs")
index_delete("testes")
}
}

