#' Calculate statistics about nests
#' @title Calculte statistics about nests
#' @author Marc Girondot
#' @return A list with three components: likelihood, summary and metric
#' @param NestsResult A NestsResult file generated by searchR
#' @param parameters A set of parameters if result is not provided.
#' @param SexualisationTRN A set of parameters used to model thermal reaction norm during TSP
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param SE Standard error for each parameter if result is not provided, or replace the one in NestsResult.
#' @param temperatures Timeseries of temperatures. Will replace the one in result.
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings
#' @param weight Weights of the different nests to estimate likelihood
#' @param stopattest True or False, does the plot stops when proxi of size reaches the mean test size.
#' @param M0 Measure of hatchling size proxi at laying date
#' @param series The name or number of the series to be displayed. Only one series can be displayed at a time.
#' @param TSP.borders The limits of TSP
#' @param embryo.stages The embryo stages. At least TSP.borders stages must be provided if show.TSP is TRUE
#' @param sampling Number of randomizations to estimate CI. If 1, no CI is estimated.
#' @param fill Number of minutes between two records. Create new one if they do not exist. NULL does not change the time of temperature recordings.
#' @param out A vector with likelihood and/or metric and/or summary
#' @param progress If FALSE, the progress bar is not shown (useful for using with sweave or knitr)
#' @description Calculate statistics about nests\cr
#' The embryo.stages is a list with stage numbers and relative size as compared to final size at the beginning of the stage.\cr
#' For example embryo.stages=list(number=21:30, size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33) indicates that the stages 21 begins at the relative size of 8.4/39.33.\cr
#' series can be indicated as the name of the series, its number or TRUE or FALSE. "all" indicates that all series must be printed.\cr
#' The summary object is a data.frame composed of these elements:
#' \itemize{
#'   \item \code{TimeWeighted.temperature.mean} Average temperature weigthed by the time at each temperature
#'   \item \code{TimeWeighted.temperature.se} Standard error for the average temperature weigthed by the time at each temperature
#'   \item \code{TSP.TimeWeighted.temperature.mean} Average temperature during the TSP weigthed by the time at each temperature
#'   \item \code{TSP.TimeWeighted.temperature.se} Standard error for the average temperature during the TSP weigthed by the time at each temperature
#'   \item \code{TSP.MassWeighted.temperature.mean} Average temperature during the TSP weigthed by the mass increase at each temperature
#'   \item \code{TSP.MassWeighted.temperature.se} Standard error for the average temperature during the TSP weigthed by the mass increase at each temperature
#'   \item \code{TSP.STRNWeighted.temperature.mean} Averaged temperature during the TSP weighted by sexualisation thermal reaction norm
#'   \item \code{TSP.STRNWeighted.temperature.se} Standard error for temperature during the TSP weighted by sexualisation thermal reaction norm
#'   \item \code{TSP.MassWeighted.STRNWeighted.temperature.mean} Averaged temperature during the TSP weighted by sexualisation thermal reaction norm and embryo growth
#'   \item \code{TSP.MassWeighted.STRNWeighted.temperature.se} Standard error for temperature during the TSP weighted by sexualisation thermal reaction norm and embryo growth
#'   \item \code{TSP.length.mean} Average length of the TSP
#'   \item \code{TSP.length.se} Standard error for length of the TSP
#'   \item \code{TSP.begin.mean} Average time of the beginning of the TSP
#'   \item \code{TSP.begin.se} Standard error for the beginning of the TSP
#'   \item \code{TSP.end.mean} Average time of the endd of the TSP
#'   \item \code{TSP.end.se} Standard error for the end of the TSP
#'   \item \code{incubation.length.mean} Average number of days for the incubation length
#'   \item \code{incubation.length.se} Standard error for number of days for the incubation length
#'   \item \code{incubation.first.third} Average day at which the middle third incubation begins
#'   \item \code{incubation.first.third.se} Standard error for day at which the middle third incubation begins
#'   \item \code{incubation.second.third} Average day at which the middle third incubation ends
#'   \item \code{incubation.second.third.se} Standard error at which the middle third incubation ends
#'   \item \code{MiddleThird.TimeWeighted.temperature.mean} Average temperature during the middle third of incubation
#'   \item \code{MiddleThird.TimeWeighted.temperature.se} Standard error for the temperature during the middle third of incubation
#' }
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p)
#' summary.nests <- info.nests(resultNest_4p, out="summary", sampling=20, 
#'   SE=c(DHA=1.396525, DHH=4.101217, T12H=0.04330405, Rho25=1.00479))
#' # to get info for all nests at the same time, use
#' infoall <- info.nests(resultNest_4p)
#' # it is easier to use after dataframe conversion
#' infoall.df <- info.nests(resultNest_4p, out="metric", sampling=20, 
#'   SE=c(DHA=1.396525, DHH=4.101217, T12H=0.04330405, Rho25=1.00479))
#' }
#' @export


  info.nests <- function(NestsResult=NULL, parameters=NULL, fixed.parameters=NULL, 
                         SE=NULL, temperatures=NULL, derivate=NULL, 
                         test=NULL, stopattest=FALSE, M0=NULL, series="all",
                         TSP.borders=c("21", "26"), embryo.stages="Caretta caretta",
                         sampling=1, weight=NULL, out="Likelihood", fill=NULL, 
                         SexualisationTRN=NULL, 
                         progress=FALSE) {
    # NestsResult=NULL; parameters=NULL; fixed.parameters=NULL;  SE=NULL; temperatures=NULL; derivate=NULL;  test=NULL; stopattest=FALSE; M0=NULL; series="all"; TSP.borders=c("21", "26"); embryo.stages="Caretta caretta"; sampling=1; weight=NULL; out="Likelihood"; fill=NULL;  SexualisationTRN=NULL;  progress=FALSE
    
    # Dans la moyenne, les temps de endTSP et beginTSP ne sont pas bons
    # je n'ai pas la moyenne de endTSP et BeginTSP
    # Transition mal géré
    
    # fill est soit NULL soit c'est le nombre de minutes entre deux temps
    
    # NestsResult <- resultNest_4p
    # fonctionfit(NestsResult=resultNest_4p$par, temperatures=resultNest_4p$data, derivate=resultNest_4p$derivate, weight=resultNest_4p$weight, test=resultNest_4p$test, M0=resultNest_4p$M0, fixed.parameters=resultNest_4p$fixed.parameters)
    # outdf <- fonctionfit(NestsResult=resultNest_4p$par, temperatures=resultNest_4p$data, derivate=resultNest_4p$derivate, weight=resultNest_4p$weight, test=resultNest_4p$test, M0=resultNest_4p$M0, fixed.parameters=resultNest_4p$fixed.parameters, out="metric")
    # fonctionfit(NestsResult=resultNest_4p$par, temperatures=resultNest_4p$data, derivate=resultNest_4p$derivate, weight=resultNest_4p$weight, test=resultNest_4p$test, M0=resultNest_4p$M0, fixed.parameters=resultNest_4p$fixed.parameters, out="temperature.mean")
    # fonctionfit(NestsResult=resultNest_4p$par, temperatures=resultNest_4p$data, derivate=resultNest_4p$derivate, weight=resultNest_4p$weight, test=resultNest_4p$test, M0=resultNest_4p$M0, fixed.parameters=resultNest_4p$fixed.parameters, out="temperature.TSP.mean")
    # fonctionfit(NestsResult=resultNest_4p$par, temperatures=resultNest_4p$data, derivate=resultNest_4p$derivate, weight=resultNest_4p$weight, test=resultNest_4p$test, M0=resultNest_4p$M0, fixed.parameters=resultNest_4p$fixed.parameters, out="weighted.temperature.mean")
    # Lout <- embryogrowth:::.fonctionfit(NestsResult=resultNest_4p)
    # outdf <- embryogrowth:::.fonctionfit(NestsResult=resultNest_4p, out="metric", SE=c(DHA=1.396525, DHH=4.101217, T12H=0.04330405, Rho25=1.00479))
    # sumdf <- embryogrowth:::.fonctionfit(NestsResult=resultNest_4p, out="summary", SE=c(DHA=1.396525, DHH=4.101217, T12H=0.04330405, Rho25=1.00479))
    # outdf <- embryogrowth:::.fonctionfit(NestsResult=resultNest_4p, out="metric", stopattest=TRUE)
    # outdf <- embryogrowth:::.fonctionfit(NestsResult=resultNest_4p, series=c(TRUE, rep(FALSE, 20)), out="metric", SE=c(DHA=1.396525, DHH=4.101217, T12H=0.04330405, Rho25=1.00479))
    
    if (is.null(SE) | out=="Likelihood") sampling <- 1
    
    if (class(NestsResult)=="NestsResult") {
      # temperatures est un objet Nests
      if (is.null(temperatures)) temperatures <- NestsResult$data
      if (is.null(derivate)) derivate <- NestsResult$derivate
      if (is.null(weight)) weight <- NestsResult$weight
      if (is.null(test)) test <- NestsResult$test
      if (is.null(M0)) M0 <- NestsResult$M0
      if (is.null(fixed.parameters)) fixed.parameters <- NestsResult$fixed.parameters
      if (is.null(SE)) SE <- NestsResult$SE
      if (is.null(parameters)) {x <- NestsResult$par} else {x <- parameters}
    }
    
    NBTs <- temperatures[["IndiceT"]][3]
    if (is.null(series) | (series=="all")) series <- rep(TRUE, NBTs)
    
    if (embryo.stages[1]=="Caretta caretta") {
      embryo.stages <- c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33
      names(embryo.stages) <- c(21:30)
    }
    
    size.begin.TSP <- unname(embryo.stages[TSP.borders[1]])
    size.end.TSP <- unname(embryo.stages[TSP.borders[2]])
    
    # dans x j'ai les paramètres à ajuster
    # Il faut que je rajoute les fixe - 16/7/2012
    x <- c(x, fixed.parameters)
    # je génère les se
    df_random <- data.frame(fake=rep(NA, sampling))
    
    if (is.null(SE)) {
      SE <- x
      SE[] <- NA
    }
    
    for (i in 1:length(x)) {
      nm <- names(x)[i]
      if (is.na(SE[nm])) {
        df_random <- cbind(df_random, rep(x[i], sampling))
      } else {
        df_random <- cbind(df_random, c(x[i], rnorm(sampling-1, x[i], SE[nm])))
      }
    }
    
    df_random <- df_random[,-1]
    colnames(df_random) <- names(x)
    
    returntotal <- list()
    
    for(sp in 1:sampling) {
      x <- as.numeric(df_random[sp, ])
      names(x) <- colnames(df_random)
      
      # Ce sont toutes les températures de tous les nids
      tempK <- as.numeric(temperatures[["Temperatures"]])
      # rlist <- embryogrowth:::.SSM(tempK, x)
      rlist <- .SSM(tempK, x)
      r <- rlist[[1]]
      r_L <- rlist[[2]]
      
      # j'utilise les indices, plus besoin
      # names(r)<-tempK_s
      # names(r_L)<-tempK_s
      
      if (!is.na(x["K"])) {
        Kval <- unname(x["K"])
      } else {
        Kval <- NULL
      }
      
      temperatures_ec <- temperatures[1:NBTs][series]
      transition_P <- x["transition_P"]
      transition_S <- x["transition_S"]
      logicTransition <- (is.na(transition_P) | is.na(transition_S))
      
      AnalyseTraces <- mclapply(temperatures_ec, function (tec) {
        namets <- names(temperatures_ec)[substitute(tec)[[3]]]
        # tec <- temperatures_ec[[1]]
        # namets <- names(temperatures_ec)[1]
        meanSCL <- as.numeric(test[namets, "Mean"])
        sdSCL <- as.numeric(test[namets, "SD"])
        if (!is.na(x["rK"])) {
          Kval <- unname(x["rK"]*meanSCL)
        }
        y <- M0
        tmin <- tec[,"Time"]
        ldeltaT0 <- length(tmin)
        if (!is.null(fill)) 
          tmin <- c(tmin, seq(from=fill, to=tail(tmin, n=1L), by=fill))
        ldeltaT <- length(tmin)-ldeltaT0
        
        df <- data.frame(Time=tmin, R=rep(NA, length(tmin)), 
                         SCL=c(M0, rep(NA, length(tmin)-1)), 
                         TempC=c(tec[, "Temperatures C"], rep(NA, ldeltaT)),
                         TempK=c(tec[, "Temperatures K"], rep(NA, ldeltaT)),
                         IndiceK=c(tec[, "IndiceK"], rep(NA, ldeltaT)),
                         DeltaT=c(diff(tmin), 0))
        
        # j'ai la fonction fill
        if (!is.null(fill)) {
          od <- order(tmin)
          df <- df[od,]
          # du <- duplicated(df$Time)
          
          ldt <- which(!is.na(df$TempC))
          
          out <- sapply(ldt[-1], function(x) {
            if (df[x, "Time"]==df[x-1, "Time"]) {
              df[x, "TempC"] <- df[x-1, "TempC"] 
              df[x, "TempC"] <- NA
              df[x, "TempK"] <- df[x-1, "TempK"] 
              df[x, "TempK"] <- NA
              df[x, "IndiceK"] <- df[x-1, "IndiceK"] 
              df[x, "IndiceK"] <- NA
            }
          })
        }
        
        tmin <- df$Time
        df$DeltaT <- c(diff(tmin), 0)
        ldt <- which(!is.na(df$TempC))
        #  tpk <- as.character(df$TempK)
        IndK <- df$IndiceK
        
        if (!stopattest) {
          if (logicTransition) {  
            for (i in 1:(length(ldt)-1)) {
              y <- df[ldt[i], "SCL"]
              range <- c(ldt[i]:ldt[i+1])
              timesunique <- tmin[range]
              a <- r[IndK[i]]
              param <- c(alpha=unname(a), K=Kval)
              out1 <- lsoda(y, timesunique, derivate, param)
              df[c(ldt[i]:(ldt[i+1]-1)), "R"] <- a
              df[range, "SCL"] <- out1[,2]
            }
          } else {
            for (i in 1:(length(tmin)-1)) {
              y <- df[i, "SCL"]
              timesunique <- c(tmin[i], tmin[i+1])
              transition <- 1/(1+exp(transition_S*(y-transition_P)))
              if (!is.na(IndK[i])) tk <- IndK[i]
              a <- r[tk]*transition+r_L[tk]*(1-transition)
              param <- c(alpha=unname(a), K=Kval)
              out1 <- lsoda(y, timesunique, derivate, param)
              df[i, "R"] <- a
              df[i+1, "SCL"] <- as.numeric(out1[2,2])
            }
          }
          # je suis en stopattest
        } else {
          if (logicTransition) {  
            for (i in 1:(length(ldt)-1)) {
              y <- df[ldt[i], "SCL"]
              range <- c(ldt[i]:ldt[i+1])
              timesunique <- tmin[range]
              a <- r[IndK[i]]
              param <- c(alpha=unname(a), K=Kval)
              out1 <- lsoda(y, timesunique, derivate, param)
              df[c(ldt[i]:(ldt[i+1]-1)), "R"] <- a
              df[range, "SCL"] <- out1[,2]
              if (y>meanSCL) {
                df <- df[1:i, ]
                break
              }
            }
          } else {
            for (i in 1:(length(tmin)-1)) {
              y <- df[i, "SCL"]
              timesunique <- c(tmin[i], tmin[i+1])
              transition <- 1/(1+exp(transition_S*(y-transition_P)))
              if (!is.na(IndK[i])) tk <- IndK[i]
              a <- r[tk]*transition+r_L[tk]*(1-transition)
              param <- c(alpha=unname(a), K=Kval)
              out1 <- lsoda(y, timesunique, derivate, param)
              y <- as.numeric(out1[2,2])
              df[i, "R"] <- a
              df[i+1, "SCL"] <- y
              if (y>meanSCL) {
                df <- df[1:i, ]
                break
              }
            }
          }
        }
        
        
        #  names(L) <- namets
        if (out=="Likelihood") {
          # dans y je n'ai pas un vecteur avec les tailles à la fin !
          return(-dnorm(y, mean=meanSCL, sd=sdSCL, log=TRUE))
        }
        
        if (any(out=="metric") | any(out=="summary")) { 
          # si metric ou summary
          # les valeurs sont fausses pour transition
          
          # indices entre la TSP
          indice.begin.tsp <- which(df[,"SCL"]>(size.begin.TSP*meanSCL))[1]-1
          # la vrai TSP commence entre indice.begin.tsp et indice.begin.tsp+1
          timei1 <- df[indice.begin.tsp,"Time"]+df[indice.begin.tsp,"DeltaT"]*1/3
          timei2 <- df[indice.begin.tsp,"Time"]+df[indice.begin.tsp,"DeltaT"]*2/3
          timesunique <- c(df[indice.begin.tsp, "Time"], timei1, timei2, df[indice.begin.tsp+1, "Time"])
          # si je suis en transition, je dois recalculer R
          a <- df[indice.begin.tsp, "R"]
          param <- c(alpha=unname(a), K=Kval)
          y <- df[indice.begin.tsp, "SCL"]
          names(y) <- "SCL"
          dfpol <- lsoda(y, timesunique, derivate, param)
          
          o <- lm(SCL ~ poly(time, degree=2, raw=T), data=dfpol)
          ocoef <- dummy.coef(o)
          # je cherche x pour lequel j'ai y=size.begin.TSP*meanSCL
          # ocoef[[3]]*x^2 + ocoef[[2]]*x + ocoef[[1]] = size.begin.TSP*meanSCL
          # ocoef[[3]]*x^2 + ocoef[[2]]*x + (ocoef[[1]] - size.begin.TSP*meanSCL) = 0
          a <- ocoef[[3]]
          b <- ocoef[[2]]
          c <- ocoef[[1]] - size.begin.TSP*meanSCL
          delta <- b^2-4*a*c
          x1 <- (-b+sqrt(delta))/(2*a)
          x2 <- (-b-sqrt(delta))/(2*a)
          time.begin.TSP <- unname(ifelse(x1<timei1 & x1>timei2, x2, x1))
          
          indice.end.tsp <- which(df[,"SCL"]>(size.end.TSP*meanSCL))[1]-1
          # la vrai TSP fini entre indice.end.tsp et indice.end.tsp+1
          timei1 <- df[indice.end.tsp,"Time"]+df[indice.end.tsp,"DeltaT"]*1/3
          timei2 <- df[indice.end.tsp,"Time"]+df[indice.end.tsp,"DeltaT"]*2/3
          timesunique <- c(df[indice.end.tsp, "Time"], timei1, timei2, df[indice.end.tsp+1, "Time"])
          # si je suis en transition, je dois recalculer R
          a <- df[indice.end.tsp, "R"]
          param <- c(alpha=unname(a), K=Kval)
          y <- df[indice.end.tsp, "SCL"]
          names(y) <- "SCL"
          dfpol <- lsoda(y, timesunique, derivate, param)
          
          o <- lm(SCL ~ poly(time, degree=2, raw=T), data=dfpol)
          ocoef <- dummy.coef(o)
          # je cherche x pour lequel j'ai y=size.begin.TSP*meanSCL
          # ocoef[[3]]*x^2 + ocoef[[2]]*x + ocoef[[1]] = size.begin.TSP*meanSCL
          # ocoef[[3]]*x^2 + ocoef[[2]]*x + (ocoef[[1]] - size.begin.TSP*meanSCL) = 0
          a <- ocoef[[3]]
          b <- ocoef[[2]]
          c <- ocoef[[1]] - size.end.TSP*meanSCL
          delta <- b^2-4*a*c
          x1 <- (-b+sqrt(delta))/(2*a)
          x2 <- (-b-sqrt(delta))/(2*a)
          time.end.TSP <- unname(ifelse(x1<timei1 & x1>timei2, x2, x1))
          
          # C'est nécessaire pour estimer les températures moyennes
          # on doit les garder
          
          df <- rbind(df[1:indice.begin.tsp,], 
                      c(Time=time.begin.TSP, 
                        R=df[indice.begin.tsp,"R"], 
                        SCL=size.begin.TSP*meanSCL, 
                        TempC=df[indice.begin.tsp,"TempC"], 
                        TempK=df[indice.begin.tsp,"TempK"], 
                        IndiceK=df[indice.begin.tsp,"IndiceK"], 
                        DeltaT=time.begin.TSP-df[indice.begin.tsp,"Time"]),
                      df[(indice.begin.tsp+1):indice.end.tsp,],
                      c(Time=time.end.TSP, 
                        R=df[indice.end.tsp,"R"], 
                        SCL=size.end.TSP*meanSCL, 
                        TempC=df[indice.end.tsp,"TempC"], 
                        TempK=df[indice.end.tsp,"TempK"], 
                        IndiceK=df[indice.end.tsp,"IndiceK"], 
                        DeltaT=time.end.TSP-df[indice.end.tsp,"Time"]),
                      df[(indice.end.tsp+1):nrow(df),])
          # il faut que je modifie encore deux DeltaT
          df[(indice.begin.tsp+2), "DeltaT"] <- df[(indice.begin.tsp+2), "Time"]-df[(indice.begin.tsp+1), "Time"]
          df[(indice.end.tsp+3), "DeltaT"] <- df[(indice.end.tsp+3), "Time"]-df[(indice.end.tsp+2), "Time"] 
          
          # si je suis en stopattest, il faut que je calcule la vraie valeur de fin
          if (stopattest) {
            indice.fin.incubation <- nrow(df)-1
            timei1 <- df[indice.fin.incubation,"Time"]+df[indice.fin.incubation,"DeltaT"]*1/3
            timei2 <- df[indice.fin.incubation,"Time"]+df[indice.fin.incubation,"DeltaT"]*2/3
            timesunique <- c(df[indice.fin.incubation, "Time"], timei1, timei2, df[indice.fin.incubation+1, "Time"])
            # si je suis en transition, je dois recalculer R
            a <- df[indice.fin.incubation, "R"]
            param <- c(alpha=unname(a), K=Kval)
            y <- df[indice.fin.incubation, "SCL"]
            names(y) <- "SCL"
            dfpol <- lsoda(y, timesunique, derivate, param)
            
            o <- lm(SCL ~ poly(time, degree=2, raw=T), data=dfpol)
            ocoef <- dummy.coef(o)
            # je cherche x pour lequel j'ai y=size.begin.TSP*meanSCL
            # ocoef[[3]]*x^2 + ocoef[[2]]*x + ocoef[[1]] = size.begin.TSP*meanSCL
            # ocoef[[3]]*x^2 + ocoef[[2]]*x + (ocoef[[1]] - size.begin.TSP*meanSCL) = 0
            a <- ocoef[[3]]
            b <- ocoef[[2]]
            c <- ocoef[[1]] - meanSCL
            delta <- b^2-4*a*c
            x1 <- (-b+sqrt(delta))/(2*a)
            x2 <- (-b-sqrt(delta))/(2*a)
            time.end.incubation <- unname(ifelse(x1<timei1 & x1>timei2, x2, x1))
            # Maintenant je tronque df de façon à ne garder que jusqu'à time.end.incubation
            # garde
            df <- df[df$SCL<meanSCL,]
            df <- rbind(df, c(Time=time.end.incubation, 
                              R=NA, SCL=meanSCL, 
                              TempC=NA, tempK=NA, IndiceK=NA, DeltaT=NA))
          }
          
          rownames(df) <- 1:nrow(df)
          attributes(df)$indice.end.tsp <- indice.end.tsp+2
          attributes(df)$time.end.tsp <- time.end.TSP
          attributes(df)$indice.begin.tsp <- indice.begin.tsp+1
          attributes(df)$time.begin.tsp <- time.begin.TSP
          return(df)
          # fin de si metric ou summary
        }
        # fin du mcapply
      })
      
      if (out=="Likelihood") {
        L <- unlist(AnalyseTraces)
        # dans L j'ai un vecteur avec le nom
        # il faut que j'applique le weight
        if (!is.null(weight))	L <- L*weight[match(names(L), names(weight))]
        returntotal <- (sum(L))
      }
      
      if (out=="metric") {
        # J'ai un retour de df
        if (!is.null(fill)) {
          # je dois intégrer des valeurs
          
          AnalyseTraces2 <- mclapply(AnalyseTraces, function(df) {
            tt <- seq(from=0, to=df[nrow(df), "Time"], by=fill)
          })
        }
        returntotal <- c(returntotal, list(AnalyseTraces))
      }
      
      if (out=="summary") {
        TimeWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, function(x) sum(x[, "TempC"]*x[, "DeltaT"])/sum(x[, "DeltaT"])))
        TSP.TimeWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                           function(x) {
                                                             x2 <- x[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp-1), ]
                                                             sum(x2[, "TempC"]*x2[, "DeltaT"])/sum(x2[, "DeltaT"])
                                                           }))
        TSP.MassWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                           function(x) {
                                                             x2 <- x[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp), ]
                                                             dSCL <- c(diff(x2[, "SCL"]), 0)
                                                             sum(x2[, "TempC"]*dSCL)/sum(dSCL)
                                                           }))
        if (is.null(SexualisationTRN)) {
          TSP.STRNWeighted.temperature.mean <- rep(NA, length(AnalyseTraces))
          TSP.MassWeighted.STRNWeighted.temperature.mean <- rep(NA, length(AnalyseTraces))
        } else {
          TSP.STRNWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                             function(x) {
                                                               # dans x[, "TempC"] j'ai les températures en Celsius
                                                               x2 <- x[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp-1), ]
                                                               # STRN_model <- embryogrowth:::.SSM(x2[, "TempK"], SexualisationTRN)[[1]]
                                                               STRN_model <- .SSM(x2[, "TempK"], SexualisationTRN)[[1]]
                                                               sum(x2[, "TempC"]*x2[, "DeltaT"]*STRN_model)/sum(x2[, "DeltaT"]*STRN_model)
                                                             }))
          
          TSP.MassWeighted.STRNWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                                          function(x) {
                                                                            # dans x[, "TempC"] j'ai les températures en Celsius
                                                                            x2 <- x[(attributes(x)$indice.begin.tsp):(attributes(x)$indice.end.tsp), ]
                                                                            # STRN_model <- embryogrowth:::.SSM(x2[, "TempK"], SexualisationTRN)[[1]]
                                                                            STRN_model <- .SSM(x2[, "TempK"], SexualisationTRN)[[1]]
                                                                            dSCL <- c(diff(x2[, "SCL"]), 0)
                                                                            sum(x2[, "TempC"]*dSCL*STRN_model)/sum(dSCL*STRN_model)
                                                                          }))
        }
        
        TSP.length.mean <- unlist(lapply(AnalyseTraces, 
                                         function(x) {
                                           t1 <- attributes(x)$time.begin.tsp
                                           t2 <- attributes(x)$time.end.tsp
                                           return(t2-t1)
                                         }))
        TSP.begin.mean <- unlist(lapply(AnalyseTraces, 
                                        function(x) {
                                          return(attributes(x)$time.begin.tsp)
                                        }))
        TSP.end.mean <- unlist(lapply(AnalyseTraces, 
                                      function(x) {
                                        return(attributes(x)$time.end.tsp)
                                      }))
        incubation.length.mean <- unlist(lapply(AnalyseTraces, 
                                                function(x) {
                                                  return(x[nrow(x), "Time"])
                                                }))
        incubation.first.third <- unlist(lapply(AnalyseTraces, 
                                                function(x) {
                                                  return(x[nrow(x), "Time"]*1/3)
                                                }))
        incubation.second.third <- unlist(lapply(AnalyseTraces, 
                                                 function(x) {
                                                   return(x[nrow(x), "Time"]*2/3)
                                                 }))
        MiddleThird.TimeWeighted.temperature.mean <- unlist(lapply(AnalyseTraces, 
                                                                   function(x) {
                                                                     tt <- c(x[, "Time"],x[nrow(x), "Time"]*1/3, x[nrow(x), "Time"]*2/3)
                                                                     ttemp <- c(x[, "TempC"],NA, NA)
                                                                     ott <- order(tt)
                                                                     ttemp <- ttemp[ott]
                                                                     tt <- tt[ott]
                                                                     pos <- which(is.na(ttemp))
                                                                     ttemp[pos] <- ttemp[pos-1]
                                                                     return(sum(ttemp[pos[1]:(pos[2]-1)]*diff(tt)[pos[1]:(pos[2]-1)])/sum(diff(tt)[pos[1]:(pos[2]-1)]))
                                                                   }))
        
        summarydf <- data.frame(
          TimeWeighted.temperature.mean=TimeWeighted.temperature.mean,
          TSP.TimeWeighted.temperature.mean=TSP.TimeWeighted.temperature.mean,
          TSP.MassWeighted.temperature.mean=TSP.MassWeighted.temperature.mean,
          TSP.STRNWeighted.temperature.mean=TSP.STRNWeighted.temperature.mean,
          TSP.MassWeighted.STRNWeighted.temperature.mean=TSP.MassWeighted.STRNWeighted.temperature.mean,
          TSP.length.mean=TSP.length.mean,
          TSP.begin.mean=TSP.begin.mean,
          TSP.end.mean=TSP.end.mean,
          incubation.length.mean=incubation.length.mean,
          incubation.first.third=incubation.first.third,
          incubation.second.third=incubation.second.third,
          MiddleThird.TimeWeighted.temperature.mean=MiddleThird.TimeWeighted.temperature.mean
        )
        returntotal <- c(returntotal, list(summarydf))
        
      }
      
      # fin de la boucle des réplicats
    }
    
    if (out=="Likelihood") return(returntotal)
    
    if (out=="metric") {
      
      # il y a un problème sur les temps créés pour début et fin de la TSP
      
      meanTotal <- lapply(returntotal[[1]], function(x) x[,"SCL"])
      meanTotal2 <- lapply(returntotal[[1]], function(x) x[,"SCL"]^2)
      meanRTotal <- lapply(returntotal[[1]], function(x) x[,"R"])
      meanRTotal2 <- lapply(returntotal[[1]], function(x) x[,"R"]^2)
      
      if (length(returntotal) != 1) {
        for (i in 2:length(returntotal)) {
          meanTotal_i <- lapply(returntotal[[i]], function(x) x[,"SCL"])
          meanTotal2_i <- lapply(returntotal[[i]], function(x) x[,"SCL"]^2)
          meanRTotal_i <- lapply(returntotal[[i]], function(x) x[,"R"])
          meanRTotal2_i <- lapply(returntotal[[i]], function(x) x[,"R"]^2)
          
          for(j in 1:length(meanTotal_i)) {
            meanTotal[[j]] <- meanTotal[[j]]+meanTotal_i[[j]]
            meanTotal2[[j]] <- meanTotal2[[j]]+meanTotal2_i[[j]]
            meanRTotal[[j]] <- meanRTotal[[j]]+meanRTotal_i[[j]]
            meanRTotal2[[j]] <- meanRTotal2[[j]]+meanRTotal2_i[[j]]
            
          }
        }
      }
      
      ret <- returntotal[[1]]
      
      # Je supprime les temps créés pour le début et la fin de la TSP
      ret2 <- lapply(ret, function(x) 
        x[-c(attributes(x)$indice.begin.tsp, attributes(x)$indice.end.tsp), ])
      # Je rétablis les deltaT
      for(j in 1:length(ret2)) ret2[[j]][,"DeltaT"] <- c(diff(ret2[[j]][,"Time"]), 0)
      
      
      # > attributes(returntotal[[1]][[1]])$indice.begin.tsp
      # [1] 44
      # > attributes(returntotal[[1]][[1]])$indice.end.tsp
      # [1] 66
      
      # sum(a^2)-length(a)*mean(a)^2)/(length(a)-1)
      for(j in 1:length(meanTotal)) {
        se_i <- (meanTotal2[[j]]-sampling*(meanTotal[[j]]/sampling)^2)/(sampling-1)
        se_i <- sqrt(ifelse(se_i<=0, 0,se_i))
        seR_i <- (meanRTotal2[[j]]-sampling*(meanRTotal[[j]]/sampling)^2)/(sampling-1)
        seR_i <- sqrt(ifelse(seR_i<=0, 0,seR_i))
        
        ret[[j]] <- cbind(ret[[j]], mean.SCL=meanTotal[[j]]/sampling, 
                          mean.R=meanRTotal[[j]]/sampling, 
                          se.SCL=se_i, se.R=seR_i)
      }
      return(ret)
    }
    
    
    if (out=="summary") {
      
      
      if (sampling != 1) {
        meanTotal2 <- returntotal[[1]]^2
        meanTotal <- returntotal[[1]]
        
        for (i in 2:length(returntotal)) {
          meanTotal <- meanTotal+returntotal[[i]]
          meanTotal2 <- meanTotal2+returntotal[[i]]^2
        }
        
        meanTotal <- meanTotal/sampling
        df_se <- (meanTotal2-sampling*meanTotal^2)/(sampling-1)
        df_se <- sqrt(ifelse(df_se<1E-9, 0,df_se))
        
        
        colnames(df_se) <- paste0(gsub("(.+)\\.mean", "\\1", colnames(df_se)), ".se")
        return(cbind(meanTotal, df_se))
      } else {
        
        df_se <- as.matrix(returntotal[[1]])
        df_se[, ] <- NA
        df_se <- as.data.frame(df_se)
        colnames(df_se) <- paste0(gsub("(.+)\\.mean", "\\1", colnames(df_se)), ".se")
        
        return(cbind(returntotal[[1]], df_se))
      }
    }
    
  }
