% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make_epiflows.R
\name{make_epiflows}
\alias{make_epiflows}
\alias{make_epiflows.data.frame}
\alias{make_epiflows.integer}
\alias{make_epiflows.numeric}
\title{Create an epiflows object}
\usage{
make_epiflows(...)

\method{make_epiflows}{data.frame}(flows, locations = NULL, from = 1L,
  to = 2L, n = 3L, id = 1L, ...)

\method{make_epiflows}{integer}(inflow, outflow, focus, locations,
  id = 1L, ...)

\method{make_epiflows}{numeric}(inflow, outflow, focus, locations,
  id = 1L, ...)
}
\arguments{
\item{...}{Any number of varaibles that can be used for mapping or modelling.
See \code{\link[=global_vars]{global_vars()}} and \code{\link[=get_vars]{get_vars()}} for details.}

\item{flows}{a data frame where each row represents a flow from one location
to the next. This must have at least three columns:
\itemize{
\item Where the flow started (as specified in \code{from}, below)
\item Where the flow ended (as specified in \code{to}, below)
\item How many cases were involved (as specified in \code{n}, below)
}}

\item{locations}{a data frame where each row represents a location. This can
have any number of columns specifying useful metadata about the location,
but it must contain at least one column specifying the location ID used in
the \code{flows} data frame (as specified by the \code{id} argument, below).}

\item{from}{the column in the \code{flows} data frame indicating where the flow
started. This can be an integer or character. Default is the first column.}

\item{to}{the column in the \code{flows} data frame indicating where the flow
terminated. This can be an integer or character. Default is the second column.}

\item{n}{the column in the \code{flows} data frame indicating how many cases were
contained in the flow. This can be an integer or character. Default is the
third column.}

\item{id}{The column to use for the identifier in the \code{locations} data frame.
This defaults to the first column.}

\item{inflow}{a \strong{named} integer or numeric vector specifying the number of
cases flowing into a location specified by \code{focus}.}

\item{outflow}{a \strong{named} integer or numeric vector specifying the number of
cases flowing from a location specified by \code{focus}.}

\item{focus}{a character vector specifying the focal location for integer
input. This is necessary for integer input to make clear what "outflow" and
"inflow" are relative to.}
}
\value{
An \code{epiflows} object in list format with four elements:
\itemize{
\item \strong{locations} (accessible via \code{\link[=get_locations]{get_locations()}}): a data frame of
locations with first column 'id' containing character vector of unique
identifiers.
\item \strong{flows} (accessible via \code{\link[=get_flows]{get_flows()}}): data.frame of flows with first
two columns named 'from' and 'to' indicating directed flows between two
locations, and a third column named 'n', specifying the number of cases in
each
\item \strong{vars} (accessible via \code{\link[=get_vars]{get_vars()}}). This contains a named list of
available variables that can be used in further plotting and/or modelling.
Default variables are found in \code{\link[=global_vars]{global_vars()}}:
\itemize{
\item \code{coordinates}: two columns specifying the lon and lat coordinates
\item \code{pop_size}: population size of each location
\item \code{duration_stay}: the average duration of stay for each location
\item \code{first_date}: the date of first recorded case
\item \code{last_date}: the date of the last recorded case
\item \code{num_cases}: the number of cases between the first and last date
}
}
}
\description{
An epiflows object contains a pair of data frames that provide information
about locations and flows between locations.
}
\details{
The \code{epiflows} object can be constructed using simply a list of locations with
optional metadata (similar to a linelist) and a list of flows that describes
the number of cases flowing from one location to another. Optional metadata
such as coordinates and duration of stay can be included in the linelist for
use in \code{\link[=estimate_risk_spread]{estimate_risk_spread()}} or \code{\link[=map_epiflows]{map_epiflows()}}.

\subsection{Developer note: object structure}{

Because a flow of cases from one location to another can be thought of as a
contact with a wider scope, the \code{epiflows} object inherits from the
\code{epicontacts} object, constructed via \code{\link[epicontacts:make_epicontacts]{epicontacts::make_epicontacts()}}.
This means that all the methods for subsetting an object of class
\code{epicontacts} also applies to \code{epiflows}, including the use of the function
\code{\link[epicontacts:thin]{epicontacts::thin()}}. One caveat is that, internally, the names of the
elements within the object do not match the terminology used in \emph{epiflows}.

}
}
\examples{
## Load data
data(YF_flows)
data(YF_locations)
YF_flows
YF_locations
## Use both to create the epiflows object.
ef <- make_epiflows(flows         = YF_flows, 
                    locations     = YF_locations, 
                    pop_size      = "location_population",
                    duration_stay = "length_of_stay",
                    num_cases     = "num_cases_time_window",
                    first_date    = "first_date_cases",
                    last_date     = "last_date_cases"
                   )
ef
# Access variable information
get_pop_size(ef)
get_vars(ef, "duration_stay")
get_vars(ef, "num_cases")
data(YF_Brazil)
(inflows   <- YF_Brazil$T_O["Espirito Santo", , drop = TRUE])
(outflows  <- YF_Brazil$T_D["Espirito Santo", , drop = TRUE])
(locations <- subset(YF_Brazil$states, location_code == "Espirito Santo", drop = FALSE))
los <- data.frame(location_code    = names(YF_Brazil$length_of_stay), 
                  length_of_stay   = YF_Brazil$length_of_stay,
                  stringsAsFactors = FALSE
                 )
locations <- merge(x   = locations, 
                   y   = los, 
                   by  = "location_code", 
                   all = TRUE)
ef <- make_epiflows(inflow = inflows, 
                    outflow = outflows, 
                    focus = "Espirito Santo", 
                    locations = locations,
                    pop_size = "location_population",
                    duration_stay = "length_of_stay",
                    num_cases = "num_cases_time_window",
                    first_date = "first_date_cases",
                    last_date = "last_date_cases"
                   )
ef
}
\seealso{
\code{\link[=global_vars]{global_vars()}} for definitions of global variables,
\code{\link[=estimate_risk_spread]{estimate_risk_spread()}} for modelling, \code{\link[=plot.epiflows]{plot.epiflows()}} for plotting,
\code{\link[=add_coordinates]{add_coordinates()}} for adding coordinates, \code{\link[=get_vars]{get_vars()}} for accession of
metadata, \code{\link[=get_locations]{get_locations()}} to access the locations data frame,
\code{\link[=get_flows]{get_flows()}} to access the flows data frame.
}
\author{
Zhian Kamvar, Thibaut Jombart
}
