% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ergmMPLE.R
\name{ergmMPLE}
\alias{ergmMPLE}
\title{ERGM Predictors and response for logistic regression calculation of MPLE}
\usage{
ergmMPLE(
  formula,
  constraints = ~.,
  obs.constraints = ~-observed,
  output = c("matrix", "array", "dyadlist", "fit"),
  expand.bipartite = FALSE,
  control = control.ergm(),
  verbose = FALSE,
  ...,
  basis = ergm.getnetwork(formula)
)
}
\arguments{
\item{formula, constraints, obs.constraints}{An ERGM formula and
(optionally) a constraint specification formulas. See
\code{\link[=ergm]{ergm()}}. This function supports only dyad-independent
constraints.}

\item{output}{Character, partially matched. See Value.}

\item{expand.bipartite}{Logical. Specifies whether the output matrices (or array slices) representing dyads for bipartite networks are represented as rectangular matrices with first mode vertices in rows and second mode in columns, or as square matrices with dimension equalling the total number of vertices, containing with structural \code{NA}s or 0s within each mode.}

\item{control}{A list of control parameters for algorithm tuning,
typically constructed with \code{\link[=control.ergm]{control.ergm()}}. Its documentation
gives the the list of recognized control parameters and their
meaning. The more generic utility \code{\link[=snctrl]{snctrl()}} (StatNet ConTRoL)
also provides argument completion for the available control
functions and limited argument name checking.}

\item{verbose}{A logical or an integer to control the amount of
progress and diagnostic information to be printed. \code{FALSE}/\code{0}
produces minimal output, with higher values producing more
detail. Note that very high values (5+) may significantly slow
down processing.}

\item{\dots}{Additional arguments, to be passed to lower-level functions.}

\item{basis}{a value (usually a \code{\link[network:network]{network}}) to override the LHS of the formula.}
}
\value{
If \code{output=="matrix"} (the default), then only the response, predictor,
and weights are returned; thus, the MPLE may be found by hand or the vector
of change statistics may be used in some other way. To save space, the
algorithm will automatically search for any duplicated rows in the predictor
matrix (and corresponding response values). \code{ergmMPLE} function will
return a list with three elements, \code{response}, \code{predictor}, and
\code{weights}, respectively the response vector, the predictor matrix, and
a vector of weights, which are really counts that tell how many times each
corresponding response, predictor pair is repeated.

If \code{output=="dyadlist"}, as \code{"matrix"}, but rather than
coalescing the duplicated rows, every relation in the network that
is not fixed and is observed will have its own row in \code{predictor}
and element in \code{response} and \code{weights}, and \code{predictor} matrix
will have two additional rows at the start, \code{tail} and \code{head},
indicating to which dyad the row and the corresponding elements
pertain.

If \code{output=="array"}, a list with similarly named three elements is
returned, but \code{response} is formatted into a sociomatrix;
\code{predictor} is a 3-dimensional array of with cell
\code{predictor[t,h,k]} containing the change score of term \code{k} for
dyad (\code{t},\code{h}); and \code{weights} is also formatted into a
sociomatrix, with an element being 1 if it is to be added into the
pseudolikelihood and 0 if it is not.

In particular, for a unipartite network, cells corresponding to self-loops,
i.e., \code{predictor[i,i,k]} will be \code{NA} and \code{weights[i,i]} will
be 0; and for a unipartite undirected network, lower triangle of each
\code{predictor[,,k]} matrix will be set to \code{NA}, with the lower
triangle of \code{weights} being set to 0.

To all of the above output types, \code{attr(., "etamap")} is attached
containing the \link[=ergm.eta]{mapping and offset information}.

If \code{output=="fit"}, then \code{ergmMPLE} simply calls the
\code{\link[=ergm]{ergm()}} function with the \code{estimate="MPLE"} option set,
returning an object of class \code{ergm} that gives the fitted
pseudolikelihood model.
}
\description{
Return the predictor matrix, response vector, and vector of weights that can
be used to calculate the MPLE for an ERGM.
}
\details{
The MPLE for an ERGM is calculated by first finding the matrix of change
statistics.  Each row of this matrix is associated with a particular pair
(ordered or unordered, depending on whether the network is directed or
undirected) of nodes, and the row equals the change in the vector of network
statistics (as defined in \code{formula}) when that pair is toggled from a 0
(no edge) to a 1 (edge), holding all the rest of the network fixed.  The
MPLE results if we perform a logistic regression in which the predictor
matrix is the matrix of change statistics and the response vector is the
observed network (i.e., each entry is either 0 or 1, depending on whether
the corresponding edge exists or not).

Using \code{output="matrix"}, note that the result of the fit may be
obtained from the \code{\link[=glm]{glm()}} function, as shown in the examples
below.
}
\examples{

data(faux.mesa.high)
formula <- faux.mesa.high ~ edges + nodematch("Sex") + nodefactor("Grade")
mplesetup <- ergmMPLE(formula)

# Obtain MPLE coefficients "by hand":
coef(glm(mplesetup$response ~ . - 1, data = data.frame(mplesetup$predictor),
         weights = mplesetup$weights, family="binomial"))

# Check that the coefficients agree with the output of the ergm function:
coef(ergmMPLE(formula, output="fit"))

# We can also format the predictor matrix into an array:
mplearray <- ergmMPLE(formula, output="array")

# The resulting matrices are big, so only print the first 8 actors:
mplearray$response[1:8,1:8]
mplearray$predictor[1:8,1:8,]
mplearray$weights[1:8,1:8]

# Constraints are handled:
faux.mesa.high\%v\%"block" <- seq_len(network.size(faux.mesa.high)) \%/\% 4
mplearray <- ergmMPLE(faux.mesa.high~edges, constraints=~blockdiag("block"), output="array")
mplearray$response[1:8,1:8]
mplearray$predictor[1:8,1:8,]
mplearray$weights[1:8,1:8]

# Or, a dyad list:
faux.mesa.high\%v\%"block" <- seq_len(network.size(faux.mesa.high)) \%/\% 4
mplearray <- ergmMPLE(faux.mesa.high~edges, constraints=~blockdiag("block"), output="dyadlist")
mplearray$response[1:8]
mplearray$predictor[1:8,]
mplearray$weights[1:8]

# Curved terms produce predictors on the canonical scale:
formula2 <- faux.mesa.high ~ gwesp
mplearray <- ergmMPLE(formula2, output="array")
# The resulting matrices are big, so only print the first 5 actors:
mplearray$response[1:5,1:5]
mplearray$predictor[1:5,1:5,1:3]
mplearray$weights[1:5,1:5]
}
\seealso{
\code{\link[=ergm]{ergm()}}, \code{\link[=glm]{glm()}}
}
\keyword{models}
\keyword{regression}
