\name{gammagpdcon}
\alias{dgammagpdcon}
\alias{gammagpdcon}
\alias{pgammagpdcon}
\alias{qgammagpdcon}
\alias{rgammagpdcon}
\title{Gamma Bulk and GPD Tail Extreme Value Mixture Model with Continuity Constraint}
\usage{
  dgammagpdcon(x, gshape = 1, gscale = 1,
    u = qgamma(0.9, gshape, 1/gscale), xi = 0, phiu = TRUE,
    log = FALSE)

  pgammagpdcon(q, gshape = 1, gscale = 1,
    u = qgamma(0.9, gshape, 1/gscale), xi = 0, phiu = TRUE,
    lower.tail = TRUE)

  qgammagpdcon(p, gshape = 1, gscale = 1,
    u = qgamma(0.9, gshape, 1/gscale), xi = 0, phiu = TRUE,
    lower.tail = TRUE)

  rgammagpdcon(n = 1, gshape = 1, gscale = 1,
    u = qgamma(0.9, gshape, 1/gscale), xi = 0, phiu = TRUE)
}
\arguments{
  \item{x}{quantile}

  \item{gshape}{gamma shape (non-negative)}

  \item{gscale}{gamma scale (non-negative)}

  \item{u}{threshold (non-negative)}

  \item{xi}{shape parameter}

  \item{phiu}{probability of being above threshold [0,1] or
  TRUE}

  \item{log}{logical, if TRUE then log density}

  \item{q}{quantile}

  \item{lower.tail}{logical, if FALSE then upper tail
  probabilities}

  \item{p}{cumulative probability}

  \item{n}{sample size (non-negative integer)}
}
\value{
  \code{\link[evmix:gammagpdcon]{dgammagpdcon}} gives the
  density, \code{\link[evmix:gammagpdcon]{pgammagpdcon}}
  gives the cumulative distribution function,
  \code{\link[evmix:gammagpdcon]{qgammagpdcon}} gives the
  quantile function and
  \code{\link[evmix:gammagpdcon]{rgammagpdcon}} gives a
  random sample.
}
\description{
  Density, cumulative distribution function, quantile
  function and random number generation for the extreme
  value mixture model with gamma for bulk distribution upto
  the threshold and conditional GPD above threshold with a
  continuity constraint. The parameters are the gamma shape
  \code{gshape} and scale \code{gscale}, threshold \code{u}
  and GPD shape \code{xi} and tail fraction \code{phiu}.
}
\details{
  Extreme value mixture model combining gamma distribution
  for the bulk below the threshold and GPD for upper tail
  with a continuity constraint. The user can pre-specify
  \code{phiu} permitting a parameterised value for the tail
  fraction \eqn{\phi_u}. Alternatively, when
  \code{phiu=TRUE} the tail fraction is estimated as the
  tail fraction from the gamma bulk model.

  The cumulative distribution function with tail fraction
  \eqn{\phi_u} defined by the upper tail fraction of the
  gamma bulk model (\code{phiu=TRUE}), upto the threshold
  \eqn{0 < x \le u}, given by: \deqn{F(x) = H(x)} and above
  the threshold \eqn{x > u}: \deqn{F(x) = H(u) + [1 - H(u)]
  G(x)} where \eqn{H(x)} and \eqn{G(x)} are the gamma and
  conditional GPD cumulative distribution functions (i.e.
  \code{pgamma(x, gshape, scale = gscale)} and
  \code{pgpd(x, u, sigmau, xi)}).

  The cumulative distribution function for pre-specified
  \eqn{\phi_u}, upto the threshold \eqn{0 < x \le u}, is
  given by: \deqn{F(x) = (1 - \phi_u) H(x)/H(u)} and above
  the threshold \eqn{x > u}: \deqn{F(x) = \phi_u + [1 -
  \phi_u] G(x)} Notice that these definitions are
  equivalent when \eqn{\phi_u = 1 - H(u)}.

  The continuity constraint means that \eqn{(1 - \phi_u)
  h(u)/H(u) = \phi_u g(u)} where \eqn{h(x)} and \eqn{g(x)}
  are the gamma and conditional GPD density functions (i.e.
  \code{dgamma(x, gshape, scale = gscale)} and
  \code{dgpd(x, u, sigmau, xi)}). The resulting GPD scale
  parameter is then: \deqn{\sigma_u = \phi_u H(u) / [1 -
  \phi_u] h(u)}. In the special case of where the tail
  fraction is defined by the bulk model this reduces to
  \deqn{\sigma_u = [1 - H(u)] / h(u)}.

  The gamma is defined on the non-negative reals, so the
  threshold must be non-negative.

  See \code{\link[evmix:gpd]{gpd}} for details of GPD upper
  tail component and \code{\link[stats:GammaDist]{dgamma}}
  for details of gamma bulk component.
}
\note{
  All inputs are vectorised except \code{log} and
  \code{lower.tail}. The main inputs (\code{x}, \code{p} or
  \code{q}) and parameters must be either a scalar or a
  vector. If vectors are provided they must all be of the
  same length, and the function will be evaluated for each
  element of vector. In the case of \code{rgammagpdcon} any
  input vector must be of length \code{n}.

  Default values are provided for all inputs, except for
  the fundamentals \code{x}, \code{q} and \code{p}. The
  default sample size for
  \code{\link[evmix:gammagpdcon]{rgammagpdcon}} is 1.

  Missing (\code{NA}) and Not-a-Number (\code{NaN}) values
  in \code{x} and \code{q} are passed through as is and
  infinite values are set to \code{NA}.

  Error checking of the inputs (e.g. invalid probabilities)
  is carried out and will either stop or give warning
  message as appropriate.
}
\examples{
\dontrun{
par(mfrow=c(2,2))
x = rgammagpdcon(1000, gshape = 2)
xx = seq(-1, 10, 0.01)
hist(x, breaks = 100, freq = FALSE, xlim = c(-1, 10))
lines(xx, dgammagpdcon(xx, gshape = 2))

# three tail behaviours
plot(xx, pgammagpdcon(xx, gshape = 2), type = "l")
lines(xx, pgammagpdcon(xx, gshape = 2, xi = 0.3), col = "red")
lines(xx, pgammagpdcon(xx, gshape = 2, xi = -0.3), col = "blue")
legend("topleft", paste("xi =",c(0, 0.3, -0.3)),
  col=c("black", "red", "blue"), lty = 1)

x = rgammagpdcon(1000, gshape = 2, u = 3, phiu = 0.2)
hist(x, breaks = 100, freq = FALSE, xlim = c(-1, 10))
lines(xx, dgammagpdcon(xx, gshape = 2, u = 3, phiu = 0.2))

plot(xx, dgammagpdcon(xx, gshape = 2, u = 3, xi=0, phiu = 0.2), type = "l")
lines(xx, dgammagpdcon(xx, gshape = 2, u = 3, xi=-0.2, phiu = 0.2), col = "red")
lines(xx, dgammagpdcon(xx, gshape = 2, u = 3, xi=0.2, phiu = 0.2), col = "blue")
legend("topright", c("xi = 0", "xi = 0.2", "xi = -0.2"),
  col=c("black", "red", "blue"), lty = 1)
  }
}
\author{
  Yang Hu and Carl Scarrott
  \email{carl.scarrott@canterbury.ac.nz}
}
\references{
  \url{http://en.wikipedia.org/wiki/Gamma_distribution}

  \url{http://en.wikipedia.org/wiki/Generalized_Pareto_distribution}

  Scarrott, C.J. and MacDonald, A. (2012). A review of
  extreme value threshold estimation and uncertainty
  quantification. REVSTAT - Statistical Journal 10(1),
  33-59. Available from
  \url{http://www.ine.pt/revstat/pdf/rs120102.pdf}

  Behrens, C.N., Lopes, H.F. and Gamerman, D. (2004).
  Bayesian analysis of extreme events with threshold
  estimation. Statistical Modelling. 4(3), 227-244.
}
\seealso{
  \code{\link[evmix:gpd]{gpd}},
  \code{\link[stats:GammaDist]{dgamma}} and
  \code{\link[evmix:gammagpd]{dgammagpd}}

  Other gammagpdcon: \code{\link{fgammagpdcon}},
  \code{\link{lgammagpdcon}}, \code{\link{nlgammagpdcon}}
}

