#' Summary of the shelf life estimation (osle)
#'
#' This is a method for the function \code{summary()} for objects of class
#' \sQuote{\code{expirest_osle}}.
#'
#' @param object An object of class \sQuote{\code{expirest_osle}} returned
#'   by the \code{\link{expirest_osle}()} function.
#' @param ... Further arguments passed to or from other methods or arguments
#'   that can be passed down to the \code{\link[base]{formatC}()} function.
#'
#' @details The function \code{\link{expirest_osle}()} estimates the shelf
#' life, or retest period, following the ICH Q1E guideline. By default, batch
#' poolability is checked as recommended by the guideline at a significance
#' level of 0.25. Other levels can be used, although not recommended, by
#' changing the default of the \code{alpha_pool} parameter. Three possible
#' models may be appropriate, i.e.
#' \itemize{
#'  \item a \emph{common intercept / common slope} model (cics),
#'  \item a \emph{different intercept / common slope} model (dics) or
#'  \item a \emph{different intercept / different slope} model (dids).
#' }
#'
#' The worst case intercept is the intercept of the batch whose confidence
#' limit is the first crossing the acceptance limit. As in case of the
#' \code{cics} model type all batches have a common intercept and a common
#' confidence interval, all batches can be regarded as equally worst case. In
#' case of the \code{dids} model type, shelf life estimation is done using the
#' models obtained from fitting the data of each batch individually.
#'
#' @return The \sQuote{\code{expirest_osle}} object passed to the \code{object}
#' parameter is returned invisibly.
#'
#' @seealso \code{\link{expirest_osle}}, \code{\link{expirest_wisle}},
#' \code{\link[base]{formatC}}, \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_summary.expirest_osle.R
#'
#' @export

summary.expirest_osle <- function(object, ...) {
  mt <- object[["Model.Type"]]$type.spec

  mf <- match.call(expand.dots = TRUE)
  m <- match("digits", names(mf), 0L)

  if (m == 0) {
    digits <- getOption("digits")
  } else {
    digits <- mf[[m]]
  }

  cat("\nSummary of shelf life estimation following the ICH Q1E guideline")
  cat("\n\nThe best model accepted at a significance level of",
      object[["Parameters"]]$alpha.pool,
      "has\n",
      c("Different intercepts and", "Common intercepts and")[mt[1] + 1],
      c("Different slopes", "Common slopes")[mt[2] + 1],
      paste0("(acronym: ",
             object[["Model.Type"]]$type.acronym,
             ")."))

  cat("\n\nWorst case intercept:",
      ifelse(is.na(object$wc.icpt),
             NA,
             formatC(as.numeric(object$wc.icpt), digits = digits)))

  cat("\nWorst case batch:",
      ifelse(is.na(object$wc.batch),
             NA,
             levels(object[["Data"]]
                    [, object[["Variables"]]$batch])[object$wc.batch]))

  cat("\n\nEstimated shelf life for",
      object[["Model.Type"]]$type.acronym,
      "model:",
      ifelse(is.na(object[["POI"]][object[["Model.Type"]]$type.acronym]),
             NA,
             formatC(object[["POI"]][object[["Model.Type"]]$type.acronym],
                     digits)),
      "\n")

  invisible(object)
}

#' Print a summary of the shelf life estimation (osle)
#'
#' This is a method for the function \code{print()} for objects of class
#' \sQuote{\code{expirest_osle}}.
#'
#' @param x An object of class \sQuote{\code{expirest_osle}} returned by
#'   the \code{\link{expirest_osle}()} function.
#' @inheritParams summary.expirest_osle
#'
#' @return The \sQuote{\code{expirest_osle}} object passed to the \code{x}
#' parameter is returned invisibly.
#'
#' @inherit summary.expirest_osle details seealso
#'
#' @example man/examples/examples_print.expirest_osle.R
#'
#' @export

print.expirest_osle <- function(x, ...) {

  summary(object = x, ...)

  invisible(x)
}

#' Plot illustrating the shelf life estimation (osle)
#'
#' This is a method for the function \code{plot()} for objects of class
#' \sQuote{\code{plot_expirest_osle}}.
#'
#' @param x An object of class \sQuote{\code{plot_expirest_osle}} returned by
#'   the \code{\link{plot_expirest_osle}()} function.
#' @inheritParams summary.expirest_osle
#'
#' @details The element \code{Graph} of the \sQuote{\code{plot_expirest_osle}}
#' object that is returned by the function \code{\link{plot_expirest_osle}()}
#' is an object of class \sQuote{\code{ggplot}}, generated by the function
#' \code{\link[ggplot2]{ggplot}()} from the \sQuote{\code{ggplot2}} package.
#' Thus, the corresponding \code{plot} method is used for plotting. Arguments
#' to the \code{\link[ggplot2]{ggplot}()} function can be passed via the
#' \code{...} parameter.
#'
#' @return The \sQuote{\code{plot_expirest_osle}} object passed to the \code{x}
#' parameter is returned invisibly.
#'
#' @seealso \code{\link{expirest_osle}}, \code{\link{plot_expirest_osle}},
#' \code{\link[ggplot2]{ggplot}()}, \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_plot.plot_expirest_osle.R
#'
#' @export

plot.plot_expirest_osle <- function(x, ...) {

  plot(x = x$Graph, ...)

  invisible(x)
}

#' Print a plot illustrating the shelf life estimation (osle)
#'
#' This is a method for the function \code{print()} for objects of class
#' \sQuote{\code{plot_expirest_osle}}.
#'
#' @param x An object of class \sQuote{\code{plot_expirest_osle}} returned by
#'   the \code{\link{plot_expirest_osle}()} function.
#' @inheritParams plot.plot_expirest_osle
#'
#' @inherit plot.plot_expirest_osle details return seealso
#'
#' @example man/examples/examples_print.plot_expirest_osle.R
#'
#' @export

print.plot_expirest_osle <- function(x, ...) {

  plot(x = x$Graph, ...)

  invisible(x)
}

#' Summary of the what-if shelf life estimation (wisle)
#'
#' This is a method for the function \code{summary()} for objects of class
#' \sQuote{\code{expirest_wisle}}.
#'
#' @param object An object of class \sQuote{\code{expirest_wisle}} returned
#'   by the \code{\link{expirest_wisle}()} function.
#' @param ... Further arguments passed to or from other methods or arguments
#'   that can be passed down to the \code{\link[base]{formatC}()} function.
#'
#' @details The function \code{\link{expirest_wisle}()} estimates the expiry
#' for the specified release and specification limit following the ARGPM
#' guidance \dQuote{Stability testing for prescription medicines}. By default,
#' batch poolability is checked as recommended by the ICH Q1E guideline at a
#' significance level of 0.25. Other levels can be used, although not
#' recommended, by changing the default of the \code{alpha_pool} parameter.
#' Three possible models may be appropriate, i.e.
#' \itemize{
#'  \item a \emph{common intercept / common slope} model (cics),
#'  \item a \emph{different intercept / common slope} model (dics) or
#'  \item a \emph{different intercept / different slope} model (dids).
#' }
#'
#' The worst case intercept is the intercept of the batch whose confidence
#' limit is the first crossing the acceptance limit. As in case of the
#' \code{cics} model type all batches have a common intercept and a common
#' confidence interval, all batches can be regarded as equally worst case. In
#' case of the \code{dids} model type, shelf life estimation is done using the
#' models obtained from fitting the data of each batch individually. In
#' addition to the shelf life estimated according to the ARGPM also the
#' estimate according to ICH Q1E is shown.
#'
#' @return The \sQuote{\code{expirest_wisle}} object passed to the
#' \code{object} parameter is returned invisibly.
#'
#' @seealso \code{\link{expirest_wisle}}, \code{\link{expirest_osle}},
#' \code{\link[base]{formatC}}, \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_summary.expirest_wisle.R
#'
#' @export

summary.expirest_wisle <- function(object, ...) {
  mt <- object[["Model.Type"]]$type.spec

  mf <- match.call(expand.dots = TRUE)
  m <- match("digits", names(mf), 0L)

  if (m == 0) {
    digits <- getOption("digits")
  } else {
    digits <- mf[[m]]
  }

  tmp <- object[["POI"]][, -c(grep("Intercept", colnames(object[["POI"]])),
                             grep("Delta", colnames(object[["POI"]])),
                             grep("WCSL", colnames(object[["POI"]])))]

  tmp <- tmp[, c("Exp.Spec.Report", "Rel.Spec.Report",
                 colnames(tmp)[grep(object[["Model.Type"]]$type.acronym,
                                    colnames(tmp))])]
  colnames(tmp) <- c("SL", "RL", "wisle", "osle")

  cat("\nSummary of shelf life estimation following the ARGPM
  guidance \"Stability testing for prescription medicines\"")
  cat("\n\nThe best model accepted at a significance level of",
      object[["Parameters"]]$alpha.pool,
      "has\n",
      c("Different intercepts and", "Common intercepts and")[mt[1] + 1],
      c("Different slopes", "Common slopes")[mt[2] + 1],
      paste0("(acronym: ",
             object[["Model.Type"]]$type.acronym,
             ")."))

  cat("\n\nWorst case intercept(s):",
      ifelse(is.na(object$wc.icpt[, object[["Model.Type"]]$type.acronym]),
             NA,
             formatC(object$wc.icpt[, object[["Model.Type"]]$type.acronym],
                     digits = digits)))

  cat("\nWorst case batch(es):",
      ifelse(is.na(object$wc.batch[[object[["Model.Type"]]$type.acronym]]),
             NA,
             levels(object[["Data"]][[object[["Variables"]]$batch]])
             [object$wc.batch[[object[["Model.Type"]]$type.acronym]]]))

  cat("\n\nEstimated shelf life (lives) for",
      object[["Model.Type"]]$type.acronym,
      "model:\n")
  print(tmp, digits = digits)
  cat("\nAbbreviations:\n")
  cat("ARGPM: Australian Regulatory Guidelines for Prescription Medicines;",
      "ICH: International Council for Harmonisation;",
      "osle: Ordinary shelf life estimation (i.e. following the ICH guidance);",
      "RL: Release Limit;",
      "SL: Specification Limit;",
      "wisle: What-if (approach for) shelf life estimation",
      "(i.e. following ARGPM guidance).\n\n")

  invisible(object)
}

#' Print a summary of the what-if shelf life estimation (wisle)
#'
#' This is a method for the function \code{print()} for objects of class
#' \sQuote{\code{expirest_wisle}}.
#'
#' @param x An object of class \sQuote{\code{expirest_wisle}} returned by
#'   the \code{expirest_wisle()} function.
#' @inheritParams summary.expirest_wisle
#'
#' @return The \sQuote{\code{expirest_wisle}} object passed to the
#' \code{x} parameter is returned invisibly.
#'
#' @inherit summary.expirest_wisle details seealso
#'
#' @example man/examples/examples_print.expirest_wisle.R
#'
#' @export

print.expirest_wisle <- function(x, ...) {

  summary(object = x, ...)

  invisible(x)
}

#' Plot illustrating the what-if shelf life estimation (wisle)
#'
#' This is a method for the function \code{plot()} for objects of class
#' \sQuote{\code{plot_expirest_wisle}}.
#'
#' @param x An object of class \sQuote{\code{plot_expirest_wisle}} returned by
#'   the \code{\link{plot_expirest_wisle}()} function.
#' @inheritParams summary.expirest_wisle
#'
#' @details The element \code{Graph} of the \sQuote{\code{plot_expirest_wisle}}
#' object that is returned by the function \code{\link{plot_expirest_wisle}()}
#' is an object of class \sQuote{\code{ggplot}}, generated by the function
#' \code{\link[ggplot2]{ggplot}()} from the \sQuote{\code{ggplot2}} package.
#' Thus, the corresponding \code{plot} method is used for plotting. Arguments
#' to the \code{\link[ggplot2]{ggplot}()} function can be passed via the
#' \code{...} parameter.
#'
#' @return The \sQuote{\code{plot_expirest_wisle}} object passed to the \code{x}
#' parameter is returned invisibly.
#'
#' @seealso \code{\link{expirest_wisle}}, \code{\link{plot_expirest_wisle}},
#' \code{\link[ggplot2]{ggplot}()}, \code{\link[utils]{methods}}.
#'
#' @example man/examples/examples_plot.plot_expirest_wisle.R
#'
#' @export

plot.plot_expirest_wisle <- function(x, ...) {

  plot(x = x$Graph, ...)

  invisible(x)
}

#' Print a plot illustrating the what-if shelf life estimation (wisle)
#'
#' This is a method for the function \code{print()} for objects of class
#' \sQuote{\code{plot_expirest_wisle}}.
#'
#' @param x An object of class \sQuote{\code{plot_expirest_wisle}} returned by
#'   the \code{\link{plot_expirest_wisle}()} function.
#' @inheritParams plot.plot_expirest_wisle
#'
#' @inherit plot.plot_expirest_wisle details return seealso
#'
#' @example man/examples/examples_print.plot_expirest_wisle.R
#'
#' @export

print.plot_expirest_wisle <- function(x, ...) {

  plot(x = x$Graph, ...)

  invisible(x)
}
