% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_edgelist.R
\name{sample_edgelist.matrix}
\alias{sample_edgelist.matrix}
\alias{sample_edgelist.Matrix}
\title{Low level interface to sample RPDG edgelists}
\usage{
\method{sample_edgelist}{matrix}(
  factor_model,
  S,
  Y,
  directed,
  poisson_edges,
  allow_self_loops,
  ...
)

\method{sample_edgelist}{Matrix}(
  factor_model,
  S,
  Y,
  directed,
  poisson_edges,
  allow_self_loops,
  ...
)
}
\arguments{
\item{factor_model}{An \code{n} by \code{k1} \code{\link[=matrix]{matrix()}} or \code{\link[Matrix:Matrix]{Matrix::Matrix()}}
of latent node positions encoding incoming edge community membership.
The \code{X} matrix in Rohe et al (2017). Naming differs only for
consistency with the S3 generic.}

\item{S}{A \code{k1} by \code{k2} mixing \code{\link[=matrix]{matrix()}} or \code{\link[Matrix:Matrix]{Matrix::Matrix()}}. In
the undirect case this is assumed to be symmetric but \strong{we do not
check that this is the case}.}

\item{Y}{A \code{d} by \code{k2} \code{\link[=matrix]{matrix()}} or \code{\link[Matrix:Matrix]{Matrix::Matrix()}} of latent
node positions encoding outgoing edge community membership.}

\item{directed}{Logical indicating whether or not the graph should be
directed. When \code{directed = FALSE}, symmetrizes \code{S} internally.
\code{Y = X} together with a symmetric \code{S} implies a symmetric
expectation (although not necessarily an undirected graph).
When \code{directed = FALSE}, samples a directed graph with
symmetric expectation, and then adds edges until symmetry
is achieved.}

\item{poisson_edges}{Whether or not to remove duplicate edges
after sampling. See Section 2.3 of Rohe et al. (2017)
for some additional details. Defaults to \code{TRUE}.}

\item{allow_self_loops}{Logical indicating whether or not
nodes should be allowed to form edges with themselves.
Defaults to \code{TRUE}. When \code{FALSE}, sampling proceeds allowing
self-loops, and these are then removed after the fact.}

\item{...}{Ignored, for generic consistency only.}
}
\value{
A single realization of a random Poisson (or Bernoulli)
Dot Product Graph, represented as a \code{\link[tibble:tibble]{tibble::tibble()}} with two
integer columns, \code{from} and \code{to}.

In the undirected case, \code{from} and \code{to} do not encode
information about edge direction, but we will always have
\code{from <= to} for convenience of edge identification.
To avoid handling such considerations yourself, we recommend using
\code{\link[=sample_sparse]{sample_sparse()}}, \code{\link[=sample_igraph]{sample_igraph()}}, and \code{\link[=sample_tidygraph]{sample_tidygraph()}}
over \code{\link[=sample_edgelist]{sample_edgelist()}}.
}
\description{
\strong{This is a breaks-off, no safety checks interface.}
We strongly recommend that you do not call
\code{sample_edgelist.matrix()} unless you know what you are doing,
and even then, we still do not recommend it, as you will
bypass all typical input validation.
\emph{\strong{extremely loud coughing}} All those who bypass input
validation suffer foolishly at their own hand.
\emph{\strong{extremely loud coughing}}
}
\details{
This function implements the \code{fastRG} algorithm as
described in Rohe et al (2017). Please see the paper
(which is short and open access!!) for details.
}
\examples{

set.seed(46)

n <- 10000
d <- 1000

k1 <- 5
k2 <- 3

X <- matrix(rpois(n = n * k1, 1), nrow = n)
S <- matrix(runif(n = k1 * k2, 0, .1), nrow = k1)
Y <- matrix(rpois(n = d * k2, 1), nrow = d)

sample_edgelist(X, S, Y, TRUE, TRUE, TRUE)

}
\references{
Rohe, Karl, Jun Tao, Xintian Han, and Norbert Binkiewicz. 2017.
"A Note on Quickly Sampling a Sparse Matrix with Low Rank Expectation."
Journal of Machine Learning Research; 19(77):1-13, 2018.
\url{https://www.jmlr.org/papers/v19/17-128.html}
}
\seealso{
Other samplers: 
\code{\link{sample_edgelist}()},
\code{\link{sample_igraph}()},
\code{\link{sample_sparse}()},
\code{\link{sample_tidygraph}()}
}
\concept{samplers}
