% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/physignal.eigen.R
\name{physignal.eigen}
\alias{physignal.eigen}
\title{Assessing dimensions of phylogenetic signal in Procrustes shape variables}
\usage{
physignal.eigen(
  Y,
  phy = NULL,
  Cov = NULL,
  Blomberg = FALSE,
  unit.tree = TRUE,
  lambda = 1,
  test = TRUE,
  iter = 999,
  seed = NULL,
  tol = 0.001
)
}
\arguments{
\item{Y}{A matrix (n x [p x k]) or 3D array (p x k x n) containing Procrustes shape variables for a 
set of specimens (it is assumed that the data have been subjected to a Generalized Procrustes Analysis)}

\item{phy}{A phylogenetic tree of class = "phylo" - see \code{\link[ape]{read.tree}} in library ape}

\item{Cov}{An object covariance matrix describing the phylogenetic relationships among species, which may be 
used in place of `tree}

\item{Blomberg}{A logical value to indicate whether GLS-centering or OLS-centering (the default) should be used}

\item{unit.tree}{A logical value to indicate whether the tree should be scaled to unit height}

\item{lambda}{An optional value for scaling the tree by its phylogenetic signal (as represented by the 
parameter lambda)}

\item{test}{A logical value to indicate whether significance testing should be performed}

\item{iter}{Number of iterations for significance testing}

\item{seed}{An optional argument for setting the seed for random permutations of the resampling procedure.
If left NULL (the default), the exact same P-values will be found for repeated runs of the analysis (with the same number of iterations).
If seed = "random", a random seed will be used, and P-values will vary.  One can also specify an integer for specific seed values,
which might be of interest for advanced users.}

\item{tol}{A value indicating the magnitude below which 
components should be omitted, following projection.  See \code{\link[RRPP]{ordinate}} 
for details.}
}
\value{
Function returns a list with the following components: 
  \item{eib.obs}{The observed eigenvalues of the phylogenetic signal matrix, K.}
  \item{rand.eigen.values}{The set of eigenvalues from the permuted datasets.}
  \item{traceK.obs}{The observed traceK statistic.}
  \item{traceK}{The set of traceK statistics from the permuted datasets.}
  \item{p.traceK}{The p-value of traceK from permutation.}
  \item{Z.traceK}{The effect size of traceK.}
  \item{detK.obs}{The observed detK statistic.}
  \item{detK}{The set of detK statistics from the permuted datasets.}
  \item{p.detK}{The p-value of detK from permutation.}
  \item{Z.detK}{The effect size of detK.}
  \item{Kmult.obs}{The observed Kmult statistic.}
  \item{Kmult}{The set of Kmult statistics from the permuted datasets.}
  \item{p.Kmult}{The p-value of multK from permutation.}
  \item{Z.Kmult}{The effect size of Kmult.}   
  \item{permutations}{The number of random permutations used in the resampling procedure.}
  \item{call}{The matched call}
}
\description{
Function calculates the degree of phylogenetic signal across dimensions of shape space from 
Procrustes shape variables
}
\details{
The function estimates the degree of phylogenetic signal across dimensions of shape space, based 
on Procrustes shape variables for a given phylogeny. The method is intended to interrogate patterns 
of phylogenetic signal that are not uniformly distributed across shape space, but rather are 
concentrated in one or a few dimensions. The degree of phylogenetic signal in data is estimated 
using a matrix generalization of Blomberg's Kappa (K), and the phylogenetic signal represented by this
matrix is decomposed into linear combinations that describe directions of phylogenetic signal within the 
dataspace containing decreasing levels of phylogenetic signal (Mitteroecker et al. 2024). Two 
summary measures based on the eigenvalues of K are calculated: the determinant of K (detK) and the 
trace of K (traceK). Both describe the degree of phylogenetic signal in multivariate data. In addition, 
the  multivariate version of the K-statistic (Kmult: Adams 2014) is also provided. All three 
statistics are evaluated via permutation, where the shape data is permuted among the tips of the 
phylogeny. Effect sizes are also provided.

It should be noted that by default the function uses OLS centering when calculating the phylogenetic 
signal matrix, unlike most authors who have used GLS centering. Both OLS and GLS centering were 
proposed in Blomberg et al.s (2003) original exploration of phylogenetic signal. However, for both 
mathematical and computational reasons OLS is used here (see justification in Mitteroecker et al. 
2024). Both measures are highly rank correlated with one another (Mitteroecker et al. 2024). Additionally,
using using BLOMBERG = TRUE will result in GLS centering, in which case the statistic Kmult 
obtained with this function will be identical to that obtained when using \code{\link{physignal}}.

Importantly, because detK and traceK are based on the covariance matrix K, singularity can become
an issue. In particular, geometric morphometric shape data will not be of full rank, as several 
dimensions are standardized during the Generalized Procrustes Analysis (one may also have fewer 
observations than variables, which will also generate redundancies). For this reason, a 
principal components analysis of the data is performed, and the redundant dimensions are 
removed so that detK and traceK may be computed (see Mitteroecker et al. 2024). Additionally, if 
n< (p X k), the last nontrivial PC dimension is also removed, as in this case, using 100% of the 
variation results in invariant K-statistics across permutations. 
  
The generic functions, \code{\link{print}}, \code{\link{summary}}, and \code{\link{plot}} all work with \code{\link{physignal.eigen}}.
}
\examples{
\dontrun{
data(plethspecies) 
Y.gpa <- gpagen(plethspecies$land)    #GPA-alignment    

#Test for phylogenetic signal in shape
PSe.shape <- physignal.eigen(Y = Y.gpa$coords, phy = plethspecies$phy)
summary(PSe.shape)
plot(PSe.shape)
plot(PSe.shape, type = "vectors")

}
}
\references{
Mitteroecker, P., M.L. Collyer, and D.C. Adams. 2024. Exploring phylogenetic signal in 
multivariate phenotypes by maximizing Blomberg's K. Systematic Biology. (In Press).

Blomberg SP, Garland T, Ives AR. 2003. Testing for phylogenetic signal in comparative 
data: behavioral traits are more labile. Evolution, 57:717-745.

Adams, D.C. 2014. A generalized K statistic for estimating phylogenetic signal from shape and 
other high-dimensional multivariate data. Systematic Biology.  63:685-697.
}
\seealso{
\code{\link{gm.prcomp}}, \code{\link{physignal}}
}
\author{
Dean Adams and Michael Collyer
}
\keyword{analysis}
