% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{llkw}
\alias{llkw}
\title{Negative Log-Likelihood of the Kumaraswamy (Kw) Distribution}
\usage{
llkw(par, data)
}
\arguments{
\item{par}{A numeric vector of length 2 containing the distribution parameters
in the order: \code{alpha} (\eqn{\alpha > 0}), \code{beta} (\eqn{\beta > 0}).}

\item{data}{A numeric vector of observations. All values must be strictly
between 0 and 1 (exclusive).}
}
\value{
Returns a single \code{double} value representing the negative
log-likelihood (\eqn{-\ell(\theta|\mathbf{x})}). Returns \code{Inf}
if any parameter values in \code{par} are invalid according to their
constraints, or if any value in \code{data} is not in the interval (0, 1).
}
\description{
Computes the negative log-likelihood function for the two-parameter
Kumaraswamy (Kw) distribution with parameters \code{alpha} (\eqn{\alpha})
and \code{beta} (\eqn{\beta}), given a vector of observations. This function
is suitable for maximum likelihood estimation.
}
\details{
The Kumaraswamy (Kw) distribution's probability density function (PDF) is
(see \code{\link{dkw}}):
\deqn{
f(x | \theta) = \alpha \beta x^{\alpha-1} (1 - x^\alpha)^{\beta-1}
}
for \eqn{0 < x < 1} and \eqn{\theta = (\alpha, \beta)}.
The log-likelihood function \eqn{\ell(\theta | \mathbf{x})} for a sample
\eqn{\mathbf{x} = (x_1, \dots, x_n)} is \eqn{\sum_{i=1}^n \ln f(x_i | \theta)}:
\deqn{
\ell(\theta | \mathbf{x}) = n[\ln(\alpha) + \ln(\beta)]
+ \sum_{i=1}^{n} [(\alpha-1)\ln(x_i) + (\beta-1)\ln(v_i)]
}
where \eqn{v_i = 1 - x_i^{\alpha}}.
This function computes and returns the \emph{negative} log-likelihood, \eqn{-\ell(\theta|\mathbf{x})},
suitable for minimization using optimization routines like \code{\link[stats]{optim}}.
It is equivalent to the negative log-likelihood of the GKw distribution
(\code{\link{llgkw}}) evaluated at \eqn{\gamma=1, \delta=0, \lambda=1}.
}
\examples{
\donttest{
# Assuming existence of rkw, grkw, hskw functions for Kw distribution

# Generate sample data from a known Kw distribution
set.seed(123)
true_par_kw <- c(alpha = 2, beta = 3)
sample_data_kw <- rkw(100, alpha = true_par_kw[1], beta = true_par_kw[2])
hist(sample_data_kw, breaks = 20, main = "Kw(2, 3) Sample")

# --- Maximum Likelihood Estimation using optim ---
# Initial parameter guess
start_par_kw <- c(1.5, 2.5)

# Perform optimization (minimizing negative log-likelihood)
# Use method="L-BFGS-B" for box constraints (params > 0)
mle_result_kw <- stats::optim(par = start_par_kw,
                              fn = llkw, # Use the Kw neg-log-likelihood
                              method = "L-BFGS-B",
                              lower = c(1e-6, 1e-6), # Lower bounds > 0
                              hessian = TRUE,
                              data = sample_data_kw)

# Check convergence and results
if (mle_result_kw$convergence == 0) {
  print("Optimization converged successfully.")
  mle_par_kw <- mle_result_kw$par
  print("Estimated Kw parameters:")
  print(mle_par_kw)
  print("True Kw parameters:")
  print(true_par_kw)
} else {
  warning("Optimization did not converge!")
  print(mle_result_kw$message)
}

# --- Compare numerical and analytical derivatives (if available) ---
# Requires 'numDeriv' package and analytical functions 'grkw', 'hskw'
if (mle_result_kw$convergence == 0 &&
    requireNamespace("numDeriv", quietly = TRUE) &&
    exists("grkw") && exists("hskw")) {

  cat("\nComparing Derivatives at Kw MLE estimates:\n")

  # Numerical derivatives of llkw
  num_grad_kw <- numDeriv::grad(func = llkw, x = mle_par_kw, data = sample_data_kw)
  num_hess_kw <- numDeriv::hessian(func = llkw, x = mle_par_kw, data = sample_data_kw)

  # Analytical derivatives (assuming they return derivatives of negative LL)
  ana_grad_kw <- grkw(par = mle_par_kw, data = sample_data_kw)
  ana_hess_kw <- hskw(par = mle_par_kw, data = sample_data_kw)

  # Check differences
  cat("Max absolute difference between gradients:\n")
  print(max(abs(num_grad_kw - ana_grad_kw)))
  cat("Max absolute difference between Hessians:\n")
  print(max(abs(num_hess_kw - ana_hess_kw)))

} else {
   cat("\nSkipping derivative comparison for Kw.\n")
   cat("Requires convergence, 'numDeriv' package and functions 'grkw', 'hskw'.\n")
}

}

}
\references{
Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.

Jones, M. C. (2009). Kumaraswamy's distribution: A beta-type distribution
with some tractability advantages. \emph{Statistical Methodology}, \emph{6}(1), 70-81.
}
\seealso{
\code{\link{llgkw}} (parent distribution negative log-likelihood),
\code{\link{dkw}}, \code{\link{pkw}}, \code{\link{qkw}}, \code{\link{rkw}},
\code{grkw} (gradient, if available),
\code{hskw} (Hessian, if available),
\code{\link[stats]{optim}}
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{kumaraswamy}
\keyword{likelihood}
\keyword{optimize}
