#' Plot quantile-quantile (QQ) graphs from residuals of linear models.
#'
#' This function takes a linear model (simple or mixed effects) and plots a QQ graph after running `augment` from \code{\link[broom.mixed]{augment}} to generate a table of model residuals on a saved ordinary (\code{\link{simple_model}}) or mixed model (\code{\link{mixed_model}} or \code{\link{mixed_model_slopes}}. The graph plots residuals from the model (sample) on Y axis & theoretical quantiles on X axis.
#'
#' The function uses \code{\link{ggplot2}[geom_qq]} and \code{\link{ggplot2}[geom_qq_line]} geometries. Symbols receive "ok_orange" colour by default.
#'
#' @param Model name of a saved model generated by \code{simple_model} or \code{mixed_model}.
#' @param symsize size of symbols, default set to 3.
#' @param symthick thickness of symbol border, default set to 1.
#' @param s_alpha fractional opacity of symbols, default set to 1 (i.e. maximum opacity & zero transparency).
#' @param fontsize parameter of \code{base_size} of fonts in \code{theme_classic}, default set to size 20.
#'
#' @return This function returns a \code{ggplot2} object of class "gg" and "ggplot".
#' @export plot_qqmodel
#' @import ggplot2
#' @importFrom broom.mixed augment
#'
#' @examples
#' #Basic usage
#' m1 <- simple_model(data = data_2w_Festing, 
#' Y_value = "GST", 
#' Fixed_Factor = c("Treatment", "Strain"))
#' plot_qqmodel(m1)
#'

plot_qqmodel <- function(Model, symsize = 2.5, symthick = 1, s_alpha = 1, fontsize = 20){
mod <- broom.mixed::augment(Model)
P <- ggplot2::ggplot(mod,
                     aes(sample = mod$`.resid`))+
  geom_qq_line(na.rm = T,
                   size = 1)+
  geom_qq(na.rm = T,
          shape = 21, fill = "#E69F00",
          size = {{ symsize }},
          stroke = {{ symthick }},
          alpha = {{ s_alpha }})+
  theme_classic(base_size = {{ fontsize }})+
  theme(strip.background = element_blank())
suppressWarnings(P)
}
