% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gsMethods.R, R/gsqplot.R
\name{summary.gsDesign}
\alias{summary.gsDesign}
\alias{print.gsDesign}
\alias{gsBoundSummary}
\alias{xprint}
\alias{print.gsBoundSummary}
\alias{gsBValue}
\alias{gsDelta}
\alias{gsRR}
\alias{gsHR}
\alias{gsCPz}
\title{Bound Summary and Z-transformations}
\usage{
\method{summary}{gsDesign}(object, information = FALSE, timeunit = "months", ...)

\method{print}{gsDesign}(x, ...)

gsBoundSummary(
  x,
  deltaname = NULL,
  logdelta = FALSE,
  Nname = NULL,
  digits = 4,
  ddigits = 2,
  tdigits = 0,
  timename = "Month",
  prior = normalGrid(mu = x$delta/2, sigma = 10/sqrt(x$n.fix)),
  POS = FALSE,
  ratio = NULL,
  exclude = c("B-value", "Spending", "CP", "CP H1", "PP"),
  r = 18,
  ...
)

xprint(
  x,
  include.rownames = FALSE,
  hline.after = c(-1, which(x$Value == x[1, ]$Value) - 1, nrow(x)),
  ...
)

\method{print}{gsBoundSummary}(x, row.names = FALSE, digits = 4, ...)

gsBValue(z, i, x, ylab = "B-value", ...)

gsDelta(z, i, x, ylab = NULL, ...)

gsRR(z, i, x, ratio = 1, ylab = "Approximate risk ratio", ...)

gsHR(z, i, x, ratio = 1, ylab = "Approximate hazard ratio", ...)

gsCPz(z, i, x, theta = NULL, ylab = NULL, ...)
}
\arguments{
\item{object}{An item of class \code{gsDesign} or \code{gsSurv}}

\item{information}{indicator of whether \code{n.I} in \code{object}
represents statistical information rather than sample size or event counts.}

\item{timeunit}{Text string with time units used for time-to-event designs
created with \code{gsSurv()}}

\item{...}{This allows many optional arguments that are standard when
calling \code{plot} for \code{gsBValue}, \code{gsDelta}, \code{gsHR},
\code{gsRR} and \code{gsCPz}}

\item{x}{An item of class \code{gsDesign} or \code{gsSurv}, except for
\code{print.gsBoundSummary()} where \code{x} is an object created by
\code{gsBoundSummary()} and \code{xprint()} which is used with \code{xtable}
(see examples)}

\item{deltaname}{Natural parameter name. If default \code{NULL} is used,
routine will default to \code{"HR"} when class is \code{gsSurv} or if
\code{nFixSurv} was input when creating \code{x} with \code{gsDesign()}.}

\item{logdelta}{Indicates whether natural parameter is the natural logarithm
of the actual parameter. For example, the relative risk or odds-ratio would
be put on the logarithmic scale since the asymptotic behavior is 'more
normal' than a non-transformed value. As with \code{deltaname}, the default
will be changed to true if \code{x} has class \code{gsDesign} or if
\code{nFixSurv>0} was input when \code{x} was created by \code{gsDesign()};
that is, the natural parameter for a time-to-event endpoint will be on the
logarithmic scale.}

\item{Nname}{This will normally be changed to \code{"N"} or, if a
time-to-event endpoint is used, \code{"Events"}. Other immediate possibility
are \code{"Deaths"} or \code{"Information"}.}

\item{digits}{Number of digits past the decimal to be printed in the body of
the table.}

\item{ddigits}{Number of digits past the decimal to be printed for the
natural parameter delta.}

\item{tdigits}{Number of digits past the decimal point to be shown for
estimated timing of each analysis.}

\item{timename}{Text string indicating time unit.}

\item{prior}{A prior distribution for the standardized effect size. Must be
of the format produced by \code{normalGrid()}, but can reflect an arbitrary
prior distribution. The default reflects a normal prior centered half-way
between the null and alternate hypothesis with the variance being equivalent
to the treatment effect estimate if 1 percent of the sample size for a fixed
design were sampled. The prior is intended to be relatively uninformative.
This input will only be applied if \code{POS=TRUE} is input.}

\item{POS}{This is an indicator of whether or not probability of success
(POS) should be estimated at baseline or at each interim based on the prior
distribution input in \code{prior}. The prior probability of success before
the trial starts is the power of the study averaged over the prior
distribution for the standardized effect size. The POS after an interim
analysis assumes the interim test statistic is an unknown value between the
futility and efficacy bounds. Based on this, a posterior distribution for
the standardized parameter is computed and the conditional power of the
trial is averaged over this posterior distribution.}

\item{ratio}{Sample size ratio assumed for experimental to control treatment
group sample sizes. This only matters when \code{x} for a binomial or
time-to-event endpoint where \code{gsRR} or \code{gsHR} are used for
approximating the treatment effect if a test statistic falls on a study
bound.}

\item{exclude}{A list of test statistics to be excluded from design boundary
summary produced; see details or examples for a list of all possible output
values. A value of \code{NULL} produces all available summaries.}

\item{r}{See \code{\link{gsDesign}}. This is an integer used to control the
degree of accuracy of group sequential calculations which will normally not
be changed.}

\item{include.rownames}{indicator of whether or not to include row names in
output.}

\item{hline.after}{table lines after which horizontal separation lines
should be set; default is to put lines between each analysis as well as at
the top and bottom of the table.}

\item{row.names}{indicator of whether or not to print row names}

\item{z}{A vector of z-statistics}

\item{i}{A vector containing the analysis for each element in \code{z}; each
element must be in 1 to \code{x$k}, inclusive}

\item{ylab}{Used when functions are passed to \code{plot.gsDesign} to
establish default y-axis labels}

\item{theta}{A scalar value representing the standardized effect size used
for conditional power calculations; see \code{gsDesign}; if NULL,
conditional power is computed at the estimated interim treatment effect
based on \code{z}}
}
\value{
\code{gsBValue()}, \code{gsDelta()}, \code{gsHR()} and
\code{gsCPz()} each returns a vector containing the B-values, approximate
treatment effect (see details), approximate hazard ratio and conditional
power, respectively, for each value specified by the interim test statistics
in \code{z} at interim analyses specified in \code{i}.

\code{summary} returns a text string summarizing the design at a high level.
This may be used with \code{gsBoundSummary} for a nicely formatted, concise
group sequential design description.

\code{gsBoundSummary} returns a table in a data frame providing a variety of
boundary characteristics. The tabular format makes formatting particularly
amenable to place in documents either through direct creation of readable by
Word (see the \code{rtf} package) or to a csv format readable by spreadsheet
software using \code{write.csv}.

\code{print.gsDesign} prints an overall summary a group sequential design.
While the design description is complete, the format is not as `document
friendly' as \code{gsBoundSummary}.

\code{print.gsBoundSummary} is a simple extension of \code{print.data.frame}
intended for objects created with \code{gsBoundSummary}. The only extension
is to make the default to not print row names. This is probably `not good R
style' but may be helpful for many lazy R programmers like the author.
}
\description{
A tabular summary of a group sequential design's bounds and their properties
are often useful. The 'vintage' \code{print.gsDesign()} function provides a
complete but minimally formatted summary of a group sequential design
derived by \code{gsDesign()}. A brief description of the overall design can
also be useful (\code{summary.gsDesign()}.  A tabular summary of boundary
characteristics oriented only towards LaTeX output is produced by
\code{\link{xtable.gsSurv}}. More flexibility is provided by
\code{gsBoundSummary()} which produces a tabular summary of a
user-specifiable set of package-provided boundary properties in a data
frame.  This can also be used to along with functions such as
\code{\link{print.data.frame}()}, \code{\link{write.table}()},
\code{\link{write.csv}()}, \code{\link{write.csv2}()} or, from the RTF
package, \code{addTable.RTF()} (from the rtf package) to produce console or
R Markdown output or output to a variety of file types. \code{xprint()} is
provided for LaTeX output by setting default options for
\code{\link{print.xtable}()} when producing tables summarizing design
bounds.

Individual transformation of z-value test statistics for interim and final
analyses are obtained from \code{gsBValue()}, \code{gsDelta()},
\code{gsHR()} and \code{gsCPz()} for B-values, approximate treatment effect
(see details), approximate hazard ratio and conditional power, respectively.

The \code{print.gsDesign} function is intended to provide an easier output
to review than is available from a simple list of all the output components.
The \code{gsBoundSummary} function is intended to provide a summary of
boundary characteristics that is often useful for evaluating boundary
selection; this outputs an extension of the \code{data.frame} class that
sets up default printing without row names using
\code{print.gsBoundSummary}. \code{summary.gsDesign}, on the other hand,
provides a summary of the overall design at a higher level; this provides
characteristics not included in the \code{gsBoundSummary} summary and no
detail concerning interim analysis bounds.

In brief, the computed descriptions of group sequential design bounds are as
follows: \code{Z:} Standardized normal test statistic at design bound.

\code{p (1-sided):} 1-sided p-value for \code{Z}. This will be computed as
the probability of a greater EXCEPT for lower bound when a 2-sided design is
being summarized.

\code{delta at bound:} Approximate value of the natural parameter at the
bound. The approximate standardized effect size at the bound is generally
computed as \code{Z/sqrt(n)}. Calling this \code{theta}, this is translated
to the \code{delta} using the values \code{delta0} and \code{delta1} from
the input \code{x} by the formula \code{delta0 +
(delta1-delta0)/theta1*theta} where \code{theta1} is the alternate
hypothesis value of the standardized parameter. Note that this value will be
exponentiated in the case of relative risks, hazard ratios or when the user
specifies \code{logdelta=TRUE}. In the case of hazard ratios, the value is
computed instead by \code{gsHR()} to be consistent with
\code{plot.gsDesign()}. Similarly, the value is computed by \code{gsRR()}
when the relative risk is the natural parameter.

\code{Spending: }Incremental error spending at each given analysis. For
asymmetric designs, futility bound will have beta-spending summarized.
Efficacy bound always has alpha-spending summarized.

\code{B-value: }\code{sqrt(t)*Z} where \code{t} is the proportion of
information at the analysis divided by the final analysis planned
information. The expected value for B-values is directly proportional to
\code{t}.

\code{CP: }Conditional power under the estimated treatment difference
assuming the interim Z-statistic is at the study bound

\code{CP H1: }Conditional power under the alternate hypothesis treatment
effect assuming the interim test statistic is at the study bound.

\code{PP: }Predictive power assuming the interim test statistic is at the
study bound and the input prior distribution for the standardized effect
size. This is the conditional power averaged across the posterior
distribution for the treatment effect given the interim test statistic
value. \code{P{Cross if delta=xx}: }For each of the parameter values in
\code{x}, the probability of crossing either bound given that treatment
effect is computed. This value is cumulative for each bound. For example,
the probability of crossing the efficacy bound at or before the analysis of
interest.
}
\note{
The manual is not linked to this help file, but is available in
library/gsdesign/doc/gsDesignManual.pdf in the directory where R is
installed.
}
\examples{
library(ggplot2)
# survival endpoint using gsSurv
# generally preferred over nSurv since time computations are shown
xgs <- gsSurv(lambdaC = .2, hr = .5, eta = .1, T = 2, minfup = 1.5)
gsBoundSummary(xgs, timename = "Year", tdigits = 1)
summary(xgs)

# survival endpoint using nSurvival
# NOTE: generally recommend gsSurv above for this!
ss <- nSurvival(
  lambda1 = .2, lambda2 = .1, eta = .1, Ts = 2, Tr = .5,
  sided = 1, alpha = .025, ratio = 2
)
xs <- gsDesign(nFixSurv = ss$n, n.fix = ss$nEvents, delta1 = log(ss$lambda2 / ss$lambda1))
gsBoundSummary(xs, logdelta = TRUE, ratio = ss$ratio)
# generate some of the above summary statistics for the upper bound
z <- xs$upper$bound
# B-values
gsBValue(z = z, i = 1:3, x = xs)
# hazard ratio
gsHR(z = z, i = 1:3, x = xs)
# conditional power at observed treatment effect
gsCPz(z = z[1:2], i = 1:2, x = xs)
# conditional power at H1 treatment effect
gsCPz(z = z[1:2], i = 1:2, x = xs, theta = xs$delta)

# information-based design
xinfo <- gsDesign(delta = .3, delta1 = .3)
gsBoundSummary(xinfo, Nname = "Information")

# show all available boundary descriptions
gsBoundSummary(xinfo, Nname = "Information", exclude = NULL)

# add intermediate parameter value
xinfo <- gsProbability(d = xinfo, theta = c(0, .15, .3))
class(xinfo) # note this is still as gsDesign class object
gsBoundSummary(xinfo, Nname = "Information")

# now look at a binomial endpoint; specify H0 treatment difference as p1-p2=.05
# now treatment effect at bound (say, thetahat) is transformed to
# xp$delta0 + xp$delta1*(thetahat-xp$delta0)/xp$delta
np <- nBinomial(p1 = .15, p2 = .10)
xp <- gsDesign(n.fix = np, endpoint = "Binomial", delta1 = .05)
summary(xp)
gsBoundSummary(xp, deltaname = "p[C]-p[E]")
# estimate treatment effect at lower bound
# by setting delta0=0 (default) and delta1 above in gsDesign
# treatment effect at bounds is scaled to these differences
# in this case, this is the difference in event rates
gsDelta(z = xp$lower$bound, i = 1:3, xp)

# binomial endpoint with risk ratio estimates
n.fix <- nBinomial(p1 = .3, p2 = .15, scale = "RR")
xrr <- gsDesign(k = 2, n.fix = n.fix, delta1 = log(.15 / .3), endpoint = "Binomial")
gsBoundSummary(xrr, deltaname = "RR", logdelta = TRUE)
gsRR(z = xp$lower$bound, i = 1:3, xrr)
plot(xrr, plottype = "RR")

# delta is odds-ratio: sample size slightly smaller than for relative risk or risk difference
n.fix <- nBinomial(p1 = .3, p2 = .15, scale = "OR")
xOR <- gsDesign(k = 2, n.fix = n.fix, delta1 = log(.15 / .3 / .85 * .7), endpoint = "Binomial")
gsBoundSummary(xOR, deltaname = "OR", logdelta = TRUE)

# for nice LaTeX table output, use xprint
xprint(xtable::xtable(gsBoundSummary(xOR, deltaname = "OR", logdelta = TRUE), 
                                          caption = "Table caption."))
}
\references{
Jennison C and Turnbull BW (2000), \emph{Group Sequential
Methods with Applications to Clinical Trials}. Boca Raton: Chapman and Hall.
}
\seealso{
\link{gsDesign}, \link{plot.gsDesign},
\code{\link{gsProbability}}, \code{\link{xtable.gsSurv}}
}
\author{
Keaven Anderson \email{keaven\_anderson@merck.}
}
\keyword{design}
