% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_update_ahr.R
\name{gs_update_ahr}
\alias{gs_update_ahr}
\title{Group sequential design using average hazard ratio under non-proportional hazards}
\usage{
gs_update_ahr(
  x = NULL,
  alpha = NULL,
  ustime = NULL,
  lstime = NULL,
  observed_data = NULL
)
}
\arguments{
\item{x}{A design created by either \code{\link[=gs_design_ahr]{gs_design_ahr()}} or \code{\link[=gs_power_ahr]{gs_power_ahr()}}.}

\item{alpha}{Type I error for the updated design.}

\item{ustime}{Default is NULL in which case upper bound spending time is determined by timing.
Otherwise, this should be a vector of length k (total number of analyses)
with the spending time at each analysis.}

\item{lstime}{Default is NULL in which case lower bound spending time is determined by timing.
Otherwise, this should be a vector of length k (total number of analyses)
with the spending time at each analysis}

\item{observed_data}{a list of observed datasets by analyses.}
}
\value{
A list with input parameters, enrollment rate, analysis, and bound.
}
\description{
Group sequential design using average hazard ratio under non-proportional hazards
}
\examples{
library(gsDesign)
library(gsDesign2)
library(dplyr)

alpha <- 0.025
beta <- 0.1
ratio <- 1

# Enrollment
enroll_rate <- define_enroll_rate(
  duration = c(2, 2, 10),
  rate = (1:3) / 3)

# Failure and dropout
fail_rate <- define_fail_rate(
  duration = c(3, Inf), fail_rate = log(2) / 9,
  hr = c(1, 0.6), dropout_rate = .0001)

# IA and FA analysis time
analysis_time <- c(20, 36)

# Randomization ratio
ratio <- 1

# ------------------------------------------------- #
# Example A: one-sided design (efficacy only)
# ------------------------------------------------- #
# Original design
upper <- gs_spending_bound
upar <- list(sf = sfLDOF, total_spend = alpha)
x <- gs_design_ahr(
  enroll_rate = enroll_rate, fail_rate = fail_rate,
  alpha = alpha, beta = beta, ratio = ratio,
  info_scale = "h0_info",
  info_frac = NULL,
  analysis_time = c(20, 36),
  upper = gs_spending_bound, upar = upar,
  lower = gs_b, lpar = rep(-Inf, 2),
  test_upper = TRUE, test_lower = FALSE) |> to_integer()

# Observed dataset at IA and FA
set.seed(123)

observed_data <- simtrial::sim_pw_surv(
  n = x$analysis$n[x$analysis$analysis == 2],
  stratum = data.frame(stratum = "All", p = 1),
  block = c(rep("control", 2), rep("experimental", 2)),
  enroll_rate = x$enroll_rate,
  fail_rate = (fail_rate |> simtrial::to_sim_pw_surv())$fail_rate,
  dropout_rate = (fail_rate |> simtrial::to_sim_pw_surv())$dropout_rate)

observed_data_ia <- observed_data |> simtrial::cut_data_by_date(x$analysis$time[1])
observed_data_fa <- observed_data |> simtrial::cut_data_by_date(x$analysis$time[2])

observed_event_ia <- sum(observed_data_ia$event)
observed_event_fa <- sum(observed_data_fa$event)

planned_event_ia <- x$analysis$event[1]
planned_event_fa <- x$analysis$event[2]

# Example A1 ----
# IA spending = observed events / final planned events
# the remaining alpha will be allocated to FA.
ustime <- c(observed_event_ia / planned_event_fa, 1)
gs_update_ahr(
  x = x,
  ustime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# Example A2 ----
# IA, FA spending = observed events / final planned events
ustime <- c(observed_event_ia, observed_event_fa) / planned_event_fa
gs_update_ahr(
  x = x,
  ustime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# Example A3 ----
# IA spending = min(observed events, planned events) / final planned events
ustime <- c(min(observed_event_ia, planned_event_ia) / planned_event_fa, 1)
gs_update_ahr(
  x = x,
  ustime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# Example A4 ----
# IA spending = min(observed events, planned events) / final planned events
ustime <- c(min(observed_event_ia, planned_event_ia),
            min(observed_event_fa, planned_event_fa)) / planned_event_fa
gs_update_ahr(
  x = x,
  ustime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# alpha is upadted to 0.05
gs_update_ahr(
  x = x,
  alpha = 0.05,
  ustime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# ------------------------------------------------- #
# Example B: Two-sided asymmetric design,
# beta-spending with non-binding lower bound
# ------------------------------------------------- #
# Original design
x <- gs_design_ahr(
  enroll_rate = enroll_rate, fail_rate = fail_rate,
  alpha = alpha, beta = beta, ratio = ratio,
  info_scale = "h0_info",
  info_frac = NULL, analysis_time = c(20, 36),
  upper = gs_spending_bound,
  upar = list(sf = sfLDOF, total_spend = alpha),
  test_upper = TRUE,
  lower = gs_spending_bound,
  lpar = list(sf = sfLDOF, total_spend = beta),
  test_lower = c(TRUE, FALSE),
  binding = FALSE) |> to_integer()

# Example B1 ----
# IA spending = observed events / final planned events
# the remaining alpha will be allocated to FA.
ustime <- c(observed_event_ia / planned_event_fa, 1)
gs_update_ahr(
  x = x,
  ustime = ustime,
  lstime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# Example B2 ----
# IA, FA spending = observed events / final planned events
ustime <- c(observed_event_ia, observed_event_fa) / planned_event_fa
gs_update_ahr(
  x = x,
  ustime = ustime,
  lstime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# Example B3 ----
ustime <- c(min(observed_event_ia, planned_event_ia) / planned_event_fa, 1)
gs_update_ahr(
  x = x,
  ustime = ustime,
  lstime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# Example B4 ----
# IA spending = min(observed events, planned events) / final planned events
ustime <- c(min(observed_event_ia, planned_event_ia),
            min(observed_event_fa, planned_event_fa)) / planned_event_fa
gs_update_ahr(
  x = x,
  ustime = ustime,
  lstime = ustime,
  observed_data = list(observed_data_ia, observed_data_fa))

# Example B5 ----
# alpha is updated to 0.05 ----
gs_update_ahr(x = x, alpha = 0.05)

# Example B6 ----
# updated boundaries only when IA data is observed
ustime <- c(observed_event_ia / planned_event_fa, 1)
gs_update_ahr(
  x = x,
  ustime = ustime,
  lstime = ustime,
  observed_data = list(observed_data_ia, NULL))

# ------------------------------------------------- #
# Example C: Two-sided asymmetric design,
# with calendar spending for efficacy and futility bounds
# beta-spending with non-binding lower bound
# ------------------------------------------------- #
# Original design
x <- gs_design_ahr(
  enroll_rate = enroll_rate, fail_rate = fail_rate,
  alpha = alpha, beta = beta, ratio = ratio,
  info_scale = "h0_info",
  info_frac = NULL, analysis_time = c(20, 36),
  upper = gs_spending_bound,
  upar = list(sf = sfLDOF, total_spend = alpha, timing = c(20, 36) / 36),
  test_upper = TRUE,
  lower = gs_spending_bound,
  lpar = list(sf = sfLDOF, total_spend = beta, timing = c(20, 36) / 36),
  test_lower = c(TRUE, FALSE),
  binding = FALSE) |> to_integer()

# Updated design due to potential change of multiplicity graph
gs_update_ahr(x = x, alpha = 0.05)
}
