% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_tf}
\alias{fmt_tf}
\title{Format \code{TRUE} and \code{FALSE} values}
\usage{
fmt_tf(
  data,
  columns = everything(),
  rows = everything(),
  tf_style = "true-false",
  pattern = "{x}",
  true_val = NULL,
  false_val = NULL,
  na_val = NULL,
  colors = NULL,
  auto_align = TRUE,
  locale = NULL
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\code{\link[=rows-columns]{<column-targeting expression>}} // \emph{default:} \code{everything()}

Can either be a series of column names provided in \code{c()}, a vector of
column indices, or a select helper function (e.g. \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}} and \code{\link[=everything]{everything()}}).}

\item{rows}{\emph{Rows to target}

\code{\link[=rows-columns]{<row-targeting expression>}} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
undergo formatting. The default \code{\link[=everything]{everything()}} results in all rows in
\code{columns} being formatted. Alternatively, we can supply a vector of row
captions within \code{c()}, a vector of row indices, or a select helper
function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{tf_style}{\emph{Predefined style for \code{TRUE}/\code{FALSE} formatting}

\verb{scalar<character>|scalar<numeric|integer>(1<=val<=10)} // \emph{default:} \code{"true-false"}

The \code{TRUE}/\code{FALSE} mapping style to use. By default this is the short name
\code{"true-false"} which corresponds to the words 'true' and 'false'. Two other
\code{tf_style} values produce words: \code{"yes-no"} and \code{"up-down"}. All three of
these options for \code{tf_style} are locale-aware through the \code{locale} option,
so, a \code{"yes"} value will instead be \code{"ja"} when \code{locale = "de"}. Options
4 through to 10 involve pairs of symbols (e.g., \code{"check-mark"} displays
a check mark for \code{TRUE} and an X symbol for \code{FALSE}).}

\item{pattern}{\emph{Specification of the formatting pattern}

\verb{scalar<character>} // \emph{default:} \code{"{x}"}

A formatting pattern that allows for decoration of the formatted value. The
formatted value is represented by the \code{{x}} (which can be used multiple
times, if needed) and all other characters will be interpreted as string
literals.}

\item{true_val}{\emph{Text to use for \code{TRUE} values}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

While the choice of a \code{tf_style} will typically supply the \code{true_val} and
\code{false_val} text, we could override this and supply text for any \code{TRUE}
values.  This doesn't need to be used in conjunction with \code{false_val}.}

\item{false_val}{\emph{Text to use for \code{FALSE} values}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

While the choice of a \code{tf_style} will typically supply the \code{true_val} and
\code{false_val} text, we could override this and supply text for any \code{FALSE}
values. This doesn't need to be used in conjunction with \code{true_val}.}

\item{na_val}{\emph{Text to use for \code{NA} values}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

None of the \code{tf_style} presets will replace any missing values encountered
in the targeted cells. While we always have the option to use
\code{\link[=sub_missing]{sub_missing()}} for \code{NA} replacement, we have the opportunity to do that
here with the \code{na_val} option. This is useful because we also have the
means to add color to the \code{na_val} text or symbol and doing that requires
that a replacement value for \code{NA}s is specified here.}

\item{colors}{\emph{Colors to use for the resulting strings or symbols}

\verb{vector<character>} // \emph{default:} \code{NULL} (\code{optional})

Providing a vector of color values to \code{colors} will progressively add color
to the formatted result depending on the number of colors provided. With a
single color, all formatted values will be in that color. Giving two colors
results in \code{TRUE} values being the first color, and \code{FALSE} values
receiving the second. With the three color option, the final color will be
given to any \code{NA} values replaced through \code{na_val}.}

\item{auto_align}{\emph{Automatic alignment of the formatted column}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

The input values may have resulted in an alignment that is not as suitable
once formatting has occurred. With \code{auto_align = TRUE}, the formatted
values will be inspected and this may result in a favorable change in
alignment. Typically, symbols will be center aligned whereas words will
receive a left alignment (for words in LTR languages).}

\item{locale}{\emph{Locale identifier}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional locale identifier that can be used for formatting values
according the locale's rules. Examples include \code{"en"} for English (United
States) and \code{"fr"} for French (France). We can call \code{\link[=info_locales]{info_locales()}} for a
useful reference for all of the locales that are supported. A locale ID can
be also set in the initial \code{\link[=gt]{gt()}} function call (where it would be used
automatically by any function with a \code{locale} argument) but a \code{locale}
value provided here will override that global locale.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
There can be times where logical values are useful in a \strong{gt} table. You
might want to express a 'yes' or 'no', a 'true' or 'false', or, perhaps use
pairings of complementary symbols that make sense in a table. The \code{fmt_tf()}
function has a set of \code{tf_style} presets that can be used to quickly map
\code{TRUE}/\code{FALSE} values to strings (which are automatically translated
according to a given \code{locale} value), or, symbols like up/down or left/right
arrows and open/closed shapes.

While the presets are nice, you can provide your own mappings through the
\code{true_val} and \code{false_val} arguments. With those you could provide text
(perhaps a Unicode symbol?) or even a \strong{fontawesome} icon by using
\code{fontawesome::fa("<icon name>")}. The function will automatically handle
alignment when \code{auto_align = TRUE} and try to give you the best look
depending on the options chosen. For extra customization, you can also apply
color to the individual \code{TRUE}, \code{FALSE}, and \code{NA} mappings. Just supply
a vector of colors (up to a length of 3) to the \code{colors} argument.
}
\section{Compatibility of formatting function with data values}{


\code{fmt_tf()} is compatible with body cells that are of the \code{"logical"}
(preferred) or \code{"numeric"} types. Any other types of body cells are ignored
during formatting. This is to say that cells of incompatible data types may
be targeted, but there will be no attempt to format them.

There is a special caveat when attempting to format numerical values: the
values must either be exactly \code{1} (the analogue for \code{TRUE}) or exactly \code{0}
(the analogue for \code{FALSE}). Any other numerical values will be disregarded
and left as is. Because of these restrictions, it is recommended that only
logical values undergo formatting.
}

\section{Compatibility of arguments with the \code{from_column()} helper function}{


\code{\link[=from_column]{from_column()}} can be used with certain arguments of \code{fmt_tf()} to obtain
varying parameter values from a specified column within the table. This means
that each row could be formatted a little bit differently. These arguments
provide support for \code{\link[=from_column]{from_column()}}:
\itemize{
\item \code{tf_style}
\item \code{pattern}
\item \code{true_val}
\item \code{false_val}
\item \code{na_val}
\item \code{locale}
}

Please note that for each of the aforementioned arguments, a \code{\link[=from_column]{from_column()}}
call needs to reference a column that has data of the correct type (this is
different for each argument). Additional columns for parameter values can be
generated with \code{\link[=cols_add]{cols_add()}} (if not already present). Columns that contain
parameter data can also be hidden from final display with \code{\link[=cols_hide]{cols_hide()}}.
Finally, there is no limitation to how many arguments the \code{\link[=from_column]{from_column()}}
helper is applied so long as the arguments belong to this closed set.
}

\section{Formatting with the \code{tf_style} argument}{


We can supply a preset \code{TRUE}/\code{FALSE} style to the \code{tf_style} argument to
handle the formatting of logical values. There are several such styles and
the first three of them can handle localization to any supported locale
(i.e., the pairs of words for each style will be translated to the language
of the \code{locale}) value.

The following table provides a listing of all valid \code{tf_style} values and a
description of their output values. The output from styles \code{4} to \code{10} are
described in terms of the Unicode character names used for the \code{TRUE} and
\code{FALSE} values.\tabular{lll}{
    \tab TF Style \tab Output (for \code{TRUE} and \code{FALSE}) \cr
   1 \tab \code{"true-false"} \tab \code{"true"}, \code{"false"} (\code{locale}-aware) \cr
   2 \tab \code{"yes-no"} \tab \code{"yes"}, \code{"no"} (\code{locale}-aware) \cr
   3 \tab \code{"up-down"} \tab \code{"up"}, \code{"down"} (\code{locale}-aware) \cr
   4 \tab \code{"check-mark"} \tab \verb{<Heavy Check Mark>}, \verb{<Heavy Ballot X>} \cr
   5 \tab \code{"circles"} \tab \verb{<Black Circle>}, \verb{<Heavy Circle>} \cr
   6 \tab \code{"squares"} \tab \verb{<Black Square>},  \verb{<White Square>} \cr
   7 \tab \code{"diamonds"} \tab \verb{<Black Diamond>}, \verb{<White Diamond>} \cr
   8 \tab \code{"arrows"} \tab \verb{<Upwards Arrow>}, \verb{<Downwards Arrow>} \cr
   9 \tab \code{"triangles"} \tab \verb{<Black Up-Pointing Triangle>}, \verb{<Black Down-Pointing Triangle>} \cr
   10 \tab \code{"triangles-lr"} \tab \verb{<Heavy Check Mark>}, \verb{<Heavy Ballot X>} \cr
}
}

\section{Adapting output to a specific \code{locale}}{


This formatting function can adapt outputs according to a provided \code{locale}
value. Examples include \code{"en"} for English (United States) and \code{"fr"} for
French (France). Note that a \code{locale} value provided here will override any
global locale setting performed in \code{\link[=gt]{gt()}}'s own \code{locale} argument (it is
settable there as a value received by all other functions that have a
\code{locale} argument). As a useful reference on which locales are supported, we
can call \code{\link[=info_locales]{info_locales()}} to view an info table.
}

\section{Examples}{


Let's use a subset of the \code{\link{sp500}} dataset to create a small \strong{gt} table
containing opening and closing price data for a week in 2013. We can add
a logical column (\code{dir}) with \code{\link[=cols_add]{cols_add()}}; the expression used determines
whether the \code{close} value is greater than the \code{open} value. That new column
is inserted between \code{open} and \code{close}. Then, we use \code{fmt_tf()} to generate
up and down arrows in the \code{dir} column. We elect to use green upward arrows
and red downward arrows (through the \code{colors} option). With a little numeric
formatting and changes to the column labels, the table becomes more presentable.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sp500 |>
  dplyr::filter(date >= "2013-01-07" & date <= "2013-01-12") |>
  dplyr::arrange(date) |>
  dplyr::select(-c(adj_close, volume, high, low)) |>
  gt(rowname_col = "date") |>
  cols_add(dir = close > open, .after = open) |>
  fmt_tf(
    columns = dir,
    tf_style = "arrows",
    colors = c("green", "red")
  ) |>
  fmt_currency(columns = c(open, close)) |>
  cols_label(
    open = "Opening",
    close = "Closing",
    dir = ""
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_tf_1.png" alt="This image of a table was generated from the first code example in the `fmt_tf()` help file." style="width:100\%;">
}}

The \code{\link{reactions}} dataset contains chemical kinetic information on a wide
variety of atmospherically-relevant compounds. It might be interesting to get
a summary (for a small subset of compounds) for which rate constants are
available for the selected compounds. We first start by selecting the
relevant rows and columns. Then we generate logical columns for each of the
reaction types (i.e., if a value is \code{NA} then there's no measurement, so
that's \code{FALSE}). Once the \strong{gt} table has been created, we can use
\code{fmt_tf()} to provide open and filled circles to indicate whether a
particular reaction has been measured and presented in the literature.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{reactions |>
  dplyr::filter(cmpd_type \%in\% c("carboxylic acid", "alkyne", "allene")) |>
  dplyr::select(cmpd_name, cmpd_type, ends_with("k298")) |>
  dplyr::mutate(across(ends_with("k298"), is.na)) |>
  gt(rowname_col = "cmpd_name", groupname_col = "cmpd_type") |>
  tab_spanner(
    label = "Has a measured rate constant",
    columns = ends_with("k298")
  ) |>
  tab_stub_indent(
    rows = everything(),
    indent = 2
  ) |>
  fmt_tf(
    columns = ends_with("k298"),
    tf_style = "circles"
  ) |>
  cols_label(
    OH_k298 = "OH",
    O3_k298 = "Ozone",
    NO3_k298 = "Nitrate",
    Cl_k298 = "Chlorine"
  ) |>
  cols_width(
    stub() ~ px(200),
    ends_with("k298") ~ px(80)
  ) |>
  opt_vertical_padding(scale = 0.35)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_tf_2.png" alt="This image of a table was generated from the second code example in the `fmt_tf()` help file." style="width:100\%;">
}}

There are census-based population values in the \code{\link{towny}} dataset and quite a
few small towns within it. Let's look at the ten smallest towns (according
to the 2021 figures) and work out whether their populations have increased or
declined since 1996. Also, let's determine which of these towns even have a
website. After that data preparation, the data is made into a \strong{gt} table
and \code{fmt_tf()} can be used in the \code{website} and \code{pop_dir} columns (which both
have \code{TRUE}/\code{FALSE} values). Each of these \code{fmt_tf()} calls will either
produce \code{"yes"}/\code{"no"} or \code{"up"}/\code{"down"} strings (set via the \code{tf_style}
option).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::arrange(population_2021) |>
  dplyr::mutate(website = !is.na(website))  |>
  dplyr::mutate(pop_dir = population_2021 > population_1996) |>
  dplyr::select(name, website, population_1996, population_2021, pop_dir) |>
  dplyr::slice_head(n = 10) |>
  gt(rowname_col = "name") |>
  tab_spanner(
    label = "Population",
    columns = starts_with("pop")
  ) |>
  tab_stubhead(label = "Town") |>
  fmt_tf(
    columns = website,
    tf_style = "yes-no",
    auto_align = FALSE
  ) |>
  fmt_tf(
    columns = pop_dir,
    tf_style = "up-down",
    pattern = "It's \{x\}."
  ) |>
  cols_label_with(
    columns = starts_with("population"),
    fn = function(x) sub("population_", "", x)
  ) |>
  cols_label(
    website = md("Has a  \\n website?"),
    pop_dir = "Pop. direction?"
  ) |>
  opt_horizontal_padding(scale = 2)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_tf_3.png" alt="This image of a table was generated from the third code example in the `fmt_tf()` help file." style="width:100\%;">
}}

If formatting to words instead of symbols (with the hyphenated \code{tf_style}
keywords), the words themselves can be translated to different languages
if providing a \code{locale} value. In this next example, we're manually creating
a tibble with locale codes and their associated languages. The \code{yes} and \code{up}
columns all receive \code{TRUE} whereas \code{no} and \code{down} will all be \code{FALSE}.
With two calls of \code{fmt_tf()} for each of these pairings, we get the columns'
namesake words. To have these words translated, the \code{locale} argument is
pointed toward values in the \code{code} column by using \code{\link[=from_column]{from_column()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  code = c("de", "fr", "is", "tr", "ka", "lt", "ca", "bg", "lv"),
  lang = c(
    "German", "French", "Icelandic", "Turkish", "Georgian",
    "Lithuanian", "Catalan", "Bulgarian", "Latvian"
  ),
  yes = TRUE,
  no = FALSE,
  up = TRUE,
  down = FALSE
) |>
  gt(rowname_col = "lang") |>
  tab_header(title = "Common words in a few languages") |>
  fmt_tf(
    columns = c(yes, no),
    tf_style = "yes-no",
    locale = from_column("code")
  ) |>
  fmt_tf(
    columns = c(up, down),
    tf_style = "up-down",
    locale = from_column("code")
  ) |>
  cols_merge(
    columns = c(lang, code),
    pattern = "\{1\} (\{2\})"
  ) |>
  cols_width(
    stub() ~ px(150),
    everything() ~ px(80)
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_tf_4.png" alt="This image of a table was generated from the fourth code example in the `fmt_tf()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-18
}

\section{Function Introduced}{

\code{v0.11.0} (July 9, 2024)
}

\seealso{
Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_chem}()},
\code{\link{fmt_country}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_date}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_email}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_icon}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
