% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modify_columns.R
\name{cols_merge_n_pct}
\alias{cols_merge_n_pct}
\title{Merge two columns to combine counts and percentages}
\usage{
cols_merge_n_pct(data, col_n, col_pct, autohide = TRUE)
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{col_n}{A column that contains values for the count component.}

\item{col_pct}{A column that contains values for the percentage component.
This column should be formatted such that percentages are displayed (e.g.,
with \code{fmt_percent()}).}

\item{autohide}{An option to automatically hide the column specified as
\code{col_pct}. Any columns with their state changed to hidden will behave
the same as before, they just won't be displayed in the finalized table.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
The \code{cols_merge_n_pct()} function is a specialized variant of the
\code{\link[=cols_merge]{cols_merge()}} function. It operates by taking two columns that constitute
both a count (\code{col_n}) and a fraction of the total population (\code{col_pct}) and
merges them into a single column. What results is a column containing both
counts and their associated percentages (e.g., \verb{12 (23.2\%)}). The column
specified in \code{col_pct} is dropped from the output table.
}
\details{
This function could be somewhat replicated using \code{\link[=cols_merge]{cols_merge()}}, however,
\code{cols_merge_n_pct()} employs the following specialized semantics for \code{NA}
and zero-value handling:
\enumerate{
\item \code{NA}s in \code{col_n} result in missing values for the merged
column (e.g., \code{NA} + \verb{10.2\%} = \code{NA})
\item \code{NA}s in \code{col_pct} (but not \code{col_n}) result in
base values only for the merged column (e.g., \code{13} + \code{NA} = \code{13})
\item \code{NA}s both \code{col_n} and \code{col_pct} result in
missing values for the merged column (e.g., \code{NA} + \code{NA} = \code{NA})
\item If a zero (\code{0}) value is in \code{col_n} then the formatted output will be
\code{"0"} (i.e., no percentage will be shown)
}

Any resulting \code{NA} values in the \code{col_n} column following the merge
operation can be easily formatted using the \code{\link[=sub_missing]{sub_missing()}} function.
Separate calls of \code{\link[=sub_missing]{sub_missing()}} can be used for the \code{col_n} and
\code{col_pct} columns for finer control of the replacement values. It is the
responsibility of the user to ensure that values are correct in both the
\code{col_n} and \code{col_pct} columns (this function neither generates nor
recalculates values in either). Formatting of each column can be done
independently in separate \code{\link[=fmt_number]{fmt_number()}} and \code{\link[=fmt_percent]{fmt_percent()}} calls.

This function is part of a set of four column-merging functions. The other
two are the general \code{\link[=cols_merge]{cols_merge()}} function and the specialized
\code{\link[=cols_merge_uncert]{cols_merge_uncert()}} and \code{\link[=cols_merge_range]{cols_merge_range()}} functions. These functions
operate similarly, where the non-target columns can be optionally hidden from
the output table through the \code{hide_columns} or \code{autohide} options.
}
\section{Examples}{


Use \code{\link{pizzaplace}} to create a \strong{gt} table that displays the counts and
percentages of the top 3 pizzas sold by pizza category in 2015. The
\code{cols_merge_n_pct()} function is used to merge the \code{n} and \code{frac} columns
(and the \code{frac} column is formatted using \code{\link[=fmt_percent]{fmt_percent()}}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace \%>\%
  dplyr::group_by(name, type, price) \%>\%
  dplyr::summarize(
    n = dplyr::n(),
    frac = n/nrow(.),
    .groups = "drop"
  ) \%>\%
  dplyr::arrange(type, dplyr::desc(n)) \%>\%
  dplyr::group_by(type) \%>\%
  dplyr::slice_head(n = 3) \%>\%
  gt(
    rowname_col = "name",
    groupname_col = "type"
  ) \%>\%
  fmt_currency(price) \%>\%
  fmt_percent(frac) \%>\%
  cols_merge_n_pct(
    col_n = n,
    col_pct = frac
  ) \%>\%
  cols_label(
    n = md("*N* (\%)"),
    price = "Price"
  ) \%>\%
  tab_style(
    style = cell_text(font = "monospace"),
    locations = cells_stub()
  ) \%>\%
  tab_stubhead(md("Cat. and  \\nPizza Code")) \%>\%
  tab_header(title = "Top 3 Pizzas Sold by Category in 2015") \%>\%
  tab_options(table.width = px(512))
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_merge_n_pct_1.png" alt="This image of a table was generated from the first code example in the `cols_merge_n_pct()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

4-11
}

\seealso{
Other column modification functions: 
\code{\link{cols_align}()},
\code{\link{cols_hide}()},
\code{\link{cols_label}()},
\code{\link{cols_merge_range}()},
\code{\link{cols_merge_uncert}()},
\code{\link{cols_merge}()},
\code{\link{cols_move_to_end}()},
\code{\link{cols_move_to_start}()},
\code{\link{cols_move}()},
\code{\link{cols_unhide}()},
\code{\link{cols_width}()}
}
\concept{column modification functions}
