% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modify_columns.R
\name{cols_label_with}
\alias{cols_label_with}
\title{Relabel columns with a function}
\usage{
cols_label_with(data, columns = everything(), fn)
}
\arguments{
\item{data}{A table object that is created using the \code{\link[=gt]{gt()}} function.}

\item{columns}{The column names to which the function or function call in
\code{fn} should be applied. By default this is set as \code{everything()} which
select every column in the table.}

\item{fn}{The function or function call to be applied to the column labels.
This can take the form of a bare function (e.g., \code{tools::toTitleCase}), a
function call as a RHS formula (e.g., \code{~ tools::toTitleCase(.)}), or an
anonymous function as in \code{function(x) tools::toTitleCase(x)}.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Column labels can be modified from their default values (the names of the
columns from the input table data). When you create a \strong{gt} table object
using \code{\link[=gt]{gt()}}, column names effectively become the column labels. While this
serves as a good first approximation, you may want to make adjustments so
that the columns names present better in the \strong{gt} output table. The
\code{cols_label_with()} function allows for modification of column labels through
a supplied function. By default, the function will be invoked on all column
labels but this can be limited to a subset via the \code{columns} argument. With
the \code{fn} argument, we provide either a bare function name, a RHS formula
(with \code{.} representing the vector of column labels), or, an anonymous
function (e.g., \code{function(x) tools::toTitleCase(x)}).
}
\section{A note on column names and column labels}{


It's important to note that while columns can be freely relabeled, we
continue to refer to columns by their original column names. Column names in
a tibble or data frame must be unique whereas column labels in \strong{gt} have
no requirement for uniqueness (which is useful for labeling columns as, say,
measurement units that may be repeated several times---usually under
different spanner column labels). Thus, we can still easily distinguish
between columns in other \strong{gt} function calls (e.g., in all of the
\verb{fmt*()} functions) even though we may lose distinguishability in column
labels once they have been relabeled.
}

\section{Examples}{


Use \code{\link{sp500}} to create a \strong{gt} table. We want all the column labels to be
entirely capitalized versions of the default labels but, instead of using
\code{\link[=cols_label]{cols_label()}} and rewriting each label manually in capital letters we can
use \code{cols_label_with()} and instruct it to apply the \code{toupper()} function to
all column labels.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sp500 |>
  dplyr::filter(
    date >= "2015-12-01" &
      date <= "2015-12-15"
  ) |>
  dplyr::select(-c(adj_close, volume)) |>
  gt() |>
  cols_label_with(fn = toupper)
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_label_with_1.png" alt="This image of a table was generated from the first code example in the `cols_label_with()` help file." style="width:100\%;">
}}

Use \code{\link{countrypops}} to create a \strong{gt} table. To improve the presentation of
the table, we are again going to change the default column labels via
function calls supplied within \code{cols_label_with()}. We can, if we prefer,
apply multiple types of column label changes in sequence with multiple calls
of \code{cols_label_with()}. Here, we use the \code{make_clean_names()} functions from
the \strong{janitor} package and follow up with the removal of a numeral with
\code{gsub()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{countrypops |>
  dplyr::filter(year == 2021) |>
  dplyr::filter(grepl("^C", country_code_3)) |>
  dplyr::select(-country_code_2, -year) |>
  head(8) |>
  gt() |>
  cols_move_to_start(columns = country_code_3) |>
  fmt_integer(columns = population) |>
  cols_label_with(
    fn = ~ janitor::make_clean_names(., case = "title")
  ) |>
  cols_label_with(
    fn = ~ gsub("[0-9]", "", .)
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_label_with_2.png" alt="This image of a table was generated from the second code example in the `cols_label_with()` help file." style="width:100\%;">
}}

We can make a svelte \strong{gt} table with the \code{\link{pizzaplace}} dataset. There are
ways to use one instance of \code{cols_label_with()} with multiple functions
called on the column labels. In the example, we use an anonymous function
call (with the \code{function(x) { ... }} construction) to perform multiple
mutations of \code{x} (the vector of column labels). We can even use the \code{\link[=md]{md()}}
helper function with that to signal to \strong{gt} that the column label should be
interpreted as Markdown text.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace |>
  dplyr::mutate(month = substr(date, 6, 7)) |>
  dplyr::group_by(month) |>
  dplyr::summarize(pizze_vendute = dplyr::n()) |>
  dplyr::ungroup() |>
  dplyr::mutate(frazione_della_quota = pizze_vendute / 4000) |>
  dplyr::mutate(date = paste0("2015/", month, "/01")) |>
  dplyr::select(-month) |>
  gt(rowname_col = "date") |>
  fmt_date(date, date_style = "month", locale = "it") |>
  fmt_percent(columns = frazione_della_quota) |>
  fmt_integer(columns = pizze_vendute) |>
  cols_width(everything() ~ px(100)) |>
  cols_label_with(
    fn = function(x) \{
      janitor::make_clean_names(x, case = "title") |>
        toupper() |>
        stringr::str_replace_all("^|$", "**") |>
        md()
    \}
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_cols_label_with_3.png" alt="This image of a table was generated from the third code example in the `cols_label_with()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

5-5
}

\seealso{
Other column modification functions: 
\code{\link{cols_align_decimal}()},
\code{\link{cols_align}()},
\code{\link{cols_hide}()},
\code{\link{cols_label}()},
\code{\link{cols_merge_n_pct}()},
\code{\link{cols_merge_range}()},
\code{\link{cols_merge_uncert}()},
\code{\link{cols_merge}()},
\code{\link{cols_move_to_end}()},
\code{\link{cols_move_to_start}()},
\code{\link{cols_move}()},
\code{\link{cols_unhide}()},
\code{\link{cols_width}()}
}
\concept{column modification functions}
