/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Bmodule.h" 

#include "H5private.h"   
#include "H5Bpkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 

static herr_t H5B__cache_get_initial_load_size(void *udata, size_t *image_len);
static void  *H5B__cache_deserialize(const void *image, size_t len, void *udata, bool *dirty);
static herr_t H5B__cache_image_len(const void *thing, size_t *image_len);
static herr_t H5B__cache_serialize(const H5F_t *f, void *image, size_t len, void *thing);
static herr_t H5B__cache_free_icr(void *thing);

const H5AC_class_t H5AC_BT[1] = {{
    H5AC_BT_ID,                       
    "v1 B-tree",                      
    H5FD_MEM_BTREE,                   
    H5AC__CLASS_NO_FLAGS_SET,         
    H5B__cache_get_initial_load_size, 
    NULL,                             
    NULL,                             
    H5B__cache_deserialize,           
    H5B__cache_image_len,             
    NULL,                             
    H5B__cache_serialize,             
    NULL,                             
    H5B__cache_free_icr,              
    NULL,                             
}};

static herr_t
H5B__cache_get_initial_load_size(void *_udata, size_t *image_len)
{
    H5B_cache_ud_t *udata = (H5B_cache_ud_t *)_udata; 
    H5B_shared_t   *shared;                           

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(udata);
    assert(image_len);

    
    shared = (H5B_shared_t *)H5UC_GET_OBJ(udata->rc_shared);
    assert(shared);

    
    *image_len = shared->sizeof_rnode;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5B__cache_deserialize(const void *_image, size_t len, void *_udata, bool H5_ATTR_UNUSED *dirty)
{
    H5B_t          *bt    = NULL;                     
    H5B_cache_ud_t *udata = (H5B_cache_ud_t *)_udata; 
    H5B_shared_t   *shared;                           
    const uint8_t  *image = (const uint8_t *)_image;  
    const uint8_t  *p_end = image + len - 1;          
    uint8_t        *native;                           
    unsigned        u;                                
    H5B_t          *ret_value = NULL;                 

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(udata);

    
    if (NULL == (bt = H5FL_MALLOC(H5B_t)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "can't allocate B-tree struct");
    memset(&bt->cache_info, 0, sizeof(H5AC_info_t));

    
    bt->rc_shared = udata->rc_shared;
    H5UC_INC(bt->rc_shared);

    
    shared = (H5B_shared_t *)H5UC_GET_OBJ(bt->rc_shared);
    if (NULL == shared)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTGET, NULL, "can't get a pointer to shared data");

    
    if (NULL == (bt->native = H5FL_BLK_MALLOC(native_block, shared->sizeof_keys)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "can't allocate buffer for native keys");
    if (NULL == (bt->child = H5FL_SEQ_MALLOC(haddr_t, (size_t)shared->two_k)))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTALLOC, NULL, "can't allocate buffer for child addresses");

    
    if (H5_IS_BUFFER_OVERFLOW(image, H5_SIZEOF_MAGIC, p_end))
        HGOTO_ERROR(H5E_BTREE, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (memcmp(image, H5B_MAGIC, (size_t)H5_SIZEOF_MAGIC) != 0)
        HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, NULL, "wrong B-tree signature");
    image += H5_SIZEOF_MAGIC;

    
    if (H5_IS_BUFFER_OVERFLOW(image, 2, p_end))
        HGOTO_ERROR(H5E_BTREE, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*image++ != (uint8_t)udata->type->id)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTLOAD, NULL, "incorrect B-tree node type");
    bt->level = *image++;

    
    if (udata->exp_level != H5B_UNKNOWN_NODELEVEL)
        if (bt->level != (unsigned)udata->exp_level)
            HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, NULL, "level is not as expected, possibly corrupted");

    
    if (H5_IS_BUFFER_OVERFLOW(image, 2, p_end))
        HGOTO_ERROR(H5E_BTREE, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(image, bt->nchildren);

    
    if (bt->nchildren > shared->two_k)
        HGOTO_ERROR(H5E_BTREE, H5E_BADVALUE, NULL, "number of children is greater than maximum");

    
    if (H5_IS_BUFFER_OVERFLOW(image, H5F_sizeof_addr(udata->f), p_end))
        HGOTO_ERROR(H5E_BTREE, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_addr_decode(udata->f, (const uint8_t **)&image, &(bt->left));

    if (H5_IS_BUFFER_OVERFLOW(image, H5F_sizeof_addr(udata->f), p_end))
        HGOTO_ERROR(H5E_BTREE, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5F_addr_decode(udata->f, (const uint8_t **)&image, &(bt->right));

    
    native = bt->native;
    for (u = 0; u < bt->nchildren; u++) {
        
        if (H5_IS_BUFFER_OVERFLOW(image, shared->sizeof_rkey, p_end))
            HGOTO_ERROR(H5E_BTREE, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        if ((udata->type->decode)(shared, image, native) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDECODE, NULL, "unable to decode key");
        image += shared->sizeof_rkey;
        native += udata->type->sizeof_nkey;

        
        if (H5_IS_BUFFER_OVERFLOW(image, H5F_sizeof_addr(udata->f), p_end))
            HGOTO_ERROR(H5E_BTREE, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5F_addr_decode(udata->f, (const uint8_t **)&image, bt->child + u);
    }

    
    if (bt->nchildren > 0) {
        
        if ((udata->type->decode)(shared, image, native) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTDECODE, NULL, "unable to decode key");
    }

    
    ret_value = bt;

done:
    if (!ret_value && bt)
        if (H5B__node_dest(bt) < 0)
            HDONE_ERROR(H5E_BTREE, H5E_CANTFREE, NULL, "unable to destroy B-tree node");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B__cache_image_len(const void *_thing, size_t *image_len)
{
    const H5B_t  *bt = (const H5B_t *)_thing; 
    H5B_shared_t *shared;                     

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(bt);
    assert(image_len);

    
    shared = (H5B_shared_t *)H5UC_GET_OBJ(bt->rc_shared);
    assert(shared);

    
    *image_len = shared->sizeof_rnode;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5B__cache_serialize(const H5F_t *f, void *_image, size_t H5_ATTR_UNUSED len, void *_thing)
{
    H5B_t        *bt = (H5B_t *)_thing;      
    H5B_shared_t *shared;                    
    uint8_t      *image = (uint8_t *)_image; 
    uint8_t      *native;                    
    unsigned      u;                         
    herr_t        ret_value = SUCCEED;       

    FUNC_ENTER_PACKAGE

    
    assert(image);
    assert(bt);
    assert(bt->rc_shared);
    shared = (H5B_shared_t *)H5UC_GET_OBJ(bt->rc_shared);
    assert(shared);
    assert(shared->type);
    assert(shared->type->encode);

    
    H5MM_memcpy(image, H5B_MAGIC, (size_t)H5_SIZEOF_MAGIC);
    image += 4;

    
    *image++ = (uint8_t)shared->type->id;

    
    if (bt->level >= pow(2, LEVEL_BITS))
        HGOTO_ERROR(H5E_BTREE, H5E_CANTENCODE, FAIL, "unable to encode node level");

    H5_CHECK_OVERFLOW(bt->level, unsigned, uint8_t);
    *image++ = (uint8_t)bt->level;

    
    UINT16ENCODE(image, bt->nchildren);

    
    H5F_addr_encode(f, &image, bt->left);
    H5F_addr_encode(f, &image, bt->right);

    
    native = bt->native;
    for (u = 0; u < bt->nchildren; ++u) {
        
        if (shared->type->encode(shared, image, native) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTENCODE, FAIL, "unable to encode B-tree key");
        image += shared->sizeof_rkey;
        native += shared->type->sizeof_nkey;

        
        H5F_addr_encode(f, &image, bt->child[u]);
    } 
    if (bt->nchildren > 0) {
        
        if (shared->type->encode(shared, image, native) < 0)
            HGOTO_ERROR(H5E_BTREE, H5E_CANTENCODE, FAIL, "unable to encode B-tree key");
        image += shared->sizeof_rkey;
    } 

    
    assert((size_t)(image - (uint8_t *)_image) <= len);

    
    memset(image, 0, len - (size_t)(image - (uint8_t *)_image));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5B__cache_free_icr(void *thing)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(thing);

    
    if (H5B__node_dest((H5B_t *)thing) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_CANTFREE, FAIL, "unable to destroy B-tree node");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
