% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/har_model.R
\name{harModel}
\alias{harModel}
\title{HAR model estimation (Heterogeneous Autoregressive model for Realized volatility)}
\usage{
harModel(
  data,
  periods = c(1, 5, 22),
  periodsJ = c(1, 5, 22),
  periodsQ = c(1),
  leverage = NULL,
  RVest = c("rCov", "rBPCov", "rQuar"),
  type = "HARRV",
  inputType = "RM",
  jumptest = "ABDJumptest",
  alpha = 0.05,
  h = 1,
  transform = NULL,
  ...
)
}
\arguments{
\item{data}{an xts object containing either: intra-day (log-)returns or realized measures already computed from such returns. In case more than one realized measure is needed, the object should have the as many columns as realized measures needed.}

\item{periods}{a vector of integers indicating over how days the realized measures in the model should be aggregated. By default  periods = c(1,5,22), which corresponds to one day, one week and one month respectively. This default is in line with Andersen et al. (2007).}

\item{periodsJ}{a vector of integers indicating over what time periods the jump components in the model should be aggregated. By default periodsJ = c(1,5,22), which corresponds to one day, one week and one month respectively.}

\item{periodsQ}{a vector of integers indicating over what time periods the realized quarticity in the model should be aggregated. By default periodsQ = c(1,5,22), which corresponds to one day, one week and one month respectively.}

\item{leverage}{a vector of integers indicating over what periods the negative returns should be aggregated.
See Corsi and Reno (2012) for more information. By default leverage = NULL and the model assumes the absence of a  leverage effect. Set leverage = c(1,5,22) to mimic the analysis in Corsi and Reno (2012).}

\item{RVest}{a character vector with one, two, or three elements. The first element always refers to the name of the function to estimate the daily integrated variance (non-jump-robust).
The second and third element depends on which type of model is estimated: If type = "HARRVJ", type = "HARRVCJ", type = "HARRVQJ" the second element refers to the name of the function to estimate the continuous component of daily volatility (jump robust). If type = "HARRVQ", the second element refers to the name of the function used to estimate the integrated quarticity.
If type = "HARRVQJ" the third element always refers to the name of the function used to estimate integrated quarticity.
By default RVest = c("rCov","rBPCov","rQuar"), i.e. using the Realized Volatility, Realized Bi-Power Variance, and Realized Quarticity.}

\item{type}{a string referring to the type of HAR model you would like to estimate. By default type = "HARRV", the most basic model. Other valid options are type = "HARRVJ", type = "HARRVCJ", type = "HARRVQ", type = "HARRVQJ", type = "CHARRV", or type = "CHARRVQ".}

\item{inputType}{a string denoting if the input data consists of realized measures or high-frequency returns, default "RM" is the only way to denote realized measures and everything else denotes returns.}

\item{jumptest}{the function name of a function used to test whether the test statistic which determines whether the jump variability is significant that day. By default jumptest = "ABDJumptest", hence using the test statistic in Equation or Equation (18) of Andersen et al. (2007).}

\item{alpha}{a real indicating the confidence level used in testing for jumps. By default alpha = 0.05.}

\item{h}{an integer indicating the number over how many days the dependent variable should be aggregated.
By default, h=1, i.e. no aggregation takes place, you just model the daily realized volatility.}

\item{transform}{optionally a string referring to a function that transforms both the dependent and explanatory variables in the model. By default transform=NULL, so no transformation is done. Typical other choices in this context would be "log" or "sqrt".}

\item{...}{extra arguments for jump test.}
}
\value{
The function outputs an object of class \code{harModel} and \code{\link{lm}} (so \code{harModel} is  a subclass of \code{\link{lm}}).
}
\description{
Function returns the estimates for the Heterogeneous Autoregressive model
for Realized volatility discussed in Andersen et al. (2007) and Corsi (2009).
This model is mainly used to forecast the next days'volatility based on the high-frequency returns of the past. Consult the vignette for more information.
}
\examples{
##### Example 1: HARRVCJ #####
dat <- sample_5minprices_jumps$stock1
dat <- makeReturns(dat) #Get the high-frequency return data

x <- harModel(dat, periods = c(1,5,10), periodsJ = c(1,5,10),
             RVest = c("rCov","rBPCov"),
             type = "HARRVCJ",transform = "sqrt", inputType = "returns")
# Estimate the HAR model of type HARRVCJ
class(x)
x
# plot(x)
predict(x)


##### Example 2: HARRV #####
# Forecasting daily Realized volatility for the S&P 500 using the basic harModel: HARRV
library(xts)
RV_SP500 <- as.xts(realized_library$rv5, order.by = realized_library$date)

x <- harModel(data = RV_SP500 , periods = c(1,5,22), RVest = c("rCov"),
              type = "HARRV", h = 1, transform = NULL, inputType = "RM")
class(x)
x
summary(x)
plot(x)
predict(x)


##### Example 3: HARRVQ #####
dat <- sample_5minprices_jumps$stock1
dat <- makeReturns(dat) #Get the high-frequency return data
#
x <- harModel(dat, periods = c(1,5,10), periodsJ = c(1,5,10),
            periodsQ = c(1), RVest = c("rCov", "rQuar"),
              type="HARRVQ", inputType = "returns")
## Estimate the HAR model of type HARRVQ
class(x)
x
# plot(x)
#predict(x)

##### Example 4: HARRVQJ with already computed realized measures #####
dat <- SP500RM[, c("RV", "BPV", "RQ")]
x <- harModel(dat, periods = c(1,5,22), periodsJ = c(1),
              periodsQ = c(1), type = "HARRVQJ")
## Estimate the HAR model of type HARRVQJ
class(x)
x
# plot(x)
predict(x)

##### Example 5: CHARRV with already computed realized measures #####
dat <- SP500RM[, c("RV", "BPV")]

x <- harModel(dat, periods = c(1, 5, 22), type = "CHARRV")
# Estimate the HAR model of type CHARRV
class(x)
x
# plot(x)
predict(x)

##### Example 6: CHARRVQ with already computed realized measures #####
dat <- SP500RM[, c("RV", "BPV", "RQ")]

x <- harModel(dat, periods = c(1,5,22), periodsQ = c(1), type = "CHARRVQ")
# Estimate the HAR model of type CHARRVQ
class(x)
x
# plot(x)
predict(x)

#' ##### Example 7: HARRV #####
# Forecasting weekly Realized volatility for the S&P 500 using the basic harModel: HARRV
library(xts)
RV_SP500 <- as.xts(realized_library$rv5, order.by = realized_library$date)

x <- harModel(data = RV_SP500 , periods = c(1,5,22), RVest = c("rCov"),
              type = "HARRV", h = 5, transform = NULL, inputType = "RM")
class(x)
x
summary(x)
plot(x)
predict(x)


}
\references{
Andersen, T. G., T. Bollerslev, and F. Diebold (2007). Roughing it up: including jump components in the measurement, modelling and forecasting of return volatility. The Review of Economics and Statistics 89, 701-720.
Corsi, F. (2009). A simple approximate long memory model of realized volatility. Journal of Financial Econometrics 7, 174-196.
Corsi, F. and Reno R. (2012). Discrete-time volatility forecasting with persistent leverage effect and the link with continuous-time volatility modeling. Journal of Business and Economic Statistics, forthcoming.
Bollerslev, T., Patton, A., Quaedvlieg, R. 2016,  Exploiting the errors: A simple approach for improved volatility forecasting, Journal of Econometrics, vol.192, issue 1, 1-18.
}
\author{
Jonathan Cornelissen and Kris Boudt
}
\keyword{forecasting}
