% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dethrz_restr.R
\name{det_hrz_restr}
\alias{det_hrz_restr}
\title{Determination of the high-risk zone.}
\usage{
det_hrz_restr(
  ppdata,
  type,
  criterion,
  cutoff,
  hole = NULL,
  integratehole = TRUE,
  obsprobs = NULL,
  obsprobimage = NULL,
  distancemap = NULL,
  intens = NULL,
  nxprob = 0.1,
  covmatrix = NULL,
  returnintens = TRUE
)
}
\arguments{
\item{ppdata}{Observed spatial point process of class ppp.}

\item{type}{Method to use, can be one of \code{"dist"}(method of fixed radius or quantile-based method), or
\code{"intens"}(intensity based method)}

\item{criterion}{criterion to limit the high-risk zone, can be one of
\code{"area"} (giving size of hrz), \code{"indirect"} (giving quantile/alpha depending on type),
or \code{"direct"} (giving radius/threshold c depending on type)}

\item{cutoff}{Value of criterion (area, radius, quantile, alpha or threshold).
Depending on criterion and type.}

\item{hole}{(optional) an object of class \code{owin} representing a region inside the observation window of
the \code{ppdata} where no observations were possible.}

\item{integratehole}{Should the \code{hole} be part of the resulting high-risk zone? Defaults to \code{TRUE}.}

\item{obsprobs}{(optional)  Vector of observation probabilities associated with the observations contained in \code{ppdata}.
Must be given in the same order as the coordinates of the observations. Only meaningful
for the intensity-based method if some observations are located in areas where not all
events can actually be observed. For example, if only one third of the events in a specific region
could be observed, the observation probability of the corresponding observations
is 1/3.}

\item{obsprobimage}{(optional) an object of class \code{im} giving the observation probabilities inside the
observation window. Ranges of the coordinates must equal those of \code{ppdata}.
Only used if \code{obsprobs} is not given.}

\item{distancemap}{(optional) distance map: distance of every pixel to the nearest observation
of the point pattern; only needed for \code{type="dist"}. If not given,
it will be computed by \code{\link[spatstat.geom]{distmap}}.}

\item{intens}{(optional) estimated intensity of the observed process (object of class "im",
see \code{\link[spatstat.explore]{density.ppp}}), only needed for type="intens". If not given,
it will be estimated.}

\item{nxprob}{Probability of having unobserved events.
Default value is 0.1.}

\item{covmatrix}{(optional) Covariance matrix of the kernel of a normal distribution, only needed for
\code{type="intens"} if no intensity is given. If not given, it will be estimated.}

\item{returnintens}{Should the image of the estimated intensity be returned? Defaults to \code{TRUE}.}
}
\value{
An object of class "\code{highriskzone}", which is a list of
   \item{ typehrz, criterion, cutoff, nxprob}{ see arguments}
   \item{ zone }{ Determined high-risk zone: Object of class "owin" based on a binary mask.
                  See \code{\link[spatstat.geom]{owin}}. }
   \item{ threshold }{ determined threshold. If type = "dist" and criterion = "direct" it is the specified radius.
   If criterion = "indirect" or "area" the determined radius used to construct a risk zone fulfilling the specified criterion 
   and cutoff. If type = "dist" it is the specified or calculated threshold c, the maximum intensitiy of unexploded bombs 
   outside the risk zone.}
   \item{ calccutoff }{ determined cutoff-value. For type="dist" and criterion="area", this is the
quantile of the nearest-neighbour distance. For type="intens" and criterion="area" or "direct", it is the failure
probability alpha. For all other criterions it is NA.}
   \item{ covmatrix }{ If not given (and \code{type="intens"}), it is estimated. See \code{\link[ks]{Hscv}}.}
   \item{ estint }{ Estimated intensity. See \code{\link[spatstat.explore]{density.ppp}}.}
}
\description{
\code{det_hrz_restr} determines the high-risk zone through the method of fixed radius
(type = "dist" and criterion = "direct"), the quantile-based method (type = "dist" and
criterion = "area"/"indirect") and the intensity-based method (type =  "intens").
Restriction areas can be taken into account.
}
\details{
Used in functions eval_method, sim_clintens, sim_intens.
\cr
This function contains the same functionalities as \code{\link[highriskzone]{det_hrz}}.
In addition, it offers the possibility to take into account so-called restriction areas. This is relevant in
situations where the observed point pattern \code{ppdata} is incomplete. If it is known that no observations
can be made in a certain area (for example because of water expanses),
this can be accounted for by integrating a hole in the observation window.
The shape and location of the hole is given by \code{hole}, whereas \code{integratehole} is used to state
whether the hole is to become part of the resulting high-risk zone.
This may also be a reasonable approach if only few observations could be made in a certain area.
Another approach consists in weighting the observed events with their reciprocal observation probability when
estimating the intensity. To do so, the observation probability can be specified by using \code{obsprobs} (value of the
observation probability for each event) or \code{obsprobsimage} (image of the observation probability). Note that the
observation probability may vary in space.
\cr
If there are no restriction areas in the observation window, \code{\link[highriskzone]{det_hrz}}
can be used instead.
\cr
Note that for \code{criterion = "area"}, \code{cutoff} specifies the area of the high-risk zone outside the hole. If
\code{integratehole = TRUE}, the area of the resulting high-risk zone will exceed \code{cutoff}.


For further information, Mahling et al. (2013) and Mahling (2013), Chapters 4 and 8 and Appendix A (References).
}
\examples{
set.seed(1211515)
data(craterA)
#change npixel = 100 to 1000 to get a nicer picture
spatstat.geom::spatstat.options(npixel=100)
# reduce number of observations for faster computation
craterA <- craterA[sample(1:craterA$n, 150)]
# define restriction area
restrwin <- spatstat.geom::owin(xrange=craterA$window$xrange, yrange=craterA$window$yrange,
                           poly=list(x=c(1500, 1500, 2000, 2000), y=c(2000, 1500, 1500, 2000)))

# create image of observation probability (30\% inside restriction area)
wim <- spatstat.geom::as.im(craterA$window, value=1)
rim <- spatstat.geom::as.im(restrwin, xy=list(x=wim$xcol, y=wim$yrow))
rim$v[is.na(rim$v)] <- 0
oim1 <- spatstat.geom::eval.im(wim - 0.7 * rim)
# determine high-risk zone by weighting the observations
hrzi1 <- det_hrz_restr(ppdata=craterA, type = "intens", criterion = "indirect",
                       cutoff = 0.4, hole=NULL, obsprobs=NULL, obsprobimage=oim1, nxprob = 0.1)

# determine high-risk zone by accounting for a hole
hrzi2 <- det_hrz_restr(ppdata=craterA, type = "intens", criterion = "indirect",
                       cutoff = 0.4, hole=restrwin, obsprobs=NULL, obsprobimage=NULL, nxprob = 0.1)
}
\seealso{
\code{\link[spatstat.geom]{distmap}}, \code{\link[spatstat.geom]{eval.im}}, \code{\link[spatstat.geom]{owin}}
}
