% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelbuilding.R
\name{emulator_from_data}
\alias{emulator_from_data}
\title{Generate Emulators from Data}
\usage{
emulator_from_data(
  input_data,
  output_names,
  ranges,
  input_names = names(ranges),
  beta,
  u,
  c_lengths,
  funcs,
  deltas,
  ev,
  quadratic = TRUE,
  beta.var = FALSE,
  adjusted = TRUE,
  discrepancies = NULL,
  has.hierarchy = FALSE,
  verbose = interactive(),
  na.rm = FALSE,
  check.ranges = FALSE,
  corr_name = "exp_sq",
  targets = NULL,
  ...
)
}
\arguments{
\item{input_data}{Required. A data.frame containing parameter and output values.}

\item{output_names}{Required. A character vector of output names.}

\item{ranges}{A named list of input parameter ranges.}

\item{input_names}{The names of the parameters (if \code{ranges} is not provided).}

\item{beta}{A list of regression coefficients for each output.}

\item{u}{A list of \code{\link{Correlator}} objects for each output.}

\item{c_lengths}{A list of correlation lengths for each output.}

\item{funcs}{A list of regression functions for each output.}

\item{deltas}{Nugget terms for each correlation structure.}

\item{ev}{Estimates of ensemble variability for each output.}

\item{quadratic}{Should a quadratic or linear fit be found?}

\item{beta.var}{Should regression coefficient uncertainty be included?}

\item{adjusted}{Are the raw emulators wanted, or Bayes Linear updated ones?}

\item{discrepancies}{Any internal or external discrepancies in the model.}

\item{has.hierarchy}{For hierarchical emulators, this will be TRUE.}

\item{verbose}{Should status updates be printed?}

\item{na.rm}{If NAs exist in the dataset, should those rows be removed?}

\item{check.ranges}{Should the ranges be modified in light of the data provided?}

\item{corr_name}{What correlation function to use. Defaults to exp_sq}

\item{targets}{If provided, outputs are checked for over/underestimation}

\item{...}{Any additional parameters (eg for custom correlation functions)}
}
\value{
A list of \code{\link{Emulator}} objects.
}
\description{
Given data from simulator runs, generates a set of univariate \code{\link{Emulator}} objects,
one for each output.
}
\details{
Many of the parameters that can be passed to this function are optional: the minimal operating
example requires \code{input_data}, \code{output_names}, and one of \code{ranges} or
\code{input_names}. If \code{ranges} is supplied, the input names are generated from that
list, data.frame, or data.matrix; if only \code{input_names} is specified, then the ranges
are assumed to be [-1, 1] for every input.

The ranges can be provided in alternative ways: either as a named list of length-2 numeric
vectors (corresponding to the maximum and minimum for each parameter); as a data.frame with
2 columns where each row corresponds to a parameter; or as a data.matrix defined similarly
as the data.frame. In the cases where the ranges are provided as a data.frame or a data.matrix,
the \code{row.names} of the data object must be provided, corresponding to the names of the
parameters.

If the minimum information is provided, then an emulator is fitted as follows. The basis
functions and associated regression coefficients are generated using \code{step} and \code{lm}
up to a desired order (default 2, determined by \code{quadratic}). These regression parameters
are assumed to be `known' unless \code{beta.var = TRUE}, in which case the derived parameter
variance is taken from the model fit too (and the regression coefficients themselves can
be modified by the maximum likelihood estimate performed below).

The correlation function c(x, x') is assumed to be \code{\link{exp_sq}} and a corresponding
\code{\link{Correlator}} object is created. The hyperparameters of the correlation structure
are determined using a combination of maximum likelihood estimation and restriction to a
'sensible' range of values, to avoid the correlation length tending to 0 or very large values.
This determines the variance \code{sigma^2}, correlation length \code{theta}, any other
hyperparameters (eg \code{nu} for the matern correlation function), and nugget
term \code{delta}. The hyperparameter priors can be overridden either by directly specifying
them using the \code{c_lengths} argument, or by supplying ranges to the \code{theta_ranges}
argument. Examples of this customisation can be found in the examples to this function.

If \code{ev} is provided, then the ensemble variability is taken into account in the
determination of the nugget term via a two-stage training process.

Some rudimentary data handling functionality is available but should be approached with
caution. The \code{na.rm} option will strip out rows of the training data that have NA values
in them; this of course may leave too few points to train to, and any consistent occurrence
of NAs in model data should be investigated. The \code{check.ranges} option allows a
redefinition of the ranges of the input parameters for emulator training; this is a common
practice in later waves in order to maximise the predictive power of the emulators, but should
only be used here if one is sure that the training set is representative of (and certainly
spanning) the full minimum enclosing hyperrectangle.
}
\examples{
# Use the SIRSample training dataset as an example.
ranges <- list(aSI = c(0.1, 0.8), aIR = c(0, 0.5), aSR = c(0, 0.05))
out_vars <- c('nS','nI','nR')
ems_linear <- emulator_from_data(SIRSample$training, out_vars, ranges, quadratic = FALSE)
ems_linear # Printout of the key information.

\donttest{

  ems_quad <- emulator_from_data(SIRSample$training, out_vars, ranges)
  ems_quad # Now includes quadratic terms (but only where needed)

  ems_unadjusted <- emulator_from_data(SIRSample$training, out_vars, ranges, adjusted = FALSE)
  ems_unadjusted # Looks the same as ems_quad, but the emulators are not BL adjusted

  # Reproduce the linear case, but with slightly changed beta values
  basis_f <- list(
   c(function(x) 1, function(x) x[[1]], function(x) x[[2]], function(x) x[[3]]),
   c(function(x) 1, function(x) x[[1]], function(x) x[[2]]),
   c(function(x) 1, function(x) x[[1]], function(x) x[[2]])
  )
  beta_vals <- list(
   list(mu = c(300, -260, 220, -120)),
   list(mu = c(120, 110, -260)),
   list(mu = c(580, 160, 130))
  )
  ems2 <- emulator_from_data(SIRSample$training, out_vars, ranges,
                             funcs = basis_f, beta = beta_vals)
  ems2
  # Custom correlation functions
  corr_structs <- list(
    list(sigma = 83, corr = Correlator$new('exp_sq', list(theta = 0.5), nug = 0.1)),
    list(sigma = 95, corr = Correlator$new('exp_sq', list(theta = 0.4), nug = 0.25)),
    list(sigma = 164, corr = Correlator$new('exp_sq', list(theta = 0.2), nug = 0.45))
  )
  ems3 <- emulator_from_data(SIRSample$training, out_vars, ranges,
                             u = corr_structs)
  # Using alternative correlation functions and c_lengths
  # Allow code to choose hyperparameters
  ems_matern <- emulator_from_data(SIRSample$training, out_vars, ranges,
                                   corr_name = 'matern')
  # Providing hyperparameters to the function directly, via c_lengths
  matern_hp <- list(list(theta = 0.8, nu = 1.5), list(theta = 0.6, nu = 2.5),
   list(theta = 1.2, nu = 0.5))
  ems_matern2 <- emulator_from_data(SIRSample$training, out_vars, ranges,
                                    corr_name = 'matern', c_lengths = matern_hp)
  # If only one set of hyperparameters are provided to c_lengths, they are used for all
  ems_matern3 <- emulator_from_data(SIRSample$training, out_vars, ranges,
                                    corr_name = 'matern', c_lengths = matern_hp[[1]])
  # "Custom" correlation function with user-specified ranges: gamma exponential
  # 'gamma_exp' can be substituted for any correlation function - see Correlator documentation
  ems_gamma <- emulator_from_data(SIRSample$training, out_vars, ranges,
                                    corr_name = 'gamma_exp',
                                    theta_ranges = list(gamma = c(0.01, 2), theta = c(1/3, 2)))
}

}
