#'Converts the HTTK-Pop population data table to a table of the parameters 
#'needed by HTTK, for a specific chemical.
#'
#'Takes the data table generated by \code{httkpop_generate}, and converts it to
#'the corresponding table of HTTK model parameters for a specified chemical and
#'HTTK model.
#'
#'@export
#'
#'@param indiv_dt A data table generated by \code{httkpop_generate()}.
#'@param chemcas The CAS number of one of the HTTK chemicals (see 
#'  \code{\link[httk]{get_cheminfo}}).
#'@param model One of the HTTK models: "1compartment", "3compartmentss", 
#'  "3compartment", or "pbtk".
#'@param poormetab TRUE (include poor metabolizers) or FALSE (exclude poor 
#'  metabolizers)
#'@param fup.censor TRUE (draw \code{Funbound.plasma} from a censored 
#'  distribution) or FALSE (draw \code{Funbound.plasma} from a non-censored 
#'  distribution)
#'@param sigma.factor The coefficient of variation to use for \code{Clint} and 
#'  \code{Funbound.plasma} distributions. Default value is 0.3.
#'@param Clint.vary TRUE (sample \code{Clint} values) or FALSE (hold 
#'  \code{Clint} fixed). Default TRUE. If \code{Clint.vary} is FALSE, then 
#'  \code{poormetab} will have no effect.
#'@param lod The average limit of detection for Funbound.plasma. if 
#'  \code{fup.censor == TRUE}, the \code{Funbound.plasma} distribution will be 
#'  censored below \code{lod/2}. Default value is 0.01.
#'  
#'@return A data.table whose columns correspond to the parameters of the HTTK 
#'  model specified in \code{model}, and whose rows correspond to the 
#'  individuals (rows) of \code{indiv_dt}.                                                                               
#'  
#'  @examples 
#'  set.seed(42)
#'  indiv_examp <- httkpop_generate(method="d", 
#'  nsamp=100)
#'  httk_param <- get_httk_param(indiv_dt=indiv_examp, 
#'  chemcas="80-05-7", 
#'  model="1compartment", 
#'  poormetab=TRUE, 
#'  fup.censor=TRUE)

get_httk_params <- function(indiv_dt,
                            chemcas,
                            model,
                            poormetab,
                            fup.censor,
                            sigma.factor=0.3,
                            Clint.vary=TRUE,
                            lod=0.01,
                            adjusted.Funbound.plasma=T,
                            regression=T,
                            well.stirred.correction=T,
                            restrictive.clearance=T){
  
  #First convert to physiological parameters used by HTTK
  indiv_bio <- httkpop_bio(indiv_dt = indiv_dt)
  
  #Next add chemical-specific Funbound.plasma and CLint values
  #Just cbind them together for now
  indiv_fc <- cbind(indiv_bio,
                    draw_fup_clint(this.chem=chemcas,
               nsamp=nrow(indiv_bio),
               sigma.factor=sigma.factor,
               poormetab=poormetab,
               fup.censor=fup.censor,
               Clint.vary=Clint.vary,
               lod=lod,
               adjusted.Funbound.plasma=adjusted.Funbound.plasma))
  
  #Next convert the whole thing to the HTTK parameters for a specified model
  indiv_httk <- convert_httk(indiv.model.bio=indiv_fc, 
                 model=model,
                 this.chem=chemcas,
                 adjusted.Funbound.plasma=adjusted.Funbound.plasma,regression=regression,
                 well.stirred.correction=well.stirred.correction,
                 restrictive.clearance=restrictive.clearance)
  
  return(indiv_httk)
}