% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lat_long2utm.R
\name{lat_long2utm}
\alias{lat_long2utm}
\title{Conversion of Latitude/Longitude to UTM Coordinates}
\source{
\enumerate{
   \item r - Converting latitude and longitude points to UTM - Stack Overflow answered and edited by Teodor Ciuraru on Feb 17 2018. See \url{https://stakoverflow.com/questions/18639967/converting-latitude-and-longitude-points-to-utm}.
   \item r - Converting latitude and longitude points to UTM - Stack Overflow answered by Josh O'Brien on Sep 5 2013 and edited by Josh O'Brien on Feb 21 2014. See \url{https://stakoverflow.com/questions/18639967/converting-latitude-and-longitude-points-to-utm}.
   \item r - data.table alternative for dplyr mutate? - Stack Overflow answered by Arun on Aug 30 2015. See \url{https://stackoverflow.com/questions/29583665/data-table-alternative-for-dplyr-mutate}.
   \item database design - What is the maximum length of latitude and longitude? - Stack Overflow answered by JasonM1 on May 24 2013 and edited by JasonM1 on Jul 16 2019. See \url{https://stackoverflow.com/questions/15965166/what-is-the-maximum-length-of-latitude-and-longitude}.
   \item r - How to not run an example using roxygen2? - Stack Overflow answered and edited by samkart on Jul 9 2017. (Also see the additional comments in response to the answer.) See \url{https://stackoverflow.com/questions/12038160/how-to-not-run-an-example-using-roxygen2}.
   \item devtools - Issues in R package after CRAN asked to replace dontrun by donttest - Stack Overflow answered by Hong Ooi on Sep 1 2020. (Also see the additional comments in response to the answer.) See \url{https://stackoverflow.com/questions/63693563/issues-in-r-package-after-cran-asked-to-replace-dontrun-by-donttest}.
   \item Latitude Longitude Coordinates to State Code in R - Stack Overflow answered by Josh O'Brien on Jan 6 2012 and edited by Josh O'Brien on Jun 18, 2020. See \url{https://stackoverflow.com/questions/8751497/latitude-longitude-coordinates-to-state-code-in-r}.
}
}
\usage{
lat_long2utm(
  latitude,
  longitude,
  units = c("us-ft", "ft", "m"),
  output = c("basic", "table")
)
}
\arguments{
\item{latitude}{numeric vector (or character vector with numbers only) that
contains the latitude as a decimal degree}

\item{longitude}{numeric vector (or character vector with numbers only) that
contains the longitude as a decimal degree}

\item{units}{character vector that contains the system of units (options are
\code{survey_ft} (United States Customary System) {US survey foot},
\code{foot}, or \code{meters} (International System of Units) {meters}}

\item{output}{character vector that contains basic for the default result
using a \code{\link[base]{list}} or table for the result as a \code{\link[data.table]{data.table}}}
}
\value{
the UTM zone along with the UTM Easting and Northing coordinates (in
   the requested unit) as either a list or a data.table
}
\description{
Takes latitude/longitude coordinates (as character or numeric vectors) and
transforms them into their respective UTM Easting and Northing coordinates
(with units of US Survey foot, foot, or meters) & UTM Zone.
}
\note{
Remember: Latitude coordinates signify North (N) or South (S) while
longitude coordinates signify East (E) and West (W). It is customary to
denote West longitude coordinates and South latitude coordinates as negative
(-).

Stack Overflow user contributions are "licensed under CC BY-SA 3.0 with
attribution required." [Stack Overflow Reference] I have decided to make my
adaptions to the Stack Overflow user contributions as CC BY-SA 4.0 thereby
enabling me to license my adaptions to the aforementioned code as GPLv3.
[Creative Commons References]
}
\examples{

# Example 1
# Test location from TGRN Reference Manual with NCAT
# using the 1983 (1995) DATUM
# GPS 1 is the station name with these coordinates
# latitude (North) = 36 22 6.43923
# longitude (West) = 82 10 46.87679

install.load::load_package("iemisc", "sp")

lats <- as.numeric(char2dms("36d22'6.43923\"N"))
lats

longs <- as.numeric(char2dms("82d10'46.87679\"W"))
longs

latsc <- as.character(lats)
latsc

longsc <- as.character(longs)
longsc

lat_long2utm(latsc, longsc, units = "m", output = "basic")

lat_long2utm(latsc, longsc, units = "m", output = "table")

lat_long2utm(lats, longs, units = "m", output = "basic")

lat_long2utm(lats, longs, units = "m", output = "table")

# From https://www.ngs.noaa.gov/NCAT/
# Latitude: 36.3684553416667
# Longitude: -82.1796879972222
# UTM Northing (m): 4,025,462.877
# UTM Easting (m): 394,172.067
# USNG: 17SLA9417225462



# Example 2
# Test against Grid [Reference: National Geospatial-Intelligence Agency Office of Geomatics]

library("iemisc")

lat_long2utm("80", "-179", units = "m", output = "basic") # = 1X

lat_long2utm("-80", "-179", units = "m", output = "basic") # = 1C



# Example 3
# Test with world cities

\donttest{
# See Source 5 and Source 6

install.load::load_package("iemisc", "maps", "rando", "utils", "data.table")

import::from(sampler, rsamp)

data(world.cities) # from maps

set_n(200) # makes the example reproducible

wc <- rsamp(world.cities, 2, over = 0, rep = FALSE)
wc

wcutm1 <- lat_long2utm(wc$lat[1], wc$long[1], units = "m", output = "table")
wcutm1

wcutm2 <- lat_long2utm(wc$lat[2], wc$long[2], units = "m", output = "table")
wcutm2

l <- list(wcutm1, wcutm2)
ll <- rbindlist(l)

wc_utm <- setDT(cbind(wc, ll))
wc_utm
}



# Example 4
# Test with 2 Web sites

library("iemisc")

latlong1 <- lat_long2utm(6.32, 7.41, units = "m", output = "table")
latlong1

latlong2 <- lat_long2utm(44.47, 19.81, units = "m", output = "table")
latlong2



# Results from https://www.latlong.net/lat-long-utm.html
# Latitude: 6.32
# Longitude: 7.41
# UTM Easting: 324118.76
# UTM Northing: 698846.97
# UTM Zone: 32N

# Latitude: 44.47
# Longitude: 19.81
# UTM Easting: 405349.04
# UTM Northing: 4924765.48
# UTM Zone: 34T


# Results from https://www.ngs.noaa.gov/NCAT/
# Latitude: 6.32
# Longitude: 7.41
# UTM Northing (m): 698,846.969
# UTM Easting (m): 324,118.758
# USNG: 32NLM2411898846

# Latitude: 44.47
# Longitude: 19.81
# UTM Northing (m): 4,924,765.484
# UTM Easting (m): 405,349.043
# USNG: 34TDQ0534924765








}
\references{
\enumerate{
   \item MapTools, 29 May 2016, "More details about UTM grid zones", \url{https://www.maptools.com/tutorials/grid_zone_details}.
   \item Wikimedia Foundation, Inc. Wikipedia, 11 August 2019, "Geographic coordinate system", \url{https://en.wikipedia.org/wiki/Geographic_coordinate_system}.
   \item PROJ 6.2.0 documentation, 28 Oct 2019, "Cartographic projection", \url{https://proj.org/usage/projections.html}.
   \item Wikimedia Foundation, Inc. Wikibooks, 19 August 2018, "PROJ.4", \url{https://en.wikibooks.org/wiki/PROJ.4}.
   \item National Geospatial-Intelligence Agency Office of Geomatics, "Military Grid Reference System (MGRS) Grid Zone Designator (GZD's)", \url{https://vdocuments.net/military-grid-reference-system-mgrs-grid-zone-designator-gzds.html}.
   \item Tennessee Department of Transportation Design Division, Tennessee Department of Transportation Tennessee Geodetic Reference Network (TGRN) Reference Manual Second Edition Issued, page ix, \url{https://www.tn.gov/content/dam/tn/tdot/documents/TgrnComposite.pdf}.
   \item LatLong.net, "Lat Long to UTM Converter", \url{https://www.latlong.net/lat-long-utm.html}.
   \item NOAA’s National Geodetic Survey (NGS), "NGS Coordinate Conversion and Transformation Tool (NCAT)", \url{https://www.ngs.noaa.gov/NCAT/}.
   \item Creative Commons. Weblog Archives, October 8, 2015, "CC BY-SA 4.0 now one-way compatible with GPLv3: The declaration increases interoperability of the commons for games, hardware designs, and more" Posted by mike, \url{https://creativecommons.org/2015/10/08/cc-by-sa-4-0-now-one-way-compatible-with-gplv3/}.
   \item Stack Overflow. Public Network Terms of Service, "6. Content Permissions, Restrictions, and Creative Commons Licensing", \url{https://stackoverflow.com/legal/terms-of-service#licensing}.
}
}
\author{
Irucka Embry, Teodor Ciuraru (Latitude/Longitude to UTM conversion code), and Josh O'Brien (Latitude/Longitude to UTM conversion code)
}
