% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/centrality.R
\name{eigen_centrality}
\alias{eigen_centrality}
\alias{evcent}
\title{Find Eigenvector Centrality Scores of Network Positions}
\usage{
eigen_centrality(
  graph,
  directed = FALSE,
  scale = TRUE,
  weights = NULL,
  options = arpack_defaults
)
}
\arguments{
\item{graph}{Graph to be analyzed.}

\item{directed}{Logical scalar, whether to consider direction of the edges
in directed graphs. It is ignored for undirected graphs.}

\item{scale}{Logical scalar, whether to scale the result to have a maximum
score of one. If no scaling is used then the result vector has unit length
in the Euclidean norm.}

\item{weights}{A numerical vector or \code{NULL}. This argument can be used
to give edge weights for calculating the weighted eigenvector centrality of
vertices. If this is \code{NULL} and the graph has a \code{weight} edge
attribute then that is used. If \code{weights} is a numerical vector then it
used, even if the graph has a \code{weight} edge attribute. If this is
\code{NA}, then no edge weights are used (even if the graph has a
\code{weight} edge attribute. Note that if there are negative edge weights
and the direction of the edges is considered, then the eigenvector might be
complex. In this case only the real part is reported.
This function interprets weights as connection strength. Higher
weights spread the centrality better.}

\item{options}{A named list, to override some ARPACK options. See
\code{\link[=arpack]{arpack()}} for details.}
}
\value{
A named list with components: \item{vector}{A vector containing the
centrality scores.} \item{value}{The eigenvalue corresponding to the
calculated eigenvector, i.e. the centrality scores.} \item{options}{A named
list, information about the underlying ARPACK computation. See
\code{\link[=arpack]{arpack()}} for the details.}
}
\description{
\code{eigen_centrality()} takes a graph (\code{graph}) and returns the
eigenvector centralities of positions \code{v} within it
}
\details{
Eigenvector centrality scores correspond to the values of the first
eigenvector of the graph adjacency matrix; these scores may, in turn, be
interpreted as arising from a reciprocal process in which the centrality of
each actor is proportional to the sum of the centralities of those actors to
whom he or she is connected.  In general, vertices with high eigenvector
centralities are those which are connected to many other vertices which are,
in turn, connected to many others (and so on).  (The perceptive may realize
that this implies that the largest values will be obtained by individuals in
large cliques (or high-density substructures).  This is also intelligible
from an algebraic point of view, with the first eigenvector being closely
related to the best rank-1 approximation of the adjacency matrix (a
relationship which is easy to see in the special case of a diagonalizable
symmetric real matrix via the \eqn{SLS^-1}{$S \Lambda S^{-1}$}
decomposition).)

The adjacency matrix used in the eigenvector centrality calculation assumes
that loop edges are counted \emph{twice}; this is because each loop edge has
\emph{two} endpoints that are both connected to the same vertex, and you
could traverse the loop edge via either endpoint.

In the directed case, the left eigenvector of the adjacency matrix is
calculated. In other words, the centrality of a vertex is proportional to
the sum of centralities of vertices pointing to it.

Eigenvector centrality is meaningful only for connected graphs. Graphs that
are not connected should be decomposed into connected components, and the
eigenvector centrality calculated for each separately. This function does
not verify that the graph is connected. If it is not, in the undirected case
the scores of all but one component will be zeros.

Also note that the adjacency matrix of a directed acyclic graph or the
adjacency matrix of an empty graph does not possess positive eigenvalues,
therefore the eigenvector centrality is not defined for these graphs.
igraph will return an eigenvalue of zero in such cases. The eigenvector
centralities will all be equal for an empty graph and will all be zeros for
a directed acyclic graph. Such pathological cases can be detected by checking
whether the eigenvalue is very close to zero.

From igraph version 0.5 this function uses ARPACK for the underlying
computation, see \code{\link[=arpack]{arpack()}} for more about ARPACK in igraph.
}
\examples{

# Generate some test data
g <- make_ring(10, directed = FALSE)
# Compute eigenvector centrality scores
eigen_centrality(g)
}
\references{
Bonacich, P.  (1987).  Power and Centrality: A Family of
Measures. \emph{American Journal of Sociology}, 92, 1170-1182.
}
\seealso{
Other centrality: 
\code{\link{alpha_centrality}()},
\code{\link{arpack_defaults}},
\code{\link{closeness}()},
\code{\link{diversity}()},
\code{\link{estimate_betweenness}()},
\code{\link{harmonic_centrality}()},
\code{\link{page_rank}()},
\code{\link{power_centrality}()},
\code{\link{spectrum}()},
\code{\link{strength}()},
\code{\link{subgraph_centrality}()}
}
\author{
Gabor Csardi \email{csardi.gabor@gmail.com} and Carter T. Butts
(\url{http://www.faculty.uci.edu/profile.cfm?faculty_id=5057}) for the
manual page.
}
\concept{centrality}
\keyword{graphs}
