# DEPRECATED
pixelplot.mesh = function(...) { stop("function removed from inlabru") }


#' Plot a map using extend of a spatial object
#' 
#' Uses get_map() to query map services like Google Maps for a region centered around
#' the spatial object provided. Then calls ggmap() to plot the map.
#' 
#' @aliases gmap
#' @name gmap
#' @export
#' @param data A Spatial* object.
#' @param ... Arguments passed on to get_map().
#' @return a ggplot object
#' 
#' @example inst/examples/gmap.R

gmap = function(data, ...) {
  data = spTransform(data, CRS("+proj=longlat"))
  df = cbind(coordinates(data), data@data)
  
  # Figure out a sensible bounding box (range of data plus 30%)
  extend = 2.5
  crange = apply(coordinates(data), MARGIN = 2, range)
  lonlim = (extend*(crange[,1] - mean(crange[,1]))) + mean(crange[,1])
  latlim = (extend*(crange[,2] - mean(crange[,2]))) + mean(crange[,2])
  
  # Create map
  requireNamespace("ggmap")
  myMap = ggmap::get_map(c(lonlim[1], latlim[1], lonlim[2], latlim[2]), ...)
  
  # Return map
  ggmap::ggmap(myMap)
}

#' ggplot2 geomes for inlabru related objects
#' 
#' @description 
#' gg is a generic function for generating geomes from various kinds of spatial objects, e.g. Spatial* data,
#' meshes, Raster objects and inla/inlabru predictions. The function invokes particular methods which depend 
#' on the \link{class} of the first argument.
#'
#' @name gg
#' @export
#' @param data an object for which to generate a geom.
#' @param ... Arguments passed on to the geom method.
#' @return The form of the value returned by gg depends on the class of its argument. See the documentation of the particular methods for details of what is produced by that method.
#'
#' @family geomes for inla and inlabru predictions
#' @family geomes for spatial data
#' @family geomes for meshes
#' @family geomes for Raster data
#' 
#' @examples 
#' 
#' # Load Gorilla data
#' 
#' data(gorillas)
#' 
#' # Invoke ggplot and add geomes for the Gorilla nests and the survey boundary
#' 
#' ggplot() + gg(gorillas$boundary) + gg(gorillas$nests)

gg = function(data, ...){UseMethod("gg")}


#' ggplot geom for spatial data
#' 
#' gm is a wrapper for the \link{gg} method. It will take the first argument and transform its
#' coordinate system to latitude and longitude. Thereafter, \link{gg} is called using the transformed
#' data and the arguments provided via \code{...}. gm is intended to replace gg whenever the
#' data is supposed to be plotted over a spatial map generated by \link{gmap}, which only works
#' if the coordinate system is latitude/longitude.
#' 
#' @aliases gm
#' @name gm
#' @export
#' @param data an object for which to generate a geom.
#' @param ... Arguments passed on to \link{gg}.
#' @return The form of the value returned by gm depends on the class of its argument. See the documentation of the particular methods for details of what is produced by that method.
#' @family geomes for inla and inlabru predictions
#' @family geomes for spatial data
#' @family geomes for meshes
#' @family geomes for Raster data
#' 
#' @example inst/examples/gmap.R

gm = function(data, ...) { gg(data, crs = CRS("+proj=longlat"), ...) }


#' Geom for matrix
#' 
#' Creates a tile geom for plotting a matrix
#' 
#' @name gg.matrix
#' @export
#' @import ggplot2
#' @param data A \code{matrix} object.
#' @param mapping a set of aesthetic mappings created by \link{aes} or \link{aes_}. These are passed on to \link{geom_tile}.
#' @param ... Arguments passed on to \link{geom_tile}.
#' @return A \link{geom_tile} with reversed y scale.
#' @family geomes for inla and inlabru predictions
#' 
#' @examples
#' 
#' A = matrix(runif(100), nrow = 10)
#' ggplot() + gg(A)
#' 

gg.matrix = function(data, mapping = NULL, ...) {
  A = as.matrix(data)
  grd = expand.grid(row = 1:nrow(A), column = 1:ncol(A))
  df = data.frame(value = as.vector(A), grd)
  
  dmap = aes_string(x="column", y="row", fill = "value")
  
  if ( !is.null(mapping) ) { dmap = modifyList(dmap, mapping) }
  
  ggp = c(geom_tile(dmap, data = df, ...), scale_y_reverse())
}


#' Geom for data.frame
#' 
#' This geom constructor will simply call \link{gg.prediction} for the data provided.
#' 
#' @name gg.data.frame
#' @export
#' @import ggplot2
#' @param ... Arguments passed on to \link{gg.prediction}.
#' @return Concatenation of a \link{geom_line} value and optionally a \link{geom_ribbon} value.
#' @family geomes for inla and inlabru predictions
#' @example inst/examples/gg.prediction.R

gg.data.frame = function(...){ gg.prediction(...) }


#' Geom for predictions
#' 
#' @description 
#' 
#' This geom serves to visualize \code{prediction} objects which usually results from a call to 
#' \link{predict.bru}. Predictions objects provide summary statistics (mean, median, sd, ...) for
#' one or more random variables. For single variables (or if requested so by setting \code{bar = TRUE}),
#' a boxplot-style geom is constructed to show the statistics. For multivariate predictions the
#' mean of each variable (y-axis) is plotted agains the row number of the varriable in the prediction
#' data frame (x-axis) using \link{geom_line}. In addition, a \link{geom_ribbon} is used to show
#' the confidence interval.
#' 
#' Note: \code{gg.prediction} also understands the format of INLA-style posterior summaries, e.g.
#' \code{fit$summary.fixed} for an inla object \code{fit}
#' 
#' @aliases gg.prediction
#' @name gg.prediction
#' @export
#' @import ggplot2
#' @importFrom utils modifyList
#' @param data A prediction object, usually the result of a \link{predict.bru} call.
#' @param mapping a set of aesthetic mappings created by \link{aes} or \link{aes_}. These are passed on to \link{geom_line}.
#' @param ribbon If TRUE, plot a ribbon around the line based on the upper and lower 2.5 percent quantiles.
#' @param alpha The ribbons numeric alpha level in [0,1].
#' @param bar If TRUE plot boxplot-style summary for each variable.
#' @param ... Arguments passed on to \link{geom_line}.
#' @return Concatenation of a \link{geom_line} value and optionally a \link{geom_ribbon} value.
#' @family geomes for inla and inlabru predictions
#' @example inst/examples/gg.prediction.R

gg.prediction = function(data, mapping = NULL, ribbon = TRUE, alpha = 0.3, bar = FALSE, ...){
  
  if ( all(c("0.025quant", "0.975quant") %in% names(data)) ) {
    names(data)[names(data) == "0.975quant"] = "q0.975"
    names(data)[names(data) == "0.025quant"] = "q0.025"
    names(data)[names(data) == "0.5quant"] = "median"
  }
  lqname = "q0.025"
  uqname = "q0.975"

  
  if ( bar | ( nrow(data) == 1) ) {
    
    sz = 10 # bar width
    med_sz = 4 # median marker size
    
    data = cbind(data.frame(data), 
                 data.frame(variable = rownames(data), 
                            summary = data$mean[1],
                            sdmax = data$mean+data$sd,
                            sdmin = data$mean-data$sd))
    
    geom = c(geom_point(data = data, mapping = aes_string(x = "variable", y = "summary", color = "variable"), shape = 95, size = 0), # Fake ylab
             geom_segment(data = data, mapping = aes_string(y = lqname, yend = uqname, x = "variable", xend = "variable", color = "variable"), size = sz))
    
    # Min and max sample
    if ( all(c("smax", "smax") %in% names(data)) ) {
      geom = c(geom, 
               geom_segment(data = data, mapping = aes_string(y = "smin", yend = "smax", x = "variable", xend = "variable", color = "variable"), size = 1),
               geom_point(data = data, mapping = aes_string(x = "variable", y = "smax", color = "variable"), shape = 95, size = 5),
               geom_point(data = data, mapping = aes_string(x = "variable", y = "smin", color = "variable"), shape = 95, size = 5))
    }

    # Mean and median       
    geom = c(geom,       
             geom_point(data = data, mapping = aes_string(x = "variable", y = "mean"), color = "black", shape = 95, size = sz),
             geom_point(data = data, mapping = aes_string(x = "variable", y = "median"), color = "black", shape = 20, size = med_sz),
             coord_flip())
             
  } else {
    
    if ( "pdf" %in% names(data) ) { 
      y.str = "pdf"
      ribbon = FALSE
    } else if ( "mean" %in% names(data) ){ 
      y.str = "mean" 
    } else if ( "median" %in% names(data) ){
      y.str = "median"
    } else {
      stop("Prediction has neither mean nor median or pdf as column. Don't know what to plot.")
    }
    
    line.map = aes_string(x = names(data)[1], 
                          y = y.str)
    
    ribbon.map = aes_string(x = names(data)[1], 
                            ymin = lqname, 
                            ymax = uqname)
    
    if ( !is.null(mapping) ) { line.map = utils::modifyList(line.map, mapping) }
    
    # Use line color for ribbon filling
    if ( "colour" %in% names(line.map) ) { 
      ribbon.map = modifyList(ribbon.map, aes_string(fill = line.map[["colour"]])) 
      }
    
    geom = geom_line(data = data, line.map, ...)
    if ( ribbon ) {
      geom = c(geom, geom_ribbon(data = data, ribbon.map, alpha = alpha))
    }
  }
  geom
}

#' Geom for SpatialPoints objects
#' 
#' This function coerces the \code{SpatialPoints} into a \code{data.frame} and uses \link{geom_point}
#' to plot the points.
#' 
#' @aliases gg.SpatialPoints
#' @name gg.SpatialPoints
#' @export
#' @import ggplot2
#' @param data A SpatialPoints object.
#' @param mapping Aesthetic mappings created by \link{aes} or \link{aes_} used to update the default
#'                mapping. The default mapping is \code{aes_string(x = coordnames(data)[1], y = coordnames(data)[2])}.
#' @param crs A \link{CRS} object defining the coordinate system to project the data to before plotting.
#' @param ... Arguments passed on to \link{geom_point}.
#' @return A \link{geom_point} return value
#' @family geomes for spatial data
#' 
#' @example inst/examples/gg.spatial.R

gg.SpatialPoints = function(data, mapping = NULL, crs = NULL, ...) {
  
  if ( !is.null(crs) ) { data = spTransform(data, crs) }
  
  df = data.frame(data)
  
  dmap = aes_string(x = coordnames(data)[1], 
                    y = coordnames(data)[2])
  
  if ( !is.null(mapping) ) { dmap = modifyList(dmap, mapping) }
  
  geom_point(data = df, mapping = dmap, ...)
}

#' Geom for SpatialLines objects
#' 
#' @description 
#' 
#' Extracts start and end points of the lines and calls \link{geom_segment} to plot  lines between them.
#' 
#' @aliases gg.SpatialLines
#' @name gg.SpatialLines
#' @export
#' @import ggplot2
#' @param data A SpatialLines object.
#' @param mapping Aesthetic mappings created by \link{aes} or \link{aes_} used to update the default
#'                mapping. The default mapping is \code{aes_string(x = coordnames(data)[1], 
#'                y = coordnames(data)[2], 
#'                xend = paste0("end.", coordnames(data)[1]), 
#'                yend = paste0("end.", coordnames(data)[2]))}.
#' @param crs A \link{CRS} object defining the coordinate system to project the data to before plotting.
#' @param ... Arguments passed on to \link{geom_segment}.
#' @return A \link{geom_segment} return value.
#' @family geomes for spatial data
#' @example inst/examples/gg.spatial.R

gg.SpatialLines = function(data, mapping = NULL, crs = NULL, ...) {
  
  if ( !is.null(crs) ) { data = spTransform(data, crs) }
  
  qq = coordinates(data)
  cnames = coordnames(data)
  if (is.null(cnames)) { cnames = c("x","y") }
  sp = do.call(rbind, lapply(qq, function(k) do.call(rbind, lapply(k, function(x) x[1:(nrow(x)-1),]))))
  ep = do.call(rbind, lapply(qq, function(k) do.call(rbind, lapply(k, function(x) x[2:(nrow(x)),]))))
  colnames(sp) = cnames
  colnames(ep) = paste0("end.", cnames)
  df = data.frame(cbind(sp, ep), data@data)
  
  dmap = aes_string(x = cnames[1], 
                    y = cnames[2], 
                    xend = paste0("end.", cnames[1]), 
                    yend = paste0("end.", cnames[2]))
  
  if ( !is.null(mapping) ) { dmap = modifyList(dmap, mapping) }
  
  geom_segment(data = df, mapping = dmap, ...)  
} 

#' Geom for SpatialPolygons objects
#' 
#' Uses the fortitfy() function to turn the SpatialPolygons objects into a data.frame. Then 
#' calls \link{geom_polygon} to plot the polygons.
#' 
#' 
#' @aliases gg.SpatialPolygons
#' @name gg.SpatialPolygons
#' @export
#' @import ggplot2
#' @param data A SpatialPolygons object.
#' @param mapping Aesthetic mappings created by \link{aes} or \link{aes_} used to update the default
#'                mapping. The default mapping is \code{aes_string(x = "long", y = "lat", group = "group")}.
#' @param crs A \link{CRS} object defining the coordinate system to project the data to before plotting.
#' @param color Filling color for the polygons.
#' @param alpha Alpha level for polygon filling.
#' @param ... Arguments passed on to \link{geom_polygon}.
#' @return A \link{geom_polygon} return value.
#' @family geomes for spatial data
#' @example inst/examples/gg.spatial.R

gg.SpatialPolygons = function(data, mapping = NULL, crs = NULL, color = "black", alpha = NULL, ...) {
  
  if ( !is.null(crs) ) { data = spTransform(data, crs) }
  df = fortify(data)
  dmap = aes_string(x = "long", y = "lat", group = "group")
  
  if (!("alpha" %in% names(dmap)) & is.null(alpha)) { alpha = 0.1 }
  if (!("color" %in% names(dmap)) & is.null(color)) { color = "black" }

  if ( !is.null(mapping) ) { dmap = modifyList(dmap, mapping) }
  
  geom_polygon(data = df, mapping = dmap, alpha = alpha, color = color, ...)
}

#' Geom for SpatialGridDataFrame objects
#' 
#' Coerces input SpatialGridDataFrame to SpatialPixelsDataFrame and calls 
#' gg.SpatialPixelDataFrame to plot it.
#' 
#' @aliases gg.SpatialGridDataFrame
#' @name gg.SpatialGridDataFrame
#' @export
#' @import ggplot2
#' @param data A SpatialGridDataFrame object.
#' @param ... Arguments passed on to gg.SpatialPixelsDataFrame().
#' @return A \link{geom_tile} value.
#' @family geomes for spatial data
#' @example inst/examples/gg.spatial.R

gg.SpatialGridDataFrame = function(data, ...) {
  data = as(data, "SpatialPixelsDataFrame")
  gg(data, ...)
}

#' Geom for SpatialPixelsDataFrame objects
#' 
#' Coerces input SpatialPixelsDataFrame to data.frame and uses \link{geom_tile} to plot it.
#' 
#' @aliases gg.SpatialPixelsDataFrame
#' @name gg.SpatialPixelsDataFrame
#' @export
#' @import ggplot2
#' @param data A SpatialPixelsDataFrame object.
#' @param mapping Aesthetic mappings created by \link{aes} or \link{aes_} used to update the default
#'                mapping. The default mapping is \code{aes_string(x = coordnames(data)[1], 
#'                y = coordnames(data)[2], fill = names(data)[[1]])}.
#' @param alpha Character array identifying the data column used for tile transparency.
#' @param crs A \link{CRS} object defining the coordinate system to project the data to before plotting.
#' @param mask A SpatialPolygon defining the region that is plotted.
#' @param ... Arguments passed on to \link{geom_tile}.
#' @return A \link{geom_tile} return value.
#' @family geomes for spatial data
#' @example inst/examples/gg.spatial.R

gg.SpatialPixelsDataFrame = function(data, 
                                     mapping = NULL,
                                     alpha = NULL, 
                                     crs = NULL,
                                     mask = NULL, ...) {
  
  if ( !is.null(crs) ) { data = spTransform(data, crs) }
  if ( !is.null(mask) ) { data = data[as.vector(!is.na(over(data, mask))), ] }
  
  df <- as.data.frame(data)
  
  dmap = aes_string(x = coordnames(data)[1], 
                    y = coordnames(data)[2], 
                    fill = names(data)[[1]])
  
  if ( !is.null(mapping) ) { dmap = modifyList(dmap, mapping) }
  if ( !is.null(alpha) ) dmap = modifyList(dmap, aes_string(alpha = alpha))
  gm = geom_tile(data = df, mapping = dmap, ...)
  
  # If data is not discrete (factor), add default color palette
  if (!inherits(data[[deparse(dmap$fill)]], "factor")) {
    gm = c(gm, scale_fill_gradientn(colours = bru.pal()))
  }
  gm
}


#' Geom for SpatialPixels objects
#' 
#' Uses \link{geom_point} to plot the pixel centers.
#' 
#' @aliases gg.SpatialPixels
#' @name gg.SpatialPixels
#' @export
#' @import ggplot2
#' @param data A \link{SpatialPixels} object.
#' @param ... Arguments passed on to \link{geom_tile}.
#' @return A \link{geom_tile} return value.
#' @family geomes for spatial data
#' @examples 
#' 
#' # Load Gorilla data
#' 
#' data(gorillas)
#' 
#' # Turn elevation covariate into SpatialPixels
#' pxl = SpatialPixels(SpatialPoints(gorillas$gcov$elevation))
#'
#' # Plot the pixel centers
#' ggplot() + gg(pxl, size = 0.1)

gg.SpatialPixels = function(data, ...) {
  gg(SpatialPoints(data), ...) 
}



#' Geom for inla.mesh objects
#' 
#' @description 
#' 
#' This function extracts the graph of an inla.mesh object and uses \link{geom_line} to visualize
#' the graph's edges. Alternatively, if the \code{color} argument is provided, interpolates the colors
#' across for a set of SpatialPixels covering the mesh area and calls gg.SpatialPixelDataFrame()
#' to plot the interpolation.
#' 
#' @aliases gg.inla.mesh
#' @name gg.inla.mesh
#' @export
#' @import ggplot2
#' @param data An \link[INLA]{inla.mesh} object.
#' @param color A vector of scalar values to fill the mesh with colors. The length of the vector mus correspond to the number of mesh vertices.
#' @param alpha A vector of scalar values setting the alpha value of the colors provided.
#' @param edge.color Color of the mesh edges.
#' @param interior If TRUE, plot the interior boundaries of the mesh.
#' @param int.color Color used to plot the interior boundaries.
#' @param exterior If TRUE, plot the exterior boundaries of the mesh.
#' @param ext.color Color used to plot the interior boundaries.
#' @param crs A \link{CRS} object defining the coordinate system to project the mesh to before plotting.
#' @param nx Number of pixels in x direction (when plotting using the color parameter).
#' @param ny Number of pixels in y direction (when plotting using the color parameter).
#' @param mask A SpatialPolygon defining the region that is plotted.
#' @param ... ignored arguments (S3 generic compatibility).
#' @return \link{geom_line} return values or, if the color argument is used, the values of gg.SpatialPixelDataFrame().
#' @family geomes for meshes
#' @example inst/examples/gg.inla.mesh.R

gg.inla.mesh = function(data, 
                        color = NULL, 
                        alpha = NULL,
                        edge.color = "grey",
                        interior = TRUE, 
                        int.color = "blue", 
                        exterior = TRUE, 
                        ext.color = "black",
                        crs = NULL,
                        mask = NULL,
                        nx = 500, ny = 500,
                        ...) {

  
if ( !is.null(color) ) {
  
  px = pixels(data, nx = nx, ny = ny)
  A = INLA::inla.spde.make.A(data, px)
  px$color = as.vector(A %*% color)
  if ( !is.null(alpha) ) { 
    px$alpha = as.vector(A %*% alpha)
    gg = gg(px, mask = mask, alpha = "alpha")
  } else {
    gg = gg(px, mask = mask)
  }
  
  return(gg)
  
} else {
  if ( data$manifold == "S2" ) { stop("Geom not implemented for spherical meshes (manifold = S2)" ) }
  if ( !is.null(crs) ) { data = INLA::inla.spTransform(data, CRSobj = crs)}
  
  df = rbind(data.frame(a=data$loc[data$graph$tv[,1],c(1,2)],b=data$loc[data$graph$tv[,2],c(1,2)]),
             data.frame(a=data$loc[data$graph$tv[,2],c(1,2)],b=data$loc[data$graph$tv[,3],c(1,2)]),
             data.frame(a=data$loc[data$graph$tv[,1],c(1,2)],b=data$loc[data$graph$tv[,3],c(1,2)]))
  
  colnames(df) = c("x","y","xend","yend")
  mp = aes_string(x = "x",y = "y", xend = "xend", yend = "yend")
  msh = geom_segment(data = df, mapping = mp, color = edge.color)
  
  # Outer boundary
  if ( exterior ) {
    df = data.frame(data$loc[data$segm$bnd$idx[,1],1:2], data$loc[data$segm$bnd$idx[,2],1:2])
    colnames(df) = c("x","y","xend","yend")
    bnd = geom_segment(data = df, mapping = mp, color = ext.color)
  } else { bnd = NULL }
  
  if ( interior ) {
    # Interior boundary
    df = data.frame(data$loc[data$segm$int$idx[,1],1:2], data$loc[data$segm$int$idx[,2],1:2])
    colnames(df) = c("x","y","xend","yend")
    if ( nrow(df) == 0 ) { int = NULL } else {
      int = geom_segment(data = df, mapping = mp, color = int.color)
    }
  } else { int = NULL }
  
  # Return combined geomes
  c(msh, bnd, int)
} 
  
  
}


#' Geom for inla.mesh.1d objects
#' 
#' This function generates a \link{geom_point} object showing the knots (vertices)
#' of a 1D mesh.
#' 
#' @aliases gg.inla.mesh.1d
#' @name gg.inla.mesh.1d
#' @export
#' @import ggplot2
#' @param data An inla.mesh.1d object.
#' @param mapping aesthetic mappings created by \link{aes} or \link{aes_}. These are passed on to \link{geom_point}.
#' @param y Single or vector numeric defining the y-coordinates of the mesh knots to plot.
#' @param shape Shape of the knot markers.
#' @param ... parameters passed on to \link{geom_point}.
#' @return An obbject generated by \link{geom_point}.
#' @family geomes for meshes
#' 
#' @examples
#' 
#' \donttest{
#' # Load INLA
#' 
#' library(INLA)
#' 
#' # Create a 1D mesh
#' 
#' mesh = inla.mesh.1d(seq(0,10,by=0.5))
#' 
#' # Plot it
#' 
#' ggplot() + gg(mesh)
#' 
#' # Plot it using a different shape and size for the mesh nodes
#' 
#' ggplot() + gg(mesh, shape = "|", size = 5)
#' }

gg.inla.mesh.1d = function(data, mapping = aes_string("x","y"), y = 0, shape = 4, ...) {
  
    df = data.frame(x = data$loc, y = y)
    geom_point(data = df, mapping = mapping, shape = shape, ...)

}



#' Geom for RasterLayer objects
#' 
#' This function takes a RasterLayer object, converts it into a SpatialPixelsDataFrame and
#' uses \link{geom_tile} to plot the data.
#' 
#' @aliases gg.RasterLayer
#' @name gg.RasterLayer
#' @export
#' @import ggplot2
#' @param data A RasterLayer object.
#' @param mapping aesthetic mappings created by \link{aes} or \link{aes_}. These are passed on to \link{geom_tile}.
#' @param ... Arguments passed on to \link{geom_tile}.
#' @return An object returned by \link{geom_tile}
#' @family geomes for Raster data 
#' 
#' @examples
#' 
#' # Load Gorilla data
#' data("gorillas", package = "spatstat.data")
#' 
#' # Convert elevation covariate to RasterLayer
#'
#' library(raster)
#' elev = as(gorillas.extra$elevation, "RasterLayer")
#'
#' # Plot the elevation
#'
#' ggplot() + gg(elev)

gg.RasterLayer = function(data, mapping = aes_string(x="x", y="y", fill = "layer"), ...) {
  
  requireNamespace("raster")
  spdf <- as(data, "SpatialPixelsDataFrame")
  df <- as.data.frame(spdf)
  # head(r.df)
  # g <- ggplot(r.df, aes(x=x, y=y)) + geom_tile(aes(fill = layer)) + coord_equal()
  geom_tile(data = df, mapping = mapping,...)
}

#' Plot method for posterior marginals estimated by \link{bru}
#' 
#' @description
#' 
#' \link{bru} uses \link[INLA]{inla} to fit models. The latter estimates the posterior densities of
#' all random effects in the model. This function serves to access and plot the posterior
#' densities in a convenient way.
#' 
#' @method plot bru
#' @export
#' @import ggplot2
#' @param x a fitted \link{bru} model.
#' @param ... A character naming the effect to plot, e.g. "Intercept".
#' @return an object of class \code{gg}
#' 
#' 
#' @examples
#' 
#' \dontrun{
#' 
#' # Generate some data and fit a simple model
#' input.df <- data.frame(x=cos(1:10))
#' input.df <- within(input.df, y <- 5 + 2*cos(1:10) + rnorm(10, mean=0, sd=0.1))
#' fit <- bru(y ~ x, "gaussian", input.df)
#' summary(fit)
#' 
#' # Plot the posterior of the model's x-effect
#' plot(fit, "x")
#' 
#' }
#' 

plot.bru = function(x, ...) {
  plotmarginal.inla(x, ...)
}

#' Plot prediction using ggplot2
#' 
#' Generates a base ggplot2 using \link{ggplot} and adds a geom for input \code{x} using \link{gg}.
#' 
#' @export
#' @method plot prediction
#' @import ggplot2
#' @param x a prediction object.
#' @param y Ignored argument but required for S3 compatibility.
#' @param ... Arguments passed on to \link{gg.prediction}.
#' @return an object of class \code{gg}
#' @example inst/examples/gg.prediction.R

plot.prediction = function(x, y = NULL, ...) {
  ggplot() + gg(x, ...)
}

plot.prediction_old = function(..., property = "median") {
  args = list(...)
  pnames = sapply(substitute(list(...))[-1], deparse)
  
  ggopts = attr(args[[1]], "misc")
  data = args[[1]]
  
  if ( attr(data,"type") == "full" ) {
    
    df = do.call(rbind, lapply(1:length(args), function(k) {
      apr = approx(args[[k]]$x, args[[k]]$y, xout = seq(range(args[[k]]$x)[1],range(args[[k]]$x)[2],length.out = 1000))
      data.frame(effect = pnames[[k]], x = apr$x, y = 0.1*apr$y/max(apr$y))}))

    qtl = do.call(rbind, lapply(1:length(args), function(k) 
      data.frame(y = args[[k]]$quantiles,
                 yend = args[[k]]$quantiles,
                 x = k, xend = k + 0.5,
                 effect = pnames[[k]])))
    
    expec = do.call(rbind, lapply(1:length(args), function(k) 
      data.frame(y = args[[k]]$mean,
                 yend = args[[k]]$mean,
                 x = k, xend = k - 0.28,
                 sdy = args[[k]]$sd,
                 cvy = args[[k]]$cv,
                 effect = pnames[[k]])))
    
    sdev = do.call(rbind, lapply(1:length(args), function(k) 
      data.frame(y = args[[k]]$mean + args[[k]]$sd,
                 yend = args[[k]]$mean - args[[k]]$sd,
                 x = k - 0.28, xend = k - 0.28,
                 effect = pnames[[k]])))

    jit = do.call(rbind, lapply(1:length(args), function(k)
      data.frame(y = INLA::inla.rmarginal(5000,args[[k]]),
                 n = 500,
                 effect = pnames[[k]])))
    
    txt_size = (5/14) * theme_get()$text$size
    
    # Function for formating numbers
    fmt = function(x) {
      th = 1E3
      if (abs(x)<th & abs(x)>0.01) {
        # sprintf("%.2f",x)
        as.character(signif(x,4))
      } else {
        sprintf("%.2E",x)
      }
    }
    
    # Workaround for non-visible bindings
    x = NULL
    y = NULL
    xend = NULL
    yend = NULL
    sdy = NULL
    cvy = NULL
    
    plt = ggplot() +  geom_violin(data = df, aes(x=as.numeric(effect),y = x, weight = y, fill=effect), width = 0.5, alpha = 0.7, adjust = 0.2) +
      geom_text(data = qtl, aes(x=xend, y=y, label = fmt(y)), size = txt_size, family = "", vjust = -0.5, hjust = 1.1) + 
      geom_text(data = expec, aes(x=xend, y=y, label = paste0(fmt(y)," +/- ", fmt(sdy), " [",round(100*cvy),"%]")), size = txt_size, family = "", vjust = -0.5, angle = 90) + 
      geom_segment(data = qtl, aes(x=x,xend=xend,y=y,yend=yend), linetype = 1, alpha = 0.2) +
      geom_segment(data = expec, aes(x=x,xend=xend,y=y,yend=yend), alpha = 0.5, linetype = 3) +
      geom_segment(data = sdev, aes(x=x,xend=xend,y=y,yend=yend), alpha = 0.5, linetype = 1) +
      geom_point(data = jit, aes(x=as.numeric(effect), y = y), shape = 95, size = 3, alpha = 0.05) +
      ylab(paste0("integral_{", paste(ggopts$idims, collapse = ","), "} (", ggopts$predictor, ")")) + 
      guides(fill=FALSE) + 
      scale_x_continuous(name = "", breaks = 1:length(pnames), labels = pnames) +
      scale_fill_brewer(palette = "YlOrRd", direction = -1) # YlOrRd, PuBu
    
    # This still causes a warning since the violin plot does not like the width argument
    # suppressWarnings(print(plt))  
    return(plt)
    
  } 
  
    else if ( attr(data,"type") == "0d" ) {
      dnames = colnames(data)
      ggp = ggplot()
      
      if ( "summary" %in% names(attributes(data)) ){
        smy = attr(data, "summary")
        ggp = ggp + geom_ribbon(data = smy$inner.marg, ymin = 0, aes_string(x = dnames[1], ymax = dnames[2]), alpha = 0.1, colour = "grey")
      } 
      
      ggp = ggp + geom_path(data = data, aes_string(x = dnames[1], y = dnames[2]))
      
      if (length(pnames) == 1) { ggp = ggp + guides(color=FALSE, fill=FALSE) } 

      ggp
    }

    else if ( attr(data,"type") == "1d" ) {
      
      data = do.call(rbind, lapply(1:length(args), function(k) data.frame(args[[k]], Prediction = pnames[[k]])))
    
      ggp = ggplot(data = data) + 
              geom_ribbon(aes_string(x = ggopts$dims, ymin = "lq", ymax = "uq",  fill = "Prediction"), alpha = 0.3) + 
              geom_line(aes_string(x = ggopts$dims, y = "median", color = "Prediction"), size = 1.1) +
              xlab(ggopts$predictor) +
              ylab(ggopts$predictor[2])
      
      # Show guide only for multiple graphs
      if (length(pnames) == 1) { ggp = ggp + guides(color=FALSE, fill=FALSE) } 
      
      # Plot graph
      ggp
    }
      
     
    else if ( attr(data,"type") == "spatial" ) {
      # ggplot() + gg.col(ggopts$data, color = data$mean) + scale_fill_gradientn(colours = topo.colors(100))
      pixelplot.mesh(mesh = ggopts$mesh, col = data[,property], property = property, ...)
  }
}


#' @title Multiple ggplots on a page.
#'
#' @description
#' Renders multiple ggplots on a single page. 
#'
#' @param ... Comma-separated \code{ggplot} objects.
#' @param plotlist A list of \code{ggplot} objects - an alternative to the comma-separated argument above.
#' @param cols Number of columns of plots on the page.
#' @param layout A matrix specifying the layout. If present, 'cols' is ignored. If the layout is 
#' something like matrix(c(1,2,3,3), nrow=2, byrow=TRUE), then plot 1 will go in the upper left, 
#' 2 will go in the upper right, and 3 will go all the way across the bottom.
#' 
#' @author David L. Borchers <\email{dlb@@st-andrews.ac.uk}>
#'
#' @source 
#' \url{http://www.cookbook-r.com/Graphs/Multiple_graphs_on_one_page_(ggplot2)/}
#'  
#' @examples 
#' df = data.frame(x=1:10,y=1:10,z=11:20)
#' pl1 = ggplot(data = df) + geom_line(mapping = aes(x,y), color = "red")
#' pl2 = ggplot(data = df) + geom_line(mapping = aes(x,z), color = "blue")
#' multiplot(pl1,pl2, cols = 2)
#' 
#' @export
#
multiplot <- function(..., plotlist=NULL, cols=1, layout=NULL) {
  
  # Make a list from the ... arguments and plotlist
  plots <- c(list(...), plotlist)
  
  numPlots = length(plots)
  
  # If layout is NULL, then use 'cols' to determine layout
  if (is.null(layout)) {
    # Make the panel
    # ncol: Number of columns of plots
    # nrow: Number of rows needed, calculated from # of cols
    layout <- matrix(seq(1, cols * ceiling(numPlots/cols)),
                     ncol = cols, nrow = ceiling(numPlots/cols))
  }
  
  if (numPlots==1) {
    print(plots[[1]])
    
  } else {
    # Set up the page
    grid::grid.newpage()
    grid::pushViewport(grid::viewport(layout = grid::grid.layout(nrow(layout), ncol(layout))))
    
    # Make each plot, in the correct location
    for (i in 1:numPlots) {
      # Get the i,j matrix positions of the regions that contain this subplot
      matchidx <- as.data.frame(which(layout == i, arr.ind = TRUE))
      
      print(plots[[i]], vp = grid::viewport(layout.pos.row = matchidx$row,
                                      layout.pos.col = matchidx$col))
    }
  }
}



# Default color palette for plots using \link{gg}
# 
# @aliases bru.pal
# @name bru.pal
# @export
# @param 
# @return Color values as returned by \code{RColorBrewer::brewer.pal(9, "YlOrRd")}

bru.pal = function() {
  # library(RColorBrewer)
  # brewer.pal(9, "YlOrRd")
  pal = c("#FFFFCC","#FFEDA0","#FED976","#FEB24C","#FD8D3C","#FC4E2A","#E31A1C","#BD0026","#800026")
}

