#' Build Package Vignettes
#'
#' Build package vignettes from their source files.
#'
#' @param dir 'character' string.
#'   Path to a package's root source directory, by default the \link[=getwd]{working directory}.
#'   Its subdirectory \sQuote{vignettes} is searched for vignette source files.
#' @param doc 'character' string.
#'   Path to write the vignette output files, by default \sQuote{inst/doc} under the working directory.
#' @param gs_quality 'character' string.
#'   Quality of compacted PDF files, the options are
#'   \code{"ebook"} (150 dpi, default), \code{"printer"} (300 dpi),
#'   \code{"screen"} (72 dpi), and \code{"none"} (no compression).
#' @param clean 'logical' flag.
#'   Whether to remove all intermediate files generated by the build.
#' @param quiet 'logical' flag.
#'   Whether to suppress most output.
#'
#' @return Used for the side-effect of creating the PDF and (or) HTML documents
#'   of the package vignettes, and their executable code files.
#'
#' @author J.C. Fisher, U.S. Geological Survey, Idaho Water Science Center
#'
#' @seealso \code{\link[tools]{buildVignettes}}, \code{\link[tools]{compactPDF}}
#'
#' @keywords utilities
#'
#' @export
#'

BuildVignettes <- function(dir=getwd(), doc=file.path(dir, "inst/doc"),
                           gs_quality=c("ebook", "printer", "screen", "none"),
                           clean=TRUE, quiet=TRUE) {

  checkmate::assertFileExists(file.path(dir, "DESCRIPTION"))
  checkmate::assertPathForOutput(doc, overwrite=TRUE)
  gs_quality <- match.arg(gs_quality)
  checkmate::assertFlag(clean)
  checkmate::assertFlag(quiet)

  tools::buildVignettes(dir=dir, quiet=quiet, clean=clean, tangle=TRUE)

  v <- tools::pkgVignettes(dir=dir, output=TRUE, source=TRUE)
  if (length(v) == 0) return(invisible(NULL))
  out <- c(v$outputs, unique(unlist(v$sources, use.names=FALSE)))

  if (v$dir != doc) {
    dir.create(doc, showWarnings=!quiet, recursive=TRUE)
    file.copy(c(v$docs, out), doc, overwrite=TRUE)
    file.remove(out)
  }

  if (gs_quality != "none") {
    gs_cmd <- Sys.getenv("R_GSCMD", tools::find_gs_cmd())
    tools::compactPDF(paths=doc, gs_cmd=gs_cmd, gs_quality=gs_quality)
  }

  invisible(TRUE)
}
