#' @rdname fit_functions
#' @aliases fit.sl fit.sb fit.su qtls
#' 
#' @title Fit Functions for Johnson Curves
#' 
#' @description Fit functions for three Johnson Curves.
#' @details Three types of transformations are SB, SL and SU. Their forms are 
#' described below:
#' \deqn{S_B:   Z = \gamma + \eta * ln((X-\epsilon) / (\lambda + \epsilon - X))}
#' \deqn{S_L:   Z = \gamma + \eta * ln(X - \epsilon)}
#' \deqn{S_U:   Z = \gamma + \eta * asinh((X-\epsilon) / \lambda)}
#' in whihc Z is the standard normal varible, and X is the non-normal original
#' data, all the necessary parameters will be returned. Before fitting these 
#' curves, sample quantiles should be calculated according to z values. the 
#' \code{qtls} function here is to provide every useful parameters for Johnson 
#' curve fitting.
#' 
#' @param x the non-normal numerical data.
#' @param q the quantiles and some statistics generated by quantiles, it must 
#'          be the return value of \code{qtls} function.
#' @param z current z value, Mandraccia, Halverson and Chou (1996) suggest 
#' using 101 z values from 0.25 to 1.25.
#' 
#' @export
#' 
#' @return return NA when fit failed, return a list with 2 component when fit 
#' succeeded. The first component \code{trans} is the transformed value
#' and the second component \code{params} is the parameters used in the 
#' transformation.
fit.sb <- function(x, q) {
  eta <- q$z / acosh(.5 * sqrt((1 + q$xm / q$xu) * (1 + q$xm / q$xl)))  
  gamma <- eta * asinh((q$xm / q$xl - q$xm / q$xu) * 
                         sqrt((1 + q$xm / q$xu) * 
                                (1 + q$xm / q$xl) - 4) / 
                         (2 * (q$xm^2 / q$xl / q$xu - 1))) 
  lambda <- (q$xm * sqrt(((1 + q$xm / q$xu) * 
                            (1 + q$xm / q$xl) - 2)^2 - 4) / 
               (q$xm^2 / q$xl / q$xu - 1))
  epsilon <- .5 * (q$x2 + q$x3 - lambda + 
                     q$xm * (q$xm / q$xl - q$xm / q$xu) / 
                     (q$xm^2 / q$xl / q$xu - 1))
  
  if (is.nan(gamma) | is.nan(epsilon) | eta <= 0 | lambda <= 0) 
    return(NA)
  
  if (all(x > epsilon) & all(x < epsilon + lambda)) {
    return(list(trans  = gamma + eta *  log((x - epsilon) / 
                                              (lambda + epsilon - x)),
                params = c(eta, gamma, lambda, epsilon, q$z)))
  } else return(NA)
}
