% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/aoristic.R
\docType{methods}
\name{aoristic}
\alias{aoristic}
\alias{aoristic-method}
\alias{aoristic,numeric,numeric-method}
\alias{aoristic,ANY,missing-method}
\title{Aoristic Analysis}
\usage{
aoristic(x, y, ...)

\S4method{aoristic}{numeric,numeric}(
  x,
  y,
  step = 1,
  start = min(x),
  end = max(y),
  calendar = CE(),
  weight = TRUE,
  groups = NULL
)

\S4method{aoristic}{ANY,missing}(
  x,
  step = 1,
  start = NULL,
  end = NULL,
  calendar = CE(),
  weight = TRUE,
  groups = NULL
)
}
\arguments{
\item{x, y}{A \code{\link{numeric}} vector giving the lower and upper boundaries of the
time intervals, respectively. If \code{y} is missing, an attempt is made to
interpret \code{x} in a suitable way (see \code{\link[grDevices:xy.coords]{grDevices::xy.coords()}}).}

\item{...}{Currently not used.}

\item{step}{A length-one \code{\link{integer}} vector giving the step size, i.e. the
width of each time step in the time series (defaults to \eqn{1},
i.e. annual level).}

\item{start}{A length-one \code{\link{numeric}} vector giving the beginning of the time
window.}

\item{end}{A length-one \code{\link{numeric}} vector giving the end of the time
window.}

\item{calendar}{An \code{\link[aion:TimeScale-class]{aion::TimeScale}} object specifying the calendar
of \code{x} and \code{y} (see \code{\link[aion:calendar]{aion::calendar()}}). Defaults to Gregorian Common Era.}

\item{weight}{A \code{\link{logical}} scalar: should the aoristic sum be weighted by
the length of periods (default). If \code{FALSE} the aoristic sum is the number
of elements within a time block.}

\item{groups}{A \code{\link{factor}} vector in the sense that \code{as.factor(groups)}
defines the grouping. If \code{x} is a \code{list} (or a \code{data.frame}), \code{groups} can
be a length-one vector giving the index of the grouping component (column)
of \code{x}.}
}
\value{
An \code{\linkS4class{AoristicSum}} object.
}
\description{
Computes the aoristic sum.
}
\details{
Aoristic analysis is used to determine the probability of contemporaneity of
archaeological sites or assemblages. The aoristic analysis distributes the
probability of an event uniformly over each temporal fraction of the period
considered. The aoristic sum is then the distribution of the total number of
events to be assumed within this period.

Muller and Hinz (2018) pointed out that the overlapping of temporal
intervals related to period categorization and dating accuracy is likely to
bias the analysis. They proposed a weighting method to overcome this
problem. This method is not implemented here (for the moment), see the
\href{https://github.com/ISAAKiel/aoristAAR}{\pkg{aoristAAR} package}.
}
\examples{
## Data from Husi 2022
data("loire", package = "folio")

## Get time range
loire_range <- loire[, c("lower", "upper")]

## Calculate aoristic sum (normal)
aorist_raw <- aoristic(loire_range, step = 50, weight = FALSE)
plot(aorist_raw, col = "grey")

## Calculate aoristic sum (weights)
aorist_weighted <- aoristic(loire_range, step = 50, weight = TRUE)
plot(aorist_weighted, col = "grey")

## Calculate aoristic sum (weights) by group
aorist_groups <- aoristic(loire_range, step = 50, weight = TRUE,
                          groups = loire$area)
plot(aorist_groups, flip = TRUE, col = "grey")
image(aorist_groups)

## Rate of change
roc_weighted <- roc(aorist_weighted, n = 30)
plot(roc_weighted)

## Rate of change by group
roc_groups <- roc(aorist_groups, n = 30)
plot(roc_groups, flip = TRUE)
}
\references{
Crema, E. R. (2012). Modelling Temporal Uncertainty in Archaeological
Analysis. \emph{Journal of Archaeological Method and Theory}, 19(3): 440-61.
\doi{10.1007/s10816-011-9122-3}.

Johnson, I. (2004). Aoristic Analysis: Seeds of a New Approach to Mapping
Archaeological Distributions through Time. \emph{In} Ausserer, K. F., Börner, W.,
Goriany, M. & Karlhuber-Vöckl, L. (ed.), \emph{Enter the Past - The E-Way into
the Four Dimensions of Cultural Heritage}, Oxford: Archaeopress, p. 448-52.
BAR International Series 1227.
\doi{10.15496/publikation-2085}

Müller-Scheeßel, N. & Hinz, M. (2018). \emph{Aoristic Research in R: Correcting
Temporal Categorizations in Archaeology}. Presented at the Human History and
Digital Future (CAA 2018), Tubingen, March 21.
\url{https://www.youtube.com/watch?v=bUBukex30QI}.

Palmisano, A., Bevan, A. & Shennan, S. (2017). Comparing Archaeological
Proxies for Long-Term Population Patterns: An Example from Central Italy.
\emph{Journal of Archaeological Science}, 87: 59-72.
\doi{10.1016/j.jas.2017.10.001}.

Ratcliffe, J. H. (2000). Aoristic Analysis: The Spatial Interpretation of
Unspecific Temporal Events. \emph{International Journal of Geographical
Information Science}, 14(7): 669-79. \doi{10.1080/136588100424963}.

Ratcliffe, J. H. (2002). Aoristic Signatures and the Spatio-Temporal
Analysis of High Volume Crime Patterns. \emph{Journal of Quantitative
Criminology}, 18(1): 23-43. \doi{10.1023/A:1013240828824}.
}
\seealso{
Other aoristic analysis: 
\code{\link{plot_aoristic}},
\code{\link{roc}()}
}
\author{
N. Frerebeau
}
\concept{aoristic analysis}
