#' Labeled loop and break
#'
#' \code{label \%._.\% for} generates labeled loop.
#' To escape from this loop, replace `break` with \code{._.(label)}.
#' Pseudo-anonymous labeled loop can be generated by \code{.\%._.\%} and
#' break by \code{._.()}.
#'
#' @usage label \%._.\% statement
#' @param label label for this loop
#' @param statement statement such as \code{for}, \code{while} etc.
#' @rdname labeledLoop
#' @aliases labeledLoop package-labeledLoop labeledLoop-package
#' @export
#' @examples
#' . %._.% for(i in 1:3) {  
#'   foo %._.% for (j in 1:3) {
#'     orz %._.% for (k in 1:3) {
#'       print(c(i, j, k))
#'       
#'       if (i == 3 && j == 3 && k == 2) {
#'         cat("escape from outmost loop\n")
#'         ._.() # break
#'       }
#' 
#'       if (i == 2 && j == 2) {
#'         cat("escape from innermost loop (orz)\n")
#'         ._.(orz) # break
#'       }
#' 
#'       if (i == 1 && j == 1 && k == 2) {
#'         cat("escape from foo\n")
#'         ._.(foo) # break
#'       }
#'     }
#'   }
#' }
`%._.%` <- function(label, statement) {
  label <- substitute(label)
  if (!is.character(label)) label <- deparse(substitute(label))
  arg <- list(as.name("statement"), label)
  names(arg) <- c("expr", label)
  do.call("withRestarts", arg)
  invisible()
}

#' @rdname labeledLoop
#' @inheritParams %._.%
#' @export
`._.` <- function(label = ".") {
  label <- substitute(label)
  if (!is.character(label)) label <- deparse(substitute(label))
  invokeRestart(label)
}
