% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metrics.r
\docType{data}
\name{stdmetrics}
\alias{stdmetrics}
\alias{.stdmetrics}
\alias{stdmetrics_z}
\alias{.stdmetrics_z}
\alias{stdmetrics_i}
\alias{.stdmetrics_i}
\alias{stdmetrics_rn}
\alias{.stdmetrics_rn}
\alias{stdmetrics_pulse}
\alias{.stdmetrics_pulse}
\alias{stdmetrics_ctrl}
\alias{.stdmetrics_ctrl}
\alias{stdtreemetrics}
\alias{.stdtreemetrics}
\title{Predefined standard metrics functions}
\format{An object of class \code{expression} of length 1.}
\usage{
stdmetrics(x, y, z, i, a, rn, class, dz = 1)

.stdmetrics

stdmetrics_z(z, dz = 1)

.stdmetrics_z

stdmetrics_i(i, z = NULL, class = NULL, rn = NULL)

.stdmetrics_i

stdmetrics_rn(rn, class = NULL)

.stdmetrics_rn

stdmetrics_pulse(pulseID, rn)

.stdmetrics_pulse

stdmetrics_ctrl(x, y, z, a)

.stdmetrics_ctrl

stdtreemetrics(x, y, z)

.stdtreemetrics
}
\arguments{
\item{x, y, z, i, a}{Coordinates of the points, Intensity and ScanAngle}

\item{rn, class}{ReturnNumber, Classification}

\item{dz}{Layer thickness for metrics requiring this data, such as \link[lidR:entropy]{entropy}}

\item{pulseID}{The number referencing each pulse}
}
\description{
Predefined functions usable in \link{grid_metrics}, \link{grid_hexametrics}, \link{lasmetrics}, \link{tree_metrics},
and their convenient shortcuts. The philosophy of the \code{lidR} package is to provide an easy way
to compute user-defined metrics rather than to provide them. However, for efficiency and to save time, a set of
standard metrics has been predefined. To use these functions please read the Details and Examples sections.
}
\details{
The function names, their parameters and the output names of the metrics rely on a nomenclature chosen for brevity:
\itemize{
\item{\code{z}: refers to the elevation}
\item{\code{i}: refers to the intensity}
\item{\code{rn}: refers to the return number}
\item{\code{q}: refers to quantile}
\item{\code{a}: refers to the ScanAngle}
\item{\code{n}: refers to a number (a count)}
\item{\code{p}: refers to a percentage}
}
For example the metric named \code{zq60} refers to the elevation, quantile, 60 i.e. the 60th percentile of elevations.
The metric \code{pground} refers to a percentage. It is the percentage of points classified as ground.
The function \code{stdmetric_i} refers to metrics of intensity. A description of each existing metric can be found
on the \href{https://github.com/Jean-Romain/lidR/wiki/stdmetrics}{lidR wiki page}.\cr\cr
Some functions have optional parameters. If these parameters are not provided the function
computes only a subset of existing metrics. For example \code{stdmetrics_i} requires the intensity
values, but if the elevation values are provided it can compute additional metrics such as cumulative
intensity at a given percentile of height.\cr\cr
Each function has a convenient associated variable. It is the name of the function, with a
dot before the name. This enables the function to be used without writing parameters. The cost
of such a feature is inflexibility. It corresponds to a predefined behaviour (see examples)\cr\cr
\code{stdtreemetrics} is a special function which works with \link{tree_metrics}. Actually
it won't fail with other functions but the output make sense more sense if computed at the
individual tree level.
}
\examples{
LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")
lidar = readLAS(LASfile, select = "*")

# All the predefined functions
lidar \%>\% grid_metrics(stdmetrics(X,Y,Z,Intensity, ScanAngle,
                                  ReturnNumber, Classification,
                                  dz = 1))

# Convenient shortcut
lidar \%>\% grid_metrics(.stdmetrics)

# Basic metrics from intensities
lidar \%>\% grid_metrics(stdmetrics_i(Intensity))

# All the metrics from intensities
lidar \%>\% grid_metrics(stdmetrics_i(Intensity, Z, Classification, ReturnNumber))

# Convenient shortcut for the previous example
lidar \%>\% grid_metrics(.stdmetrics_i)

# Compute the metrics only on first return
lidar \%>\% lasfilterfirst \%>\% grid_metrics(.stdmetrics_z)

# Compute the metrics with a threshold at 2 meters
lidar \%>\% lasfilter(Z > 2) \%>\% grid_metrics(.stdmetrics_z)

# Works also with lasmetrics and grid_hexametrics
lidar \%>\% lasmetrics(.stdmetrics)
lidar \%>\% grid_hexametrics(.stdmetrics)

# Combine some predefined function with your own new metrics
# Here convenient shortcuts are no longer usable.
myMetrics = function(z, i)
{
  metrics = list(
     zwimean = sum(z*i)/sum(i), # Mean elevation weighted by intensities
     zimean  = mean(z*i),       # Mean products of z by intensity
     zsqmean = sqrt(mean(z^2))  # Quadratic mean
   )

  return( c(metrics, stdmetrics_z(z)) )
}

lidar \%>\% grid_metrics(myMetrics(Z, Intensity))

# You can write your own convenient shorcuts like this:
.myMetrics = expression(myMetrics(Z,Intensity))

lidar \%>\% grid_metrics(.myMetrics)
}
\seealso{
\link{grid_metrics}
\link{lasmetrics}
\link{grid_hexametrics}
\link{grid_metrics3d}
\link{tree_metrics}
}
\keyword{datasets}
